import torch
import random
import argparse
import numpy as np
from copy import deepcopy
from model_utils import Net
from eval_utils import infer_result, save_result
from data_utils import prepare_dataloader, partition_data, adjacency
import os

def main(args):
    (
        source_dataset,
        source_dataloader_train,
        source_dataloader_eval,
        target_dataset,
        target_dataloader_train,
        target_dataloader_eval,
        gene_num,
        type_num,
        label_map,
        source_adata,
        target_adata,
    ) = prepare_dataloader(args)
    # After preprocessing, obtain in sequence:
    # Preprocessed RNA-seq, take batch RNA-seq for training, take batch RNA-seq for evaluation
    # Preprocessed ATAC-seq, take batch ATAC-seq for training, take batch ATAC-seq for evaluation
    # RNA (shared) gene number, RNA cell type number, cell type from int==>string lookup table
    # anndata: RNA-seq, ATAC-seq

    source_dataloader_eval_all = deepcopy(source_dataloader_eval)
    target_dataloader_eval_all = deepcopy(target_dataloader_eval)
    if args.novel_type:
        target_adj = adjacency(target_dataset.tensors[0])
    else:
        target_adj = None

    # Count the number of each cell type; and make the smaller the number of a cell type, the larger the weight allocated
    source_label = source_dataset.tensors[1]
    count = torch.unique(source_label, return_counts=True, sorted=True)[1]
    ce_weight = 1.0 / count
    ce_weight = ce_weight / ce_weight.sum() * type_num
    ce_weight = ce_weight.cuda()

    print("======= Training Start =======")

    net = Net(gene_num, type_num, ce_weight, args).cuda()
    preds, prob_feat, prob_logit = net.run(
        source_dataloader_train,
        source_dataloader_eval,
        target_dataloader_train,
        target_dataloader_eval,
        target_adj,
        args,
            )   # Get a clustering result; and two parameters generated by GMM during the clustering process    initialization/warmup

    for iter in range(args.max_iteration):
        (
            source_dataloader_train,
            source_dataloader_eval,
            target_dataloader_train,
            target_dataloader_eval,
            source_dataset,
            target_dataset,
        ) = partition_data(
            preds,
            prob_feat,
            prob_logit,
            source_dataset,
            target_dataset,
            args,
        )   # Iteratively update dataset (training set, validation set)

        # Iteration convergence check discard data smaller than batch-size
        if target_dataset.__len__() <= args.batch_size:
            break
        print("======= Iteration:", iter, "=======")

        source_label = source_dataset.tensors[1]
        count = torch.unique(source_label, return_counts=True, sorted=True)[1]  # Training set updated, so cross-entropy weights for each cell type should also be updated
        ce_weight = 1.0 / count
        ce_weight = ce_weight / ce_weight.sum() * type_num
        ce_weight = ce_weight.cuda()

        net = Net(gene_num, type_num, ce_weight, args).cuda()
        preds, prob_feat, prob_logit = net.run(
            source_dataloader_train,
            source_dataloader_eval,
            target_dataloader_train,
            target_dataloader_eval,
            target_adj,
            args,
        )
    print("======= Training Done =======")

    # Inference and save results
    infer_result(
        net,
        source_dataloader_eval_all,
        target_dataloader_eval_all,
        source_adata,
        target_adata,
        label_map,
        args,
    )

if __name__ == "__main__":
    # User input parameters
    # Note: The actual file paths will be constructed as data_path + source_data and data_path + target_data
    data_path = "Please enter data directory path"
    source_data = "Please enter Gene Expression Matrix filename"
    target_data = "Please enter Gene Activity Scores filename"
    exp_id = "Please enter experiment ID"
    output_path = "Please enter output path"
    
    parser = argparse.ArgumentParser()
    parser.add_argument("--data_path", type=str, default=data_path)
    parser.add_argument("--source_data", type=str, default=source_data)
    parser.add_argument("--target_data", type=str, default=target_data)
    parser.add_argument("--exp_id", type=str, default=exp_id)
    parser.add_argument("--output_path", type=str, default=output_path)
    parser.add_argument("--batch_size", type=int, default=128)
    parser.add_argument("--max_iteration", type=int, default=10)
    parser.add_argument("--novel_type", action="store_true")
    parser.add_argument("--lr", type=float, default=0.001)
    parser.add_argument("--epochs", type=int, default=100)
    parser.add_argument("--hidden_dim", type=int, default=128)
    parser.add_argument("--dropout", type=float, default=0.1)
    parser.add_argument("--weight_decay", type=float, default=1e-5)
    parser.add_argument("--patience", type=int, default=20)
    parser.add_argument("--seed", type=int, default=42)
    
    args = parser.parse_args([])
    
    # Set random seed
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    if torch.cuda.is_available():
        torch.cuda.manual_seed(args.seed)
    
    main(args)

