#this publicly available data and code can be found at https://github.com/mihem/pns_atlas

library(qs)
library(ggplot2)
organism = "org.Hs.eg.db"
library(organism, character.only = TRUE)

if (!file.exists("de_pseudo_pnp_ctrl.qs")) {
  download.file(
    "https://zenodo.org/records/14551660/files/de_pseudo_pnp_ctrl.qs?download=1",
    "de_pseudo_pnp_ctrl.qs"
  )
}
de_pseudo_pnp_ctrl <- qread("de_pseudo_pnp_ctrl.qs")

# volcano plot function
volcanoPlot <- function(cluster, input, FCcutoff = 2, selectLab = NULL, drawConnectors = TRUE, condition1, condition2) {
  input <- input[[cluster]]
  volcano <- EnhancedVolcano::EnhancedVolcano(
    data.frame(input),
    lab = paste0("italic('", input[["gene"]], "')"),
    x = "avg_logFC",
    y = "p_val_adj",
    xlim = c(min(input[["avg_logFC"]], max(input[["avg_logFC"]]))),
    ylim = c(0, max(-log10(input[["p_val_adj"]]))),
    pCutoff = 0.1,
    FCcutoff = FCcutoff,
    axisLabSize = 15,
    pointSize = 2,
    labSize = 5,
    subtitle = NULL,
    caption = NULL,
    border = "full",
    gridlines.major = FALSE,
    gridlines.minor = FALSE,
    drawConnectors = drawConnectors,
    lengthConnectors = unit(0.0001, "npc"),
    title = paste(condition1, "vs", condition2, "in ", cluster),
    boxedLabels = TRUE,
    selectLab = selectLab,
    xlab = bquote(~ Log[2] ~ "fold change"),
    ylab = bquote(~ -Log[10] ~ "adjusted p-value"),
    parseLabels = TRUE,
    legendLabels = c(
      "NS", "logFC",
      "p-val", "p-val + logFC"
    ),
    legendPosition = "right",
  )
}

# define clusters of interest
cluster_de <- c("mySC", "nmSC", "repairSC", "PC2")

# define genes of interests
lab_pnp_ctrl <- list(
  "mySC" = paste0("italic('", c("XIST"), "')"),
  "nmSC" = paste0("italic('", c("XIST"), "')"),
  "repairSC" = paste0("italic('", c("XIST"), "')"),
  "PC2" = paste0("italic('", c("XIST"), "')")
)

#plots
volcano_plots <- lapply(cluster_de, function(cluster) {
  volcanoPlot(
    cluster = cluster,
    input = de_pseudo_pnp_ctrl,
    FCcutoff = 2,
    condition1 = "PNP",
    condition2 = "CTRL",
    selectLab = lab_pnp_ctrl[[cluster]]
  )
})

#arrange
ggpubr::ggarrange(
  volcano_plots[[1]], volcano_plots[[2]], volcano_plots[[3]],
  align = "h",
  common.legend = TRUE
)

#get overlap between cilia
cilia_genes <- AnnotationDbi::select(org.Hs.eg.db, 
                                     keys = "GO:0003351", 
                                     keytype = "GOALL", 
                                     columns = c("SYMBOL", "ENSEMBL", "GENENAME"))

#get actin movement genes
muscle_genes <- AnnotationDbi::select(org.Hs.eg.db, 
                                      keys = "GO:0030048", 
                                      keytype = "GOALL", 
                                      columns = c("SYMBOL", "ENSEMBL", "GENENAME"))

#get means
mean(de_pseudo_pnp_ctrl$nmSC[de_pseudo_pnp_ctrl$nmSC$gene %in% cilia_genes$SYMBOL,]$avg_logFC)

mean(de_pseudo_pnp_ctrl$nmSC[de_pseudo_pnp_ctrl$nmSC$gene %in% muscle_genes$SYMBOL,]$avg_logFC)
