/*
 * Author: Richard M. Leggett
 * © Copyright 2021-22 Earlham Institute
 */
package uk.ac.earlham.marti.centrifuge;

import java.io.BufferedReader;
import java.io.File;
import uk.ac.earlham.marti.core.MARTiAlert;
import uk.ac.earlham.marti.core.MARTiEngineOptions;
import uk.ac.earlham.marti.schedule.SlurmScheduler;

/**
 *
 * @author martins
 */
public class CentrifugeProcess {
    
    private MARTiEngineOptions options;
    private String centrifugeName = null;
    private String centrifugeDatabase = null;
    private int numThreads = 1;
    private boolean classifyThis = false;
    private String centrifugeMemory = null;
    private String jobQueue = null;
    private int minHitLen = 100;
    private int primaryAssignments = 1;
    private String taxaFilter = "";
    private String processOptions = "";
    
    public CentrifugeProcess(MARTiEngineOptions o) {
        options = o;
    }
    
    
    public String readConfigFile(BufferedReader br) {
        String line = null;
        
        // Default job queue
        jobQueue = options.getQueue();
        
        boolean keepReading = true;
        try {
            do {
                line = br.readLine();
                if (line != null) {
                    line = line.trim();
                    if (line.length() > 1) {
                        if (!line.startsWith("#")) {
                            String[] tokens = line.split(":");
                            if (tokens[0].compareToIgnoreCase("Name") == 0) {
                                centrifugeName = tokens[1];
                            } else if (tokens[0].compareToIgnoreCase("Database") == 0) {
                                centrifugeDatabase = tokens[1];
                            } else if (tokens[0].compareToIgnoreCase("Memory") == 0) {
                                centrifugeMemory = tokens[1];
                            } else if (tokens[0].compareToIgnoreCase("Queue") == 0) {
                                jobQueue = tokens[1];
                            } else if (tokens[0].compareToIgnoreCase("CentrifugeThreads") == 0) {
                                numThreads = Integer.parseInt(tokens[1]);
                            } else if (tokens[0].compareTo("UseToClassify") == 0) {
                                classifyThis = true;
                            } else if (tokens[0].compareToIgnoreCase("MinHitLen") == 0) {
                                minHitLen = Integer.parseInt(tokens[1]);
                            } else if (tokens[0].compareToIgnoreCase("options") == 0) {
                                processOptions = tokens[1];
                            } else if (tokens[0].compareToIgnoreCase("TaxaFilter") == 0) {
                                if (tokens[1].charAt(0) != '"') {
                                    taxaFilter = tokens[1];
                                } else {
                                    taxaFilter = '"' + tokens[1] + '"';
                                }
                            } else {
                                keepReading = false;
                            }
                        }
                    }
                }              
            } while ((line != null) && (keepReading));
        } catch (Exception e) {
            System.out.println("readConfigFile Exception:");
            e.printStackTrace();
            System.exit(1);
        }
        
        if (centrifugeName == null) {
            System.out.println("Error: missing Centrifuge name.");
            options.addAlert(new MARTiAlert(MARTiAlert.TYPE_ERROR, "Error: missing name in Centrifuge process. Analysis stopped."));
            options.writeAlertsFile();
            System.exit(1);
        }

        if (centrifugeDatabase == null) {
            System.out.println("Error: missing Centrifuge database.");
            options.addAlert(new MARTiAlert(MARTiAlert.TYPE_ERROR, "Error: missing database in Centrifuge process. Analysis stopped."));
            options.writeAlertsFile();
            System.exit(1);
        }

        if (options.getJobScheduler() instanceof SlurmScheduler) {    
            if (centrifugeMemory == null) {
                System.out.println("Error: missing memory in Centrifuge process config.");
                options.addAlert(new MARTiAlert(MARTiAlert.TYPE_ERROR, "Error: missing memory in Centrifuge process. Analysis stopped."));
                options.writeAlertsFile();
                System.exit(1);
            }
            
            if (jobQueue == null) {
                System.out.println("Error: missing job queue in Centrifuge process config.");
                options.addAlert(new MARTiAlert(MARTiAlert.TYPE_ERROR, "Error: missing queue in Centrifuge process. Analysis stopped."));
                options.writeAlertsFile();
                System.exit(1);
            }
        }

        return line;
    }
    
    public String getName() {
        return centrifugeName;
    }
    
    public String getDatabase() {
        return centrifugeDatabase;
    }
      
    public String getJobQueue() {
        return jobQueue;
    }
    
    public String getMemory() {
        return centrifugeMemory;
    }
    
    public boolean useForClassifying() {
        return classifyThis;
    }
    
    public String getProcessDir() {
        return options.getSampleDirectory() + File.separator + "centrifuge_" + centrifugeName + File.separator;
    }
    
    public int getMinHitLen() {
        return minHitLen;
    }
    
    public int getNumThreads() {
        return numThreads;
    }
    
    public int getNumPrimaryAssignments() {
        return primaryAssignments;
    }

    public String getProcessOptions() {
        return processOptions;
    }   

    public String getTaxaFilter() {
        return taxaFilter;
    }
}
