# pylint: disable=W0621,C0114,C0116,W0212,W0613
import pathlib
import textwrap

import pytest
from dae.annotation.annotatable import VCFAllele
from dae.annotation.annotation_factory import load_pipeline_from_yaml
from dae.annotation.annotation_pipeline import AnnotationPipeline
from dae.annotation.score_annotator import AlleleScoreAnnotator
from dae.genomic_resources import build_genomic_resource_repository
from dae.genomic_resources.repository import (
    GR_CONF_FILE_NAME,
)
from dae.genomic_resources.testing import (
    build_filesystem_test_repository,
    convert_to_tab_separated,
    setup_directories,
    setup_vcf,
)


@pytest.fixture
def annotation_pipeline(tmp_path: pathlib.Path) -> AnnotationPipeline:
    root_path = tmp_path
    setup_directories(
        root_path / "grr", {
            "allele_score": {
                GR_CONF_FILE_NAME: """
                    type: allele_score
                    allele_score_mode: alleles
                    table:
                        filename: data.txt
                        reference:
                          name: reference
                        alternative:
                          name: alternative
                    scores:
                        - id: ID
                          type: str
                          desc: "variant ID"
                          name: ID
                        - id: freq
                          type: float
                          desc: ""
                          name: freq
                    default_annotation:
                    - source: freq
                      name: allele_freq

                """,
                "data.txt": convert_to_tab_separated("""
                    chrom  pos_begin  reference  alternative ID  freq
                    1      10         A          G           ag  0.02
                    1      10         A          C           ac  0.03
                    1      10         A          T           at  0.04
                    1      16         CA         G           .   0.03
                    1      16         C          T           ct  0.04
                    1      16         C          A           ca  0.05
                    1      16         C          CA          ca  1.0
                    1      16         C          CG          ca  2.0
                """),
            },
        })
    local_repo = build_genomic_resource_repository({
        "id": "allele_score_local",
        "type": "directory",
        "directory": str(root_path / "grr"),
    })
    annotation_configuration = textwrap.dedent("""
        - allele_score:
            resource_id: allele_score
    """)
    return load_pipeline_from_yaml(annotation_configuration, local_repo)


def test_allele_score_annotator_attributes(
    annotation_pipeline: AnnotationPipeline,
) -> None:

    pipeline = annotation_pipeline
    annotator = pipeline.annotators[0]

    assert isinstance(annotator, AlleleScoreAnnotator)
    assert not annotator.is_open()

    attributes = annotator.get_info().attributes
    assert len(attributes) == 1
    assert attributes[0].name == "allele_freq"
    assert attributes[0].source == "freq"
    assert attributes[0].type == "float"
    assert attributes[0].description == ""


@pytest.mark.parametrize("variant, expected", [
    (("1", 10, "A", "G"), 0.02),
    (("1", 10, "A", "C"), 0.03),
    (("1", 10, "A", "T"), 0.04),
    (("1", 16, "C", "T"), 0.04),
    (("1", 16, "C", "A"), 0.05),
    (("1", 16, "CA", "G"), 0.03),
    (("1", 16, "C", "CG"), 2.0),
    (("1", 16, "C", "CA"), 1.0),
])
def test_allele_score_with_default_score_annotation(
    variant: tuple, expected: float,
    tmp_path: pathlib.Path,
) -> None:
    root_path = tmp_path
    setup_directories(
        root_path / "grr", {
            "allele_score": {
                GR_CONF_FILE_NAME: """
                    type: allele_score
                    allele_score_mode: alleles
                    table:
                        filename: data.txt
                        reference:
                          name: reference
                        alternative:
                          name: alternative
                    scores:
                        - id: ID
                          type: str
                          desc: "variant ID"
                          name: ID
                        - id: freq
                          type: float
                          desc: ""
                          name: freq
                    default_annotation:
                    - source: freq
                      name: allele_freq

                """,
                "data.txt": convert_to_tab_separated("""
                    chrom  pos_begin  reference  alternative ID  freq
                    1      10         A          G           ag  0.02
                    1      10         A          C           ac  0.03
                    1      10         A          T           at  0.04
                    1      16         CA         G           .   0.03
                    1      16         C          T           ct  0.04
                    1      16         C          A           ca  0.05
                    1      16         C          CA          ca  1.0
                    1      16         C          CG          ca  2.0
                """),
            },
        })
    local_repo = build_genomic_resource_repository({
        "id": "allele_score_local",
        "type": "directory",
        "directory": str(root_path / "grr"),
    })
    annotation_configuration = textwrap.dedent("""
        - allele_score:
            resource_id: allele_score
    """)
    pipeline = load_pipeline_from_yaml(annotation_configuration, local_repo)

    annotatable = VCFAllele(*variant)
    result = pipeline.annotate(annotatable)
    assert len(result) == 1
    assert result["allele_freq"] == expected


@pytest.mark.parametrize("allele, expected", [
    (("chr1", 1, "C", "A"), 0.001),
    (("chr1", 11, "C", "A"), 0.1),
    (("chr1", 21, "C", "A"), 0.2),
    (("chr1", 31, "C", "CA"), 0.3),
    (("chr1", 31, "C", "CG"), 1.0),
])
def test_allele_annotator_add_chrom_prefix_vcf_table(
        tmp_path: pathlib.Path, allele: tuple, expected: float) -> None:

    setup_directories(
        tmp_path, {
            "allele_score1": {
                "genomic_resource.yaml": textwrap.dedent("""
                    type: allele_score
                    allele_score_mode: alleles
                    table:
                        filename: data.vcf.gz
                        format: vcf_info
                        chrom_mapping:
                            add_prefix: chr
                    scores:
                    - id: test100way
                      type: float
                      desc: "test values"
                      name: test100way
                    """),
            },
        })

    setup_vcf(
        tmp_path / "allele_score1" / "data.vcf.gz",
        textwrap.dedent("""
        ##fileformat=VCFv4.1
        ##INFO=<ID=test100way,Number=1,Type=Float,Description="test values">
        ##contig=<ID=1>
        #CHROM POS ID REF ALT QUAL FILTER INFO
        1      1   .  C   A   .    .      test100way=0.001;
        1      11  .  C   A   .    .      test100way=0.1;
        1      21  .  C   A   .    .      test100way=0.2;
        1      31  .  C   CA  .    .      test100way=0.3;
        1      31  .  C   G   .    .      test100way=0.4;
        1      31  .  C   CG  .    .      test100way=1.0;
        """))
    repo = build_filesystem_test_repository(tmp_path)

    pipeline_config = textwrap.dedent("""
            - allele_score:
                resource_id: allele_score1
                attributes:
                - source: test100way
            """)

    pipeline = load_pipeline_from_yaml(pipeline_config, repo)
    with pipeline.open() as work_pipeline:
        annotatable = VCFAllele(*allele)
        result = work_pipeline.annotate(annotatable)

        print(annotatable, result)
        assert result.get("test100way") == pytest.approx(expected, rel=1e-3)
