#!/bin/Rscript
#
# Written by Joonoh Lim (Ju lab; joonoh.lim@kaist.ac.kr)
#
# First created on 2019-05-21
# Last modified on 2021-08-05
#
################################################################################
NOTE="
#=================#
# VERSION HISTORY #
#=================#
Version.1.0.0 2019-05-22: the first code has been implemented
Version.1.0.1 2019-05-23: default for signature db is set to its absolute path; VCF including indel calls is supported
Version 1.0.2 2019-05-25: display help message if no input is supplied
Version 1.0.3 2019-05-31: chromosomes or contigs other than 1~22, X, and Y can now be properly handled (e.g., GL000211.1 -> chrUn_gl000211)
Version 1.0.4 2019-06-05: added a new argument for output prefix. Fixed error occrring when no extra chromosome is in in vcf
Version 1.0.5 2019-06-11: suggested by CJYoon, added a functionality to make use of vaf information to give weights to each mutations (i.e., SNVs)
Version 1.0.6 2019-06-12: corrected a minor bug regarding output directory
Version 1.0.7 2019-06-17: resolved a bottleneck for computation speed.
Version 1.0.8 2019-06-20: added mm10 genome.
Version 1.0.9 2019-07-04: added new arguments to allow input of a table format
Version 1.1.0 2019-08-08:
    1) removed chromosomesExtra, added chromosomes_of_interest and chromosome_prefix, for more robustness.
    2) now it generates one more output, a INFO file which currently only contains: cosine similarity.
Version 1.2.0 2019-08-19:
    1) removed the argument '--signature_data' and added a new '--signatureSet'
    2) modify the argument '--signature_number'. Previously '1:30' should now be 'all'.
Version 1.2.1 2019-08-22: contained the signature data inside the code. No need to care about the file paths.
Version 1.2.1 2019-08-26: corrected a small bug
Version 1.3.1 2019-10-21: added a new option \"COSMIC_v2v3\" to --signatureSet (-s) (requested by JHPark).
Version 1.3.2 2019-12-09: The code has been re-factored.
    1) New argument '-m or --mode' takes either \"SISO\" or \"MISO\" which stand for 'single-input single-output' and 'multiple-input single-output', respectively.
       The MISO mode takes in multiple files (either VCFs or TSVs or mixed (NOTE: mixed is only allowed when -t --type is NOT specified, i.e., NULL)), merge the counts across input data, and analyze mutational spectra.
    2) New argument '-z or --log' displays this update log message.
    3) New argument '-b or --outputBase', if set, overrides the input filename (suggested by CJYoon)
    4) A small problem with naming output file has been resolved. (reported by CJYoon)
    5) A small problem with specifying signature_name has been resolved. (reported by CJYoon)
\nVersion 1.3.3 2019-12-10:
    1) New argument -e --regions_of_interest has been added (requested by CJYoon)
    2) Some minor bugs have been fixed.
\nVersion 1.3.4 2019-12-16:
    1) A major problem with handling missing SNVs in ROI has ben resolved. (reported by CJYoon)
    2) A new flag -l or --list has been added, which when ON lets -i or --input takes a text file that contains a list of input file (e.g., vcf) paths. This for the case of arbitrarily many multiple inputs.
    3) For displaying \"Missing input file ...\", changed the coloring from \\e[35;5;1m to \\e[38;5;1m
\nVersion 1.3.5 2019-12-16:
    1) A minor problem with saving the output into file when SNVs are missing from all samples has been resolved.
\nVersion 1.3.6 2020-1-29:
    1) A custom signature in the form of VCF or table (currently only VCF format is supported) (requested by SYPark)
       NOTE: it should be a table of two columns (signature_name, vcf file path) WITHOUT a header
\nVersion 2.0.0 2020-1-29:
    1) Now it generates a new file that contains the input signature data (e.g., \"<some names>.signature_individual_spectra.tsv\")
    2) Now the analysis figure (i.e., PNG) contains a new information: total mutation count. (requested by YHAn)
\nVersion 3.0.0 2020-4-6, 2020-4-14:
    1) A new argument '--algorithm' (-a) has been added. For the details, please see the help mesasge.
    2) 2020-4-14, minor problem has been resolved. 'inputHeader <- opt$header...' should have been placed outside the functions that defines it for the first time, due to the scope of variable.
\nVersion 4.0.0 2020-6-22 ~ 2020-7-1:
    1) A code has been partially refactored and made more flexible regarding the combined usage of input arguments.
    2) --signatureSet (-s) has been deprecated and replaced by --specificSamples (-s)
    3) A help message for --list (-l) has been changed to clarify some ambiguities.
    4) An argument --mode (-m) has been deprecated and replaced by --mergeData (-m).
    5) A new argument --mergeData (-m) has been added. It may be used with --list (-l). Now, input files in a list can be processed separately, i.e. without being merged into one, by not invoking --mergeData flag.
    6) --type='TABLE' is for an input in the format of mutation table (type, subtype, SAMPLE1, SMAPLE2, ...) cf. it may only contain a single sample.
    7) plotCosineSim() and plotCosineSim() internally used a global variable mutDF. It should use its argument 'df'. Now fixed.
    8) --algorithm (-a) takes new values 4 := Progressive weighted least squares (PWLS) and 5 := Weighted KLD (WKLD). cf. 3 := WLS.
    9) An issue with the signature proportion has been fixed. The issue was that it was merely taking respective proportions conditioned on the fitted values, which is unrealistic and should be conditioned on the original values. It was resolved by by explicitly stating 'unexplained' mutations.
\nVersion 4.0.1 2020-7-20:
    1) a new syntax for --signature_number has been added, which is to be used when --algorithm 4. Now signature numbers wrapped by parentheses '(' and ')' are fitted together. This requires to put arguments in the quotation marks. For example, \"v2_1,(5,7)\" fits v2_1 first and then v_5,7.
    2) There was an issue with counting the 'unexplained'. It is now okay.
    3) Minor fix for handling a data table. Removed the line 1352. 2020-9-8.
\nVersion 4.0.2 2020-10-26:
    1) A critical issue with data.frame(...,stringsAsFactors=TRUE) was found and resolved accordingly 'options(stringsAsFactors=FALSE)'; issue reported by Yohan An.
\nVersion 4.1.2 2021-3-8:
    1) Variants in mitochondrial genomes can be included now. (requested by Jisong An)
    2) The behavior of --refGenome has been changed. Now, one has to distinguish between UCSC and Ensembl.
    3) -g, --refGenome now takes the following values: GRCh37, GRCh38, hg19, hg38, mm10.
\nVerseion 4.1.3 2021-3-10:
    1) GRCh38 has been replaced by the primary assembly version of GRCh38 (now no extra info other than REF such as HAP is contained). This is to boost up the speed of querying the reference genome (2bit format cat store indexes up to 4GB).
\nVersion 4.1.4 2021-05-26:
    1) From now on, not only png but also pdf file is generated. (requested by Ryul Kim).
    2) Signature plot has been greatly improved.
\nVersion 4.1.5 2021-06-11:
    1) Updated signature set COSMIC SBS v3.2 has been added (v3.1 is skipped) (requested by HSPark).
\nVersion 4.1.6 2021-06-16:
    1) A potentially critical issue has been found and corrected. MNV has not been treated properly when it is present.
\nVersion 4.1.7 2021-07-02:
    1) The first column field in '<BASENAME>.signature_exposures.tsv' has been changed from 'Signature #' to 'Signature'
    2) A new output file in rds is generated that contains grob objects (e.g., BASENAME.grob_objects.rds).
    3) Some help messages have been corrected.
    4) Now for -n (--signature_number), it recognizes v2 as v2_all, v3 as v3_all, and v3.2 as v3.2_all.
    5) Displaying progress has been changed from 'Saving files...' to 'Saving results...'
    6) A critical issue regarding filtering MNV has been found and resolved. Do not use SigAn_4.1.6.
    7) Mutational exposure has been re-defined regarding what its values mean.
    In the exposure table,
    'Exposure' := fitted values 'x' and 'unexplained' are rescaled by a scaling factor where
      scaling factor := (total original count) / (total fitted count)
    'Proportion' := Exposure / sum(Exposure)
\nVersion 4.1.8 2021-08-05:
    1) A 'too many open devices' issue that occurs when too many files are in the vcf list has been resolved. (reported by YHAn)
    Note that 'unexplained' counts are not necessarily a part of total mutation count.
\nVersion 4.1.9 2021-08-20:
    1) An issue that when an input vcf is empty the exposure table gets empty entries (found by AYH) has been resolved.
       If the vcf is empty, signature_exposures.tsv's 'exposure' gets 0 and 'prop' is NA.
\nA code to do mutational signature anaysis on a single sample using 30 COSMIC mutationsl signatures (or custom signatures).
Version M.m.n: M -> Major change of the outlook; m -> major changes, backward incompatible; n -> major or minor changes, backward compatible.\n
"
.libPaths(c("/home/users/jolim/usr/tools/R/library/3.6.0","", .libPaths())) 
suppressMessages(library(optparse))
option_list = list(
    make_option(c("-i","--input"), type="character",
                default=NULL,
                help="Input file (supported: VCF in .vcf, .gz, or .bz2; a tab-delimited text file (TSV) with mutation table (column header: Substitution   Trinucleotide   Count); the code automatically detects the type of input, unless -t or --type is specified)",metavar="character"), 
    make_option(c("-m","--mergeData"), action="store_true",
                default=FALSE,
                help="[Optional] A flag whether to merge data in case of multiple input data."), 
    make_option(c("-t","--type"), type="character",
                default=NULL,
                help="[Optional] For fail-proof determination of the input data type (either 1 or 2; where 1: VCF, 2: TABLE; default: NULL).",metavar="character"), 
    make_option(c("-r","--header"), type="character",
                default="Substitution,Trinucleotide", 
                help="[Optional] Column names for the signature features, comma-separated, used when the input is of table form instead of vcf (default: 'Substitution,Trinucleotide')",metavar="character"), 
    make_option(c("-o","--outputDir"), type="character",
                default="./output/",
                help="[Optional] Output directory (default: ./output/)",metavar="character"),
    make_option(c("-b","--outputBase"), type="character",
                default=NULL,
                help="[Optional] a base name for output file. If NULL, the input file name is used as a base name for the output file. (default: NULL)", metavar="character"),    
    make_option(c("-p","--outputPrefix"), type="character",
                default="",
                help="[Optional] a prefix for output file (default: \"\")", metavar="character"),
    make_option(c("-x","--outputSuffix"), type="character",
                default="",
                help="[Optional] a suffix for output file (default: \"\")", metavar="character"),
    make_option(c("-s","--specificSamples"), type="character",
                default=NULL,
                help="[Optional][MIMO] Specify samples to process. Only for the case the input is mutation tables and not VCFs. It processes all samples in the table by default. Use double or single quotation marks and comma to indicate more than one sample. (e.g., 'SAMPLE1,SAMPLE2,...')",metavar="character"), 
    make_option(c("-n","--signature_number"), type="character",
                default="v2_all",
                help="[Optional] Signature numbers to fit, comma-separated or colon-ranged or combination of the two. COSMIC SBS v2, v3, v3.2 are available (default: 'v2_all'). A special inputs: 'v2_all', 'v3_all', or 'v2_all__v3_all' or any other combinations like 'v2_all__v3_all__v3.2_all'. For --algorithm 4, the fitting is done in the order specified here (e.g., v2_5,1,7 will fit with 5 first and then progressively 1 and 7) (default: 'v2_all'). The syntax is \'v2_<signature_numbers>__v3_<signature_numbers>\' where __ is double underscore (e.g. v2_1,2:5__v3_1:4,7a,7b). Mixing with 'all' is allowed (e.g. v2_all_v3_1:5).",metavar="character"), 
    make_option(c("-g","--refGenome"), type="character",
                default="GRCh37",
                help="[Optional] a reference genome name (supported: hg19, hg38, mm10, GRCh37; default: GRCh37). It's case-sensitive.",metavar="character"), 
    make_option(c("-c","--chromosomes_of_interest"), type="character",
                default="1:22,X,Y,MT", 
                help="[Optional] chromosome names to include for processing, comma-separated and/or colon-ranged (default: 1:22,X,Y)",metavar="character"), 
    make_option(c("-e","--regions_of_interest"), type="character",
                default=NULL,
                help="[Optional] regions to include for processing, comma-separated and/or colon-ranged (e.g., 1:100000,200000:300000) (default: NULL). NOTE: chromosomes_of_interest should be specified and single for this argument to work properly.",metavar="character"), 
    make_option(c("-d","--chromosome_prefix"), type="character",
                default="chr", 
                help="[Optional] A chromosome name prefix, e.g. because sometimes it is 1 and sometimes chr1. Case-insensitive (e.g. chr and CHR are treated the same). This doesn't affect the case of no prefix (i.e. chromosome name '1' is not affected by '-d chr'), so just leave it as default when no prefix is the case. (def.: chr)",metavar="character"), 
    make_option(c("-u","--custom_signatures"), type="character", 
                default=NULL, 
                help="[Optional] a file containing a table of two columns (signature_name, vcf_file_path). NOTE that names for custom signatures should not overlap with those of COSMIC signatures (currently supported: VCF) (def.: NULL)"), 
    make_option(c("-v","--vafVarName"), type="character", 
                default=NULL, 
                help="[Optional] the name of the variable that contains VAF information; this is to give weights (:= vaf * 2) to each mutation. Note that the header name for the column containing the variable is assumed to be 'INFO' (supported: VEP format (e.g. <name>=<value> with field delimiter '|' or ';'); default: NULL)", metavar="character"),
    make_option(c("-a","--algorithm"), type="character", 
                default=1, 
                help="[Optional] a number to indicate which algorithm to use (default: 1) (supported: 1 := OLS, 2 := KLD, 3 := WLS, 4 := Stepwise WLS, 5 := WKLD)", metavar="character"), 
    make_option(c("--pruning_threshold"), type="character", 
                default=0,
                help="[Optional][Experimental] if the signal is below the threshold of the average, they are ignored while fitting. This option is applied iteratively and valid only when --algorithm 4 (-a 4) (default: 0)", metavar="character"), 
    make_option(c("-l","--list"), action="store_true", 
                default=FALSE, 
                help="[Optional] If this flag is used, -i (--input) takes a text file as an input that contains a list of arbitrarily many input file paths."), 
    make_option(c("-z","--log"), action="store_true", 
                default=FALSE, 
                help="[Optional] Display update logs")
)
infoText <- "
\t\tVERSION\t\tDATE\t\tCHANGES
\t\t1.3.1\t\t19.10.21\t\"COSMIC_v2v3\" is now available for -s --signatureSet. Refer to the above for its usage along with --signature_number (-n).
\t\t3.0.0\t\t20.04.06\t\"-a, --algorithm\" has been added.
\t\t4.0.0\t\t20.06.22\t\"-s, --signatureSet\" is now deprecated. It has been replaced by --specificSamples (-s). Now, it is required to specify -n as v2_<signatures> and/or v3_<signatures> w/ or w/o \"__\" in between.
\t\t4.0.0\t\t20.07.01\t\"-m, --mode\" is now deprecated and replaced by --mergeData (-m).
\t\t4.1.2\t\t21.03.09\t\"-g, --refGenome\" should be specified properly. The default value has been changed to GRCh37 (Ensembl) from hg19 (UCSC). Chromosome names of the UCSC reference genomes starts with \"chr\", e.g., chr1, chr2, ..., chrM. That of Ensemble goes 1, 2, ..., MT.
Please see --log (-z) for the most up-to-date info regarding changes made.
" 
opt_parser = OptionParser(option_list = option_list, epilogue = infoText,
                          usage = c(paste0("Usage: SigAn_<version>.R -i <input vcf>.vcf -n <signature names> [options]"),
                                    paste0("Usage: SigAn_<version>.R -i <input mutation table>.tsv -n <signature names> [options]")))
opt = parse_args(opt_parser)
options(stringsAsFactors=FALSE)
# HELPER FUNCTION: stop_quietly 
stop_quietly <- function() {
  opt <- options(show.error.messages = FALSE)
  on.exit(options(opt))
  stop()
} 
# HELPER FUNCTION: ConvTest 
ConvTest <- function(X=NULL, prev_test_measure_proxy=NULL){
  n <- dim(X)[2];
  index <- ramify::argmax(X, rows=FALSE)
  mat1 <- do.call(rbind, replicate(n, index, simplify = FALSE))
  mat2 <- do.call(cbind, replicate(n, index, simplify = FALSE))
  test_measure_proxy <- mat1 == mat2 
  if (!is.null(prev_test_measure_proxy)){
    if (all(prev_test_measure_proxy == test_measure_proxy)) {conv <- 1} else {conv <- 0}
  }
  out <- list(conv=conv, prev_test_measure_proxy=test_measure_proxy)
  return(out)
} 
# HELPER FUNCTION: KLD 
# DEPENDENCY: ConvTest()
KLD <- function(v=NULL, w=NULL, max.conv=1000){
  # Initialization
  k <- dim(w)[2]
  n <- dim(v)[2]
  h <- pracma::rand(k,n)
  converged <- FALSE
  conv <- 0
  init <- TRUE
  i <- 1
  # Iteration
  while (!converged) {
    if (init) {
      convResult <- ConvTest(h)
      prev_test_measure_proxy <- convResult$prev_test_measure_proxy
      init <- FALSE
    }
    w <- pmax(w, eps()) # For numerical stability
    x1 <- do.call(cbind, replicate(n, t(t(colSums(w))), simplify=FALSE))
    h <- h * (t(w) %*% (as.matrix(v) / (w %*% h))) / x1
    # Convergence test
    convResult <- ConvTest(h, prev_test_measure_proxy)
    conv <- conv + convResult$conv
    prev_test_measure_proxy <- convResult$prev_test_measure_proxy
    # Escape the loop
    if (conv > max.conv) {break}
  }
  # For numerical stability
  h <- pmax(as.vector(h), eps())
  # OUTPUT
  resid.norm <- sum((v - (w %*% h)) * (v - (w %*% h))) 
  out <- list(x=h, resid.norm=resid.norm)
  return(out)
} 
# HELPER FUNCTION: normalize 
normalize <- function(X){if(is.matrix(X)){X/colSums(X)}else if(is.vector(X)){X/sum(X)}} 
# HELPER FUNCTION: plotMutationalSpectrum2 
# INPUT: df, ...
# OUTPUT: a plot
# NOTE: color.palette is now 'COSMIC.SNV' by default
plotMutationalSpectrum2 <- function(df = NULL, y.max = NULL, bar.width = 0.5,
                                    type.name = "type", subtype.name = "subtype", value.name = "count",
                                    color.palette = "COSMIC.SNV", newpage = TRUE,
                                    upperLabels.y = 0.05, upperLabels.height = 0.25,
                                    upperLabels.text.y = 0.35, upperLabels.text.fontsize = 12,
                                    xLabels.text.y = 0.5, xLabels.text.fontsize = 10,
                                    yLabels.text.x = 0.9, yLabels.text.fontsize = 10,
                                    yLabels.text.decimals = 1, yLabels.text.suffix = "",
                                    title.innerFrame.upperLeft = "", title.innerFrame.upperLeft.fontsize = 20,
                                    bar.edge.color = NA, background.alpha = 0.1,
                                    left.margin = 0.05, right.margin = 0.05){
  # Packages
  suppressMessages(library(grid))
  suppressMessages(library(magrittr))
  suppressMessages(library(data.table))
  # Color palette
  ## Presets
  palette.COSMIC.SNV.96 <- paste0("#",c(`C>A`="1EBFF0",`C>G`="050708",`C>T`="E62725",`T>A`="CBCACB",`T>C`="A1CF64",`T>G`="EDC8C5"))
  palette.COSMIC.SNV.192 <- paste0("#",c(`A>C`="B6CCFF",`A>G`="B3853C",`A>T`="808080",`G>A`="0000FF",`G>C`="000000",`G>T`="FFD000",
                                         `C>A`="1EBFF0",`C>G`="050708",`C>T`="E62725",`T>A`="CBCACB",`T>C`="A1CF64",`T>G`="EDC8C5"))
  palette.COSMIC.DBS <- paste0("#",c(AC="A6CEE3",AT="1F78B4",CC="B2DF8A",CG="33A02C",CT="FB9A99",GC="E31A1C",TA="FDBF6F",TC="FF7F00",TG="CAB2D6",TT="6A3D9A"))
  palette.COSMIC.INDEL <- paste0("#",c(`D1_C`="FCBF6F",
                                       `D1_T`="FD8103",
                                       `I1_C`="B2DB8D",
                                       `I1_T`="36A034",
                                       `D2_2`="FEC9B7",
                                       `D2_3`="FE886A",
                                       `D2_4`="F34433",
                                       `D2_5+`="BD181C",
                                       `I2_2`="D0E1F3",
                                       `I2_3`="96C4DE",
                                       `I2_4`="4798CD",
                                       `I2_5+`="1764AA",
                                       `MH_2`="DCE2F2",
                                       `MH_3`="B6B6D8",
                                       `MH_4`="8584BE",
                                       `MH_5+`="63409C"))
  if (is.null(color.palette)) {
    stop("Please provide input for 'color.palette' (currently, 'COSMIC.SNV', 'COSMIC.SNV.192', 'COSMIC.DBS' are available)")
  } else if (color.palette == "COSMIC.SNV") {
    color.palette <- palette.COSMIC.SNV.96
  } else if (color.palette == "COSMIC.SNV.192") {
    color.palette <- palette.COSMIC.SNV.192
  } else if (color.palette == "COSMIC.DBS") {
    color.palette <- palette.COSMIC.DBS
  } else if (color.palette == "COSMIC.INDEL"){
    color.palette <- palette.COSMIC.INDEL
  } else {
    stop("Invalid input for 'color.palette'")
  }
  # Reformatting input df
  df <- df %>% as.data.frame
  df <- df[, c(type.name, subtype.name, value.name)]
  df[["type_subtype"]] <- paste0(df$type,":",df$subtype)
  setnames(df, old=value.name, "count")
  # Plot
  if (newpage == TRUE){grid.newpage()} else {
    initial.vp <- current.viewport()
    if (initial.vp$name == "ROOT") {
      backToTheInitialVp <- parse(text="upViewport(0)")
    } else {
      backToTheInitialVp <- parse(text="seekViewport(initial.vp$name)")
    }
  }
  # Outer frame
  x.outer <- 0; y.outer <- 0; w.outer <- 1; h.outer <- 1
  pushViewport(viewport(x = x.outer, y = y.outer, width = w.outer, height = h.outer, just = c("left","bottom"))) 
  vp.outerFrame <- current.viewport()$name 
  # Inner frame
  x.inner <- left.margin; y.inner <- 0.1; w.inner <- 1-left.margin-right.margin; h.inner <- 0.7
  pushViewport(viewport(x = x.inner, y = y.inner, width = w.inner, height = h.inner, just = c("left","bottom")))
  vp.innerFrame <- current.viewport()$name 
  grid.rect(name = "rect.innerFrame", gp = gpar(col="gray",lwd=1))
  # Y-axis labels
  x.y_axis <- 0; y.y_axis <- 0; w.y_axis <- x.inner; h.y_axis <- 1
  upViewport()
  pushViewport(viewport(x = x.y_axis, y = y.y_axis, width = w.y_axis, height = h.y_axis, just = c("left","bottom")))
  vp.y_axis <- current.viewport()$name
  # X-axis labels
  x.x_axis <- x.inner; y.x_axis <- 0; w.x_axis <- w.inner; h.x_axis <- y.inner
  upViewport()
  pushViewport(viewport(x = x.x_axis, y = y.x_axis, width = w.x_axis, height = h.x_axis, just = c("left","bottom"))) 
  vp.x_axis <- current.viewport()$name 
  # Inner drawings (loop)
  ## Some initial workups
  df$type <- factor(df$type, levels=unique(df$type), ordered=TRUE) 
  df.type <- df$type %>% table %>% as.data.frame
  w.bar_field.scaleFactor <- sum(df.type$Freq)
  if (is.null(y.max)) {
    y.max <- max(df[["count"]]) * 1/(3/4 + 1/4*1/10)
  }
  h.bar.scaleFactor <- 1/y.max # h.bar.scaleFactor scales each bar height into the [0,1] scale.
  df.count.pointer <- 1
  df.subtype.pointer <- 1
  x.bar.field.pointer <- x.inner
  x.xLabels.field.pointer <- 0
  ## Sanity checks
  if (bar.width > 1 | bar.width < 0) { stop("Wrong range input for bar.width") }
  if (is.null(color.palette)) { stop("Input for 'color.palette' is missing") }
  ## Drawing bars
  ### Bar field
  for (i in 1:nrow(df.type)) {
    seekViewport(vp.outerFrame)
    x.bar_field <- x.bar.field.pointer
    w.bar_field <- df.type$Freq[i]/w.bar_field.scaleFactor*w.inner
    y.bar_field <- y.inner
    h.bar_field <- h.inner
    pushViewport(viewport(x = x.bar_field, y = y.bar_field, width = w.bar_field, height = h.bar_field, just = c("left","bottom"))) 
    vp.barField.name <- paste0("vp.barField.",df.type$.[i]) 
    assign(vp.barField.name,current.viewport()$name) 
    grid.rect(name = paste0(vp.barField.name,".rect.barField.",df.type$.[i]),
              gp=gpar(col=NA, 
                      fill=color.palette[i],
                      alpha=background.alpha))
    ### Bar drawings
    #### Now in vp.barField
    x.bar.pointer <- 1/2*1/df.type$Freq[i] - 1/2*bar.width/df.type$Freq[i]
    for (j in 1:df.type$Freq[i]) {
      grid.rect(name = paste0(vp.barField.name,".rect.bar.",df$type_subtype[df.count.pointer]),
                just = c("left","bottom"),
                x = x.bar.pointer,
                y = 0,
                width = bar.width/df.type$Freq[i],
                height = df[["count"]][df.count.pointer] * h.bar.scaleFactor,
                gp=gpar(fill=color.palette[i],
                        col=bar.edge.color))
      df.count.pointer <- df.count.pointer + 1
      # Do not let x.bar.pointer be updated in the last loop
      if (j != df.type$Freq[i]) {
        x.bar.pointer <- x.bar.pointer + 1/df.type$Freq[i]
      }
    }
    ### X labels
    seekViewport(vp.x_axis) 
    x.xLabels.pointer <- 1/2*1/df.type$Freq[i]
    for (j in 1:df.type$Freq[i]){
      grid.text(label = df$subtype[df.subtype.pointer],
                name = paste0(vp.x_axis,".text.xLabel.",df$subtype[df.subtype.pointer]), 
                just = c("center","center"),
                x = x.xLabels.field.pointer + x.xLabels.pointer*df.type$Freq[i]/w.bar_field.scaleFactor,
                y = xLabels.text.y,
                rot = 90,
                gp = gpar(fontsize = xLabels.text.fontsize, fontfamily = "mono") # fontframily = "mono" is Courier
      )
      df.subtype.pointer <- df.subtype.pointer + 1
      x.xLabels.pointer <- x.xLabels.pointer + 1/df.type$Freq[i]
    }
    ### Upper labels
    seekViewport(vp.outerFrame) 
    if (upperLabels.y < 0 | upperLabels.y > 1) {stop("Invalid range for 'upperLabels.y'.")}
    if (upperLabels.height < 0 | upperLabels.height > 1) {stop("Invalid range for 'upperLabels.height'.")}
    x.upperLabels <- x.bar.field.pointer + w.bar_field*(1/2*1/df.type$Freq[i] - 1/2*bar.width/df.type$Freq[i])
    w.upperLabels <- w.bar_field*(x.bar.pointer + 1/2*bar.width/df.type$Freq[i])
    y.upperLabels <- y.inner + h.inner + (1 - (y.inner + h.inner))*upperLabels.y
    grid.rect(name = paste0(vp.outerFrame,".rect.upperLabel.",df.type$.[i]),
              just = c("left","bottom"),
              x = x.upperLabels,
              y = y.upperLabels,
              width = w.upperLabels,
              height = upperLabels.height*(1 - (y.inner + h.inner)),
              gp = gpar(fill=color.palette[i],col=NA)
    )
    ### Upper texts
    y.upperLabels.text <- y.inner + h.inner + (1 - (y.inner + h.inner))*upperLabels.text.y
    grid.text(name = paste0(vp.outerFrame,".text.upperLabel.",df.type$.[i]),
              label = df.type$.[i],
              just = c("center","bottom"),
              x = x.upperLabels + 0.5*w.upperLabels,
              y = y.upperLabels.text,
              gp = gpar(fontsize = upperLabels.text.fontsize)
    )
    ### Y labels
    seekViewport(vp.y_axis) 
    x.yLabels <- yLabels.text.x
    y.yLabels <- y.inner
    labels.yLabels <- seq(0,y.max,length.out = 5)
    h.bar_field.quartiles <- seq(0,h.inner,length.out = 5) # w.r.s.t. vp.y_axis
    for (j in 1:5){
      vp.y_axis
      grid.text(name = paste0(vp.y_axis,".text.yLabels.",j),
                label = paste0(round(labels.yLabels[j],yLabels.text.decimals),yLabels.text.suffix),
                just = c("right","center"),
                x = x.yLabels,
                y = y.inner + h.bar_field.quartiles[j],
                gp = gpar(fontsize = yLabels.text.fontsize, fontfamily = "mono")
      )
    }
    ### Y inward ticks and horizontal lines
    seekViewport(vp.innerFrame) 
    h.bar_field.quartiles2 <- seq(0,1,length.out = 5) # w.r.s.t. vp.innerFrame
    for (j in 2:4){
      #### orizontal lines
      grid.segments(x0 = 0,
                    x1 = 1,
                    y0 = h.bar_field.quartiles2[j],
                    y1 = h.bar_field.quartiles2[j],
                    gp = gpar(col="gray", lwd = 0.75, alpha = 1/5)
      )
      #### 3 inward ticks
      grid.segments(x0 = 0,
                    x1 = 1/w.bar_field.scaleFactor,
                    y0 = h.bar_field.quartiles2[j],
                    y1 = h.bar_field.quartiles2[j],
                    gp = gpar(col="darkgray", lwd = 0.75, alpha = 2/5)
      )
    }
    ### Update field pointers
    x.bar.field.pointer <- x.bar.field.pointer + w.bar_field #df.type$Freq[i]/w.bar_field.scaleFactor
    x.xLabels.field.pointer <- x.xLabels.field.pointer + df.type$Freq[i]/w.bar_field.scaleFactor
  }
  ## Title at the upper-left corner of the innerFrame
  seekViewport(vp.innerFrame) 
  grid.text(
    label = title.innerFrame.upperLeft,
    just = c("left","top"),
    x = 0.01,
    y = 0.95,
    gp = gpar(fontsize = title.innerFrame.upperLeft.fontsize,
              fontface = "bold")
  )
  ## Back to the initial viewport
  if (newpage != TRUE) { eval(backToTheInitialVp) }
} 
# HELPER FUNCTION: plotSNVspectra 
# DEPENDENCY: plotMutationalSpectrum2
plotSNVspectra <- function(df=NULL, newpage=FALSE){
  # PARAMETER
  bot_margin <- 0.1
  top_margin <- 0.1
  ys <- c(2/3, 1/3, 0) * (1 - bot_margin - top_margin) + bot_margin
  hs <- 1/3 * (1 - bot_margin - top_margin)
  if(newpage){grid.newpage()}else{}
  # Orig
  pushViewport(viewport(x=0,y=ys[1],width=1,height=1/3,just=c("left","bottom")))
  plotMutationalSpectrum2(df = df,
                          y.max = NULL,
                          value.name = "orig",
                          color.palette = "COSMIC.SNV",
                          newpage = FALSE,
                          upperLabels.text.fontsize = 12,
                          xLabels.text.fontsize = 0,
                          yLabels.text.fontsize = 9,
                          yLabels.text.decimals = 2,
                          xLabels.text.y = 0.3, 
                          yLabels.text.x = 0.9,
                          title.innerFrame.upperLeft = "ORIGINAL",
                          title.innerFrame.upperLeft.fontsize = 13,
                          background.alpha = 0.05,
                          left.margin = 0.1)
  popViewport()
  # Recon
  pushViewport(viewport(x=0,y=ys[2],width=1,height=1/3,just=c("left","bottom")))
  plotMutationalSpectrum2(df = df,
                          y.max = max(df$orig) * 1/(3/4 + 1/4*1/10),
                          value.name = "recon",
                          color.palette = "COSMIC.SNV",
                          newpage = FALSE,
                          upperLabels.text.fontsize = 0,
                          xLabels.text.fontsize = 0,
                          yLabels.text.fontsize = 9,
                          yLabels.text.decimals = 2,
                          xLabels.text.y = 0.3, 
                          yLabels.text.x = 0.9,
                          title.innerFrame.upperLeft = "RECONSTRUCTED",
                          title.innerFrame.upperLeft.fontsize = 13,
                          background.alpha = 0.05,
                          left.margin = 0.1)
  popViewport()
  # Resid
  pushViewport(viewport(x=0,y=ys[3],width=1,height=1/3,just=c("left","bottom")))
  plotMutationalSpectrum2(df = df,
                          y.max = max(df$orig) * 1/(3/4 + 1/4*1/10),
                          value.name = "resid",
                          color.palette = "COSMIC.SNV",
                          newpage = FALSE,
                          upperLabels.text.fontsize = 0,
                          xLabels.text.fontsize = 9,
                          yLabels.text.fontsize = 9,
                          yLabels.text.decimals = 2,
                          xLabels.text.y = 0.3, 
                          yLabels.text.x = 0.9,
                          title.innerFrame.upperLeft = "RESIDUAL",
                          title.innerFrame.upperLeft.fontsize = 13,
                          background.alpha = 0.05,
                          left.margin = 0.1)
  popViewport()
} 
if(opt$log){cat(NOTE)} else # display update log
if(is.null(opt$input)){parse_args(opt_parser, args = c("--help"))} else { # Show help message if input is missing
run <- function(){
  system(paste0("echo -e \"\\e[01m\\e[38;5;15mMERGE OUTPUT: ",opt$merge,"\\e[0m\"")) 
  system(paste0("echo -e \"\\e[01m\\e[38;5;15mINPUT: ",opt$input,"\\e[0m\"")) 
  system("echo -e \"\\e[38;5;15mInitiating...\\e[0m\"")
  suppressMessages(library(tidyverse))
  suppressMessages(library(data.table))
  suppressMessages(library(BSgenome.Hsapiens.UCSC.hg19))
  suppressMessages(library(BSgenome.Hsapiens.UCSC.hg38))
  suppressMessages(library(BSgenome.Mmusculus.UCSC.mm10)) 
  suppressMessages(library(BSgenome.Hsapiens.Ensembl.GRCh37.jolim)) 
  suppressMessages(library(BSgenome.Hsapiens.Ensembl.GRCh38.primary.jolim)) 
  suppressMessages(library(pracma))
  suppressWarnings(library(tolerance)) 
  suppressMessages(library(gtools)) 
  suppressMessages(library(RColorBrewer))
  suppressMessages(library(grid))
  suppressMessages(library(gridExtra))
  ## M0 : Argument handling ----------------------------------------------------
  mergeData <- opt$mergeData
  # Sanity checks :: input & opt$list
  if (opt$list & (length(strsplit(opt$input, split=",")[[1]]) > 1)) {
    system("echo -e \"\\e[38;5;1mERROR: It seems that multiple input files have been specified through (-i) and at the same time --list (-l) is ON. Please do either one. \\e[0m\"")
    stop_quietly()
  } 
  outputDir <- opt$outputDir
  if (opt$outputPrefix=="") {outputPrefix <- opt$outputPrefix} else {outputPrefix <- paste0(opt$outputPrefix,".")}
  if (opt$outputSuffix=="") {outputSuffix <- opt$outputSuffix} else {outputSuffix <- paste0(".",opt$outputSuffix)}
  signature_chunking <- opt$signature_number %>% grepl("[(].*[)]",.) 
  signature_number <- opt$signature_number %>% gsub("[(]|[)]","",.) 
  tmp.signature_number <- unlist(strsplit(signature_number, split="__"))
  tmp.signature_number <- tolower(tmp.signature_number) 
  # Sanity check
  if (length(tmp.signature_number)>3) { 
    system("echo -e \"\\e[38;5;1mUnrecoginzable input for -n --signature_number\\e[0m\""); stop_quietly()
  }
  # 2021-07-02, to recognize v2, v3, and v3.2 as v2_all, v3_all, and v3.2_all, respectively.
  if ("v2" %in% tmp.signature_number) {
    tmp.signature_number <- gsub("v2","v2_all",tmp.signature_number)
  }
  if ("v3" %in% tmp.signature_number) {
    tmp.signature_number <- gsub("v3","v3_all",tmp.signature_number)
  }
  if ("v3.2" %in% tmp.signature_number) {
    tmp.signature_number <- gsub("v3.2","v3.2_all",tmp.signature_number)
  }
  # v2_all and/or v3_all and/or v3.2_all
  v2_all <- any(grepl("^v2_",tmp.signature_number)) 
  v3_all <- any(grepl("^v3_",tmp.signature_number)) 
  v3.2_all <- any(grepl("^v3.2_",tmp.signature_number)) # 2021-06-11
  # HELPER FUNCTION: ger_sigDF 
  get_sigDF <- function(){
    sigDF.v2 <- as.data.table(read.csv(textConnection(paste0(COSMIC_v2,collapse="\n")),sep="\t",check.names=FALSE))
    sigDF.v3 <- as.data.table(read.csv(textConnection(paste0(COSMIC_v3,collapse="\n")),sep="\t",check.names=FALSE))
    sigDF.v3.2 <- as.data.table(read.csv(textConnection(paste0(COSMIC_v3.2,collapse="\n")),sep="\t",check.names=FALSE)) 
    sigDF.v2[,`Somatic Mutation Type`:=NULL]
    sigDF.v2 <- sigDF.v2[,c(1:32)] %>% dplyr::rename(type="Substitution Type",subtype="Trinucleotide")
    names(sigDF.v2) <- names(sigDF.v2) %>% gsub("Signature ","v2_",.)
    sigDF.v3 <- sigDF.v3 %>% dplyr::rename(type="Type",subtype="SubType")
    names(sigDF.v3) <- names(sigDF.v3) %>% gsub("SBS","v3_",.)
    sigDF.v3.2$Type <- sigDF.v3.2$Type %>% gsub("(.)[[](.)>(.)[]](.)","\\2>\\3:\\1\\2\\4",.) 
    sigDF.v3.2 <- sigDF.v3.2 %>%
      tidyr::separate(col="Type",into=c("type","subtype"), sep=":") %>%
      arrange(type, subtype)  
    names(sigDF.v3.2) <- names(sigDF.v3.2) %>% gsub("SBS","v3.2_",.) 
    return(list(v2=sigDF.v2, v3=sigDF.v3, v3.2=sigDF.v3.2)) 
  }
  sigDF.coll <- get_sigDF()
  # HELPER FUNCTION: spread_ranges 
  spread_ranges <- function(sigString=NULL, sigDF=NULL){
    str.ver <- gsub("^(v2_|v3_|v3.2_).*","\\1",sigString) 
    str <- sigString %>% gsub("^(v2_|v3_|v3.2_)(.*)","\\2",.) 
    str <- strsplit(str, split=",")[[1]]
    if (any(grepl("all",str))) {
      str <- sigDF[,-c(1,2)] %>% names
    } else {
      str.nonrange <- str[!grepl(":",str)]
      str.range <- str[grepl(":",str)] %>% sapply(function(x){eval(parse(text=x))}) %>% unlist(use.names=FALSE) %>% as.vector
      str <- c(str.nonrange, str.range) %>% paste0(str.ver,.)
    }
    return(str)
  } 
#  if (tmp.v2 & tmp.v3) {
#    signature_numbers.v2 <- grep("v2_",tmp.signature_number,value=TRUE)
#    signature_numbers.v3 <- grep("v3_",tmp.signature_number,value=TRUE)
#    signatureNames.v2 <- signature_numbers.v2 %>% spread_ranges(sigDF=sigDF.coll$v2)
#    signatureNames.v3 <- signature_numbers.v3 %>% spread_ranges(sigDF=sigDF.coll$v3)
#    signatureNames <- c(signatureNames.v2, signatureNames.v3)
#  } else if (tmp.v2) {
#    signature_numbers.v2 <- grep("v2_",tmp.signature_number,value=TRUE)
#    signatureNames.v2 <- signature_numbers.v2 %>% spread_ranges(sigDF=sigDF.coll$v2)
#    signatureNames <- signatureNames.v2
#  } else if (tmp.v3) {
#    signature_numbers.v3 <- grep("v3_",tmp.signature_number,value=TRUE)
#    signatureNames.v3 <- signature_numbers.v3 %>% spread_ranges(sigDF=sigDF.coll$v3)
#    signatureNames <- signatureNames.v3
#  } else {
#    system("echo -e \"\\e[38;5;1mWrong input for -n --signature_number\\e[0m\""); stop_quietly()
#  }
  if (v2_all) {
    sigNumbers.v2 <- grep("v2_",tmp.signature_number,value=TRUE) %>% spread_ranges(sigDF.coll$v2)
  } else {sigNumbers.v2 <- NULL}
  if (v3_all) {
    sigNumbers.v3 <- grep("v3_",tmp.signature_number,value=TRUE) %>% spread_ranges(sigDF.coll$v3)
  } else {sigNumbers.v3 <- NULL}
  if (v3.2_all) {
    sigNumbers.v3.2 <- grep("v3.2_",tmp.signature_number,value=TRUE) %>% spread_ranges(sigDF.coll$v3.2)
  } else {sigNumbers.v3.2 <- NULL}
  signatureNames <- c(sigNumbers.v2, sigNumbers.v3, sigNumbers.v3.2) 
  # If the wrap (...) is used,
  # Sanity check
  if (signature_chunking & (opt$algorithm != 4)) {
    system("echo -e \"\\e[033;0;33mWARNING: The alorithm has not been set to 4. the '(' and ')' in the --signature_number will be ignored.\\e[0m\"")
    signature_chunking <- FALSE
  } else if (signature_chunking & (opt$algorithm == 4)) {
    signature_number <- opt$signature_number
    # signature_number <- c("v2_1,(2,3)__v3_3:4,(5:7)") # DEBUGGING
    tmp.signature_number <- unlist(strsplit(signature_number, split="__"))
    signatureNames <- tmp.signature_number %>% lapply(function(x){
      x <- tolower(x)
      x.header <- x %>% gsub("(^(v2_)|^(v3_)|^(v3.2_)).*","\\1",.) 
      x.body <- x %>% gsub("^(v2_|v3_|v3.2_)(.*)","\\2",.) 
      x.body <- x.body %>% strsplit(split="[(]|[)]") %>% unlist %>% strsplit(split=",") %>% lapply(function(x){paste0(x.header,x)}) #%>% lapply(function(x){sapply(x,function(x){spread_ranges(x,sigDF=sigDF)})})
      x.body <- x.body %>% lapply(function(x){lapply(x,function(x){spread_ranges(x,sigDF)})} %>% unlist)
      }) %>% unlist(recursive=FALSE)
  } else {
    # Do nothing
  } 
  # OTHER ARGUMENTS
  refGenome <- opt$refGenome
  refGenomes <- list(hg19=BSgenome.Hsapiens.UCSC.hg19,
                     hg38=BSgenome.Hsapiens.UCSC.hg38,
                     mm10=BSgenome.Mmusculus.UCSC.mm10,
                     GRCh37=BSgenome.Hsapiens.Ensembl.GRCh37.jolim, 
                     GRCh38=BSgenome.Hsapiens.Ensembl.GRCh38.primary.jolim 
                     )
  vafVarName <- opt$vafVarName
  inputHeader <- strsplit(opt$header,split=",")[[1]] 
  inputType <- opt$inputType
  ## M1 : Data pre-processing --------------------------------------------------
  # HELPER FUNCTION :: setupTable 
  setupTable <- function(mutTable = NULL) {
    # make sure it's DF
    mutDF <- as.data.frame(mutTable)
    # helper functions
    c_ntd <- function(ntd){if(ntd=="A"){return("T")};if(ntd=="C"){return("G")};if(ntd=="G"){return("C")};if(ntd=="T"){return("A")}}
    c_chg <- function(chg){if(substr(chg,1,1)=="A"|substr(chg,1,1)=="G"){out<-paste0(c_ntd(substr(chg,1,1)),">",c_ntd(substr(chg,3,3)))} else {out <- chg};return(out)}
    c_seq <- function(seq){temp <- strsplit(seq,split="")[[1]]; out <- paste0(c_ntd(temp[3]),c_ntd(temp[2]),c_ntd(temp[1])); return(out)}
    # abide by C & T convention
    for (i in 1:nrow(mutDF)) {
      toConvert <- substr(mutDF[i,inputHeader[1]],1,1)
      if (toConvert =="A"| toConvert =="G"){
        mutDF[i,inputHeader[1]] <- c_chg(mutDF[i,inputHeader[1]])
        mutDF[i,inputHeader[2]] <- c_seq(mutDF[i,inputHeader[2]])
      }
    }
    # aggregate complemented features
    mutDF <- aggregate(formula(paste(". ~",inputHeader[1],"+",inputHeader[2],sep=" ")),data=mutDF,sum) 
    # arrange
    ntd <- c("A","C","G","T")
    chg <- c("C>A","C>G","C>T","T>A","T>C","T>G")
    ctx_pre <- rep(ntd,each=4)
    ctx_suf <- rep(ntd,4)
    type <- rep(chg,each=16) 
    subtype <- paste0(rep(ctx_pre,6),c(rep("C",48),rep("T",48)),rep(ctx_suf,6))
    # return
    out <- left_join(data.frame(type=type, subtype=subtype, stringsAsFactors = FALSE), mutDF, by=c("type"=inputHeader[1], "subtype"=inputHeader[2]))
    out[is.na(out)] <- 0 
    return(out)
  } 
  # HELPER FUNCTION :: processVCF 
  processVCF <- function(vcf = NULL, filtering = TRUE){
    # vcf must be data.table since it is from fread, but to make sure
    if (!is.data.table(vcf)){vcf <- as.data.table(vcf)}
    ## Process input data
    system("echo -e \"\\e[38;5;15mProcessing input data...\\e[0m\"")
    vcf <- vcf[nchar(REF)==1 & nchar(ALT)==1 & (REF %in% c("A","C","G","T")) & (ALT %in% c("A","C","G","T")),] 
    # 2021-06-16, remove consecutive positions (i.e., MNVs)
    mnv.ir <- reduce(IRanges(start=vcf$POS,width=1)) 
    mnv.pos <- pos(IPos(mnv.ir[(width(mnv.ir)>1)])) 
    vcf <- vcf[!vcf$POS %in% mnv.pos,] ## 2021-07-05
    if (!is.null(vafVarName)){
      vafInfo <- strsplit(vcf[["INFO"]],split=";|\\|") 
      vafInfo <- unlist(lapply(vafInfo, function(x){grep(vafVarName,x,value=TRUE)})) 
      ### Sanity check: Stop if vafInfo is not provided: 2019-06-11
      if (length(vafInfo)==0) { stop("INFO column doesn't have the given varaible name. Do the double check.") } 
      vaf <- as.numeric(unlist(lapply(strsplit(vafInfo,split="="), function(x){x[2]}))) 
      vcf$weights <- vaf*2 
    }
    # FILTERING STEP 
    if(filtering){
      COI <- strsplit(opt$chromosomes_of_interest,split=",")[[1]] 
      COI <- unlist(sapply(COI, function(x){if(grepl("[0-9]+:[0-9]+",x)){eval(parse(text=x))}else{x}}),use.names=FALSE) 
      chromosome_prefix <- opt$chromosome_prefix 
      chrs_to_include_idx <- gsub(paste0("^",chromosome_prefix),"",vcf$`#CHROM`,ignore.case=TRUE) %in% COI 
      vcf <- vcf[chrs_to_include_idx,] 
      if (!is.null(opt$regions_of_interest)){
          # Sanity check: whether a single COI is specified
          if(length(COI)!=1){stop("-c (--chromosomes_of_interest) must be single when -e (--regions_of_interest) is specified.")}
          #
          ROI <- strsplit(opt$regions_of_interest,split=",")[[1]] 
          lapply(ROI,function(roi){
              tmp.range <- strsplit(roi,split="(:|-)")[[1]]
              if(length(tmp.range)!=2){stop("Invalid input for -e or --regions_of_interest")} # sanity check: whether the format is num:num or num-num
              tmp.start <- tmp.range[1]
              tmp.end <- tmp.range[2]
              fc <- paste0("(",tmp.start," < as.numeric(POS) & as.numeric(POS) < ",tmp.end,")")
          }) -> tmp.ROI.fc
          ROI.fc <- paste0(tmp.ROI.fc,collapse=" | ")
          vcf <- eval(parse(text=paste0("vcf[",ROI.fc,",]")))
      }
    }
    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING
    if(nrow(vcf)==0){
      ntd <- c("A","C","G","T")
      chg <- c("C>A","C>G","C>T","T>A","T>C","T>G")
      ctx_pre <- rep(ntd,each=4)
      ctx_suf <- rep(ntd,4)
      type <- rep(chg,each=16) 
      subtype <- paste0(rep(ctx_pre,6),c(rep("C",48),rep("T",48)),rep(ctx_suf,6))
      mutDF <- data.frame(type=type,subtype=subtype,stringsAsFactors = FALSE)
      mutDF[["orig"]] <- 0
      return(mutDF)
    } # otherwise, proceed
    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING
    str <- as.vector(sapply(vcf[["REF"]], function(x){if (x == "A" | x == "G") {"-"} else {"+"}}))
    gr <- GRanges(seqnames = vcf[["#CHROM"]], ranges = IRanges(start = vcf[["POS"]], width = 1), strand = str) 
    df <- data.frame(REF=vcf[["REF"]],ALT=vcf[["ALT"]],stringsAsFactors=FALSE)
    c_ntd_vec <- function(x){A_i<-x=="A";C_i<-x=="C";G_i<-x=="G";T_i<-x=="T";x[A_i]<-"T";x[C_i]<-"G";x[G_i]<-"C";x[T_i]<-"A";return(x)} 
    REF_A_idx <- df$REF == "A";
    REF_G_idx <- df$REF == "G";
    df$REF[REF_A_idx] <- "T"; df$ALT[REF_A_idx] <- c_ntd_vec(df$ALT[REF_A_idx]); df$REF[REF_G_idx] <- "C"; df$ALT[REF_G_idx] <- c_ntd_vec(df$ALT[REF_G_idx])
    #
    tryCatch({mutCtx <- as.character(getSeq(refGenomes[[refGenome]], gr+1))},
             error = function(err){
               system(paste0("echo -e \"It seems that a wrong reference genome has been chosen.\nCurrent choice: ",refGenome,"\""));
               if (grepl("chr",vcf[["#CHROM"]][1])) { # UCSC
                 system("echo -e \"Your vcf seems to be based on UCSC reference genome. Make sure you specified --refGenome correctly, e.g., hg19 or hg38 for human in UCSC.\\e[0m\"")
               } else { # Ensembl
                 system("echo -e \"Your vcf seems to be based on Ensembl reference genome. Make sure you specified --refGenome correctly, e.g., GRCh37 or GRCh38 for human in Ensembl.\\e[0m\"")
               } 
               system("echo -e \"Terminating...\"");
               opt <- options(show.error.messages = FALSE);
               on.exit(options(opt));
               stop_quietly()
             },
             warning = function(war){
               system(paste0("echo -e \"It seems that a wrong reference genome has been chosen.\nCurrent choice: ",refGenome,"\""));
               if (grepl("chr",vcf[["#CHROM"]][1])) { # UCSC
                 system("echo -e \"Your vcf seems to be based on UCSC reference genome. Make sure you specified --refGenome correctly, e.g., hg19 or hg38 for human in UCSC.\\e[0m\"")
               } else { # Ensembl
                 system("echo -e \"Your vcf seems to be based on Ensembl reference genome. Make sure you specified --refGenome correctly, e.g., GRCh37 or GRCh38 for human in Ensembl.\\e[0m\"")
               } 
               system("echo -e \"Terminating...\"")
               opt <- options(show.error.messages = FALSE);
               on.exit(options(opt));
               stop_quietly()
             }, 
             finally = NULL 
    )
    ### Sanity check
    system("echo -e \"\\e[38;5;15mChecking if the reference genome has been chosen properly...\\c\\e[0m\"") # \c --> echo without newline
    if (all(substr(mutCtx,2,2)==df$REF)) {system("echo \"OK\"")} else {system("echo -e \"NOT OK\nThe reference genome does not match with REF in the VCF.\""); stop_quietly()}
    ##
    ntd <- c("A","C","G","T")
    chg <- c("C>A","C>G","C>T","T>A","T>C","T>G")
    ctx_pre <- rep(ntd,each=4)
    ctx_suf <- rep(ntd,4)
    type <- rep(chg,each=16) 
    subtype <- paste0(rep(ctx_pre,6),c(rep("C",48),rep("T",48)),rep(ctx_suf,6))
    mutDF <- data.frame(type=type,subtype=subtype,stringsAsFactors = FALSE)
    if (!is.null(vafVarName)) {
      mutDT <- data.table(feature=paste0(df$REF,">",df$ALT,":",mutCtx))
      mutDT$vaf <- vcf$weights
      mutDT <- mutDT[, .(count =.N, weightSum = sum(vaf)), by = feature ]
      mutDT$Freq <- mutDT$count * mutDT$weightSum # Freq column is weighted count
      mutDT <- mutDT[,c("feature","Freq")]
      mutDT <- separate(mutDT,col="feature",into=c("type","subtype"),sep=":")
      mutDF <- left_join(mutDF,mutDT,by=c("type","subtype"))
    } else {
      mutDF <- left_join(mutDF,separate(as.data.frame(table(paste0(df$REF,">",df$ALT,":",mutCtx)),stringsAsFactors=FALSE),col="Var1",into=c("type","subtype"),sep=":"),by=c("type","subtype"))
    }
    #
    names(mutDF)[names(mutDF)=="Freq"] <- "orig"
    mutDF$orig[is.na(mutDF$orig)] <- 0
    #}
    return(mutDF)
  }
  # HELPER FUNCTION :: infer_inputType_and_get_mutDF 
  infer_inputType_and_get_mutDF <- function(inputFile = NULL, inputType = NULL){
    if (is.null(inputType)) {
      system("echo -e \"\\e[38;5;15mFiguring out the type of input data...\\e[0m\"")
      # Automatic determination of input type
      is.vcf <- !is.null(tryCatch({vcf <- fread(inputFile, skip = "#CHROM", sep = "\t")}, error = function(err){})) 
      is.tsv <- FALSE 
      if (!is.vcf){
        is.tsv <- !is.null(tryCatch({mutDF <- fread(inputFile, skip = paste(inputHeader, collapse="\t"), sep = "\t")}, error = function(err){}))
      }
      if (!is.vcf & !is.tsv){
        system("echo -e \"\\e[38;5;1mERROR: Unrecognizable input types (vcf or table). Please check the data formats. Or make sure if -l (--list) in case your input is of a file list.\\e[0m\"") 
        stop_quietly()  
      } 
      if (is.vcf == TRUE) {
        system("echo -e \"\\e[38;5;15mInput data type: VCF\\e[0m\"")
        inputFileName <- basename(inputFile) %>% gsub("(.*)[.]gz$","\\1",.) %>% gsub("(.*)[.]vcf$","\\1",.)
      } else if (is.vcf == FALSE & is.tsv == TRUE) {
        system("echo -e \"\\e[38;5;15mInput data type: TSV\\e[0m\"")
        mutDF <- setupTable(mutDF)
        inputFileName <- basename(inputFile) %>% gsub("(.*)[.]tsv$","\\1",.)
      } else {
        system("echo -e \"\\e[38;5;1mMissing input file or unrecognizable input format. In case of the latter, re-check names of the column header, i.e. -r or --header. Else, try -t or --type to manually specify the format.\\e[0m\"")
        stop_quietly()}
    } else if (inputType == "1") {
      is.vcf <- TRUE
      is.tsv <- FALSE 
      system("echo -e \"\\e[38;5;15mInput data type: VCF\\e[0m\"")
      vcf <- fread(inputFile, skip = "#CHROM", sep = "\t")
      inputFileName <- basename(inputFile) %>% gsub("(.*)[.]gz$","\\1",.) %>% gsub("(.*)[.]vcf$","\\1",.)
    } else if (inputType == "2") {
      is.vcf <- FALSE
      is.tsv <- TRUE
      system("echo -e \"\\e[38;5;15mInput data type: VCF\\e[0m\"")
      mutDF <- fread(inputFile, skip = paste(inputHeader,collapse="\t"), sep = "\t")
      mutDF <- setupTable(mutDF)
    }
    if (is.vcf == TRUE) {mutDF <- processVCF(vcf = vcf) %>% set_names(c("type","subtype",inputFileName))}
    return(list(is.vcf=is.vcf, is.tsv=is.tsv, mutDF=mutDF))
  } 
  if(!is.null(opt$custom_signatures)){
    custom_signatures_list_path <- opt$custom_signatures
    custom_signatures_vcf_list <- fread(custom_signatures_list_path, header=FALSE)
    if(ncol(custom_signatures_vcf_list)!=2){
      custom_signatures_vcf_list <- fread(custom_signatures_list_path, header=FALSE, sep="\t")
    }
    if(ncol(custom_signatures_vcf_list)!=2){
      custom_signatures_vcf_list <- fread(custom_signatures_list_path, header=FALSE, sep=",")
    }
    custom_signatures_vcf_list <- set_names(custom_signatures_vcf_list, c("SIGNAME", "FILE"))
    lapply(1:nrow(custom_signatures_vcf_list), function(i){
        tmp.vcf <- fread(custom_signatures_vcf_list$FILE[i])
        tmp.df <- processVCF(vcf = tmp.vcf, filtering = FALSE)
        tmp.df <- set_names(tmp.df, c("type","subtype",custom_signatures_vcf_list$SIGNAME[i]))
        if (i==1) {tmp.df} else {tmp.df[,3]}
    }) -> custom_signatures_table_list
    custom_signatures_table <- do.call(cbind, custom_signatures_table_list)
    if (ncol(custom_signatures_table)==3) { 
      # i.e., if only one custom signature
      custom_signatures_table[,3] <- custom_signatures_table[,3]/sum(custom_signatures_table[,3])
    } else {
      custom_signatures_table[,3:ncol(custom_signatures_table)] <- custom_signatures_table[,3:ncol(custom_signatures_table)]/colSums(custom_signatures_table[,3:ncol(custom_signatures_table)])
    }
  }
  # CUSTOM SIGNATURE Pt.1: END
  # Get inputFiles
  if (opt$list) {
    inputFiles <- fread(opt$input, header=FALSE)[[1]]
  } else { 
    inputFiles <- strsplit(opt$input, split=",")[[1]]
  }
  # Process each inputFile in inputFiles
  mutDF.list <- list()
  sample_count <- 0 
  for (i in 1:length(inputFiles)) {
    system(paste0("echo -e \"\\e[38;5;2mProcessing (",i,"/",length(inputFiles),") ",basename(inputFiles[i]),"...\\e[0m\"")) 
    tmp <- infer_inputType_and_get_mutDF(inputFile=inputFiles[i], inputType=inputType) 
    if(tmp$is.vcf){
      sample_count <- sample_count + 1
      mutDF.list <- mutDF.list %>% append(list(tmp$mutDF))
    } else if(tmp$is.tsv){
      sample_count <- sample_count + length(tmp$mutDF[,-c(1,2)])
      mutDF.list <- mutDF.list %>% append(list(tmp$mutDF))
    } 
    # Sanity check 
    if (sample_count > 1) {
      if (!is.null(opt$outputBase) & !opt$mergeData){
        system("echo -e \"\\e[033;0;33mWARNING: The outputBase (-b) has been specified, but --mergeData (-m) is FALSE. --outputBase (-b) will be ignored.\\e[0m\"")
      }
      if (is.null(opt$outputBase) & opt$mergeData){
        system("echo -e \"\\e[38;5;1mERROR: If --mergeData (-m) is ON , --outputBase (-b) should be specified.\\e[0m\"")
        stop_quietly()
      }
    }
  }
  # Aggregate data.frames
  if(mergeData){
    mutDF <- mutDF.list %>% lapply(function(x){x %>% set_names(c("type","subtype","orig"))}) %>% do.call(rbind,.) %>% aggregate(orig ~ type + subtype,.,sum) %>% arrange(type,subtype)
  } else {
    mutDF <- Reduce(function(d1,d2){left_join(d1,d2,by=c("type","subtype"))}, mutDF.list)
  }
	#----
	## M2 : Signature data processing ----------------------------------------------------
	system("echo -e \"\\e[38;5;15mPreparing signature data...\\e[0m\"")
  # HELPER FUNCTION :: process_signatureData 
  process_signatureData <- function(signatureNames = NULL){
    sigDF <- left_join(sigDF.coll$v2, sigDF.coll$v3, by=c("type","subtype")) %>% arrange(type, subtype)
    sigDF <- left_join(sigDF, sigDF.coll$v3.2, by=c("type","subtype")) 
    sigDF <- sigDF[,names(sigDF) %in% c("type","subtype",signatureNames %>% unlist), with=FALSE] 
    return(sigDF)
  } 
  sigDF <- process_signatureData(signatureNames = signatureNames %>% unlist) 
  if(!is.null(opt$custom_signatures)){
    sigDF <- cbind(custom_signatures_table, sigDF[-c(1,2)])
    sigDF <- set_names(sigDF, sigN)
	}
	# CUSTOM SIGNATURE Pt.2: END
	## M3 : Signature analysis ---------------------------------------------------
	system("echo -e \"\\e[38;5;15mRunning signature extraction...\\e[0m\"")
	# HELPER FUNCTION: SigAn_Engine 
	SigAn_Engine <- function(mutDF=NULL, sigDF=NULL, algorithm=NULL, signatureNames_chunks=NULL){
	  # HELPER FUNCTION : sigBarPlot
	  sigBarPlot <- function(x=NULL, ...){
	    # Setup colors
	    palette <- c(`C>A`="#1EBFF0",`C>G`="#050708",`C>T`="#E62725",`T>A`="#CBCACB",`T>C`="#A1CF64",`T>G`="#EDC8C5")
	    palette.df <- data.frame(type=names(palette), col=palette)
	    Colors <- left_join(mutDF["type"], palette.df, by="type")[["col"]]
	    xticks <- paste0(rep(c("A","C","G","T"), each=4, times=6),
	                     rep(c("C","T"), each=48),
	                     rep(c("A","C","G","T"), times=24))
	    barplot(x, col=Colors, names=xticks, las=2, cex.names = 0.8, ...)
	  }
	  # PREPROCESSING
	  mutDF_class <- class(mutDF)[1]
	  mutDF <- as.data.frame(mutDF)
	  sigDF <- as.data.frame(sigDF)
	  # Convert DF to a matrix
	  # Convert DF to a matrix 
	  remove_col.sigDF <- (names(sigDF) %>% tolower) %in% c("type","subtype","subsubtype")
	  remove_col.mutDF <- (names(mutDF) %>% tolower) %in% c("type","subtype","subsubtype")
	  C <- as.matrix(sigDF[,!remove_col.sigDF])#[,-c(1,2)])
	  Y <- as.matrix(mutDF[,!remove_col.mutDF])#-c(1,2)])
	  sampleNames <- names(mutDF)[!remove_col.mutDF]#[-c(1,2)]
	  sigNames <- colnames(C)
	  # Look through mutDF
	  lapply(1:ncol(Y), function(i) {
	    orig <- Y[,i] 
	    if (algorithm == 1) {
	      res <- lsqnonneg(C, orig)
	      res$w <- C 
	    }
	    if (algorithm == 2) {
	      res <- KLD(v=as.matrix(orig), w=C, max.conv=1000) 
	      res$w <- C
	    }
	    if (algorithm == 3) {
	      wts <- 1/(orig*(normalize(orig))*(1-normalize(orig)))
        wts[wts == Inf] <- 0 
	      res <- lsqnonneg(C * sqrt(wts), as.vector(sqrt(wts) * orig))
	      res$w <- C 
	    }
	    if (algorithm == 4) { 
	      # NOTE: Different order may result in different results. Use this with care.
	      # HELPER FUNCTION : SWLS
	      SWLS <- function(orig, sigDF, signatureNames_chunks=NULL, smoothing=FALSE, weighting=TRUE, pruning=FALSE, pruning_threshold=0, to_be_weighted_threshold_percent=-Inf){
	        if (!is.null(signatureNames_chunks)) {
	          sig_n_iter <- length(signatureNames_chunks)
	        } else {
	          sigNames <- names(sigDF)[-c(1,2)]
	          sig_n_iter <- length(sigNames)
	          signatureNames_chunks <- sigNames %>% as.list
	        } 
	        SWLS.out <- list()
	        for (i in 1:sig_n_iter) {
	          if (i == 1) {orig_iter <- orig}
	          if (smoothing) {
	            orig_iter <- (orig_iter + 1)/(sum(orig_iter) + length(orig_iter)) * sum(orig_iter) 
	          } else {
	            # Do nothing
	          }
	          if (weighting) {
              orig_tbw.idx <- !(orig_iter < to_be_weighted_threshold_percent*sum(orig_iter)/96) 
	            wts <- 1/(abs(orig_iter)*(normalize(abs(orig_iter)))*(1-normalize(abs(orig_iter)))) 
              wts[!orig_tbw.idx] <- 0 
	            if (!smoothing) {wts[wts == Inf] <- 0} 
	          } else {
	            wts <- rep(1,length(orig_iter))
	          }
            if (pruning) {
              pruning_idx <- orig_iter < (pruning_threshold * sum(orig_iter) / length(orig_iter)) 
              wts[pruning_idx] <- 0 
            }
            C <- as.matrix(sigDF[,signatureNames_chunks[[i]]]) 
            res <- lsqnonneg(C * sqrt(wts), as.vector(sqrt(wts) * orig_iter)) 
            recon <- C %*% res$x %>% as.vector 
	          resid <- orig_iter - recon
	          # Update
	          SWLS.out[[signatureNames_chunks[[i]] %>% paste0(collapse=",")]] <- list(orig_iter=orig_iter, expo=res$x, recon=recon, resid=resid) 
	          # For next iteration
            orig_iter <- resid
	        }
	        return(SWLS.out)
	      }
	      # HELPER FUNCTION : expos
	      expos <- function(SWLS.out){data.frame(signatures=names(SWLS.out) %>% strsplit(split=",") %>% unlist, exposures=SWLS.out %>% lapply(function(x){x$expo}) %>% unlist(use.names=FALSE))} 
          tbw_thr_percent <- 0.01 # peaks contributing less than 1 % gets removed.
	      SWLS.out <- SWLS(orig = orig, sigDF = sigDF, signatureNames_chunks = signatureNames_chunks,
	                       smoothing=FALSE, weighting=FALSE, pruning=FALSE,
	                       pruning_threshold=as.numeric(opt$pruning_threshold), 
	                       to_be_weighted_threshold_percent=tbw_thr_percent) 
	      res <- NULL
	      res$x <- expos(SWLS.out)$exposures
	      res$w <- as.matrix(sigDF[,which(names(sigDF) %in% expos(SWLS.out)$signatures)]) 
          recon <- as.vector(as.matrix(res$w) %*% as.numeric(res$x)) 
	      res$resid.norm <- sum((orig - recon)^2) 
	    }
	    if (algorithm == 5) {
	      wts <- 1/(orig*(normalize(orig))*(1-normalize(orig)))
        wts[wts == Inf] <- 0 
	      res <- KLD(v=as.matrix(orig) * sqrt(wts), w=C* sqrt(wts), max.conv=1000) 
	      res$w <- C
	    }
	    # Post-processing the result
        recon <- as.vector(as.matrix(res$w) %*% as.numeric(res$x)) 
        resid <- orig - recon 
	    res$orig <- mutDF[, c(which(remove_col.mutDF), (sum(remove_col.mutDF) + i)) ]#[,c(1, 2, 2+i)]
	    res$orig <- eval(parse(text=paste0("as.",mutDF_class,"(res$orig)"))) # the class of mutDF was changed to data.frame, and this step puts it back to its original class
	    res$recon <- cbind(mutDF[,remove_col.mutDF], recon) %>% set_names(names(mutDF[,c(which(remove_col.mutDF), (sum(remove_col.mutDF) + i))])) 
	    res$recon <- eval(parse(text=paste0("as.",mutDF_class,"(res$recon)")))
	    res$resid <- cbind(mutDF[,remove_col.mutDF], resid) 
	    res$abs_resid <- abs(orig - recon) %>% as.vector 
	    res$totalCount <- sum(orig)
	    res$overexplained <- resid[resid < 0] %>% sum %>% abs
	    res$underexplained <- resid[resid > 0] %>% sum %>% abs
	    res$abs_resid_count <- 0
        recon <- res$recon[[3]] 
	    for (i in 1:length(recon)) {
	      tol <- suppressWarnings(tolerance::bintol.int(recon[i], n=res$totalCount, P=0, side=2, method="JF")) 
	      upper_offset <- tol$`2-sided.upper` - recon[i] # (+)
	      lower_offset <- tol$`2-sided.lower` - recon[i] # (-)
	      if ((upper_offset >= resid[i]) & (resid[i] >= lower_offset)) { 
	        # Do nothing
	      } else {
	        if (resid[i] > 0) { 
	          res$abs_resid_count <- res$abs_resid_count + (resid[i] - upper_offset) 
	        } else if (resid[i] < 0) { 
	          res$abs_resid_count <- res$abs_resid_count + (lower_offset - resid[i]) 
	        }
	      }
	    } 
	    res$unexplained <- res$abs_resid_count 
	    res$pseudo_totalCount <- res$totalCount + res$unexplained 
        # Rescaling unexplained and pseudo_total count 
        res$fitted_totalCount <- sum(res$recon[[3]]) 
        res$scaling_factor <- res$totalCount / res$fitted_totalCount 
        if (is.nan(res$scaling_factor)) {res$scaling_factor <- 0} 
        res$x_rs <- res$x * res$scaling_factor 
        res$unexplained_rs <- res$unexplained * res$scaling_factor 
	    return(res)
	  }) %>% set_names(names(mutDF)[-c(1,2)]) -> res.coll 
	  return(res.coll)
	}
	if (signature_chunking){
	  signatureNames_chunks <- signatureNames
	} else {
	  signatureNames_chunks <- NULL
	} 
	sigAn.results.list <- SigAn_Engine(mutDF=mutDF, sigDF=sigDF, algorithm=opt$algorithm, signatureNames_chunks=signatureNames_chunks) 
  # PLOTTING
	# HELPER FUNCTION: plotMutationalSpectrum 
	plotMutationalSpectrum <- function(df, features = c("type","subtype"), bar_width = 0.8, alpha = 0.01) {
	  chg <- c("C>A","C>G","C>T","T>A","T>C","T>G")
	  # Colors
	  palette <- c(rgb(30,191,240, max = 255),rgb(5,7,8, max = 255),rgb(230,39,37, max = 255),rgb(203,202,203, max = 255),rgb(161,207,100, max = 255),rgb(237,200,197, max = 255))
	  names(palette) <- chg
	  # Plot
	  names(df) <- c("type","subtype","Original","Reconstructed","Residual")
	  p <- ggplot(data = melt(df, id.vars = features),aes_string(x = features[ features != "type" ], y = "value", fill = "type")) +
	    geom_bar(stat = "identity", color = "black", width = bar_width, size = 0.5) +
	    geom_rect(xmin = -Inf, xmax = Inf, ymin = -Inf, ymax = Inf, alpha = alpha) +
	    scale_fill_manual("Type", values = palette) +
	    theme(legend.position = "none",axis.text.x = element_text(angle = 90, hjust = 1),strip.text.x = element_text(size = 12, face = "bold"),strip.text.y = element_text(size = 12, face = "bold"),
	          axis.title.x=element_blank()) +
	    facet_grid(reformulate("type","variable"), scales = "free_x") +
	    ylab("Counts")
	  return(p)
	}
	# HELPER FUNCTION: plotExposure 
  plotExposure <- function(df, mode = NULL){
    qual_col_pals <- brewer.pal.info[brewer.pal.info$category == 'qual',]
    colors <- unlist(mapply(brewer.pal, qual_col_pals$maxcolors, rownames(qual_col_pals)))
    if (mode == "pie"){
      p <- ggplot(df, aes(x = "", y = prop, fill = Signature)) +
        geom_bar(width = 0.5, stat = "identity", color = "black", size = 0.2) +
        coord_polar(theta = "y", start = 0, direction = -1) +
        scale_fill_manual(values = colors[1:nrow(df)], labels = df$Signature) +
        theme_void()
    }
    if (mode == "bar"){
      df <- df[nrow(df):1,] 
      df$Signature <- factor(df$Signature, levels = df$Signature)
      p <- ggplot(df, aes(x=Signature, y=exp, fill=Signature)) +
        geom_bar(stat='identity', show.legend = FALSE, color = "black", size = 0.2) + 
        scale_fill_manual(values = {tmp.colors <- rev(colors[1:nrow(df)]); tmp.colors[1] <- "grey50"; tmp.colors}) + 
        xlab("Signature") + ylab("Exposure (Counts)") + 
        theme(legend.position="none") +
        scale_x_discrete(limits=df$Signature, labels=gsub("Signature ","",df$Signature)) +
        coord_flip(clip = 'off')
      # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING 
      if(!is.na(sum(df$prop))){
        p <- p + annotate("label", x = df$Signature, y = df$exp, label = c(paste0(df$prop[df$prop > 0]," %"),rep(NA,sum(df$prop == 0))), size = 2.5, hjust=-0.1)
      } else {}
      # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING 
    }
    return(p)
  }
  # HELPER FUNCTION: plotNucleotideChanges 
  plotNucleotideChanges <- function(df){
    df <- aggregate(df$orig, by=list(df$type), sum)
    names(df) <- c("Substitution","Count")
    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING 
    if(sum(df$Count)!=0){
      df$Prop <- trunc(df$Count/sum(df$Count)*10^3)/10
    } else {df$Prop <- NA}
    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING 
    df$Pos <- cumsum(rev(df$Count)) - rev(df$Count)/2
    palette <- c(rgb(30,191,240, max = 255),rgb(5,7,8, max = 255),rgb(230,39,37, max = 255),rgb(203,202,203, max = 255),rgb(161,207,100, max = 255),rgb(237,200,197, max = 255))
    p <- ggplot(df, aes(x = "", y = Count, fill = Substitution)) +
      geom_bar(width = 0.5, stat = "identity", color = "black", size = 0.2) +
      coord_polar(theta = "y", start = 0, direction = -1) +
      scale_fill_manual(values = palette, labels = df$Substitution) +
      annotate("label", x = 1.25, y = df$Pos, label = rev(paste0(df$Prop," %")), size = 2.5) +
      theme_void()
    return(p)
  } 
  # HELPER FUNCTION: plotCosineSim 
  plotCosineSim <- function(df) {
    A <- as.matrix(df$orig)
    B <- as.matrix(df$recon)
    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING, 2019-12-16 
    if(sum(A)==0|sum(B)==0){
      cosSim <- NA
    } else {
      cosSim <- (t(A) %*% B)/sqrt(t(A) %*% A)/sqrt(t(B) %*% B) 
    }
    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING, 2019-12-16 
    text <- paste0("Cosine similarity: ",trunc(cosSim*10^3)/10^3) 
    p <- ggplot() +
      annotate("text", x = 0, y = 0, size=5, label = text) +
      theme_void()
    return(list(p=p,cosSim=cosSim)) 
  }
  # HELPER FUNCTION: plotTotalCounts 
	plotTotalCounts <- function(df){
	  tot <- sum(df$orig)
	  text <- paste0("Total mutation count: ",tot)
	  p <- ggplot() +
	    annotate("text", x=0, y=0, size=5, label=text) +
	    theme_void()
	  return(list(p=p,totCount=tot))
	}
	# HELPER FUNCTION: plotUnexplainedCounts 
	plotUnexplainedCounts <- function(res){
      unxp <- ceil(res$unexplained_rs) 
	  text <- paste0("Unexplained mutation count: ",unxp)
	  p <- ggplot() +
	    annotate("text", x=0, y=0, size=5, label=text) +
	    theme_void()
	  return(list(p=p,unxpCount=unxp))
	}
	# HELPER FUNCTION: generate_and_save_plots 
	generate_and_save_plots <- function(sigAn.results.list=NULL, mutDF=NULL, sigDF=NULL, mergeData=FALSE){
	  for (i in 1:length(sigAn.results.list)){
	    # Argument handling
	    res <- sigAn.results.list[[i]]
	    sampleName <- names(sigAn.results.list[i])
	    mutDF.i <- mutDF[,c("type","subtype",sampleName)] %>% set_names(c("type","subtype","orig"))
	    C <- as.matrix(sigDF[,-c(1,2)])
	    sigNames <- colnames(C)
	    if (mergeData) {
	      outputBase <- opt$outputBase
	    } else {
	      outputBase <- names(sigAn.results.list[i]) # if mergeData is FALSE, follow the input filenames
	    }
	    # Body
        # Exposure table
	    exp <- data.frame(Signature = sigNames,
	                      exp = res$x_rs, # 2021-07-05, muted # 2021-07-06, fitted counts rescaled to original counts.
	                      stringsAsFactors=FALSE) 
	    exp <- arrange(exp, desc(exp))
	    exp <- rbind(exp,
                     data.frame(Signature = "Unexplained",
                                exp = res$unexplained_rs, 
                                stringsAsFactors=FALSE))
        exp$prop <- if(sum(res$x)!=0){
            round(exp$exp / (res$unexplained_rs + res$totalCount) * 100, 2) 
        } else {NA} 
	    exp$Signature <- factor(exp$Signature, levels = exp$Signature)#, ordered = TRUE)
	    mutDF.i$recon <- C %*% res$x
	    mutDF.i$resid <- mutDF.i[,3] - mutDF.i$recon
	    # Save to files
	    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING, 2019-12-16
	    ## Plot
	    # system("echo -e \"\\e[38;5;15mSaving files...\"")
	    system(paste0("echo -ne \"\\e[38;5;15mSaving results...(",i,"/",length(sigAn.results.list),")\\r\"")) 
	    pdf(NULL) # To prevent Rplot.pdf from being created.
        p.m <- grid.grabExpr(plotSNVspectra(mutDF.i, newpage=FALSE), wrap.grobs=TRUE) 
	    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING, 2019-12-16
	    # When SNVs are missing from all samples... sum(res$x)==0
	    if(!is.na(sum(exp$prop))){exp.nonzero <- filter(exp, prop > 0)}
	    if(is.na(sum(exp$prop))){exp.nonzero <- exp} # Note that it is not actually exp.nonzero, but regarded as if for convenience
	    # EXCEPTION HANDLING EXCEPTION HANDLING EXCEPTION HANDLING, 2019-12-16
	    p.ep <- plotExposure(exp.nonzero, mode = "pie")
	    suppressWarnings(p.eb <- plotExposure(exp.nonzero, mode = "bar"))
	    plotCosineSim.out <- plotCosineSim(mutDF.i) 
	    cosSim <- plotCosineSim.out$cosSim 
	    p.c <- plotCosineSim.out$p 
	    p.chg <- plotNucleotideChanges(mutDF.i)
	    plotTotalCounts.out <- plotTotalCounts(mutDF.i) 
	    p.tot <- plotTotalCounts.out$p
	    totCount <- plotTotalCounts.out$totCount
	    plotUnexplainedCounts.out <- plotUnexplainedCounts(res) 
	    p.unxp <- plotUnexplainedCounts.out$p
	    unxpCount <- plotUnexplainedCounts.out$unxpCount
	    p <- grid.arrange(grobs=list(p.chg,p.m,p.eb,p.tot,p.c,p.unxp),
	                      layout_matrix=rbind(c(1,1,2,2,2,2,2,2),
	                                          c(1,1,2,2,2,2,2,2),
	                                          c(1,1,2,2,2,2,2,2),
	                                          c(1,1,2,2,2,2,2,2),
	                                          c(3,3,2,2,2,2,2,2),
	                                          c(3,3,2,2,2,2,2,2),
	                                          c(3,3,2,2,2,2,2,2),
	                                          c(3,3,2,2,2,2,2,2),
	                                          c(4,4,2,2,2,2,2,2),
	                                          c(6,6,2,2,2,2,2,2),
	                                          c(5,5,2,2,2,2,2,2),
	                                          c(NA,NA,2,2,2,2,2,2) 
	                      ),
	                      top=textGrob(outputBase, gp=gpar(fontsize=20,font=2)))
	    # Save the results to files
	    system(paste0("if [ ! -d ",outputDir," ];then mkdir -p ",outputDir,"; fi")) 
	    suppressMessages(ggsave(filename = paste0(outputDir,"/",outputPrefix,outputBase,".signature_analysis",outputSuffix,".png"), plot = p,
	                            width = 45, height = 15, units = "cm",
	                            dpi = 300))
	    suppressMessages(ggsave(filename = paste0(outputDir,"/",outputPrefix,outputBase,".signature_analysis",outputSuffix,".pdf"), plot = p,
	                            width = 45, height = 15, units = "cm",
	                            dpi = 300)) 
	    names(sigDF) <- c("Substitution","Trinucleotide",sigNames) 
	    fwrite(sigDF,file = paste0(outputDir,"/",outputPrefix,outputBase,".signature_individual_spectra",outputSuffix,".tsv"),sep="\t") 
	    names(mutDF.i) <- c("Substitution","Trinucleotide","Original","Reconstructed","Residual")
	    fwrite(mutDF.i,file = paste0(outputDir,"/",outputPrefix,outputBase,".signature_spectra",outputSuffix,".tsv"),sep="\t")
	    names(exp) <- c("Signature","Exposure","Proportion") 
	    fwrite(exp, file = paste0(outputDir,"/",outputPrefix,outputBase,".signature_exposures",outputSuffix,".tsv"),sep="\t",na="NA",quote=FALSE) 
	    INFO.file <- paste0(outputDir,"/",outputPrefix,outputBase,".INFO",outputSuffix,".tsv")
	    system(paste0("touch ",INFO.file))
	    info <- paste0("# Cosine similarity: ",cosSim,"\n",
	                   "# Total mutation count: ",totCount,"\n",
	                   "# Unexplained mutation count: ",unxpCount 
	                   )
	    system(paste0("echo \"",info,"\" > ",INFO.file))
        saveRDS(p,paste0(outputDir,"/",outputPrefix,outputBase,".grob_objects",outputSuffix,".rds")) 
        dev.off() 
	  }
	  system("echo \"\"") 
	} 
	# Process results
	generate_and_save_plots(sigAn.results.list=sigAn.results.list, mutDF=mutDF, sigDF=sigDF, mergeData=mergeData)  
	# Completed.
	system("echo -e \"\\e[42mJob completed.\\e[0m\"")
}
# Signature Data :: COSMIC_v2, COSMIC_3, COSMIC_3.2
# COSMIC_v2 
COSMIC_v2 <- c(
"Substitution Type\tTrinucleotide\tSomatic Mutation Type\tSignature 1\tSignature 2\tSignature 3\tSignature 4\tSignature 5\tSignature 6\tSignature 7\tSignature 8\tSignature 9\tSignature 10\tSignature 11\tSignature 12\tSignature 13\tSignature 14\tSignature 15\tSignature 16\tSignature 17\tSignature 18\tSignature 19\tSignature 20\tSignature 21\tSignature 22\tSignature 23\tSignature 24\tSignature 25\tSignature 26\tSignature 27\tSignature 28\tSignature 29\tSignature 30\t\t\t\t\t\t\t\r",
"C>A\tACA\tA[C>A]A\t0.011098326166\t0.000682708227\t0.022172306775\t0.036500000000\t0.014941547714\t0.001700000000\t0.000400000000\t0.036718003777\t0.012000000000\t0.000700000000\t0.000200000000\t0.007700000000\t0.000334757187\t0.000100000000\t0.001300000000\t0.016100000000\t0.001832019172\t0.050536418625\t0.010700000000\t0.001179961596\t0.000100000000\t0.001504070375\t0.000453360683\t0.028645992536\t0.009896768001\t0.002039772910\t0.005205626909\t0.001397438842\t0.069981987280\t0.000000000000\t\t\t\t\t\t\t\r",
"C>A\tACC\tA[C>A]C\t0.009149340734\t0.000619107232\t0.017871675353\t0.030900000000\t0.008960917997\t0.002800000000\t0.000500000000\t0.033245722246\t0.006700000000\t0.001000000000\t0.001000000000\t0.004700000000\t0.000648736139\t0.004200000000\t0.004000000000\t0.009700000000\t0.000342235622\t0.010939824789\t0.007400000000\t0.002211505148\t0.000700000000\t0.002451011015\t0.000366800484\t0.020214638433\t0.006998928770\t0.001487162284\t0.004738227415\t0.000917187665\t0.055152357207\t0.000000000000\t\t\t\t\t\t\t\r",
"C>A\tACG\tA[C>A]G\t0.001490070468\t0.000099278956\t0.002138339617\t0.018300000000\t0.002207846012\t0.000500000000\t0.000000000000\t0.002525311256\t0.000500000000\t0.000300000000\t0.000000000000\t0.001700000000\t0.000038144594\t0.000500000000\t0.000000000000\t0.002200000000\t0.000001576225\t0.002288072697\t0.000500000000\t0.000000161691\t0.000000000000\t0.000000000000\t0.000000000000\t0.020478996485\t0.001448442998\t0.000283945614\t0.000782697937\t0.000000000000\t0.017846984044\t0.001967300000\t\t\t\t\t\t\t\r",
"C>A\tACT\tA[C>A]T\t0.006233885236\t0.000323891363\t0.016265145590\t0.024300000000\t0.009206905292\t0.001900000000\t0.000400000000\t0.033598549516\t0.006800000000\t0.009200000000\t0.000200000000\t0.004600000000\t0.000846658460\t0.029600000000\t0.005700000000\t0.008800000000\t0.003179664772\t0.019424091364\t0.007400000000\t0.003008009987\t0.000600000000\t0.000922452518\t0.000000000000\t0.024600145425\t0.004966565004\t0.000597865638\t0.002718242455\t0.000513409971\t0.026804715973\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tACA\tA[C>G]A\t0.001801068369\t0.000263481044\t0.024002615202\t0.009700000000\t0.011671021651\t0.001300000000\t0.000000000000\t0.008356844806\t0.004800000000\t0.000500000000\t0.000700000000\t0.003100000000\t0.003775165298\t0.000100000000\t0.001100000000\t0.004800000000\t0.001661704195\t0.001516887580\t0.005800000000\t0.000697041070\t0.000500000000\t0.000527644040\t0.000000000000\t0.011993038821\t0.008032567646\t0.001272881297\t0.001324163246\t0.000255194996\t0.009301971097\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tACC\tA[C>G]C\t0.002580908520\t0.000269866049\t0.012160303668\t0.005400000000\t0.007292088583\t0.001200000000\t0.000000000000\t0.004306380304\t0.002300000000\t0.000300000000\t0.000300000000\t0.001500000000\t0.000920824828\t0.000000000000\t0.000100000000\t0.002400000000\t0.001626898399\t0.002498784527\t0.001900000000\t0.002059310006\t0.000800000000\t0.000000000000\t0.000000000000\t0.008425119792\t0.001635970413\t0.001528194862\t0.001771084129\t0.000268898964\t0.003479048871\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tACG\tA[C>G]G\t0.000592548022\t0.000219233911\t0.005275419458\t0.003100000000\t0.002303839151\t0.000000000000\t0.000000000000\t0.000584415303\t0.000000000000\t0.000000000000\t0.000000000000\t0.000000000000\t0.000019890489\t0.000000000000\t0.000600000000\t0.000000000000\t0.000025801897\t0.002614508983\t0.000000000000\t0.000012734958\t0.000000000000\t0.000000000000\t0.000000000000\t0.000000000000\t0.000000000000\t0.000307246308\t0.000000000000\t0.000000000000\t0.000154433523\t0.004820000000\t\t\t\t\t\t\t\r",
"C>G\tACT\tA[C>G]T\t0.002963986287\t0.000610973492\t0.023277656342\t0.005400000000\t0.011696245747\t0.001800000000\t0.000100000000\t0.008634906912\t0.003800000000\t0.000200000000\t0.000900000000\t0.002500000000\t0.003860632488\t0.000100000000\t0.001000000000\t0.007300000000\t0.001328507581\t0.003983011170\t0.007200000000\t0.000848587116\t0.001800000000\t0.000299454735\t0.000000000000\t0.003881383185\t0.003428449701\t0.002498251826\t0.001322719819\t0.000307698016\t0.003976789927\t0.000000000000\t\t\t\t\t\t\t\r",
"C>T\tACA\tA[C>T]A\t0.029514532745\t0.007441556817\t0.017872170732\t0.012000000000\t0.021839187574\t0.031200000000\t0.000000000000\t0.018066687156\t0.009300000000\t0.000000000000\t0.022500000000\t0.012100000000\t0.001480180764\t0.029300000000\t0.011700000000\t0.013500000000\t0.009350488403\t0.003825623912\t0.022100000000\t0.032858047272\t0.005100000000\t0.003113389695\t0.019767154372\t0.006315518136\t0.020987618053\t0.005907232039\t0.013723285787\t0.005433792471\t0.005197221326\t0.065119000000\t\t\t\t\t\t\t\r",
"C>T\tACC\tA[C>T]C\t0.014322747041\t0.002726312428\t0.008896034323\t0.007500000000\t0.012756105638\t0.016300000000\t0.019700000000\t0.005650015029\t0.005600000000\t0.003200000000\t0.109900000000\t0.005400000000\t0.000040508707\t0.032100000000\t0.016900000000\t0.007600000000\t0.004224289013\t0.002465214261\t0.020300000000\t0.022196046913\t0.006200000000\t0.001524860290\t0.081880579925\t0.003844326917\t0.013141283697\t0.010626457816\t0.005424306001\t0.001815415488\t0.008976469990\t0.054397000000\t\t\t\t\t\t\t\r",
"C>T\tACG\tA[C>T]G\t0.171646931305\t0.003322083281\t0.003572708350\t0.002800000000\t0.016760177116\t0.090800000000\t0.000100000000\t0.019265105465\t0.012500000000\t0.012600000000\t0.007200000000\t0.001900000000\t0.000316594671\t0.049600000000\t0.030900000000\t0.003500000000\t0.009229213313\t0.008463956886\t0.036800000000\t0.034572014703\t0.025600000000\t0.003625641201\t0.009248715085\t0.003110664425\t0.019659894285\t0.019930323239\t0.006706429454\t0.005432039331\t0.031638996360\t0.020460400000\t\t\t\t\t\t\t\r",
"C>T\tACT\tA[C>T]T\t0.012623763151\t0.003326528417\t0.014797612209\t0.005900000000\t0.016477954692\t0.014900000000\t0.004300000000\t0.020805967497\t0.007600000000\t0.004700000000\t0.063900000000\t0.006700000000\t0.000000000000\t0.027700000000\t0.006600000000\t0.010100000000\t0.005735961173\t0.005605739927\t0.026600000000\t0.018819559611\t0.003000000000\t0.000000000000\t0.033209652209\t0.003420418673\t0.005728519312\t0.011334530552\t0.004475614034\t0.001629213189\t0.004769666594\t0.021935900000\t\t\t\t\t\t\t\r",
"T>A\tATA\tA[T>A]A\t0.004021520333\t0.000000132377\t0.008428563566\t0.004800000000\t0.008902904323\t0.000600000000\t0.001000000000\t0.013365082811\t0.012100000000\t0.000000000000\t0.000200000000\t0.005800000000\t0.001243618346\t0.000000000000\t0.002100000000\t0.008100000000\t0.000613002450\t0.003037210804\t0.001100000000\t0.000574401358\t0.008300000000\t0.049619229712\t0.000000000000\t0.001166747126\t0.020430527125\t0.004459339879\t0.143076389009\t0.000964030891\t0.000000000000\t0.007574300000\t\t\t\t\t\t\t\r",
"T>A\tATC\tA[T>A]C\t0.002371144163\t0.000113069562\t0.007373096670\t0.003900000000\t0.007399202496\t0.003300000000\t0.000800000000\t0.012431202077\t0.004200000000\t0.000200000000\t0.000100000000\t0.006500000000\t0.001060001878\t0.000800000000\t0.009000000000\t0.007300000000\t0.001878587472\t0.000811167214\t0.005300000000\t0.001971462059\t0.029200000000\t0.011666740464\t0.000000000000\t0.000015173057\t0.008892560661\t0.012822242916\t0.001464085387\t0.003365667809\t0.003049595749\t0.003738000000\t\t\t\t\t\t\t\r",
"T>A\tATG\tA[T>A]G\t0.002810909959\t0.000532929416\t0.007357300020\t0.010000000000\t0.011508051457\t0.000000000000\t0.000900000000\t0.014036657067\t0.006800000000\t0.000000000000\t0.000700000000\t0.004600000000\t0.000102554360\t0.000000000000\t0.000900000000\t0.008000000000\t0.001087349685\t0.004503207138\t0.004500000000\t0.000190586616\t0.002700000000\t0.067915454213\t0.000237418521\t0.006332455573\t0.006241948480\t0.001172344272\t0.001226280897\t0.001785753650\t0.003909981984\t0.006590600000\t\t\t\t\t\t\t\r",
"T>A\tATT\tA[T>A]T\t0.008360909345\t0.000149110959\t0.008753938991\t0.003000000000\t0.011193623195\t0.005300000000\t0.003500000000\t0.024109916596\t0.018500000000\t0.001200000000\t0.000300000000\t0.006000000000\t0.000971812099\t0.000900000000\t0.007900000000\t0.008400000000\t0.002045493651\t0.002736523185\t0.003300000000\t0.007421103528\t0.004300000000\t0.014599917580\t0.000000000000\t0.000004967216\t0.014850877622\t0.003993137549\t0.002535197951\t0.005653089100\t0.005105527437\t0.009344700000\t\t\t\t\t\t\t\r",
"T>C\tATA\tA[T>C]A\t0.013915773030\t0.001303509558\t0.013035709432\t0.008400000000\t0.035366735114\t0.007500000000\t0.000500000000\t0.016206510101\t0.025200000000\t0.003100000000\t0.000600000000\t0.046600000000\t0.000549454166\t0.002600000000\t0.011500000000\t0.071700000000\t0.002697314025\t0.002933053644\t0.014500000000\t0.016395062964\t0.026500000000\t0.016839930949\t0.001068448820\t0.004726009142\t0.020663282773\t0.055028652133\t0.005047301081\t0.004092746888\t0.010119181163\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tATC\tA[T>C]C\t0.006274960600\t0.000425576677\t0.009186289690\t0.002000000000\t0.013771102129\t0.005600000000\t0.000100000000\t0.007787802820\t0.010500000000\t0.005200000000\t0.001500000000\t0.031700000000\t0.000145506661\t0.004600000000\t0.007300000000\t0.017100000000\t0.008078150072\t0.001513284763\t0.004800000000\t0.019251793639\t0.016500000000\t0.001108262382\t0.000221573159\t0.000627949341\t0.009502478424\t0.027594604208\t0.001845691228\t0.007880729211\t0.006349682655\t0.008754700000\t\t\t\t\t\t\t\r",
"T>C\tATG\tA[T>C]G\t0.010137636154\t0.000575473007\t0.011716920077\t0.008100000000\t0.028449051657\t0.021700000000\t0.000700000000\t0.009852583553\t0.017600000000\t0.002600000000\t0.000800000000\t0.048700000000\t0.000279178024\t0.003500000000\t0.007300000000\t0.034800000000\t0.004432870698\t0.003329668955\t0.013500000000\t0.054784918009\t0.022400000000\t0.004913616405\t0.000498801514\t0.002723758627\t0.019911492456\t0.051791044050\t0.005688813879\t0.005520833604\t0.014659610308\t0.009049800000\t\t\t\t\t\t\t\r",
"T>C\tATT\tA[T>C]T\t0.009256316389\t0.001488164093\t0.016978870798\t0.003600000000\t0.027302974158\t0.002300000000\t0.001100000000\t0.021829463826\t0.024700000000\t0.004000000000\t0.000300000000\t0.025000000000\t0.000267461432\t0.002500000000\t0.006300000000\t0.049700000000\t0.003140370209\t0.005491317771\t0.016000000000\t0.007341269916\t0.018900000000\t0.002630523933\t0.000313240541\t0.007122368470\t0.014488390255\t0.039072244419\t0.002592754577\t0.003232484000\t0.007892413403\t0.000000000000\t\t\t\t\t\t\t\r",
"T>G\tATA\tA[T>G]A\t0.001587636423\t0.000033718411\t0.002351368076\t0.000000000000\t0.003462151975\t0.000000000000\t0.000000000000\t0.004219719284\t0.033500000000\t0.000000000000\t0.000000000000\t0.001200000000\t0.000386199790\t0.000000000000\t0.000000000000\t0.005200000000\t0.000053139249\t0.000386138197\t0.001400000000\t0.000001765291\t0.000000000000\t0.001651998615\t0.000000000000\t0.000000000000\t0.005281679682\t0.000080653426\t0.001136337392\t0.006364675555\t0.000000000000\t0.005311800000\t\t\t\t\t\t\t\r",
"T>G\tATC\tA[T>G]C\t0.001784091288\t0.000024844826\t0.001464231129\t0.000200000000\t0.002246592659\t0.001700000000\t0.000000000000\t0.000793671329\t0.005100000000\t0.004000000000\t0.000000000000\t0.001000000000\t0.000112383214\t0.002000000000\t0.000600000000\t0.004200000000\t0.006468661271\t0.001254982080\t0.000000000000\t0.000381387913\t0.003100000000\t0.000000000000\t0.000000000000\t0.001463950688\t0.000000000000\t0.000163171210\t0.000958976380\t0.016610905647\t0.000000000000\t0.002459200000\t\t\t\t\t\t\t\r",
"T>G\tATG\tA[T>G]G\t0.001385830552\t0.000273454078\t0.009053777642\t0.001500000000\t0.005490216837\t0.000700000000\t0.000900000000\t0.006865800374\t0.005200000000\t0.001500000000\t0.000100000000\t0.001800000000\t0.000036000696\t0.000700000000\t0.001200000000\t0.006100000000\t0.003481549012\t0.002369216073\t0.001000000000\t0.000658758087\t0.001600000000\t0.001086872758\t0.000000000000\t0.000847612530\t0.007136421644\t0.001255079507\t0.000067660610\t0.006592842245\t0.002925171448\t0.006295500000\t\t\t\t\t\t\t\r",
"T>G\tATT\tA[T>G]T\t0.003158539312\t0.000217694872\t0.007030976419\t0.000200000000\t0.003821226712\t0.002900000000\t0.000000000000\t0.002939922259\t0.019300000000\t0.013700000000\t0.000000000000\t0.003000000000\t0.000192493754\t0.005600000000\t0.002900000000\t0.007600000000\t0.030926375169\t0.000338602881\t0.001700000000\t0.000532125222\t0.001800000000\t0.000000000000\t0.000000000000\t0.001075455294\t0.000000000000\t0.001850383492\t0.000256027748\t0.086392897435\t0.000000000000\t0.009541600000\t\t\t\t\t\t\t\r",
"C>A\tCCA\tC[C>A]A\t0.006595870109\t0.000677444988\t0.018781725573\t0.046100000000\t0.009674904330\t0.010100000000\t0.001200000000\t0.031723756590\t0.009800000000\t0.003100000000\t0.000700000000\t0.013500000000\t0.001710089561\t0.005600000000\t0.010600000000\t0.015900000000\t0.001032430218\t0.088768108819\t0.011200000000\t0.017377110647\t0.002000000000\t0.004549692944\t0.000164739382\t0.063559283793\t0.014832947875\t0.003705850083\t0.005065073268\t0.001168515637\t0.051410211701\t0.000000000000\t\t\t\t\t\t\t\r",
"C>A\tCCC\tC[C>A]C\t0.007342367815\t0.000213681013\t0.015760457842\t0.061400000000\t0.004952300642\t0.024100000000\t0.000600000000\t0.025505407135\t0.005700000000\t0.000900000000\t0.001700000000\t0.011200000000\t0.001159256581\t0.010200000000\t0.008400000000\t0.010000000000\t0.000421880127\t0.020641390557\t0.015900000000\t0.036502462984\t0.001400000000\t0.003764473854\t0.000736874802\t0.033757004716\t0.007822175302\t0.003980723373\t0.002234153344\t0.000334291811\t0.025825650831\t0.000000000000\t\t\t\t\t\t\t\r",
"C>A\tCCG\tC[C>A]G\t0.000892840370\t0.000006770460\t0.001963389846\t0.008800000000\t0.002800627307\t0.009100000000\t0.000000000000\t0.001159624303\t0.000000000000\t0.000700000000\t0.001000000000\t0.002800000000\t0.000244166533\t0.000900000000\t0.001500000000\t0.002200000000\t0.000297462769\t0.017178402480\t0.001800000000\t0.012482587542\t0.002700000000\t0.000900163331\t0.000163953661\t0.022428985806\t0.001276976732\t0.000811741970\t0.000266312161\t0.000053651998\t0.014496183324\t0.002262400000\t\t\t\t\t\t\t\r",
"C>A\tCCT\tC[C>A]T\t0.007186581642\t0.000416332906\t0.014722861126\t0.043200000000\t0.011013465824\t0.057100000000\t0.001300000000\t0.028791172971\t0.009100000000\t0.016000000000\t0.001400000000\t0.007100000000\t0.001256768244\t0.125700000000\t0.022800000000\t0.008400000000\t0.000031479429\t0.037676958929\t0.009600000000\t0.103401226249\t0.005600000000\t0.004439846231\t0.000722731835\t0.020086515438\t0.012563654719\t0.019038431345\t0.003100570009\t0.000186671902\t0.040355074099\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tCCA\tC[C>G]A\t0.001284983446\t0.000027772077\t0.016832571989\t0.010500000000\t0.007537523174\t0.000000000000\t0.000100000000\t0.006619077216\t0.001800000000\t0.000000000000\t0.000000000000\t0.001900000000\t0.005333651375\t0.000000000000\t0.000900000000\t0.004000000000\t0.001924978751\t0.000458463822\t0.000200000000\t0.000013316260\t0.000100000000\t0.000582017663\t0.000199049174\t0.013476821609\t0.007200790410\t0.001279106069\t0.004730649426\t0.000103366092\t0.002618863169\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tCCC\tC[C>G]C\t0.000702134818\t0.000279643862\t0.013531441527\t0.009700000000\t0.007633226665\t0.000000000000\t0.000400000000\t0.006077702895\t0.002000000000\t0.000000000000\t0.000200000000\t0.001900000000\t0.000912303471\t0.000000000000\t0.000300000000\t0.005000000000\t0.000709367774\t0.003654228330\t0.002800000000\t0.000548144708\t0.000000000000\t0.000825299647\t0.000283776026\t0.016105687737\t0.004598473117\t0.001214977303\t0.001261644842\t0.000144310792\t0.001111512270\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tCCG\tC[C>G]G\t0.000506289594\t0.000019161576\t0.004176457524\t0.006300000000\t0.002613760399\t0.000000000000\t0.000600000000\t0.000656033433\t0.000000000000\t0.000000000000\t0.000800000000\t0.000000000000\t0.000285467331\t0.000000000000\t0.000400000000\t0.002800000000\t0.000003517869\t0.005405774245\t0.000200000000\t0.000000000000\t0.000000000000\t0.000382114636\t0.000000000000\t0.000797471998\t0.000823709283\t0.000207832588\t0.000360856587\t0.000000000000\t0.001520046950\t0.003246100000\t\t\t\t\t\t\t\r",
"C>G\tCCT\tC[C>G]T\t0.001381542722\t0.000312781573\t0.024046390424\t0.009400000000\t0.009417097876\t0.000200000000\t0.000300000000\t0.007805026957\t0.003900000000\t0.000000000000\t0.000200000000\t0.001900000000\t0.006383097568\t0.000000000000\t0.001700000000\t0.008000000000\t0.000953776334\t0.005519604820\t0.000800000000\t0.000120730556\t0.000300000000\t0.001760645911\t0.000195775338\t0.010388665882\t0.000196360325\t0.002297026211\t0.000268026230\t0.000290999509\t0.000139750691\t0.000000000000\t\t\t\t\t\t\t\r",
"C>T\tCCA\tC[C>T]A\t0.020896446965\t0.015019506931\t0.014395059491\t0.021000000000\t0.022769209355\t0.008500000000\t0.075400000000\t0.004865810021\t0.009800000000\t0.001200000000\t0.031000000000\t0.018700000000\t0.003647278451\t0.006000000000\t0.005100000000\t0.021000000000\t0.021265205902\t0.009767511475\t0.043800000000\t0.011145185306\t0.003400000000\t0.007731349522\t0.060572627805\t0.006035447394\t0.014333994691\t0.006594444885\t0.017438484778\t0.004391532554\t0.009126471767\t0.069447200000\t\t\t\t\t\t\t\r",
"C>T\tCCC\tC[C>T]C\t0.018501704770\t0.003516918098\t0.008544781192\t0.014400000000\t0.017509476305\t0.009900000000\t0.100700000000\t0.003980150431\t0.006900000000\t0.002400000000\t0.151800000000\t0.009400000000\t0.001004229295\t0.016700000000\t0.008800000000\t0.009800000000\t0.003847235312\t0.005654828462\t0.075300000000\t0.018152746866\t0.009100000000\t0.007110850526\t0.186375634076\t0.000000000000\t0.005936643659\t0.006510631971\t0.007144870207\t0.002647660966\t0.007067603037\t0.063840300000\t\t\t\t\t\t\t\r",
"C>T\tCCG\tC[C>T]G\t0.095577217268\t0.004979275715\t0.003518465840\t0.007600000000\t0.012862237589\t0.090100000000\t0.020800000000\t0.008338639182\t0.007600000000\t0.010900000000\t0.013500000000\t0.004600000000\t0.000129202148\t0.029300000000\t0.015500000000\t0.001600000000\t0.013176112314\t0.016183222964\t0.023800000000\t0.044384942487\t0.031800000000\t0.005502030965\t0.015149082593\t0.002924419074\t0.011559391627\t0.011904935324\t0.007503561655\t0.002284563551\t0.019875180409\t0.017312600000\t\t\t\t\t\t\t\r",
"C>T\tCCT\tC[C>T]T\t0.017113307642\t0.008956552775\t0.016075545712\t0.020100000000\t0.020429513385\t0.008700000000\t0.078800000000\t0.018843627434\t0.009700000000\t0.004300000000\t0.081600000000\t0.020500000000\t0.000611769970\t0.018600000000\t0.009600000000\t0.021100000000\t0.009648025316\t0.005153318613\t0.115900000000\t0.013847136214\t0.004000000000\t0.004169777241\t0.156015396977\t0.003971201575\t0.013632596885\t0.006238556327\t0.007148027702\t0.002423984267\t0.001634058687\t0.034231800000\t\t\t\t\t\t\t\r",
"T>A\tCTA\tC[T>A]A\t0.001182587416\t0.000154598981\t0.007571320260\t0.007500000000\t0.005043163511\t0.000100000000\t0.000000000000\t0.012024217522\t0.005500000000\t0.000000000000\t0.000400000000\t0.001700000000\t0.000285158991\t0.000000000000\t0.001600000000\t0.005400000000\t0.000030079241\t0.000412346353\t0.004300000000\t0.000045042364\t0.000000000000\t0.086898545516\t0.000594528453\t0.002766634838\t0.026296323089\t0.003561422681\t0.330601017471\t0.002566579967\t0.001765447503\t0.005311800000\t\t\t\t\t\t\t\r",
"T>A\tCTC\tC[T>A]C\t0.001903166857\t0.000464019028\t0.012725463528\t0.011100000000\t0.006208843676\t0.002600000000\t0.001400000000\t0.017881092423\t0.004900000000\t0.000100000000\t0.001100000000\t0.008600000000\t0.000222287328\t0.003400000000\t0.004300000000\t0.012800000000\t0.010223747696\t0.002491411310\t0.005300000000\t0.004389055645\t0.003700000000\t0.053706846809\t0.000000000000\t0.002694462454\t0.015799883085\t0.003901610981\t0.002813598808\t0.019610897429\t0.000383359117\t0.006787300000\t\t\t\t\t\t\t\r",
"T>A\tCTG\tC[T>A]G\t0.001487960630\t0.000230409809\t0.011508561865\t0.034200000000\t0.010507568970\t0.000800000000\t0.000700000000\t0.016356774533\t0.005100000000\t0.000300000000\t0.000600000000\t0.003100000000\t0.000286951828\t0.000200000000\t0.002700000000\t0.013000000000\t0.004467036232\t0.005523045713\t0.002100000000\t0.001127835842\t0.002500000000\t0.213220866558\t0.000173120400\t0.006911164721\t0.062860606381\t0.002390304180\t0.004396857083\t0.008833600032\t0.003556845215\t0.008262800000\t\t\t\t\t\t\t\r",
"T>A\tCTT\tC[T>A]T\t0.002179344412\t0.000574885587\t0.016456177696\t0.011500000000\t0.009784640746\t0.001100000000\t0.001800000000\t0.026247801917\t0.006900000000\t0.000900000000\t0.000400000000\t0.005400000000\t0.000000800446\t0.002000000000\t0.000700000000\t0.009600000000\t0.031244883388\t0.003313713273\t0.002200000000\t0.002385541321\t0.001000000000\t0.061827507476\t0.000457813099\t0.002915200165\t0.036622351112\t0.001635922667\t0.005430891633\t0.010942609820\t0.002232755492\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tCTA\tC[T>C]A\t0.004176674882\t0.000546619204\t0.007895012611\t0.005200000000\t0.014238629786\t0.006200000000\t0.001000000000\t0.005173842417\t0.015400000000\t0.000500000000\t0.000100000000\t0.031200000000\t0.000042751466\t0.000600000000\t0.003200000000\t0.014600000000\t0.007426999646\t0.000192169320\t0.006600000000\t0.014867696528\t0.021900000000\t0.004027346334\t0.000000000000\t0.001848876535\t0.009028014894\t0.037889377691\t0.004899079238\t0.020738003352\t0.000000000000\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tCTC\tC[T>C]C\t0.005252593331\t0.000392332573\t0.014431170362\t0.002600000000\t0.012489500440\t0.004000000000\t0.000800000000\t0.010936893768\t0.011800000000\t0.001900000000\t0.000400000000\t0.038500000000\t0.000043307542\t0.005300000000\t0.005000000000\t0.019400000000\t0.039657001207\t0.003050192581\t0.007000000000\t0.018356161025\t0.010000000000\t0.001352443929\t0.000000000000\t0.004453689857\t0.010771632420\t0.021740734953\t0.005724448467\t0.043283797246\t0.002774953681\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tCTG\tC[T>C]G\t0.007013225310\t0.000362024508\t0.008422972144\t0.010000000000\t0.018741631229\t0.027000000000\t0.001000000000\t0.005657773602\t0.015400000000\t0.001500000000\t0.000100000000\t0.041500000000\t0.000054272534\t0.001600000000\t0.012800000000\t0.019000000000\t0.028609696263\t0.002596464193\t0.004000000000\t0.074150562563\t0.021100000000\t0.002749666137\t0.000320704886\t0.000000000000\t0.013350661310\t0.047240103343\t0.005661478992\t0.037088261728\t0.006902414709\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tCTT\tC[T>C]T\t0.006713813119\t0.000560900126\t0.011932430465\t0.005400000000\t0.019054082944\t0.003300000000\t0.004500000000\t0.011120389322\t0.027500000000\t0.002000000000\t0.000900000000\t0.032400000000\t0.000000763944\t0.002700000000\t0.003500000000\t0.026300000000\t0.101869657701\t0.001310028744\t0.005300000000\t0.011001118253\t0.020400000000\t0.002363353537\t0.000000000000\t0.001597389007\t0.008961029966\t0.020740621431\t0.003900589062\t0.005052716018\t0.003041424895\t0.000000000000\t\t\t\t\t\t\t\r",
"T>G\tCTA\tC[T>G]A\t0.000302691186\t0.000113784603\t0.001974181146\t0.000000000000\t0.002041648941\t0.000100000000\t0.000000000000\t0.002424050552\t0.021600000000\t0.002200000000\t0.000000000000\t0.001300000000\t0.000015094208\t0.000000000000\t0.000000000000\t0.000200000000\t0.000091469787\t0.000061002548\t0.000000000000\t0.000023264084\t0.000000000000\t0.001120356515\t0.000000000000\t0.000000000000\t0.000000000000\t0.000095396260\t0.000000000000\t0.009589841941\t0.001116854814\t0.004623300000\t\t\t\t\t\t\t\r",
"T>G\tCTC\tC[T>G]C\t0.002098502440\t0.000022095087\t0.005824295545\t0.001300000000\t0.003478502118\t0.004000000000\t0.000800000000\t0.002096210616\t0.006400000000\t0.001800000000\t0.000100000000\t0.004900000000\t0.000147624840\t0.002000000000\t0.001900000000\t0.008200000000\t0.019814293551\t0.002015696913\t0.001100000000\t0.008476494525\t0.000500000000\t0.001064583571\t0.000190144342\t0.002185437348\t0.002886454337\t0.003919455100\t0.000381244984\t0.033913223722\t0.000000000000\t0.006000400000\t\t\t\t\t\t\t\r",
"T>G\tCTG\tC[T>G]G\t0.001599548500\t0.000228245948\t0.010464654525\t0.004600000000\t0.007147456226\t0.005000000000\t0.000900000000\t0.006604046294\t0.012600000000\t0.003700000000\t0.001100000000\t0.004500000000\t0.000045987300\t0.000500000000\t0.000700000000\t0.006700000000\t0.013449837633\t0.005311967254\t0.001300000000\t0.015383776115\t0.000400000000\t0.004188568126\t0.000000000000\t0.000113909674\t0.011769492499\t0.005497357314\t0.000403257236\t0.018516481606\t0.002169693346\t0.007377500000\t\t\t\t\t\t\t\r",
"T>G\tCTT\tC[T>G]T\t0.002758537619\t0.000067111340\t0.008724387302\t0.001200000000\t0.011486811509\t0.008600000000\t0.001300000000\t0.004866713859\t0.050900000000\t0.018200000000\t0.000900000000\t0.006300000000\t0.000463714722\t0.006300000000\t0.004500000000\t0.018600000000\t0.261456614079\t0.002341629178\t0.001900000000\t0.002185394659\t0.003200000000\t0.001629409573\t0.000210834980\t0.002533718333\t0.008550807482\t0.006788718691\t0.001439096069\t0.118967102954\t0.001143663251\t0.009148100000\t\t\t\t\t\t\t\r",
"C>A\tGCA\tG[C>A]A\t0.008232603989\t0.000352013353\t0.009696539676\t0.037600000000\t0.011892168979\t0.002400000000\t0.000300000000\t0.023682328869\t0.011800000000\t0.001400000000\t0.000400000000\t0.006200000000\t0.000132109643\t0.001800000000\t0.002400000000\t0.009600000000\t0.006535404875\t0.128724158080\t0.003200000000\t0.001116123789\t0.000100000000\t0.001298370161\t0.000349907495\t0.054676448679\t0.013465295053\t0.001375311846\t0.010755871858\t0.002136629077\t0.078046610108\t0.008853000000\t\t\t\t\t\t\t\r",
"C>A\tGCC\tG[C>A]C\t0.005758021410\t0.000133816879\t0.010843341116\t0.039900000000\t0.009247857495\t0.005800000000\t0.000100000000\t0.015821896445\t0.009200000000\t0.002200000000\t0.001000000000\t0.005600000000\t0.000754244011\t0.011400000000\t0.009900000000\t0.009400000000\t0.001293803953\t0.016092878663\t0.003100000000\t0.003271428605\t0.003800000000\t0.001809622152\t0.000568337772\t0.109945988997\t0.006468298058\t0.001961639088\t0.012442335117\t0.000429208379\t0.073460914669\t0.009344900000\t\t\t\t\t\t\t\r",
"C>A\tGCG\tG[C>A]G\t0.000616335232\t0.000178441660\t0.000929140538\t0.022700000000\t0.002809188466\t0.002100000000\t0.000000000000\t0.000850913754\t0.000000000000\t0.000200000000\t0.000000000000\t0.001500000000\t0.000096820775\t0.001100000000\t0.001300000000\t0.003600000000\t0.001283663378\t0.009271416762\t0.000000000000\t0.003751612508\t0.000300000000\t0.000333838055\t0.000149417834\t0.046524869010\t0.000658103089\t0.000013229664\t0.001629628347\t0.000000000000\t0.017383296915\t0.000885300000\t\t\t\t\t\t\t\r",
"C>A\tGCT\tG[C>A]T\t0.004459080311\t0.000122832046\t0.012215382639\t0.025800000000\t0.010301267456\t0.008700000000\t0.000100000000\t0.021061235802\t0.008500000000\t0.008800000000\t0.000600000000\t0.002700000000\t0.000551103672\t0.073600000000\t0.030900000000\t0.006300000000\t0.003103766672\t0.072017070126\t0.007200000000\t0.012937194450\t0.002300000000\t0.000519148161\t0.000000000000\t0.050428709482\t0.010473080347\t0.001934996293\t0.007412355152\t0.001318507256\t0.059602472118\t0.008164500000\t\t\t\t\t\t\t\r",
"C>G\tGCA\tG[C>G]A\t0.000602122711\t0.000044838514\t0.011916844569\t0.007000000000\t0.005559423089\t0.000000000000\t0.000000000000\t0.003724567946\t0.001100000000\t0.000000000000\t0.000000000000\t0.001200000000\t0.001349419584\t0.000200000000\t0.002200000000\t0.003100000000\t0.001296324191\t0.000923879703\t0.002000000000\t0.001339769321\t0.000400000000\t0.000000000000\t0.000000000000\t0.004945060708\t0.004169655524\t0.001320607475\t0.001418166387\t0.000081591411\t0.000265759858\t0.007377500000\t\t\t\t\t\t\t\r",
"C>G\tGCC\tG[C>G]C\t0.002393352209\t0.000014520103\t0.009823653397\t0.009100000000\t0.005388801221\t0.003000000000\t0.000400000000\t0.003046596953\t0.002900000000\t0.000200000000\t0.000200000000\t0.001100000000\t0.000788807743\t0.000100000000\t0.000500000000\t0.003200000000\t0.000260760911\t0.000648647922\t0.000000000000\t0.001811151475\t0.004200000000\t0.000000000000\t0.000137501074\t0.009369875774\t0.002392804944\t0.001845501071\t0.001346716782\t0.000105219031\t0.000945012884\t0.006590600000\t\t\t\t\t\t\t\r",
"C>G\tGCG\tG[C>G]G\t0.000000248534\t0.000040665889\t0.001671054250\t0.006200000000\t0.001100589633\t0.000000000000\t0.000000000000\t0.000321212793\t0.000000000000\t0.000000000000\t0.000000000000\t0.000000000000\t0.000081791214\t0.000000000000\t0.000600000000\t0.000000000000\t0.000004377552\t0.000542839968\t0.000000000000\t0.000000637538\t0.000000000000\t0.000214695851\t0.000000000000\t0.005625726540\t0.000418193916\t0.000205060777\t0.000000000000\t0.000000000000\t0.000000000000\t0.001573900000\t\t\t\t\t\t\t\r",
"C>G\tGCT\tG[C>G]T\t0.000890080731\t0.000268429459\t0.017914334306\t0.006000000000\t0.006041247837\t0.001700000000\t0.000000000000\t0.005775663598\t0.004400000000\t0.000400000000\t0.000000000000\t0.000900000000\t0.002680123244\t0.000300000000\t0.001800000000\t0.002900000000\t0.001820392786\t0.001376724888\t0.000000000000\t0.001238264182\t0.000400000000\t0.000538438803\t0.000000000000\t0.003055770064\t0.001994012217\t0.001225532856\t0.004560685974\t0.000159679354\t0.004796744744\t0.000000000000\t\t\t\t\t\t\t\r",
"C>T\tGCA\tG[C>T]A\t0.024943814154\t0.006390807876\t0.016127273332\t0.008700000000\t0.020038258645\t0.065300000000\t0.000000000000\t0.002261262686\t0.006200000000\t0.000000000000\t0.026100000000\t0.012400000000\t0.002934137577\t0.055700000000\t0.047600000000\t0.012500000000\t0.002068754725\t0.002703068727\t0.037400000000\t0.056793947985\t0.016300000000\t0.001915370854\t0.053282058965\t0.023679051295\t0.010038112013\t0.009606515027\t0.014215313743\t0.003384240869\t0.011638793032\t0.048593400000\t\t\t\t\t\t\t\r",
"C>T\tGCC\tG[C>T]C\t0.027161494035\t0.001995816784\t0.008208632437\t0.008000000000\t0.018022312238\t0.077300000000\t0.021000000000\t0.001616649187\t0.006900000000\t0.013400000000\t0.097500000000\t0.009200000000\t0.002515171146\t0.061800000000\t0.134500000000\t0.009200000000\t0.000325483399\t0.004199307536\t0.061400000000\t0.054429857270\t0.039200000000\t0.003229133547\t0.162918343743\t0.015919201840\t0.006953875155\t0.019507426720\t0.020143646182\t0.001687414029\t0.003781751661\t0.049478700000\t\t\t\t\t\t\t\r",
"C>T\tGCG\tG[C>T]G\t0.103570762296\t0.000303020977\t0.001212922866\t0.002300000000\t0.013193817648\t0.133900000000\t0.000200000000\t0.011606711600\t0.008800000000\t0.027000000000\t0.009000000000\t0.002300000000\t0.003205264550\t0.097800000000\t0.182900000000\t0.000900000000\t0.007511101358\t0.011897360825\t0.027800000000\t0.004592688609\t0.001900000000\t0.003300598879\t0.013046494744\t0.012648077606\t0.010468363027\t0.022502717383\t0.009368468496\t0.002189308960\t0.019684388045\t0.015738700000\t\t\t\t\t\t\t\r",
"C>T\tGCT\tG[C>T]T\t0.017689854381\t0.003265818188\t0.010611649208\t0.008200000000\t0.019502915335\t0.052400000000\t0.016100000000\t0.006284928996\t0.010100000000\t0.015200000000\t0.052200000000\t0.011500000000\t0.001687730545\t0.055500000000\t0.080000000000\t0.011600000000\t0.000925697167\t0.001913380905\t0.091700000000\t0.049580863753\t0.016600000000\t0.001114059570\t0.107590399754\t0.018129778378\t0.009823820735\t0.017307472184\t0.007589625950\t0.001043993631\t0.005761336823\t0.018886500000\t\t\t\t\t\t\t\r",
"T>A\tGTA\tG[T>A]A\t0.000689289439\t0.000114994210\t0.004434782559\t0.006900000000\t0.006740387717\t0.000000000000\t0.000000000000\t0.008139178608\t0.003000000000\t0.000000000000\t0.000300000000\t0.001500000000\t0.000137960405\t0.000000000000\t0.000000000000\t0.003000000000\t0.000008545689\t0.001875434081\t0.002600000000\t0.000163129836\t0.000000000000\t0.053709145693\t0.000000000000\t0.002225127336\t0.036217800837\t0.002242886513\t0.109851871980\t0.000551460783\t0.002930098053\t0.003738000000\t\t\t\t\t\t\t\r",
"T>A\tGTC\tG[T>A]C\t0.000552409528\t0.000293862060\t0.005615432411\t0.005200000000\t0.004021596480\t0.002800000000\t0.000000000000\t0.007910080318\t0.002900000000\t0.000100000000\t0.000300000000\t0.003900000000\t0.000252867405\t0.002500000000\t0.003800000000\t0.004000000000\t0.002573466959\t0.001510309757\t0.000000000000\t0.003799179503\t0.014200000000\t0.009489396694\t0.000183203812\t0.004853149157\t0.005847109647\t0.005207287095\t0.000023455678\t0.003163987597\t0.000000000000\t0.003344500000\t\t\t\t\t\t\t\r",
"T>A\tGTG\tG[T>A]G\t0.001200228847\t0.000088721344\t0.008070136472\t0.013300000000\t0.006814352288\t0.000600000000\t0.000600000000\t0.009057263056\t0.002800000000\t0.000000000000\t0.000400000000\t0.001900000000\t0.000153517461\t0.000000000000\t0.002600000000\t0.005100000000\t0.003538822593\t0.001505368776\t0.001600000000\t0.000828333700\t0.003100000000\t0.050293902428\t0.000000000000\t0.000195758628\t0.038241179315\t0.001358121778\t0.001565937159\t0.002152847013\t0.000533632401\t0.005016700000\t\t\t\t\t\t\t\r",
"T>A\tGTT\tG[T>A]T\t0.002107136837\t0.000215755466\t0.008679121744\t0.004500000000\t0.005101032909\t0.002100000000\t0.001000000000\t0.017812238647\t0.003600000000\t0.000400000000\t0.000100000000\t0.003100000000\t0.000122745552\t0.000900000000\t0.004300000000\t0.003300000000\t0.003879009961\t0.003611025202\t0.005200000000\t0.001900746587\t0.004100000000\t0.010601457233\t0.000000000000\t0.000000000000\t0.020167954419\t0.002513112942\t0.000000000000\t0.002007733244\t0.000000000000\t0.006393900000\t\t\t\t\t\t\t\r",
"T>C\tGTA\tG[T>C]A\t0.011247835116\t0.000004673852\t0.006850055574\t0.006100000000\t0.016294982792\t0.012200000000\t0.000200000000\t0.005720778549\t0.009100000000\t0.006400000000\t0.001000000000\t0.057300000000\t0.001054081222\t0.007800000000\t0.019500000000\t0.016100000000\t0.000011401335\t0.000529126468\t0.003900000000\t0.002469317074\t0.142600000000\t0.004217454113\t0.000251168628\t0.001269749840\t0.017489823879\t0.098052824719\t0.004777380354\t0.003655048127\t0.007090803763\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tGTC\tG[T>C]C\t0.006999724257\t0.000186065927\t0.006260749014\t0.001600000000\t0.009574560593\t0.005900000000\t0.000100000000\t0.005181051502\t0.009700000000\t0.010100000000\t0.001600000000\t0.027400000000\t0.000529899747\t0.012700000000\t0.049900000000\t0.009900000000\t0.009657713318\t0.002279130829\t0.003000000000\t0.001565632906\t0.066500000000\t0.000000000000\t0.000538218489\t0.006710339996\t0.004045612446\t0.040226363963\t0.001964954330\t0.007071175327\t0.007111327534\t0.006590600000\t\t\t\t\t\t\t\r",
"T>C\tGTG\tG[T>C]G\t0.004977592617\t0.000000495044\t0.006098762993\t0.004200000000\t0.014103929348\t0.011500000000\t0.000300000000\t0.005040707471\t0.009400000000\t0.005400000000\t0.001900000000\t0.041000000000\t0.000630095718\t0.008200000000\t0.015400000000\t0.014800000000\t0.004654129131\t0.005871056581\t0.006000000000\t0.027341755021\t0.088400000000\t0.001474884534\t0.000000000000\t0.000277684976\t0.013402246274\t0.044620700983\t0.001229167749\t0.008232238440\t0.005643063554\t0.007869400000\t\t\t\t\t\t\t\r",
"T>C\tGTT\tG[T>C]T\t0.010667406133\t0.000578598104\t0.007509343859\t0.002400000000\t0.015666874692\t0.004200000000\t0.008100000000\t0.008488220258\t0.012600000000\t0.012400000000\t0.002300000000\t0.040500000000\t0.000676187888\t0.008600000000\t0.009400000000\t0.014200000000\t0.008149234601\t0.004771565739\t0.009900000000\t0.001498264796\t0.100300000000\t0.001561042738\t0.000995769681\t0.004238477669\t0.014159693736\t0.055528068280\t0.003747225013\t0.000407884510\t0.011237674345\t0.008459600000\t\t\t\t\t\t\t\r",
"T>G\tGTA\tG[T>G]A\t0.000099045003\t0.000095552392\t0.004144488018\t0.000000000000\t0.001627664547\t0.000000000000\t0.000000000000\t0.000974578665\t0.007200000000\t0.000000000000\t0.000000000000\t0.000000000000\t0.000018489857\t0.000000000000\t0.000000000000\t0.000000000000\t0.000039193821\t0.001341532464\t0.000000000000\t0.000003579149\t0.000000000000\t0.000000000000\t0.000000000000\t0.000000000000\t0.005546226924\t0.000037393232\t0.000000000000\t0.000000000000\t0.000000000000\t0.003246100000\t\t\t\t\t\t\t\r",
"T>G\tGTC\tG[T>G]C\t0.000202365646\t0.000047002381\t0.004501985313\t0.000000000000\t0.000327734937\t0.001600000000\t0.000000000000\t0.000524821555\t0.000600000000\t0.002000000000\t0.000000000000\t0.000400000000\t0.000093373513\t0.001900000000\t0.002200000000\t0.003200000000\t0.009078461515\t0.000000920431\t0.000000000000\t0.002830575073\t0.000600000000\t0.000000000000\t0.000000000000\t0.001073632302\t0.000000000000\t0.002460704995\t0.000050790565\t0.000000000000\t0.000000000000\t0.001869000000\t\t\t\t\t\t\t\r",
"T>G\tGTG\tG[T>G]G\t0.001188353185\t0.000109925703\t0.016391452608\t0.001800000000\t0.005948879802\t0.001000000000\t0.001700000000\t0.006087753467\t0.005000000000\t0.000900000000\t0.001000000000\t0.001100000000\t0.000010579194\t0.001200000000\t0.003700000000\t0.000800000000\t0.004782775461\t0.009694999981\t0.004300000000\t0.002733219405\t0.000400000000\t0.001877589181\t0.000187132414\t0.002924696996\t0.008685244444\t0.000817201563\t0.004742918550\t0.004217808299\t0.003640593813\t0.003344500000\t\t\t\t\t\t\t\r",
"T>G\tGTT\tG[T>G]T\t0.000800723342\t0.000086477180\t0.007067236643\t0.000200000000\t0.003307466591\t0.003500000000\t0.000900000000\t0.005427433785\t0.018500000000\t0.003000000000\t0.000000000000\t0.003200000000\t0.000053741207\t0.003800000000\t0.019000000000\t0.004400000000\t0.063497756582\t0.002890553519\t0.002500000000\t0.003558558565\t0.000800000000\t0.000000000000\t0.000000000000\t0.000825098263\t0.002768571683\t0.007833561271\t0.002298476030\t0.031648997321\t0.006181023776\t0.005606900000\t\t\t\t\t\t\t\r",
"C>A\tTCA\tT[C>A]A\t0.012250063666\t0.015127427519\t0.011653224909\t0.033000000000\t0.014774016248\t0.001700000000\t0.001000000000\t0.027032383039\t0.022200000000\t0.037400000000\t0.000900000000\t0.006600000000\t0.048166047937\t0.000500000000\t0.005700000000\t0.012200000000\t0.001824558257\t0.067080643529\t0.005100000000\t0.000688299692\t0.000100000000\t0.004051634552\t0.002071289985\t0.031451263182\t0.006938146585\t0.002680062424\t0.007079284522\t0.000775902768\t0.026894406511\t0.000000000000\t\t\t\t\t\t\t\r",
"C>A\tTCC\tT[C>A]C\t0.011162229329\t0.006532492454\t0.016606775011\t0.053800000000\t0.012043464964\t0.002900000000\t0.002000000000\t0.018089773290\t0.004300000000\t0.010300000000\t0.002500000000\t0.009900000000\t0.017329557089\t0.016600000000\t0.006200000000\t0.014500000000\t0.001286518338\t0.041869009693\t0.006300000000\t0.004058229060\t0.000500000000\t0.002739770937\t0.000696410201\t0.062538293699\t0.010246607400\t0.002031643866\t0.006268800628\t0.000829292467\t0.031608136996\t0.000000000000\t\t\t\t\t\t\t\r",
"C>A\tTCG\tT[C>A]G\t0.002275495686\t0.001656455398\t0.001357239439\t0.010400000000\t0.003902362388\t0.001100000000\t0.000200000000\t0.001694875197\t0.000000000000\t0.003100000000\t0.000100000000\t0.001900000000\t0.002293157306\t0.000000000000\t0.002500000000\t0.004000000000\t0.002172941968\t0.015162379290\t0.001800000000\t0.001001102231\t0.000000000000\t0.000513151070\t0.000099393633\t0.012802590728\t0.000000000000\t0.000265071700\t0.001446403415\t0.000000000000\t0.012500468152\t0.001672200000\t\t\t\t\t\t\t\r",
"C>A\tTCT\tT[C>A]T\t0.015259102491\t0.012394610714\t0.016328075975\t0.037000000000\t0.018243395705\t0.005800000000\t0.001300000000\t0.038141330868\t0.032200000000\t0.308300000000\t0.000400000000\t0.004900000000\t0.018634581512\t0.051600000000\t0.014500000000\t0.015700000000\t0.005012316394\t0.117877846666\t0.008700000000\t0.012229104507\t0.000400000000\t0.003605650899\t0.000207430191\t0.030297679613\t0.024625787226\t0.003017187457\t0.009806277750\t0.001998073667\t0.041207176390\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tTCA\tT[C>G]A\t0.001874853199\t0.037241853149\t0.016041405438\t0.003200000000\t0.002681056961\t0.000000000000\t0.000200000000\t0.004117878837\t0.003300000000\t0.000000000000\t0.000100000000\t0.002400000000\t0.280234946548\t0.000000000000\t0.000900000000\t0.005000000000\t0.000392684657\t0.001802969051\t0.000500000000\t0.000024298469\t0.000200000000\t0.000225490615\t0.000230608944\t0.010220857436\t0.009798724567\t0.004201896751\t0.002870433721\t0.000521370368\t0.000820444889\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tTCC\tT[C>G]C\t0.002067418791\t0.000019413411\t0.020149920428\t0.010500000000\t0.007924048307\t0.000200000000\t0.001000000000\t0.003799649592\t0.002500000000\t0.000000000000\t0.001100000000\t0.002700000000\t0.063885286587\t0.000000000000\t0.002600000000\t0.006400000000\t0.000655602418\t0.007863006794\t0.001100000000\t0.000151998550\t0.000300000000\t0.000628695020\t0.000165787009\t0.007578365252\t0.003266507150\t0.002807761534\t0.003107967569\t0.000667325890\t0.000444119287\t0.000000000000\t\t\t\t\t\t\t\r",
"C>G\tTCG\tT[C>G]G\t0.000304897004\t0.001625465457\t0.002527910936\t0.003100000000\t0.001319075649\t0.000000000000\t0.000300000000\t0.000025653018\t0.000000000000\t0.000000000000\t0.000300000000\t0.000000000000\t0.009528486008\t0.000000000000\t0.000000000000\t0.000000000000\t0.000013641541\t0.001513223334\t0.000000000000\t0.000000000000\t0.000000000000\t0.000395807994\t0.000000000000\t0.001592714217\t0.002057436753\t0.000000006197\t0.000012810409\t0.000000000000\t0.000143297155\t0.001967300000\t\t\t\t\t\t\t\r",
"C>G\tTCT\tT[C>G]T\t0.003151574460\t0.066879899684\t0.032673640762\t0.005000000000\t0.006644595708\t0.000100000000\t0.000700000000\t0.006247553738\t0.004900000000\t0.000100000000\t0.000800000000\t0.001500000000\t0.330225481745\t0.000100000000\t0.001500000000\t0.016900000000\t0.002852380395\t0.003567102065\t0.010500000000\t0.000683059976\t0.000300000000\t0.000257375147\t0.000438955809\t0.006100727190\t0.016400916850\t0.008019382986\t0.002130587503\t0.001245779535\t0.001369869047\t0.000000000000\t\t\t\t\t\t\t\r",
"C>T\tTCA\tT[C>T]A\t0.014492099634\t0.419941399594\t0.008880209061\t0.003500000000\t0.010997560324\t0.007400000000\t0.120200000000\t0.007180668540\t0.005000000000\t0.003700000000\t0.030200000000\t0.013800000000\t0.113842093163\t0.009200000000\t0.011200000000\t0.017200000000\t0.016100784572\t0.014923391701\t0.021800000000\t0.010710322070\t0.006400000000\t0.002016721688\t0.014558613696\t0.026571894346\t0.011077520800\t0.011303288768\t0.018757776459\t0.007622257672\t0.012076583098\t0.084989200000\t\t\t\t\t\t\t\r",
"C>T\tTCC\tT[C>T]C\t0.017680775357\t0.081972496144\t0.013529572798\t0.007000000000\t0.020644964803\t0.006700000000\t0.288700000000\t0.004061032938\t0.008400000000\t0.021100000000\t0.158900000000\t0.009500000000\t0.015024839001\t0.028300000000\t0.015600000000\t0.013100000000\t0.016779984064\t0.009962744310\t0.033000000000\t0.008175509770\t0.008600000000\t0.001990834246\t0.043510446939\t0.012055691082\t0.006448787540\t0.010807526975\t0.007989184406\t0.005116619241\t0.000000000000\t0.090301000000\t\t\t\t\t\t\t\r",
"C>T\tTCG\tT[C>T]G\t0.076002221712\t0.047720186368\t0.001705408858\t0.001100000000\t0.007534489574\t0.039100000000\t0.099200000000\t0.005535121520\t0.004700000000\t0.214100000000\t0.008000000000\t0.001000000000\t0.006102097956\t0.009400000000\t0.010400000000\t0.003600000000\t0.016272095596\t0.011549715126\t0.009000000000\t0.023163899202\t0.012600000000\t0.001498473091\t0.002752640544\t0.005363359680\t0.014083416181\t0.010364329328\t0.004609130971\t0.001918028110\t0.015830196802\t0.015148500000\t\t\t\t\t\t\t\r",
"C>T\tTCT\tT[C>T]T\t0.013761704021\t0.228674918306\t0.010304416928\t0.007700000000\t0.011787461775\t0.004700000000\t0.084400000000\t0.012209070497\t0.009600000000\t0.039200000000\t0.095400000000\t0.009000000000\t0.028140661233\t0.022900000000\t0.007600000000\t0.019300000000\t0.019827417550\t0.005398013994\t0.026700000000\t0.009540087238\t0.006400000000\t0.001182226502\t0.024328785234\t0.005532499180\t0.012418585885\t0.007249145360\t0.011880932482\t0.006193158626\t0.013462137300\t0.045544000000\t\t\t\t\t\t\t\r",
"T>A\tTTA\tT[T>A]A\t0.005600155423\t0.000080752486\t0.007132681354\t0.004500000000\t0.009205637343\t0.000200000000\t0.001000000000\t0.015139996236\t0.021500000000\t0.000000000000\t0.000300000000\t0.005700000000\t0.000403261316\t0.000000000000\t0.000800000000\t0.008500000000\t0.000698812311\t0.004663278043\t0.003900000000\t0.000924547968\t0.002700000000\t0.040810502825\t0.000000000000\t0.000000000000\t0.021830329049\t0.000628145256\t0.052813887494\t0.000203237563\t0.001191207434\t0.007672600000\t\t\t\t\t\t\t\r",
"T>A\tTTC\tT[T>A]C\t0.001999079260\t0.000004802898\t0.009102797827\t0.004600000000\t0.006835344247\t0.000800000000\t0.001500000000\t0.012093554606\t0.002700000000\t0.000800000000\t0.000000000000\t0.013200000000\t0.000771813274\t0.001600000000\t0.001900000000\t0.008700000000\t0.002579358604\t0.002578155543\t0.004800000000\t0.000708206814\t0.004500000000\t0.025570395801\t0.000000000000\t0.007994129155\t0.003405686728\t0.005073753861\t0.000047091785\t0.010180964868\t0.001682664015\t0.006492200000\t\t\t\t\t\t\t\r",
"T>A\tTTG\tT[T>A]G\t0.001090065693\t0.000066605043\t0.006565863041\t0.008200000000\t0.007144208769\t0.000000000000\t0.001000000000\t0.008362546590\t0.014000000000\t0.000000000000\t0.000000000000\t0.003700000000\t0.000245194218\t0.000000000000\t0.000600000000\t0.006300000000\t0.000021133639\t0.002016738361\t0.005200000000\t0.000097003127\t0.000000000000\t0.069845218085\t0.000231525618\t0.011202143591\t0.010191520401\t0.000020138391\t0.000000000000\t0.003250427227\t0.002438756000\t0.003639600000\t\t\t\t\t\t\t\r",
"T>A\tTTT\tT[T>A]T\t0.003981022761\t0.000276315891\t0.014712135403\t0.004500000000\t0.010240850413\t0.000700000000\t0.005000000000\t0.027653296254\t0.013900000000\t0.002800000000\t0.000800000000\t0.009400000000\t0.000160345975\t0.001200000000\t0.000300000000\t0.008000000000\t0.003883130589\t0.002818679447\t0.008500000000\t0.002313489060\t0.000800000000\t0.023579061797\t0.000000000000\t0.000270578768\t0.014527539313\t0.001235538470\t0.001929770813\t0.016492812256\t0.000000000000\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tTTA\tT[T>C]A\t0.008073616351\t0.000101932352\t0.009115495077\t0.002800000000\t0.017059640976\t0.005900000000\t0.002300000000\t0.004901072379\t0.015600000000\t0.003000000000\t0.000200000000\t0.050000000000\t0.000784056118\t0.001800000000\t0.006800000000\t0.024600000000\t0.001708586711\t0.000302303565\t0.005200000000\t0.005734229562\t0.038900000000\t0.004803669740\t0.000000000000\t0.001682291048\t0.014515050090\t0.028789680542\t0.001721015277\t0.009929552068\t0.011498665964\t0.000000000000\t\t\t\t\t\t\t\r",
"T>C\tTTC\tT[T>C]C\t0.004857381178\t0.000470343659\t0.010953790932\t0.001600000000\t0.014195709147\t0.003500000000\t0.001800000000\t0.006081011299\t0.013700000000\t0.009700000000\t0.000900000000\t0.034200000000\t0.000132974888\t0.005600000000\t0.009900000000\t0.018300000000\t0.011074152072\t0.006260791115\t0.002000000000\t0.011234254668\t0.031300000000\t0.000912157512\t0.000230608944\t0.002344527656\t0.007406083457\t0.036639492442\t0.003489212553\t0.029010563206\t0.003584215038\t0.009148100000\t\t\t\t\t\t\t\r",
"T>C\tTTG\tT[T>C]G\t0.008325454207\t0.000192354203\t0.006113033219\t0.003600000000\t0.012596501722\t0.010600000000\t0.001900000000\t0.001712484699\t0.009800000000\t0.006500000000\t0.000700000000\t0.032000000000\t0.000037402404\t0.003400000000\t0.003900000000\t0.016500000000\t0.005932002763\t0.001541265329\t0.005700000000\t0.023767611345\t0.024900000000\t0.001781735680\t0.000239513775\t0.000481274164\t0.004892531217\t0.019143694280\t0.001152846581\t0.015683218557\t0.001694296670\t0.006000400000\t\t\t\t\t\t\t\r",
"T>C\tTTT\tT[T>C]T\t0.006257105605\t0.000585330275\t0.010774336259\t0.002200000000\t0.017375139145\t0.002900000000\t0.002400000000\t0.010003441830\t0.030900000000\t0.009900000000\t0.000500000000\t0.037100000000\t0.000179702846\t0.002600000000\t0.004700000000\t0.017400000000\t0.007208218396\t0.001357797089\t0.003800000000\t0.005395186301\t0.027400000000\t0.000000000000\t0.000000000000\t0.000358148002\t0.017014780759\t0.023071688831\t0.001527957003\t0.017291563618\t0.003799705772\t0.000000000000\t\t\t\t\t\t\t\r",
"T>G\tTTA\tT[T>G]A\t0.001397553749\t0.000071736950\t0.005427184241\t0.000000000000\t0.005202874420\t0.000900000000\t0.000000000000\t0.001743221362\t0.050200000000\t0.005000000000\t0.000000000000\t0.001900000000\t0.000546581847\t0.000000000000\t0.000000000000\t0.006800000000\t0.000133410642\t0.000036318404\t0.003200000000\t0.000000269147\t0.000000000000\t0.001651698761\t0.000000000000\t0.000000000000\t0.002081479927\t0.000008534935\t0.001189022454\t0.009389658706\t0.000000000000\t0.008656300000\t\t\t\t\t\t\t\r",
"T>G\tTTC\tT[T>G]C\t0.001291736985\t0.000014281456\t0.006160250423\t0.000300000000\t0.005131607914\t0.001900000000\t0.001000000000\t0.002549838329\t0.008100000000\t0.009200000000\t0.000000000000\t0.002700000000\t0.000235347085\t0.001500000000\t0.000400000000\t0.006900000000\t0.009613345245\t0.003233837990\t0.001800000000\t0.000377234437\t0.001800000000\t0.000000000000\t0.000000000000\t0.000000000000\t0.000578978775\t0.002718509845\t0.000280295354\t0.030117076984\t0.000000000000\t0.004328200000\t\t\t\t\t\t\t\r",
"T>G\tTTG\tT[T>G]G\t0.002031076880\t0.000206615168\t0.011076526261\t0.003000000000\t0.006055254100\t0.001100000000\t0.001000000000\t0.006030395185\t0.008800000000\t0.002200000000\t0.000300000000\t0.001100000000\t0.000000479022\t0.000200000000\t0.000900000000\t0.004900000000\t0.004522462318\t0.000754601773\t0.001100000000\t0.000515421599\t0.000300000000\t0.002572751958\t0.000247501933\t0.001360504850\t0.009429195885\t0.001369161173\t0.002353055589\t0.012698750811\t0.000353695951\t0.008262800000\t\t\t\t\t\t\t\r",
"T>G\tTTT\tT[T>G]T\t0.004030128160\t0.000023598204\t0.013000984209\t0.001100000000\t0.013369935834\t0.007200000000\t0.001400000000\t0.007223998893\t0.054500000000\t0.063300000000\t0.000300000000\t0.003200000000\t0.000670588327\t0.002500000000\t0.003300000000\t0.016300000000\t0.058040407760\t0.002126441532\t0.001300000000\t0.000615656691\t0.000300000000\t0.000000000000\t0.000000000000\t0.000069515778\t0.007869671577\t0.002568076726\t0.000139561285\t0.233659783276\t0.006104834136\t0.000000000000\t\t\t\t\t\t\t"
)
# COSMIC_v3 
COSMIC_v3 <- c(
 "Type\tSubType\tSBS1\tSBS2\tSBS3\tSBS4\tSBS5\tSBS6\tSBS7a\tSBS7b\tSBS7c\tSBS7d\tSBS8\tSBS9\tSBS10a\tSBS10b\tSBS11\tSBS12\tSBS13\tSBS14\tSBS15\tSBS16\tSBS17a\tSBS17b\tSBS18\tSBS19\tSBS20\tSBS21\tSBS22\tSBS23\tSBS24\tSBS25\tSBS26\tSBS27\tSBS28\tSBS29\tSBS30\tSBS31\tSBS32\tSBS33\tSBS34\tSBS35\tSBS36\tSBS37\tSBS38\tSBS39\tSBS40\tSBS41\tSBS42\tSBS43\tSBS44\tSBS45\tSBS46\tSBS47\tSBS48\tSBS49\tSBS50\tSBS51\tSBS52\tSBS53\tSBS54\tSBS55\tSBS56\tSBS57\tSBS58\tSBS59\tSBS60\tSBS84\tSBS85\r",
 "C>A\tACA\t8.86E-04\t5.80E-07\t2.08E-02\t4.22E-02\t1.20E-02\t4.25E-04\t6.70E-05\t2.33E-03\t4.83E-03\t4.04E-05\t4.41E-02\t5.58E-04\t2.19E-03\t1.82E-04\t1.46E-04\t4.52E-03\t1.82E-03\t1.12E-03\t9.44E-04\t1.60E-02\t2.07E-03\t6.08E-04\t5.15E-02\t1.27E-03\t6.19E-04\t1.57E-04\t6.01E-03\t8.35E-04\t3.64E-02\t9.90E-03\t8.73E-04\t5.21E-03\t7.84E-04\t6.32E-02\t1.80E-03\t9.54E-03\t2.23E-02\t3.11E-03\t4.87E-03\t8.83E-03\t2.52E-02\t3.95E-03\t1.28E-02\t1.17E-02\t2.82E-02\t2.11E-03\t1.16E-03\t2.95E-02\t7.68E-18\t9.11E-03\t4.40E-03\t6.78E-02\t8.55E-04\t2.51E-02\t1.19E-01\t1.41E-01\t1.52E-02\t5.38E-03\t2.16E-03\t5.88E-03\t1.26E-02\t1.23E-02\t5.89E-02\t3.59E-03\t6.15E-03\t0.003471994832\t0.006080257390\r",
 "C>A\tACC\t2.28E-03\t1.48E-04\t1.65E-02\t3.33E-02\t9.44E-03\t5.24E-04\t1.79E-04\t4.61E-04\t1.15E-03\t7.65E-04\t4.78E-02\t4.09E-03\t1.77E-03\t6.54E-03\t5.52E-04\t1.13E-03\t7.21E-04\t1.31E-02\t4.97E-04\t2.90E-03\t9.18E-04\t1.29E-04\t1.58E-02\t6.41E-04\t1.39E-03\t2.36E-03\t9.54E-05\t3.99E-04\t2.65E-02\t7.00E-03\t5.28E-04\t4.74E-03\t2.53E-03\t5.12E-02\t5.06E-04\t1.85E-02\t1.88E-02\t2.23E-03\t6.96E-03\t4.62E-02\t8.32E-03\t1.45E-03\t1.01E-02\t7.15E-03\t1.34E-02\t1.22E-03\t2.06E-02\t5.90E-03\t1.50E-04\t2.85E-03\t4.70E-03\t2.97E-02\t8.41E-04\t7.03E-03\t1.27E-01\t1.69E-03\t6.54E-03\t1.96E-03\t7.96E-04\t2.05E-03\t1.57E-02\t1.43E-03\t6.75E-03\t2.37E-03\t7.79E-04\t0.005007308997\t0.000879936873\r",
 "C>A\tACG\t1.77E-04\t5.23E-05\t1.75E-03\t1.56E-02\t1.85E-03\t5.20E-05\t7.12E-05\t1.86E-04\t3.77E-04\t2.50E-04\t4.62E-03\t4.26E-04\t1.50E-04\t5.35E-05\t9.41E-05\t5.40E-04\t2.64E-04\t4.13E-04\t4.61E-05\t1.02E-03\t4.76E-05\t5.82E-05\t2.43E-03\t2.46E-04\t2.18E-05\t2.94E-04\t7.65E-04\t9.85E-08\t1.48E-02\t1.45E-03\t1.14E-04\t7.83E-04\t3.53E-04\t1.92E-02\t9.13E-05\t1.66E-03\t4.46E-03\t4.14E-04\t5.20E-05\t1.39E-03\t2.24E-03\t1.06E-03\t1.90E-03\t2.67E-03\t2.92E-03\t6.14E-05\t3.34E-05\t6.26E-04\t9.16E-07\t1.66E-03\t3.00E-04\t1.38E-03\t2.58E-04\t7.11E-03\t5.77E-03\t4.80E-03\t4.14E-03\t3.81E-02\t1.64E-03\t4.49E-05\t2.06E-04\t3.33E-03\t8.25E-04\t1.42E-04\t2.23E-16\t0.000452641972\t0.000306011349\r",
 "C>A\tACT\t1.28E-03\t9.78E-05\t1.22E-02\t2.95E-02\t6.61E-03\t1.80E-04\t2.48E-04\t7.10E-04\t1.96E-03\t4.05E-03\t4.70E-02\t3.05E-03\t1.70E-02\t1.63E-05\t2.66E-04\t1.22E-03\t3.48E-04\t8.26E-02\t1.11E-03\t1.06E-02\t6.18E-05\t4.56E-04\t2.14E-02\t5.71E-04\t1.24E-03\t6.21E-04\t1.72E-03\t5.61E-18\t2.20E-02\t4.97E-03\t6.19E-04\t2.72E-03\t3.97E-03\t3.56E-02\t5.56E-04\t6.28E-03\t1.46E-02\t1.90E-03\t1.23E-03\t2.16E-02\t1.79E-02\t1.85E-03\t8.85E-03\t7.40E-03\t1.48E-02\t1.33E-03\t7.97E-03\t2.07E-03\t5.78E-03\t9.64E-03\t3.40E-03\t1.19E-02\t3.89E-05\t7.97E-03\t8.32E-02\t2.12E-02\t9.24E-03\t1.71E-05\t4.14E-04\t1.17E-03\t2.30E-02\t8.64E-03\t5.58E-03\t1.45E-02\t4.44E-04\t0.009295948365\t0.002717279233\r",
 "C>A\tCCA\t3.12E-04\t2.08E-04\t2.25E-02\t8.07E-02\t7.43E-03\t1.82E-03\t4.55E-04\t1.14E-03\t1.09E-04\t1.45E-02\t4.01E-02\t4.80E-03\t3.18E-03\t5.21E-04\t6.18E-04\t5.42E-03\t1.40E-03\t1.63E-02\t4.18E-02\t1.04E-02\t2.95E-04\t2.71E-04\t7.40E-02\t1.32E-03\t3.76E-02\t2.22E-16\t1.83E-03\t1.51E-04\t2.54E-02\t1.48E-02\t1.41E-03\t5.07E-03\t2.03E-03\t5.42E-02\t7.68E-04\t1.07E-02\t1.24E-03\t2.70E-03\t1.95E-03\t2.99E-02\t5.58E-02\t3.22E-03\t3.81E-01\t1.01E-02\t2.08E-02\t2.38E-03\t1.16E-02\t6.63E-03\t2.84E-05\t2.38E-01\t1.56E-02\t2.46E-02\t1.18E-03\t2.29E-01\t2.25E-02\t6.98E-03\t5.74E-02\t3.34E-01\t6.32E-05\t6.25E-03\t2.48E-03\t8.70E-03\t6.18E-03\t2.23E-03\t1.18E-03\t0.000000000000\t0.004264235123\r",
 "C>A\tCCC\t1.79E-03\t9.53E-05\t2.53E-02\t7.97E-02\t6.14E-03\t4.09E-03\t2.40E-04\t1.55E-03\t3.09E-03\t3.22E-03\t3.88E-02\t1.92E-03\t2.02E-04\t3.18E-03\t9.86E-04\t4.28E-03\t9.68E-04\t1.97E-02\t5.05E-03\t9.03E-03\t1.16E-03\t4.11E-05\t1.96E-02\t7.67E-04\t9.89E-02\t2.60E-03\t4.43E-03\t6.30E-04\t3.49E-02\t7.82E-03\t1.51E-03\t2.23E-03\t1.09E-04\t3.69E-02\t3.88E-04\t1.16E-02\t4.92E-03\t1.91E-03\t3.67E-05\t3.30E-02\t2.02E-02\t8.14E-04\t6.95E-02\t9.06E-03\t1.47E-02\t9.19E-04\t3.28E-02\t8.20E-05\t3.03E-02\t1.51E-01\t7.20E-03\t5.27E-03\t3.57E-05\t2.86E-02\t2.89E-03\t3.17E-03\t1.90E-02\t3.55E-03\t2.57E-03\t3.75E-03\t6.68E-03\t3.94E-03\t1.83E-03\t5.93E-03\t2.53E-04\t0.005218571801\t0.000306234865\r",
 "C>A\tCCG\t9.32E-05\t2.23E-16\t2.51E-03\t2.45E-02\t3.46E-03\t1.65E-03\t2.23E-16\t4.08E-04\t1.91E-03\t8.51E-05\t3.41E-03\t1.37E-03\t4.64E-05\t8.38E-05\t9.20E-06\t1.17E-03\t4.33E-06\t1.38E-03\t6.62E-05\t1.74E-03\t2.22E-16\t2.23E-16\t1.20E-02\t2.36E-04\t7.43E-03\t1.98E-03\t1.36E-04\t1.30E-04\t1.60E-02\t1.28E-03\t3.62E-04\t2.66E-04\t2.22E-16\t1.48E-02\t4.07E-04\t3.29E-04\t1.66E-05\t3.71E-04\t2.98E-04\t2.20E-03\t7.10E-03\t7.47E-04\t5.10E-02\t5.74E-03\t2.54E-03\t2.61E-04\t1.88E-03\t2.56E-05\t3.74E-03\t6.79E-02\t4.50E-03\t2.71E-04\t1.37E-01\t5.15E-01\t5.18E-18\t4.30E-03\t1.36E-02\t6.77E-02\t5.93E-03\t6.41E-04\t6.72E-18\t1.07E-03\t1.46E-03\t8.15E-04\t3.79E-04\t0.000069231298\t0.000157256840\r",
 "C>A\tCCT\t2.23E-16\t4.21E-04\t1.54E-02\t6.90E-02\t6.49E-03\t9.60E-03\t3.72E-04\t1.49E-03\t1.35E-02\t1.17E-04\t3.30E-02\t3.06E-03\t1.73E-02\t8.33E-03\t1.50E-04\t1.10E-03\t2.22E-16\t2.60E-01\t2.87E-02\t2.49E-03\t1.37E-04\t1.82E-04\t3.63E-02\t3.44E-05\t2.52E-01\t1.91E-02\t1.10E-04\t5.80E-04\t1.71E-02\t1.26E-02\t7.14E-03\t3.10E-03\t2.03E-04\t5.19E-02\t2.97E-04\t2.79E-02\t2.67E-03\t2.53E-03\t6.20E-04\t8.25E-02\t6.05E-02\t1.77E-03\t1.84E-01\t6.03E-03\t1.47E-02\t2.65E-03\t2.11E-02\t5.22E-18\t8.67E-02\t1.94E-01\t8.30E-03\t6.82E-04\t8.45E-04\t4.90E-02\t1.15E-03\t5.07E-03\t1.88E-02\t4.92E-03\t5.53E-03\t2.84E-05\t7.13E-02\t8.21E-03\t1.96E-03\t2.02E-03\t5.41E-05\t0.005640381180\t0.000860850681\r",
 "C>A\tGCA\t1.58E-03\t8.62E-05\t7.13E-03\t3.16E-02\t1.02E-02\t4.27E-04\t4.06E-05\t7.46E-04\t1.05E-03\t1.68E-04\t2.42E-02\t7.72E-03\t1.21E-03\t7.31E-04\t1.65E-04\t1.98E-03\t9.49E-04\t2.08E-03\t2.29E-03\t8.61E-03\t1.78E-03\t9.98E-04\t1.09E-01\t8.14E-04\t1.08E-02\t2.14E-03\t2.23E-03\t3.37E-04\t6.22E-02\t1.35E-02\t4.96E-04\t1.08E-02\t1.56E-04\t9.71E-02\t9.43E-04\t1.31E-02\t4.51E-03\t4.41E-03\t2.81E-03\t9.87E-03\t6.77E-02\t2.04E-03\t1.20E-02\t6.88E-03\t1.42E-02\t3.09E-03\t7.71E-03\t9.30E-03\t7.68E-18\t7.55E-03\t7.20E-03\t1.11E-02\t7.52E-04\t7.88E-03\t1.42E-02\t3.94E-03\t8.24E-03\t4.32E-04\t5.19E-03\t8.55E-04\t2.17E-04\t1.89E-03\t2.74E-03\t1.03E-01\t5.17E-04\t0.002703472389\t0.001439703098\r",
 "C>A\tGCC\t3.39E-04\t2.23E-16\t1.09E-02\t3.48E-02\t7.65E-03\t9.13E-04\t1.38E-04\t2.15E-04\t7.42E-04\t2.22E-16\t2.62E-02\t6.68E-03\t4.41E-04\t2.55E-03\t4.05E-04\t4.17E-04\t4.51E-04\t2.24E-02\t2.49E-03\t4.25E-03\t1.28E-03\t2.78E-06\t1.73E-02\t2.72E-04\t5.65E-03\t7.07E-03\t1.73E-03\t5.41E-04\t1.28E-01\t6.47E-03\t6.66E-04\t1.24E-02\t1.02E-03\t9.03E-02\t4.94E-04\t2.15E-02\t3.62E-03\t9.68E-04\t4.68E-04\t5.02E-02\t1.92E-02\t6.29E-04\t6.28E-03\t6.95E-03\t1.02E-02\t7.23E-04\t2.68E-02\t1.69E-03\t4.67E-03\t5.73E-03\t3.00E-03\t3.48E-03\t3.46E-04\t2.48E-03\t7.58E-03\t3.28E-03\t7.32E-03\t3.44E-03\t9.13E-03\t1.92E-04\t7.34E-03\t7.52E-04\t1.08E-03\t1.29E-01\t8.66E-04\t0.010228858800\t0.000504268015\r",
 "C>A\tGCG\t5.87E-04\t1.39E-05\t1.43E-03\t1.53E-02\t2.34E-03\t3.57E-04\t4.25E-05\t7.29E-05\t2.22E-16\t1.82E-05\t2.74E-03\t1.71E-04\t1.05E-05\t8.52E-05\t1.39E-05\t4.47E-04\t1.64E-04\t1.11E-03\t4.04E-04\t1.36E-03\t1.06E-04\t9.24E-05\t7.56E-03\t2.05E-04\t1.57E-03\t1.69E-03\t5.15E-04\t1.19E-04\t6.59E-02\t6.58E-04\t1.29E-04\t1.63E-03\t3.95E-05\t2.30E-02\t2.22E-16\t9.72E-04\t8.77E-04\t3.42E-04\t2.27E-04\t5.07E-04\t4.83E-03\t5.06E-04\t1.37E-03\t4.53E-03\t2.55E-03\t2.20E-04\t1.06E-03\t1.77E-04\t6.05E-04\t2.09E-03\t4.00E-04\t7.83E-04\t7.39E-04\t6.83E-03\t2.44E-04\t4.28E-03\t4.04E-03\t5.08E-02\t5.64E-03\t7.72E-05\t9.79E-06\t6.57E-18\t2.73E-03\t1.21E-02\t3.90E-04\t0.000763601342\t0.000362282705\r",
 "C>A\tGCT\t2.23E-16\t5.12E-05\t1.00E-02\t2.12E-02\t6.82E-03\t1.23E-03\t2.23E-16\t1.47E-04\t3.63E-04\t7.40E-05\t2.69E-02\t8.56E-03\t1.11E-02\t2.80E-03\t2.22E-16\t2.75E-05\t6.15E-05\t1.26E-01\t1.82E-02\t5.80E-03\t4.70E-05\t5.34E-04\t6.23E-02\t2.73E-04\t2.49E-03\t8.90E-03\t5.51E-04\t3.09E-05\t5.44E-02\t1.05E-02\t8.17E-04\t7.41E-03\t1.58E-04\t8.46E-02\t8.62E-04\t3.51E-03\t3.27E-03\t3.19E-03\t1.35E-05\t2.29E-02\t6.90E-02\t1.29E-03\t9.09E-03\t5.89E-03\t1.08E-02\t1.95E-03\t1.81E-02\t1.39E-03\t1.13E-02\t6.99E-03\t2.40E-03\t1.41E-03\t1.99E-05\t5.84E-03\t2.04E-03\t3.45E-03\t4.21E-03\t1.14E-03\t6.73E-03\t7.38E-04\t1.57E-02\t1.95E-03\t3.50E-04\t4.99E-01\t1.33E-04\t0.010179165989\t0.000762947056\r",
 "C>A\tTCA\t6.58E-05\t6.39E-04\t8.41E-03\t3.98E-02\t7.86E-03\t3.56E-04\t3.90E-04\t2.90E-03\t2.52E-03\t1.14E-02\t2.98E-02\t1.70E-02\t9.40E-02\t4.46E-04\t1.80E-04\t3.48E-03\t7.90E-02\t3.47E-04\t1.72E-02\t1.08E-02\t1.90E-04\t1.24E-05\t7.38E-02\t1.38E-03\t1.71E-03\t1.46E-03\t7.84E-04\t2.13E-03\t2.98E-02\t6.94E-03\t9.38E-04\t7.08E-03\t1.43E-02\t4.52E-02\t6.96E-04\t1.86E-03\t9.73E-03\t1.19E-02\t4.27E-03\t5.27E-03\t1.41E-01\t2.00E-03\t1.77E-02\t5.98E-03\t2.60E-02\t3.31E-03\t5.10E-03\t6.30E-03\t8.53E-04\t6.40E-02\t1.23E-02\t1.69E-02\t1.24E-01\t3.29E-02\t1.90E-02\t3.57E-02\t4.64E-01\t2.67E-03\t6.69E-04\t1.74E-03\t2.19E-01\t1.43E-02\t7.37E-03\t2.37E-05\t2.02E-03\t0.002083564327\t0.001944387330\r",
 "C>A\tTCC\t2.53E-03\t1.70E-04\t1.68E-02\t4.69E-02\t9.17E-03\t4.98E-04\t6.36E-04\t3.52E-03\t1.42E-02\t7.76E-03\t2.51E-02\t2.46E-03\t9.97E-03\t3.80E-03\t9.38E-04\t2.88E-03\t2.94E-02\t2.43E-02\t5.81E-03\t5.87E-03\t2.59E-05\t2.23E-16\t4.36E-02\t1.13E-03\t3.37E-03\t2.36E-04\t1.74E-03\t6.64E-04\t5.19E-02\t1.02E-02\t6.30E-04\t6.27E-03\t4.52E-04\t4.11E-02\t2.43E-04\t9.38E-03\t9.29E-03\t5.11E-03\t1.22E-03\t2.31E-02\t7.72E-02\t1.59E-03\t1.15E-02\t7.86E-03\t2.10E-02\t1.72E-03\t5.40E-02\t5.37E-06\t2.22E-03\t6.42E-02\t7.10E-03\t8.11E-05\t3.79E-03\t5.11E-03\t8.86E-04\t2.86E-03\t4.68E-02\t3.64E-03\t4.05E-05\t8.28E-18\t1.32E-02\t7.09E-04\t4.29E-04\t2.02E-02\t1.63E-04\t0.000009064386\t0.000169427568\r",
 "C>A\tTCG\t2.23E-16\t8.53E-05\t1.41E-03\t1.09E-02\t2.30E-03\t2.00E-04\t3.44E-05\t3.00E-04\t2.22E-16\t2.22E-16\t1.90E-03\t4.45E-04\t1.16E-03\t5.73E-04\t2.22E-16\t5.05E-04\t1.21E-02\t7.08E-06\t7.69E-05\t1.49E-03\t2.22E-16\t9.51E-05\t1.27E-02\t2.85E-04\t2.22E-16\t6.83E-05\t2.22E-16\t9.12E-05\t9.79E-03\t0.00E+00\t1.23E-04\t1.45E-03\t1.29E-04\t1.74E-02\t8.26E-05\t9.55E-04\t2.38E-03\t1.37E-03\t2.22E-16\t9.47E-05\t1.47E-02\t1.12E-03\t2.38E-03\t3.01E-03\t3.17E-03\t2.07E-04\t9.04E-04\t3.07E-18\t3.15E-07\t1.83E-02\t2.50E-03\t2.39E-05\t7.09E-01\t4.05E-02\t4.19E-05\t4.54E-03\t1.34E-01\t4.47E-04\t8.28E-04\t9.91E-05\t5.46E-03\t2.23E-04\t2.73E-03\t6.34E-04\t2.05E-04\t0.000000000000\t0.000000003989\r",
 "C>A\tTCT\t5.88E-06\t4.40E-04\t9.97E-03\t4.13E-02\t1.27E-02\t7.47E-04\t6.13E-04\t1.41E-03\t3.73E-03\t1.91E-04\t3.19E-02\t2.73E-02\t6.70E-01\t8.16E-02\t4.73E-05\t2.35E-03\t3.55E-02\t1.51E-01\t8.05E-03\t9.25E-03\t2.88E-04\t2.57E-03\t1.22E-01\t1.38E-03\t9.04E-04\t2.22E-16\t3.96E-04\t2.40E-04\t2.28E-02\t2.46E-02\t1.40E-03\t9.81E-03\t5.57E-04\t4.11E-02\t6.65E-04\t4.49E-03\t9.64E-03\t1.28E-02\t3.20E-05\t1.53E-02\t2.08E-01\t2.44E-03\t2.78E-02\t6.55E-03\t2.41E-02\t5.90E-03\t2.73E-02\t6.52E-18\t1.66E-02\t6.20E-02\t3.00E-03\t1.92E-04\t1.87E-04\t8.75E-03\t1.85E-03\t3.70E-04\t6.10E-02\t8.71E-04\t7.20E-03\t4.06E-03\t4.02E-01\t6.19E-03\t1.47E-02\t3.83E-03\t4.06E-04\t0.001203779266\t0.002057492391\r",
 "C>G\tACA\t1.86E-03\t2.23E-16\t1.97E-02\t6.89E-03\t1.01E-02\t4.71E-04\t6.49E-05\t8.56E-06\t1.12E-03\t1.18E-03\t4.33E-03\t4.84E-03\t2.23E-16\t3.05E-04\t1.70E-04\t1.22E-03\t3.87E-03\t2.22E-16\t1.13E-04\t1.74E-03\t1.01E-03\t1.46E-04\t1.73E-03\t3.19E-03\t8.77E-03\t1.69E-05\t1.66E-03\t5.74E-18\t3.34E-03\t8.03E-03\t4.27E-04\t1.32E-03\t1.54E-04\t3.14E-03\t1.07E-03\t8.32E-03\t1.99E-03\t1.41E-03\t7.62E-04\t3.58E-03\t1.84E-03\t3.43E-02\t2.69E-03\t4.73E-02\t1.22E-02\t5.33E-03\t4.06E-04\t3.61E-04\t3.18E-03\t3.11E-03\t1.20E-03\t1.03E-03\t1.98E-04\t2.88E-04\t3.85E-03\t3.96E-03\t1.73E-03\t7.03E-03\t1.55E-03\t2.85E-03\t4.18E-04\t1.54E-02\t2.06E-02\t2.94E-03\t3.77E-04\t0.006758077485\t0.007234927130\r",
 "C>G\tACC\t1.22E-03\t1.33E-04\t1.17E-02\t2.84E-03\t5.70E-03\t2.89E-04\t2.16E-05\t1.89E-04\t3.19E-04\t2.55E-03\t2.95E-03\t1.96E-03\t2.43E-05\t1.34E-04\t2.90E-04\t6.42E-04\t9.51E-04\t5.78E-06\t2.69E-04\t2.01E-03\t5.69E-04\t4.34E-05\t2.59E-03\t1.79E-03\t1.51E-03\t7.19E-04\t1.95E-04\t4.08E-05\t3.67E-03\t1.64E-03\t3.78E-04\t1.77E-03\t2.24E-04\t2.63E-03\t4.67E-04\t3.16E-03\t2.03E-03\t1.30E-03\t1.13E-03\t4.55E-03\t1.67E-03\t1.12E-02\t1.58E-03\t2.25E-02\t8.68E-03\t3.16E-03\t5.06E-04\t3.13E-05\t2.33E-04\t1.93E-03\t1.50E-03\t6.17E-04\t1.14E-04\t1.35E-04\t1.10E-03\t4.54E-03\t1.52E-03\t5.46E-03\t4.96E-04\t1.68E-04\t1.66E-04\t3.13E-03\t1.74E-03\t2.44E-03\t5.88E-04\t0.010811956859\t0.002551355408\r",
 "C>G\tACG\t1.15E-04\t1.52E-05\t2.53E-04\t1.28E-03\t1.72E-03\t2.22E-16\t5.08E-05\t5.05E-04\t2.22E-16\t9.06E-04\t2.86E-04\t1.36E-03\t2.23E-16\t2.22E-16\t5.96E-05\t7.51E-05\t2.64E-04\t3.91E-05\t3.07E-05\t8.47E-04\t1.52E-04\t2.23E-16\t1.92E-03\t6.16E-04\t4.69E-04\t2.22E-16\t2.31E-04\t2.86E-18\t9.20E-04\t0.00E+00\t6.83E-05\t0.00E+00\t2.22E-16\t8.96E-04\t1.31E-04\t3.00E-03\t1.13E-03\t2.53E-04\t2.22E-16\t3.46E-04\t5.00E-04\t5.35E-03\t6.36E-04\t1.36E-02\t2.25E-03\t1.21E-03\t1.77E-05\t4.43E-03\t3.14E-05\t2.70E-04\t0.00E+00\t3.28E-04\t8.03E-05\t2.04E-04\t8.27E-04\t1.97E-03\t8.18E-04\t2.51E-03\t1.70E-03\t5.10E-04\t4.28E-04\t2.77E-03\t1.69E-03\t8.30E-04\t2.76E-04\t0.000045986523\t0.000422196736\r",
 "C>G\tACT\t1.14E-03\t9.12E-05\t1.74E-02\t3.55E-03\t1.01E-02\t2.54E-04\t9.56E-05\t2.85E-04\t1.93E-03\t2.91E-03\t5.71E-03\t6.41E-03\t2.23E-16\t2.22E-16\t5.35E-04\t1.44E-03\t4.00E-03\t2.04E-04\t3.25E-04\t1.15E-02\t1.96E-05\t3.19E-04\t4.08E-03\t2.14E-03\t4.42E-03\t6.19E-05\t6.20E-05\t5.61E-18\t2.85E-03\t3.43E-03\t7.28E-04\t1.32E-03\t8.33E-05\t2.49E-03\t7.31E-04\t3.78E-03\t1.75E-03\t1.30E-03\t2.03E-03\t9.48E-03\t1.91E-03\t1.88E-02\t2.44E-03\t4.34E-02\t1.46E-02\t8.27E-03\t4.31E-04\t3.54E-04\t3.29E-03\t1.43E-03\t2.20E-03\t1.72E-03\t6.86E-04\t1.76E-04\t1.97E-03\t9.00E-04\t1.71E-03\t3.04E-03\t5.18E-03\t1.27E-03\t1.23E-04\t5.07E-03\t2.03E-03\t3.83E-03\t2.50E-04\t0.029594216358\t0.013019760586\r",
 "C>G\tCCA\t2.41E-05\t2.23E-16\t1.93E-02\t7.83E-03\t6.96E-03\t4.11E-05\t2.23E-16\t4.58E-04\t2.06E-04\t1.32E-03\t5.00E-03\t8.93E-04\t4.57E-05\t2.99E-05\t2.61E-04\t1.34E-03\t5.61E-03\t8.55E-05\t8.36E-05\t3.55E-03\t5.53E-04\t1.94E-04\t6.13E-04\t1.77E-03\t7.69E-05\t1.97E-05\t1.05E-03\t1.74E-04\t2.61E-03\t7.20E-03\t3.64E-04\t4.73E-03\t1.44E-05\t3.29E-03\t1.27E-03\t2.05E-03\t1.15E-03\t1.50E-03\t1.19E-04\t3.86E-03\t1.24E-03\t4.42E-04\t1.69E-03\t2.88E-02\t7.56E-03\t2.08E-03\t4.26E-03\t2.09E-05\t5.36E-05\t1.43E-03\t5.00E-04\t9.56E-04\t9.93E-05\t2.16E-04\t6.56E-05\t2.85E-03\t1.18E-03\t2.72E-03\t1.13E-03\t2.94E-03\t2.04E-04\t8.88E-04\t7.94E-04\t2.08E-03\t4.71E-05\t0.000007117033\t0.001223945047\r",
 "C>G\tCCC\t7.68E-05\t3.51E-04\t1.41E-02\t6.87E-03\t9.07E-03\t3.72E-05\t1.31E-04\t9.60E-04\t2.26E-03\t2.86E-04\t5.46E-03\t1.54E-03\t2.23E-16\t2.22E-16\t1.77E-04\t8.91E-04\t6.48E-04\t7.86E-05\t1.70E-05\t4.54E-03\t2.35E-04\t4.66E-05\t2.26E-03\t1.81E-03\t4.38E-06\t2.73E-04\t7.85E-04\t2.55E-04\t5.22E-03\t4.60E-03\t3.22E-04\t1.26E-03\t7.55E-05\t6.34E-03\t7.93E-04\t1.86E-03\t8.24E-04\t4.10E-04\t1.18E-04\t6.30E-03\t1.37E-03\t3.13E-04\t1.26E-03\t3.77E-02\t9.35E-03\t1.20E-03\t6.62E-03\t2.12E-04\t2.49E-05\t8.00E-04\t3.00E-04\t2.37E-04\t1.22E-04\t2.56E-04\t2.89E-04\t1.20E-04\t8.35E-04\t6.03E-03\t8.83E-04\t2.86E-03\t2.25E-04\t2.87E-03\t1.28E-03\t1.73E-03\t1.42E-04\t0.003440786012\t0.000210178860\r",
 "C>G\tCCG\t3.52E-04\t4.87E-05\t1.50E-03\t3.40E-03\t2.49E-03\t8.43E-05\t7.75E-05\t7.91E-04\t2.62E-04\t1.32E-03\t1.05E-05\t4.37E-04\t2.01E-05\t2.22E-16\t1.12E-05\t1.73E-04\t4.88E-04\t1.91E-05\t5.02E-05\t1.46E-03\t2.32E-04\t2.23E-16\t3.58E-03\t4.81E-05\t1.88E-04\t1.01E-05\t2.39E-04\t1.86E-18\t9.45E-04\t8.24E-04\t4.17E-05\t3.61E-04\t2.22E-16\t1.30E-03\t1.20E-04\t8.54E-04\t5.37E-04\t3.18E-04\t1.90E-04\t8.70E-04\t4.19E-04\t3.21E-04\t9.57E-04\t1.23E-02\t1.68E-03\t4.11E-04\t4.70E-04\t2.54E-03\t7.68E-18\t6.43E-04\t1.00E-04\t6.70E-04\t3.63E-05\t2.38E-04\t1.42E-03\t2.03E-03\t5.68E-04\t1.62E-03\t4.34E-04\t2.64E-04\t6.66E-05\t7.63E-04\t5.11E-05\t6.90E-04\t3.37E-04\t0.000002195949\t0.000040247509\r",
 "C>G\tCCT\t2.23E-16\t6.36E-05\t2.23E-02\t6.82E-03\t9.53E-03\t1.44E-05\t9.56E-05\t8.90E-04\t4.24E-03\t2.22E-16\t5.49E-03\t3.03E-03\t3.01E-05\t2.22E-16\t9.64E-06\t1.56E-03\t6.40E-03\t1.78E-04\t8.45E-05\t9.62E-03\t3.90E-05\t1.93E-04\t3.79E-03\t3.22E-03\t1.01E-05\t1.73E-04\t3.20E-04\t2.27E-04\t4.71E-03\t1.96E-04\t6.25E-04\t2.68E-04\t2.22E-16\t6.69E-03\t1.02E-03\t3.42E-03\t4.95E-04\t6.62E-04\t7.36E-05\t1.62E-02\t2.09E-03\t5.81E-04\t1.27E-03\t4.02E-02\t1.39E-02\t5.03E-03\t6.84E-03\t8.53E-04\t8.64E-05\t1.91E-03\t1.50E-03\t1.37E-04\t2.45E-04\t1.91E-04\t5.12E-04\t2.61E-03\t1.20E-03\t5.08E-04\t1.33E-03\t3.31E-03\t7.69E-04\t2.87E-03\t1.25E-03\t3.46E-03\t1.46E-04\t0.013308248775\t0.000484083725\r",
 "C>G\tGCA\t9.59E-06\t6.54E-05\t1.27E-02\t5.29E-03\t4.68E-03\t1.94E-05\t1.96E-05\t5.11E-05\t1.82E-04\t1.96E-04\t2.58E-03\t1.36E-03\t1.50E-05\t1.72E-05\t4.67E-05\t5.64E-04\t1.50E-03\t2.22E-16\t9.94E-04\t1.05E-03\t8.78E-04\t5.99E-05\t6.35E-04\t1.31E-03\t6.69E-04\t3.88E-04\t1.40E-03\t4.32E-18\t2.72E-03\t4.17E-03\t3.95E-04\t1.42E-03\t1.83E-04\t2.27E-03\t5.15E-04\t1.83E-03\t8.76E-04\t7.71E-04\t3.34E-05\t3.38E-03\t8.86E-04\t1.32E-03\t1.20E-03\t2.40E-02\t5.06E-03\t2.04E-03\t2.40E-03\t1.36E-02\t2.19E-04\t3.33E-04\t1.00E-04\t1.68E-03\t7.84E-05\t1.52E-04\t2.67E-02\t3.41E-03\t1.38E-03\t1.38E-03\t2.19E-03\t8.51E-04\t1.87E-04\t3.79E-03\t2.12E-03\t1.44E-03\t1.63E-04\t0.011151059622\t0.001595924101\r",
 "C>G\tGCC\t1.64E-04\t2.61E-04\t8.63E-03\t5.75E-03\t5.08E-03\t6.85E-04\t3.32E-05\t4.52E-04\t4.73E-04\t2.22E-16\t2.39E-03\t2.83E-03\t8.31E-05\t1.36E-04\t1.14E-04\t2.89E-04\t4.79E-04\t4.33E-05\t6.08E-04\t3.09E-03\t8.35E-04\t5.55E-05\t1.34E-03\t1.29E-03\t2.86E-03\t1.42E-03\t9.30E-04\t1.27E-04\t5.79E-03\t2.39E-03\t5.79E-04\t1.35E-03\t2.22E-16\t1.64E-03\t2.56E-04\t7.51E-03\t9.00E-04\t7.76E-04\t1.93E-04\t2.14E-02\t1.08E-03\t9.38E-04\t1.08E-03\t2.13E-02\t5.37E-03\t2.69E-03\t1.42E-03\t1.52E-02\t1.19E-03\t1.01E-03\t8.00E-04\t2.28E-03\t2.17E-04\t1.10E-04\t3.30E-02\t1.14E-03\t1.45E-03\t8.47E-02\t2.20E-04\t1.41E-03\t9.83E-05\t2.57E-03\t1.49E-03\t1.72E-03\t1.35E-04\t0.015758341831\t0.005060362002\r",
 "C>G\tGCG\t1.66E-04\t2.23E-16\t2.42E-03\t3.30E-03\t1.51E-03\t2.22E-16\t4.51E-05\t3.37E-04\t1.80E-04\t5.45E-04\t4.88E-04\t5.21E-04\t2.23E-16\t2.22E-16\t3.24E-06\t9.54E-05\t2.41E-04\t2.22E-16\t1.01E-05\t8.13E-05\t2.37E-05\t2.23E-16\t6.39E-04\t5.97E-04\t6.92E-04\t1.17E-04\t2.11E-04\t1.83E-18\t3.14E-03\t4.18E-04\t7.38E-05\t0.00E+00\t2.22E-16\t6.88E-06\t1.17E-04\t4.57E-04\t6.86E-05\t4.66E-05\t2.22E-16\t5.91E-04\t1.72E-04\t5.77E-04\t1.02E-03\t9.86E-03\t1.19E-03\t1.56E-04\t2.23E-04\t1.28E-02\t7.68E-18\t6.95E-04\t0.00E+00\t4.65E-04\t4.31E-05\t3.67E-05\t2.80E-03\t2.67E-04\t1.01E-03\t3.00E-03\t1.59E-03\t5.72E-03\t5.62E-06\t1.86E-03\t1.71E-03\t3.59E-04\t2.02E-04\t0.001697967239\t0.000026827588\r",
 "C>G\tGCT\t2.23E-16\t1.34E-04\t1.47E-02\t3.04E-03\t6.72E-03\t3.21E-04\t4.20E-05\t2.77E-04\t1.28E-03\t8.82E-04\t4.86E-03\t8.57E-03\t3.54E-05\t1.13E-04\t1.86E-04\t5.04E-04\t2.70E-03\t8.03E-05\t4.51E-04\t3.23E-03\t7.87E-04\t2.23E-16\t1.23E-03\t3.20E-04\t3.02E-03\t2.07E-04\t4.55E-04\t4.04E-18\t3.38E-03\t1.99E-03\t3.05E-04\t4.56E-03\t2.22E-16\t3.21E-03\t3.99E-04\t3.36E-03\t2.28E-04\t9.31E-04\t2.69E-05\t8.74E-03\t1.05E-03\t1.27E-03\t9.17E-04\t3.24E-02\t6.98E-03\t3.56E-03\t2.77E-03\t1.80E-02\t1.85E-03\t9.32E-04\t6.00E-04\t2.22E-03\t6.53E-05\t2.08E-04\t1.43E-02\t2.85E-03\t1.02E-03\t5.73E-04\t4.94E-03\t6.82E-05\t1.75E-06\t4.36E-04\t1.08E-03\t2.26E-03\t6.43E-05\t0.076057560709\t0.003186103940\r",
 "C>G\tTCA\t2.23E-16\t2.23E-16\t1.33E-02\t3.58E-03\t7.81E-03\t4.97E-05\t2.23E-16\t2.54E-04\t1.94E-03\t3.74E-03\t1.47E-03\t2.69E-03\t5.55E-05\t2.22E-16\t1.56E-05\t9.69E-04\t3.17E-01\t1.97E-05\t2.14E-04\t2.51E-03\t3.75E-03\t1.60E-04\t2.14E-03\t1.75E-03\t1.29E-04\t6.54E-04\t3.56E-03\t2.11E-04\t3.75E-03\t9.80E-03\t1.31E-03\t2.87E-03\t1.15E-04\t1.07E-04\t1.27E-04\t3.96E-04\t4.71E-18\t1.30E-02\t6.18E-07\t6.63E-03\t3.98E-03\t1.79E-03\t2.66E-03\t3.47E-02\t1.25E-02\t1.99E-02\t5.59E-04\t4.42E-06\t6.73E-04\t3.30E-03\t2.80E-03\t1.38E-04\t1.37E-04\t4.80E-05\t9.32E-04\t2.05E-03\t1.98E-03\t3.74E-03\t4.69E-03\t7.06E-03\t1.13E-03\t2.28E-03\t2.13E-03\t2.35E-06\t5.29E-04\t0.000000000000\t0.000947995470\r",
 "C>G\tTCC\t2.03E-04\t9.87E-05\t2.03E-02\t5.95E-03\t7.60E-03\t1.13E-04\t5.09E-05\t1.62E-03\t4.61E-03\t5.88E-04\t1.34E-03\t2.00E-03\t2.23E-16\t2.22E-16\t2.24E-04\t1.55E-03\t6.35E-02\t1.50E-04\t6.12E-05\t5.23E-03\t7.23E-04\t6.18E-05\t4.64E-03\t3.12E-03\t2.82E-05\t6.31E-04\t1.52E-03\t2.66E-04\t5.58E-03\t3.27E-03\t7.45E-04\t3.11E-03\t1.36E-04\t6.83E-03\t9.86E-04\t2.92E-03\t1.10E-03\t2.51E-03\t1.40E-04\t5.63E-03\t2.41E-03\t1.11E-03\t2.18E-03\t3.67E-02\t1.41E-02\t1.42E-02\t1.87E-03\t1.03E-03\t9.40E-06\t2.01E-03\t2.40E-03\t1.19E-03\t3.50E-04\t1.53E-04\t4.93E-04\t3.61E-03\t1.44E-03\t4.99E-03\t5.17E-04\t3.28E-03\t2.94E-04\t2.80E-03\t1.80E-03\t2.09E-03\t3.48E-04\t0.000113943392\t0.001869404519\r",
 "C>G\tTCG\t6.94E-05\t4.12E-05\t5.20E-04\t1.83E-03\t1.71E-03\t1.95E-06\t6.35E-06\t3.64E-04\t8.75E-06\t8.73E-04\t7.82E-05\t4.04E-04\t2.23E-16\t2.22E-16\t2.22E-16\t2.14E-04\t1.38E-02\t1.83E-05\t5.84E-05\t4.39E-04\t3.80E-04\t1.55E-05\t8.99E-04\t4.41E-04\t1.24E-04\t6.27E-05\t3.81E-04\t2.56E-18\t1.35E-03\t2.06E-03\t9.41E-05\t1.28E-05\t2.46E-05\t8.97E-04\t1.06E-04\t4.16E-04\t9.97E-04\t8.04E-05\t2.22E-16\t1.55E-04\t3.02E-04\t8.60E-04\t1.10E-03\t9.62E-03\t1.74E-03\t1.93E-03\t2.07E-04\t1.47E-03\t7.68E-18\t1.78E-04\t1.00E-04\t0.00E+00\t2.82E-05\t6.33E-05\t1.03E-04\t2.16E-03\t5.81E-04\t1.87E-03\t1.23E-03\t2.29E-04\t8.39E-05\t1.33E-03\t6.30E-04\t4.05E-04\t6.59E-05\t0.000000720571\t0.000169996975\r",
 "C>G\tTCT\t1.56E-03\t1.18E-03\t2.35E-02\t3.38E-03\t1.25E-02\t1.28E-04\t2.81E-04\t7.14E-04\t1.23E-02\t8.53E-05\t2.49E-04\t5.63E-03\t5.45E-05\t4.34E-05\t1.85E-04\t3.05E-03\t3.69E-01\t3.21E-04\t1.74E-04\t2.85E-02\t2.83E-04\t1.50E-03\t3.18E-03\t1.61E-03\t1.12E-03\t6.87E-04\t3.88E-04\t4.66E-04\t4.07E-03\t1.64E-02\t2.21E-03\t2.13E-03\t1.43E-04\t3.36E-04\t1.09E-04\t4.24E-03\t6.92E-04\t2.01E-02\t2.15E-03\t2.12E-02\t7.59E-03\t2.55E-03\t3.95E-03\t5.07E-02\t2.37E-02\t4.46E-02\t3.81E-04\t2.51E-04\t2.59E-03\t3.29E-03\t4.90E-03\t9.24E-04\t1.48E-04\t5.44E-04\t4.21E-04\t3.33E-03\t2.42E-03\t3.74E-03\t3.11E-04\t2.11E-03\t4.11E-04\t1.29E-02\t1.81E-02\t1.52E-03\t7.61E-04\t0.004953383455\t0.006244142553\r",
 "C>T\tACA\t2.50E-02\t6.11E-05\t1.42E-02\t8.70E-03\t3.26E-02\t6.41E-02\t1.22E-04\t1.41E-03\t1.33E-03\t3.33E-03\t6.81E-03\t8.10E-03\t1.04E-03\t5.49E-04\t2.58E-02\t5.51E-03\t7.18E-03\t1.24E-02\t5.05E-03\t1.76E-03\t1.22E-03\t5.71E-04\t9.20E-03\t2.80E-02\t4.93E-02\t2.17E-03\t1.76E-03\t2.26E-02\t1.18E-02\t2.10E-02\t2.66E-03\t1.37E-02\t7.77E-04\t5.09E-03\t1.06E-01\t1.77E-02\t8.64E-02\t5.31E-03\t3.52E-03\t2.21E-03\t5.96E-03\t6.24E-03\t3.45E-03\t9.60E-03\t1.97E-02\t9.88E-03\t2.24E-02\t2.90E-05\t8.99E-02\t1.33E-03\t1.52E-02\t4.27E-03\t2.87E-04\t5.75E-04\t3.52E-03\t3.58E-02\t2.69E-03\t7.22E-03\t1.48E-03\t5.11E-03\t7.40E-03\t3.26E-02\t8.05E-02\t3.92E-03\t2.76E-03\t0.053885768706\t0.001374780580\r",
 "C>T\tACC\t6.32E-03\t1.38E-03\t1.24E-02\t4.18E-03\t1.79E-02\t2.29E-02\t1.77E-02\t3.81E-02\t7.68E-03\t1.40E-03\t2.82E-03\t5.42E-03\t8.34E-04\t5.25E-03\t1.47E-01\t2.12E-03\t2.22E-16\t1.01E-02\t6.91E-03\t3.73E-03\t1.71E-03\t7.03E-05\t4.65E-03\t1.41E-02\t1.30E-02\t3.92E-03\t2.20E-04\t8.09E-02\t1.12E-02\t1.31E-02\t2.54E-03\t5.42E-03\t3.38E-03\t4.31E-03\t9.04E-02\t9.28E-03\t1.23E-01\t2.13E-03\t3.71E-03\t3.99E-03\t3.28E-03\t3.06E-03\t2.15E-03\t4.04E-03\t1.21E-02\t3.36E-03\t3.98E-02\t1.28E-04\t1.69E-02\t6.01E-04\t8.90E-03\t9.40E-04\t4.38E-04\t3.03E-04\t1.80E-03\t5.65E-03\t2.10E-03\t1.96E-03\t5.29E-04\t2.33E-04\t9.82E-04\t1.29E-02\t6.76E-03\t2.86E-03\t1.34E-03\t0.042106831614\t0.001478441742\r",
 "C>T\tACG\t3.65E-01\t3.27E-05\t2.57E-03\t7.83E-04\t6.18E-03\t1.20E-01\t2.23E-16\t5.43E-04\t9.27E-04\t1.21E-03\t3.23E-03\t4.38E-03\t5.57E-03\t4.14E-04\t7.58E-03\t1.18E-04\t3.03E-04\t2.65E-02\t7.45E-02\t3.27E-03\t3.95E-05\t2.87E-04\t1.22E-02\t1.55E-03\t1.33E-02\t7.40E-03\t1.36E-03\t8.72E-03\t5.14E-04\t1.97E-02\t5.33E-03\t6.71E-03\t7.05E-03\t2.01E-02\t1.44E-02\t7.59E-03\t3.66E-02\t7.16E-03\t1.18E-04\t5.80E-04\t7.07E-03\t6.33E-03\t5.18E-03\t6.00E-03\t2.67E-04\t3.43E-03\t4.40E-04\t1.22E-03\t1.43E-02\t5.41E-03\t4.60E-03\t3.05E-03\t1.21E-03\t2.01E-03\t1.39E-02\t5.65E-03\t4.08E-03\t6.95E-03\t3.43E-04\t5.32E-03\t1.52E-03\t1.76E-02\t8.63E-03\t2.56E-03\t4.21E-03\t0.002540673114\t0.000443775624\r",
 "C>T\tACT\t9.58E-03\t1.86E-03\t1.21E-02\t4.25E-03\t2.20E-02\t2.13E-02\t7.42E-03\t1.08E-02\t7.48E-03\t3.29E-02\t1.25E-02\t6.49E-03\t2.99E-03\t7.93E-03\t1.09E-01\t2.38E-03\t3.18E-04\t1.61E-02\t1.78E-03\t6.00E-03\t7.37E-04\t1.47E-04\t7.97E-03\t3.68E-02\t1.30E-02\t4.17E-05\t6.16E-04\t3.50E-02\t9.15E-03\t5.73E-03\t2.79E-03\t4.48E-03\t5.06E-03\t1.83E-03\t2.93E-02\t1.01E-02\t1.52E-01\t3.07E-03\t3.87E-03\t9.00E-04\t4.31E-03\t5.65E-03\t2.44E-03\t1.28E-02\t1.69E-02\t6.35E-03\t3.52E-02\t9.10E-04\t3.03E-02\t1.22E-03\t6.80E-03\t2.49E-03\t2.35E-04\t4.15E-04\t6.92E-03\t3.20E-03\t1.76E-03\t1.94E-03\t4.99E-03\t7.76E-04\t2.84E-03\t1.54E-02\t7.31E-03\t3.39E-03\t2.98E-04\t0.054345874146\t0.004563583260\r",
 "C>T\tCCA\t2.00E-03\t4.32E-03\t1.61E-02\t1.27E-02\t1.94E-02\t1.36E-02\t5.00E-02\t1.07E-01\t2.71E-03\t3.56E-02\t2.67E-03\t7.24E-03\t3.68E-05\t1.51E-04\t4.07E-03\t6.59E-03\t1.93E-05\t2.73E-03\t1.08E-02\t7.17E-03\t8.21E-04\t4.57E-04\t1.17E-02\t1.19E-01\t1.06E-02\t1.35E-03\t1.21E-03\t6.22E-02\t5.38E-03\t1.43E-02\t1.69E-03\t1.74E-02\t2.92E-04\t7.61E-03\t1.10E-01\t4.12E-02\t1.69E-02\t4.59E-03\t4.75E-03\t5.10E-03\t6.75E-03\t3.61E-03\t4.24E-03\t9.11E-03\t1.96E-02\t6.91E-03\t1.87E-02\t7.62E-06\t3.57E-03\t2.14E-03\t2.84E-02\t8.64E-04\t1.82E-05\t4.70E-04\t3.84E-03\t3.23E-03\t3.27E-03\t1.46E-02\t3.37E-03\t7.25E-03\t2.98E-04\t1.49E-02\t8.09E-03\t2.87E-03\t1.52E-03\t0.018028129532\t0.001636999066\r",
 "C>T\tCCC\t2.70E-04\t1.70E-04\t2.02E-02\t1.15E-02\t1.90E-02\t1.48E-02\t5.53E-02\t1.82E-01\t5.97E-03\t2.60E-03\t4.02E-03\t8.76E-03\t2.23E-16\t8.44E-05\t1.26E-01\t2.54E-04\t1.35E-03\t1.38E-03\t1.11E-02\t2.84E-03\t1.72E-03\t5.89E-05\t6.76E-03\t9.20E-02\t1.30E-02\t5.16E-03\t3.19E-03\t1.89E-01\t8.85E-03\t5.94E-03\t2.83E-03\t7.14E-03\t2.10E-04\t1.64E-03\t1.03E-01\t1.64E-01\t2.84E-02\t1.53E-03\t5.14E-04\t8.32E-02\t6.68E-03\t2.10E-03\t4.73E-03\t3.86E-03\t1.73E-02\t1.85E-03\t4.81E-02\t4.35E-18\t2.86E-03\t5.07E-05\t1.33E-02\t4.61E-03\t3.01E-04\t4.93E-04\t4.25E-03\t4.70E-03\t1.82E-03\t4.60E-03\t2.26E-04\t4.00E-03\t5.31E-04\t1.21E-02\t9.92E-03\t4.09E-04\t1.88E-03\t0.040621303139\t0.000369373076\r",
 "C>T\tCCG\t1.96E-01\t2.77E-03\t2.24E-03\t3.31E-03\t1.75E-02\t1.22E-01\t6.71E-03\t2.21E-02\t8.94E-03\t2.64E-03\t1.49E-03\t6.51E-03\t2.23E-16\t3.70E-03\t9.20E-04\t4.09E-05\t1.99E-03\t1.16E-02\t2.48E-02\t6.91E-04\t1.90E-03\t5.43E-04\t1.68E-02\t1.57E-02\t1.65E-02\t6.22E-03\t4.26E-04\t1.42E-02\t6.35E-03\t1.16E-02\t3.34E-03\t7.50E-03\t2.55E-03\t1.53E-02\t1.68E-02\t8.20E-03\t1.12E-02\t5.88E-03\t9.45E-03\t2.67E-03\t5.33E-03\t3.27E-03\t3.50E-03\t7.57E-03\t2.36E-03\t1.01E-02\t1.92E-03\t9.72E-05\t1.55E-02\t2.20E-03\t8.30E-03\t1.52E-03\t1.87E-05\t1.38E-03\t1.08E-03\t2.07E-02\t2.38E-03\t1.37E-02\t2.83E-02\t6.59E-05\t7.95E-03\t6.40E-03\t7.99E-03\t1.10E-03\t4.06E-03\t0.000000000000\t0.000024057242\r",
 "C>T\tCCT\t1.96E-04\t1.45E-03\t2.30E-02\t1.60E-02\t2.31E-02\t1.38E-02\t5.66E-02\t1.26E-01\t2.06E-02\t1.38E-04\t1.45E-02\t1.11E-02\t1.03E-04\t5.95E-04\t6.51E-02\t3.38E-03\t1.45E-03\t1.36E-03\t1.56E-02\t1.93E-02\t1.74E-03\t1.36E-04\t6.65E-03\t2.58E-01\t1.17E-02\t3.25E-04\t1.60E-03\t1.56E-01\t4.29E-03\t1.36E-02\t1.65E-03\t7.15E-03\t3.32E-03\t6.73E-04\t4.62E-02\t1.45E-01\t4.28E-02\t2.70E-03\t3.53E-05\t4.70E-02\t5.75E-03\t6.06E-03\t4.10E-03\t6.99E-03\t2.16E-02\t7.73E-03\t3.29E-02\t7.44E-04\t5.04E-03\t2.73E-03\t6.40E-03\t3.00E-03\t4.79E-04\t5.19E-04\t1.19E-02\t4.06E-03\t2.57E-03\t1.60E-02\t3.71E-03\t7.19E-04\t5.08E-04\t3.20E-02\t2.77E-02\t2.42E-03\t2.51E-03\t0.042237294613\t0.002350306147\r",
 "C>T\tGCA\t4.44E-03\t2.23E-16\t1.64E-02\t4.09E-03\t1.94E-02\t9.36E-02\t8.49E-05\t8.55E-04\t1.57E-04\t3.65E-04\t2.76E-04\t6.98E-03\t4.22E-05\t3.63E-04\t1.97E-02\t4.48E-03\t7.18E-03\t1.71E-02\t7.37E-02\t6.50E-03\t1.64E-03\t6.11E-05\t9.68E-03\t6.10E-02\t4.22E-02\t9.32E-03\t2.38E-03\t5.37E-02\t2.99E-02\t1.00E-02\t2.75E-03\t1.42E-02\t3.25E-03\t1.04E-02\t4.97E-02\t6.80E-03\t2.26E-02\t5.35E-03\t2.33E-04\t7.29E-03\t3.72E-03\t2.55E-03\t2.37E-03\t8.03E-03\t1.43E-02\t6.01E-03\t3.36E-02\t7.66E-04\t9.55E-02\t1.00E-03\t1.26E-02\t1.06E-03\t3.45E-04\t1.84E-04\t2.42E-03\t4.47E-03\t1.78E-03\t9.18E-03\t2.22E-04\t6.50E-03\t9.49E-06\t5.14E-03\t6.78E-03\t2.44E-03\t1.24E-03\t0.089177561445\t0.000887033367\r",
 "C>T\tGCC\t9.28E-05\t1.29E-07\t1.34E-02\t4.46E-03\t2.16E-02\t1.11E-01\t1.17E-02\t2.63E-02\t9.51E-06\t1.81E-04\t2.45E-04\t6.53E-03\t1.25E-02\t1.51E-02\t1.15E-01\t1.82E-03\t9.21E-04\t8.16E-03\t1.29E-01\t5.29E-03\t2.22E-16\t1.57E-04\t8.17E-03\t3.97E-02\t5.32E-02\t1.96E-02\t1.87E-03\t1.54E-01\t4.37E-02\t6.95E-03\t5.45E-03\t2.01E-02\t1.88E-04\t7.96E-03\t6.10E-02\t1.14E-02\t3.79E-02\t5.14E-03\t2.22E-16\t1.78E-02\t3.40E-03\t2.31E-03\t2.96E-03\t4.50E-03\t1.16E-02\t8.64E-03\t1.52E-01\t4.10E-18\t8.14E-02\t9.33E-04\t4.50E-03\t1.71E-03\t7.39E-06\t1.18E-04\t1.58E-03\t3.09E-03\t1.22E-03\t1.19E-02\t1.31E-02\t1.51E-03\t5.06E-04\t6.93E-03\t3.15E-03\t2.35E-03\t1.44E-03\t0.111629551116\t0.000484366313\r",
 "C>T\tGCG\t2.18E-01\t2.23E-16\t5.28E-04\t4.82E-04\t1.23E-02\t1.78E-01\t5.52E-06\t1.76E-04\t3.43E-05\t1.57E-04\t1.91E-03\t4.76E-03\t1.49E-02\t1.26E-02\t1.78E-03\t7.46E-05\t1.78E-04\t1.92E-05\t2.77E-01\t1.17E-03\t1.61E-03\t8.55E-04\t1.48E-02\t3.21E-03\t2.05E-02\t4.34E-05\t2.22E-16\t1.23E-02\t2.54E-02\t1.05E-02\t6.68E-03\t9.37E-03\t2.15E-04\t1.83E-02\t5.24E-03\t1.97E-03\t1.04E-02\t5.84E-03\t1.67E-02\t2.53E-03\t5.70E-03\t4.32E-03\t4.48E-03\t5.48E-03\t3.93E-18\t1.71E-02\t4.55E-03\t9.66E-06\t4.96E-02\t1.51E-03\t1.50E-03\t1.72E-03\t3.03E-04\t1.04E-03\t5.50E-03\t4.28E-03\t2.83E-03\t5.99E-03\t1.91E-02\t1.19E-02\t2.06E-03\t8.44E-03\t7.36E-03\t3.24E-18\t3.36E-03\t0.009223925314\t0.000116423626\r",
 "C>T\tGCT\t3.84E-05\t5.04E-05\t1.13E-02\t3.81E-03\t1.86E-02\t7.44E-02\t1.61E-02\t5.39E-03\t7.58E-04\t4.21E-02\t4.61E-03\t1.07E-02\t1.35E-02\t3.19E-02\t6.96E-02\t2.67E-03\t1.27E-03\t1.05E-02\t7.72E-02\t6.72E-03\t6.60E-05\t2.23E-16\t6.69E-03\t1.09E-01\t3.31E-02\t8.05E-03\t1.76E-03\t1.07E-01\t2.03E-02\t9.82E-03\t4.98E-03\t7.59E-03\t1.02E-04\t5.12E-03\t1.77E-02\t5.70E-03\t5.32E-02\t3.93E-03\t5.17E-04\t6.03E-03\t3.63E-03\t2.32E-03\t3.20E-03\t8.61E-03\t1.27E-02\t8.41E-03\t8.65E-02\t1.86E-03\t6.66E-02\t1.99E-03\t4.00E-03\t4.04E-03\t1.10E-04\t3.08E-04\t1.63E-02\t5.95E-03\t1.02E-03\t3.10E-03\t2.48E-04\t5.29E-03\t8.30E-04\t6.18E-03\t5.03E-03\t2.96E-03\t8.94E-04\t0.180440456121\t0.004167105016\r",
 "C>T\tTCA\t1.11E-03\t5.36E-01\t5.76E-03\t3.74E-03\t2.03E-02\t1.20E-02\t2.38E-01\t6.61E-03\t1.73E-02\t2.07E-04\t5.04E-03\t4.27E-03\t3.66E-04\t2.73E-03\t9.78E-03\t5.52E-03\t6.93E-03\t3.54E-03\t2.65E-02\t6.36E-03\t5.32E-04\t1.08E-04\t1.14E-02\t4.39E-02\t8.07E-03\t3.60E-03\t1.59E-03\t1.53E-02\t2.40E-03\t1.11E-02\t3.29E-03\t1.88E-02\t4.84E-03\t6.39E-03\t9.80E-02\t7.48E-03\t2.48E-02\t1.47E-02\t1.86E-03\t2.96E-03\t1.07E-02\t4.02E-03\t6.45E-03\t3.88E-03\t3.00E-02\t1.17E-02\t4.82E-03\t3.72E-05\t1.40E-03\t2.40E-03\t1.85E-02\t7.24E-03\t2.14E-04\t1.87E-04\t1.94E-02\t1.70E-03\t1.47E-03\t5.73E-03\t7.45E-18\t5.82E-11\t2.00E-04\t8.14E-03\t4.29E-03\t1.61E-04\t4.07E-05\t0.000070752091\t0.003823610731\r",
 "C>T\tTCC\t3.73E-05\t9.73E-02\t1.33E-02\t6.99E-03\t2.30E-02\t1.05E-02\t3.31E-01\t2.11E-01\t1.37E-02\t5.65E-03\t2.52E-04\t7.33E-03\t2.76E-03\t4.85E-02\t1.58E-01\t2.64E-03\t4.48E-03\t1.37E-04\t9.56E-03\t3.68E-03\t9.06E-05\t6.04E-04\t1.14E-02\t2.50E-02\t9.19E-03\t8.07E-05\t1.32E-03\t4.36E-02\t1.22E-02\t6.45E-03\t2.53E-03\t7.99E-03\t4.13E-03\t3.61E-18\t1.34E-01\t2.46E-02\t3.87E-02\t3.17E-03\t2.89E-04\t1.26E-02\t6.86E-03\t2.50E-03\t7.23E-03\t5.28E-03\t2.31E-02\t6.47E-03\t3.27E-02\t5.06E-18\t2.25E-03\t2.02E-03\t1.26E-02\t3.71E-02\t3.17E-04\t5.82E-04\t1.04E-01\t1.77E-03\t1.53E-03\t5.77E-03\t6.61E-04\t4.11E-03\t1.84E-04\t1.73E-02\t1.53E-02\t2.80E-04\t9.58E-04\t0.026909334242\t0.002998120759\r",
 "C>T\tTCG\t1.10E-01\t4.42E-02\t9.89E-04\t2.81E-04\t1.59E-02\t5.20E-02\t7.41E-02\t5.74E-02\t4.64E-05\t1.75E-03\t4.17E-05\t3.26E-03\t2.79E-03\t4.37E-01\t2.94E-07\t8.85E-05\t1.19E-03\t5.32E-03\t2.12E-02\t3.02E-03\t1.77E-04\t6.24E-05\t1.18E-02\t4.32E-03\t2.64E-03\t3.17E-04\t3.77E-05\t2.87E-03\t2.69E-03\t1.41E-02\t2.63E-03\t4.61E-03\t1.74E-03\t1.44E-02\t1.47E-02\t8.41E-03\t9.85E-03\t3.85E-03\t3.00E-04\t7.65E-04\t3.89E-03\t3.43E-03\t3.47E-03\t4.86E-03\t2.12E-03\t4.92E-03\t3.64E-04\t3.07E-18\t7.68E-18\t1.75E-03\t3.70E-03\t1.94E-03\t4.02E-04\t9.34E-04\t3.97E-03\t6.49E-03\t2.19E-03\t1.02E-02\t1.19E-13\t4.81E-03\t1.54E-04\t4.60E-03\t3.34E-03\t4.54E-18\t1.73E-03\t0.000006923779\t0.000157958968\r",
 "C>T\tTCT\t2.23E-16\t3.00E-01\t8.17E-03\t7.22E-03\t2.10E-02\t6.99E-03\t1.07E-01\t1.02E-01\t2.32E-03\t5.00E-02\t5.27E-03\t5.46E-03\t2.28E-02\t1.28E-01\t1.19E-01\t6.12E-03\t1.42E-03\t3.59E-04\t1.74E-03\t2.21E-02\t3.48E-04\t8.95E-04\t3.75E-03\t6.29E-02\t5.00E-05\t7.31E-05\t2.22E-16\t2.63E-02\t4.71E-03\t1.24E-02\t2.13E-03\t1.19E-02\t1.99E-03\t4.28E-03\t5.25E-02\t1.95E-02\t6.70E-02\t9.04E-03\t8.08E-04\t8.04E-03\t8.33E-03\t5.84E-03\t6.58E-03\t4.23E-03\t2.24E-02\t1.15E-02\t1.51E-02\t8.36E-04\t3.02E-03\t2.95E-03\t4.40E-03\t8.09E-02\t5.85E-06\t3.12E-04\t8.18E-02\t4.24E-02\t1.65E-03\t7.71E-03\t2.36E-04\t3.14E-02\t8.20E-05\t3.72E-02\t1.79E-01\t2.09E-03\t3.91E-03\t0.016839668838\t0.038650889984\r",
 "T>A\tATA\t8.00E-04\t9.59E-05\t5.49E-03\t9.42E-03\t7.38E-03\t9.82E-05\t1.94E-03\t2.49E-03\t1.24E-02\t8.89E-05\t1.95E-02\t1.27E-02\t5.07E-05\t1.82E-03\t4.17E-04\t5.17E-03\t2.22E-16\t6.13E-04\t2.04E-03\t2.31E-02\t1.01E-03\t6.26E-05\t2.49E-03\t7.86E-04\t5.11E-04\t3.46E-03\t6.39E-02\t8.28E-18\t1.58E-03\t2.04E-02\t1.52E-03\t1.43E-01\t1.72E-04\t6.19E-04\t3.63E-04\t7.86E-03\t7.13E-04\t8.68E-04\t1.03E-01\t4.89E-03\t2.44E-03\t2.25E-03\t1.99E-03\t5.31E-03\t6.10E-03\t7.46E-03\t1.33E-02\t1.53E-02\t4.78E-03\t5.25E-04\t2.70E-03\t1.38E-01\t6.60E-04\t1.29E-04\t5.36E-03\t1.13E-01\t1.77E-03\t5.89E-03\t9.56E-04\t4.26E-03\t5.05E-06\t3.28E-03\t2.45E-02\t2.00E-03\t3.97E-03\t0.001562338887\t0.111858717410\r",
 "T>A\tATC\t2.23E-03\t8.78E-04\t7.21E-03\t3.53E-03\t6.93E-03\t5.46E-04\t6.70E-04\t4.67E-04\t3.42E-03\t1.08E-03\t2.13E-02\t2.82E-03\t1.94E-04\t3.37E-04\t4.41E-04\t2.71E-03\t6.52E-05\t3.72E-04\t8.41E-04\t7.74E-03\t9.97E-04\t2.89E-04\t8.78E-04\t2.55E-04\t5.19E-03\t5.65E-03\t7.01E-03\t5.56E-18\t2.25E-03\t8.89E-03\t3.52E-03\t1.46E-03\t4.78E-04\t2.00E-03\t3.93E-04\t6.07E-03\t8.87E-04\t1.14E-03\t4.44E-02\t7.36E-03\t1.43E-03\t1.66E-03\t1.94E-03\t2.88E-03\t4.44E-03\t4.30E-03\t4.01E-03\t3.78E-03\t2.60E-03\t4.97E-04\t1.80E-03\t7.16E-03\t2.66E-04\t7.78E-05\t3.94E-03\t1.35E-03\t1.18E-03\t2.59E-03\t3.19E-03\t6.18E-04\t4.29E-03\t6.43E-03\t1.46E-03\t2.50E-03\t2.23E-04\t0.000397461148\t0.013051252824\r",
 "T>A\tATG\t1.14E-03\t2.55E-05\t9.64E-03\t1.24E-02\t9.12E-03\t2.22E-16\t5.87E-04\t1.56E-03\t2.11E-04\t3.70E-03\t2.28E-02\t2.76E-03\t2.23E-16\t2.34E-05\t3.93E-05\t3.12E-03\t5.90E-04\t3.54E-04\t5.40E-04\t8.81E-03\t2.00E-04\t6.57E-05\t2.60E-03\t9.52E-04\t3.03E-04\t2.48E-04\t6.06E-02\t3.54E-04\t2.00E-03\t6.24E-03\t6.25E-04\t1.23E-03\t1.52E-04\t2.48E-03\t6.38E-04\t5.03E-03\t7.12E-04\t6.96E-04\t3.46E-02\t1.33E-03\t2.53E-03\t2.49E-03\t1.08E-03\t6.94E-03\t6.93E-03\t3.03E-03\t5.01E-03\t1.43E-03\t1.08E-06\t2.13E-03\t5.00E-04\t7.45E-03\t4.65E-04\t2.41E-04\t5.73E-04\t3.68E-02\t1.19E-03\t2.87E-03\t4.23E-03\t8.10E-03\t5.13E-04\t5.80E-03\t4.69E-03\t2.19E-03\t3.52E-04\t0.000070829550\t0.021107255813\r",
 "T>A\tATT\t1.83E-04\t2.23E-16\t6.11E-03\t4.21E-03\t6.20E-03\t1.66E-03\t3.71E-03\t8.29E-03\t5.07E-02\t4.11E-02\t2.50E-02\t1.89E-02\t6.66E-03\t2.85E-03\t3.40E-06\t2.82E-03\t6.06E-04\t1.85E-03\t9.54E-03\t1.10E-02\t1.46E-03\t6.33E-04\t7.84E-03\t6.54E-04\t2.56E-02\t1.60E-03\t2.51E-03\t1.61E-04\t1.80E-03\t1.49E-02\t3.07E-03\t2.54E-03\t1.70E-03\t1.96E-03\t4.90E-04\t4.16E-03\t8.44E-04\t5.07E-03\t3.16E-01\t7.37E-03\t3.07E-03\t2.77E-03\t2.72E-03\t6.12E-03\t1.07E-02\t6.96E-02\t6.85E-03\t4.28E-03\t3.08E-02\t6.05E-04\t2.00E-03\t6.76E-02\t6.75E-04\t3.54E-04\t2.41E-03\t2.95E-02\t1.19E-03\t4.63E-03\t3.01E-04\t3.92E-03\t2.90E-02\t6.31E-03\t8.77E-03\t2.80E-03\t7.28E-04\t0.010652827921\t0.070288409275\r",
 "T>A\tCTA\t4.30E-05\t2.43E-04\t7.94E-03\t1.44E-02\t3.54E-03\t6.66E-05\t1.77E-04\t1.12E-03\t3.47E-03\t1.61E-03\t1.90E-02\t4.79E-03\t2.23E-16\t7.09E-05\t1.84E-04\t5.46E-04\t2.09E-04\t1.55E-04\t3.48E-04\t3.21E-03\t4.34E-04\t9.36E-05\t5.42E-04\t9.60E-05\t1.44E-04\t6.19E-06\t1.28E-01\t4.62E-04\t1.32E-03\t2.63E-02\t9.44E-04\t3.31E-01\t1.40E-04\t1.95E-03\t2.14E-04\t2.06E-02\t3.25E-04\t7.46E-04\t2.37E-02\t5.26E-02\t1.26E-03\t1.57E-03\t8.97E-04\t4.94E-03\t1.09E-03\t3.62E-03\t1.26E-02\t1.36E-03\t1.24E-05\t6.97E-04\t4.20E-03\t1.40E-02\t8.62E-05\t7.41E-05\t1.93E-04\t6.11E-03\t1.22E-03\t3.69E-03\t7.45E-18\t7.39E-05\t1.19E-03\t6.39E-03\t2.48E-03\t7.03E-05\t6.64E-05\t0.000136550096\t0.009896320186\r",
 "T>A\tCTC\t3.93E-04\t2.44E-04\t1.78E-02\t1.22E-02\t5.19E-03\t5.51E-04\t2.61E-04\t9.58E-04\t1.34E-02\t8.13E-05\t2.92E-02\t3.13E-03\t2.23E-16\t1.06E-04\t2.12E-04\t5.60E-03\t3.42E-04\t2.95E-03\t2.44E-04\t5.70E-03\t4.05E-02\t2.24E-03\t3.06E-03\t4.03E-04\t6.48E-03\t6.88E-04\t5.51E-02\t4.47E-18\t1.87E-03\t1.58E-02\t1.45E-03\t2.81E-03\t1.35E-03\t1.41E-03\t7.77E-04\t2.11E-02\t6.50E-04\t9.92E-04\t4.09E-04\t3.30E-02\t1.46E-03\t1.16E-03\t2.02E-03\t6.44E-03\t7.62E-03\t3.19E-03\t1.72E-03\t1.21E-05\t3.25E-03\t1.04E-03\t3.30E-03\t5.71E-04\t1.02E-04\t5.14E-05\t5.35E-04\t1.49E-03\t2.31E-04\t4.50E-03\t7.32E-04\t1.10E-06\t9.74E-04\t6.59E-03\t2.23E-02\t1.64E-03\t5.14E-04\t0.000000000000\t0.001136017362\r",
 "T>A\tCTG\t3.24E-04\t1.49E-04\t1.32E-02\t3.25E-02\t5.01E-03\t1.15E-04\t1.12E-04\t8.83E-04\t9.61E-04\t1.20E-03\t2.76E-02\t5.31E-03\t2.23E-16\t2.22E-16\t3.05E-04\t1.41E-03\t2.26E-04\t3.17E-04\t8.20E-05\t1.03E-03\t1.24E-02\t7.17E-04\t3.21E-03\t2.94E-04\t1.23E-04\t1.25E-04\t1.68E-01\t1.32E-04\t5.03E-03\t6.29E-02\t1.14E-03\t4.40E-03\t8.06E-04\t1.98E-03\t5.87E-04\t1.87E-02\t1.41E-03\t4.79E-04\t1.17E-03\t5.82E-02\t2.14E-03\t1.60E-03\t1.78E-03\t6.80E-03\t3.46E-03\t3.12E-03\t1.49E-02\t2.96E-04\t5.82E-04\t3.41E-03\t4.30E-03\t5.82E-05\t3.10E-04\t2.32E-04\t5.16E-04\t6.52E-03\t1.23E-03\t3.30E-03\t1.92E-03\t1.60E-02\t1.59E-03\t1.29E-03\t8.58E-04\t1.39E-04\t3.81E-07\t0.000125175761\t0.000178582568\r",
 "T>A\tCTT\t2.60E-04\t2.23E-16\t1.27E-02\t1.18E-02\t4.30E-03\t1.79E-04\t3.71E-04\t1.12E-04\t4.87E-02\t1.15E-03\t4.18E-02\t4.44E-03\t3.76E-04\t2.34E-04\t1.62E-04\t1.50E-03\t5.23E-04\t1.38E-03\t1.05E-03\t6.53E-03\t7.23E-02\t7.79E-03\t2.26E-03\t2.18E-04\t1.02E-04\t2.22E-16\t6.03E-02\t3.75E-04\t4.80E-03\t3.66E-02\t6.42E-04\t5.43E-03\t1.03E-03\t1.98E-03\t6.12E-04\t3.10E-02\t6.63E-04\t1.12E-03\t2.68E-04\t6.64E-02\t1.96E-03\t2.06E-03\t2.36E-03\t8.27E-03\t9.28E-03\t9.01E-03\t7.57E-03\t2.03E-04\t1.38E-05\t1.34E-03\t3.60E-03\t7.60E-05\t3.62E-04\t1.06E-04\t2.09E-04\t3.62E-03\t8.56E-04\t8.61E-03\t6.66E-05\t5.23E-03\t6.09E-06\t1.07E-02\t1.30E-03\t3.71E-03\t1.32E-04\t0.000000000000\t0.010551155388\r",
 "T>A\tGTA\t8.12E-05\t2.23E-16\t7.06E-03\t9.22E-03\t3.67E-03\t5.10E-05\t4.01E-04\t5.26E-04\t1.73E-03\t2.37E-05\t1.28E-02\t2.65E-03\t3.31E-06\t2.22E-16\t1.54E-06\t8.26E-04\t6.70E-06\t5.77E-05\t5.11E-04\t9.40E-04\t6.46E-05\t9.73E-05\t1.60E-03\t1.89E-04\t5.56E-05\t1.17E-03\t5.37E-02\t6.26E-18\t1.57E-03\t3.62E-02\t7.83E-04\t1.10E-01\t1.16E-04\t3.43E-03\t3.29E-04\t1.90E-03\t8.79E-03\t8.10E-04\t1.90E-02\t7.17E-04\t8.80E-04\t1.15E-03\t8.93E-04\t4.96E-03\t2.65E-03\t2.33E-03\t1.20E-02\t6.18E-03\t1.16E-03\t3.00E-04\t6.10E-03\t1.60E-02\t1.30E-04\t5.04E-05\t7.32E-04\t2.04E-02\t5.24E-04\t4.45E-03\t2.33E-03\t2.43E-03\t4.20E-04\t3.33E-03\t2.07E-03\t2.55E-04\t1.09E-04\t0.001098638590\t0.017239232975\r",
 "T>A\tGTC\t1.29E-04\t1.31E-04\t7.48E-03\t3.60E-03\t2.91E-03\t4.89E-04\t2.24E-04\t2.56E-04\t2.27E-03\t1.12E-04\t1.35E-02\t2.12E-03\t2.23E-16\t1.23E-04\t3.75E-04\t1.81E-03\t1.30E-04\t2.73E-03\t5.89E-04\t6.08E-04\t3.40E-03\t3.47E-04\t1.95E-03\t7.78E-05\t2.56E-03\t3.26E-03\t1.02E-02\t1.40E-04\t2.35E-03\t5.85E-03\t1.15E-03\t2.35E-05\t1.18E-04\t7.89E-04\t1.82E-04\t2.91E-03\t8.22E-03\t3.62E-04\t2.42E-03\t5.56E-03\t6.31E-04\t7.74E-04\t1.12E-03\t3.52E-03\t3.43E-03\t2.29E-03\t1.84E-03\t1.33E-03\t2.36E-03\t6.38E-04\t1.00E-03\t2.28E-04\t1.24E-04\t9.10E-05\t1.05E-03\t3.85E-03\t3.99E-04\t2.70E-03\t8.99E-03\t1.85E-04\t2.37E-03\t5.97E-04\t6.26E-04\t1.98E-03\t1.10E-04\t0.000065576983\t0.001005109264\r",
 "T>A\tGTG\t2.46E-04\t6.38E-05\t1.16E-02\t1.31E-02\t4.11E-03\t1.95E-04\t2.90E-04\t6.56E-04\t2.22E-16\t1.35E-03\t1.56E-02\t7.88E-04\t7.30E-05\t2.22E-16\t1.19E-04\t2.51E-03\t4.70E-05\t1.20E-04\t1.87E-04\t5.55E-03\t6.95E-04\t2.41E-04\t1.83E-03\t4.86E-04\t2.13E-03\t7.42E-04\t3.75E-02\t4.68E-18\t2.49E-03\t3.82E-02\t6.08E-04\t1.57E-03\t7.17E-05\t3.66E-03\t9.86E-05\t5.77E-03\t1.31E-02\t8.27E-04\t6.53E-03\t1.12E-03\t1.17E-03\t1.08E-03\t1.21E-03\t6.67E-03\t4.41E-03\t1.43E-03\t7.77E-03\t4.50E-03\t2.21E-03\t1.03E-03\t2.40E-03\t1.31E-05\t1.43E-04\t1.10E-04\t1.99E-04\t4.29E-03\t1.29E-03\t3.96E-03\t2.64E-03\t1.25E-02\t4.36E-04\t2.46E-03\t2.04E-02\t5.05E-04\t3.60E-04\t0.000337380913\t0.003520166086\r",
 "T>A\tGTT\t2.58E-04\t6.28E-05\t1.23E-02\t5.52E-03\t3.22E-03\t2.48E-04\t3.06E-04\t5.59E-05\t2.03E-02\t8.76E-03\t2.81E-02\t2.56E-03\t4.26E-04\t1.08E-04\t1.64E-04\t9.15E-04\t3.58E-04\t8.57E-04\t8.40E-04\t9.66E-04\t4.80E-03\t1.58E-03\t3.46E-03\t2.83E-04\t4.30E-03\t9.37E-04\t9.32E-03\t5.78E-18\t1.31E-03\t2.02E-02\t8.34E-04\t0.00E+00\t3.24E-04\t1.07E-03\t1.91E-04\t7.89E-04\t6.16E-03\t1.76E-03\t7.48E-03\t2.86E-03\t1.12E-03\t1.68E-03\t1.84E-03\t4.09E-03\t5.77E-03\t4.52E-03\t3.25E-03\t1.31E-03\t3.18E-03\t5.10E-04\t1.20E-03\t3.28E-06\t2.31E-04\t1.31E-04\t7.86E-04\t1.46E-04\t8.11E-04\t3.11E-03\t1.85E-03\t5.12E-03\t5.37E-03\t7.93E-03\t2.77E-03\t1.51E-03\t6.37E-04\t0.000135756563\t0.011658501206\r",
 "T>A\tTTA\t6.72E-03\t4.22E-04\t6.51E-03\t8.88E-03\t6.31E-03\t4.54E-04\t3.56E-04\t3.24E-03\t1.54E-02\t1.67E-02\t1.15E-02\t2.60E-02\t2.04E-03\t3.06E-03\t1.86E-04\t3.75E-03\t2.96E-04\t2.00E-04\t1.09E-03\t1.23E-02\t1.19E-04\t1.19E-03\t1.08E-02\t4.94E-04\t1.20E-02\t2.57E-04\t7.18E-02\t7.77E-18\t4.12E-03\t2.18E-02\t1.13E-03\t5.28E-02\t1.88E-04\t7.64E-04\t2.82E-04\t6.70E-03\t5.12E-04\t4.10E-03\t2.72E-01\t4.30E-03\t2.46E-03\t2.45E-03\t2.44E-03\t6.69E-03\t7.13E-03\t8.06E-02\t8.50E-03\t7.63E-05\t6.96E-03\t4.07E-04\t4.50E-03\t1.56E-01\t2.35E-05\t4.20E-04\t1.69E-03\t2.01E-02\t1.05E-03\t3.10E-03\t1.55E-03\t3.61E-03\t9.42E-03\t3.02E-03\t1.19E-02\t2.09E-03\t1.13E-03\t0.006569907486\t0.060179901809\r",
 "T>A\tTTC\t2.23E-16\t1.92E-04\t1.05E-02\t4.31E-03\t4.14E-03\t1.14E-04\t3.89E-04\t2.24E-03\t3.42E-02\t1.23E-03\t1.95E-02\t1.85E-03\t3.60E-04\t6.65E-04\t2.02E-04\t6.45E-03\t2.32E-04\t1.73E-03\t1.57E-04\t4.67E-03\t5.70E-03\t2.42E-04\t2.73E-03\t4.75E-04\t3.18E-04\t7.95E-04\t2.36E-02\t5.27E-18\t3.27E-03\t3.41E-03\t1.26E-03\t4.71E-05\t1.25E-03\t1.93E-03\t2.41E-04\t3.75E-03\t2.45E-03\t2.17E-04\t2.22E-16\t5.30E-03\t1.02E-03\t1.13E-03\t9.28E-04\t3.69E-03\t1.03E-02\t1.43E-02\t2.26E-03\t9.38E-04\t2.07E-05\t2.62E-04\t4.30E-03\t1.23E-05\t7.64E-06\t5.70E-05\t6.64E-04\t2.58E-03\t4.69E-04\t3.56E-03\t1.79E-04\t6.95E-05\t8.05E-04\t9.23E-03\t8.98E-04\t1.86E-03\t2.23E-16\t0.000864507132\t0.007794895232\r",
 "T>A\tTTG\t2.80E-05\t5.01E-05\t6.71E-03\t9.54E-03\t4.79E-03\t2.95E-05\t2.23E-16\t1.70E-03\t1.22E-02\t5.60E-03\t1.29E-02\t1.67E-02\t6.93E-04\t7.06E-05\t2.01E-04\t2.77E-03\t1.88E-04\t3.57E-04\t1.49E-04\t5.18E-03\t1.83E-03\t2.23E-16\t1.79E-03\t3.73E-04\t1.76E-04\t9.85E-05\t7.32E-02\t1.85E-04\t1.99E-03\t1.02E-02\t2.78E-04\t0.00E+00\t3.21E-04\t9.24E-04\t1.90E-04\t3.27E-03\t1.76E-03\t6.60E-04\t7.77E-03\t1.38E-03\t8.72E-04\t1.28E-03\t8.04E-04\t5.46E-03\t5.02E-03\t9.29E-03\t5.33E-03\t5.14E-05\t5.02E-06\t6.54E-04\t5.00E-04\t4.52E-04\t8.32E-06\t7.53E-05\t7.54E-05\t4.03E-03\t7.78E-04\t1.60E-03\t3.29E-04\t9.48E-03\t6.36E-03\t4.40E-03\t8.29E-04\t4.53E-04\t4.24E-05\t0.000784032307\t0.006733400083\r",
 "T>A\tTTT\t2.25E-03\t2.05E-04\t1.30E-02\t5.62E-03\t8.75E-03\t4.26E-04\t7.98E-04\t2.22E-16\t2.79E-01\t2.73E-02\t3.79E-02\t1.70E-02\t5.29E-03\t1.52E-03\t6.74E-04\t5.25E-03\t6.20E-05\t1.82E-03\t4.75E-04\t7.16E-03\t1.98E-03\t1.31E-03\t7.10E-03\t6.73E-04\t6.71E-03\t7.64E-05\t3.01E-02\t7.32E-18\t2.19E-03\t1.45E-02\t1.31E-03\t1.93E-03\t5.72E-03\t1.20E-03\t7.01E-04\t8.88E-03\t7.11E-04\t2.72E-03\t4.60E-02\t5.41E-03\t2.70E-03\t1.95E-03\t2.50E-03\t1.07E-02\t2.10E-02\t5.35E-02\t3.52E-03\t8.77E-18\t1.39E-02\t1.39E-03\t3.40E-03\t4.35E-03\t4.26E-04\t3.31E-04\t2.95E-03\t6.38E-03\t1.34E-03\t5.08E-03\t7.45E-18\t2.10E-03\t7.44E-04\t2.89E-02\t1.46E-02\t2.99E-03\t5.30E-04\t0.007059955557\t0.062798279518\r",
 "T>C\tATA\t1.09E-03\t6.17E-05\t1.65E-02\t7.95E-03\t4.62E-02\t2.21E-03\t4.90E-04\t1.02E-03\t2.88E-04\t4.36E-03\t9.77E-03\t2.03E-02\t7.45E-04\t9.35E-03\t1.14E-03\t7.36E-02\t1.91E-03\t1.26E-02\t3.89E-03\t2.42E-01\t9.65E-05\t9.73E-05\t2.02E-03\t2.59E-03\t1.07E-02\t1.67E-02\t2.50E-02\t8.86E-04\t6.40E-03\t2.07E-02\t8.48E-02\t5.05E-03\t5.49E-04\t1.96E-03\t6.11E-04\t2.04E-02\t9.36E-04\t1.25E-02\t4.62E-03\t2.04E-03\t5.25E-03\t7.33E-02\t2.79E-03\t1.12E-02\t1.87E-02\t1.44E-02\t1.16E-02\t8.88E-06\t1.93E-02\t1.37E-03\t1.85E-02\t2.83E-03\t1.10E-03\t6.13E-04\t2.63E-03\t2.59E-03\t3.75E-03\t1.13E-02\t2.92E-02\t4.35E-03\t1.25E-02\t2.66E-02\t8.05E-02\t9.95E-03\t1.29E-03\t0.000000000000\t0.029567020575\r",
 "T>C\tATC\t3.04E-03\t2.15E-05\t7.76E-03\t7.15E-04\t1.34E-02\t1.13E-03\t2.27E-04\t4.56E-04\t5.33E-04\t3.93E-03\t5.01E-03\t1.05E-02\t3.62E-03\t5.10E-04\t9.38E-04\t4.03E-02\t1.42E-04\t6.30E-03\t1.96E-03\t1.29E-02\t1.27E-02\t2.52E-04\t2.09E-03\t5.94E-04\t4.55E-03\t1.00E-02\t8.99E-04\t1.78E-04\t2.18E-03\t9.50E-03\t2.61E-02\t1.85E-03\t4.41E-04\t2.29E-03\t4.58E-04\t7.40E-03\t7.01E-04\t3.75E-03\t1.23E-03\t8.21E-04\t1.69E-03\t3.84E-02\t1.64E-03\t5.64E-03\t7.34E-03\t5.79E-03\t3.51E-03\t3.13E-02\t3.24E-03\t4.18E-04\t4.78E-02\t5.67E-03\t1.89E-04\t8.14E-05\t9.28E-03\t1.93E-02\t1.88E-03\t3.70E-03\t7.75E-03\t3.36E-06\t6.16E-03\t9.66E-03\t4.93E-03\t2.61E-03\t2.77E-04\t0.000323133451\t0.020410486307\r",
 "T>C\tATG\t1.06E-04\t1.32E-05\t1.23E-02\t5.00E-03\t3.82E-02\t4.24E-03\t1.87E-04\t1.50E-03\t1.07E-03\t9.19E-03\t6.98E-03\t1.43E-02\t2.14E-03\t5.31E-04\t3.20E-04\t7.05E-02\t4.28E-04\t1.24E-02\t2.40E-04\t6.39E-02\t3.76E-04\t2.23E-16\t2.08E-03\t2.31E-03\t1.86E-02\t3.88E-03\t7.75E-03\t4.70E-04\t1.76E-03\t1.99E-02\t7.85E-02\t5.69E-03\t4.05E-04\t2.75E-03\t2.77E-04\t8.12E-03\t1.09E-03\t7.34E-02\t1.95E-05\t9.20E-04\t3.84E-03\t1.68E-02\t2.17E-03\t9.64E-03\t1.55E-02\t8.04E-03\t8.56E-03\t5.86E-04\t2.72E-02\t2.07E-03\t3.29E-02\t2.09E-03\t2.85E-04\t2.00E-04\t1.13E-03\t3.95E-03\t2.62E-03\t8.25E-03\t2.02E-01\t4.02E-03\t9.55E-03\t1.03E-02\t6.75E-03\t3.52E-03\t3.52E-06\t0.000011504801\t0.005070940432\r",
 "T>C\tATT\t5.74E-03\t1.55E-04\t1.73E-02\t1.56E-03\t3.83E-02\t5.02E-04\t6.15E-04\t2.15E-03\t2.65E-03\t5.10E-02\t1.49E-02\t2.90E-02\t3.36E-03\t1.50E-04\t1.19E-03\t6.38E-02\t9.59E-04\t5.97E-03\t9.29E-04\t1.36E-01\t7.79E-03\t5.44E-04\t4.24E-03\t1.74E-03\t1.03E-04\t2.56E-02\t7.55E-04\t2.61E-04\t2.44E-03\t1.45E-02\t3.72E-02\t2.59E-03\t5.95E-04\t1.15E-03\t5.45E-04\t8.47E-03\t6.10E-04\t9.21E-03\t2.34E-03\t2.03E-03\t5.12E-03\t4.80E-02\t3.49E-03\t1.36E-02\t1.94E-02\t1.77E-02\t2.70E-03\t8.80E-03\t2.67E-04\t1.46E-03\t3.06E-02\t1.30E-03\t1.11E-03\t4.82E-04\t4.92E-03\t1.87E-03\t4.92E-03\t7.18E-03\t7.83E-03\t1.37E-02\t1.24E-02\t1.79E-02\t8.16E-03\t1.08E-02\t3.63E-05\t0.009382160052\t0.059234079950\r",
 "T>C\tCTA\t2.23E-16\t3.02E-04\t8.83E-03\t5.18E-03\t1.33E-02\t1.66E-03\t4.51E-04\t1.53E-03\t7.57E-03\t1.02E-02\t2.39E-03\t1.37E-02\t1.22E-04\t1.32E-03\t3.28E-04\t4.54E-02\t4.74E-05\t3.16E-03\t7.50E-04\t1.07E-02\t1.11E-02\t6.47E-04\t3.86E-04\t4.52E-04\t1.17E-02\t1.15E-02\t6.74E-03\t6.86E-18\t2.27E-03\t9.03E-03\t5.20E-02\t4.90E-03\t1.32E-03\t1.18E-03\t1.27E-03\t8.15E-03\t5.93E-04\t1.51E-02\t5.94E-04\t1.33E-03\t1.50E-03\t5.00E-02\t1.95E-03\t7.34E-03\t1.09E-02\t1.03E-02\t3.54E-03\t2.72E-03\t1.46E-02\t1.11E-03\t2.23E-02\t1.91E-03\t1.45E-04\t1.21E-04\t8.39E-04\t4.18E-03\t2.39E-03\t1.01E-02\t1.14E-02\t4.18E-03\t3.37E-03\t1.20E-02\t3.92E-03\t2.20E-03\t8.79E-04\t0.002269665991\t0.007184033893\r",
 "T>C\tCTC\t2.50E-03\t2.85E-05\t1.45E-02\t2.36E-03\t1.04E-02\t1.03E-03\t2.59E-04\t8.40E-04\t3.58E-03\t8.89E-05\t7.00E-03\t1.53E-02\t4.12E-05\t1.24E-04\t7.13E-04\t4.97E-02\t1.22E-04\t5.34E-03\t2.15E-03\t3.17E-03\t1.73E-01\t6.46E-03\t2.46E-03\t8.14E-05\t4.83E-03\t2.82E-03\t1.34E-03\t4.47E-18\t2.24E-03\t1.08E-02\t3.12E-02\t5.72E-03\t2.62E-03\t1.84E-03\t1.38E-03\t9.91E-03\t9.27E-04\t3.01E-02\t4.04E-05\t2.02E-03\t1.88E-03\t2.18E-02\t2.32E-03\t7.91E-03\t9.87E-03\t5.25E-03\t3.13E-03\t1.05E-01\t3.20E-03\t2.87E-03\t8.34E-02\t6.28E-03\t4.50E-04\t1.78E-04\t1.77E-02\t7.11E-02\t1.31E-03\t4.07E-03\t1.70E-02\t4.98E-03\t2.64E-03\t4.56E-03\t9.50E-03\t5.73E-03\t2.79E-03\t0.000003247249\t0.006209066076\r",
 "T>C\tCTG\t3.60E-04\t2.23E-16\t1.01E-02\t5.50E-03\t2.08E-02\t5.32E-03\t1.04E-04\t7.30E-04\t2.17E-03\t9.35E-04\t1.26E-03\t1.54E-02\t1.05E-03\t2.53E-05\t1.31E-04\t3.74E-02\t2.22E-16\t4.47E-03\t1.39E-03\t9.61E-03\t8.73E-02\t3.92E-03\t1.62E-03\t4.98E-04\t2.61E-02\t1.44E-02\t2.03E-03\t3.89E-04\t4.22E-03\t1.34E-02\t7.87E-02\t5.66E-03\t2.49E-03\t3.91E-03\t1.27E-03\t4.46E-03\t1.84E-03\t4.00E-02\t8.11E-04\t1.56E-03\t1.85E-03\t9.95E-03\t2.54E-03\t7.77E-03\t1.05E-02\t5.57E-03\t3.56E-03\t9.25E-03\t3.38E-02\t1.56E-03\t5.04E-02\t3.10E-03\t3.21E-05\t1.08E-04\t2.42E-03\t2.93E-03\t1.23E-03\t8.49E-03\t1.93E-01\t9.49E-04\t4.81E-04\t8.30E-03\t5.93E-03\t1.64E-03\t5.80E-04\t0.003628314755\t0.000839951905\r",
 "T>C\tCTT\t4.26E-05\t1.86E-04\t1.58E-02\t2.56E-03\t1.49E-02\t1.27E-03\t1.82E-03\t5.73E-03\t4.31E-02\t3.97E-02\t6.46E-03\t3.29E-02\t2.44E-06\t8.15E-04\t8.45E-05\t4.59E-02\t1.12E-04\t2.06E-03\t1.75E-03\t1.55E-02\t4.26E-01\t1.83E-02\t2.56E-03\t3.91E-04\t6.90E-03\t2.17E-02\t5.36E-04\t4.71E-05\t3.63E-03\t8.96E-03\t5.55E-02\t3.90E-03\t3.67E-04\t1.78E-03\t4.10E-03\t1.38E-02\t1.09E-03\t3.07E-03\t1.59E-03\t3.18E-03\t2.16E-03\t5.04E-02\t4.27E-03\t8.59E-03\t1.36E-02\t7.87E-03\t3.24E-03\t9.45E-03\t7.71E-03\t6.02E-04\t9.58E-02\t5.45E-03\t6.77E-05\t9.74E-05\t1.38E-03\t6.64E-03\t2.34E-03\t5.42E-03\t2.69E-02\t6.84E-04\t2.06E-04\t2.43E-02\t2.82E-02\t1.22E-02\t2.10E-03\t0.000918581801\t0.016913140039\r",
 "T>C\tGTA\t1.05E-03\t2.23E-16\t9.98E-03\t2.90E-03\t1.41E-02\t2.57E-03\t1.01E-04\t2.66E-04\t2.22E-16\t8.00E-04\t2.60E-03\t9.52E-03\t2.20E-03\t1.47E-02\t8.09E-04\t7.11E-02\t2.87E-04\t1.10E-02\t1.29E-02\t2.06E-03\t1.23E-04\t5.95E-05\t7.08E-04\t9.54E-04\t2.76E-03\t2.23E-01\t7.21E-04\t2.18E-04\t3.70E-03\t1.75E-02\t8.35E-02\t4.78E-03\t4.91E-04\t1.08E-03\t9.35E-04\t4.53E-03\t9.14E-03\t1.22E-02\t1.09E-04\t5.65E-04\t2.34E-03\t3.07E-02\t1.70E-03\t7.47E-03\t6.01E-03\t6.06E-03\t5.82E-03\t7.50E-18\t2.70E-02\t6.54E-04\t2.21E-02\t4.17E-04\t1.57E-04\t6.59E-05\t1.38E-03\t1.18E-03\t2.42E-03\t5.61E-03\t1.14E-02\t2.61E-03\t7.30E-04\t3.86E-03\t2.18E-03\t7.10E-04\t7.61E-04\t0.000000000000\t0.021590706956\r",
 "T>C\tGTC\t1.90E-03\t2.74E-05\t5.64E-03\t1.18E-03\t7.94E-03\t1.11E-03\t7.41E-05\t2.22E-16\t4.23E-04\t2.32E-03\t2.62E-03\t9.78E-03\t6.28E-03\t5.33E-03\t8.60E-04\t3.04E-02\t1.82E-04\t6.54E-03\t1.16E-02\t1.19E-03\t1.49E-02\t1.50E-03\t2.26E-03\t3.71E-04\t2.26E-03\t6.63E-02\t1.24E-03\t4.18E-04\t3.65E-03\t4.05E-03\t2.48E-02\t1.96E-03\t5.87E-04\t3.59E-03\t5.55E-04\t6.57E-03\t7.67E-03\t7.94E-03\t6.46E-04\t2.03E-03\t1.40E-03\t1.82E-02\t2.16E-03\t6.89E-03\t4.58E-03\t3.74E-03\t3.31E-03\t2.19E-02\t1.16E-03\t5.70E-04\t5.76E-02\t1.09E-03\t1.29E-04\t1.51E-04\t5.38E-03\t2.27E-03\t2.05E-03\t6.68E-03\t4.43E-03\t3.17E-03\t2.73E-05\t2.29E-03\t7.80E-04\t3.18E-03\t6.35E-04\t0.000865945066\t0.011606134984\r",
 "T>C\tGTG\t1.17E-03\t1.07E-04\t1.09E-02\t2.37E-03\t1.50E-02\t2.45E-03\t7.25E-05\t5.29E-04\t2.32E-04\t2.57E-03\t2.81E-03\t6.31E-03\t2.23E-16\t1.38E-03\t5.61E-04\t4.15E-02\t1.95E-04\t1.10E-02\t5.95E-03\t4.28E-03\t3.57E-03\t1.01E-03\t3.98E-03\t9.53E-04\t4.06E-03\t1.17E-01\t5.81E-04\t4.68E-18\t2.67E-03\t1.34E-02\t7.12E-02\t1.23E-03\t7.31E-04\t2.99E-03\t5.01E-04\t5.13E-03\t9.62E-03\t2.07E-01\t4.40E-04\t1.91E-04\t1.84E-03\t9.83E-03\t1.87E-03\t5.74E-03\t5.65E-03\t3.51E-03\t3.21E-03\t1.20E-03\t1.73E-02\t9.07E-04\t3.53E-02\t3.10E-03\t4.67E-05\t6.53E-05\t3.71E-03\t4.56E-03\t9.14E-04\t2.56E-03\t1.21E-01\t6.01E-03\t1.65E-03\t4.72E-03\t1.11E-02\t9.86E-04\t5.00E-04\t0.000197907041\t0.006462687955\r",
 "T>C\tGTT\t7.13E-05\t1.25E-05\t1.02E-02\t7.55E-04\t1.37E-02\t9.06E-04\t8.11E-04\t6.85E-03\t7.07E-03\t3.18E-01\t6.29E-03\t9.86E-03\t5.29E-03\t1.15E-02\t2.37E-04\t4.63E-02\t2.22E-16\t1.12E-02\t5.86E-03\t2.40E-03\t5.24E-03\t1.47E-03\t3.15E-03\t2.36E-03\t9.28E-04\t1.33E-01\t8.34E-04\t7.99E-04\t3.99E-03\t1.42E-02\t5.52E-02\t3.75E-03\t8.51E-04\t1.55E-03\t4.45E-04\t4.69E-03\t1.29E-02\t9.71E-03\t2.22E-16\t4.21E-04\t2.65E-03\t4.00E-02\t3.05E-03\t7.51E-03\t7.31E-03\t3.89E-03\t2.39E-03\t5.18E-04\t1.11E-02\t1.39E-03\t6.07E-02\t3.19E-03\t1.10E-04\t1.75E-04\t1.17E-03\t4.01E-03\t3.11E-03\t6.99E-03\t5.41E-03\t1.26E-02\t2.58E-03\t5.36E-03\t2.02E-03\t1.19E-03\t6.96E-04\t0.003662473198\t0.054089001127\r",
 "T>C\tTTA\t2.55E-04\t2.23E-16\t1.52E-02\t2.41E-03\t1.80E-02\t1.48E-03\t2.15E-03\t6.00E-03\t3.02E-02\t8.12E-02\t2.08E-03\t1.40E-02\t7.30E-04\t8.86E-03\t1.13E-04\t8.61E-02\t2.21E-05\t6.27E-03\t3.42E-03\t1.30E-02\t2.09E-03\t1.06E-03\t1.80E-03\t1.24E-03\t1.38E-02\t8.42E-02\t4.63E-03\t7.77E-18\t1.46E-03\t1.45E-02\t4.23E-02\t1.72E-03\t1.31E-03\t2.59E-03\t5.44E-04\t4.59E-03\t8.97E-04\t1.10E-02\t1.40E-03\t3.68E-04\t2.97E-03\t4.77E-02\t2.11E-03\t5.94E-03\t1.02E-02\t6.36E-02\t6.82E-03\t1.37E-04\t1.99E-02\t1.28E-03\t9.30E-03\t1.37E-03\t6.05E-04\t1.73E-04\t1.09E-03\t4.46E-03\t2.77E-03\t7.11E-03\t4.73E-03\t4.88E-03\t5.68E-04\t4.90E-03\t3.31E-03\t2.05E-03\t2.40E-04\t0.000432939443\t0.012963549217\r",
 "T>C\tTTC\t3.39E-03\t4.35E-05\t8.99E-03\t5.03E-04\t9.71E-03\t1.09E-03\t5.96E-04\t1.73E-03\t2.09E-02\t1.43E-02\t2.64E-03\t1.36E-02\t5.17E-03\t8.10E-05\t4.67E-04\t4.63E-02\t2.49E-05\t2.88E-03\t5.92E-03\t3.67E-03\t1.53E-02\t1.10E-03\t6.18E-03\t5.53E-04\t1.66E-02\t5.48E-02\t2.22E-16\t1.90E-04\t3.00E-03\t7.41E-03\t3.14E-02\t3.49E-03\t2.05E-03\t1.50E-03\t2.86E-04\t1.03E-02\t1.22E-03\t1.25E-02\t6.24E-03\t8.22E-04\t2.28E-03\t4.51E-02\t2.19E-03\t6.44E-03\t7.55E-03\t2.68E-02\t3.03E-03\t3.59E-02\t1.05E-02\t1.10E-03\t5.43E-02\t1.75E-02\t4.04E-04\t8.47E-05\t2.90E-02\t3.93E-03\t2.18E-03\t5.03E-03\t8.19E-03\t9.80E-04\t1.32E-03\t2.67E-02\t1.96E-02\t2.24E-03\t1.09E-03\t0.000538464673\t0.023324448028\r",
 "T>C\tTTG\t4.16E-04\t1.17E-04\t6.93E-03\t1.74E-03\t1.27E-02\t2.13E-03\t3.19E-04\t3.81E-03\t1.29E-02\t2.28E-02\t2.22E-16\t9.31E-03\t3.63E-04\t6.68E-04\t1.55E-04\t4.00E-02\t9.98E-05\t4.31E-03\t1.32E-03\t7.23E-03\t2.35E-03\t1.21E-03\t3.26E-03\t1.08E-03\t1.20E-02\t2.76E-02\t1.22E-03\t3.02E-04\t2.03E-03\t4.89E-03\t3.03E-02\t1.15E-03\t1.41E-03\t3.79E-04\t2.03E-04\t4.01E-03\t1.69E-03\t3.20E-01\t1.10E-03\t4.38E-04\t1.38E-03\t1.00E-02\t1.92E-03\t4.98E-03\t6.01E-03\t2.05E-02\t3.41E-03\t2.78E-05\t1.91E-02\t1.24E-03\t2.37E-02\t1.58E-03\t2.97E-04\t6.31E-05\t1.02E-03\t2.47E-03\t1.07E-03\t4.72E-03\t8.84E-02\t4.16E-03\t1.85E-03\t2.96E-03\t2.16E-03\t1.35E-03\t1.09E-04\t0.000022365153\t0.003823534542\r",
 "T>C\tTTT\t4.33E-03\t3.58E-05\t1.39E-02\t6.36E-04\t1.85E-02\t1.74E-03\t8.12E-04\t2.97E-03\t8.81E-02\t4.31E-02\t4.57E-03\t3.84E-02\t3.25E-03\t2.51E-05\t1.06E-04\t5.97E-02\t5.31E-04\t2.92E-03\t1.35E-03\t8.39E-03\t2.10E-02\t1.00E-03\t3.37E-03\t1.47E-03\t1.93E-02\t4.07E-02\t2.22E-16\t7.32E-18\t3.67E-03\t1.70E-02\t5.74E-02\t1.53E-03\t2.06E-03\t8.22E-04\t4.45E-04\t1.21E-02\t6.84E-04\t9.12E-03\t7.08E-03\t1.06E-03\t2.80E-03\t6.14E-02\t3.06E-03\t9.48E-03\t1.02E-02\t4.61E-02\t2.36E-03\t1.84E-03\t3.71E-02\t5.45E-04\t3.65E-02\t1.15E-02\t5.32E-04\t2.08E-04\t5.11E-03\t2.13E-02\t2.00E-03\t2.19E-03\t1.25E-02\t8.87E-04\t2.85E-04\t1.95E-01\t3.20E-02\t2.61E-03\t1.32E-03\t0.009820763708\t0.094054377313\r",
 "T>G\tATA\t1.72E-04\t2.38E-04\t3.95E-03\t1.05E-03\t3.86E-03\t2.57E-05\t4.73E-04\t8.44E-05\t1.19E-02\t3.70E-03\t1.64E-03\t4.51E-02\t1.92E-03\t3.47E-05\t2.18E-04\t2.01E-03\t6.14E-05\t2.12E-04\t4.68E-04\t1.12E-02\t8.51E-04\t1.41E-03\t7.21E-04\t2.33E-03\t1.05E-05\t3.92E-04\t2.33E-03\t8.28E-18\t3.21E-03\t5.28E-03\t2.92E-03\t1.14E-03\t9.87E-03\t4.07E-04\t3.90E-05\t3.68E-03\t3.86E-04\t9.79E-04\t2.82E-03\t3.05E-03\t1.09E-03\t2.17E-02\t1.15E-03\t3.88E-03\t7.23E-03\t7.53E-03\t1.20E-03\t2.63E-03\t2.40E-08\t1.17E-04\t1.00E-04\t1.71E-04\t5.17E-04\t1.96E-04\t7.43E-04\t7.23E-04\t1.73E-03\t4.59E-03\t3.87E-05\t2.62E-04\t1.21E-04\t7.06E-03\t1.35E-02\t1.79E-03\t2.50E-02\t0.000127137772\t0.023838221813\r",
 "T>G\tATC\t2.07E-04\t7.46E-05\t2.60E-03\t1.59E-04\t2.60E-03\t2.04E-04\t2.47E-04\t9.32E-04\t1.05E-03\t2.66E-03\t6.68E-04\t6.93E-03\t1.24E-03\t5.33E-05\t4.67E-05\t6.64E-04\t2.22E-16\t2.97E-03\t4.15E-04\t4.41E-03\t7.00E-04\t4.48E-03\t8.58E-04\t1.89E-03\t2.13E-05\t4.37E-04\t5.09E-05\t5.43E-18\t1.40E-03\t0.00E+00\t2.53E-03\t9.59E-04\t1.73E-02\t8.17E-04\t1.10E-04\t2.48E-03\t4.29E-04\t3.02E-04\t4.62E-04\t9.66E-04\t6.23E-04\t1.06E-02\t1.07E-03\t2.97E-03\t7.05E-03\t1.64E-03\t4.05E-04\t4.07E-03\t3.26E-05\t5.84E-04\t8.00E-04\t1.72E-03\t4.10E-05\t8.28E-05\t6.07E-04\t4.09E-03\t5.29E-04\t2.90E-03\t1.58E-03\t9.20E-04\t7.99E-18\t7.89E-04\t7.88E-04\t1.56E-03\t1.05E-02\t0.001270955861\t0.003434920439\r",
 "T>G\tATG\t2.68E-04\t2.05E-06\t6.30E-03\t1.42E-03\t7.94E-03\t1.07E-04\t2.76E-04\t1.47E-03\t6.04E-04\t1.06E-02\t4.77E-03\t5.33E-03\t3.20E-04\t5.23E-04\t1.79E-04\t8.08E-04\t8.25E-05\t7.09E-04\t1.85E-04\t4.62E-03\t7.19E-04\t9.90E-04\t1.99E-03\t1.77E-03\t2.69E-05\t1.87E-04\t2.95E-03\t5.68E-18\t2.06E-03\t7.14E-03\t2.29E-03\t6.77E-05\t7.68E-03\t4.20E-04\t4.10E-04\t4.02E-03\t1.27E-03\t4.75E-04\t1.97E-04\t4.48E-03\t9.36E-04\t1.68E-02\t2.07E-03\t7.66E-03\t8.06E-03\t2.70E-03\t6.37E-04\t3.49E-02\t7.47E-06\t9.84E-04\t1.60E-03\t3.68E-03\t3.51E-04\t1.26E-04\t3.94E-03\t2.78E-02\t1.19E-03\t5.06E-03\t1.15E-03\t1.45E-01\t1.27E-05\t1.23E-03\t2.71E-03\t1.70E-03\t2.61E-02\t0.000952698782\t0.004485500819\r",
 "T>G\tATT\t1.12E-04\t3.76E-06\t3.97E-03\t2.16E-04\t4.83E-03\t3.86E-04\t3.09E-04\t7.33E-04\t2.22E-16\t1.43E-02\t7.40E-04\t2.39E-02\t9.22E-03\t2.22E-16\t2.85E-04\t1.28E-03\t3.20E-04\t1.11E-02\t3.63E-03\t7.96E-03\t1.25E-02\t6.91E-02\t1.84E-03\t1.62E-03\t5.82E-05\t6.95E-05\t4.18E-05\t7.43E-18\t4.71E-03\t0.00E+00\t5.93E-03\t2.56E-04\t1.15E-01\t8.10E-04\t7.94E-04\t7.83E-04\t6.45E-18\t7.58E-04\t1.15E-03\t2.63E-03\t9.84E-04\t2.47E-02\t2.86E-03\t6.42E-03\t1.80E-02\t7.61E-03\t9.11E-04\t5.64E-03\t1.42E-03\t1.66E-04\t6.00E-04\t8.08E-03\t1.99E-04\t1.26E-04\t1.03E-04\t2.13E-03\t1.86E-03\t2.06E-03\t3.40E-04\t2.28E-03\t5.75E-03\t5.97E-03\t2.08E-03\t3.44E-03\t1.64E-02\t0.001764975843\t0.003178596391\r",
 "T>G\tCTA\t3.55E-05\t1.53E-04\t4.37E-03\t8.70E-04\t2.59E-03\t4.95E-05\t8.42E-05\t3.59E-04\t3.28E-03\t1.48E-03\t1.22E-03\t2.60E-02\t1.55E-04\t2.27E-04\t2.22E-16\t8.93E-04\t4.14E-05\t4.85E-05\t2.06E-04\t3.96E-03\t3.10E-04\t2.05E-03\t1.11E-04\t7.14E-04\t9.63E-04\t4.35E-06\t4.91E-03\t6.85E-18\t1.43E-03\t0.00E+00\t1.42E-03\t0.00E+00\t1.46E-02\t7.81E-04\t1.98E-04\t1.97E-03\t5.81E-05\t7.28E-04\t1.06E-03\t3.71E-03\t4.78E-04\t2.60E-03\t9.83E-04\t5.30E-03\t3.35E-03\t3.47E-03\t8.70E-04\t2.73E-03\t2.34E-04\t1.02E-04\t0.00E+00\t1.37E-04\t1.34E-04\t1.06E-04\t8.94E-04\t2.78E-03\t8.03E-04\t1.49E-03\t4.36E-03\t1.65E-03\t5.82E-03\t1.40E-04\t5.03E-04\t1.01E-03\t3.23E-03\t0.000897193499\t0.003736474707\r",
 "T>G\tCTC\t2.12E-04\t1.55E-04\t7.04E-03\t4.41E-04\t4.61E-03\t7.84E-04\t1.72E-04\t7.92E-04\t5.58E-03\t4.98E-05\t1.95E-03\t7.41E-03\t2.23E-16\t1.15E-03\t2.22E-16\t1.14E-03\t2.82E-06\t1.21E-03\t1.57E-04\t1.08E-03\t1.04E-03\t3.14E-02\t1.99E-03\t2.09E-03\t1.60E-03\t5.44E-05\t8.80E-04\t1.48E-04\t8.80E-04\t2.89E-03\t4.94E-03\t3.81E-04\t2.21E-02\t5.69E-06\t2.15E-04\t1.43E-03\t4.50E-04\t9.49E-04\t2.22E-16\t5.18E-04\t5.66E-04\t2.38E-03\t1.41E-03\t7.39E-03\t5.67E-03\t2.41E-03\t7.53E-04\t7.26E-03\t2.53E-03\t2.81E-04\t1.80E-03\t5.15E-05\t9.28E-05\t2.67E-05\t6.18E-05\t2.56E-03\t7.75E-04\t4.24E-03\t1.89E-02\t4.66E-04\t3.93E-07\t8.47E-04\t1.04E-03\t2.62E-03\t4.16E-03\t0.002148873221\t0.002478170057\r",
 "T>G\tCTG\t1.28E-04\t1.21E-04\t1.07E-02\t3.79E-03\t6.13E-03\t9.24E-04\t2.23E-16\t1.29E-03\t1.19E-03\t2.58E-05\t4.70E-03\t1.42E-02\t5.35E-05\t4.79E-04\t2.22E-16\t5.93E-04\t2.22E-16\t4.35E-04\t2.72E-04\t2.24E-03\t1.94E-06\t1.37E-02\t2.93E-03\t1.33E-03\t2.71E-03\t2.01E-05\t8.03E-03\t6.52E-05\t3.04E-03\t1.18E-02\t5.56E-03\t4.03E-04\t1.55E-02\t3.32E-04\t7.85E-05\t1.50E-03\t1.09E-03\t1.07E-03\t4.36E-04\t6.36E-03\t8.87E-04\t3.77E-03\t1.59E-03\t9.83E-03\t5.59E-03\t3.04E-03\t5.68E-03\t4.65E-02\t3.47E-03\t6.49E-04\t1.60E-03\t3.67E-03\t1.04E-04\t1.12E-04\t3.66E-03\t5.93E-03\t7.96E-04\t2.98E-03\t2.21E-02\t1.13E-01\t4.45E-03\t2.19E-03\t6.99E-04\t2.02E-03\t7.11E-03\t0.000788671403\t0.001120299176\r",
 "T>G\tCTT\t1.71E-04\t2.03E-04\t7.03E-03\t1.20E-03\t7.30E-03\t1.30E-03\t2.65E-05\t3.68E-06\t3.40E-03\t1.56E-04\t5.74E-04\t5.65E-02\t1.30E-03\t5.23E-03\t1.33E-04\t3.44E-04\t5.55E-05\t9.62E-04\t2.49E-03\t2.54E-03\t5.31E-03\t5.50E-01\t4.14E-03\t2.20E-03\t5.11E-03\t1.85E-04\t1.19E-03\t2.47E-04\t4.71E-03\t8.55E-03\t7.80E-03\t1.44E-03\t1.01E-01\t7.67E-04\t2.65E-04\t9.49E-04\t6.76E-04\t1.92E-04\t6.55E-07\t2.71E-03\t1.07E-03\t4.11E-03\t6.88E-03\t4.94E-03\t1.31E-02\t1.05E-02\t3.48E-03\t6.26E-18\t3.84E-04\t7.28E-04\t2.00E-03\t1.07E-04\t6.57E-05\t9.29E-05\t4.71E-04\t2.30E-03\t8.63E-04\t2.72E-03\t8.86E-04\t1.18E-03\t6.78E-18\t5.16E-03\t4.73E-04\t2.12E-02\t1.80E-05\t0.000000000000\t0.001756870383\r",
 "T>G\tGTA\t2.23E-16\t2.09E-04\t4.42E-03\t5.98E-04\t2.52E-03\t4.35E-05\t9.06E-05\t3.23E-04\t2.48E-03\t7.48E-05\t1.31E-03\t8.30E-03\t2.04E-04\t4.81E-04\t9.57E-05\t4.12E-04\t8.77E-05\t7.79E-05\t2.03E-04\t9.02E-04\t4.95E-04\t2.81E-04\t8.70E-04\t6.36E-04\t2.22E-16\t2.86E-04\t1.59E-03\t6.26E-18\t1.07E-03\t5.55E-03\t1.37E-03\t0.00E+00\t1.16E-03\t4.49E-05\t1.88E-04\t3.04E-04\t1.15E-03\t5.81E-04\t3.65E-04\t1.76E-03\t1.85E-04\t1.18E-02\t5.04E-04\t3.33E-03\t2.49E-03\t1.80E-03\t1.14E-03\t6.11E-02\t7.68E-18\t4.40E-04\t0.00E+00\t2.20E-02\t7.64E-06\t6.61E-05\t8.67E-03\t2.19E-03\t1.13E-03\t4.38E-03\t1.70E-04\t5.67E-03\t6.72E-18\t1.40E-03\t8.07E-03\t1.23E-03\t2.82E-02\t0.000018879250\t0.007699623603\r",
 "T>G\tGTC\t2.23E-16\t1.33E-04\t2.34E-03\t2.53E-04\t1.71E-03\t2.13E-04\t1.21E-04\t3.21E-04\t1.66E-03\t2.23E-05\t6.56E-04\t5.85E-03\t8.33E-04\t9.42E-04\t2.22E-16\t2.13E-04\t2.22E-16\t2.10E-03\t5.80E-04\t9.01E-04\t2.04E-04\t6.25E-03\t3.70E-05\t7.03E-04\t6.65E-05\t5.33E-04\t2.36E-04\t8.69E-05\t1.53E-03\t0.00E+00\t1.21E-03\t5.08E-05\t1.93E-03\t9.17E-05\t3.53E-04\t8.72E-04\t7.40E-04\t6.37E-04\t2.22E-16\t5.07E-03\t3.13E-04\t6.86E-03\t9.39E-04\t5.80E-03\t2.24E-03\t9.50E-04\t7.05E-05\t7.56E-02\t1.83E-05\t5.20E-04\t1.00E-04\t1.04E-02\t1.35E-04\t4.37E-05\t9.59E-03\t5.30E-03\t5.61E-04\t1.74E-03\t1.53E-03\t3.20E-03\t9.19E-07\t1.71E-04\t2.69E-03\t1.65E-03\t2.88E-02\t0.001229435537\t0.000800067973\r",
 "T>G\tGTG\t3.48E-04\t3.71E-05\t1.09E-02\t2.43E-03\t5.30E-03\t2.74E-04\t3.25E-04\t2.41E-03\t6.03E-03\t3.80E-03\t3.97E-03\t3.97E-03\t2.23E-16\t1.73E-03\t1.30E-04\t4.96E-04\t4.26E-04\t7.81E-04\t3.47E-04\t6.75E-04\t8.35E-04\t1.48E-03\t5.83E-03\t5.10E-04\t8.70E-05\t2.13E-04\t4.50E-04\t1.57E-04\t3.13E-03\t8.69E-03\t2.35E-03\t4.74E-03\t3.13E-03\t7.70E-04\t1.38E-03\t2.55E-03\t3.01E-03\t5.42E-03\t5.65E-05\t4.04E-03\t1.53E-03\t9.83E-03\t2.16E-03\t7.72E-03\t4.45E-03\t1.20E-03\t4.10E-04\t2.41E-01\t1.03E-04\t1.79E-04\t1.20E-03\t2.72E-02\t6.63E-05\t1.47E-04\t3.51E-02\t7.62E-02\t1.37E-03\t4.73E-03\t7.45E-18\t2.99E-01\t6.72E-18\t8.61E-04\t2.34E-02\t2.29E-03\t6.62E-01\t0.000212751233\t0.001946513805\r",
 "T>G\tGTT\t1.46E-05\t2.23E-16\t5.83E-03\t2.52E-04\t2.35E-03\t7.87E-04\t8.35E-04\t1.83E-03\t6.75E-03\t6.84E-03\t1.29E-03\t1.94E-02\t2.21E-03\t5.16E-03\t1.54E-04\t1.65E-04\t2.89E-06\t2.25E-03\t1.39E-03\t2.39E-03\t7.24E-03\t1.14E-01\t2.14E-03\t3.41E-04\t3.56E-03\t2.59E-03\t1.96E-05\t5.78E-18\t2.68E-03\t2.77E-03\t6.05E-03\t2.30E-03\t2.06E-02\t1.15E-03\t1.60E-04\t2.81E-03\t7.85E-04\t4.51E-04\t4.17E-04\t3.35E-03\t4.89E-04\t1.71E-02\t3.34E-03\t5.94E-03\t5.48E-03\t3.47E-03\t9.20E-04\t7.46E-02\t2.80E-03\t4.96E-04\t6.00E-04\t8.84E-02\t1.15E-05\t1.02E-04\t3.65E-02\t2.09E-03\t8.64E-04\t2.03E-03\t1.21E-05\t1.44E-03\t1.14E-04\t2.83E-03\t1.69E-02\t7.45E-03\t4.74E-02\t0.001340078646\t0.002927905616\r",
 "T>G\tTTA\t2.23E-16\t1.67E-05\t7.25E-03\t3.77E-04\t5.22E-03\t1.05E-04\t1.28E-04\t9.55E-04\t1.93E-02\t2.11E-04\t2.22E-16\t6.57E-02\t2.69E-03\t1.13E-02\t8.51E-05\t1.63E-03\t1.84E-04\t9.92E-05\t2.31E-04\t9.78E-03\t2.22E-16\t1.22E-05\t6.96E-04\t1.66E-03\t2.22E-16\t4.65E-04\t5.06E-03\t1.86E-04\t1.23E-04\t2.08E-03\t1.43E-03\t1.19E-03\t4.23E-02\t4.67E-04\t1.18E-04\t2.25E-03\t2.26E-05\t4.68E-04\t2.22E-16\t4.29E-03\t7.88E-04\t1.83E-02\t1.17E-03\t8.39E-03\t8.46E-03\t4.19E-02\t1.15E-03\t2.04E-04\t7.68E-18\t2.85E-04\t7.00E-04\t2.41E-03\t2.50E-04\t1.50E-04\t3.27E-04\t2.55E-03\t1.43E-03\t1.30E-03\t3.24E-05\t6.06E-03\t9.46E-03\t3.22E-03\t7.98E-04\t1.60E-03\t1.61E-03\t0.001875698241\t0.007106411217\r",
 "T>G\tTTC\t5.51E-05\t7.04E-05\t6.28E-03\t1.74E-04\t6.56E-03\t2.87E-04\t1.16E-04\t1.55E-03\t1.74E-02\t1.15E-04\t1.14E-03\t8.62E-03\t2.23E-16\t5.55E-03\t6.73E-05\t1.13E-03\t2.22E-16\t1.20E-03\t2.94E-04\t5.51E-03\t1.16E-04\t8.75E-03\t2.10E-03\t2.02E-03\t2.22E-16\t3.79E-04\t7.52E-04\t5.27E-18\t2.64E-03\t5.79E-04\t1.72E-03\t2.80E-04\t3.57E-02\t9.93E-05\t9.64E-05\t2.38E-03\t2.24E-04\t1.64E-04\t2.22E-16\t1.94E-03\t7.44E-04\t1.95E-02\t9.03E-04\t5.08E-03\t1.14E-02\t1.55E-02\t2.74E-04\t8.25E-03\t7.68E-18\t1.35E-04\t1.00E-03\t2.25E-05\t2.22E-04\t1.38E-04\t6.55E-04\t3.47E-03\t1.12E-03\t4.36E-03\t7.06E-04\t5.65E-03\t1.00E-06\t3.96E-03\t1.59E-03\t2.33E-03\t1.06E-02\t0.000737916543\t0.006404453857\r",
 "T>G\tTTG\t5.83E-04\t9.54E-05\t8.05E-03\t2.32E-03\t6.94E-03\t3.24E-04\t2.23E-16\t1.35E-03\t7.64E-03\t1.25E-04\t3.09E-03\t1.09E-02\t2.16E-05\t2.76E-03\t1.01E-04\t8.75E-04\t3.67E-05\t1.14E-04\t1.48E-04\t5.24E-03\t9.20E-04\t4.78E-03\t1.45E-03\t2.07E-03\t2.79E-03\t1.31E-04\t5.57E-03\t1.82E-04\t3.04E-03\t9.43E-03\t2.84E-03\t2.35E-03\t1.40E-02\t1.80E-04\t8.13E-04\t5.35E-03\t8.18E-04\t9.59E-04\t1.52E-03\t4.82E-03\t9.02E-04\t3.02E-02\t1.90E-03\t7.28E-03\t8.66E-03\t1.94E-02\t2.93E-03\t2.28E-02\t9.17E-04\t7.08E-04\t9.00E-04\t1.54E-03\t3.50E-04\t1.12E-04\t8.72E-03\t2.76E-03\t1.48E-03\t6.15E-03\t7.69E-03\t7.95E-02\t1.01E-04\t5.40E-03\t6.74E-03\t1.96E-03\t2.39E-02\t0.000000000000\t0.003409999931\r",
 "T>G\tTTT\t2.23E-16\t2.23E-16\t1.05E-02\t5.68E-04\t1.35E-02\t1.01E-03\t8.29E-05\t1.77E-03\t2.17E-02\t1.66E-04\t9.96E-04\t6.39E-02\t1.89E-02\t9.09E-02\t5.55E-05\t2.21E-03\t1.89E-05\t6.05E-04\t5.99E-03\t1.58E-02\t4.58E-03\t1.22E-01\t5.16E-03\t4.23E-03\t1.51E-03\t2.22E-16\t5.95E-04\t7.32E-18\t3.96E-03\t7.87E-03\t9.45E-03\t1.40E-04\t4.79E-01\t9.28E-04\t8.89E-03\t7.30E-03\t6.35E-04\t2.25E-03\t2.32E-04\t3.62E-03\t1.81E-03\t2.91E-02\t5.29E-03\t1.09E-02\t2.50E-02\t8.79E-02\t1.74E-03\t1.32E-03\t9.00E-03\t7.24E-04\t1.00E-03\t3.62E-03\t3.20E-07\t3.64E-04\t1.60E-03\t2.91E-03\t2.53E-03\t7.05E-03\t4.99E-04\t4.12E-02\t2.90E-02\t1.19E-01\t2.17E-02\t1.30E-17\t3.18E-02\t0.006603134330\t0.012708761755\r")
# COSMIC_v3.2 
COSMIC_v3.2 <- c("Type\tSBS1\tSBS2\tSBS3\tSBS4\tSBS5\tSBS6\tSBS7a\tSBS7b\tSBS7c\tSBS7d\tSBS8\tSBS9\tSBS10a\tSBS10b\tSBS10c\tSBS10d\tSBS11\tSBS12\tSBS13\tSBS14\tSBS15\tSBS16\tSBS17a\tSBS17b\tSBS18\tSBS19\tSBS20\tSBS21\tSBS22\tSBS23\tSBS24\tSBS25\tSBS26\tSBS27\tSBS28\tSBS29\tSBS30\tSBS31\tSBS32\tSBS33\tSBS34\tSBS35\tSBS36\tSBS37\tSBS38\tSBS39\tSBS40\tSBS41\tSBS42\tSBS43\tSBS44\tSBS45\tSBS46\tSBS47\tSBS48\tSBS49\tSBS50\tSBS51\tSBS52\tSBS53\tSBS54\tSBS55\tSBS56\tSBS57\tSBS58\tSBS59\tSBS60\tSBS84\tSBS85\tSBS86\tSBS87\tSBS88\tSBS89\tSBS90\tSBS91\tSBS92\tSBS93\tSBS94",
                 "A[C>A]A\t0.000886157230877471\t5.80016751463797e-07\t0.0208083233293317\t0.0421964976906917\t0.011997600479904\t0.000425233006425871\t6.70435065427006e-05\t0.00232938611358363\t0.00483040594731581\t4.03937995517688e-05\t0.0440982184319753\t0.000557953689843743\t0.00219017006670567\t0.000181997452035671\t0.00433145266043067\t0.0101139695135519\t0.000146207934878072\t0.00452059219757788\t0.00181991985072977\t0.00112049160448655\t0.000944202153681102\t0.0159950367400995\t0.00207026139120325\t0.000607958306219358\t0.0515338577445382\t0.00126938219168731\t0.000619283396467891\t0.000156992630765912\t0.00601276346608901\t0.000835522871466248\t0.0364134001312483\t0.00989846573781064\t0.000873117958236158\t0.00520781844485345\t0.000783905147477154\t0.0632092620531686\t0.00179968073663732\t0.0095349845981014\t0.022297772452532\t0.00311135966417324\t0.00486795023758959\t0.00882694852389529\t0.0251937771370472\t0.00395082177092835\t0.0127948692574278\t0.0117015211977557\t0.0282037510988962\t0.0021101932937057\t0.00116025223883672\t0.0295062922168152\t7.68194564478318e-18\t0.00910813556464992\t0.0044\t0.0677878741050801\t0.000855467975201154\t0.0251166272072112\t0.11900027370063\t0.140963913238211\t0.0151962769121565\t0.00538350950984947\t0.00216035753917248\t0.00588112588239664\t0.0125974817382056\t0.012306325451282\t0.0588746191516836\t0.00358820822822124\t0.0061527065694672\t0.00347199473960039\t0.00608025719462903\t0.00295430070903217\t0.00897255127904656\t0\t0.032168854189218\t0.00220164987697197\t0.00294455046139974\t0.0113293395468264\t0.0115732275143254\t0.0155796699205714",
                 "A[C>A]C\t0.00228040461219034\t0.000148004274511452\t0.0165066026410564\t0.0332972363293847\t0.00943811237752451\t0.000524287283216839\t0.000179116233897663\t0.00046087854006955\t0.00115009665412281\t0.000764882590522354\t0.0477980689580141\t0.00408966055817367\t0.00177013745117308\t0.0065399084412818\t0.0148302194712279\t0.0184460598341763\t0.000552786164744493\t0.00113014804939447\t0.000720968248558332\t0.0131057500167623\t0.000497106430486766\t0.00289910040914304\t0.000918115921316224\t0.000128991153786673\t0.0158103874245379\t0.000640688177064223\t0.0013906363830216\t0.00235988922679969\t9.54438660008139e-05\t0.000399249851155728\t0.0265097555900571\t0.00699891516814894\t0.000528071342438363\t0.0047380152454137\t0.00252969390703724\t0.0512075034354784\t0.00050591025152138\t0.0184902741158151\t0.0187981220676055\t0.00223097493604705\t0.00695707056542578\t0.0461840341793842\t0.00831794546746954\t0.00145030166274585\t0.0100959515234391\t0.00715092962085071\t0.0134017824370641\t0.00122011176223742\t0.0206044794138246\t0.00590125844336305\t0.000150038000874672\t0.00284941672439652\t0.0047\t0.0296946882141723\t0.000841460312449322\t0.00703465694289619\t0.127000292100672\t0.00168956747072749\t0.00653839809246734\t0.00196127855749163\t0.000796131759806154\t0.00205039252702604\t0.0156968621658593\t0.00143073539799457\t0.00674709132892809\t0.00236881713116555\t0.000779342832132512\t0.00500730862445185\t0.000879937028166784\t0.00377450090588022\t0.00457257517091682\t0\t0.0176937280715253\t0.000708222617756176\t0.0529970882409236\t0.00974472961021081\t0.00809634412960846\t0.0247462165428657",
                 "A[C>A]G\t0.000177031410683197\t5.23015105199252e-05\t0.00175070028011204\t0.0155987053074595\t0.0018496300739852\t5.20285090215183e-05\t7.12462338185117e-05\t0.000185950994474916\t0.000377031685742869\t0.000249961630889658\t0.00461981335954027\t0.00042596464493447\t0.000150011648404498\t5.34992510104857e-05\t0.00065711618103902\t0.000726534557978129\t9.42340183015522e-05\t0.000540070749268154\t0.000263988373952011\t0.000413181279154416\t4.61098721236216e-05\t0.00101968359218135\t4.76060107349154e-05\t5.81960089177082e-05\t0.00243159755959666\t0.000245880330043368\t2.18099806833603e-05\t0.000293986200287758\t0.000765351755666904\t9.85616800472162e-08\t0.014805448405013\t0.00144977528483085\t0.00011401540348101\t0.000782672138641122\t0.000352957292167647\t0.0192028137883044\t9.12838062527707e-05\t0.00165912729904071\t0.0044595544905064\t0.000414180997095731\t5.19781134198478e-05\t0.00138951964305939\t0.00223944685662641\t0.00106022052586938\t0.00189923840539943\t0.00267034714512887\t0.00292038841165875\t6.14056247552276e-05\t3.34072627389194e-05\t0.000626133522973774\t9.16232058674661e-07\t0.00165966026754323\t3e-04\t0.00137975318974942\t0.000258141213569471\t0.00711470993797893\t0.00577001327103052\t0.00479877151449229\t0.00413898594844263\t0.0381248535920567\t0.00164027146492725\t4.49085972992533e-05\t0.000205958828418282\t0.00333171250022512\t0.000824644495757878\t0.000141929127690088\t2.23098140648973e-16\t0.000452641966051853\t0.000306011009795412\t0.000385490092517622\t0.00620856628748506\t0\t0.0096708342916345\t0.000138565642613613\t0.000204200082036665\t0.00469656981213721\t0.00176062063415735\t0.00157427088821657",
                 "A[C>A]T\t0.00128022715070335\t9.78028246433782e-05\t0.0122048819527811\t0.0294975517032086\t0.00660867826434713\t0.000180098685074487\t0.000248161039143131\t0.000709812935898873\t0.0019601647322441\t0.00404937842041246\t0.0469981012767084\t0.00304974687100971\t0.0170013201525098\t1.62997718031947e-05\t0.0131277619202124\t0.0141966525908043\t0.00026637884025731\t0.00122015984093916\t0.000347984674754923\t0.0826362558308832\t0.00111023770189198\t0.0105967118403159\t6.18078038533145e-05\t0.000455968729664519\t0.0214140690433615\t0.000570722229490907\t0.00124056770859481\t0.000620970851628224\t0.00172079087548637\t5.61351294482114e-18\t0.0220080989804248\t0.00496922976938575\t0.000619083638199521\t0.00271886106909816\t0.00396951968811773\t0.0356052172324811\t0.000555901383094639\t0.00627669845661182\t0.0145985416056936\t0.00190083066299973\t0.00122948229820025\t0.0215925354604913\t0.0178955797917914\t0.00185038488005505\t0.00884645257251842\t0.00740096212507626\t0.014801968661832\t0.00133012183916047\t0.00797173305476611\t0.00207044152165449\t0.00578146430037068\t0.00963802709585348\t0.0034\t0.0118978717087087\t3.8921291503304e-05\t0.00797527963511844\t0.0832001913604401\t0.0211945741890076\t0.00923773675449515\t1.71111547617892e-05\t0.000414068528341391\t0.00117022402761974\t0.0229954031729149\t0.00864444324382733\t0.00557759549858055\t0.0144927630387766\t0.000444195401112754\t0.00929594730280395\t0.0027172790869801\t0.00362400086976021\t0.00495747308092117\t0.00173110200692441\t0.0207438194151116\t0.00175516113967604\t0.000130809594502265\t0.00775797968968081\t0.0084208185648137\t0.0110761448121778",
                 "A[C>G]A\t0.0018603300783658\t2.23006440649012e-16\t0.0197078831532613\t0.00688942817746127\t0.0100979804039192\t0.000471258225944906\t6.49421429047951e-05\t8.55774469196388e-06\t0.00112009413271091\t0.00117981889779919\t0.00432982507506697\t0.00483959831333999\t2.23017317294687e-16\t0.000304995730059778\t0.000347633892310801\t0.000128896740565299\t0.00017024211595392\t0.00122015984093916\t0.00386982957270561\t2.22097443032156e-16\t0.000113024198480895\t0.00173946024548583\t0.00101012753870304\t0.000145989988006622\t0.00173113735724371\t0.00318844818226969\t0.00877401516481972\t1.68992067512351e-05\t0.00166076328680661\t5.7435943499596e-18\t0.00334122957248267\t0.00802875554289085\t0.000427057695494661\t0.00131944728353293\t0.000153981368254441\t0.00314046017162895\t0.00106981021566774\t0.00831562598073414\t0.00198980121885824\t0.00141061643938401\t0.000761679277421616\t0.00357876282169254\t0.00183954563222884\t0.0343071358842639\t0.00268892174238131\t0.0473061497994739\t0.0122016228158345\t0.00533048827272578\t0.000406088283592853\t0.00036107699967018\t0.00318080561854304\t0.00310936351328883\t0.0012\t0.0010298157865521\t0.000198108373204478\t0.000288190782297881\t0.00385000885502037\t0.00395898649945614\t0.00172957625381781\t0.00703458584651334\t0.00155025656746173\t0.0028505457083045\t0.00041791645766428\t0.0154079196707107\t0.0205911231668028\t0.00293853264372436\t0.000377165914908802\t0.00675807649314426\t0.00723492723159002\t0.052516012603843\t0.00786645728513694\t0\t0.0148171691477631\t0.000508073356232389\t0.000243319459323657\t0.00305585987776561\t0.008856798221547\t0.00700387239009481",
                 "A[C>G]C\t0.00122021650301413\t0.00013300384128394\t0.0117046818727491\t0.00283976429956314\t0.00569886022795441\t0.000289158444369592\t2.16140259898856e-05\t0.000188950204063221\t0.000319026811013197\t0.00254960863507451\t0.00294988082481468\t0.00195983733350132\t2.43018870415287e-05\t0.000133998124026263\t0.000325131346638492\t0.000292095511604645\t0.000290413021333157\t0.000642084113018806\t0.000950958119804402\t5.78253703029667e-06\t0.000269057605233281\t0.00200937649047501\t0.000569071851011908\t4.33970238321055e-05\t0.0025917027487059\t0.00178912923080337\t0.00151069132256303\t0.000718966251724144\t0.000195089663209211\t4.08255486896083e-05\t0.00367135105718905\t0.00163974583939489\t0.000378051074700192\t0.00176925885746461\t0.000223972899279187\t0.00263038543037711\t0.000466917168894238\t0.00315833871383652\t0.0020297972232574\t0.00130056834836824\t0.00112952438777746\t0.00454842760857572\t0.00166958761185987\t0.0112023300846576\t0.00157936667396374\t0.0225029253802994\t0.00868115459356094\t0.0031602894825166\t0.000506110028320157\t3.13066761486887e-05\t0.000233059028025323\t0.00192960500985448\t0.0015\t0.000616889650779269\t0.000114062396693487\t0.000135089429202132\t0.00110000253000582\t0.00453883805745729\t0.00151962769121565\t0.00546356169586954\t0.000496082101587754\t0.000168032168068476\t0.000165966822900169\t0.00313160964735874\t0.0017392502092348\t0.00243878219411137\t0.00058825877444662\t0.0108119561891033\t0.00255135508166888\t0.0783210187970445\t0.00393397863849599\t0.00403923801615695\t0.0058832129144495\t0.000277130285200937\t0.000124573614466966\t0.0016898599324056\t0.00541810111432403\t0.00399009063919146",
                 "A[C>G]G\t0.000115020408071004\t1.52004390038788e-05\t0.000253101240496198\t0.00127989376881719\t0.00171965606878624\t2.22121711591867e-16\t5.08329870502865e-05\t0.000504866947364691\t2.22018658448056e-16\t0.000905860950344122\t0.000285988446066779\t0.00135988712936826\t2.23017317294687e-16\t2.21996892043511e-16\t4.24335818106468e-05\t2.69267447978162e-05\t5.96848830050214e-05\t7.510983938896e-05\t0.000263988373952011\t3.91171622637716e-05\t3.07065742775528e-05\t0.00084673725742902\t0.000152019193943427\t2.22984707708745e-16\t0.00192126226931094\t0.000615700338645181\t0.0004692147220411\t2.21989579809124e-16\t0.000231106216417065\t2.86179091304607e-18\t0.000920338684635946\t0\t6.83092285767807e-05\t0\t2.21973141249908e-16\t0.000896131310120872\t0.000130976764721938\t0.00299842282959164\t0.00112988712427629\t0.00025311060933628\t2.21906561138581e-16\t0.000345880429135648\t0.000499876530496967\t0.00535111303151055\t0.000635745066228442\t0.0136017682298699\t0.00225029928980554\t0.00121011084615351\t1.77038488167328e-05\t0.00443094489899971\t3.14079548497646e-05\t0.000269944742311249\t0\t0.000327941337853485\t8.03439513551493e-05\t0.000204135137460999\t0.000827001902104375\t0.00196949580907288\t0.000817799639088423\t0.00251163733637959\t0.00170028139657093\t0.000510097653065015\t0.000427914459043808\t0.00277142451219927\t0.00168927175494644\t0.000829585746357557\t0.000276121465556577\t4.59999965500003e-05\t0.000422197013514526\t0.00496530119167229\t0.0306618335062441\t0\t0.00605056156875587\t4.62012145375278e-05\t4.72600016687395e-05\t0.000907299963708003\t0.00163156569476336\t0.000550341742011534",
                 "A[C>G]T\t0.00114020230609517\t9.12026340232729e-05\t0.017406962785114\t0.00354970537445392\t0.0100979804039192\t0.000254139255605109\t9.56620779922714e-05\t0.000284924910888984\t0.0019301622108322\t0.00290955338355562\t0.00570976932531926\t0.00640946801415483\t2.23017317294687e-16\t2.21996892043511e-16\t0.000516305633623826\t0.000106580770036192\t0.0005357619531491\t0.00144018866471508\t0.00399982384775774\t0.000204089542245765\t0.000325069597400804\t0.0114964326569465\t1.96024750084946e-05\t0.000318978124480222\t0.00408268232228574\t0.00213895896866995\t0.00442202360644278\t6.18970945503818e-05\t6.20285083024157e-05\t5.61351294482114e-18\t0.00285104918610049\t0.00342946843239298\t0.000728098366089259\t0.00131944728353293\t8.32899219194476e-05\t0.00249036491317073\t0.000730870343601045\t0.00377801276528546\t0.00174982519246327\t0.00130056834836824\t0.00202914558158252\t0.00947672389654897\t0.00190952834649842\t0.0188039112135324\t0.00243902195219717\t0.0434056427335554\t0.0146019420582938\t0.00827075760139629\t0.000431093719774679\t0.000354075506601783\t0.00329083348585113\t0.00142970733890773\t0.0022\t0.00171969238142681\t0.000686375474839756\t0.000176116589182039\t0.00197000453101042\t0.000899769658967304\t0.00170958115261761\t0.00304198306876253\t0.00518085743190436\t0.00127024317527955\t0.000122975416968197\t0.00507260732016256\t0.00202912524410727\t0.00382808844403547\t0.000250110023149073\t0.029594213780434\t0.0130197614167626\t0.0920820220996853\t0.0097203472185146\t0.00403923801615695\t0.0109006856569006\t0.000923769284046939\t8.0876204563098e-05\t0.00411590983536361\t0.013901694724708\t0.00712363521813952",
                 "A[C>T]A\t0.0250044365371747\t6.11017646800655e-05\t0.0142056822729092\t0.00869927795992932\t0.0325934813037393\t0.0641351428515255\t0.000122079220868798\t0.0014096285065034\t0.00133011178259421\t0.00332948892345025\t0.00680972488711456\t0.00809932775579627\t0.00104008076227119\t0.000548992314107601\t0.00486706685605187\t0.00136790196583553\t0.0258367446565361\t0.0055107219045695\t0.00717968380672514\t0.0124054427639583\t0.00505108143653555\t0.00175945404141095\t0.00122015405665119\t0.00057096084350535\t0.00920604837378158\t0.0279863790293264\t0.0493225709949386\t0.00216989814498107\t0.00176080926793954\t0.0226141519702242\t0.0118043439985915\t0.0209967455044468\t0.00266035941455691\t0.0136942634730311\t0.000776905994374679\t0.00509074594700361\t0.105981198935309\t0.0176906946945907\t0.0863913695021867\t0.00531232148448872\t0.00351851844688201\t0.00220923626702249\t0.00595852824352385\t0.00624129819002352\t0.00344861710454108\t0.00960124816226109\t0.0197026204485197\t0.00988090509090633\t0.022404870818916\t2.90061855690726e-05\t0.0899227751908864\t0.00132972780471837\t0.0152\t0.00426923631900726\t0.000287157086412551\t0.000575380902157227\t0.00352000809601862\t0.0357908375455883\t0.0026893411114277\t0.00722470978831099\t0.0014802449805441\t0.00511097844541613\t0.00739852102085089\t0.0326167650172189\t0.0804653114042536\t0.00391804352496581\t0.00276121465556577\t0.0538857649585676\t0.00137478104400674\t0.0087243020938325\t0.0094250488219849\t0.0103866130415465\t0.049898107913646\t0.000708018612393283\t0.0157634101530637\t0.0237663690493452\t0.00894100654472624\t0.00753036175765247",
                 "A[C>T]C\t0.00632112155659776\t0.0013800398569311\t0.0124049619847939\t0.00417965308879363\t0.0178964207158568\t0.0229125549344763\t0.0177114935194896\t0.0380899617714747\t0.00768064548144624\t0.00139978513298209\t0.0028198860766025\t0.00541955017733528\t0.000834064765129009\t0.00524992650102897\t0.00524943631308188\t0.00196342211058735\t0.147209359089566\t0.00212027775638609\t2.21990223550554e-16\t0.0101044332190305\t0.006911479747814\t0.00372884294003571\t0.00171021593186356\t7.02951791566132e-05\t0.00465305705848743\t0.0140931408683394\t0.0130059517836552\t0.00391981600383678\t0.000220101158492443\t0.0809506590438556\t0.0112041231173072\t0.0130979698146787\t0.00254034320036637\t0.00541773051268824\t0.00337959106948059\t0.00431063163685375\t0.0903839658844521\t0.00927512128620346\t0.122987713527419\t0.0021309312169418\t0.0037084384766853\t0.00398862113367409\t0.00327919004006011\t0.00306063661241538\t0.00214913819558357\t0.00404052526828488\t0.0121016095140654\t0.00336030780419486\t0.0398086544014669\t0.000128027301822114\t0.0169042814318797\t0.000600877000478002\t0.00890000000000001\t0.000939831882872791\t0.000438239734664451\t0.000303200718875896\t0.00180000414000952\t0.00564855397018363\t0.00209948562602162\t0.00196127855749163\t0.000529087563991778\t0.00023304461404735\t0.000981803735469672\t0.0129066340098811\t0.00675708701978576\t0.00285857257178628\t0.00134058972407903\t0.0421068288419878\t0.00147844204732493\t0.00204950049188012\t0.00355558069319684\t0\t0.013521816266249\t0.000292526689941625\t0.00121493438737799\t0.0140975394360984\t0.00507120824756748\t0.0142901047508419",
                 "A[C>T]G\t0.365064773442751\t3.27009444359762e-05\t0.00257102841136455\t0.000782935016393639\t0.00617876424715057\t0.120065790049658\t2.23144805358541e-16\t0.000542856935483223\t0.000927077911627691\t0.00120981429350595\t0.00322986951327166\t0.00437963649017132\t0.00557043254408703\t0.000413994204081142\t0.013019985315525\t0.000702930371761392\t0.00759079552312184\t0.000118015460025263\t0.000302986656467649\t0.0265116317132979\t0.0745159538657226\t0.00326898563375784\t3.95049878997722e-05\t0.00028698031888973\t0.0122080206695799\t0.00154924598198057\t0.0133060891325088\t0.00739965266030413\t0.00136062534340783\t0.0087254604062104\t0.000514189221633561\t0.0196969469732192\t0.00533072018029636\t0.00670719035795905\t0.00704914705320655\t0.0201029456846312\t0.0143974458930986\t0.00758600975886684\t0.0365963440252319\t0.00716313028793583\t0.000117950334298885\t0.000579799563290971\t0.00706825414122712\t0.00633131691391809\t0.0051779236526153\t0.00600078010141318\t0.000267035515723591\t0.00343031421678226\t0.000440095676800136\t0.00122026022049202\t0.014303622750052\t0.00540889279964391\t0.0046\t0.00304945451357661\t0.00121066228069403\t0.00201133150145396\t0.0139000319700735\t0.00564855397018363\t0.00407900064484201\t0.00695453366049328\t0.00034305677589637\t0.00532101865550173\t0.00151969620968829\t0.0176090510522409\t0.00862628121017029\t0.00255872230201849\t0.00421185278983039\t0.00254067280944954\t0.00044377601420527\t0.0240910057818414\t0.121719339063989\t0.012694749050779\t0.00610562848067364\t9.24024290750555e-05\t0.00222754056693685\t0.00978577960856883\t0.0176377359880682\t0.0230993698363128",
                 "A[C>T]T\t0.00958170008104535\t0.00186005372021149\t0.0121048419367747\t0.00424964727927582\t0.021995600879824\t0.0213116777338142\t0.00742481818726625\t0.0107971545178984\t0.00748062867203358\t0.032894950625079\t0.0124994950204012\t0.0064894613747059\t0.00299023219152966\t0.00792988898155424\t0.00615621277751885\t0.00250772109609097\t0.109155239052807\t0.00238031182084853\t0.00031798599589674\t0.0161070668144942\t0.00178038117961056\t0.00599813877753733\t0.000737093065370432\t0.000146989919431325\t0.00797523973250426\t0.036782098152829\t0.0130059517836552\t4.16980426938759e-05\t0.00061628324377884\t0.0350219167680463\t0.00915336843958577\t0.00572911198764192\t0.00279037697992999\t0.00447812411380873\t0.00505938781407449\t0.00183026818919776\t0.0292948031019297\t0.0100946901929585\t0.15198481671681\t0.00307134217653114\t0.00386837113336175\t0.000899688977520471\t0.00430893569288386\t0.00565117544449245\t0.00243902195219717\t0.0128016642163481\t0.0169022479989839\t0.00635058171328494\t0.0352076541440109\t0.000910194098891589\t0.0303076761766836\t0.00121975031711009\t0.0068\t0.00248955466846091\t0.00023512862476289\t0.000415274911991739\t0.00692001591603661\t0.00319918100966153\t0.00175956890561812\t0.00194126551098661\t0.00499082598169938\t0.0007761485858401\t0.00283943239178602\t0.0154079196707107\t0.00730685001695768\t0.00338830804837604\t0.000298131147593695\t0.0543458699240598\t0.0045635831460803\t0.00457510109802426\t0.0107019418884555\t0\t0.0296365272125246\t0.000646637998819713\t0.00232619294212912\t0.0164520693419172\t0.00650902375548142\t0.0165099999568924",
                 "A[T>A]A\t0.00080014196918959\t9.59027697678933e-05\t0.0054921968787515\t0.00941921820488899\t0.00737852429514098\t9.82538381906365e-05\t0.00194125974168417\t0.00248934395829323\t0.0124010421835851\t8.88863559443625e-05\t0.0194992122318258\t0.012698945987483\t5.07039371607203e-05\t0.00181997452035671\t0.00402005807774053\t0.00190397473635455\t0.00041759389619285\t0.005170677358734\t2.21990223550554e-16\t0.000613269065669872\t0.0020404368575312\t0.0230928342935187\t0.00101012753870304\t6.25957071864009e-05\t0.00249163700551262\t0.000785617639894663\t0.000511233950880601\t0.00345983759522328\t0.0639293819439414\t8.2851848811264e-18\t0.00158058165404869\t0.0203968384900341\t0.0015202053797468\t0.142940122382734\t0.000171979190517947\t0.000619090715362522\t0.00036293561522186\t0.00785586781353009\t0.000712928778415037\t0.000868379481833561\t0.102956647735468\t0.00488831011119456\t0.0024393974688252\t0.00225046809736425\t0.00198920232986572\t0.00531069038975067\t0.00610081140791725\t0.00746068339859931\t0.0133028920487314\t0.0153032634209245\t0.0047812109612062\t0.000524892554494095\t0.0027\t0.137975318974942\t0.000660361244014926\t0.000129085454570926\t0.00536001232802835\t0.112971079403673\t0.00176956645621823\t0.0058938421957274\t0.000956158244189301\t0.00426081569030777\t5.04899069666176e-06\t0.00328168678700852\t0.0244894426012946\t0.00199900179845194\t0.00397174716760728\t0.00156233888282458\t0.111858720580598\t0.00184770044344811\t0.00989704625903881\t0.0478938261915753\t0.0115162573150389\t0.0613536678652208\t0.000215264822829132\t0.0100720295971188\t0.0104135321713916\t0.0105248312583337",
                 "A[T>A]C\t0.00223039573911599\t0.000878025358250369\t0.00721288515406162\t0.00352970703431615\t0.00692861427714457\t0.000546299344725942\t0.000670435065427006\t0.00046687695924616\t0.00342028744095653\t0.00107983424544332\t0.0212991395147636\t0.00281976595942537\t0.000194015065269817\t0.00033699528206605\t0.013642364920594\t0.00318662835672214\t0.000441628077268698\t0.0027103550565124\t6.51971287184511e-05\t0.000372163282918748\t0.000841180096658694\t0.00773759902302315\t0.000997125897115768\t0.000288980181739136\t0.000878577225236981\t0.000254875951874223\t0.00519237613516697\t0.00564973480144841\t0.00701322327741829\t5.5634816351525e-18\t0.00225082830481617\t0.00888862226354915\t0.00352047561625576\t0.00145938866208945\t0.000477942168997551\t0.00200029310294837\t0.000392930294165815\t0.00606680885854041\t0.000886911397551384\t0.00114049839779984\t0.0443813122277162\t0.00735745652727852\t0.00142964687722133\t0.00166034535183318\t0.0019392223718289\t0.00288037444867833\t0.00444059059854961\t0.00430039391608271\t0.00401087196356488\t0.00378080625693429\t0.00260065868182764\t0.000496898284921077\t0.0018\t0.0071587194482651\t0.000266145592284804\t7.78515377179692e-05\t0.00394000906202084\t0.00134965448845096\t0.00117971097081215\t0.00259168952239965\t0.00319052803238898\t0.000618118332537607\t0.00428914259181761\t0.0064333067196539\t0.00145937086522\t0.00249875224806493\t0.000223098140648973\t0.000397460970190427\t0.0130512534177706\t0.00379990091197622\t0.000694526228722578\t0.0236583960946336\t0.00306767467684393\t0.000354111308878088\t0.000169235885178583\t0.00358814985647401\t0.00578886625138323\t0.00560861933000566",
                 "A[T>A]G\t0.00114020230609517\t2.55007364867704e-05\t0.00964385754301721\t0.0123989708854165\t0.00911817636472707\t2.22121711591867e-16\t0.000587381169262168\t0.00155958898591865\t0.000211017733930359\t0.00369943213716694\t0.0227990789172117\t0.00275977093901206\t2.23017317294687e-16\t2.33996724045863e-05\t0.00123009128026161\t0.000660449904293088\t3.93559715117003e-05\t0.00312040877354934\t0.000589974017544266\t0.000354155382132356\t0.000540115638758257\t0.00880726710501731\t0.00020002525518872\t6.569549460298e-05\t0.00260170932302523\t0.000951536886997099\t0.000303138722342118\t0.000247988359426408\t0.0606278645665547\t0.000354221672453954\t0.00200073627094771\t0.00623903294989277\t0.000625084448909048\t0.00122948496874659\t0.000151981610225163\t0.00248036344765598\t0.000637886838874783\t0.00502735561094864\t0.000711928878305058\t0.000696304284972533\t0.0345854370062834\t0.00132954037789136\t0.00252937524431465\t0.00249051802774977\t0.00107956709359547\t0.00694090231730125\t0.00693092181260108\t0.00303027757342573\t0.00501108941083792\t0.00143030501254393\t1.08027360629764e-06\t0.00212956407823319\t5e-04\t0.00744866758234286\t0.000465254512828698\t0.000241159647686769\t0.000573001317903031\t0.0367905816111076\t0.00118970852141225\t0.00287187217346989\t0.00423070018087943\t0.00810155096044436\t0.000512897470769798\t0.00580298273312483\t0.00468797901224782\t0.00218890696930488\t0.000352154912593895\t7.07999946900004e-05\t0.0211072566756433\t0.00199410047858411\t0.00384337913045132\t0.000577034002308136\t0.0106705673827554\t0.000354111308878088\t0.000160199906352066\t0.00701172971953081\t0.00531744621143067\t0.00402893050643422",
                 "A[T>A]T\t0.000183032475452119\t2.23006440649012e-16\t0.0061124449779912\t0.00420965059900028\t0.00619876024795041\t0.00166091009568693\t0.00371240909363312\t0.00828781582901642\t0.05070426118611\t0.0410936921182598\t0.0249989900408023\t0.0188984314301913\t0.00666051718915971\t0.00284996010055858\t0.0717767269598036\t0.0221637936906852\t3.4048423190784e-06\t0.00282036946840036\t0.000605973312935297\t0.00185081202526796\t0.00954204295139589\t0.0109965877588184\t0.00146018436287766\t0.000632956591836929\t0.007845154266353\t0.000653681853042124\t0.0256117204355057\t0.00159992489952521\t0.00251115412643651\t0.000161100817133013\t0.00180066264385294\t0.014897690857917\t0.00307041481304124\t0.00253893643952548\t0.00169979432488669\t0.00196028724088941\t0.000489913089417937\t0.00415781299036707\t0.000843915692822287\t0.00507221655863612\t0.315866996935998\t0.00736745307147319\t0.00306924189725138\t0.00277057627986621\t0.0027189097172034\t0.00612079570344145\t0.0107014232892975\t0.0696063759440365\t0.0068514895138203\t0.00428091290467692\t0.0308078028462659\t0.000604876181845576\t0.002\t0.0675879098746816\t0.000675369454106174\t0.000354234503241146\t0.00241000554301275\t0.0294924499328172\t0.00118970852141225\t0.00463302026591135\t0.000301049823745794\t0.00392075058826443\t0.0289942040006319\t0.00631324500793408\t0.00876622088217769\t0.00279860251783272\t0.000728320387410101\t0.010652827201038\t0.070288411249932\t0.00725840174201642\t0.00420617716045802\t0.0427005191708021\t0.0044484873178094\t0.0234637358146871\t0.00037385509066041\t0.00918123963275042\t0.0132763562322627\t0.0159102586276951",
                 "A[T>C]A\t0.00109019343302082\t6.1701782009166e-05\t0.0165066026410564\t0.007949340204763\t0.0461907618476305\t0.00221121163341453\t0.00049031818217796\t0.00101973126002373\t0.000288024205554234\t0.00435933084271564\t0.00976960530794555\t0.0202983152398351\t0.00074505785374234\t0.00934986910183255\t0.0182225378596284\t0.00521479296601786\t0.00114162360110276\t0.0736096428632151\t0.00190991588730432\t0.0126055305504737\t0.00389083302735116\t0.241924930694006\t9.65121856285573e-05\t9.72933276235913e-05\t0.00202132801250422\t0.00258874006021269\t0.0107048987757778\t0.0166992161387944\t0.0250114952832321\t0.000886554807328259\t0.00640235606703267\t0.0206967919972404\t0.0848114580279796\t0.00504788544078885\t0.000548933579036936\t0.00196028724088941\t0.000610891627825224\t0.0203892752412231\t0.000935906502940356\t0.0125054648881561\t0.00461805546153263\t0.00203929501571307\t0.00524870357021816\t0.0733152495719111\t0.00278888165845496\t0.0112014561893046\t0.0187024874308283\t0.0144013191608351\t0.0116025223883672\t8.88189406390913e-06\t0.0193048894458744\t0.00136971961839412\t0.0185\t0.0028294938601383\t0.00110060207335821\t0.000613406074821531\t0.00263000604901391\t0.0025893371296948\t0.00374908147503861\t0.0113073712753344\t0.0292048333999242\t0.0043508329232016\t0.0124975017244103\t0.0266136794312277\t0.0804653114042536\t0.00994503394729842\t0.00129056771944922\t0\t0.0295670219464404\t0.00361080086659221\t0.0092150499622787\t0.11136757144547\t0.0157823045507937\t0.00571197215632405\t0.000299841243488636\t0.0891880464324782\t0.0344443967528774\t0.0136339516523377",
                 "A[T>C]C\t0.00304053948292045\t2.15006209594339e-05\t0.00776310524209684\t0.000714940659925226\t0.0133973205358928\t0.00113061952296761\t0.000227147402764075\t0.000455879857422375\t0.000533044797084746\t0.00392939683758543\t0.00500979760417679\t0.0104991285723285\t0.00362028111482855\t0.000509992860099957\t0.00907955457303495\t0.00322795502350425\t0.000939335910381041\t0.0403052799916789\t0.0001419937465954\t0.00630276527523685\t0.00196041972586331\t0.0128959983717053\t0.0127016037044837\t0.000251982719025129\t0.00209137403273951\t0.000593711040836425\t0.00455208312427933\t0.0099995306220326\t0.000899413370385028\t0.00017811146242035\t0.002180802535333\t0.00949852772820213\t0.0261035265864418\t0.00184922535949691\t0.000440946645455899\t0.00229033560287589\t0.000457918765211052\t0.00739610964632604\t0.000700929977095289\t0.00375163946644684\t0.00122948229820025\t0.000820716278382563\t0.00168958267307975\t0.0384079888616832\t0.00163934262360793\t0.00564073329532839\t0.00734097634985453\t0.00579053041258579\t0.00351076323992836\t0.0313066761486887\t0.0032408208188929\t0.000417914452911489\t0.0478\t0.00566898593179652\t0.000189103447149729\t8.14539224966927e-05\t0.00928002134404909\t0.0192950604645211\t0.00187953951281936\t0.00370241360342808\t0.00775128283730866\t3.36064336136951e-06\t0.00615876884978939\t0.00966496779344583\t0.00492787559283193\t0.00260869734697979\t0.000277121905649173\t0.000323132975765027\t0.0204104866533397\t0.00206440049545612\t0.00311898306392196\t0.0455856901823428\t0.00345339505071307\t0.000184753856809388\t0.000148825793972971\t0.00463735981450561\t0.00329174316146106\t0.00515384409540948",
                 "A[T>C]G\t0.000106018810917621\t1.32003812402106e-05\t0.0123049219687875\t0.00499958503444214\t0.0381923615276945\t0.00424232458175457\t0.000187121428708732\t0.00149960479415255\t0.00107008993035775\t0.00918858955150384\t0.00697971801939201\t0.0142988131985045\t0.00214016618390417\t0.000530992566104073\t0.016943033887876\t0.00855587489051188\t0.000320455747677967\t0.070509236710009\t0.000427981151710078\t0.0124054427639583\t0.00024005139500367\t0.0638801779807726\t0.000376047479754793\t2.22984707708745e-16\t0.00208136745842018\t0.00230887626991943\t0.0186085156289221\t0.00387981788134865\t0.00775356353780197\t0.000470294310885193\t0.00176064791843398\t0.0198969159780234\t0.0785106067829764\t0.00568761745704724\t0.000404951000928887\t0.00275040301655402\t0.000276950868915854\t0.00811573112542803\t0.00108989111987712\t0.0734320898232528\t1.94917925324429e-05\t0.000919682065909815\t0.00383905175421671\t0.0168034951269864\t0.0021691301787983\t0.00964125336293718\t0.015502061774216\t0.00804073653146628\t0.0085618613486572\t0.000586124991154364\t0.0272068908252738\t0.00206957635771958\t0.0329\t0.00208962620766397\t0.000285155991733718\t0.000200132487706862\t0.00113000259900598\t0.00394898905880095\t0.00261935825722698\t0.00825538168331936\t0.202033436533722\t0.00402076973592424\t0.00954809131744946\t0.0103052969226182\t0.00674709132892809\t0.00351824316527542\t3.52154912593895e-06\t1.14999991375001e-05\t0.0050709401623208\t0.0035218008452322\t0.000973214715444095\t0.0063473740253895\t0.0146903909550654\t0.000107772833132237\t0.000610604240688731\t0.0401119883955205\t0.00629280563528245\t0.00282878244393038",
                 "A[T>C]T\t0.00574101862893531\t0.000155004476684291\t0.0173069227691076\t0.00155987053074595\t0.0382923415316937\t0.000502275221707735\t0.000615399351100909\t0.00214943353828532\t0.00265022272471778\t0.0509921727014903\t0.0148993980643182\t0.0289975931997644\t0.00336026092426076\t0.000149997900029399\t0.0127789875253921\t0.00568142006493223\t0.00119169481167744\t0.0638083588950153\t0.000958957767499917\t0.00597262042748635\t0.000929198941493373\t0.135957812290846\t0.00779098368960063\t0.000543962695038373\t0.00424278751139499\t0.00173915355396529\t0.00010304715643973\t0.0255987983924034\t0.000755347157553611\t0.000261163436470288\t0.0024408982505562\t0.0144977528483085\t0.0372050263990665\t0.00258891550329567\t0.00059492801371034\t0.00115016853419532\t0.000544903334148522\t0.00846554712221372\t0.000609939067087198\t0.00921402652959343\t0.00233901510389315\t0.0020292984715184\t0.00511873567228895\t0.048009986077104\t0.00348860107097054\t0.0136017682298699\t0.0194025805432123\t0.0177016214685265\t0.0027005871076372\t0.00880187700027031\t0.000267067641556915\t0.00145970119916453\t0.0306\t0.00129976749759003\t0.00111060754675238\t0.000482319295373537\t0.00492001131602603\t0.00186952140252095\t0.00491879489525066\t0.00718468369530097\t0.00783129607950024\t0.0137026232293935\t0.012397521710615\t0.0179092053315404\t0.00815648373985974\t0.0107946097116405\t3.63159753612454e-05\t0.00938215929633805\t0.059234081896083\t0.00446870107248826\t0.00557556972465639\t0.169648009678592\t0.0118120880974314\t0.00286368428053237\t0.000235211867934782\t0.0295669388173225\t0.0106766584656483\t0.0229376809921258",
                 "A[T>G]A\t0.000172030523375762\t0.000238006873876524\t0.0039515806322529\t0.00104991285723285\t0.00385922815436913\t2.5714090035635e-05\t0.000473307143204439\t8.43777630843168e-05\t0.0119010001600534\t0.00369943213716694\t0.00163993374667663\t0.0450962570106681\t0.00192014909957757\t3.4699514206801e-05\t0.00251923582225346\t0.00693223462200924\t0.000218310478105615\t0.00201026334449813\t6.13972960630813e-05\t0.000212093053706383\t0.000468100220257156\t0.0111965257180697\t0.000851107460828002\t0.00140990330883108\t0.000721474008423534\t0.00232886654065466\t1.05048072098754e-05\t0.000391981600383678\t0.00233107136039724\t8.2851848811264e-18\t0.00321118171487107\t0.00527918172683234\t0.00292039454530307\t0.00113952265396026\t0.00986880587448917\t0.000407059646449994\t3.89930826271419e-05\t0.00367806533763241\t0.000385961442451899\t0.000979428010040388\t0.00281881307392252\t0.00304894597937493\t0.00108973083648339\t0.0217045145390241\t0.00114953903484703\t0.00388050446558052\t0.00723096171790848\t0.0075306898111867\t0.00120026093672764\t0.00263056096712624\t2.40060801399474e-08\t0.000116976055001541\t1e-04\t0.000170969416990689\t0.000517282974478359\t0.000196129837952725\t0.00074300170890393\t0.000722814959370401\t0.00172957625381781\t0.00459299417290132\t3.87064059101735e-05\t0.000262050166868694\t0.000120975816692292\t0.00706363070618298\t0.0134941826578562\t0.00178910660961449\t0.0250110023149073\t0.000127137990464651\t0.0238382227630615\t0.00478110114746428\t0.00795505680404155\t0.00807847703231391\t0.00523782038708874\t0.00543484187112311\t0.00238815121697493\t0.0020696599172136\t0.025483724390257\t0.00371962869080269",
                 "A[T>G]C\t0.000207036734527807\t7.46021545848263e-05\t0.00260104041616647\t0.00015898680409526\t0.0025994801039792\t0.000204111843084418\t0.000247160389791747\t0.00093175444543345\t0.00105008824941648\t0.00265959175266596\t0.000667973013890239\t0.00692942485773681\t0.00124009629347718\t5.32992538104465e-05\t0.000978867919018153\t0.000613639618064527\t4.67665106767532e-05\t0.000664086995396397\t2.21990223550554e-16\t0.00297130362975452\t0.00041508887052718\t0.00440863200148994\t0.00070008839316052\t0.00447969278266895\t0.000858564076598325\t0.00188908058447953\t2.13097517686043e-05\t0.000436979488182824\t5.09234043966607e-05\t5.43340023001404e-18\t0.0014005153896634\t0\t0.00253034184918382\t0.000958598443112179\t0.0172979069532586\t0.000817119732554411\t0.00010998048946117\t0.00247869620579575\t0.000428957147180997\t0.000302132031697852\t0.000461805546153263\t0.000965666169205306\t0.000622846156999221\t0.0106022052586938\t0.0010695711019881\t0.00297038615019953\t0.00705093777472404\t0.00164015023776178\t0.00040508806614558\t0.00407086811262502\t3.26082588567619e-05\t0.000583880479665812\t0.000800000000000001\t0.00171969238142681\t4.10224409160787e-05\t8.28548499106408e-05\t0.000607001396103211\t0.00408895322797364\t0.000528870426745447\t0.00290189174322741\t0.00158026153328357\t0.000920176158470222\t7.98840310224306e-18\t0.000789405754557843\t0.000787660439584494\t0.00155922140279252\t0.0105046209722611\t0.00127095590467831\t0.00343492010995179\t7.31000175440042e-05\t0.000673416343349256\t0.00519330602077322\t0.00130024169613512\t4.62012145375278e-05\t0.000131587060660205\t0.000358969985641201\t0.00239302312464458\t0.000917381024486118",
                 "A[T>G]G\t0.000268047559678513\t2.05005920775997e-06\t0.00630252100840336\t0.00141988214978157\t0.0079384123175365\t0.000107058662794278\t0.000276179220981871\t0.0014696126982695\t0.000604050764426242\t0.0105983731497215\t0.00476980729978509\t0.00532955764671532\t0.000320024849929596\t0.000522992678102505\t0.00103876762418443\t0.000939706937043501\t0.000179254933857363\t0.000808105861867905\t8.24963668600033e-05\t0.00070931120319729\t0.000185039616981996\t0.00461856685870374\t0.000719090792403448\t0.000989932110455862\t0.00199130828954623\t0.00176913896006814\t2.69123156138712e-05\t0.000186991222632009\t0.00295135644342139\t5.68355677835723e-18\t0.00206075835907614\t0.00713889347151192\t0.00229030942080275\t6.76716523448328e-05\t0.00767907083242926\t0.000420061551619159\t0.000409927278900723\t0.00401788659165279\t0.00126987313967335\t0.000475207665749933\t0.000196917083532885\t0.00447845179921301\t0.000935768865090323\t0.0168034951269864\t0.00206917026272465\t0.00766099592947083\t0.00806107212259231\t0.00270024734265659\t0.000637138513912925\t0.0349074440124357\t7.47189244355864e-06\t0.000983798616423218\t0.0016\t0.00367934183933178\t0.000351192116135211\t0.000126083467255323\t0.00394000906202084\t0.0277928850214345\t0.00118970852141225\t0.00506330076576921\t0.00115019035650386\t0.14502776410672\t1.26974617520009e-05\t0.0012306325451282\t0.00270883222242891\t0.00169915152868415\t0.0261114864167632\t0.00095269892854758\t0.00448550114358089\t0.0017220004132801\t0.00268778540532525\t0.00230813600923254\t0.00511041321824118\t0.000230942071005163\t0.00122049851083922\t0.0012383699504652\t0.00513872125032146\t0.0020771085449869",
                 "A[T>G]T\t0.000112019875686543\t3.76010859569634e-06\t0.00397158863545418\t0.0002159820734879\t0.00482903419316137\t0.000386211624659732\t0.00030920064957753\t0.000732806876075879\t2.22018658448056e-16\t0.0142978052868885\t0.000739970105207749\t0.0238980164646334\t0.00922071598859648\t2.21996892043511e-16\t0.0130558582565527\t0.00485570574068897\t0.000285405900275689\t0.00128016770196896\t0.000319985907820619\t0.0111048721516078\t0.00363077734943051\t0.00795753077819952\t0.012501578449295\t0.0690952614469697\t0.00184120967475632\t0.00161921192955389\t5.82266456775949e-05\t6.94967378231266e-05\t4.18192201135641e-05\t7.43465261675955e-18\t0.00471173391808185\t0\t0.00593080125124904\t0.000255892806503356\t0.114986086683511\t0.000810118706694092\t0.000793859169383351\t0.000782588358523417\t6.44935570936464e-18\t0.000758331390817787\t0.00114951596986202\t0.00262909112319871\t0.000983757012018032\t0.0247051386688431\t0.00285885359970652\t0.00642083470851211\t0.0180023943184444\t0.00761069713985801\t0.000911198094465737\t0.00564120298653688\t0.00142035974161356\t0.000165966026754323\t6e-04\t0.00807855490809803\t0.000199108920543894\t0.000126083467255323\t0.000103000236900545\t0.00212945485955595\t0.00185954441161915\t0.00206134379001671\t0.000340056279314186\t0.0022804365666436\t0.00574885079322873\t0.00597307015806124\t0.00207910369839562\t0.00343828309333734\t0.0164072175185792\t0.00176497586762681\t0.00317859610174686\t0.00112850027084006\t0.0021739881952441\t0.00750144303000577\t0.00399421654330289\t0.00204768782961767\t0.000213945233538338\t0.00354119985835201\t0.00796299935405534\t0.00228964526974765",
                 "C[C>A]A\t0.000312055367983941\t0.0002080060074215\t0.0225090036014406\t0.0806933024558961\t0.00742851429714057\t0.00182099781575314\t0.000455295454879534\t0.00113969964355594\t0.000109009161129901\t0.0144977745916002\t0.040098380025447\t0.00479960163306446\t0.00318024694617536\t0.000520992706102113\t0.0174329338237838\t0.0100036827393083\t0.000618880162703073\t0.00542071011302481\t0.00139993834671521\t0.0163071546010096\t0.0418089512964726\t0.0103967738810647\t0.000295037251403362\t0.000270981416094484\t0.0740486499630257\t0.00131935786852539\t0.037617214389649\t2.21989579809124e-16\t0.00183084145473259\t0.000151094555199286\t0.0254093506410359\t0.0147977063555149\t0.00141019051673881\t0.00506787706629693\t0.00202975439971763\t0.0542079430899009\t0.000767863780965257\t0.0106943747588768\t0.00123987613637398\t0.00270118041584172\t0.00194917925324429\t0.029889667142069\t0.0557862208034616\t0.00322066989933906\t0.380847280240624\t0.0101013131707122\t0.0208027667679801\t0.00238021802797136\t0.0116025223883672\t0.00663141414906729\t2.84071948322711e-05\t0.237951291370656\t0.0156\t0.0245956003390114\t0.00118064586051153\t0.229151698424357\t0.022500051750119\t0.00697821357732421\t0.0573859404445911\t0.334217876633778\t6.32104613313429e-05\t0.00625119672873793\t0.002479504342123\t0.00870447409968724\t0.00617733695004083\t0.00222888700527392\t0.00118051930926363\t0\t0.00426423513649817\t0.00611610146786435\t0.00855745353302731\t0\t0.0246714694825487\t0.00118550416453348\t0.0081911341798605\t0.018550349257986\t0.00649848719181487\t0.0799255116578844",
                 "C[C>A]C\t0.00179031765606171\t9.53027524387929e-05\t0.0253101240496198\t0.0796933854490077\t0.00613877224555089\t0.0040922423441925\t0.000240155844332062\t0.0015495916206243\t0.00309025970542564\t0.0032195058058588\t0.0387984325433252\t0.00191984065322578\t0.000202015686518057\t0.00317995548062326\t0.00967042181943386\t0.0148486780051483\t0.000987404272532735\t0.0042805607534587\t0.000967957371157372\t0.0197086469717724\t0.00505108143653555\t0.00902719886019368\t0.00116014648009458\t4.10971815552889e-05\t0.0196128856658825\t0.000766626882696192\t0.0989452793387309\t0.00259987796172847\t0.00443203696418874\t0.000630394501824834\t0.0349128479280375\t0.00781878808784639\t0.00151020402856426\t0.00222906624415033\t0.000108986812595676\t0.0369054077493975\t0.000387931181008489\t0.0115939016077543\t0.00491950854109675\t0.00191083503491026\t3.66845531251618e-05\t0.0329885958424173\t0.0201950118320775\t0.000814169347224224\t0.0694721416711899\t0.00906117795313391\t0.0147019553600629\t0.000919084188111631\t0.0328071322705556\t8.20174902297915e-05\t0.0303076761766836\t0.150969096625921\t0.0072\t0.00526905747099959\t3.5719540017171e-05\t0.0286189457420813\t0.00289000664701529\t0.00316918868769595\t0.0189953461401956\t0.00355231575464045\t0.00257042540540429\t0.00375071803724276\t0.00667866492152486\t0.00394202620146755\t0.00182921142695384\t0.00592704033241001\t0.000253111343426862\t0.00521857160860713\t0.000306235009802583\t0.00315170075640818\t0.0060403672008061\t0\t0.0145144626260917\t0.000446488737295926\t0.00457950360123646\t0.0139555194417792\t0.0047957320633761\t0.0759941300715491",
                 "C[C>A]G\t9.32165394105873e-05\t2.23006440649012e-16\t0.0025110044017607\t0.0244979666687665\t0.00345930813837233\t0.00165090461318279\t2.23144805358541e-16\t0.000407892504009493\t0.00191016052989093\t8.50869391548397e-05\t0.00340986224156544\t0.00136988629943715\t4.64036032397914e-05\t8.37988268164243e-05\t0.00240460582852613\t0.00101960742077299\t9.21310274574155e-06\t0.001170153290081\t4.32980931519775e-06\t0.00138060572695664\t6.6214176455179e-05\t0.00173946024548583\t2.22028033259479e-16\t2.22984707708745e-16\t0.0120078891831934\t0.000235885194675751\t0.00743340167327372\t0.00197990706316245\t0.000136062534340783\t0.000130081405138458\t0.0160058901675817\t0.00127980163074723\t0.00036204891280812\t0.000265888619257393\t2.21973141249908e-16\t0.014802168961818\t0.000406927811006327\t0.000328827036978549\t1.65983418256516e-05\t0.000371162197880474\t0.000297874573059897\t0.00219923972282782\t0.00709824673305693\t0.000747155408324932\t0.0509795571975638\t0.00574074629701861\t0.00254033786493604\t0.000261023909790137\t0.00188040880087331\t2.56054603644227e-05\t0.00374094748847514\t0.0678861037145696\t0.0045\t0.000270951532189922\t0.137074985500068\t0.51534115584517\t5.1800119140274e-18\t0.00429889948173268\t0.01359666881614\t0.0677441624194813\t0.00593098157745037\t0.000641122736499362\t6.71865692704297e-18\t0.0010705502628351\t0.00145937086522\t0.000814593232869167\t0.000379166795093995\t6.91999948100004e-05\t0.000157257005033797\t0.000554120132988832\t0.0200068913625799\t0\t0.00328987525122035\t0.000216005678357273\t0.00232635145819486\t0.00528202978871881\t0.00150826226668765\t0.00622497001285507",
                 "C[C>A]T\t2.23039573911599e-16\t0.00042101215925217\t0.015406162464986\t0.0689942734753015\t0.00648870225954809\t0.00960526320397261\t0.000372241558714696\t0.0014896074288582\t0.0135011346353547\t0.00011698204325636\t0.0329986668538591\t0.00305974604107859\t0.0173013434493188\t0.00832988338163263\t0.0448873939777216\t0.018549972114909\t0.000150213631724047\t0.00110014411887957\t2.21990223550554e-16\t0.260114122470092\t0.0287061459858556\t0.00248922759267799\t0.000137017299804273\t0.000181987519295926\t0.0363238647791599\t3.4383265664601e-05\t0.252115373037009\t0.0190991034880822\t0.000110050579246221\t0.000580363192156196\t0.0171062951166029\t0.0125980473026681\t0.00714096474433696\t0.00309870195375157\t0.000202975439971763\t0.0519076060215103\t0.000296947321545158\t0.0278853323152022\t0.00266973329364397\t0.0025311060933628\t0.000619739044621262\t0.0824714896060432\t0.060485060190133\t0.00177036823659321\t0.183926245575524\t0.00603078400192025\t0.0147019553600629\t0.00265024276223702\t0.0211045881374611\t5.22111340243307e-18\t0.0867219645055601\t0.193960296327342\t0.00830000000000001\t0.00068187802565877\t0.000845462501806989\t0.0490324594881812\t0.00115000264500608\t0.00506870241218248\t0.0187953951281936\t0.00492320944023409\t0.00553091536649251\t2.84054379353852e-05\t0.0712857498360363\t0.00821422211016463\t0.00195915540810357\t0.00201899181643646\t5.41238090094594e-05\t0.00564038057697146\t0.000860851027555841\t0.00331460079550419\t0.00488557347133605\t0\t0.0167336984748875\t0.00101614571243849\t0.00323962559535773\t0.012543099498276\t0.0100621584282679\t0.0399040508151155",
                 "C[C>G]A\t2.41042768218364e-05\t2.23006440649012e-16\t0.0193077230892357\t0.00782935016393639\t0.00695860827834433\t4.11225330920078e-05\t2.23144805358541e-16\t0.000457879330481245\t0.000206017313695043\t0.0013197974110974\t0.00499979800816047\t0.000892925887151366\t4.57035488805704e-05\t2.98995814058602e-05\t0.000110201810811223\t8.17359684772684e-05\t0.000261371719199842\t0.00134017556299875\t0.00560975294648023\t8.55375287353573e-05\t8.36179025929451e-05\t0.00354889877670959\t0.000553069830596811\t0.000193986696392361\t0.000613403005774794\t0.00176913896006814\t7.69352070894682e-05\t1.96990753254042e-05\t0.00105048280189575\t0.000174108957646859\t0.00261096083358676\t0.00719888417295319\t0.000364049183044629\t0.00472801943265966\t1.43982578108049e-05\t0.00329048215435008\t0.00126977474196078\t0.00204892226688762\t0.00114988512647587\t0.00150065578657874\t0.000118949913403113\t0.00385866605914336\t0.00123969379563248\t0.000442091955126666\t0.00168932258164476\t0.0288037444867833\t0.00756100561374663\t0.00208019054545396\t0.00426092632538314\t2.0904457875642e-05\t5.3613578979216e-05\t0.00142970733890773\t5e-04\t0.000955829021304668\t9.93543508040639e-05\t0.000216143086723411\t6.5600150880347e-05\t0.0028492705867298\t0.00117971097081215\t0.00272177432468226\t0.00113018704595597\t0.00294056294119832\t0.000203959228142376\t0.000888456666726698\t0.000793657854099097\t0.00207896187039002\t4.71207283612854e-05\t7.11999946600004e-06\t0.00122394503917848\t0.0212180050923212\t0.00676566326244848\t0.00461627201846509\t0.0154254290831546\t0.0003387149041374\t6.63858120804583e-05\t0.0024095899036164\t0.00337806001700955\t0.00919576116832207",
                 "C[C>G]C\t7.68136290422007e-05\t0.000351010137523781\t0.0141056422569028\t0.0068694298373235\t0.00906818636272747\t3.72203949153939e-05\t0.000131085065031251\t0.000959747068257631\t0.00226018994636309\t0.000285956105737769\t0.00545977942491123\t0.00153987219060818\t2.23017317294687e-16\t2.21996892043511e-16\t0.000207582609750355\t0.000124852916162653\t0.000177252085434375\t0.000891116736292455\t0.000647971463336753\t7.86345001005741e-05\t1.70036404794267e-05\t0.00453859167500325\t0.000235029674846746\t4.65968043911548e-05\t0.00226148579616808\t0.0018091195015386\t4.3820052932623e-06\t0.00027298718598149\t0.000785360951893489\t0.000255159679310052\t0.00522192166717352\t0.00459928711049787\t0.000322043508077941\t0.0012594724070087\t7.54908656052617e-05\t0.00634092913634635\t0.000792859346751886\t0.00185902215434681\t0.000823917690622706\t0.000410179248331521\t0.000117950334298885\t0.0062978228426433\t0.00136966169356169\t0.000313065117544449\t0.00125949494252805\t0.0377049016372128\t0.00935124371541415\t0.00120010993006959\t0.00662143950094751\t0.000212045218642876\t2.49063081451955e-05\t0.000799836273514812\t3e-04\t0.000236957613022183\t0.00012206677540882\t0.000256169584264783\t0.000289000664701529\t0.000119969287862307\t0.000834795475108598\t0.00603393352126251\t0.00088314616068949\t0.00286054762307048\t0.000224955031039385\t0.00287147593863246\t0.00127944842978192\t0.00172913655566093\t0.000142062493148674\t0.00344078574194107\t0.00021017900672783\t0.0591170141880834\t0.00333918186824246\t0.00115406800461627\t0.0106184963572366\t7.70020242292129e-05\t0.00020205722084773\t0.00396221984151121\t0.00320840580067268\t0.0150747665722057",
                 "C[C>G]G\t0.00035206246644342\t4.87014065453223e-05\t0.00150060024009604\t0.00339971782342066\t0.00248950209958008\t8.43462175098845e-05\t7.75503247322284e-05\t0.000790791594783111\t0.000262022020330588\t0.0013197974110974\t1.0499575817137e-05\t0.000436963732010243\t2.01015608862027e-05\t2.21996892043511e-16\t0.00018022253098145\t2.90258615279342e-05\t1.12159511687288e-05\t0.000173022665969242\t0.000487978509426444\t1.9108383612226e-05\t5.0210750121601e-05\t0.00145954710253408\t0.000232029296018915\t2.22984707708745e-16\t0.00358235360631935\t4.80766011182357e-05\t0.000188086071948245\t1.00995259282529e-05\t0.000239109894907699\t1.86116471967332e-18\t0.000945347888022792\t0.000823872299793532\t4.17056344312117e-05\t0.000360848840420748\t2.21973141249908e-16\t0.00130019051691644\t0.000119978715775821\t0.000853551032157086\t0.00053694635905873\t0.000318139026754692\t0.00018992002980329\t0.000869699344936456\t0.000418896532556459\t0.000321066781890633\t0.000956616396824873\t0.012301599207897\t0.00168022346972147\t0.000411037651048836\t0.000470102200218327\t0.00254054177053257\t7.68194564478318e-18\t0.00064286840483753\t1e-04\t0.000669880171834862\t3.63198684208209e-05\t0.000238157660371166\t0.00142000326600751\t0.00202948045300403\t0.000567860874085849\t0.00162105676690635\t0.000434071838889285\t0.00026405054982189\t6.6586689187658e-05\t0.000763392383685214\t5.10779802827001e-05\t0.00068965562046592\t0.000337148311204951\t2.19999983500001e-06\t4.0200001286802e-05\t0.00315600075744018\t0.0859845331039852\t0\t0.00416984188376561\t1.54004048458426e-05\t5.34188829007216e-05\t0.0016010099359596\t0.000775635363837088\t0.00184268124210204",
                 "C[C>G]T\t2.23039573911599e-16\t6.36018368846509e-05\t0.0223089235694278\t0.00681943398697908\t0.00952809438112379\t1.44078948059589e-05\t9.56620779922714e-05\t0.000889765511197179\t0.00424035635954845\t2.21965928230017e-16\t0.00548977821296019\t0.00302974853087194\t3.01023374465026e-05\t2.21996892043511e-16\t0.000238267798260736\t0.000230805477983167\t9.65372939879875e-06\t0.00156020438677467\t0.00639971815641238\t0.000178078129998756\t8.45180953242088e-05\t0.00961701583998485\t3.90049247618004e-05\t0.000192986764967658\t0.00379249166702524\t0.00321843358837254\t1.01046240780706e-05\t0.000172991879761164\t0.000320147139625372\t0.000227142145895615\t0.00471173391808185\t0.00019596962470817\t0.000625084448909048\t0.000267887781808201\t2.21973141249908e-16\t0.00669098042936231\t0.00101981908409448\t0.00341820202573447\t0.000494950554439612\t0.000662289420476748\t7.35690220711692e-05\t0.0161944015953685\t0.00208948389747732\t0.000581120873141613\t0.00126949093413541\t0.0402052266794683\t0.0139018489459098\t0.00503046079020838\t0.00684148733934757\t0.000853181941048929\t8.64218885038108e-05\t0.00190960910301661\t0.0015\t0.000136975497822949\t0.000245134098157056\t0.000191126525760053\t0.000512001177602709\t0.00260933201100518\t0.00119970607201236\t0.000508331381227422\t0.0013302201514349\t0.00331063378753961\t0.000768846306085721\t0.00287147593863246\t0.00124946135720891\t0.00345827311132186\t0.000146064253519059\t0.0133082480018814\t0.000484084015495529\t0.0402330096559223\t0.00961064781418237\t0.00288517001154068\t0.0158336186050161\t0.000292526689941625\t0.000100648612470242\t0.00536599978536001\t0.00605184642633379\t0.00855143692710394",
                 "C[C>T]A\t0.00200035492297398\t0.00432012476952345\t0.0161064425770308\t0.012698945987483\t0.0193961207758448\t0.0136074562056279\t0.0500324675691796\t0.106971808649548\t0.00271022776754158\t0.0355945362386873\t0.00266989213635769\t0.00723939912987222\t3.68028577419035e-05\t0.000150997886029595\t0.00376117335066708\t0.000907688410990866\t0.00407579654077914\t0.00659086340310581\t1.92991500654311e-05\t0.00273119828593597\t0.0108023127751652\t0.00716777583915711\t0.000821103672549694\t0.000456968661089222\t0.0117076919536135\t0.118942110874637\t0.0106048529928266\t0.0013499366339744\t0.00121055637170844\t0.0622389492277852\t0.00538198056884934\t0.0142977838435043\t0.00169022834985006\t0.017392714192025\t0.000291964672274654\t0.00761111525671856\t0.10998048946117\t0.0411783401930585\t0.0168983118586453\t0.00459200670693093\t0.00474800074508225\t0.00509823753928267\t0.00674833316170906\t0.00361075103621553\t0.00423830044152295\t0.00911118445397902\t0.0196026071467505\t0.00691063301398408\t0.0187040662640058\t7.6216253115977e-06\t0.00357090442081718\t0.00213956203165212\t0.0284\t0.000863845475321375\t1.82099615773813e-05\t0.000470311346111126\t0.00384000883202031\t0.0032291733316271\t0.00326919904623367\t0.0146095239486621\t0.00337055782732002\t0.00725138820533599\t0.000297940441109941\t0.0149076625385448\t0.00808651390385604\t0.00286856758077854\t0.00152066894074637\t0.0180281286478903\t0.00163699905240034\t0.0035426008502242\t0.00838315447947118\t0\t0.00944421488481578\t0.000615846189364626\t0.0366140866274658\t0.0366073485357061\t0.00870141864094458\t0.0136574653755014",
                 "C[C>T]C\t0.000270047914601487\t0.000170004909911803\t0.0202080832332933\t0.0114990455792169\t0.018996200759848\t0.0148081141061244\t0.0553359091315126\t0.181952048357176\t0.00597050176096798\t0.00259960096125245\t0.00401983759856102\t0.00875927298034263\t2.23017317294687e-16\t8.43988184165419e-05\t0.00664382240632547\t0.00220279990566597\t0.126179450648199\t0.000254033278359465\t0.00134994054861824\t0.00138060572695664\t0.0111023770189198\t0.00283911902136767\t0.00172021719462299\t5.88959609150002e-05\t0.00676444423986559\t0.0919552453820725\t0.0130059517836552\t0.00515975780096882\t0.00319146679814042\t0.18911835054745\t0.00885325799894361\t0.00593907944268638\t0.00283038238466017\t0.00713701030638266\t0.000209974593074238\t0.00164024034441767\t0.102981731040913\t0.163913781351009\t0.028397163123404\t0.00153066890231031\t0.000513783659573111\t0.0831712476996702\t0.00667835044743948\t0.0021004368908733\t0.00472810403028386\t0.00386050186524248\t0.0173023012060604\t0.00185016947552396\t0.0481104592138331\t4.35092783536089e-18\t0.0028607245500104\t5.06896238340012e-05\t0.0133\t0.00460917551068465\t0.000301164749164383\t0.000493326582197415\t0.00425000977502248\t0.00469879710794037\t0.00181955420921874\t0.00460300069615382\t0.000226037409191194\t0.00400076590639228\t0.000530893873252949\t0.0121062225984156\t0.00991572533080988\t0.000408795867783422\t0.00188082737408103\t0.0406212999534025\t0.00036937301182363\t0.00484050116172028\t0.00804525631425821\t0\t0.0115589756952006\t0.000307923094682313\t0.00027078261418399\t0.0347613086095477\t0.00399147765653172\t0.0179458356991259",
                 "C[C>T]G\t0.19603478245145\t0.00277008000268055\t0.00224089635854342\t0.0033097252928007\t0.01749650069986\t0.122066886550485\t0.0067143571477839\t0.0220941773005142\t0.00894075138074603\t0.00263959482219479\t0.00148993980643182\t0.00650945971484367\t2.23017317294687e-16\t0.00369994820072518\t0.0094715933278099\t0.000591946436778997\t0.000921310274574155\t4.09053586019769e-05\t0.00198991236425947\t0.0116050916178964\t0.0248053108170459\t0.000690785649213049\t0.00190023992429284\t0.00054296276361367\t0.0168110448564707\t0.015692362527158\t0.016507554186947\t0.00621970804690428\t0.000426195879626276\t0.0142088919458931\t0.00635233766025898\t0.0115982022786468\t0.00334045129496995\t0.00749685956552799\t0.00254969148733003\t0.0153022422375551\t0.016797020208615\t0.00819568906755047\t0.011198881231765\t0.00588257068338864\t0.00944602253495311\t0.0026690772999774\t0.00532868381509767\t0.00327068030150271\t0.00349859706257791\t0.00757098422794963\t0.00236031392175159\t0.0101009252447524\t0.00192041749876423\t9.72207323211675e-05\t0.0155039267570494\t0.00219954975216573\t0.00830000000000001\t0.00151972815102834\t1.87102352470896e-05\t0.00138091416517735\t0.00108000248400571\t0.020694702156248\t0.00237941704282451\t0.0137089368559364\t0.028304684425269\t6.59126183078127e-05\t0.00794841109672495\t0.00640329129172395\t0.00798655699527933\t0.00109945098914857\t0.00406178677594095\t0\t2.4100000771441e-05\t0.00591480141955234\t0.186898985138511\t0\t0.00233381950389823\t0.000200150261550076\t0.00127656739327648\t0.0108239595670416\t0.00724524879481628\t0.0188252293661871",
                 "C[C>T]T\t0.00019603478245145\t0.00145004187865949\t0.0230092036814726\t0.0159986721102149\t0.0230953809238152\t0.0138075658557106\t0.0566367532883113\t0.125966802708814\t0.0206017313695043\t0.000137978820251091\t0.0144994142236654\t0.0110990787764616\t0.000103007998571089\t0.000594991670116617\t0.0098372604814541\t0.00254107581447825\t0.0651927161682364\t0.00338044283801178\t0.00144993614481218\t0.0013605969483051\t0.0156033406752386\t0.0192940130677451\t0.00174021972014186\t0.000135990673759593\t0.00665437192235299\t0.257874492484508\t0.0117053566052897\t0.000324984745216059\t0.00160073569812686\t0.156097686166149\t0.00429157930118284\t0.0135978923266894\t0.00165022294511989\t0.00714700611913669\t0.00331959832860223\t0.000673098629142128\t0.0461918055736912\t0.144923770096929\t0.0427957247071018\t0.00270118041584172\t3.52851423792428e-05\t0.046983757714958\t0.00574858010071512\t0.00606126074223438\t0.00409835655901983\t0.00699090881814636\t0.0216028731821332\t0.00773070813286497\t0.0329071540152829\t0.000744158691841035\t0.00504127682938896\t0.00272944128336929\t0.0064\t0.002999463455977\t0.00047926217558053\t0.000519343805599307\t0.011900027370063\t0.00405896090600806\t0.00256937050422646\t0.0160104372040133\t0.00371061410663421\t0.000719137671674012\t0.000507898470080034\t0.0320164564586197\t0.0276880636757494\t0.00241879217612685\t0.00251110463241669\t0.0422372918322031\t0.0023503060752333\t0.00235840056601614\t0.00696206219600227\t0.00403923801615695\t0.012886270365514\t0.000369507713618776\t0.00122197632877838\t0.0453234781870609\t0.00690584883919963\t0.0250481863476623",
                 "C[T>A]A\t4.30076308439405e-05\t0.000243007018285694\t0.00794317727090836\t0.0143988048991934\t0.00353929214157169\t6.663651347756e-05\t0.000177114935194896\t0.00111970491296724\t0.0034702916433097\t0.0016097529029294\t0.0189992324310098\t0.00478960246299557\t2.23017317294687e-16\t7.0899007413896e-05\t0.000673175995345079\t0.00121612355995871\t0.000184262054914831\t0.000546071535371134\t0.000208990796045342\t0.000155068034549478\t0.000348074522755322\t0.00320900424598247\t0.000434054803759522\t9.35935813521906e-05\t0.000542356328107567\t9.59532995291192e-05\t0.000144065927449719\t6.18970945503818e-06\t0.128058855850149\t0.000462289301338211\t0.00132048593882549\t0.0262959241317596\t0.000944127551632226\t0.330861402158636\t0.000139983062049492\t0.00195028577537466\t0.000213962043133548\t0.0205891700965292\t0.000324967535743179\t0.000746326144525158\t0.0236900247701999\t0.0525818224639742\t0.00125968885685236\t0.00157032662793861\t0.000896640447180681\t0.00494064228349685\t0.00109014498928357\t0.00362033162237661\t0.0126027398356403\t0.00136029008185996\t1.24031414056395e-05\t0.00069685735329978\t0.0042\t0.0139974961278926\t8.6247180657707e-05\t7.41490866953923e-05\t0.000193000443901021\t0.00610843624032248\t0.00121970117321256\t0.00369240708017557\t7.45123317909026e-18\t7.39141501205973e-05\t0.00118976216416386\t0.00639328614908063\t0.00247893133270247\t7.02649132155858e-05\t6.64292221483938e-05\t0.000136549989758751\t0.00989632031678121\t0.00211800050832012\t0.00835215464780026\t0\t0.0315528595471188\t0.102061093982039\t0.000317266974836884\t0.00704591971816321\t0.0106330580703377\t0.0119326265875612",
                 "C[T>A]C\t0.000393069742364387\t0.000244007047167528\t0.0178071228491397\t0.0121989874840388\t0.00518896220755849\t0.000551302085978011\t0.000261169480711117\t0.000957747595198761\t0.0134011262306484\t8.12875223653169e-05\t0.0291988203676571\t0.00312974023156078\t2.23017317294687e-16\t0.000105998516020775\t0.00279159772691936\t0.000614976306505402\t0.000212301932836653\t0.00560073369611419\t0.000341984938983287\t0.00295129485110297\t0.000244052251587065\t0.00569823183866046\t0.0405051141757158\t0.00223984639133448\t0.00306201174171431\t0.000402803955314948\t0.00648296673523737\t0.000687967706795842\t0.0551253356042437\t4.4727990843762e-18\t0.00187068841333611\t0.0157975513795362\t0.00145019592146899\t0.00280882338388449\t0.00134983666976296\t0.0014102066375786\t0.000776862184648443\t0.0210889072347945\t0.000649935071486359\t0.00099243369352407\t0.000408827853629188\t0.0329885958424173\t0.00145963946905114\t0.00116024133019668\t0.00201919030468782\t0.00644083730885015\t0.00762101359480811\t0.00319029223076834\t0.00172037400930962\t1.21025808753717e-05\t0.00325082335228455\t0.00103978715556926\t0.0033\t0.000570897877787622\t0.000102055828620489\t5.14340493406635e-05\t0.00053500123050283\t0.00148961865762365\t0.000230943418862379\t0.00450293546362874\t0.000732121166052896\t1.10021062425788e-06\t0.00097380533436605\t0.006593389001947\t0.0222903906126069\t0.00163918147473059\t0.000514226207594494\t0\t0.00113601703636391\t0.000507990121917629\t0.0049108733339578\t0.00346220401384882\t0.0177097554637826\t0.000354111308878088\t0.000141907106357913\t0.00393188984272441\t0.00851737033945347\t0.00878804385891402",
                 "C[T>A]G\t0.000324057497521784\t0.000149004303393286\t0.0132052821128451\t0.0324973027238739\t0.00500899820035993\t0.000115063048797589\t0.000112072727354962\t0.000882767355491134\t0.000961080769227844\t0.00119981582827036\t0.0275988850050458\t0.00530955930657755\t2.23017317294687e-16\t2.21996892043511e-16\t0.000900188120613321\t0.00100382247105333\t0.000305434384505562\t0.00141018473420018\t0.000225990047398312\t0.000317139141626997\t8.20175599595873e-05\t0.00102968049014391\t0.0124015658217006\t0.000716950831511972\t0.00321211035650422\t0.000293856979807927\t0.000123056313029969\t0.000124994132775407\t0.16807724830332\t0.000132082657525203\t0.00503185172143349\t0.0628902520109383\t0.0011401540348101\t0.00439815761177643\t0.000805902485799217\t0.00198029017191889\t0.000586895884670059\t0.0186901689711212\t0.00140985915507041\t0.000479209414514143\t0.00116950755194658\t0.0581798872129905\t0.00213947155052702\t0.0016003328692368\t0.00177928650611105\t0.00680088411493494\t0.00346046024121208\t0.00312028581818095\t0.0149032399643683\t0.000296063135463638\t0.000582147443393726\t0.00340930211585688\t0.0043\t5.81895910459537e-05\t0.000310169675219132\t0.00023215368573996\t0.00051600118680273\t0.00651833130718536\t0.00122969872381267\t0.00330215267332774\t0.00192031781259776\t0.0160030636255691\t0.00158968221934499\t0.00129066340098811\t0.000857630275588193\t0.00013893062499241\t3.81167675279187e-07\t0.000125175990611801\t0.000178583005716442\t0.00262320062956815\t0.00276728497364259\t0\t0.0368745947522416\t0.000369507713618776\t0.000246237299740317\t0.0102447595902096\t0.0075042415528327\t0.00773463882784318",
                 "C[T>A]T\t0.000260046139986617\t2.23006440649012e-16\t0.0127050820328131\t0.0117990206812834\t0.00429914017196561\t0.000179098136824073\t0.000371240909363312\t0.000111970491296724\t0.0487040930919834\t0.00114982350209243\t0.0417983113482215\t0.00443963151058462\t0.000376029198667275\t0.000233996724045863\t0.00335316785685668\t0.00210910441016432\t0.000162230722261971\t0.00150019652574487\t0.000522976968094324\t0.00138060572695664\t0.00105022485314106\t0.00652797436955313\t0.0723091297507223\t0.00778946579843552\t0.00226148579616808\t0.000217893951014041\t0.000102046698610218\t2.21989579809124e-16\t0.0603277266231559\t0.000375234822514782\t0.0048017670502745\t0.0365943278791787\t0.000642086745919374\t0.00542772632544227\t0.0010298753850784\t0.00198029017191889\t0.000611891450456689\t0.0309837025724469\t0.000662933772916086\t0.00112048965397879\t0.000267887199933062\t0.0663770534526215\t0.00195951599954811\t0.00206042856914238\t0.00235905401933824\t0.00827107523978117\t0.00928123440417576\t0.00901082539160587\t0.00757164607585689\t0.000203043298983508\t1.38034960804698e-05\t0.00133972575813731\t0.0036\t7.59864075514171e-05\t0.000362198136868793\t0.000106070218484637\t0.000209000480701106\t0.0036190735171796\t0.000855790331368814\t0.00861561652040965\t6.66110241244847e-05\t0.0052310014226079\t6.08878284013269e-06\t0.010705502628351\t0.00129943981149726\t0.00370814833612836\t0.000132058092222711\t0\t0.0105511553377425\t0.00333410080018419\t0.0101919446577405\t0\t0.0199476776040129\t0.100521479508653\t0.000460321644647702\t0.00644867974205281\t0.0245884091953206\t0.0144405810037397",
                 "C[T>C]A\t2.23039573911599e-16\t0.000302008722313908\t0.00883353341336535\t0.00517957009568206\t0.0132973405318936\t0.00166091009568693\t0.000451292857474\t0.0015295968900356\t0.00757063623626928\t0.0101984345402981\t0.0023899034479007\t0.0136988629943715\t0.000122009474035658\t0.00131998152025871\t0.0119714886529815\t0.00259925753895609\t0.000328467141369916\t0.0454059481792115\t4.73979125959292e-05\t0.0031613870269442\t0.000750160609386469\t0.0106966808199416\t0.011101401662974\t0.00064695563178277\t0.000386253768726053\t0.000451780118616269\t0.0117053566052897\t0.0114994602153375\t0.00674309912835939\t6.86429568653708e-18\t0.00227083566752565\t0.00902860056691213\t0.0520070261492328\t0.00489794824947829\t0.00131984029932378\t0.00118017293073954\t0.00126977474196078\t0.00814571535372394\t0.000592940765217555\t0.0151066015848926\t0.000593749987911338\t0.00132954037789136\t0.0014996295914909\t0.05001040216365\t0.00194921836343626\t0.00734095432406213\t0.0109014498928357\t0.0103009435664307\t0.00354076976334655\t0.00272058016371991\t0.0146036987518014\t0.0011097728295018\t0.0223\t0.00190965840030535\t0.0001450793642154\t0.000121080155062652\t0.000839001929704438\t0.00417893019387037\t0.00238941459342461\t0.0101065884850334\t0.0114018870122992\t0.00418080037217993\t0.00336932646490101\t0.0120061711719824\t0.00391831081620713\t0.00219890197829714\t0.000879386841392141\t0.00226966582977506\t0.00718403422996094\t0.00128400030816007\t0.00512197218769102\t0\t0.00488962259040099\t0.00477280746756271\t0.000154627505182636\t0.0307645887694164\t0.0295588309410924\t0.00629482606434607",
                 "C[T>C]C\t0.00250044365371747\t2.85008231322728e-05\t0.0145058023209284\t0.00235980413625669\t0.0103979204159168\t0.00103056469792623\t0.00025916818200835\t0.000839778684725427\t0.00358030088848666\t8.88863559443625e-05\t0.00699971721142466\t0.015298730205393\t4.12031994284355e-05\t0.000123998264024303\t0.00978328880945162\t0.00226580111408942\t0.00071401546279497\t0.0497065115530135\t0.000121994627356611\t0.00534234389996267\t0.00215046041357455\t0.00316901665413222\t0.173021845738243\t0.00645955700358067\t0.00246161728255464\t8.1360401892399e-05\t0.00483221131654267\t0.00281986763541319\t0.00134061614718124\t4.4727990843762e-18\t0.00224082462346143\t0.0107983262594298\t0.0312042156895397\t0.00571760489530936\t0.00261968301835478\t0.0018402696547125\t0.00137975523142195\t0.00990479008041771\t0.000926907401950545\t0.0301131594506799\t4.03829958108048e-05\t0.00201930192732372\t0.0018795357546686\t0.0218045353433514\t0.00231907005290878\t0.00791102843369638\t0.00987131288461365\t0.00525048094405448\t0.00313068060996461\t0.105022396025953\t0.00320081068532633\t0.00286941263123439\t0.0834000000000001\t0.00627887683451184\t0.00045024630273745\t0.000178117914059107\t0.0177000407100936\t0.0710818030584171\t0.00130967912861349\t0.00407265496377088\t0.0170028139657093\t0.00498095355345838\t0.00263947236419545\t0.00456234504535331\t0.00949590631478769\t0.00572714015256482\t0.00279122785834366\t3.24999975625002e-06\t0.00620906619875221\t0.00295190070845617\t0.00183399004143407\t0.00173110200692441\t0.00421591913656226\t0.000215545666264475\t0.000146081725298389\t0.00401077983956881\t0.00835610731221965\t0.0149264721337954",
                 "C[T>C]G\t0.000360063886135316\t2.23006440649012e-16\t0.0101040416166467\t0.00549954353788635\t0.0207958408318336\t0.00532291669220149\t0.000104067532543894\t0.000729807666487574\t0.00217018238212739\t0.000934856499527322\t0.00125994909805644\t0.0153987219060818\t0.00105008153883149\t2.52996458049587e-05\t0.0146277873497835\t0.00331748907837822\t0.000131186571705668\t0.0374049000419055\t2.21990223550554e-16\t0.00447196202862043\t0.00139029766272959\t0.00960701894202229\t0.0873110238898761\t0.00391973118483534\t0.0016210650397311\t0.000497757741307306\t0.0261119493502616\t0.0143993240957269\t0.00203093341699845\t0.000389243589222\t0.00422155353169967\t0.0133979233218851\t0.0787106338066273\t0.00565763001878513\t0.00248969874645168\t0.00391057301626407\t0.00126977474196078\t0.00445765527332623\t0.00183981620236139\t0.0400174876420996\t0.00081065865352878\t0.00155946089436882\t0.00184954316283878\t0.00995207003056637\t0.00253898186827082\t0.00777101023133007\t0.0105013966857592\t0.0055705102587397\t0.00356077411229201\t0.00925197298323868\t0.0338085628637593\t0.00155968073335388\t0.0504\t0.00309944557117623\t3.21175695952714e-05\t0.000108071543361705\t0.0024200055660128\t0.00292925011197134\t0.00122969872381267\t0.00849553824137956\t0.19303194678717\t0.000949181711291566\t0.000480903866355308\t0.00830426839395449\t0.00592744467859906\t0.00163918147473059\t0.00058025525370585\t0.0036283147278764\t0.000839952026886864\t0.000557810133874432\t0.000836095460001652\t0\t0.00809754694911724\t0.000200150261550076\t0.000150462619689685\t0.0141228094350876\t0.00237811593560737\t0.00704415205886581",
                 "C[T>C]T\t4.26075598593457e-05\t0.000186005372021149\t0.0158063225290116\t0.00255978753763438\t0.0148970205958808\t0.00127069627802554\t0.00182118181951814\t0.00572849031366274\t0.0431036224284288\t0.0396939069852777\t0.00645973902654333\t0.0328972695266293\t2.44018948071317e-06\t0.000814988590159735\t0.0192489869472935\t0.00474692720659948\t8.46203458712131e-05\t0.0459060136877931\t0.000111995067737217\t0.00206090420110919\t0.0017503747552351\t0.0154951918419714\t0.426053793551974\t0.0182987450720629\t0.00256168302574792\t0.000390809792873808\t0.00690315902363239\t0.0216989814498107\t0.000536246458872497\t4.71294937078566e-05\t0.00363133633177009\t0.00895861141523064\t0.0555074990631234\t0.00389836697407456\t0.000366955598372596\t0.00178026086162405\t0.00409927278900723\t0.0137927450161215\t0.00108989111987712\t0.00307134217653114\t0.00158933077572227\t0.00317890105390567\t0.0021594666117469\t0.0504104853809592\t0.00426828841634505\t0.00859111684518987\t0.0136018090406024\t0.00787072095803976\t0.00324070452916464\t0.00945201564233573\t0.00771195324495812\t0.000601876795819896\t0.0958000000000001\t0.00544902527835821\t6.77370548785008e-05\t9.74645215132417e-05\t0.0013800031740073\t0.00663830059504767\t0.0023394268404241\t0.00542353560285951\t0.0269044526869165\t0.000684130969993079\t0.000205958828418282\t0.0243124966232644\t0.028187848218633\t0.0121939109705569\t0.00210092419445221\t0.000918581931106355\t0.0169131405413896\t0.0017926004302241\t0.00885715190566515\t0\t0.00431372742776967\t0.00635861115517005\t0.000163439824179005\t0.013670349453186\t0.0148583189280317\t0.0161063564789715",
                 "C[T>G]A\t3.55062998827881e-05\t0.000153004418920622\t0.00437174869947979\t0.000869927795992932\t0.00258948210357928\t4.95271383954838e-05\t8.42546753864984e-05\t0.000358905414067177\t0.00328027567436767\t0.00147977285486678\t0.00121995071399115\t0.0259978421790991\t0.000155012036684648\t0.000226996822044491\t0.0108141450892023\t0.00634234576750458\t2.22316174951589e-16\t0.000893116998326781\t4.13981768242926e-05\t4.8521288229998e-05\t0.000206044114044817\t0.00395877159317464\t0.000310039145542516\t0.00204985942064093\t0.000111072974944539\t0.000713652665247824\t0.000963440889819999\t4.34979582058418e-06\t0.00491225767362679\t6.85428942460335e-18\t0.00143052643372761\t0\t0.00142019186792136\t0\t0.0145982336137327\t0.00078111445670134\t0.000197964881030105\t0.00196896432476517\t5.80941963897807e-05\t0.000728318275086213\t0.00105955385048151\t0.00370871789622328\t0.000477881963155101\t0.0026005409125098\t0.000982605975004023\t0.00530068908958164\t0.00335044560926603\t0.00347031788111791\t0.000870189179127542\t0.00273058229667477\t0.000234059281364488\t0.000101979124873138\t0\t0.000136975497822949\t0.000134073343481818\t0.000106070218484637\t0.000894002056204729\t0.00277928850214345\t0.000802803313188269\t0.00149097196462374\t0.00436072169944074\t0.00165031593638681\t0.00581883680288543\t0.000140071997006461\t0.000502783250140864\t0.00100949590821823\t0.00323142149908603\t0.00089719293271053\t0.00373647511960457\t0.00407170097720823\t0.00851535376162907\t0\t0.00690301724448911\t0.00894516415047519\t0.000758216701397419\t0.000928399962864003\t0.00932069227426447\t0.00222808095446599",
                 "C[T>G]C\t0.000212037621835242\t0.000155004476684291\t0.00704281712685074\t0.000440963400037797\t0.00460907818436313\t0.00078442982832443\t0.000172111688437978\t0.000791791331312546\t0.00558046898261329\t4.97923568732199e-05\t0.00194992122318258\t0.00740938502104325\t2.23017317294687e-16\t0.00114998390022539\t0.00229699542020507\t0.000494857505780763\t2.22316174951589e-16\t0.0011401493595661\t2.8198758126692e-06\t0.00121053110841851\t0.000157033620898234\t0.00107966497995672\t0.00104013132698134\t0.0313978467356708\t0.00199130828954623\t0.00208898329183187\t0.0016007325272191\t5.43974465838573e-05\t0.000880404633969772\t0.000148092676619167\t0.000880323959216992\t0.00288955211942149\t0.00494066748417711\t0.000380840465928822\t0.0220973262235269\t5.69083387788813e-06\t0.000214961865765013\t0.00142924821543868\t0.000449955049490556\t0.000949414894308813\t2.21906561138581e-16\t0.000517820989284005\t0.000565860232522567\t0.00238049514298974\t0.00140943481663853\t0.00739096082490724\t0.00567075421030997\t0.0024102207762231\t0.000753163737796597\t0.00726154852522301\t0.00253064094808613\t0.000280942491072078\t0.0018\t5.14907893276051e-05\t9.28507930978563e-05\t2.67176871088661e-05\t6.18001421403269e-05\t0.00255934480772922\t0.00077481017150798\t0.00424276585906352\t0.0189031284677592\t0.0004660892280947\t3.92921454215459e-07\t0.000847435581889091\t0.00103955184919781\t0.00261869235597205\t0.00416183078520058\t0.00214887283883454\t0.00247817007932622\t0.000164090039381609\t0.00108389411445496\t0.00230813600923254\t0.00321981890768684\t4.62012145375278e-05\t5.88139112417706e-05\t0.000418779983248801\t0.00636669561753989\t0.00205256494043526",
                 "C[T>G]G\t0.000128022715070335\t0.00012100349470193\t0.0107042817126851\t0.00378968545610714\t0.00612877424515097\t0.000924506583382364\t2.23144805358541e-16\t0.00128966012297119\t0.00119010001600534\t2.57960403078127e-05\t0.00469981012767084\t0.0141988214978157\t5.35041545976043e-05\t0.000478993294093881\t0.00686482897267188\t0.0030767934346793\t2.22316174951589e-16\t0.000593077693177806\t2.21990223550554e-16\t0.000435190935671116\t0.000272058247670826\t0.00223930514361394\t1.94024497533058e-06\t0.0136990605184296\t0.00293192627556305\t0.00132935300389301\t0.00271124071797736\t2.00990565502855e-05\t0.00803369228497417\t6.52408278079034e-05\t0.00304111913184052\t0.0117981712834511\t0.00556075125749489\t0.000402831253987705\t0.015498124726908\t0.00033204865508943\t7.84860765700165e-05\t0.00149921141479582\t0.00108989111987712\t0.00107046779442616\t0.000435816489443339\t0.00635780210781133\t0.00088678096510162\t0.00377078432313921\t0.00158936266557111\t0.0098312780661486\t0.00559074356889466\t0.00304027848950964\t0.00568123510051085\t0.0465099182400647\t0.00347087908690073\t0.000648867176888891\t0.0016\t0.00366934362781186\t0.000104056923299322\t0.000112074193115843\t0.00366000841801936\t0.00592848230852902\t0.000795805027768197\t0.00298194392924748\t0.0221036581554221\t0.113021636855582\t0.00444911061389006\t0.00219112623888679\t0.00069869879095122\t0.00201899181643646\t0.00711312905835964\t0.000788670940849679\t0.00112029903586077\t0.0017689004245361\t0.00195898936268776\t0.00230813600923254\t0.00903011058723664\t0.000123169237872925\t0.000428859838118185\t0.0020905199163792\t0.00255246177759665\t0.00251354027301379",
                 "C[T>G]T\t0.000171030345914275\t0.000203005863012329\t0.0070328131252501\t0.00119990040826611\t0.00729854029194162\t0.00130071272553796\t2.65172078116652e-05\t3.67903042832092e-06\t0.00340028576001527\t0.000155976057675147\t0.000573976811336822\t0.0564953108891962\t0.00130010095283898\t0.00522992678102505\t0.0300124607933175\t0.00592491785696071\t0.000133189420128655\t0.000344045069904158\t5.54975558876386e-05\t0.000962422253139341\t0.00249053322316308\t0.0025392120824908\t0.00531067052526051\t0.549962283586591\t0.00414272176820171\t0.00219892978087565\t0.00511233950880601\t0.000184991316507603\t0.00119054717548185\t0.00024715466976307\t0.00471173391808185\t0.00854867495538192\t0.00780105392238491\t0.00143939703658138\t0.100987780478562\t0.000767112404980702\t0.000264952997338272\t0.000948501088427488\t0.000675932474345813\t0.000192083940682078\t6.54724313269237e-07\t0.00270906347675609\t0.00106973577526351\t0.00411085505785203\t0.00687724222586743\t0.00494064228349685\t0.0131017425317567\t0.0105009618881089\t0.00348075671651017\t6.26133522973774e-18\t0.000384097282239159\t0.000727851008898479\t0.002\t0.00010698086326318\t6.57359601996676e-05\t9.29615405398373e-05\t0.000471001083302492\t0.00229941135069422\t0.000862788616788886\t0.00272177432468226\t0.000886146657271674\t0.00118022594238572\t6.77864493532014e-18\t0.00516265360395243\t0.000472796177567851\t0.0211894190635906\t1.80079216667333e-05\t0\t0.00175687005623741\t0.00242130058111214\t0.00308218326374488\t0\t0.00417984150458999\t0.00970025500030345\t0.000122273228951029\t0.000545189978192401\t0.0382202930034292\t0.00279692661914025",
                 "G[C>A]A\t0.00158028038914944\t8.62024896141022e-05\t0.0071328531412565\t0.0315973774176743\t0.0101979604079184\t0.000427234102926699\t4.06263636661738e-05\t0.000745803450958534\t0.00105008824941648\t0.00016797421595785\t0.0241990223594967\t0.00771935929317867\t0.00121009396379628\t0.000730989766143272\t0.00841770798986352\t0.00241907175719314\t0.000165234994896452\t0.00198025941398323\t0.000948958207880523\t0.00208091297976074\t0.00229049039399335\t0.00860732914576607\t0.00178022477117961\t0.000997931561853485\t0.109071660080673\t0.00081360401892399\t0.010804944558729\t0.00213989955311497\t0.00223102537926431\t0.000337211027166617\t0.0622228980264737\t0.0134979078242872\t0.00049606701865422\t0.0107954777743603\t0.000155981126283719\t0.0971142301481436\t0.000942832741471663\t0.0130931130225501\t0.00450954949600535\t0.00441192801254148\t0.00280881728288024\t0.00986658912014116\t0.0676832822292894\t0.00204042440827692\t0.0119951899288385\t0.00688089451628712\t0.0142018888512172\t0.00309028306992921\t0.00771167651847512\t0.00930198364801294\t7.68194564478318e-18\t0.00754845483129604\t0.0072\t0.0110980147871149\t0.000752411599241249\t0.00788522001565035\t0.0142000326600751\t0.00393899161814576\t0.00823798169448485\t0.000432281804508359\t0.00519085908717831\t0.000855163712491349\t0.000216956629935763\t0.00189097195958723\t0.00273881929500193\t0.102948592620275\t0.000517227527872283\t0.00270347179723962\t0.00143970304608489\t0.00484250116220028\t0.00710276143200542\t0.00865551103462204\t0.00678199483353791\t0.000969957498242714\t0.00352466455435275\t0.00893930964242762\t0.00224220596287792\t0.0138926537338449",
                 "G[C>A]C\t0.000339060159444089\t2.23006440649012e-16\t0.0109043617446979\t0.0347971118397173\t0.00764847030593882\t0.000913500552627812\t0.000138089610490936\t0.000214943353828532\t0.000742062362920978\t2.21965928230017e-16\t0.0261989415627609\t0.0066794456060147\t0.000441034246309224\t0.00254996430049979\t0.00743435619580612\t0.0250397320739673\t0.000405576805654926\t0.000417054634157075\t0.000450980138834685\t0.022409832089731\t0.00249053322316308\t0.00424868163408894\t0.00128016163320781\t2.77980936067404e-06\t0.0173113735724371\t0.000271867681999171\t0.00565258673674246\t0.00706966814977705\t0.00173079547359966\t0.000541338770614659\t0.128047121340653\t0.00646899730541766\t0.000666089988757481\t0.0123948078150063\t0.00101987659493201\t0.0903132335981191\t0.000493912379943798\t0.0214886969454067\t0.00361963839812403\t0.00096842320093881\t0.00046780302077863\t0.050182651857253\t0.0191952587710835\t0.000629130859218717\t0.0062774827294255\t0.00695090361747027\t0.0102013567804518\t0.000723066232866931\t0.0268058275869174\t0.00169036046937009\t0.00467118309389811\t0.00572882730904984\t0.003\t0.00347937760893332\t0.000346189379438128\t0.00248164284756509\t0.0075800174340401\t0.00327916053490307\t0.00731820703927537\t0.00344224399886286\t0.0091315112651133\t0.000192036763506829\t0.00733853301257372\t0.000752386726777564\t0.0010795346126285\t0.12893561600015\t0.000866381120188389\t0.0102288582328356\t0.000504268016141619\t0.00362910087098421\t0.00466597466375758\t0\t0.00602570551127272\t0.00041569592781455\t0.00587028924961475\t0.00782491968700321\t0.00170024864534753\t0.0304499594706549",
                 "G[C>A]G\t0.000587104169892862\t1.39004014574944e-05\t0.00143057222889156\t0.0152987302053929\t0.00233953209358128\t0.000357195725397732\t4.25275974338026e-05\t7.28807929958139e-05\t2.22018658448056e-16\t1.81972067287671e-05\t0.00273988930847194\t0.000170985808177921\t1.05008153883149e-05\t8.51988072166987e-05\t0.000892211501283164\t8.7330782087956e-05\t1.39197965397617e-05\t0.000447058564671972\t0.000163992777758067\t0.00111048721516078\t0.000404086514922845\t0.00135957812290846\t0.000106013385250022\t9.23936636425471e-05\t0.00756497018541182\t0.00020490027503614\t0.00157071879233375\t0.00168992067512351\t0.000515236802834582\t0.000119074517011357\t0.065924260127727\t0.000657898025806\t0.000129017430254827\t0.00162931747890809\t3.94952210782495e-05\t0.0230033706839063\t2.21960624185269e-16\t0.00097148899678769\t0.000876912396451594\t0.000342149519339952\t0.00022690445665972\t0.000506824790669866\t0.0048288072846007\t0.000506105269896138\t0.00136945085020907\t0.00453058897656695\t0.00255033919511295\t0.000220020153846092\t0.00106023049410942\t0.000177037753300891\t0.000605153270194508\t0.00208957226455745\t4e-04\t0.000782859962009995\t0.000739404483828834\t0.00683452445518933\t0.000244000561201291\t0.00427890460042229\t0.0040390104424416\t0.0508331381227422\t0.00564093357450591\t7.72147819933709e-05\t9.78804335055814e-06\t6.57337871666036e-18\t0.00272882360414425\t0.0120939608806343\t0.000390171636112554\t0.000763600942729929\t0.000362283011596679\t0.000223040053529613\t0.00828785499694737\t0.000577034002308136\t0.0016799362984955\t0.000123003233509002\t5.90525973571342e-05\t0.00445573982177041\t0.000295683073906087\t0.00176450493428396",
                 "G[C>A]T\t2.23039573911599e-16\t5.12014787499076e-05\t0.0100040016006403\t0.0211982405460347\t0.00681863627274545\t0.00123067434800899\t2.23144805358541e-16\t0.00014696126982695\t0.000363030509083983\t7.39886427433388e-05\t0.0268989132839033\t0.00855928957896495\t0.0111008619819329\t0.00279996080054878\t0.0120575353062343\t0.0098893416946934\t2.22316174951589e-16\t2.75036029719893e-05\t6.14972916592752e-05\t0.126055305504737\t0.018203897454445\t0.00579820081828609\t4.70059349693492e-05\t0.000533963380791344\t0.0623409580094122\t0.000272867195535933\t0.00249113999548473\t0.00889958225360902\t0.000551253356042436\t3.0919349375218e-05\t0.0544200265697777\t0.0104983727522234\t0.000817110391613907\t0.00740689725074166\t0.000157980884312998\t0.0846123982547162\t0.000861847108322983\t0.00350815471062221\t0.00326967335963137\t0.00319139463945744\t1.34943179070759e-05\t0.0228920862057987\t0.0689829612085815\t0.00129026837582217\t0.00908635637109519\t0.00589076579955394\t0.0108014365910666\t0.00195017863636309\t0.018103935795642\t0.00139029648072451\t0.0113028627325586\t0.00698856943983567\t0.0024\t0.00140974782430919\t1.99108920543894e-05\t0.00584386864104037\t0.00204000469201079\t0.00344911702604133\t0.00420896880264335\t0.00114074365078595\t0.00673111399936611\t0.000738141309729374\t0.0156968621658593\t0.00195100281544714\t0.000349849180018494\t0.49875094871376\t0.000133058532315307\t0.0101791652365626\t0.000762947024421934\t0.00837060200894448\t0.00106749420350647\t0.00346220401384882\t0.00723720757232607\t0.000400299523073863\t0.00116701690210707\t0.00655608973775641\t0.00310058115756285\t0.0113442323309291",
                 "G[C>G]A\t9.59170185566022e-06\t6.54018888719523e-05\t0.0127050820328131\t0.00528956096643978\t0.00467906418716257\t1.9410636058028e-05\t1.96127272871184e-05\t5.10865366541302e-05\t0.000182015296565523\t0.000195969918617492\t0.0025798957722108\t0.00135988712936826\t1.50011648404498e-05\t1.71997592033711e-05\t0.000170107336017754\t0.000104301008384182\t4.67665106767532e-05\t0.000564073893680072\t0.00149993394290915\t2.22097443032156e-16\t0.000994212860973534\t0.00104967428606903\t0.000878110870278479\t5.98958923397031e-05\t0.000635417469277315\t0.00130936273315777\t0.000669306287943488\t0.000387981788134864\t0.001400643735861\t4.32270515537029e-18\t0.00272100132848888\t0.00416935375016873\t0.000395053371710518\t0.0014194054110733\t0.000182977859678979\t0.0022703326718464\t0.000514908655204567\t0.0018290379260509\t0.000875912496341615\t0.00077133707430147\t3.33859420812099e-05\t0.0033788319377991\t0.000885781212040626\t0.00132027461712036\t0.00119951899288385\t0.0240031204056527\t0.00506067306951825\t0.00204018688111831\t0.00240052187345529\t0.0136029008185996\t0.00021905548127702\t0.00033293184885054\t1e-04\t0.00167969953534712\t7.84429114102579e-05\t0.000152100690657215\t0.0267000614101412\t0.00340912726342057\t0.00137966198281421\t0.00138090020884615\t0.00219036250499432\t0.000851162946584955\t0.000186962625797178\t0.00379194906181777\t0.00211908646182631\t0.0014392812948854\t0.000163071735093196\t0.0111510591636706\t0.00159592405108553\t0.00555880133411232\t0.00261758578650918\t0\t0.00652198769274868\t0.000431092332555238\t3.75650264867377e-05\t0.0023488099060476\t0.00353212078569927\t0.00311400758950028",
                 "G[C>G]C\t0.000164029103683866\t0.000261007538158709\t0.00863345338135254\t0.00574952278960846\t0.00507898420315937\t0.000685375551533463\t3.32215584659352e-05\t0.000451880911304635\t0.000473039754260947\t2.21965928230017e-16\t0.0023899034479007\t0.00282976512949425\t8.31064532160919e-05\t0.000135998096026655\t0.000388351914346128\t0.00032515004252862\t0.000114162360110276\t0.000289037863960179\t0.000478978905768989\t4.33190057805961e-05\t0.000608130200675964\t0.00308904147043172\t0.000835105440412905\t5.54961940710105e-05\t0.00134088095878993\t0.00128937246242254\t0.00286130939240415\t0.00141993334832863\t0.000930427624536236\t0.000127079526558339\t0.00579213150439362\t0.00238962960741085\t0.000579078233469342\t0.00134943472179504\t2.21973141249908e-16\t0.00164024034441767\t0.000255954593655086\t0.0075060518167444\t0.000899910098981111\t0.000776339260256732\t0.000192918767115974\t0.0213926045765979\t0.00107973330587345\t0.000938195144590076\t0.00107956709359547\t0.0213027693600168\t0.00537071430500257\t0.00269024642657267\t0.00142030877512771\t0.015203242091376\t0.00119030147360573\t0.00100979329531245\t0.000800000000000001\t0.00227959222654252\t0.000217118772653392\t0.000110072868238774\t0.0330000759001746\t0.00113970823469192\t0.00144964483701493\t0.0847552519487453\t0.000220036416026826\t0.00141026998200328\t9.82803535607625e-05\t0.0025713216593329\t0.00148935793779302\t0.00171914154666867\t0.000135059412500499\t0.0157583408181244\t0.00506036216198219\t0.109340026241606\t0.00119829349326633\t0.00173110200692441\t0.00459041393609396\t0.000292526689941625\t0.000189375798481573\t0.0021613199135472\t0.00466474094465029\t0.00688704108540809",
                 "G[C>G]G\t0.00016602945860684\t2.23006440649012e-16\t0.00242096838735494\t0.00329972612273181\t0.00150969806038792\t2.22121711591867e-16\t4.51292857474e-05\t0.000336911210419606\t0.000180015128471397\t0.000544916355339455\t0.000487980285596462\t0.000520956760588871\t2.23017317294687e-16\t2.21996892043511e-16\t3.28475745365971e-05\t1.91048511930844e-05\t3.24461444523941e-06\t9.54124990373739e-05\t0.000240989386827404\t2.22097443032156e-16\t1.01021628730711e-05\t8.12747804356308e-05\t2.37029927398633e-05\t2.22984707708745e-16\t0.000639420099005047\t0.00059670958144671\t0.000692316818022263\t0.000116994508277781\t0.000211097020190479\t1.83114593387214e-18\t0.0031411559453879\t0.000417935220040894\t7.38099717271803e-05\t0\t2.21973141249908e-16\t6.88100827414241e-06\t0.000116979247881426\t0.000456759744374459\t6.85931475445603e-05\t4.6620373103046e-05\t2.21906561138581e-16\t0.00059079576190511\t0.000171957526490957\t0.000577120040968521\t0.00101959114395128\t0.00986128196665567\t0.00119015829105271\t0.000156014290909047\t0.000223048490741887\t0.0128027301822114\t7.68194564478318e-18\t0.000694857762615993\t0\t0.000464916835676434\t4.31235903288535e-05\t3.67243114942092e-05\t0.00280000644001481\t0.000266931665493634\t0.0010097526106104\t0.00300195697575249\t0.00159026318855752\t0.00572109524614095\t5.61887677529487e-06\t0.00186095653165727\t0.00170926313666179\t0.000358820822822124\t0.000202088898704451\t0.00169796687265248\t2.6800000857868e-05\t0.000763800183312044\t0.00545547037679585\t0.000577034002308136\t0.00207992113147062\t4.62012145375278e-05\t6.39000484353512e-05\t0.000893369964265202\t0.000911111222338981\t0.000704123400744544",
                 "G[C>G]T\t2.23039573911599e-16\t0.000134003870165774\t0.0147058823529412\t0.00303974770094082\t0.00671865626874625\t0.000321175988382834\t4.20272727581108e-05\t0.000276927018653504\t0.00128010758024104\t0.000881864633778714\t0.00485980366393198\t0.00856928874903383\t3.54027490234615e-05\t0.000112998418022147\t0.000154679646344372\t0.000370064932523411\t0.000186264903337818\t0.000504066032650277\t0.00269988109723647\t8.03352462859554e-05\t0.00045109657977773\t0.0032289980419076\t0.000787099379167612\t2.22984707708745e-16\t0.00123080864127732\t0.000319844331763731\t0.00302138264512606\t0.000206990283876075\t0.000455209214154825\t4.04252982122592e-18\t0.00338124429790163\t0.00198969159780234\t0.000305041211067615\t0.00455809061584103\t2.21973141249908e-16\t0.00321047043023214\t0.000398929229954606\t0.00335823356914263\t0.000227977225075215\t0.000931407024869868\t2.6888677903729e-05\t0.00873697962614324\t0.00104974071404363\t0.00127026421495671\t0.000916632430395411\t0.0324042125476312\t0.00698092846348564\t0.00356032612587313\t0.00277060232894631\t0.0180038393187347\t0.00185046867745428\t0.000931809258644756\t6e-04\t0.00221960295742298\t6.5335741263901e-05\t0.000208137787215136\t0.0143000328900756\t0.0028492705867298\t0.0010197501612105\t0.000573373782368726\t0.00494081770532965\t6.82130587039883e-05\t1.74965024141744e-06\t0.000436224219248694\t0.0010795346126285\t0.0022588720322507\t6.43282979539416e-05\t0.0760575552956834\t0.00318610410198719\t0.0124540029889607\t0.00449917556947666\t0.00692440902769764\t0.00571656723348707\t0.000261734880486538\t0.000157899032918614\t0.0015274799389008\t0.00567373797547205\t0.00369182139572837",
                 "G[C>T]A\t0.00444078792900222\t2.23006440649012e-16\t0.01640656262505\t0.00408966055817367\t0.0193961207758448\t0.093651316238733\t8.49551299324669e-05\t0.000854774732666953\t0.00015701319538894\t0.000364943981098901\t0.000275988850050458\t0.00697942070808123\t4.22032770844654e-05\t0.000362994918071146\t0.00361386724788926\t0.00115448152460712\t0.0197280569664248\t0.00448058695689135\t0.00717968380672514\t0.0171075057470715\t0.0737157825490437\t0.00649798367566544\t0.0016402070925475\t6.10958100493466e-05\t0.00968636394110931\t0.0609703257424611\t0.0422193204054039\t0.00931956253973438\t0.0023810943509637\t0.0537336265841168\t0.0299110072506682\t0.00999845024021277\t0.00275037157519981\t0.014194054110733\t0.00324960679757749\t0.0104015241353315\t0.0496911847838193\t0.00679642508040771\t0.0225977424855257\t0.00535233897213082\t0.000232901931285087\t0.00728748071791582\t0.00371908138689744\t0.00255053051034615\t0.00236905001094561\t0.00803104403572464\t0.0143019021529863\t0.00601055056643188\t0.033607306228374\t0.000766163384341711\t0.0955241938902075\t0.000999795341893515\t0.0126\t0.00105981042111187\t0.000345188832098711\t0.000184121888690313\t0.0024200055660128\t0.00446885597287095\t0.00177956400681833\t0.00918598834580262\t0.000222036747081616\t0.00650124459788745\t9.48810330917229e-06\t0.00514264331866579\t0.00677707840150111\t0.00243878219411137\t0.0012405457148194\t0.0891775543116834\t0.000887033028393927\t0.00453360108806426\t0.00188168978242448\t0.000577034002308136\t0.00685369311481078\t0.000446488737295926\t0.0113072549018697\t0.0128258294869668\t0.00611424609789149\t0.00559135784390141",
                 "G[C>T]C\t9.28164684259925e-05\t1.29003725756603e-07\t0.0134053621448579\t0.00445962985072239\t0.0215956808638272\t0.111060855795933\t0.011707597411188\t0.0262930707241414\t9.51079928757211e-06\t0.000180972220764113\t0.000244990102399863\t0.00652945805498144\t0.0125009707003748\t0.0150997886029595\t0.00902990736919313\t0.00556684838966248\t0.115163784321769\t0.00182023845123711\t0.000920959440946219\t0.00816358168983059\t0.129027624814473\t0.00528835902219541\t2.22028033259479e-16\t0.000156989233678354\t0.00817537121889081\t0.0396806874094378\t0.0532243565300352\t0.0195990800191839\t0.00187085984718576\t0.154096433779404\t0.0437160875202074\t0.00694892291694787\t0.0054507363944869\t0.020091583635615\t0.000187977254752175\t0.00796116654973453\t0.0609891805193759\t0.0113940067524482\t0.0378962141682046\t0.0051422471620098\t2.21906561138581e-16\t0.017793848666516\t0.00339916040737938\t0.00231048057996063\t0.00295881351578017\t0.00450058507605989\t0.0116015430052197\t0.00864079149650108\t0.152033051985502\t4.10087451148958e-18\t0.0814206218079884\t0.000932809053986649\t0.0045\t0.00170969416990689\t7.39404483828834e-06\t0.000118078167747049\t0.00158000363400836\t0.00308920916245441\t0.00121970117321256\t0.0119077626704849\t0.0131021684088701\t0.00151028912966308\t0.000505898869804129\t0.00693356385181984\t0.00314864262016645\t0.00234882711318103\t0.00144063373333866\t0.111629542627784\t0.000484366015504556\t0.00337430080983219\t0.00261808578379419\t0\t0.00852925957900602\t0.000323319499423001\t0.0001314549393008\t0.0135109094595636\t0.00513214572788774\t0.0167729365930717",
                 "G[C>T]G\t0.218038686604164\t2.23006440649012e-16\t0.000528211284513806\t0.000481959997320222\t0.0122975404919016\t0.178097588573659\t5.52358441963743e-06\t0.000175953629180566\t3.43028828142717e-05\t0.000156975904198705\t0.0019099228391173\t0.00475960495278892\t0.0149011570748468\t0.0125998236024695\t0.00922420356462418\t0.00204662440235061\t0.00178253509645869\t7.46097738803783e-05\t0.000177992161225219\t1.92084275054837e-05\t0.27705931840007\t0.00116963706161978\t0.0016102033042692\t0.000854941368120971\t0.0148097299926051\t0.00320843845300492\t0.0205093855049948\t4.33979628996214e-05\t2.22102078115101e-16\t0.0123077021784848\t0.0254093506410359\t0.0104983727522234\t0.0066809025899399\t0.00936607655053297\t0.000214973988147434\t0.0183026818919776\t0.00523907058887753\t0.00196896432476517\t0.0103989611437817\t0.00584255319574654\t0.016692971040605\t0.00252912568125199\t0.00569859244766543\t0.00432089874693936\t0.00447820424009972\t0.00548071249262404\t3.93052275952702e-18\t0.0171015665034917\t0.00455098938509232\t9.66206043438763e-06\t0.049612565622558\t0.00150969096625921\t0.0015\t0.00171969238142681\t0.000303165843843216\t0.00104068893607568\t0.0055000126500291\t0.00427890460042229\t0.00282930681982914\t0.00599390742825248\t0.0191031615732381\t0.011902278571517\t0.00205958828418282\t0.00844434039096095\t0.00735682847124604\t3.23838291349215e-18\t0.00336147871112354\t0.00922392430820568\t0.000116424003726732\t0.00797200191328046\t0.0177089038406521\t0.0063473740253895\t0.00187992871498306\t0.000169004442788792\t0.00119186093295401\t0.00593887976244481\t0.00901405588618694\t0.0169330112577997",
                 "G[C>T]T\t3.84068145211004e-05\t5.04014556444403e-05\t0.0113045218087235\t0.00380968379624491\t0.0185962807438512\t0.0744407898307877\t0.0161104545572758\t0.00538857989365483\t0.000758063707673991\t0.0420935386418184\t0.00460981376352395\t0.0106991119737062\t0.0135010483564048\t0.0318995534062522\t0.0103831172514751\t0.00224814230488493\t0.0696991251199578\t0.00267034981582587\t0.00126994407166308\t0.0105046087920614\t0.0772165320595139\t0.00671791543084181\t6.60083342122776e-05\t2.22984707708745e-16\t0.0066943982196303\t0.108946975507021\t0.0331151541568452\t0.00804962215073625\t0.00176080926793954\t0.107067002690884\t0.0203074731501192\t0.00981847813588894\t0.00498067288890729\t0.00758682188031433\t0.000101987659493201\t0.00512075034354784\t0.0176968605769337\t0.00569700337622411\t0.0531946858508835\t0.00393171816083629\t0.000516782396885795\t0.00602791614938716\t0.00362910361140798\t0.00232048266039336\t0.00319871731435694\t0.00861111944552792\t0.0127016893246802\t0.00841077042657107\t0.0865188091891177\t0.00186039672960259\t0.0666168723883542\t0.00198959273036809\t0.004\t0.00403927745404902\t0.000110060207335821\t0.000308204031068568\t0.0163000374900862\t0.0059484771898394\t0.0010197501612105\t0.00310202220827758\t0.000248041050793877\t0.00529101291120379\t0.000829834114500843\t0.00618317815357094\t0.00502783250140864\t0.00295852266170888\t0.000894393442781086\t0.180440442466967\t0.00416710513338904\t0.00713600171264041\t0.00144639214609065\t0.00346220401384882\t0.00922585116494968\t0.000508073356232389\t0.000150440050983366\t0.0168461293261548\t0.00451193335196447\t0.00843809425152292",
                 "G[T>A]A\t8.12144098727434e-05\t2.23006440649012e-16\t0.00706282513005202\t0.00921923480351131\t0.00366926614677065\t5.10279607711045e-05\t0.00040126038990482\t0.000525861414482827\t0.00173014540141953\t2.36963626083396e-05\t0.0127994829008908\t0.00264978006825433\t3.31025704145926e-06\t2.21996892043511e-16\t0.00190231516704291\t0.00154636609790061\t1.54219328570021e-06\t0.000826108220176843\t6.69970494499421e-06\t5.7725326409709e-05\t0.000511109428528648\t0.000939708408480848\t6.46081574259565e-05\t9.72933276235913e-05\t0.00160105189109245\t0.000188908058447953\t5.56254553208639e-05\t0.00116994508277781\t0.0537246918683827\t6.26391997051342e-18\t0.00157057797269395\t0.0361943898695702\t0.000783105797593255\t0.109953940294411\t0.00011598596569815\t0.00343050267155646\t0.000328941645752044\t0.00189900112540804\t0.00878912196671552\t0.000810354124752517\t0.018992002980329\t0.000716752218757976\t0.000879782693674662\t0.00115023924976395\t0.000892642050537734\t0.0049606448838349\t0.00265035249688209\t0.0023302134475518\t0.0120026093672764\t0.00618131816609892\t0.00116029387343079\t0.000299938602568054\t0.0061\t0.0159971384318773\t0.000130071154124152\t5.04333869021292e-05\t0.000732001683603872\t0.0203947789365922\t0.000523871651445396\t0.0044529028473662\t0.00233038567882957\t0.00243046528813331\t0.000419916057940186\t0.00333171250022512\t0.00206910800753795\t0.000254872729302623\t0.000109047970092996\t0.00109863891760208\t0.0172392335518279\t0.00125920030220807\t0.00940164894904621\t0\t0.00581045367340716\t0.0389676613818824\t0.000273731615485016\t0.00562395977504161\t0.00577854418996209\t0.00566955886960164",
                 "G[T>A]C\t0.000129022892531822\t0.000131003783520271\t0.00748299319727891\t0.00359970122479834\t0.00290941811637672\t0.000489268094452355\t0.000224145454709925\t0.000255932551535368\t0.00227019078683372\t0.000111982810638567\t0.0134994546220333\t0.00211982405460347\t2.23017317294687e-16\t0.000122998278024107\t0.00688938564485282\t0.00335296207575771\t0.000375534079310117\t0.00181023714106548\t0.000129994275052126\t0.00273119828593597\t0.00058912613190484\t0.000607811396123783\t0.00340042933820824\t0.000346976204371903\t0.00195128199226892\t7.7762153160057e-05\t0.00256117204355057\t0.00325984698278262\t0.0102046900755587\t0.000140087667072185\t0.00235086511836356\t0.00584909339052447\t0.00115015538599265\t2.34901599719877e-05\t0.000117985723727429\t0.000789115629113134\t0.000181967718926662\t0.00290847014470389\t0.00821917890402749\t0.000362158263161001\t0.00241898143223138\t0.00555807857223758\t0.000630844181487173\t0.000774161025493303\t0.00111955106002493\t0.00352045765949573\t0.00343045625068134\t0.00229020978321614\t0.00184040010298239\t0.0013302836829954\t0.00236059788042817\t0.000637869428128062\t0.001\t0.000227959222654252\t0.000124067870087653\t9.10602819066221e-05\t0.00105000241500555\t0.00384901465224903\t0.000398902268944109\t0.00270176127817724\t0.00899148809127805\t0.000185035423170643\t0.00236952632694819\t0.000597307015806124\t0.00062573024769022\t0.00197901178046742\t0.000110048410185592\t6.55999950800004e-05\t0.00100510903217354\t0.000356500085560021\t0.00339898154353022\t0.00519330602077322\t0.00252120239852625\t0.0001693574520687\t0.00031824731533806\t0.0019249599230016\t0.00437303972652808\t0.00149114247036663",
                 "G[T>A]G\t0.000246043655525799\t6.38018426610177e-05\t0.0116046418567427\t0.0130989127902384\t0.00410917816436713\t0.000195106908830694\t0.000290188311901242\t0.000655827163309381\t2.22018658448056e-16\t0.00134979280680415\t0.0155993697854607\t0.000787934601428081\t7.3005668890189e-05\t2.21996892043511e-16\t0.00193027891615182\t0.000539585718329903\t0.000119169481167744\t0.00251032885307975\t4.69979302111534e-05\t0.000120052671909273\t0.000187040045273693\t0.00554827836922203\t0.000695087761780802\t0.000240983473353397\t0.00183120310043699\t0.000485763578866166\t0.00213097517686043\t0.000741965172154819\t0.0375172429248482\t4.68293058498448e-18\t0.0024909166573299\t0.0381940799176128\t0.000608082151898722\t0.00156934260238386\t7.16913253496326e-05\t0.00366053637839553\t9.85825114624666e-05\t0.00576696657558125\t0.0130986914407251\t0.000827361557000409\t0.00652725155060781\t0.00111961294980325\t0.0011697110813629\t0.00108022468673484\t0.00120951498449122\t0.00667086721273765\t0.00441058660801887\t0.0014301309999996\t0.0077716895653115\t0.00450095982968368\t0.00221055987955349\t0.00102978920215032\t0.0024\t1.30976570910995e-05\t0.000143078269536567\t0.000110072868238774\t0.000199000457701053\t0.00428890204107749\t0.00128968402741328\t0.00396258320799329\t0.00264043699232192\t0.0125023934574759\t0.000435912860147431\t0.00246126509025639\t0.0203912093496494\t0.000504747954109116\t0.000360158433334665\t0.000337380974696427\t0.00352016611268052\t0.00306190073485618\t0.00681056301864281\t0.00288517001154068\t0.00913242170770127\t0.000153961047328012\t0.000119027016444554\t0.00599619976015201\t0.00443842279260158\t0.00222596439670548",
                 "G[T>A]T\t0.000258045785063643\t6.28018137791835e-05\t0.0123049219687875\t0.00551954187802412\t0.00321935612877425\t0.000248135966102626\t0.000306198701523379\t5.58852719954183e-05\t0.0203017061553853\t0.00875865554637362\t0.0280988648058618\t0.00255978753763438\t0.000426033081468774\t0.000107998488021167\t0.0128149995877592\t0.00687477669595284\t0.000164233570684958\t0.000915119880704372\t0.000357984234374317\t0.000857376165218727\t0.000840179882512845\t0.00096570034318351\t0.00480060612452928\t0.00157989165103057\t0.00346227471448742\t0.000282862330903549\t0.00430196866690134\t0.000936956019284455\t0.00932428544158894\t5.7836193976945e-18\t0.00131048225747075\t0.0201968694852298\t0.000834112688624233\t0\t0.00032396080074311\t0.00107015681007738\t0.000190966122609849\t0.0007885852041826\t0.00615938467747072\t0.00176076945625238\t0.00747685169962426\t0.00285901163967616\t0.00111972342831321\t0.00168034951269864\t0.00183926245575524\t0.00409053176912999\t0.00577076751207911\t0.0045204140699288\t0.00325070670363737\t0.00131027941708569\t0.00318080561854304\t0.000509895624365692\t0.0012\t3.27941337853485e-06\t0.000231126435405224\t0.000131086779447995\t0.000786001807804158\t0.000145962633565807\t0.000810801353668351\t0.00311202873153008\t0.00185030622568013\t0.00512098036018211\t0.00536892674080666\t0.0079340781161517\t0.00276880636757494\t0.00150924635783122\t0.000637280338983838\t0.000135756989818226\t0.0116585013731886\t0.00264030063367215\t0.00695546223184008\t0.0150028850600115\t0.003874896067816\t0.0166278451127923\t0.000727982263783311\t0.00411784983528601\t0.00519663237741131\t0.00725627587636772",
                 "G[T>C]A\t0.00105018633456134\t2.23006440649012e-16\t0.00998399359743897\t0.00289975931997644\t0.0140971805638872\t0.0025714090035635\t0.000101065584489743\t0.000265929916829719\t2.22018658448056e-16\t0.000799877218846906\t0.00259989496424344\t0.00951920990557784\t0.00220017084326597\t0.0146997942028811\t0.0127393649132335\t0.0024316491372068\t0.00081015218709836\t0.071109315320307\t0.000286987361076618\t0.0110048282583501\t0.0129027624814473\t0.00205936098028782\t0.000123015531941063\t5.9495919769822e-05\t0.000708465461808408\t0.000953535914070622\t0.00276126360945295\t0.222989532871327\t0.000721331523968415\t0.00021813651015526\t0.00370136210125326\t0.0174972879203723\t0.0835112823742488\t0.00477799849642985\t0.000490940596187861\t0.00108015827559212\t0.000934834160419941\t0.00452761847268337\t0.00913908700520818\t0.0122053337308404\t0.000108954122360835\t0.000564804746998963\t0.00233942216272581\t0.0307063869284811\t0.00169931857325213\t0.00747097122625941\t0.00601079943632503\t0.00606055514685145\t0.00582126554312908\t7.50159971613947e-18\t0.0270068401574409\t0.000653866153598359\t0.0221\t0.000416925420380802\t0.000157085932288399\t6.5943654699411e-05\t0.0013800031740073\t0.00117969799731269\t0.00241940724522492\t0.00561365954465716\t0.0114018870122992\t0.00261049975392096\t0.000729854100705561\t0.00386198506032101\t0.00217906060697234\t0.00070964563845044\t0.000761334910465779\t0\t0.0215907076911186\t0.00444830106759226\t0.00300958365796074\t0\t0.00504980851631087\t0.00328008622690673\t0.00017140656212447\t0.0292093388316264\t0.0114008381237605\t0.00414472215563504",
                 "G[T>C]C\t0.00190033717682528\t2.74007913622552e-05\t0.0056422569027611\t0.00117990206812835\t0.0079384123175365\t0.00111060855795933\t7.41481169375241e-05\t2.21941509534577e-16\t0.000423035551907782\t0.00231964393465603\t0.00261989415627609\t0.00977918832736883\t0.00628048767986832\t0.00532992538104465\t0.00606969548274699\t0.00273773723835453\t0.000861224821884536\t0.0304039829217628\t0.000181991985072977\t0.0065428706190554\t0.0116024840918441\t0.0011896308575449\t0.0149018815115596\t0.00149989713705434\t0.00226148579616808\t0.000370819522138575\t0.00226103469469698\t0.0662968880240762\t0.00124057016604831\t0.00041826174882981\t0.00365134369447957\t0.00404937234728617\t0.024803350932711\t0.00195917929979132\t0.000586928981593226\t0.00359052611979233\t0.000554901560463174\t0.00656654599680568\t0.00766923384353903\t0.00794347129695677\t0.000645728101331186\t0.0020292984715184\t0.00139965428539151\t0.0182037863875686\t0.00215913418719094\t0.00689089581645614\t0.0045806092210264\t0.00374034261538357\t0.00331071975047375\t0.0219046711711272\t0.00116029387343079\t0.000569883344879303\t0.0576\t0.00108980505567164\t0.000129070606784736\t0.000151100028218681\t0.00538001237402846\t0.00226941902872865\t0.00204949787302111\t0.00668435753267555\t0.00443073328635837\t0.00317060698081588\t2.72945437661121e-05\t0.00229117766531997\t0.000779663886898357\t0.00317841285953859\t0.000635279458798646\t0.00086594493505413\t0.0116061353715124\t0.00686040164649639\t0.00389477885135084\t0.00346220401384882\t0.00164487462799898\t9.24024290750555e-05\t0.000135451876613749\t0.0026857598925696\t0.00281095962617991\t0.00113866741112337",
                 "G[T>C]G\t0.00117020762993978\t0.000107003090356252\t0.0109043617446979\t0.00236980330632557\t0.01499700059988\t0.00245134321351384\t7.25470779753104e-05\t0.000528860624071132\t0.000232019498918689\t0.00256960556554569\t0.00280988648058618\t0.00630947631346598\t2.23017317294687e-16\t0.00137998068027047\t0.0127299289447\t0.00353347592400027\t0.000561798982647936\t0.0415054372122748\t0.00019499141257819\t0.0110048282583501\t0.00595127416779932\t0.00427867232797663\t0.00357045080511865\t0.00100993073894992\t0.00398261657909246\t0.00095253640053386\t0.00406185878781848\t0.116994508277781\t0.000581267150382315\t4.68293058498448e-18\t0.00267098292171519\t0.0133979233218851\t0.0712096204197187\t0.00122948496874659\t0.000730911559701275\t0.00299043818890782\t0.000500911138364054\t0.0051273030386017\t0.0096190390579981\t0.207090498547865\t0.000439814805860251\t0.000190933994118233\t0.00183954563222884\t0.00983204506537361\t0.00186925043057734\t0.00574074629701861\t0.00565075154995614\t0.00351032154545356\t0.00321069800574645\t0.00120025595458231\t0.0173043827675454\t0.000906814375097418\t0.0353\t0.00309944557117623\t4.67255607507531e-05\t6.53432572362903e-05\t0.00371000853301963\t0.00455883293876768\t0.000913776124849411\t0.00256166995264213\t0.121020028814754\t0.00601115077435439\t0.00164967022762216\t0.00472242732764641\t0.0110952168520151\t0.000985507886636808\t0.000500220046298146\t0.000197906985156976\t0.00646268820687065\t0.00268070064336815\t0.00396177848754281\t0.000577034002308136\t0.0072597247184984\t7.70020242292129e-05\t0.000144828657547873\t0.00925980962960762\t0.000207274535150315\t0.000947576613718293",
                 "G[T>C]T\t7.13126530040222e-05\t1.25003610229267e-05\t0.0102040816326531\t0.000754937340200763\t0.0136972605478904\t0.000906496714874915\t0.000811526623972093\t0.00684819522662997\t0.00707059421273763\t0.317951194491645\t0.00628974589426587\t0.0098591816879199\t0.00529041080039863\t0.0114998390022539\t0.0105699462609989\t0.00216680622420104\t0.000237337538123994\t0.0463060660946584\t2.21990223550554e-16\t0.0112049160448655\t0.00586125489467294\t0.00239925551101493\t0.00524066168594446\t0.00146989919431325\t0.00315207091058826\t0.00235885194675751\t0.00092842486578708\t0.132993757273033\t0.000834383482648624\t0.000799500328504829\t0.00399146886054068\t0.0141977993411021\t0.0552074585276471\t0.003748429782764\t0.000850897041457982\t0.00155022715478499\t0.000444921071002004\t0.00468753435692826\t0.0128987114187293\t0.00971424512511968\t2.21906561138581e-16\t0.000420854510595687\t0.00264934561163393\t0.04000832173092\t0.00304877744024646\t0.0075109764269355\t0.00731097235932379\t0.00389035635664227\t0.00239051969898256\t0.000518110487061366\t0.0111028120647257\t0.00138971552523199\t0.0607\t0.00318942947485554\t0.000110060207335821\t0.000175115926743504\t0.00117000269100619\t0.0040089737027321\t0.00310923823663202\t0.00699455975350331\t0.00541089550320514\t0.0126024126051357\t0.00257948435591828\t0.00536275645681881\t0.00201912955324959\t0.00118940607007891\t0.00069630630444702\t0.00366247272531455\t0.054089002731389\t0.00854990205197649\t0.00654096448256286\t0.0132717830530871\t0.00405927007653797\t0.0013240688071208\t0.000152534376067502\t0.012672949493082\t0.00587890627612481\t0.0107456711411948",
                 "G[T>G]A\t2.23039573911599e-16\t0.000209006036303334\t0.00442176870748299\t0.00059795037011928\t0.00251949610077984\t4.35238488930009e-05\t9.06588312353534e-05\t0.000322914899007516\t0.00248020843671702\t7.47885199621857e-05\t0.00130994707813804\t0.00829931115717396\t0.000204015841830117\t0.000480993266094273\t0.000355060718443313\t0.000393074534536407\t9.5836297039942e-05\t0.000412053979071258\t8.76961378620884e-05\t7.793419284777e-05\t0.000203043471607271\t0.000901720196223112\t0.000495062506592082\t0.000280980730341513\t0.000870571965781519\t0.000635690609380415\t2.22101638151651e-16\t0.000285986575790132\t0.00159073110001356\t6.26391997051342e-18\t0.00107039390495702\t0.00554913988331809\t0.00137018511200863\t0\t0.0011598596569815\t4.4906580161191e-05\t0.000187966654715453\t0.000303840180065286\t0.00114988512647587\t0.000581254008001497\t0.000364846373043163\t0.00175939177826226\t0.000184954316283878\t0.0118024549106214\t0.000503797977011218\t0.00333043295628432\t0.00249033121405147\t0.00180016489510439\t0.00114024788989126\t0.0611130323541495\t7.68194564478318e-18\t0.000439909950433146\t0\t0.0219960653438313\t7.64418167314248e-06\t6.61437871871179e-05\t0.00867001994104587\t0.00218943950348711\t0.00112972321781164\t0.00438285718459864\t0.000170028139657093\t0.00567108567231105\t6.71865692704297e-18\t0.00140071997006461\t0.0080665225221407\t0.00122938610604795\t0.0282124106112155\t1.88999985825001e-05\t0.00769962424646497\t0.00139540033489608\t0.00352798084306402\t0\t0.012761223107181\t0.00635861115517005\t0.000100750411769035\t0.000550489977980401\t0.00373978845388092\t0.00127893452322581",
                 "G[T>G]C\t2.23039573911599e-16\t0.00013300384128394\t0.00234093637454982\t0.000252979002742772\t0.00170965806838632\t0.000213116777338142\t0.000121078571517415\t0.000320915425948645\t0.0016601395181251\t2.22965774753575e-05\t0.000655973498670653\t0.00584951449029731\t0.000833064687472979\t0.000941986812184627\t0.000620308334384202\t0.00157296029073594\t2.22316174951589e-16\t0.000213027906655772\t2.21990223550554e-16\t0.00210092175841228\t0.000580124204592202\t0.000900720506426856\t0.000204025760292494\t0.00624957140439306\t3.70243249815129e-05\t0.000702658016343446\t6.6530445662544e-05\t0.000532974982154337\t0.000236108515473711\t8.69544162040922e-05\t0.001530563247275\t0\t0.00121016349308792\t5.07787287905097e-05\t0.00192976649825371\t9.1713438770183e-05\t0.000352937388907208\t0.000871541569134636\t0.000739926081384469\t0.000637278490700436\t2.21906561138581e-16\t0.00506824790669866\t0.000312922708091102\t0.00686142717685278\t0.000938623611931615\t0.00580075409803274\t0.00224029795962863\t0.000950087027971762\t7.05153300327491e-05\t0.0756161251386858\t1.83046361067099e-05\t0.000519893577784628\t1e-04\t0.0103981399807203\t0.000135073890821235\t4.37289485639493e-05\t0.00959002205705073\t0.00529864354725191\t0.000560862588665777\t0.00174113504593645\t0.00153025325691384\t0.00320061272511382\t9.18816326778644e-07\t0.000171087939200749\t0.00268884084071357\t0.00164917648372285\t0.0288126746667732\t0.00122943590779231\t0.000800068025610178\t0.00068724016493764\t0.00123869327389552\t0.00173110200692441\t0.00127016383665748\t4.62012145375278e-05\t0.00139143019984904\t0.0002942299882308\t0.00270422389368792\t0.000632934787189933",
                 "G[T>G]G\t0.000348061756597472\t3.71010715160463e-05\t0.0109043617446979\t0.00242979832673888\t0.00529894021195761\t0.000274150220613385\t0.000325211039199667\t0.00240936503593843\t0.00603050680379178\t0.00379941678952281\t0.00396983961847941\t0.00396967051734706\t2.23017317294687e-16\t0.00172997578033907\t0.00061424515440568\t0.000200060798324658\t0.000130185147494174\t0.000496064984512971\t0.000425981239786199\t0.000781342806342854\t0.000347074308609473\t0.00067479061247295\t0.000835105440412905\t0.00147989850856028\t0.00583383282816811\t0.000509751903748446\t8.70398311675388e-05\t0.000212990002249294\t0.000450206915098179\t0.000157098312359522\t0.00313115226403316\t0.0086886532587449\t0.00235031752789802\t0.0047380152454137\t0.00312962131582078\t0.000770112844635124\t0.00137975523142195\t0.00254865940515289\t0.00300969933103683\t0.00542236957550449\t5.64762193888731e-05\t0.00403860385464745\t0.00152962218332072\t0.00983204506537361\t0.00215913418719094\t0.00772100373048496\t0.00445059192872652\t0.00120010993006959\t0.000410089153381945\t0.241051404211948\t0.000103026093933941\t0.000178963366198939\t0.0012\t0.0271951353341914\t6.63362886033176e-05\t0.000147097378464544\t0.0351000807301857\t0.0761804977925651\t0.00136966443221411\t0.00473308549843643\t7.45123317909026e-18\t0.299057251502823\t6.71865692704297e-18\t0.000861442781589737\t0.0233899166069508\t0.00228885705922748\t0.662291341298746\t0.000212750984043676\t0.00194651406230792\t0.000454480109075226\t0.00323648242590043\t0.00288517001154068\t0.0484828785777272\t0.000230942071005163\t7.33346651268891e-05\t0.001544349938226\t0.000865989082632163\t0.00197441641414727",
                 "G[T>G]T\t1.460259093771e-05\t2.23006440649012e-16\t0.00583233293317327\t0.000251979085735884\t0.0023495300939812\t0.000787431473075672\t0.000835542208405299\t0.00182951784886611\t0.00675056731767737\t0.00683895022114105\t0.0012899478861054\t0.0193983899336355\t0.00221017161982627\t0.00515992776101133\t0.00747367436050711\t0.00094783508936354\t0.000154219328570021\t0.000165021617831936\t2.88987273000496e-06\t0.00225098759829888\t0.00139029766272959\t0.00238925861305237\t0.00724091423783165\t0.11399218241613\t0.00214140690433615\t0.000340834116035725\t0.00356162987306251\t0.00258987843110644\t1.9609012302054e-05\t5.7836193976945e-18\t0.00268098660306993\t0.00276957071653894\t0.00605081746543958\t0.00229903693342859\t0.0205975077015681\t0.00115016853419532\t0.000159971621034428\t0.00280852271705083\t0.000784921586333525\t0.000451197173164673\t0.00041682448646301\t0.00334884230521509\t0.000488879246826034\t0.0171035575399683\t0.00333866119686006\t0.00594077230039905\t0.00548072893694861\t0.00347031788111791\t0.000920200051491194\t0.0746159118432006\t0.00280070934966054\t0.000495898489579183\t6e-04\t0.0883841898361221\t1.15062944032904e-05\t0.0001020675687305\t0.0365000839501931\t0.00208946509693518\t0.000863788371848896\t0.00203132422025919\t1.21020028814754e-05\t0.00144027572630122\t0.000113977215726622\t0.00283145536805918\t0.0168927175494644\t0.00744628169923349\t0.0474208603890643\t0.00134007889949408\t0.00292790609372227\t0.00220790052989613\t0.00246058663901455\t0.0132717830530871\t0.00376985705079049\t0.0028800757114303\t5.39387770965705e-05\t0.000199599992016\t0.00614792975056932\t0.00212149680761824",
                 "T[C>A]A\t6.58116769658438e-05\t0.000639018455492011\t0.00841336534613846\t0.0397966968741594\t0.00785842831433714\t0.000356195177147318\t0.000390253247039601\t0.00289923593536159\t0.00252021179859955\t0.0113982503685684\t0.0297987961286364\t0.0169985891171033\t0.0940072996668187\t0.000445993756087414\t0.06018580305725\t0.258971764875439\t0.000180256358068856\t0.0034804559397281\t0.0789965209932153\t0.000347152309604316\t0.0172036833085964\t0.0107966497995672\t0.000190023992429284\t1.23991496663159e-05\t0.0738485184766392\t0.00137932868073109\t0.00171078288846542\t0.00145993147081676\t0.00078436049208216\t0.00213133379188396\t0.0298109704371209\t0.00693892446670766\t0.000938126740922699\t0.00707703542985843\t0.0142982699093409\t0.0452066241266333\t0.000695876551499764\t0.00185902215434681\t0.00972902807009579\t0.0119052025735246\t0.00426820277505289\t0.00526817879059209\t0.140965181600145\t0.002000416086546\t0.0176929051450368\t0.00598077750107514\t0.0260034584599752\t0.0033103032237753\t0.00510110898109249\t0.00630134376155715\t0.000853216098307299\t0.0639869018811849\t0.0123\t0.0168969774686704\t0.124067870087653\t0.0329217942277788\t0.0190000437001005\t0.0356908631390364\t0.463886347844778\t0.00267174170841972\t0.000669110737827032\t0.00174033316928064\t0.218956230211668\t0.0143073539799457\t0.00736682416210371\t2.36881713116555e-05\t0.00202088898704451\t0.00208356384373271\t0.00194438706223983\t0.00856170205480849\t0.0040054782502531\t0\t0.0157400161543274\t0.00201689602016258\t0.333883761308447\t0.0148485794060568\t0.0098906568275025\t0.0252394210311844",
                 "T[C>A]C\t0.00253044897756208\t0.000170004909911803\t0.0168067226890756\t0.0468961076230673\t0.00916816636672666\t0.000498273028706079\t0.000636412987479964\t0.00351907258361132\t0.0142011934682991\t0.00775880902281499\t0.0250989860009656\t0.00245979583694553\t0.00997077423061897\t0.00379994680074478\t0.00854315600707057\t0.0144067591957389\t0.000939335910381041\t0.00288037732943016\t0.0293987052810194\t0.0243106660616279\t0.00581124418738051\t0.00586817910402402\t2.59032705469392e-05\t2.22984707708745e-16\t0.0436286640322692\t0.00112945029654067\t0.00337154288545524\t0.000235988922679969\t0.00174080007171296\t0.000664415792399507\t0.0519191062310931\t0.010198419245017\t0.00063008512450032\t0.00626737459678141\t0.000451945314616931\t0.0411060232655891\t0.000242956899446038\t0.00937506871385652\t0.00928907202170503\t0.00511223404627822\t0.00121948650715797\t0.0230920170896921\t0.0771809363087318\t0.00159033078880407\t0.0114953903484703\t0.00786102193285127\t0.0210027933715184\t0.00172015756643308\t0.054011742152744\t5.37114539675586e-06\t0.00222056241294514\t0.0641868609495636\t0.00710000000000001\t8.1085495426578e-05\t0.00379207441638874\t0.00511338506091032\t0.000886002037804687\t0.00285926802738499\t0.0467885368084819\t0.00364237446391303\t4.05067038594839e-05\t8.281585426232e-18\t0.0131973618209773\t0.000709364613411293\t0.000428815137794097\t0.0201899181643646\t0.000163071735093196\t9.05999932050005e-06\t0.00016942800542339\t0.00394860094766423\t0.00417897730815322\t0.00230813600923254\t0.0167847745381343\t0.000785203641433326\t0.0452385733549299\t0.0128317794867288\t0.0040374220561095\t0.02030369551327",
                 "T[C>A]G\t2.23039573911599e-16\t8.53024636204515e-05\t0.00141056422569028\t0.0108990953750839\t0.0022995400919816\t0.000200109650082763\t3.44223376875956e-05\t0.00029992095883051\t2.22018658448056e-16\t2.21965928230017e-16\t0.00189992324310098\t0.000444963068065351\t0.00116009008099478\t0.000572991978112305\t0.00265737281610898\t0.00769778265916047\t2.22316174951589e-16\t0.000505066163667441\t0.0120994671394672\t7.08310764264713e-06\t7.69164678157593e-05\t0.00148953779642177\t2.22028033259479e-16\t9.50934784892449e-05\t0.0127083493855463\t0.000284861357977073\t2.22101638151651e-16\t6.82967941484826e-05\t2.22102078115101e-16\t9.1257108835595e-05\t0.00979360404628904\t0\t0.000123016619545301\t0.00144939284933541\t0.00012898439288846\t0.0174025499956509\t8.25853493590237e-05\t0.000954497934086671\t0.00237976226175005\t0.00137059895174191\t2.21906561138581e-16\t9.46672735235429e-05\t0.0146963699966108\t0.00112023300846576\t0.00237904600255298\t0.00301039135087561\t0.00317042166608159\t0.000207018962937005\t0.000904196572334826\t3.07065481713976e-18\t3.1507980183681e-07\t0.0182962547566513\t0.0025\t2.38957255326167e-05\t0.709388063646337\t0.0405268287606396\t4.19000963702217e-05\t0.00453883805745729\t0.13396717804138\t0.000447291589387122\t0.000828137056682784\t9.91189753308685e-05\t0.00545890875322241\t0.000223114680946006\t0.00272882360414425\t0.000633683570109266\t0.00020509021898224\t0\t3.9900001277199e-09\t0.00107750025860006\t0.00317228277450453\t0.000577034002308136\t0.0019499260607537\t7.70020242292129e-05\t0.318417550218876\t0.0023931699042732\t0.000174699396492363\t0.00260766706038171",
                 "T[C>A]T\t5.88104347354349e-06\t0.000440012708007018\t0.00997398959583833\t0.0412965723844921\t0.0126974605078984\t0.000747409543059119\t0.000613398052398142\t0.0014096285065034\t0.00373031349554616\t0.000190970685999699\t0.0318987112920638\t0.0272977342880541\t0.670052029540091\t0.0815988576159932\t0.188803966741945\t0.341876445926685\t4.73673652036494e-05\t0.00235030789033363\t0.0354984366488499\t0.151066278819169\t0.00805172387408143\t0.00924713061537005\t0.000288036367471757\t0.00256982376148643\t0.122080206695799\t0.00137932868073109\t0.000904413877878794\t2.21989579809124e-16\t0.000396182085286397\t0.00024015028640946\t0.0228083934888039\t0.0245961875909234\t0.00140018916555627\t0.00980589231171061\t0.00055693261115405\t0.0411060232655891\t0.000664882049924343\t0.00448763950162215\t0.00963903706019768\t0.0128055960454719\t3.1986531335291e-05\t0.015294712617848\t0.207948636686738\t0.00244050762558612\t0.0277888566684759\t0.00655085161070939\t0.0241032057263616\t0.00590054048950884\t0.0273059363105539\t6.52139068656391e-18\t0.0166042054301303\t0.0619873111973979\t0.003\t0.000191965661182528\t0.000187102352470896\t0.0087557963371752\t0.00185000425500979\t0.000369905304242114\t0.0609850586606281\t0.000871568175293474\t0.0072011917972416\t0.00406077739498816\t0.401919655457035\t0.00619318329621425\t0.0146936655607768\t0.00382808844403547\t0.000406178677594095\t0.00120377890971658\t0.00205749206586032\t0.00908270217984852\t0.00413377755358788\t0.0103866130415465\t0.0187139283865495\t0.00144723804499373\t0.0764892532904738\t0.0131862594725496\t0.00874880731385209\t0.00815282519395623",
                 "T[C>G]A\t2.23039573911599e-16\t2.23006440649012e-16\t0.0133053221288515\t0.00357970288466057\t0.00780843831233754\t4.97272480455666e-05\t2.23144805358541e-16\t0.000253933078476498\t0.00194016305130283\t0.00373942599810929\t0.00146994061439918\t0.00268977674852987\t5.55043099096643e-05\t2.21996892043511e-16\t0.000258750324351168\t0.000120905995970519\t1.56222176993009e-05\t0.000969126955631188\t0.316986039934801\t1.97086469717724e-05\t0.000214045827211606\t0.00250922138860312\t0.0037504735347885\t0.000159989027952463\t0.00214140690433615\t0.0017491486893329\t0.00012905906000704\t0.000653969302680932\t0.00356163692833226\t0.000211132126801651\t0.00375138050802695\t0.00979848123540851\t0.00131017700491336\t0.00286879826040872\t0.000114986086683511\t0.000107015681007738\t0.000126977474196078\t0.000395791813506096\t4.70952951800115e-18\t0.0130056834836824\t6.17739886412807e-07\t0.00662770880106747\t0.00397901718275586\t0.00179037239745867\t0.00265893376755921\t0.0347045115865062\t0.0125016627211419\t0.0199018230069874\t0.000559121553025628\t4.42094276604486e-06\t0.000673170497257693\t0.0032993246282486\t0.0028\t0.000137975318974942\t0.000137074985500068\t4.80317970496469e-05\t0.00093200214360493\t0.00204947533431442\t0.00197951501882039\t0.00374243969643811\t0.00469077632348098\t0.00706135182478237\t0.00112977415588669\t0.00228117252267666\t0.00212908215268398\t2.34882711318103e-06\t0.000529232808983439\t0\t0.000947995030345321\t0.0422250101340024\t0.000639536527316657\t0\t0.0224716528973938\t0.000646637998819713\t4.87840117744935e-05\t0.0194511592219536\t0.0128107001126837\t0.00609267381682005",
                 "T[C>G]C\t0.000203036024681859\t9.87028506370289e-05\t0.0203081232492997\t0.00594950619098615\t0.00759848030393922\t0.000113061952296761\t5.09330519854248e-05\t0.00161957317768475\t0.00461038745696188\t0.000587909755852476\t0.00133994586618701\t0.00199983401377686\t2.23017317294687e-16\t2.21996892043511e-16\t0.000395676572849671\t6.54299669336123e-05\t0.000224319023374577\t0.00155020307660304\t0.0634972035831541\t0.000150065839886592\t6.12131057259358e-05\t0.00522837763442004\t0.000723091297507223\t6.17957620466386e-05\t0.0046430504841681\t0.00311848223469637\t2.82129107922367e-05\t0.000630970382250258\t0.00152069891322051\t0.000266166567437152\t0.00558205419594411\t0.00326949322854957\t0.000745100663099585\t0.00310869776650561\t0.000135983545990935\t0.0068310009465687\t0.000985825114624665\t0.00291846488746919\t0.00109989012097692\t0.00251109734954175\t0.000139941074591898\t0.00562805438160028\t0.00240940487699538\t0.00111023092803303\t0.00217912617040567\t0.0367047716203106\t0.0141018755494481\t0.0142013008391569\t0.00187040662640058\t0.00103021969434982\t9.40238138814608e-06\t0.00200958863720596\t0.0024\t0.00118978717087087\t0.000350191568795794\t0.000153101353095749\t0.000493001133902608\t0.00360907607652441\t0.00143964728641483\t0.00499325510300165\t0.000517085577663042\t0.00328062804324167\t0.00029394124055813\t0.00280143994012923\t0.00179922435438083\t0.00208895687938228\t0.00034815315222351\t0.000113942991454276\t0.00186940505983966\t0.0473870113728827\t0.00284118457236777\t0\t0.0165910528838657\t0.000323319499423001\t0.000153375021203397\t0.00408749983650001\t0.00706230590923424\t0.00534882802928319",
                 "T[C>G]G\t6.9412315827197e-05\t4.12011899315663e-05\t0.000520208083233293\t0.00182984812260582\t0.00170965806838632\t1.95106908830694e-06\t6.35412338128581e-06\t0.000363904096714352\t8.75073541180399e-06\t0.000872866015066687\t7.81968408476297e-05\t0.000403966470782925\t2.23017317294687e-16\t2.21996892043511e-16\t1.03871028532391e-05\t1.9573417104251e-05\t2.22316174951589e-16\t0.000214028037672935\t0.0137993922747642\t1.83080324661642e-05\t5.84125061175597e-05\t0.000438863820556481\t0.000380047984858568\t1.54989370828948e-05\t0.000899591031307569\t0.000440785469711891\t0.000124056770859481\t6.26970570001445e-05\t0.000381175188116458\t2.56160305503424e-18\t0.0013504969828897\t0.00205968074948383\t9.41127146277462e-05\t1.27946403251678e-05\t2.4597023760125e-05\t0.000897131456672346\t0.000105981198935309\t0.000415781299036707\t0.000996900409649076\t8.04351501606202e-05\t2.21906561138581e-16\t0.000154946435017415\t0.000301925424420168\t0.000860178917214781\t0.0010995590768102\t0.00962125076259914\t0.00174023145078295\t0.00193017680419526\t0.000207045011585519\t0.00147031354436334\t7.68194564478318e-18\t0.000177963570857046\t1e-04\t0\t2.82154349715468e-05\t6.33419323592217e-05\t0.000103000236900545\t0.00215944718152153\t0.000580857689865983\t0.00187121984821905\t0.00123020359869544\t0.000229043848140958\t8.38832315742419e-05\t0.00133068397156138\t0.000629728524033288\t0.000404797864186518\t6.59290021020957e-05\t7.20999945925004e-07\t0.000169997005441604\t0.00463590111261627\t0.0261168581854601\t0.000577034002308136\t0.00363986198007358\t6.16016193833704e-05\t2.57875467348049e-05\t0.000298399988064\t0.000700167290411347\t0.00104268803902658",
                 "T[C>G]T\t0.0015602768399197\t0.00118003408056428\t0.0235094037615046\t0.00337971948328289\t0.0124975004999\t0.000128070176052968\t0.000281182467738789\t0.000713811882016613\t0.0123010337788788\t8.52869084595514e-05\t0.000248989940806391\t0.00562953274878185\t5.45042322536343e-05\t4.33993924085062e-05\t0.000347148441492567\t0.000168726001410671\t0.000185263479126324\t0.00305039960234791\t0.368983749955651\t0.000321140897357306\t0.000174037261377661\t0.0284911591933023\t0.000283035736092039\t0.00149989713705434\t0.00318209063354624\t0.00160921679418627\t0.00112051276905337\t0.00068696775373364\t0.000388178406795763\t0.000466291806111702\t0.00407149831137859\t0.0163974583939489\t0.00221029861134239\t0.00212910811660995\t0.000142982699093409\t0.000336049241295327\t0.000108980666829704\t0.00423777093248951\t0.000691930876105477\t0.020108787540155\t0.00214909507408986\t0.0211926736927044\t0.00758812573294396\t0.00255053051034615\t0.00394841668490935\t0.0507065918569414\t0.0237031525192851\t0.0446040857342533\t0.000381082847411027\t0.000251053537166801\t0.00259065614843599\t0.00328932667482966\t0.0049\t0.000923834744440914\t0.00014808100623365\t0.000544360366562665\t0.000421000968302227\t0.00332914773817903\t0.00241940724522492\t0.00374243969643811\t0.000311051479019741\t0.00211040401562193\t0.00041091785669861\t0.0129066340098811\t0.018092200452385\t0.00151924136682348\t0.000761334910465779\t0.0049533826284963\t0.00624414319987502\t0.110840026601606\t0.0126309314140424\t0.00403923801615695\t0.0260800780695197\t0.00143184264027933\t0.000267230512554382\t0.0229892790804288\t0.0257271170250575\t0.0126107089624258",
                 "T[C>T]A\t0.00111019698225056\t0.536015480663095\t0.00576230492196879\t0.00373968960576272\t0.0202959408118376\t0.0120065790049658\t0.238154545629295\t0.00660825845956557\t0.0173014540141953\t0.000206968230376637\t0.00503979639222575\t0.00426964561941359\t0.000366028422106975\t0.00272996178053506\t0.00240343433458873\t0.00133201814186702\t0.00979392878840786\t0.00552072321474113\t0.00692969481624028\t0.00354155382132357\t0.0265056748649886\t0.00635802710418957\t0.000532067178801995\t0.000107992593867912\t0.0114074947240337\t0.0438786442638368\t0.00807369468416136\t0.00359983102393173\t0.00159073110001356\t0.0153095807586031\t0.00240088352513725\t0.0110982797666362\t0.00329044453905723\t0.0187921279775902\t0.00483941443085386\t0.00639093646392006\t0.0979826178835874\t0.00747606758844848\t0.0247975227274795\t0.0147064267084716\t0.00185921713386379\t0.00295897708162288\t0.0106973577526351\t0.00402083633395746\t0.00644741458675071\t0.00388050446558053\t0.0300039905307406\t0.0117010718181785\t0.00482104809585604\t3.72079345920518e-05\t0.00140035467483027\t0.00239950882054443\t0.0185\t0.00723870514042447\t0.000214117130635143\t0.000187123876005916\t0.0194000446201026\t0.00169956491138269\t0.00146963993821514\t0.00573373782368726\t7.45123317909026e-18\t5.82111439380076e-11\t0.000199960027590565\t0.00814418611166139\t0.00428815137794097\t0.000160919644775381\t4.07179117686691e-05\t7.07999946900004e-05\t0.00382361112239379\t0.0085897020615285\t8.71995265065711e-06\t0\t0.00946964092068593\t0.00211005546913817\t0.0364543980712274\t0.0171045893158164\t0.00856419140410797\t0.0094820374196133",
                 "T[C>T]C\t3.73066193134647e-05\t0.0973028102024611\t0.0133053221288515\t0.00698941987815011\t0.022995400919816\t0.010505756629345\t0.331214935307969\t0.210944407710792\t0.0137011514447674\t0.00564913285810628\t0.000251989819611288\t0.00732939166049218\t0.00276021433064276\t0.0484993210095057\t0.00901811024212494\t0.00358401474573189\t0.158225025415996\t0.00264034588531098\t0.00447980270948867\t0.000137060133763087\t0.00956204723431286\t0.00367885845022289\t9.061144060049e-05\t0.000603958580520546\t0.0114074947240337\t0.0249878384190415\t0.00919420745321473\t8.06962121198031e-05\t0.00132060695095466\t0.043627302031052\t0.012204491252781\t0.00644900040493723\t0.00253034184918382\t0.00798665439047582\t0.00412950033046001\t3.61052905082182e-18\t0.133976232616334\t0.0245870672026514\t0.0386961342561878\t0.00317138589563639\t0.000288878361121846\t0.0125956456852866\t0.00685830599841839\t0.0025005201081825\t0.00722710193212522\t0.0052806864892436\t0.0231030727086703\t0.0064705927062919\t0.0327071105258283\t5.06107927515543e-18\t0.00225057001312007\t0.0020195865906249\t0.0126\t0.0370933647389155\t0.000317173506595048\t0.000582385539226968\t0.10400023920055\t0.00176954699596903\t0.00152962524181575\t0.0057737639166973\t0.000661109413607875\t0.00411078696881806\t0.000183963225383319\t0.0173088967729413\t0.015293407012237\t0.000279860251783272\t0.000958421608707248\t0.0269093319818001\t0.00299812109596986\t0.00812190194925647\t0.00617526646830308\t0\t0.0105440891786824\t0.000985353902983402\t0.000133887309625883\t0.017369399305224\t0.00782670065392005\t0.0197463573319126",
                 "T[C>T]G\t0.110019520763569\t0.0442012765770687\t0.000989395758303321\t0.000280976678935648\t0.0158968206358728\t0.0520285090215183\t0.0741481169375241\t0.0573848767895709\t4.64038997837378e-05\t0.00174973141622761\t4.16983153880583e-05\t0.00325972944245628\t0.00279021666032366\t0.43699388208565\t0.00859141509881589\t0.00531416419424056\t2.94418718179132e-07\t8.85115950189475e-05\t0.00118994759470793\t0.00532233512131112\t0.0212045398919909\t0.00301906318469379\t0.000177022350842017\t6.23957209014603e-05\t0.0118077576968068\t0.00431789847881036\t0.00264120866991152\t0.000316985120718433\t3.77173348871141e-05\t0.0028717971749798\t0.00269099028442467\t0.0140978148387\t0.00263035536100927\t0.00460806967961121\t0.00173978948547226\t0.0144021103412283\t0.0146973926825381\t0.00840557866562189\t0.00984901608329327\t0.00385168318555209\t0.000299873731268353\t0.000764735630892401\t0.00388903940726641\t0.00343071358842639\t0.00346860908775581\t0.00486063188214468\t0.00212028199750567\t0.00492045071328534\t0.000364079150807386\t3.07065481713976e-18\t7.68194564478318e-18\t0.00174964184831365\t0.0037\t0.00193965303486512\t0.000402220030445455\t0.000934618717591045\t0.003970009131021\t0.00648833898521979\t0.00218946358142255\t0.0102066537175585\t1.19019697759965e-13\t0.00481092100243671\t0.000153969221244735\t0.00460236561592659\t0.0033385607464622\t4.53773408248591e-18\t0.00173076136019159\t6.91999948100004e-06\t0.000157959005056268\t0.0136910032858408\t0.059555676612676\t0.00923254503693018\t0.00244070345096584\t0.000292526689941625\t0.00204749866320557\t0.00359964985601401\t0.00968007143676171\t0.0050855383925427",
                 "T[C>T]T\t2.23039573911599e-16\t0.30000866455024\t0.00817326930772309\t0.00721940078973445\t0.020995800839832\t0.00699383227039256\t0.107069480598044\t0.101973126002373\t0.00232019498918689\t0.0499923261779316\t0.00526978710060113\t0.00545954685761082\t0.0228017705574837\t0.127998208025087\t0.0128012759670545\t0.00811321959520525\t0.119169481167744\t0.00612080182503908\t0.001419937465954\t0.000359157576795243\t0.00174037261377661\t0.0220931444972625\t0.000348043944028373\t0.000894938625109087\t0.00375246536974793\t0.0628694014623083\t5.0022891475597e-05\t7.30965688470582e-05\t2.22102078115101e-16\t0.0263164688857034\t0.00471173391808185\t0.0123980782978638\t0.00213028780188203\t0.0118950171773044\t0.00198975923913206\t0.00428062724030952\t0.0524906881519218\t0.0194897483923456\t0.0669933073685939\t0.00904395220711451\t0.000807659916216097\t0.00803722153251621\t0.00832794299807948\t0.00584121497271432\t0.00657736247764646\t0.00423054997149629\t0.0224029795962863\t0.0115010534965003\t0.0151032834538229\t0.000836178315025679\t0.00302076508427672\t0.00294939625858587\t0.0044\t0.0808855311961796\t5.85320193558685e-06\t0.000312206680822705\t0.0818001881404327\t0.0423891483780152\t0.00164959584901699\t0.0077150294276839\t0.000236039064465141\t0.0314060123651794\t8.19836113121315e-05\t0.0372191306331454\t0.178922866352316\t0.00208895687938228\t0.0039117207620515\t0.0168396677370249\t0.038650891237215\t0.0129850031164007\t0.0109479405526828\t0\t0.00948223644307632\t0.00277130785214082\t0.000139780296819048\t0.02458274901669\t0.0104521242705514\t0.0129271032819206",
                 "T[T>A]A\t0.00672119254119256\t0.000422012188134004\t0.00651260504201681\t0.00887926302116924\t0.00630873825234953\t0.000454248905687871\t0.000356231169092559\t0.00323914635536951\t0.015401294324775\t0.0166974369434292\t0.0114995354187691\t0.0259978421790991\t0.00204015841830117\t0.00305995716059974\t0.00996265158325054\t0.00892578842626951\t0.000186264903337818\t0.00375049131436218\t0.000295986964734073\t0.000200087786515456\t0.001090233418975\t0.0122961844939515\t0.000119015026837288\t0.00118991839539644\t0.010807100264874\t0.000493759687160259\t0.0120054939541433\t0.000256987936986238\t0.0718330144534427\t7.77486552250628e-18\t0.00412151671815228\t0.0217966215236638\t0.00113015268362756\t0.0527778913413171\t0.000187977254752175\t0.000764111965326279\t0.00028194998207318\t0.00669647765275465\t0.000511948856309255\t0.00410179248331521\t0.271885516349973\t0.00429851400370892\t0.00245939253004508\t0.00245050970601885\t0.00243902195219717\t0.0066908698130757\t0.00713094841613935\t0.0806073836363411\t0.00850184830182082\t7.63162744455255e-05\t0.00696176324058476\t0.00040691670415066\t0.0045\t0.155972099710804\t2.3512862476289e-05\t0.00042027822418441\t0.00169000388700894\t0.0200948557169365\t0.00104974281301081\t0.00310202220827758\t0.00155025656746173\t0.00361069123051903\t0.00941811729951559\t0.00302155307828224\t0.0118948721206288\t0.00208895687938228\t0.00113049730463381\t0.00656990650725701\t0.0601799039263587\t0.00392230094135223\t0.00827965504147312\t0.021350260085401\t0.00708973116448397\t0.357009385062715\t0.00229357875931564\t0.00817814967287402\t0.0206638037112401\t0.0103104675289367",
                 "T[T>A]C\t2.23039573911599e-16\t0.000192005545312153\t0.0105042016806723\t0.00430964229968912\t0.00413917216556689\t0.000114062500547175\t0.000389252597688217\t0.00223940982593447\t0.0342028744095653\t0.00122981122397712\t0.0194992122318258\t0.00184984646274359\t0.000360027956170795\t0.000664990690130336\t0.00247466418137113\t0.000761366633447982\t0.000202287690721716\t0.00645084506070295\t0.000231989783169949\t0.00173075935335869\t0.000157033620898234\t0.00466855134851655\t0.00570071977287852\t0.0002419834047781\t0.00273179478917649\t0.000474768929961788\t0.000318145589784797\t0.000794962684451592\t0.0236108515473711\t5.2733000390744e-18\t0.0032712038029995\t0.00340947153191255\t0.00126017024900064\t4.70802780715158e-05\t0.00124984876829903\t0.00193028284434518\t0.000240957254183108\t0.00374802853698955\t0.00244975526944858\t0.00021709487045839\t2.21906561138581e-16\t0.00529816842317611\t0.00101974812221381\t0.00113023508889849\t0.000927628021163513\t0.00369047976236911\t0.0103013700822209\t0.014301309999996\t0.00226049143083706\t0.000938200071165176\t2.07052441207047e-05\t0.000261946379576101\t0.0043\t1.22978001695057e-05\t7.64418167314248e-06\t5.70377589964557e-05\t0.000664001527203513\t0.00257933968903961\t0.000468885123144829\t0.00356232227789296\t0.000179029629403645\t6.95133076235658e-05\t0.000804839111052023\t0.00923474665978313\t0.000897613039018878\t0.00185907167256031\t2.23098140648973e-16\t0.00086450693516198\t0.0077948952495146\t0.0017123004109521\t0.00372257978639176\t0\t0.00316470999736161\t0.000877581069851164\t0.000224179383212792\t0.0028471198861152\t0.0103708121643981\t0.00418389162990814",
                 "T[T>A]G\t2.80049689216357e-05\t5.010144697989e-05\t0.00671268507402961\t0.0095392082457156\t0.00478904219156169\t2.95161733872075e-05\t2.23144805358541e-16\t0.00169955210003956\t0.0122010253741724\t0.00559914053192835\t0.012899478861054\t0.0166986140150368\t0.000693053815628781\t7.05990116138372e-05\t0.000565804888015597\t0.00378194247212571\t0.000201286266510223\t0.0027703629175422\t0.000187991720844614\t0.000357156698930088\t0.000149031907731445\t0.00517839314460723\t0.00183023108497679\t2.22984707708745e-16\t0.00179117680315968\t0.000372818549212098\t0.000176080577994101\t9.84953766270211e-05\t0.0732336581893037\t0.000185115845773959\t0.00199073258959297\t0.010198419245017\t0.000278037562874744\t0\t0.000320961163699192\t0.000924135413562149\t0.000189966299978384\t0.00326828088425488\t0.00175982419356306\t0.000660288546094643\t0.00776672963985034\t0.00137952309886472\t0.000871784669186711\t0.00128026629538944\t0.000803677725232182\t0.005460709892286\t0.00502066774881059\t0.00929085104195544\t0.00533115899396529\t5.14109633879425e-05\t5.02127176260567e-06\t0.000653866153598358\t5e-04\t0.000451919160700534\t8.32455386394573e-06\t7.53498816216334e-05\t7.54001734203989e-05\t0.00402896858404249\t0.000777809436688011\t0.00160104372040133\t0.000329054458512845\t0.00948181519814968\t0.00635872887737995\t0.00440226276306021\t0.000828642772100947\t0.000452773907349365\t4.24186599260828e-05\t0.000784031941197604\t0.00673340021553614\t0.00139250033420008\t0.00486627357613448\t0.00403923801615695\t0.0100626154356853\t0.000600449784623938\t0.000741770135367468\t0.00675145972994161\t0.0103166378717631\t0.00428278644639064",
                 "T[T>A]T\t0.00225039928834573\t0.000205005920775997\t0.0130052020808323\t0.00561953357871297\t0.00874825034993003\t0.000426233554676285\t0.000798518182404106\t2.21941509534577e-16\t0.279023449130665\t0.0272958100931507\t0.0378984689018564\t0.0169985891171033\t0.00529041080039863\t0.00151997872029791\t0.00483127101989216\t0.0069698058173539\t0.000674959918546718\t0.00525068784010706\t6.19972696402449e-05\t0.00182079885729065\t0.000475101719278097\t0.00715777894119455\t0.00198025002636833\t0.00130991016636079\t0.00710466776672274\t0.000672672610240596\t0.00671307203602512\t7.63964139523291e-05\t0.0301138403210115\t7.32458373548854e-18\t0.00219080621668774\t0.0144977528483085\t0.00131017700491336\t0.00192919186152921\t0.00571930796373638\t0.00120017586176902\t0.00070087566465709\t0.00887533157559124\t0.000710928978195079\t0.00272118915966277\t0.0459806387944808\t0.0054081304093175\t0.00269933326468362\t0.00195040568438235\t0.00249899790184136\t0.0107013911808535\t0.0210027933715184\t0.0535049010489361\t0.00352076541440109\t8.77187060140575e-18\t0.0139035214143862\t0.00138971552523199\t0.0034\t0.00434922201116664\t0.000426233166591452\t0.000331219267154857\t0.00295000678501561\t0.00637836713801267\t0.0013396717804138\t0.00508331381227422\t7.45123317909026e-18\t0.00210040210085594\t0.0007438513026369\t0.0289148622391909\t0.0145937086522\t0.00298850768868566\t0.000530233249076035\t0.00705995547050334\t0.062798282010173\t0.00236530056767214\t0.00526077143401111\t0.0744373922977496\t0.00635055019348721\t0.162675746420022\t0.000508383568197773\t0.00604417975823281\t0.0309867817371561\t0.0173227732323144",
                 "T[T>C]A\t0.000255045252679182\t2.23006440649012e-16\t0.0152060824329732\t0.00240979998660111\t0.017996400719856\t0.00148081141061244\t0.00215139610547472\t0.0059984191766102\t0.0302025382213121\t0.081187537712961\t0.00207991597139476\t0.013998838096438\t0.00073005668890189\t0.00885987596173651\t0.0169987694058205\t0.00564833421335546\t0.000113160935898782\t0.0861112805777557\t2.20990267588615e-05\t0.00627275210725954\t0.0034207323788023\t0.0129959673513309\t0.00209026391672212\t0.00105992731018507\t0.001801183377479\t0.00123939678558446\t0.0138063180472648\t0.0841960478375145\t0.00463212892645459\t7.77486552250628e-18\t0.00146053747779183\t0.0144977528483085\t0.0423057155021643\t0.00171927979369442\t0.00130984150917739\t0.00259037956831814\t0.000543903511517057\t0.0045875869292752\t0.000896910398651174\t0.0110048091015774\t0.00139941074591898\t0.000367872826363926\t0.00296926659115199\t0.0477099236641221\t0.00210915422915411\t0.00594077230039905\t0.0102013567804518\t0.0636058262936885\t0.00682148299040211\t0.000137029221481481\t0.0199050414493731\t0.0012797380376237\t0.0093\t0.00136975497822949\t0.000605331140347015\t0.000173114601866436\t0.00109000250700577\t0.00445885853221575\t0.00276932151622852\t0.00711463803253341\t0.00473078294457677\t0.00488093440579858\t0.000567886478357204\t0.00490251989522615\t0.00330857367388919\t0.00204897684341324\t0.00024010562222311\t0.000432938967529577\t0.0129635494149632\t0.00153390036813609\t0.00976894695461804\t0.00403923801615695\t0.0075170119624234\t0.010115271910268\t0.000318737242809893\t0.0345361386185545\t0.0943424208843152\t0.00888349531623618",
                 "T[T>C]C\t0.00339060159444089\t4.35012563597848e-05\t0.00899359743897559\t0.000502958254464879\t0.00970805838832235\t0.00109059759295106\t0.000596387013424621\t0.00172954419592261\t0.0209017565836233\t0.0142978052868885\t0.00263989334830873\t0.0135988712936826\t0.00517040148167503\t8.09988660158755e-05\t0.0167597020133725\t0.00647906233358883\t0.000467665106767533\t0.0463060660946584\t2.48989034522919e-05\t0.00288126412582256\t0.00592126774342386\t0.00366886155226033\t0.0153019320219371\t0.00109992456717318\t0.00618406292934458\t0.000552730985829197\t0.0166075999698982\t0.0547974278087386\t2.22102078115101e-16\t0.000190118976740823\t0.00300110440642156\t0.00740885162799766\t0.0314042427131906\t0.00348853865115903\t0.00204975198001042\t0.00150021982721128\t0.000285949272599041\t0.0102945850482646\t0.0012198781341744\t0.0125054648881561\t0.00623737361038174\t0.000821715932802031\t0.00227943697906617\t0.0451093827516123\t0.00218912216201303\t0.00644083730885015\t0.00755100428356972\t0.0268024551048876\t0.0030306588652373\t0.0359076573079209\t0.010502660061227\t0.00109977487608287\t0.0543\t0.0174968701598658\t0.000404221125124288\t8.4756108543856e-05\t0.0290000667001534\t0.00392899417749056\t0.00217946603082245\t0.00503328119601168\t0.00819135566936231\t0.000980187647066107\t0.00131973618209773\t0.0267137308576608\t0.0195915540810357\t0.00223888201426618\t0.00109047970092996\t0.000538464959615128\t0.0233244487466156\t0.00199500047880011\t0.00715886112738408\t0.00230813600923254\t0.00255683604733392\t0.000446488737295926\t0.000177915248462985\t0.00435019982599201\t0.0132895487751766\t0.00485704424721514",
                 "T[T>C]G\t0.000416073823978587\t0.000117003379174594\t0.0069327731092437\t0.00173985559198586\t0.0126974605078984\t0.00213116777338142\t0.000319207143091366\t0.00380899617714747\t0.0129010842071168\t0.0227965007371368\t2.21991031562325e-16\t0.00930922733413127\t0.000363028189138885\t0.000667990648130924\t0.0151153165387459\t0.00496822833920109\t0.000155220752781515\t0.0400052406865299\t9.97956050015555e-05\t0.00431189179940807\t0.00132028267252019\t0.00722775722693248\t0.00235029674846746\t0.0012099170238905\t0.00326214322810086\t0.00107947461970259\t0.0120054939541433\t0.0275987045168099\t0.00122056096982173\t0.000302189110398571\t0.00203074731501192\t0.00488924216746404\t0.0303040940831106\t0.00114951846671429\t0.00140982941064131\t0.000379055543008717\t0.000202963994187431\t0.00400789184888749\t0.00168983118586453\t0.320139901136797\t0.00109953701465063\t0.000437848635726629\t0.00137965922417163\t0.01000208043273\t0.00191923038861417\t0.00498064748417294\t0.00601079943632503\t0.0205018779720222\t0.00341074149520106\t2.78059296144903e-05\t0.0191048387780415\t0.00123974622394796\t0.0237\t0.00157971742014788\t0.000297162559806717\t6.31417998715149e-05\t0.0010200023460054\t0.00246936784183249\t0.00106973791421102\t0.00472307897518392\t0.0884146326216885\t0.00416079654264797\t0.00184963025521272\t0.00296152222242233\t0.00215906922525699\t0.00134932621395506\t0.000109047970092996\t2.23999983200001e-05\t0.00382353512239136\t0.00184590044301611\t0.00171129070769146\t0\t0.00869055846271365\t0.000138565642613613\t0.00015027841462437\t0.0129352794825888\t0.0109556074906094\t0.00210027657597401",
                 "T[T>C]T\t0.00433076840823866\t3.5801033969662e-05\t0.01390556222489\t0.00063594721638104\t0.018496300739852\t0.00174095395572004\t0.000812527273323476\t0.00296921749242205\t0.0881074045462779\t0.0430933851653771\t0.00456981537945867\t0.0383968130645156\t0.00325025238209746\t2.50996486049195e-05\t0.0199208052091363\t0.00755489945060561\t0.000106150966418326\t0.0597078217246459\t0.00053097661578984\t0.00292128168312565\t0.00135028909689565\t0.0083873973905897\t0.0210026517948156\t0.000999931424702891\t0.00337221554561347\t0.00146928489903964\t0.0193088361095804\t0.0406980896316726\t2.22102078115101e-16\t7.32458373548854e-18\t0.00367135105718905\t0.0169973654083617\t0.0574077557878069\t0.00152935935136771\t0.00205975077015681\t0.000822120465311782\t0.000444921071002004\t0.0120936387460196\t0.000683931675225645\t0.00912398718239871\t0.00707702005793313\t0.00105963368463522\t0.00279930857078302\t0.0614127738569622\t0.00305877343185383\t0.00948123256023283\t0.0102013567804518\t0.0461042231468403\t0.00236051317556437\t0.00184039246369288\t0.0371093988830021\t0.000544888461331966\t0.0365\t0.0114979432479118\t0.000532291184569607\t0.000208137787215136\t0.00511001175302703\t0.0212945485955595\t0.00199951012002059\t0.00219142859229932\t0.0125020690924333\t0.000887169839742486\t0.000284943039316555\t0.195100281544714\t0.031986210744548\t0.00260869734697979\t0.00132058092222711\t0.00982076326344276\t0.0940543800106807\t0.0034393008254322\t0.00658776422844024\t0.0963646863854588\t0.00678974253975264\t0.00400010515476431\t0.00127376175934652\t0.0142869094285236\t0.0140800925909764\t0.0167659576522223",
                 "T[T>G]A\t2.23039573911599e-16\t1.670048232663e-05\t0.00725290116046418\t0.000376968711596937\t0.00521895620875825\t0.00010505756629345\t0.0001280831169771\t0.000954748385610456\t0.019301622108322\t0.000210967616470872\t2.21991031562325e-16\t0.0656945473525697\t0.00269020889472066\t0.0112998418022147\t0.00873076870500676\t0.0198976389143985\t8.52212003981093e-05\t0.0016302135579761\t0.000183991896996856\t9.9243542111666e-05\t0.000231049467691033\t0.00977696620738585\t2.22028033259479e-16\t1.21991633813753e-05\t0.000696457572625215\t0.00165919247102435\t2.22101638151651e-16\t0.000464978173924516\t0.00506232664532619\t0.000186116471967332\t0.000123045280663284\t0.00207967764996426\t0.0014301932191039\t0.00118950171773044\t0.0422948823192393\t0.000467068439538445\t0.000117979070512891\t0.00224881712219373\t2.25977424855257e-05\t0.000468204605412565\t2.21906561138581e-16\t0.00428851745951425\t0.00078780541206322\t0.0183038071918959\t0.00116953101806176\t0.00839109084180943\t0.00846112532966885\t0.0419038383915967\t0.00115025006436399\t0.000204043512278994\t7.68194564478318e-18\t0.000284941672439652\t7e-04\t0.00240956897630152\t0.000250136834854139\t0.000150099365780147\t0.00032700075210173\t0.00254934736707403\t0.00142964973581472\t0.00130084802282608\t3.24053630875871e-05\t0.0060611603481843\t0.00945810930503371\t0.00322165593114861\t0.000797656130442166\t0.00159920143876155\t0.00161070854908003\t0.00187569785932266\t0.00710641122747622\t0.00713550171252041\t0.00820445544980691\t0.0305828041223312\t0.00686551166666311\t0.0185831555139922\t0.0059546843707242\t0.0022084299116628\t0.0536735480767485\t0.00407180522349783",
                 "T[T>G]C\t5.5109778127933e-05\t7.0402033281123e-05\t0.00628251300520208\t0.000173985559198586\t0.00655868826234753\t0.000287157347868765\t0.000116075324760497\t0.0015495916206243\t0.0174014624189017\t0.000114982350209243\t0.00113995394586059\t0.00861928459937825\t2.23017317294687e-16\t0.00554992230108777\t0.00095215516990376\t0.000738317706180713\t6.73958494335224e-05\t0.00113014804939447\t2.21990223550554e-16\t0.00120052671909273\t0.000294062958879496\t0.00550829077737178\t0.000116014648009458\t0.00874939996615029\t0.00210138060705884\t0.00201901734425855\t2.22101638151651e-16\t0.000378982210575035\t0.000752345778119623\t5.2733000390744e-18\t0.00264097187765098\t0.000578910268908319\t0.0017202324033977\t0.000279882757113045\t0.0356956808226204\t9.93145525613868e-05\t9.63829016732432e-05\t0.0023787487781427\t0.000223977624635299\t0.000164071699332608\t2.21906561138581e-16\t0.00193932957376635\t0.000743816277379487\t0.0195040568438235\t0.0009026380421451\t0.00508066048586316\t0.0114015164016814\t0.0155014199300656\t0.000274059580552812\t0.00825175968775342\t7.68194564478318e-18\t0.000134972371155624\t0.001\t2.24959759198275e-05\t0.000222121509350475\t0.000138091416517735\t0.000655001506503465\t0.00346911190735172\t0.00111972566721153\t0.00436284413809362\t0.000706116862340635\t0.00565108184277909\t9.99800137952823e-07\t0.00396203648675419\t0.00158931484636973\t0.00232883709519651\t0.0106046649815207\t0.000737916944656229\t0.00640445420500658\t0.00236850056844014\t0.00249728643973463\t0.0207732260830929\t0.00292901393472061\t0.000184753856809388\t0.000143443225519016\t0.0003005799879768\t0.0132760163201577\t0.00123543466055577",
                 "T[T>G]G\t0.000583103460046914\t9.54027553269763e-05\t0.00805322128851541\t0.00231980745598115\t0.00693861227754449\t0.000324177633134076\t2.23144805358541e-16\t0.00134964431473729\t0.00764064211956371\t0.000124980815444829\t0.00308987516904317\t0.0108990953750839\t2.16016773702477e-05\t0.00275996136054094\t0.0011083602339478\t0.00314796334803787\t0.000101143845360858\t0.000875114640017842\t3.66983838031772e-05\t0.00011405003831381\t0.000148031693585597\t0.0052383745323826\t0.000920116173868111\t0.00477967221007982\t0.00145095327630253\t0.00206899302109663\t0.00279127734433831\t0.000130993851148627\t0.00557256114910412\t0.000182113967193841\t0.00304111913184052\t0.00942853857652064\t0.00284038373584271\t0.00234901599719877\t0.0139983062049492\t0.000180026379265354\t0.00081285579938119\t0.00534718737943842\t0.000817918289962832\t0.000959419266219338\t0.00151936023842632\t0.00481833430183186\t0.000901777261016529\t0.0302062829068446\t0.00189923840539943\t0.007280946523048\t0.00866115193320712\t0.0194017772027918\t0.00293063712051\t0.022804863137064\t0.000917232312013825\t0.000707855102060608\t0.000900000000000001\t0.00153972457406819\t0.000350191568795794\t0.000112074193115843\t0.00872002005604613\t0.00275929362083307\t0.00147963748881524\t0.00615401180029261\t0.00769127290566498\t0.0795152223895464\t0.000100979813933235\t0.00540277702739208\t0.00673709563807042\t0.0019590217624829\t0.0239105182130514\t0\t0.0034100001091541\t0.00201070048256812\t0.00199598916177885\t0\t0.00503726599191085\t0.000292526689941625\t0.000628245461919583\t0.0017428399302864\t0.0127052233227372\t0.00304846818437678",
                 "T[T>G]T\t2.23039573911599e-16\t2.23006440649012e-16\t0.0105042016806723\t0.000567952859912627\t0.013497300539892\t0.00101055373291795\t8.29538312296997e-05\t0.00176953365710001\t0.0217018238212739\t0.000165974522910733\t0.000995959763225565\t0.0638946967401706\t0.0189014676989667\t0.0908987274178159\t0.025114705806213\t0.0145282083597554\t5.55790437378973e-05\t0.00221028954793078\t1.88991676806553e-05\t0.000605265554209253\t0.00599128273363326\t0.0157950987808483\t0.00458057834382169\t0.121991633813753\t0.00516339234877314\t0.00422794226050181\t0.00151069132256303\t2.21989579809124e-16\t0.000595273587740925\t7.32458373548854e-18\t0.00396145781647646\t0.00786878033904745\t0.0094512768675048\t0.000139941378556523\t0.47894204801219\t0.000928135999768046\t0.00888842319372543\t0.00729616221867298\t0.000634936569836674\t0.0022509836798681\t0.000231902352180859\t0.00361874899847123\t0.00180955304039902\t0.0291060540592443\t0.00528787956029632\t0.0109014171842339\t0.0250033254422838\t0.0879080523775978\t0.00174037835825508\t0.00132028155004055\t0.00900228005248029\t0.000723851827530905\t0.001\t0.00361935257021224\t3.20175148613297e-07\t0.000364241127626489\t0.00160000368000846\t0.00290925523066095\t0.00252938030182605\t0.00705459889301836\t0.000499082598169938\t0.0412078888358404\t0.0289942040006319\t0.119061197455492\t0.0216906491611466\t1.29935116899376e-17\t0.0318139949445621\t0.00660313350476499\t0.0127087624068075\t0.00488720117292828\t0.0062507660583403\t0.10213502640854\t0.00568978424907106\t0.00833021898479667\t0.00100868176471552\t0.001794349928226\t0.0549535891254385\t0.00333337392778025")
# MAIN
run()
}
