library(MCMCglmm)
library(Matrix)
library(data.table)
library(dplyr)


# Load phenotype and covariates

pheno_total <- read.table("pheno.txt", header = TRUE, sep = "\t", stringsAsFactors = FALSE) # Row = individuals; Column = Traits
covar <- read.table("covariates.txt", header = TRUE, sep = "\t", stringsAsFactors = FALSE) #Row = individuals; Column = PCs

phenotype_cols <- setdiff(colnames(pheno_total), "ID")
covariate_cols <- setdiff(colnames(covar), "ID")

pheno <- merge(pheno_total, covar, by = "ID")


# Z-score normalization

cat("Standardizing phenotypes (Z-score)...\n")
for (col in phenotype_cols) {
  if (sd(pheno[[col]], na.rm = TRUE) > 0) {
    pheno[[col]] <- scale(pheno[[col]])
  } else {
    cat(paste0("Skipping ", col, ": zero standard deviation.\n"))
  }
}

# Load GRM and compute inverse

GRM_raw <- read.table("GRM.txt", header = TRUE, sep = "\t", row.names = 1, check.names = FALSE)
GRM <- as.matrix(GRM_raw)
GRM <- (GRM + t(GRM)) / 2

eig <- eigen(GRM, symmetric = TRUE)$values
if (any(eig <= 1e-6)) {
  GRM <- GRM + diag(nrow(GRM)) * 1e-5
}

GRM_inv_sparse <- as(as(as(solve(GRM), "dMatrix"), "generalMatrix"), "TsparseMatrix")
GRM_inv_sparse <- as(GRM_inv_sparse, "CsparseMatrix")


# Prior setting

prior <- list(
  G = list(G1 = list(V = 1, nu = 1)),
  R = list(V = 1, nu = 1)
)

# Prepare result table

heritability_results <- data.frame(
  Phenotype = phenotype_cols,
  h2_mean = NA,
  h2_lower = NA,
  h2_upper = NA,
  stringsAsFactors = FALSE
)


# Main loop for heritability estimation

covariate_formula <- paste(covariate_cols, collapse = " + ")

for (i in seq_along(phenotype_cols)) {
  pheno_col <- phenotype_cols[i]
  cat("Processing: ", pheno_col, "...\n")
  
  matched_ids <- intersect(pheno$ID, rownames(GRM_inv_sparse))
  
  data_i <- pheno[match(matched_ids, pheno$ID),
                  c("ID", pheno_col, covariate_cols)]
  colnames(data_i)[2] <- "y"
  
  GRM_inv_sub <- GRM_inv_sparse[matched_ids, matched_ids]
  
  fixed_formula <- as.formula(paste("y ~", covariate_formula))
  
  model <- MCMCglmm(
    fixed = fixed_formula, 
    random = ~ ID,
    ginverse = list(ID = GRM_inv_sub),
    data = data_i,
    prior = prior,
    nitt = 15000,
    burnin = 5000,
    thin = 10,
    verbose = FALSE
  )
  
  V_animal <- model$VCV[, "ID"]
  V_resid  <- model$VCV[, "units"]
  h2 <- V_animal / (V_animal + V_resid)
  
  heritability_results[i, "h2_mean"]  <- mean(h2)
  heritability_results[i, "h2_lower"] <- quantile(h2, 0.025)
  heritability_results[i, "h2_upper"] <- quantile(h2, 0.975)
  
  cat(sprintf("%s: h2 = %.3f [%.3f - %.3f]\n", 
              pheno_col, mean(h2), quantile(h2, 0.025), quantile(h2, 0.975)))
}



out_name <- "heritability_results_SCALED.txt"

write.table(
  heritability_results, 
  file = out_name, 
  sep = "\t", quote = FALSE, row.names = FALSE
)

cat("All phenotypes completed. Results saved to: ", out_name, "\n")
