""" This script takes in a config file containing 
parameter and input file list and draws a figure with
a HiC matrix heatmap, TADs, and a set of signal tracks
usage: python pyrmaid.py [config].ini
dependencies: heatmap.py, bar.py, readdata.py
"""
import matplotlib.patches as patches
import sys
import csv
import numpy as np
from configparser import ConfigParser
import matplotlib.pyplot as plt
import matplotlib.cm as cm
from matplotlib import gridspec
from matplotlib.colors import Normalize
import heatmap
import bar
import readdata as rd
import seaborn as sns

def main(config_file):
	# Read in parameters from config file:
	config = ConfigParser()
	config.read(config_file)
	chro = config.get('coordinates','chro')
	start = config.getint('coordinates','start')
	end = config.getint('coordinates', 'end')
	binsize = config.getint('coordinates','binsize')
	contexts = config.get('content','contexts').split(',')
	signals = config.get('content','signals').split(',')
	nCols = len(contexts)
	nSignals = len(signals)
	# hic file names
	hicfiles = []
	genefiles = []
	for i in range(nCols):
		fname = config.get('hicfiles','file'+str(i))
		hicfiles.append(fname)
		fname = config.get('genefiles','file'+str(i))
		genefiles.append(fname)
	#pval = np.loadtxt(config.get('files','pval'))
	score = np.loadtxt(config.get('files','score'))
	sig = np.loadtxt(config.get('files','sig'))
	# output file:
	output=config.get('files','output')

	#read in gene expression
	geneloc = []
	rnaseq = []
	rnamax = 0
	for i in range(nCols):
		r, g = rd.processGeneExpressionFile(genefiles[i],chro,start,end,binsize)
		rnamax = max(rnamax, np.amax(r))
		geneloc.append(g)
		rnaseq.append(r)
	
	#Drawing a grid to fit all plots
	fig = plt.figure(figsize = (nCols*4,4))
	height_ratios = [3.5, 0.5, 0.1,0.2,0.4]
	gs = gridspec.GridSpec(figure=fig,ncols=nCols,nrows=nSignals+1,
		height_ratios=height_ratios)

	#Draw subplots to fill in
	heatmaps = []
	tracks = []
	for i in range(nCols):
		heatax = fig.add_subplot(gs[0,i],frame_on=False)
		heatmaps.append(heatax)
	for i in range(nSignals):
		row = []
		for j in range(nCols):
			trackax = fig.add_subplot(gs[i+1,j],frame_on=False)
			row.append(trackax)
		tracks.append(row)

	# draw HiC matrices
	for i in range(nCols):
		C = rd.readSymmetricSparseMatrix(hicfiles[i],0,200)
		np.fill_diagonal(C, 0)
		C = np.sqrt(C)
		#C = np.log2(C+1)
		vmax = np.percentile(C, 95)
		#vmax = C.max()
		ax = heatmaps[i]
		print('Drawing heatmap {:d}...'.format(i+1))
		heatmap.drawRotatedHalfHeatmapUp(fig,ax,C,vmax)
		#ax.axvline(x=101,ymin=-0.5,c='grey',linestyle='--',alpha=0.5,linewidth=3,clip_on=False)

	# draw boundary scores and sig boundary marks 
	max_score = np.percentile(score, 99.7) 
	colors = ['#AA4499','#88ccee','#44aa99','#117733']
	markers = ['*','^','d','p']
	for j in range(nCols):
		bar.drawTrack(tracks[0][j],score[:,j],'#0077bb',max_score)
		bar.drawMarks(tracks[1][j],200,np.nonzero(sig[:,j])[0],'#ee3377','*')

	# draw rnaseq and genes
	colormap = 'Purples'
	for t in range(nCols):
		heatmap.drawVectorHeatmap(tracks[3][t],rnaseq[t],colormap,rnamax)
		bar.drawGenes(tracks[2][t],geneloc[t],int((end-start)/binsize))
		ax = tracks[3][t]
		ax.set_xticklabels([int(x) for x in np.arange(start,end,(end-start)/4)],ha='left')
		ax.tick_params(which='both', direction='out',
			right=False,left=False,top=False,
			labelbottom=True,labelleft=False,
			length=4)
		ax = heatmaps[t]
		ax.axvline(x=101,ymin=-0.7,c='grey',linestyle='--',alpha=0.5,linewidth=3,clip_on=False)
		
	gs.tight_layout(fig)
	plt.savefig(output +'.png')

if __name__ == '__main__':
	main(sys.argv[1])
