""" This script takes in a config file containing 
parameter and input file list and draws a figure with
a HiC matrix heatmap, TADs, and a set of signal tracks
usage: python pyrmaid.py [config].ini
dependencies: heatmap.py, bar.py, readdata.py
"""
import matplotlib.patches as patches
import sys
import csv
import numpy as np
from configparser import ConfigParser
import matplotlib.pyplot as plt
import matplotlib.cm as cm
from matplotlib import gridspec
from matplotlib.colors import Normalize
import heatmap
import bar
import readdata as rd
import seaborn as sns

def main(config_file):
	# Read in parameters from config file:
	config = ConfigParser()
	config.read(config_file)
	chro = config.get('coordinates','chro')
	start = config.getint('coordinates','start')
	end = config.getint('coordinates', 'end')
	binsize = config.getint('coordinates','binsize')
	contexts = config.get('content','contexts').split(',')
	signals = config.get('content','signals').split(',')
	nCols = 1
	nSignals = 3 #len(signals)
	# hic file names
	hicfiles = []
	genefiles = []
	for i in range(2):
		fname = config.get('hicfiles','file'+str(i))
		hicfiles.append(fname)
		fname = config.get('genefiles','file'+str(i))
		genefiles.append(fname)
	score = np.loadtxt(config.get('files','score'))
	sig = np.loadtxt(config.get('files','sig'))
	# output file:
	output=config.get('files','output')

	#read in gene expression
	geneloc = []
	rnaseq = []
	for i in range(2):
		r, g = rd.processGeneExpressionFile(genefiles[i],chro,start,end,binsize)
		geneloc.append(g)
		rnaseq.append(r)
	mask = np.logical_or((rnaseq[0]==0),(rnaseq[1]==0))
	rnaseq[0][mask] = 1
	rnaseq[1][mask] = 1
	#Drawing a grid to fit all plots
	fig = plt.figure(figsize = (6,6))
	height_ratios = [70, 20, 20,5]
	gs = gridspec.GridSpec(figure=fig,ncols=nCols,nrows=nSignals+1,
		height_ratios=height_ratios)

	#Draw subplots to fill in
	heatmap_ax = fig.add_subplot(gs[0,0],frame_on=False)
	tracks= []
	for i in range(nSignals):
		trackax = fig.add_subplot(gs[i+1,0],frame_on=False)
		tracks.append(trackax)

	# draw HiC matrices
	hicmats = []
	for i in range(2):
		C = rd.readSymmetricSparseMatrix(hicfiles[i],0,200)
		hicmats.append(C)
	mask = np.logical_or((hicmats[0] == 0),(hicmats[1]==0))
	hicmats[0][mask] = 1
	hicmats[1][mask] = 1
	D = np.log(hicmats[0]/hicmats[1])
	vmax = np.percentile(np.absolute(D), 95)
	#vmax = np.amax(D)
	#vmin = np.amin(D)
	print('Drawing heatmap...')
	heatmap.drawRotatedHalfHeatmapUp2(fig,heatmap_ax,D,-1*vmax,vmax,'RdBu_r')
	#ax.axvline(x=101,ymin=-0.5,c='grey',linestyle='--',alpha=0.5,linewidth=3,clip_on=False)

	# draw boundary scores and sig boundary marks
	print('Drawing boundary score diff...') 
	score_diff = (score[:,0] - score[:,1])
	max_score = np.percentile(np.absolute(score_diff), 99.7) 
	bar.drawTrack2(tracks[0],score_diff,'#44bb99',-1*max_score,max_score)

	# draw rnaseq and genes
	print('Drawing genes and expression...')
	bar.drawGenes(tracks[2],geneloc[0],int((end-start)/binsize))

	colormap = 'RdBu_r'
	rna_diff = np.log((rnaseq[0])/(rnaseq[1]))
	rnamax = np.amax(np.absolute(rna_diff))
	#heatmap.drawVectorHeatmap2(tracks[2],rna_diff,colormap,-1*rnamax,rnamax)
	bar.drawTrack2(tracks[1],rna_diff, '#ee8866',-1*rnamax,rnamax)

	ax = heatmap_ax
	ax.set_xticklabels([int(x) for x in np.arange(start,end,(end-start)/4)],ha='left')
	ax.tick_params(which='both', direction='out',
			right=False,left=False,top=False,
			labelbottom=True,labelleft=False,
			length=4)
	#ax = heatmap_ax
	#ax.axvline(x=101,ymin=-0.7,c='grey',linestyle='--',alpha=0.5,linewidth=3,clip_on=False)
		
	gs.tight_layout(fig)
	plt.savefig(output +'_diff.png')

if __name__ == '__main__':
	main(sys.argv[1])
