library(dplyr)
library(ggplot2)
library(stringr)

# Function to handle multiple input files and create a single combined stack plot
circulomicTablesCombinedPlot <- function(seqSummaryList, runIDList, ystack = 10, workDir, startTime = 0, endTime, maxRead = 2e+6, binSize = 1e+4, saveDir, filename) 
{
    # cols <- c("5" = "#009E73",  # Green for < 100 kb
    #           "4" = "#999999",  # Grey for 100-199 kb
    #           "3" = "#0072B2",  # Blue for 200-499 kb
    #           "2" = "#CC79A7",  # Pink for 500-999 kb
    #           "1" = "#000000")  # Black for >= 1 Mb
  cols <- c("1" = "#009E73", "2" = "#CC79A7", "3" = "#0072B2", "4" = "#000000", "5" = "#999999")
  setwd(workDir)
  message("------- Preparing Tables and Plots -------")
  
  combinedTable <- data.frame()  # Initialize an empty dataframe to store combined data
  
  # Loop through each file in the list and combine data
  for (i in 1:length(seqSummaryList)) {
    
    seqSummary <- seqSummaryList[[i]]
    runID <- runIDList[[i]]
    
    # Read and process sequence summary data
    tmp <- read.table(file = seqSummary, header = TRUE, sep = "\t") %>% 
      select(sequence_length_template, start_time, duration) %>% 
      mutate(minute = floor((start_time + duration)/60)) %>% 
      arrange(minute) %>% 
      filter(minute >= startTime & minute <= endTime) %>% 
      rename(bp_length = 1)
    
    # Create the length table and bin the read lengths
    lengthTable <- tmp %>% 
      arrange(bp_length) %>% 
      mutate(bin = cut(bp_length, breaks = seq(0, maxRead, by = binSize), include.lowest = FALSE, 
                       dig.lab = 0, labels = seq(10, maxRead/1000, by = 10), ordered_result = TRUE)) %>% 
      group_by(bin) %>% 
      summarise(bin_yield = sum(bp_length), .groups = "drop") %>%
      mutate(runID = runID)  # Add runID to the table
    
    # Combine data into one table for all inputs
    combinedTable <- bind_rows(combinedTable, lengthTable)
  }
  
  # Build the combined stack plot for all runs
  combinedTable <- combinedTable %>%
    mutate(bin = as.numeric(as.character(bin))) %>%
    mutate(group = ifelse(bin <= 100, "1",
                        ifelse(bin > 100 & bin <= 200, "2",
                               ifelse(bin > 200 & bin <= 500, "3",
                                      ifelse(bin > 500 & bin < 1000, "4",
                                             ifelse(bin >= 1000, "5", NA)))))) %>%
    mutate(group = factor(group, levels = c("5", "4", "3", "2", "1"))) %>%  # Set levels from smallest to largest
    mutate(category = factor(runID, levels = unique(runIDList)))  # Keep the order of runIDs consistent
  
  stackplotLength <- ggplot(combinedTable, aes(x = category, y = (bin_yield / 1e+9), fill = group)) +
    geom_col(position = "stack", width = 0.8) +  # Stack bars for all runs
    labs(x = NULL, y = "Yield (Gb)") +
    theme(legend.position = "right", panel.grid.minor = element_blank(),
          axis.text = element_text(size = 36), axis.title = element_text(size = 40),
          axis.title.y = element_text(margin = margin(t = 0, r = 20, b = 0, l = 0)),
          legend.text = element_text(size = 30, margin = margin(t = 6, r = 0, b = 6, l = 0)),
          legend.title = element_text(size = 30), legend.spacing.y = unit(20, "points"),
          title = element_text(size = 24)) +
    scale_x_discrete(expand = expansion(mult = c(0.15, 0.5))) +
    ylim(0, ystack) +
    scale_fill_manual(values = cols, name = "Read Length",
                      labels = c(" >=1 Mb", " 500-999 kb", " 200-499 kb", " 100-199 kb", " <100 kb"))

  # Output combined plot
  setwd(saveDir)
  plot(stackplotLength)
  ggsave(paste0(filename, "combined_stackplot.png"), plot = stackplotLength, units = "mm", width = 297, height = 210, bg = "white")
  
  # Reset working directory
  setwd(workDir)
}
#
# Example usage
# circulomicTablesCombinedPlot(seqSummaryList, runIDList, ystack = 20,
#                              workDir = "/path/to/run/folder", 
#                              saveDir = "/path/to/saving/directory",
#                              startTime = 0, endTime = 3900, 
#                              filename = "filename_to_save")