#
compileTable <- function(seqSummary, workDir, runStatus, startPoint = 0, stopPoint, saveDir, saveFile, startPores = NULL) {
  # Set working directory
  setwd(workDir)
  
  # Read in the sequencing summary file
  tab <- read.table(file = seqSummary, header = TRUE, sep = "\t")
  
  # Select relevant columns and rename them
  tab <- tab[, -c(1, 2, 4, 19, 22)]  # Remove columns 1, 2, 4 and 19, 22
  colnames(tab)[which(names(tab) == "start_time")] <- "start"
  colnames(tab)[which(names(tab) == "sequence_length_template")] <- "bp_length"
  
  # Calculate end time and minute, and filter by time
  tab$end <- round(tab$start + tab$duration, 2)
  tab$minute <- floor(tab$end / 60)
  tab <- tab[tab$minute >= startPoint & tab$minute <= stopPoint, ]
  
  # Group by minute and sort
  tab <- tab[order(tab$minute), ]
  
  # Long read yield
  long <- tab[tab$bp_length >= 100000, ]
  compile <- data.frame(runID = unique(long$experiment_id), 
                        run_info = runStatus, 
                        longread = sum(long$bp_length))
  
  message("Part 1 - Long read yield created \nPart 2 - N50 calculations started")
  
  # Correct N50 calculation
  tab <- tab[order(-tab$bp_length), ]
  bp_sum <- sum(tab$bp_length)
  tab$bp_cumsum <- cumsum(tab$bp_length)
  N50_row <- tab[tab$bp_cumsum >= bp_sum * 0.5, ][1, ]
  N50_value <- N50_row$bp_length
  compile$finalN50 <- N50_value
  message(runStatus, " --- N50 at minute ", stopPoint, " is ", N50_value, " bp")
  
  # Adding N50 to compile
  compile$finalN50 <- N50_value
  
  # Total yield
  total_yield <- sum(tab$bp_length)
  compile$total_yield <- total_yield
  message("Part 3 - Total yield added to table")
  
  # Active pore number
  pore <- unique(tab[, c("channel", "mux")])
  seq_pores <- nrow(pore)
  compile$seq_pores <- seq_pores
  message("Part 4 - Active pore number calculated")
  
  # Passed bases/reads calculation
  pass <- tab[tab$passes_filtering == TRUE, ]
  compile$pass_yield <- sum(pass$bp_length)
  compile$pass_reads <- nrow(pass)
  compile$total_reads <- nrow(tab)
  
  # Additional parameters
  compile$run_minutes <- stopPoint
  compile$ratio_longread <- sum(long$bp_length) / total_yield
  compile$ratio_shortread <- 1 - (sum(long$bp_length) / total_yield)
  compile$norm_yield <- total_yield / seq_pores
  
  message("Part 5 - Passed bases/reads calculated and more parameters added")
 
  # Rename columns
  colnames(compile) <- c("runID", "run_info", "longread", "finalN50", 
                          "total_yield", "seq_pores", "pass_yield", "pass_reads", "total_reads", 
                          "run_minutes", "ratio_longread", "ratio_shortread", "norm_yield")
  
  if (!is.null(startPores)) {
    compile$starting_pores <- startPores
  }
  
  # Save the compiled data to the specified file
  setwd(saveDir)
  write.table(compile, file = saveFile, append = TRUE, row.names = FALSE,
              col.names = !file.exists(saveFile), sep = "\t", quote = FALSE)
  message("End part - Table is written to saved directory")
  
  # Reset working directory
  setwd(workDir)
}
#
# Example usage
# compileTable(seqSummary = "/path/to/sequencing_summary.txt", 
#              workDir = "/path/to/sequencing_summary/run_folder", 
#              runStatus = "Plot Title", startPoint = 11, stopPoint = 360, 
#              saveFile = "filename_to_save",
#              saveDir = "/path/to/saving/directory")