library(dplyr)
library(ggplot2)
library(tools)
#
muxRecovery <- function(muxFile = NULL, workDir, flowcellID = NULL, ymax = 2048, # change ymax 12,000 to when using PromethION
                        FCrunNo = NULL, runID, reloadMux = NULL, muxTab = NULL) 
  {
  ### muxFile   : CSV file of mux_scan_data; full path from workDir
  ### workDir   : path to working directory; preferably the parent folder where all the runs are
  ### flowcellID: series number of flowcell XXX#####
  ### FCrunNo   : the ordinal number of run on the flowcell (1 for first, 2 for second, etc.)
  ### runID     : sequencing ID/folder name
  ### reloadMux : the time points (hours) of mux scan repeats
  ### muxTab    : the table output from multiple muxFiles, if this is used, put in the run folder path in muxFile
  #
  setwd(workDir)
  if(is.null(muxTab)) {
    muxTab_each <- read.csv(muxFile) %>%
    select(mux_scan_assessment, repeat.) %>%
    rename(pore_state = 1, mux_no = 2) %>%
    group_by(mux_no) %>%
    count(pore_state) %>%
    mutate(pore_pct = n/sum(n)) %>%
    filter(pore_state == "single_pore")
  muxTab_each <- muxTab_each %>%
    mutate(x = muxTab_each %>%
             filter(mux_no == 1) %>%
             pull(n)) %>%
    mutate(relative_portion = n/x) %>%
    select(-x)
  } else {
    muxTab_each <- read.table(muxTab, header = TRUE, sep = "\t")
  }
  #
  singlePoreMux <- muxTab_each %>% 
    ggplot(aes(x = mux_no)) + 
    theme_classic() +
    coord_cartesian(ylim = c(0, ymax)) +
    geom_col(aes(y = n), width = 0.8, position = position_dodge(width = 0.9)) +
    scale_y_continuous(expand = c(0,0)) +
    scale_x_discrete(expand = c(0,0.3), limits = factor(1:max(muxTab_each$mux_no))) +
    geom_text(aes(y = n, label = n, vjust = -1.5), size = 7) +
    geom_text(aes(y = n, label = percent(round(pore_pct, 2))), vjust = -0.2, size = 5) +
    geom_vline(xintercept = NULL, color = "#999999", linetype = "dashed", size = .5) +
    labs(x = "Mux Scan Repeat", y = "Pore Count", 
         title = paste0("Mux of ", runID, " (", flowcellID, ") - Run ", FCrunNo)) +
    theme(plot.title = element_text(size = 24), plot.subtitle = element_text(size = 24),
          axis.line = element_line(color = "#999999", size = 0.5, linetype = "solid"),
          axis.title = element_text(size = 24), axis.text = element_text(size = 20),
          axis.title.y = element_text(margin = margin(t = 0, b = 0, l = 0, r = 20)), 
          axis.title.x = element_text(margin = margin(t = 20, b = 0, l = 0, r = 0)),
          legend.position = "none")
#
path = dirname(muxFile)
setwd(path)
plot(singlePoreMux)
ggsave(paste0(runID, flowcellID, FCrunNo, "_muxplot.png"), device = "png",
       width = 29.7, height = 21, units = "cm", bg = "white")
setwd(workDir)
}
#