# 加载必要的库
library(ggplot2)
library(dplyr)


# 读取数据
st_data <- read.csv("./csv/st_results.csv")  # 请确保文件路径正确
sa_data <- read.csv("./csv/sa_results.csv")  # 请确保文件路径正确
linear_st_data <- read.csv("./csv/linear_st_results.csv")  # 已正确读取

# 重命名列以确保一致性
colnames(st_data) <- c("N", "build_time_s", "query_time_s")
colnames(sa_data) <- c("N", "build_time_s")
colnames(linear_st_data) <- c("N", "build_time_s", "query_time_s")

# 添加算法标签
st_data$method <- "Sparse Table Construction"
sa_data$method <- "Enhanced Suffix Array Construction"
linear_st_data$method <- "Block Sparse Table Construction"

# 合并数据
combined_data <- bind_rows(st_data %>% select(N, build_time_s, method),
                           sa_data %>% select(N, build_time_s, method),
                           linear_st_data %>% select(N, build_time_s, method))


# 绘制美化后的对比图，增加不同的线型和点的形状
ggplot(combined_data, aes(x = N / 1e6, y = build_time_s, color = method, linetype = method, shape = method)) +  
  geom_line(size = 1.5) +  # 增加线条粗细
  geom_point(size = 4) +  # 增加点的大小
  labs(title = "Comparison of Different Method Construction Time",
       x = expression(paste("Array Size ", "(x ", 10^6, ")")),  # 设置横坐标标题，添加10的六次方
       y = "Time (s)") +
  scale_x_continuous(breaks = seq(0, 20, 5)) +  # 设置横坐标的刻度范围和间隔
  scale_color_manual(values = c("red", "green", "blue")) +  # 自定义颜色
  scale_linetype_manual(values = c("solid", "solid", "solid")) +  # 自定义线型
  scale_shape_manual(values = c(16, 17, 15)) +  # 自定义点的形状 (16 = 实心圆, 17 = 实心三角形, 15 = 实心方形)
  theme_minimal(base_size = 14) +  # 调整字体大小
  theme(
    plot.title = element_text(hjust = 0.5, size = 20, face = "bold"),  # 加粗标题
    axis.title = element_text(size = 16),  # 加大轴标题字体
    axis.text = element_text(size = 12),  # 调整轴文本的字体
    legend.position = "top",  # 将图例移到顶部
    legend.title = element_blank(),  # 移除图例标题
    legend.text = element_text(size = 14),  # 调整图例文字大小
    panel.grid.major = element_line(size = 0.5, linetype = 'solid', color = "grey80"),  # 调整网格线
    panel.grid.minor = element_blank()  # 移除次要网格线
  )

# 保存美化后的图表
ggsave("./plot/comparison_construction_time.svg", width = 12, height = 7)

# 创建数据框并添加方法标签
st_query_data <- st_data %>% select(N, query_time_s) %>% mutate(method = "Sparse Table Query")
sa_construction_data <- sa_data %>% select(N, build_time_s) %>% mutate(method = "Enhanced Suffix Array Construction")
linear_st_query_data <- linear_st_data %>% select(N, query_time_s) %>% mutate(method = "Block Sparse Table Query")

# 重命名列使其一致
colnames(st_query_data) <- c("N", "time_s", "method")
colnames(sa_construction_data) <- c("N", "time_s", "method")
colnames(linear_st_query_data) <- c("N", "time_s", "method")
# 合并数据
combined_query_data <- bind_rows(st_query_data, sa_construction_data, linear_st_query_data)

# 绘制对比图
ggplot(combined_query_data, aes(x = N / 1e6, y = time_s, color = method, linetype = method, shape = method)) +  
  geom_line(size = 1.5) +  # 增加线条粗细
  geom_point(size = 4) +  # 增加点的大小
  labs(title = "Comparison of Suffix Array Construction Time,\n Block and Regular Sparse Table Query Time",
       x = expression(paste("Array Size or Query Count ", "(x ", 10^6, ")")),  # 设置横坐标标题，添加10的六次方
       y = "Time (s)") +
  scale_x_continuous(breaks = seq(0, 20, 5)) +  # 设置横坐标的刻度范围和间隔
  scale_color_manual(values = c("blue", "red", "green")) +  # 自定义颜色
  scale_linetype_manual(values = c("solid", "solid", "solid")) +  # 自定义线型
  scale_shape_manual(values = c(16, 17, 15)) +  # 自定义点的形状 (16 = 实心圆, 17 = 实心三角形, 15 = 实心方形)
  theme_minimal(base_size = 14) +  # 调整字体大小
  theme(
    plot.title = element_text(hjust = 0.5, size = 20, face = "bold"),  # 加粗标题
    axis.title = element_text(size = 16),  # 加大轴标题字体
    axis.text = element_text(size = 12),  # 调整轴文本的字体
    legend.position = "top",  # 将图例移到顶部
    legend.title = element_blank(),  # 移除图例标题
    legend.text = element_text(size = 14),  # 调整图例文字大小
    panel.grid.major = element_line(size = 0.5, linetype = 'solid', color = "grey80"),  # 调整网格线
    panel.grid.minor = element_blank()  # 移除次要网格线
  )

# 保存图表
ggsave("./plot/comparison_query_time.svg", width = 10, height = 6)