library(ggplot2)

# 读取CSV文件
data <- read.csv("./csv/full_results.csv")

# 检查数据的维度和列名
print(dim(data))
print(colnames(data))

# 修改列名以确保清晰
colnames(data) <- c("N", "SA_Build", "SA_Query", "Linear_Query", "Linear_16Thread_Query", "Linear_Build")

# 计算三个方法的总时间
data$Enhanced_SA_Time <- data$SA_Build + data$SA_Query
data$Linear_Sparse_Time <- data$SA_Build + data$Linear_Query + data$Linear_Build
data$Linear_Sparse_16Thread_Time <- data$SA_Build + data$Linear_16Thread_Query + data$Linear_Build

# 重塑数据为长格式，以便绘制多个系列在同一个图中
combined_query_data <- data.frame(
  N = rep(data$N, 3),  # 重复 N 列3次，每个方法一次
  time_s = c(data$Enhanced_SA_Time, data$Linear_Sparse_Time, data$Linear_Sparse_16Thread_Time),  # 汇总时间列
  method = rep(c("Enhanced Suffix Array", "Block Sparse Table & Query", "Block Sparse Table & 16 Thread Query"), each = nrow(data))  # 方法列
)

# 绘制对比图
ggplot(combined_query_data, aes(x = N / 1e6, y = time_s, color = method, linetype = method, shape = method)) +  
  geom_line(size = 1.5) +  # 增加线条粗细
  geom_point(size = 4) +  # 增加点的大小
  labs(title = "Comparison of the execution time of three strategies in RaMA",
       x = expression(paste("Array Size ", "(x ", 10^6, ")")),  # 设置横坐标标题，添加10的六次方
       y = "Time (s)") +
  scale_x_continuous(breaks = seq(0, 20, 5)) +  # 设置横坐标的刻度范围和间隔
  scale_color_manual(values = c("blue", "red", "green")) +  # 自定义颜色
  scale_linetype_manual(values = c("solid", "solid", "solid")) +  # 自定义线型
  scale_shape_manual(values = c(16, 17, 15)) +  # 自定义点的形状 (16 = 实心圆, 17 = 实心三角形, 15 = 实心方形)
  theme_minimal(base_size = 14) +  # 调整字体大小
  theme(
    plot.title = element_text(hjust = 0.5, size = 20, face = "bold"),  # 加粗标题
    axis.title = element_text(size = 16),  # 加大轴标题字体
    axis.text = element_text(size = 12),  # 调整轴文本的字体
    legend.position = "top",  # 将图例移到顶部
    legend.title = element_blank(),  # 移除图例标题
    legend.text = element_text(size = 14),  # 调整图例文字大小
    panel.grid.major = element_line(size = 0.5, linetype = 'solid', color = "grey80"),  # 调整网格线
    panel.grid.minor = element_blank()  # 移除次要网格线
  )

# 保存图表
ggsave("./plot/comparison_RaMA_3_method.svg", width = 10, height = 6)
