
"""
Classes and methods used to read, store and manipulate alignment information generated by minimap2 and
written in PAF format
"""


class PAFAlignment:
    """Simple container for alignment information"""

    def __init__(self, alignment, original_names):
        """
        Takes a string representing an alignment in PAF format. Each PAF field is stored by type in the object.

        :param alignment: String representing one alignment in PAF format.
        """

        self.alignment = alignment.strip()
        align_parts = self.alignment.split("\t")
        self.query_name = align_parts[0] if not original_names else get_original_name(align_parts[0])
        self.query_length = int(align_parts[1])
        self.query_start = int(align_parts[2])
        self.query_end = int(align_parts[3])
        self.strand = align_parts[4]
        self.target_name = align_parts[5] if not original_names else get_original_name(align_parts[5])
        self.target_length = int(align_parts[6])
        self.target_start = int(align_parts[7])
        self.target_end = int(align_parts[8])
        self.match_count = int(align_parts[9])
        self.align_length = int(align_parts[10])
        self.map_qual = int(align_parts[11])
        self.identity = None
        self.identityNM = None

        for tag in align_parts[12:]:
            key, type, value = tag.split(":")
            if key == "dv":
                self.identity = 1 - float(value)
            elif key == "NM":
                self.identityNM = (self.align_length - float(value)) / self.align_length

        self.query_overlap = self.align_length / self.query_length


def load_query_alignments(paf_file, original_names=True, length=None, filtered_read_dict=None):
    """
    Reads through an alignment file in PAF format. Returns all alignments for each query sequence. Assumes
    the PAF file isn't sorted differently than the default.

    :param paf_file: Path to alignment file in PAF format.
    :param original_names: If true, read names will be set to the PacBio originals, otherwise unmodified
    :param length: If set, if the query length is shorter than length, nothing is returned for the query
    :param filtered_read_dict: if set, if either the query or target in the dictionary, the alignment is not returned
    :return: list of PAFAlignment objects for each query sequence
    """

    last_query_name = None
    query_align_list = []
    with open(paf_file, 'r') as if_align:
        for line in if_align:
            pa = PAFAlignment(line, original_names)
            if length is not None and pa.align_length < length:
                continue
            if filtered_read_dict is not None and (pa.query_name in filtered_read_dict or pa.target_name in filtered_read_dict):
                continue
            if pa.query_name != last_query_name:
                if last_query_name is not None:
                    yield query_align_list
                query_align_list = []
                last_query_name = pa.query_name
            query_align_list.append(pa)
    yield query_align_list


def get_original_name(read_name):
    return "/".join(read_name.split(":")[0].split("/")[0:3])
