/*
 * Decompiled with CFR 0.152.
 */
package picard.illumina;

import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import java.io.BufferedWriter;
import java.io.File;
import java.text.NumberFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.BaseCallingProgramGroup;
import picard.illumina.NewIlluminaBasecallsConverter;
import picard.illumina.parser.BaseIlluminaDataProvider;
import picard.illumina.parser.ClusterData;
import picard.illumina.parser.IlluminaDataProviderFactory;
import picard.illumina.parser.IlluminaDataType;
import picard.illumina.parser.IlluminaFileUtil;
import picard.illumina.parser.ParameterizedFileUtil;
import picard.illumina.parser.ReadDescriptor;
import picard.illumina.parser.ReadStructure;
import picard.illumina.parser.ReadType;
import picard.illumina.parser.readers.AbstractIlluminaPositionFileReader;
import picard.illumina.parser.readers.BclQualityEvaluationStrategy;
import picard.illumina.parser.readers.LocsFileReader;
import picard.util.IlluminaUtil;
import picard.util.TabbedTextFileWithHeaderParser;
import picard.util.ThreadPoolExecutorUtil;
import picard.util.ThreadPoolExecutorWithExceptions;

@CommandLineProgramProperties(summary="Tool determines the barcode for each read in an Illumina lane.  <p>This tool determines the numbers of reads containing barcode-matching sequences and provides statistics on the quality of these barcode matches.</p> <p>Illumina sequences can contain at least two types of barcodes, sample and molecular (index).  Sample barcodes (B in the read structure) are used to demultiplex pooled samples while index barcodes (M in the read structure) are used to differentiate multiple reads of a template when carrying out paired-end sequencing.  Note that this tool only extracts sample (B) and not molecular barcodes (M).</p><p>Barcodes can be provided in the form of a list (BARCODE_FILE) or a string representing the barcode (BARCODE).  The BARCODE_FILE contains multiple fields including 'barcode_sequence' (or 'barcode_sequence_1'), 'barcode_sequence_2' (optional), 'barcode_name', and 'library_name'. In contrast, the BARCODE argument is used for runs with reads containing a single barcode (nonmultiplexed) and can be added directly as a string of text e.g. BARCODE=CAATAGCG.</p><p>Data is output per lane/tile within the BaseCalls directory with the file name format of 's_{lane}_{tile}_barcode.txt'.  These files contain the following tab-separated columns:<ul> <li>Read subsequence at barcode position</li><li>Y or N indicating if there was a barcode match</li><li>Matched barcode sequence (empty if read did not match one of the barcodes)</li>  <li>The number of mismatches if there was a barcode match</li>  <li>The number of mismatches to the second best barcode if there was a barcode match</li>  </ul>If there is no match but we're close to the threshold of calling it a match, we output the barcode that would have been matched but in lower case.  Threshold values can be adjusted to accommodate barcode sequence mismatches from the reads.  The metrics file produced by the ExtractIlluminaBarcodes program indicates the number of matches (and mismatches) between the barcode reads and the actual barcodes.  These metrics are provided both per-barcode and per lane and can be found in the BaseCalls directory.</p><p>For poorly matching barcodes, the order of specification of barcodes can cause arbitrary output differences.</p><h4>Usage example:</h4> <pre>java -jar picard.jar ExtractIlluminaBarcodes \\<br />              BASECALLS_DIR=/BaseCalls/ \\<br />              LANE=1 \\<br />          READ_STRUCTURE=25T8B25T \\<br />              BARCODE_FILE=barcodes.txt \\<br />              METRICS_FILE=metrics_output.txt </pre>Please see the ExtractIlluminaBarcodes.BarcodeMetric <a href='http://broadinstitute.github.io/picard/picard-metric-definitions.html#ExtractIlluminaBarcodes.BarcodeMetric'>definitions</a> for a complete description of the metrics produced by this tool.</p><hr />", oneLineSummary="Tool determines the barcode for each read in an Illumina lane.  ", programGroup=BaseCallingProgramGroup.class)
@DocumentedFeature
public class ExtractIlluminaBarcodes
extends CommandLineProgram {
    public static final String BARCODE_SEQUENCE_COLUMN = "barcode_sequence";
    public static final String BARCODE_SEQUENCE_1_COLUMN = "barcode_sequence_1";
    public static final String BARCODE_NAME_COLUMN = "barcode_name";
    public static final String LIBRARY_NAME_COLUMN = "library_name";
    static final String USAGE_SUMMARY = "Tool determines the barcode for each read in an Illumina lane.  ";
    static final String USAGE_DETAILS = "<p>This tool determines the numbers of reads containing barcode-matching sequences and provides statistics on the quality of these barcode matches.</p> <p>Illumina sequences can contain at least two types of barcodes, sample and molecular (index).  Sample barcodes (B in the read structure) are used to demultiplex pooled samples while index barcodes (M in the read structure) are used to differentiate multiple reads of a template when carrying out paired-end sequencing.  Note that this tool only extracts sample (B) and not molecular barcodes (M).</p><p>Barcodes can be provided in the form of a list (BARCODE_FILE) or a string representing the barcode (BARCODE).  The BARCODE_FILE contains multiple fields including 'barcode_sequence' (or 'barcode_sequence_1'), 'barcode_sequence_2' (optional), 'barcode_name', and 'library_name'. In contrast, the BARCODE argument is used for runs with reads containing a single barcode (nonmultiplexed) and can be added directly as a string of text e.g. BARCODE=CAATAGCG.</p><p>Data is output per lane/tile within the BaseCalls directory with the file name format of 's_{lane}_{tile}_barcode.txt'.  These files contain the following tab-separated columns:<ul> <li>Read subsequence at barcode position</li><li>Y or N indicating if there was a barcode match</li><li>Matched barcode sequence (empty if read did not match one of the barcodes)</li>  <li>The number of mismatches if there was a barcode match</li>  <li>The number of mismatches to the second best barcode if there was a barcode match</li>  </ul>If there is no match but we're close to the threshold of calling it a match, we output the barcode that would have been matched but in lower case.  Threshold values can be adjusted to accommodate barcode sequence mismatches from the reads.  The metrics file produced by the ExtractIlluminaBarcodes program indicates the number of matches (and mismatches) between the barcode reads and the actual barcodes.  These metrics are provided both per-barcode and per lane and can be found in the BaseCalls directory.</p><p>For poorly matching barcodes, the order of specification of barcodes can cause arbitrary output differences.</p><h4>Usage example:</h4> <pre>java -jar picard.jar ExtractIlluminaBarcodes \\<br />              BASECALLS_DIR=/BaseCalls/ \\<br />              LANE=1 \\<br />          READ_STRUCTURE=25T8B25T \\<br />              BARCODE_FILE=barcodes.txt \\<br />              METRICS_FILE=metrics_output.txt </pre>Please see the ExtractIlluminaBarcodes.BarcodeMetric <a href='http://broadinstitute.github.io/picard/picard-metric-definitions.html#ExtractIlluminaBarcodes.BarcodeMetric'>definitions</a> for a complete description of the metrics produced by this tool.</p><hr />";
    @Argument(doc="The Illumina basecalls directory. ", shortName="B")
    public File BASECALLS_DIR;
    @Argument(doc="Where to write _barcode.txt files.  By default, these are written to BASECALLS_DIR.", optional=true)
    public File OUTPUT_DIR;
    @Argument(doc="Lane number. ", shortName="L")
    public Integer LANE;
    @Argument(doc="A description of the logical structure of clusters in an Illumina Run, i.e. a description of the structure IlluminaBasecallsToSam assumes the  data to be in. It should consist of integer/character pairs describing the number of cycles and the type of those cycles (B for Sample Barcode, M for molecular barcode, T for Template, and S for skip).  E.g. If the input data consists of 80 base clusters and we provide a read structure of \"28T8M8B8S28T\" then the sequence may be split up into four reads:\n* read one with 28 cycles (bases) of template\n* read two with 8 cycles (bases) of molecular barcode (ex. unique molecular barcode)\n* read three with 8 cycles (bases) of sample barcode\n* 8 cycles (bases) skipped.\n* read four with 28 cycles (bases) of template\nThe skipped cycles would NOT be included in an output SAM/BAM file or in read groups therein.", shortName="RS")
    public String READ_STRUCTURE;
    @Argument(doc="Barcode sequence.  These must be unique, and all the same length.  This cannot be used with reads that have more than one barcode; use BARCODE_FILE in that case. ", mutex={"BARCODE_FILE"})
    public List<String> BARCODE = new ArrayList<String>();
    @Argument(doc="Tab-delimited file of barcode sequences, barcode name and, optionally, library name.  Barcodes must be unique and all the same length.  Column headers must be 'barcode_sequence' (or 'barcode_sequence_1'), 'barcode_sequence_2' (optional), 'barcode_name', and 'library_name'.", mutex={"BARCODE"})
    public File BARCODE_FILE;
    @Argument(doc="Per-barcode and per-lane metrics written to this file.", shortName="M")
    public File METRICS_FILE;
    @Argument(doc="Maximum mismatches for a barcode to be considered a match.")
    public int MAX_MISMATCHES = 1;
    @Argument(doc="Minimum difference between number of mismatches in the best and second best barcodes for a barcode to be considered a match.")
    public int MIN_MISMATCH_DELTA = 1;
    @Argument(doc="Maximum allowable number of no-calls in a barcode read before it is considered unmatchable.")
    public int MAX_NO_CALLS = 2;
    @Argument(shortName="Q", doc="Minimum base quality. Any barcode bases falling below this quality will be considered a mismatch even in the bases match.")
    public int MINIMUM_BASE_QUALITY = 0;
    @Argument(doc="The minimum quality (after transforming 0s to 1s) expected from reads.  If qualities are lower than this value, an error is thrown.The default of 2 is what the Illumina's spec describes as the minimum, but in practice the value has been observed lower.")
    public int MINIMUM_QUALITY = 2;
    @Argument(shortName="GZIP", doc="Compress output s_l_t_barcode.txt files using gzip and append a .gz extension to the file names.")
    public boolean COMPRESS_OUTPUTS = false;
    @Argument(doc="Run this many PerTileBarcodeExtractors in parallel.  If NUM_PROCESSORS = 0, number of cores is automatically set to the number of cores available on the machine. If NUM_PROCESSORS < 0 then the number of cores used will be the number available on the machine less NUM_PROCESSORS.")
    public int NUM_PROCESSORS = 1;
    private static final Log LOG = Log.getInstance(ExtractIlluminaBarcodes.class);
    private ReadStructure readStructure;
    private IlluminaDataProviderFactory factory;
    private final Map<String, BarcodeMetric> barcodeToMetrics = new LinkedHashMap<String, BarcodeMetric>();
    private final NumberFormat tileNumberFormatter = NumberFormat.getNumberInstance();
    private BclQualityEvaluationStrategy bclQualityEvaluationStrategy;

    public ExtractIlluminaBarcodes() {
        this.tileNumberFormatter.setMinimumIntegerDigits(4);
        this.tileNumberFormatter.setGroupingUsed(false);
    }

    @Override
    protected int doWork() {
        Object laneDir;
        IOUtil.assertFileIsWritable(this.METRICS_FILE);
        if (this.OUTPUT_DIR == null) {
            this.OUTPUT_DIR = this.BASECALLS_DIR;
        }
        IOUtil.assertDirectoryIsWritable(this.OUTPUT_DIR);
        String[] noMatchBarcode = new String[this.readStructure.sampleBarcodes.length()];
        int index = 0;
        for (ReadDescriptor d : this.readStructure.descriptors) {
            if (d.type != ReadType.Barcode) continue;
            noMatchBarcode[index++] = StringUtil.repeatCharNTimes('N', d.length);
        }
        BarcodeMetric noMatchMetric = new BarcodeMetric(null, null, IlluminaUtil.barcodeSeqsToString(noMatchBarcode), noMatchBarcode);
        int numProcessors = this.NUM_PROCESSORS == 0 ? Runtime.getRuntime().availableProcessors() : (this.NUM_PROCESSORS < 0 ? Runtime.getRuntime().availableProcessors() + this.NUM_PROCESSORS : this.NUM_PROCESSORS);
        LOG.info("Processing with " + numProcessors + " PerTileBarcodeExtractor(s).");
        ThreadPoolExecutorWithExceptions pool = new ThreadPoolExecutorWithExceptions(numProcessors);
        ArrayList<PerTileBarcodeExtractor> extractors = new ArrayList<PerTileBarcodeExtractor>(this.factory.getAvailableTiles().size());
        if (IlluminaFileUtil.hasCbcls(this.BASECALLS_DIR, this.LANE)) {
            laneDir = new File(this.BASECALLS_DIR, IlluminaFileUtil.longLaneStr(this.LANE));
            File[] fileArray = IOUtil.getFilesMatchingRegexp(laneDir, IlluminaFileUtil.CYCLE_SUBDIRECTORY_PATTERN);
            List<File> cbcls = Arrays.stream(fileArray).flatMap(cycleDir -> Arrays.stream(IOUtil.getFilesMatchingRegexp(cycleDir, "^" + IlluminaFileUtil.longLaneStr(this.LANE) + "_(\\d{1,5}).cbcl$"))).collect(Collectors.toList());
            if (cbcls.size() == 0) {
                throw new PicardException("No CBCL files found.");
            }
            IOUtil.assertFilesAreReadable(cbcls);
            ArrayList<AbstractIlluminaPositionFileReader.PositionInfo> locs = new ArrayList<AbstractIlluminaPositionFileReader.PositionInfo>();
            File locsFile = new File(this.BASECALLS_DIR.getParentFile(), "s.locs");
            IOUtil.assertFileIsReadable(locsFile);
            try (LocsFileReader locsFileReader = new LocsFileReader(locsFile);){
                while (locsFileReader.hasNext()) {
                    locs.add(locsFileReader.next());
                }
            }
            Pattern laneTileRegex = Pattern.compile(ParameterizedFileUtil.escapePeriods(ParameterizedFileUtil.makeLaneTileRegex(".filter", this.LANE)));
            File[] filterFiles = NewIlluminaBasecallsConverter.getTiledFiles(laneDir, laneTileRegex);
            IOUtil.assertFilesAreReadable(Arrays.asList(filterFiles));
            for (int tile : this.factory.getAvailableTiles()) {
                PerTileBarcodeExtractor extractor = new PerTileBarcodeExtractor(tile, this.getBarcodeFile(tile), this.barcodeToMetrics, noMatchMetric, this.factory, this.MINIMUM_BASE_QUALITY, this.MAX_NO_CALLS, this.MAX_MISMATCHES, this.MIN_MISMATCH_DELTA, cbcls, locs, filterFiles);
                extractors.add(extractor);
            }
        } else {
            laneDir = this.factory.getAvailableTiles().iterator();
            while (laneDir.hasNext()) {
                int n = laneDir.next();
                PerTileBarcodeExtractor extractor = new PerTileBarcodeExtractor(n, this.getBarcodeFile(n), this.barcodeToMetrics, noMatchMetric, this.factory, this.MINIMUM_BASE_QUALITY, this.MAX_NO_CALLS, this.MAX_MISMATCHES, this.MIN_MISMATCH_DELTA);
                extractors.add(extractor);
            }
        }
        for (PerTileBarcodeExtractor perTileBarcodeExtractor : extractors) {
            pool.submit(perTileBarcodeExtractor);
        }
        pool.shutdown();
        ThreadPoolExecutorUtil.awaitThreadPoolTermination("Per tile extractor executor", pool, Duration.ofMinutes(5L));
        LOG.info("Processed " + extractors.size() + " tiles.");
        for (PerTileBarcodeExtractor perTileBarcodeExtractor : extractors) {
            for (String key : this.barcodeToMetrics.keySet()) {
                this.barcodeToMetrics.get(key).merge(perTileBarcodeExtractor.getMetrics().get(key));
            }
            noMatchMetric.merge(perTileBarcodeExtractor.getNoMatchMetric());
            if (perTileBarcodeExtractor.getException() == null) continue;
            LOG.error("Abandoning metrics calculation because one or more PerTileBarcodeExtractors failed.");
            return 4;
        }
        ExtractIlluminaBarcodes.finalizeMetrics(this.barcodeToMetrics, noMatchMetric);
        for (Map.Entry entry : this.bclQualityEvaluationStrategy.getPoorQualityFrequencies().entrySet()) {
            LOG.warn(String.format("Observed low quality of %s %s times.", entry.getKey(), entry.getValue()));
        }
        this.bclQualityEvaluationStrategy.assertMinimumQualities();
        MetricsFile metrics = this.getMetricsFile();
        for (BarcodeMetric barcodeMetric : this.barcodeToMetrics.values()) {
            metrics.addMetric(barcodeMetric);
        }
        metrics.addMetric(noMatchMetric);
        metrics.write(this.METRICS_FILE);
        return 0;
    }

    public static void finalizeMetrics(Map<String, BarcodeMetric> barcodeToMetrics, BarcodeMetric noMatchMetric) {
        long totalReads = noMatchMetric.READS;
        long totalPfReads = noMatchMetric.PF_READS;
        long totalPfReadsAssigned = 0L;
        for (BarcodeMetric barcodeMetric : barcodeToMetrics.values()) {
            totalReads += barcodeMetric.READS;
            totalPfReads += barcodeMetric.PF_READS;
            totalPfReadsAssigned += barcodeMetric.PF_READS;
        }
        if (totalReads > 0L) {
            noMatchMetric.PCT_MATCHES = (double)noMatchMetric.READS / (double)totalReads;
            double bestPctOfAllBarcodeMatches = 0.0;
            for (BarcodeMetric barcodeMetric : barcodeToMetrics.values()) {
                barcodeMetric.PCT_MATCHES = (double)barcodeMetric.READS / (double)totalReads;
                if (!(barcodeMetric.PCT_MATCHES > bestPctOfAllBarcodeMatches)) continue;
                bestPctOfAllBarcodeMatches = barcodeMetric.PCT_MATCHES;
            }
            if (bestPctOfAllBarcodeMatches > 0.0) {
                noMatchMetric.RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = noMatchMetric.PCT_MATCHES / bestPctOfAllBarcodeMatches;
                for (BarcodeMetric barcodeMetric : barcodeToMetrics.values()) {
                    barcodeMetric.RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = barcodeMetric.PCT_MATCHES / bestPctOfAllBarcodeMatches;
                }
            }
        }
        if (totalPfReads > 0L) {
            noMatchMetric.PF_PCT_MATCHES = (double)noMatchMetric.PF_READS / (double)totalPfReads;
            double bestPctOfAllBarcodeMatches = 0.0;
            for (BarcodeMetric barcodeMetric : barcodeToMetrics.values()) {
                barcodeMetric.PF_PCT_MATCHES = (double)barcodeMetric.PF_READS / (double)totalPfReads;
                if (!(barcodeMetric.PF_PCT_MATCHES > bestPctOfAllBarcodeMatches)) continue;
                bestPctOfAllBarcodeMatches = barcodeMetric.PF_PCT_MATCHES;
            }
            if (bestPctOfAllBarcodeMatches > 0.0) {
                noMatchMetric.PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = noMatchMetric.PF_PCT_MATCHES / bestPctOfAllBarcodeMatches;
                for (BarcodeMetric barcodeMetric : barcodeToMetrics.values()) {
                    barcodeMetric.PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = barcodeMetric.PF_PCT_MATCHES / bestPctOfAllBarcodeMatches;
                }
            }
        }
        if (totalPfReadsAssigned > 0L) {
            double mean = (double)totalPfReadsAssigned / (double)barcodeToMetrics.values().size();
            for (BarcodeMetric m : barcodeToMetrics.values()) {
                m.PF_NORMALIZED_MATCHES = (double)m.PF_READS / mean;
            }
        }
    }

    private File getBarcodeFile(int tile) {
        return new File(this.OUTPUT_DIR, "s_" + this.LANE + "_" + this.tileNumberFormatter.format(tile) + "_barcode.txt" + (this.COMPRESS_OUTPUTS ? ".gz" : ""));
    }

    @Override
    protected String[] customCommandLineValidation() {
        IlluminaDataType[] illuminaDataTypeArray;
        ArrayList<String> messages = new ArrayList<String>();
        this.bclQualityEvaluationStrategy = new BclQualityEvaluationStrategy(this.MINIMUM_QUALITY);
        this.readStructure = new ReadStructure(this.READ_STRUCTURE.replaceAll("T|M", "S"));
        if (this.MINIMUM_BASE_QUALITY > 0) {
            IlluminaDataType[] illuminaDataTypeArray2 = new IlluminaDataType[3];
            illuminaDataTypeArray2[0] = IlluminaDataType.BaseCalls;
            illuminaDataTypeArray2[1] = IlluminaDataType.PF;
            illuminaDataTypeArray = illuminaDataTypeArray2;
            illuminaDataTypeArray2[2] = IlluminaDataType.QualityScores;
        } else {
            IlluminaDataType[] illuminaDataTypeArray3 = new IlluminaDataType[2];
            illuminaDataTypeArray3[0] = IlluminaDataType.BaseCalls;
            illuminaDataTypeArray = illuminaDataTypeArray3;
            illuminaDataTypeArray3[1] = IlluminaDataType.PF;
        }
        IlluminaDataType[] datatypes = illuminaDataTypeArray;
        this.factory = IlluminaFileUtil.hasCbcls(this.BASECALLS_DIR, this.LANE) ? new IlluminaDataProviderFactory(this.BASECALLS_DIR, this.OUTPUT_DIR, this.LANE, this.readStructure, this.bclQualityEvaluationStrategy) : new IlluminaDataProviderFactory(this.BASECALLS_DIR, this.LANE, this.readStructure, this.bclQualityEvaluationStrategy, datatypes);
        if (this.BARCODE_FILE != null) {
            this.parseBarcodeFile(messages);
        } else {
            HashSet<String> barcodes = new HashSet<String>();
            for (String barcode : this.BARCODE) {
                if (barcodes.contains(barcode)) {
                    messages.add("Barcode " + barcode + " specified more than once.");
                }
                barcodes.add(barcode);
                BarcodeMetric metric = new BarcodeMetric(null, null, barcode, new String[]{barcode});
                this.barcodeToMetrics.put(barcode, metric);
            }
        }
        if (this.barcodeToMetrics.keySet().isEmpty()) {
            messages.add("No barcodes have been specified.");
        }
        if (messages.isEmpty()) {
            return null;
        }
        return messages.toArray(new String[messages.size()]);
    }

    public static void main(String[] argv) {
        new ExtractIlluminaBarcodes().instanceMainWithExit(argv);
    }

    private void parseBarcodeFile(ArrayList<String> messages) {
        try (TabbedTextFileWithHeaderParser barcodesParser = new TabbedTextFileWithHeaderParser(this.BARCODE_FILE);){
            String sequenceColumn;
            String string = barcodesParser.hasColumn(BARCODE_SEQUENCE_COLUMN) ? BARCODE_SEQUENCE_COLUMN : (sequenceColumn = barcodesParser.hasColumn(BARCODE_SEQUENCE_1_COLUMN) ? BARCODE_SEQUENCE_1_COLUMN : null);
            if (sequenceColumn == null) {
                messages.add(this.BARCODE_FILE + " does not have " + BARCODE_SEQUENCE_COLUMN + " or " + BARCODE_SEQUENCE_1_COLUMN + " column header");
                return;
            }
            boolean hasBarcodeName = barcodesParser.hasColumn(BARCODE_NAME_COLUMN);
            boolean hasLibraryName = barcodesParser.hasColumn(LIBRARY_NAME_COLUMN);
            int numBarcodes = this.readStructure.sampleBarcodes.length();
            HashSet<String> barcodes = new HashSet<String>();
            for (TabbedTextFileWithHeaderParser.Row row : barcodesParser) {
                String[] bcStrings = new String[numBarcodes];
                int barcodeNum = 0;
                for (ReadDescriptor rd : this.readStructure.descriptors) {
                    if (rd.type != ReadType.Barcode) continue;
                    String header = barcodeNum == 0 ? sequenceColumn : "barcode_sequence_" + (1 + barcodeNum);
                    String field = row.getField(header);
                    if (field == null) {
                        messages.add(String.format("Null barcode in column %s of row: %s", header, row.getCurrentLine()));
                        bcStrings[barcodeNum] = "";
                    } else {
                        bcStrings[barcodeNum] = field;
                    }
                    ++barcodeNum;
                }
                String bcStr = IlluminaUtil.barcodeSeqsToString(bcStrings);
                if (barcodes.contains(bcStr)) {
                    messages.add("Barcode " + bcStr + " specified more than once in " + this.BARCODE_FILE);
                }
                barcodes.add(bcStr);
                String barcodeName = hasBarcodeName ? row.getField(BARCODE_NAME_COLUMN) : "";
                String libraryName = hasLibraryName ? row.getField(LIBRARY_NAME_COLUMN) : "";
                BarcodeMetric metric = new BarcodeMetric(barcodeName, libraryName, bcStr, bcStrings);
                this.barcodeToMetrics.put(StringUtil.join("", bcStrings), metric);
            }
        }
    }

    public static class PerTileBarcodeExtractor
    implements Runnable {
        private final int tile;
        private final File barcodeFile;
        private final Map<String, BarcodeMetric> metrics;
        private final BarcodeMetric noMatch;
        private Exception exception = null;
        private final boolean usingQualityScores;
        private BaseIlluminaDataProvider provider = null;
        private final ReadStructure outputReadStructure;
        private final int maxNoCalls;
        private final int maxMismatches;
        private final int minMismatchDelta;
        private final int minimumBaseQuality;
        private List<File> cbcls = null;
        private List<AbstractIlluminaPositionFileReader.PositionInfo> locs = null;
        private File[] filterFiles = null;
        private IlluminaDataProviderFactory factory = null;

        public PerTileBarcodeExtractor(int tile, File barcodeFile, Map<String, BarcodeMetric> barcodeToMetrics, BarcodeMetric noMatchMetric, IlluminaDataProviderFactory factory, int minimumBaseQuality, int maxNoCalls, int maxMismatches, int minMismatchDelta, List<File> cbcls, List<AbstractIlluminaPositionFileReader.PositionInfo> locs, File[] filterFiles) {
            this.tile = tile;
            this.barcodeFile = barcodeFile;
            this.usingQualityScores = minimumBaseQuality > 0;
            this.maxNoCalls = maxNoCalls;
            this.maxMismatches = maxMismatches;
            this.minMismatchDelta = minMismatchDelta;
            this.minimumBaseQuality = minimumBaseQuality;
            this.metrics = new LinkedHashMap<String, BarcodeMetric>(barcodeToMetrics.size());
            for (String key : barcodeToMetrics.keySet()) {
                this.metrics.put(key, BarcodeMetric.copy(barcodeToMetrics.get(key)));
            }
            this.noMatch = BarcodeMetric.copy(noMatchMetric);
            this.cbcls = cbcls;
            this.locs = locs;
            this.factory = factory;
            this.filterFiles = filterFiles;
            this.outputReadStructure = factory.getOutputReadStructure();
        }

        public PerTileBarcodeExtractor(int tile, File barcodeFile, Map<String, BarcodeMetric> barcodeToMetrics, BarcodeMetric noMatchMetric, IlluminaDataProviderFactory factory, int minimumBaseQuality, int maxNoCalls, int maxMismatches, int minMismatchDelta) {
            this.tile = tile;
            this.barcodeFile = barcodeFile;
            this.usingQualityScores = minimumBaseQuality > 0;
            this.maxNoCalls = maxNoCalls;
            this.maxMismatches = maxMismatches;
            this.minMismatchDelta = minMismatchDelta;
            this.minimumBaseQuality = minimumBaseQuality;
            this.metrics = new LinkedHashMap<String, BarcodeMetric>(barcodeToMetrics.size());
            for (String key : barcodeToMetrics.keySet()) {
                this.metrics.put(key, BarcodeMetric.copy(barcodeToMetrics.get(key)));
            }
            this.noMatch = BarcodeMetric.copy(noMatchMetric);
            this.provider = factory.makeDataProvider(Arrays.asList(tile));
            this.outputReadStructure = factory.getOutputReadStructure();
        }

        public synchronized Map<String, BarcodeMetric> getMetrics() {
            return this.metrics;
        }

        public synchronized BarcodeMetric getNoMatchMetric() {
            return this.noMatch;
        }

        public synchronized Exception getException() {
            return this.exception;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public synchronized void run() {
            try {
                Object qualityScores;
                if (this.provider == null) {
                    this.provider = this.factory.makeDataProvider(this.cbcls, this.locs, this.filterFiles, this.tile, null);
                }
                LOG.info("Extracting barcodes for tile " + this.tile);
                int[] barcodeIndices = this.outputReadStructure.sampleBarcodes.getIndices();
                BufferedWriter writer = IOUtil.openFileForBufferedWriting(this.barcodeFile);
                byte[][] barcodeSubsequences = new byte[barcodeIndices.length][];
                Object object = qualityScores = this.usingQualityScores ? (Object)new byte[barcodeIndices.length][] : (byte[][])null;
                while (this.provider.hasNext()) {
                    ClusterData cluster = (ClusterData)this.provider.next();
                    for (int i = 0; i < barcodeIndices.length; ++i) {
                        barcodeSubsequences[i] = cluster.getRead(barcodeIndices[i]).getBases();
                        if (!this.usingQualityScores) continue;
                        qualityScores[i] = cluster.getRead(barcodeIndices[i]).getQualities();
                    }
                    boolean passingFilter = cluster.isPf();
                    BarcodeMatch match = PerTileBarcodeExtractor.findBestBarcodeAndUpdateMetrics(barcodeSubsequences, (byte[][])qualityScores, passingFilter, this.metrics, this.noMatch, this.maxNoCalls, this.maxMismatches, this.minMismatchDelta, this.minimumBaseQuality);
                    String yOrN = match.matched ? "Y" : "N";
                    for (byte[] bc : barcodeSubsequences) {
                        writer.write(StringUtil.bytesToString(bc));
                    }
                    writer.write("\t" + yOrN + "\t" + match.barcode + "\t" + String.valueOf(match.mismatches) + "\t" + String.valueOf(match.mismatchesToSecondBest));
                    writer.newLine();
                }
                writer.close();
            }
            catch (Exception e) {
                LOG.error(e, "Error processing tile ", this.tile);
                this.exception = e;
            }
            finally {
                CloserUtil.close(this.provider);
                this.provider = null;
            }
        }

        public static BarcodeMatch findBestBarcodeAndUpdateMetrics(byte[][] readSubsequences, byte[][] qualityScores, boolean passingFilter, Map<String, BarcodeMetric> metrics, BarcodeMetric noMatchBarcodeMetric, int maxNoCalls, int maxMismatches, int minMismatchDelta, int minimumBaseQuality) {
            BarcodeMetric bestBarcodeMetric = null;
            int totalBarcodeReadBases = 0;
            int numNoCalls = 0;
            for (byte[] bc : readSubsequences) {
                totalBarcodeReadBases += bc.length;
                for (byte b : bc) {
                    if (!SequenceUtil.isNoCall(b)) continue;
                    ++numNoCalls;
                }
            }
            int numMismatchesInBestBarcode = totalBarcodeReadBases + 1;
            int numMismatchesInSecondBestBarcode = totalBarcodeReadBases + 1;
            for (BarcodeMetric barcodeMetric : metrics.values()) {
                int numMismatches = PerTileBarcodeExtractor.countMismatches(barcodeMetric.barcodeBytes, readSubsequences, qualityScores, minimumBaseQuality);
                if (numMismatches < numMismatchesInBestBarcode) {
                    if (bestBarcodeMetric != null) {
                        numMismatchesInSecondBestBarcode = numMismatchesInBestBarcode;
                    }
                    numMismatchesInBestBarcode = numMismatches;
                    bestBarcodeMetric = barcodeMetric;
                    continue;
                }
                if (numMismatches >= numMismatchesInSecondBestBarcode) continue;
                numMismatchesInSecondBestBarcode = numMismatches;
            }
            boolean matched = bestBarcodeMetric != null && numNoCalls <= maxNoCalls && numMismatchesInBestBarcode <= maxMismatches && numMismatchesInSecondBestBarcode - numMismatchesInBestBarcode >= minMismatchDelta;
            BarcodeMatch match = new BarcodeMatch();
            if (numNoCalls + numMismatchesInBestBarcode < totalBarcodeReadBases && bestBarcodeMetric != null) {
                match.mismatches = numMismatchesInBestBarcode;
                match.mismatchesToSecondBest = numMismatchesInSecondBestBarcode;
                match.barcode = bestBarcodeMetric.BARCODE_WITHOUT_DELIMITER.toLowerCase();
            } else {
                match.mismatches = totalBarcodeReadBases;
                match.barcode = "";
            }
            if (matched) {
                ++bestBarcodeMetric.READS;
                if (passingFilter) {
                    ++bestBarcodeMetric.PF_READS;
                }
                if (numMismatchesInBestBarcode == 0) {
                    ++bestBarcodeMetric.PERFECT_MATCHES;
                    if (passingFilter) {
                        ++bestBarcodeMetric.PF_PERFECT_MATCHES;
                    }
                } else if (numMismatchesInBestBarcode == 1) {
                    ++bestBarcodeMetric.ONE_MISMATCH_MATCHES;
                    if (passingFilter) {
                        ++bestBarcodeMetric.PF_ONE_MISMATCH_MATCHES;
                    }
                }
                match.matched = true;
                match.barcode = bestBarcodeMetric.BARCODE_WITHOUT_DELIMITER;
            } else {
                ++noMatchBarcodeMetric.READS;
                if (passingFilter) {
                    ++noMatchBarcodeMetric.PF_READS;
                }
            }
            return match;
        }

        private static int countMismatches(byte[][] barcodeBytes, byte[][] readSubsequence, byte[][] qualities, int minimumBaseQuality) {
            int numMismatches = 0;
            for (int j = 0; j < barcodeBytes.length; ++j) {
                for (int i = 0; i < barcodeBytes[j].length && readSubsequence[j].length > i; ++i) {
                    if (SequenceUtil.isNoCall(readSubsequence[j][i])) continue;
                    if (!SequenceUtil.basesEqual(barcodeBytes[j][i], readSubsequence[j][i])) {
                        ++numMismatches;
                        continue;
                    }
                    if (qualities == null || qualities[j][i] >= minimumBaseQuality) continue;
                    ++numMismatches;
                }
            }
            return numMismatches;
        }

        public static class BarcodeMatch {
            boolean matched;
            String barcode;
            int mismatches;
            int mismatchesToSecondBest;

            public boolean isMatched() {
                return this.matched;
            }

            public String getBarcode() {
                return this.barcode;
            }
        }
    }

    public static class BarcodeMetric
    extends MetricBase {
        public String BARCODE;
        public String BARCODE_WITHOUT_DELIMITER;
        public String BARCODE_NAME = "";
        public String LIBRARY_NAME = "";
        public long READS = 0L;
        public long PF_READS = 0L;
        public long PERFECT_MATCHES = 0L;
        public long PF_PERFECT_MATCHES = 0L;
        public long ONE_MISMATCH_MATCHES = 0L;
        public long PF_ONE_MISMATCH_MATCHES = 0L;
        public double PCT_MATCHES = 0.0;
        public double RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = 0.0;
        public double PF_PCT_MATCHES = 0.0;
        public double PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = 0.0;
        public double PF_NORMALIZED_MATCHES;
        protected byte[][] barcodeBytes;

        public BarcodeMetric(String barcodeName, String libraryName, String barcodeDisplay, String[] barcodeSeqs) {
            this.BARCODE = barcodeDisplay;
            this.BARCODE_WITHOUT_DELIMITER = barcodeDisplay.replaceAll("-", "");
            this.BARCODE_NAME = barcodeName;
            this.LIBRARY_NAME = libraryName;
            this.barcodeBytes = new byte[barcodeSeqs.length][];
            for (int i = 0; i < barcodeSeqs.length; ++i) {
                this.barcodeBytes[i] = StringUtil.stringToBytes(barcodeSeqs[i]);
            }
        }

        public BarcodeMetric() {
            this.barcodeBytes = null;
        }

        public static BarcodeMetric copy(BarcodeMetric metric) {
            BarcodeMetric result = new BarcodeMetric();
            result.BARCODE = metric.BARCODE;
            result.BARCODE_WITHOUT_DELIMITER = metric.BARCODE_WITHOUT_DELIMITER;
            result.BARCODE_NAME = metric.BARCODE_NAME;
            result.LIBRARY_NAME = metric.LIBRARY_NAME;
            result.barcodeBytes = metric.barcodeBytes;
            return result;
        }

        public void merge(BarcodeMetric metric) {
            this.READS += metric.READS;
            this.PF_READS += metric.PF_READS;
            this.PERFECT_MATCHES += metric.PERFECT_MATCHES;
            this.PF_PERFECT_MATCHES += metric.PF_PERFECT_MATCHES;
            this.ONE_MISMATCH_MATCHES += metric.ONE_MISMATCH_MATCHES;
            this.PF_ONE_MISMATCH_MATCHES += metric.PF_ONE_MISMATCH_MATCHES;
        }
    }
}

