# python3

'''
This is used for RIMSseq project.
Use to predict the %methylation of target regions based on the error rate. 
Consider the calibaration in the context dependent mannner.

Logic:
Count P100 based on the region-context of ultra stable region CpG sites.
Use P100 to predict context dependent expected methylation count.
DO not calibarate P0 based on context since it has little effect.
For each region, combine the total and expected methylation count to predict % methylation and standard deviation.

Usage:
$python PredictMethylation_Context.py --target Tar_NA12878_RIMseq_Rep1.CGI.CpG.RegionCount \
    --standard-0 Tar_NA12878_RIMseq_Rep1.candidateStableRegion.nonCpG.RegionCount \
    --standard-100 Tar_NA12878_RIMseq_Rep1.candidateStableRegion.CpG.RegionCount \
    --output Tar_NA12878_RIMseq_Rep1.CGI.CpG.RegionMethylationPrediction \
    --WGBSannotation ENCODE_human_WGBS_ultraStable_1kb_methylation.summary.txt \
    --stdev

--target: RegionCount file generated by CountErrorRegion.py --context, containing the counts and context for target regions
e.g.
chr1    ucsc    CpGisland       923591  926252  .       .       1-coordination  cpgNum=325;gcNum=1919;  count_total=35595;count_error=13;   AC

--standard-0: RegionCount file generated by CountErrorRegion.py --context, containing the counts for nonCpG sites in the ultraStable regions and context
e.g.
chr1    PMID_25493099   Ultra-stable    214384006       214385006       .       .       1-coordination  cg11156304;Methylated;  count_total=55033;count_error=2;    AC

--standard-100: RegionCount file generated by CountErrorRegion.py --context, containing the counts for CpG sites in the ultraStable regions

--WGBSannotation: a file containing all the human WGBS methylation information for ultraStable regions, not required
ENCODE_human_WGBS_ultraStable_1kb_methylation.summary.txt is provided in ~/annotation folder with the scripts.
This file is used to adjust the error rate for P100 methylation regions. 
If not provided: --WBSannotation empty, the methylation level is assumed to be 100% for all the regions listed in --starndard-100.

--cutoff: cutoff of total counts for methylation prediction, Default 3000
Combine the count_total from all the contexts of the same region. Only report regions having count_total>=cutoff in the output file. 

--stdev: Optional
Add this option to calculate and add the standard deviation in the output file.
By dafault use 95% confidence interval (alpha=0.05) for stdev calculation.
'''
try:
    import numpy as np
    import statistics
    import pandas as pd
    import argparse
    import re
    import time
    import os
    import datetime
    from subprocess import check_call
except:
    print('module error')
    quit()

__version__ = '2022.11.13' # version of the script

class ConvertFormat:
    '''
    convert the RegionCount file generated by CountErrorRegion.py --context to proper format for ErrorRate2Methylation
    '''
    def __init__(self, input_standard100, input_standard0, input_CGI, annotation):
        localtime = time.asctime(time.localtime())
        self.prefix = ''.join(localtime.split()[-2].split(':')) # '151542'
        
        self.output_standard100 = self.ConvertContainWGBS(input_standard100, self.ExtractUltraStableWGBS(annotation), 'input_standard100_{}'.format(self.prefix))
        self.output_standard0 = self.Convert(input_standard0, 'input_standard0_{}'.format(self.prefix))
        self.output_CGI = self.Convert(input_CGI, 'input_CGI_{}'.format(self.prefix))

    def extractTotal(self, word):
        return re.findall('count_total=(\d.*?);', word)[0]
    def extractError(self, word):
        return re.findall('count_error=(\d.*?);', word)[0]

    def Convert(self, url, output_file):
        '''
        extract information from url= a RegionCount file with context in the last col
        e.g. for CGI and 0% standard
        output_file e.g region, total, error, context
        '''
        output = open(output_file, 'w')
        with open(url) as f:
            for line in f:
                region = '-'.join([line.split('\t')[0],line.split('\t')[3], line.split('\t')[4]])
                total, error = self.extractTotal(line.strip().split('\t')[9]), self.extractError(line.strip().split('\t')[9])
                context = line.strip().split('\t')[10]
                print('\t'.join([region, total, error, context]), file=output, end='\n')
        output.close()
        return output_file

    def ExtractUltraStableWGBS(self, annotation):
        '''
        extract the WGBS methylation of each ultra stable region from the WGBS annotation file
        return a dict {ultra stable region: [(methylation, coverage), ...]} for sample having coverage>=50
        '''
        with open(annotation) as f:
            dic = {}
            for line in f:
                key = line.strip().split('\t')[0]
                if key in dic:
                    if int(line.strip().split('\t')[2]) >=50:
                        dic[key].append(line.strip().split('\t')[3]) # [methylation, ...]
                else:
                    dic[key] = [line.strip().split('\t')[3]]
        return dic # [region: methylation]

    def ConvertContainWGBS(self, url, dic_WGBS, output_file):
        '''
        extract information from url= 100% standard RegionCount file
        also extract the WGBS methylation for all the regions listed in the standard file
        output_file e.g region, total, error, context, All_WGBS_Methylation from human WGBS data
        Note: 
        All_WGBS_Methylation from human WGBS data is context independent, so different contexts of the same region have the same WGBS data
        '''
        # if WGBS methylation of ultrastable region is not availalbe, assume the methylation = 100%
        if not dic_WGBS:
            with open(url) as f:
                key = ['-'.join([line.split('\t')[0],line.split('\t')[3], line.split('\t')[4]]) for line in f] # key: chr-start-end
                dic_WGBS = dict.fromkeys(key, ['1'])

        output = open(output_file, 'w')
        with open(url) as f:
            for line in f:
                region = '-'.join([line.split('\t')[0],line.split('\t')[3], line.split('\t')[4]])
                total, error = self.extractTotal(line.strip().split('\t')[9]), self.extractError(line.strip().split('\t')[9])
                context = line.strip().split('\t')[10]
                WGBS = ','.join(dic_WGBS[region])
                print('\t'.join([region, total, error, context, WGBS]), file=output, end='\n')
        output.close()
        return output_file

class ErrorRate2Methylation:
    def __init__(self, input_standard100, input_standard0, target, output_file, cutoff_cov, stdev):
        '''
        Use to predict methylation based on error rate with context calibration

        input_standard100, e.g. region, total, error, context, All_WGBS_Methylation; using , as delimiter, no header
        input_standard0, e.g. region, total, error, context; no header
        target, e.g. region, total, error, context; no header

        stdev: True to output standard deviation
        cutoff_cov: output regions having total>=cutoff_cov
        '''
        self.P100 = self.Standard100(input_standard100) # a dictionary, {AC: error rate, GC: error rate..}
        self.P0 = self.Standard0(input_standard0) # float
        self.MethylationPredictionFile(target, output_file, cutoff_cov, stdev)

    def GetMean(self, line):
        '''get the mean of all the WGBS methylation seperated by comma'''
        if isinstance(line, float) or isinstance(line, int): # All_WGBS_Methylation = 1 if no WGBS annotation present
            return line
        else: 
            line = line.strip().split(',') # All_WGBS_Methylation is provided in WGBS annotation using ',' as delimiter
            return statistics.mean([float(item) for item in line])

    def Standard100(self, input_file):
        '''
        set up the 100% methylation standard
        logic:
            Use the mean of all the human WGBS samples for each ultraStable region-context to adjust the error
            P100 for each context = sum(adjusted error of each region-context)/sum(total of each region-context)
            For context NC, use all total and all error, in other words P100 without considering the context

        input_file is the candidateStableRegion.CpG.Regioncount
        e.g. region, total, error, context, All_WGBS_Methylation: methylation level for all the human WGBS samples, assumed to 1=100% if no WGBS annotation file used
        '''
        df = pd.read_csv(input_file, 
                        skiprows=0, sep='\t', 
                        names=['region', 'total', 'error', 'context', 'WGBS']) # usecols=[0, 1, 2, 3] to select cols
            
        df['WGBS_mean'] = df['WGBS'].apply(self.GetMean)
        df['adjustedError'] = df['error']/df['WGBS_mean'] # adjusted error if methylation is not 100%

        total = df['total'].sum()
        error = df['adjustedError'].sum()
        P100_NC = float(error/total)

        # count the P100 for each context based on the total counts and errors of this context
        dic_total = df.groupby('context')['total'].sum().to_dict()
        dic_error = df.groupby('context')['adjustedError'].sum().to_dict()
        dic_P100 = dict(zip(dic_total.keys(), [float(dic_error[key]/dic_total[key]) for key in dic_total]))
        dic_P100.update({'NC':P100_NC})
        print('The context dependent P100 total counts of CpG sites in the ultrastable regions is: {}'.format(dic_total))
        print('The context dependent P100 error counts of CpG sites in the ultrastable regions is: {}'.format(dic_error))
        print('The context dependent P100 error rate of CpG sites in the ultrastable regions is: {}'.format(dic_P100))
        return dic_P100
        # a dictionary {AC: error rate, TC: error rate, GC: error rate, CC: error rate, NC: error rate}, e.g. {'AC': 0.009364406779661017, 'CC': 0.013400160105809057, 'GC': 0.012662023924560995, 'TC': 0.013363983480090392}

    def Standard0(self, input_file):
        '''
        set up the 0% methylation standard
        logic:
            P0 = sum(error of each region)/sum(total of each region)
            Do not need to calibrate for context for Standard0 since its effects can be ignored

        input_file is the candidateStableRegion.nonCpG.Regioncount
        e.g. region, total, error, All_WGBS_Methylation: methylation level for all the human WGBS samples
        return P0
        '''
        df = pd.read_csv(input_file, 
                     skiprows=2, sep='\t',
                     names=['region', 'total', 'error', 'context'])
        total = df['total'].sum()
        error = df['error'].sum()
        P0 = float(error/total)
        print('The P0 error rate of nonCpG sites in the ultrastable regions is: {}'.format(P0))
        return P0

    def ExpectedMethCount(self, total, error, P0, P100):
        '''
        Predict the expected methylation count
        methylation%=methylation count/total
        Use region-context, error and P100_context for this calculation
        '''
        ExpectedMethUnbounded = float(error-total*P0)/(P100-P0)
        ExpectedMeth = min(max(ExpectedMethUnbounded, 0), total) # expected methylation count
        return ExpectedMeth

    def MethylationPredictionFile(self, input_file, output_file, cutoff_cov, stdev):
        '''
        logic: 
        count the expected methylation count based on the total and error for each region-context
        Combine all the methylation count and total of all four context to calculate the % methylation = sum(expected methylation count)/sum(total)
        For context NC, use average error rate of P100 and P0

        input_file e.g. region, total, error, context
        ignore regions having total<cutoff_cov

        calculate standard deviation if stdev=True 
        Use Stdev Error (normal) Col Q in 220318_RIMSeq_context-dependent_analysis.xlsx for calculation.
        '''
        df = pd.read_csv(input_file, 
                skiprows=2,  sep='\t', 
                names=['region', 'total', 'error', 'context'])

        df['P100_context'] = df['context'].apply(lambda x: self.P100[x]) # The P100 of each context
        df['ExpectedMeth'] = df.apply(lambda x: self.ExpectedMethCount(x.total, x.error, self.P0, x.P100_context), axis=1)
        
        df['P100_context*total'] =  df['P100_context'] * df['total'] # P100_context * region-context total
        
        # combine the different context for the same region
        df_combinecontext_temp = df.groupby('region').aggregate('sum').reset_index() # the context col will be ignored because it can not be added by sum function; reset_index() to add 'region' as a col
        print('ignore regions having total<{}'.format(cutoff_cov))
        df_combinecontext = df_combinecontext_temp.loc[(df_combinecontext_temp['total']>=cutoff_cov)].copy(deep=True)
        df_combinecontext['Methylation'] = df_combinecontext['ExpectedMeth']/df_combinecontext['total'] # % methylation = sum(expected methylation count)/sum(total), combining all the four context
        
        # calculate the stdev
        if stdev:
            print('Calculate and save the deviation in output.')
            df_combinecontext['EquivP100'] = df_combinecontext['P100_context*total']/df_combinecontext['total']
            df_combinecontext['EquivError'] = df_combinecontext['ExpectedMeth']*df_combinecontext['EquivP100']+(df_combinecontext['total']-df_combinecontext['ExpectedMeth'])*self.P0
            df_combinecontext['Proportion'] = df_combinecontext['EquivError']/df_combinecontext['total']
            df_combinecontext['stdevError'] = (np.sqrt(df_combinecontext['Proportion']*(1-df_combinecontext['Proportion'])/df_combinecontext['total']))*df_combinecontext['total']

            df_combinecontext['stdevMethylation'] = (df_combinecontext['stdevError']/df_combinecontext['EquivP100'])/df_combinecontext['total']  # standard deviation of %methylation
            ls_col = ['region', 'total', 'error', 'Methylation', 'stdevMethylation']
            # print(df_combinecontext.loc[(df_combinecontext['region']=='chr1-967274-.')])
        else:
            ls_col = ['region', 'total', 'error', 'Methylation']
        df_combinecontext.to_csv(output_file, index=False, header=True, sep='\t', columns = ls_col)
        # print(df_combinecontext.loc[(df_combinecontext['region']=='chr1-1044925-.')])

        return output_file

def GetFilePath(input_file):
    '''
    return the abspath of the input_file
    if use relative path and input_file is not in cwd, the abspath will be wrong, return False
    '''
    filepath = os.path.abspath(input_file)
    
    if os.path.exists(filepath):
        return filepath
    else:
        print('Can not find file: {}'.format(filepath))
        return None

def Arg():
    '''parser if run from command line'''
    parser = argparse.ArgumentParser()
    parser.add_argument('--target', help='gtf annotation file containing the target regions for error counting, e.g. CpG islands', dest='target')
    parser.add_argument('--standard-100', help='candidateStableRegion.CpG.Regioncount as 100% methylation standard', dest='input_standard100')
    parser.add_argument('--standard-0', help='candidateStableRegion.CpG.Regioncount as 0% methylation standard', dest='input_standard0')
    parser.add_argument('--output', help='output file', dest='output_file')
    parser.add_argument('--cutoff', help='cutoff of total counts for methylation prediction', dest='cutoff', type=int, default=3000)
    parser.add_argument('--WGBSannotation', help='human ultraStable region WGBS summary', dest='annotation', required=False)
    parser.add_argument('--stdev', help='add this option to report standard deviation in the output file', dest='stdev', action='store_true')

    args = parser.parse_args()
    return vars(args) # convert Namespace object into a dictionary

def main(dic_args):
    '''
    dic_args {'input_standard100': , 'input_standard0':, 'target':, 'output_file': , 'annotation':, 'cutoff':, 'stdev': True or False}
    '''
    dic_args['input_standard100'] = GetFilePath(dic_args['input_standard100'])
    dic_args['input_standard0'] = GetFilePath(dic_args['input_standard0'])
    dic_args['target'] = GetFilePath(dic_args['target'])
    dic_args['output_file'] = os.path.abspath(dic_args['output_file'])
    dic_args['annotation'] = GetFilePath(dic_args['annotation'])

    print(datetime.datetime.now())
    print("The verion of PredictMethylation_Context.py script is: {}".format(__version__))
    print("The parameters used are:")
    print("Input target region file: {}".format(dic_args['target']))
    print("Input standard0 file: {}".format(dic_args['input_standard0']))
    print("Input standard100 file: {}".format(dic_args['input_standard100']))
    print("WGBS annotation file: {}".format(dic_args['annotation']))
    print("region total cutoff: {}".format(dic_args['cutoff']))

    # generate a tmp dir to save all the temp files, delete when finished
    cwd = os.getcwd()
    localtime = time.asctime(time.localtime())
    prefix = ''.join(localtime.split()[-2].split(':')) # '151542'

    try:
        tmp = os.path.join(cwd, 'tmp{}'.format(prefix))
        print('Generate a tmp dir {}.'.format(tmp))
        os.mkdir(tmp)
        os.chdir(tmp)

        temp = ConvertFormat(dic_args['input_standard100'], dic_args['input_standard0'], dic_args['target'], dic_args['annotation'])
        ErrorRate2Methylation(temp.output_standard100, temp.output_standard0, temp.output_CGI, dic_args['output_file'], dic_args['cutoff'], dic_args['stdev'])
    except Exception as e: # print the error message
        print(e)
        print("Error! Methylation prediction failed.")
    finally: # clean up, delete the tmp dir when finished or failed
        os.chdir(cwd)
        check_call(['rm', '-r', '{}'.format(tmp)], shell=False)
        print('Done.')

##-------mainbody
if __name__ == '__main__':
    # run from command line
    main(Arg())