#include "Color.hpp"

#include <stdexcept>
#include <cmath>
#include <iostream>

using std::runtime_error;
using std::to_string;
using std::round;
using std::cerr;


#include <string>
#include <sstream>
#include <iomanip>

using std::string;


namespace gfase{

/// Convert integer value `val` to text in hexadecimal format.
/// The minimum width is padded with leading zeros; if not
/// specified, this `width` is derived from the type of the
/// argument. Function suitable from char to long long.
/// Pointers, floating point values, etc. are not supported;
/// passing them will result in an (intentional!) compiler error.
/// Basics from: http://stackoverflow.com/a/5100745/2932052
/// stolen from https://stackoverflow.com/a/48643043
template <typename T>
inline std::string int_to_hex(T val, size_t width)
{
    std::stringstream ss;
    ss << std::setfill('0') << std::setw(width) << std::hex << (val|0);
    return ss.str();
}


string rgb_to_hex(double r, double g, double b){
    string hex;

    hex += int_to_hex(int(r*255),2);
    hex += int_to_hex(int(g*255),2);
    hex += int_to_hex(int(b*255),2);

    return hex;
}



const vector <array <double,3> > MplGnuplot::rgbs = {
        {0.062622, 0.000000, 0.024637},
        {0.108465, 0.000002, 0.073853},
        {0.140028, 0.000008, 0.122888},
        {0.165683, 0.000021, 0.171626},
        {0.187867, 0.000044, 0.219946},
        {0.207695, 0.000080, 0.267733},
        {0.225788, 0.000132, 0.314870},
        {0.242536, 0.000204, 0.361242},
        {0.258199, 0.000296, 0.406737},
        {0.265684, 0.000352, 0.429121},
        {0.286972, 0.000559, 0.494656},
        {0.293725, 0.000642, 0.515918},
        {0.313112, 0.000942, 0.577774},
        {0.319313, 0.001060, 0.597707},
        {0.337232, 0.001471, 0.655284},
        {0.342997, 0.001628, 0.673696},
        {0.359738, 0.002167, 0.726434},
        {0.370479, 0.002586, 0.759405},
        {0.375735, 0.002814, 0.775204},
        {0.391077, 0.003577, 0.819740},
        {0.400979, 0.004157, 0.846958},
        {0.410643, 0.004795, 0.872120},
        {0.415390, 0.005137, 0.883910},
        {0.429318, 0.006261, 0.916034},
        {0.438357, 0.007095, 0.934680},
        {0.447214, 0.008000, 0.951057},
        {0.451577, 0.008480, 0.958381},
        {0.464420, 0.010034, 0.976848},
        {0.472789, 0.011169, 0.986201},
        {0.481012, 0.012386, 0.993159},
        {0.485071, 0.013027, 0.995734},
        {0.497050, 0.015080, 0.999829},
        {0.504878, 0.016562, 0.999526},
        {0.512587, 0.018139, 0.996795},
        {0.520181, 0.019812, 0.991645},
        {0.527666, 0.021585, 0.984086},
        {0.535046, 0.023461, 0.974139},
        {0.538699, 0.024439, 0.968276},
        {0.549510, 0.027533, 0.947177},
        {0.556600, 0.029735, 0.930229},
        {0.563602, 0.032050, 0.911023},
        {0.570517, 0.034484, 0.889604},
        {0.577350, 0.037037, 0.866025},
        {0.584103, 0.039713, 0.840344},
        {0.590779, 0.042516, 0.812622},
        {0.594089, 0.043965, 0.798017},
        {0.603909, 0.048510, 0.751332},
        {0.610368, 0.051707, 0.717912},
        {0.616759, 0.055042, 0.682749},
        {0.623085, 0.058517, 0.645928},
        {0.629348, 0.062136, 0.607539},
        {0.635548, 0.065901, 0.567675},
        {0.641689, 0.069815, 0.526432},
        {0.644737, 0.071829, 0.505325},
        {0.653797, 0.078101, 0.440216},
        {0.659768, 0.082480, 0.395451},
        {0.665686, 0.087019, 0.349727},
        {0.671551, 0.091722, 0.303153},
        {0.677365, 0.096591, 0.255843},
        {0.683130, 0.101630, 0.207912},
        {0.688847, 0.106840, 0.159476},
        {0.691687, 0.109511, 0.135105},
        {0.700140, 0.117790, 0.061561},
        {0.705719, 0.123535, 0.012320},
        {0.711254, 0.129464, 0.000000},
        {0.716746, 0.135579, 0.000000},
        {0.722197, 0.141884, 0.000000},
        {0.727607, 0.148382, 0.000000},
        {0.732977, 0.155075, 0.000000},
        {0.738308, 0.161966, 0.000000},
        {0.743600, 0.169058, 0.000000},
        {0.748855, 0.176355, 0.000000},
        {0.754074, 0.183858, 0.000000},
        {0.759257, 0.191572, 0.000000},
        {0.764404, 0.199498, 0.000000},
        {0.766965, 0.203542, 0.000000},
        {0.774597, 0.216000, 0.000000},
        {0.779643, 0.224582, 0.000000},
        {0.784657, 0.233388, 0.000000},
        {0.789639, 0.242421, 0.000000},
        {0.794590, 0.251685, 0.000000},
        {0.799510, 0.261181, 0.000000},
        {0.804400, 0.270914, 0.000000},
        {0.809260, 0.280885, 0.000000},
        {0.814092, 0.291098, 0.000000},
        {0.818895, 0.301556, 0.000000},
        {0.823669, 0.312261, 0.000000},
        {0.828417, 0.323217, 0.000000},
        {0.833137, 0.334425, 0.000000},
        {0.837831, 0.345890, 0.000000},
        {0.842499, 0.357615, 0.000000},
        {0.844823, 0.363575, 0.000000},
        {0.851757, 0.381852, 0.000000},
        {0.856349, 0.394370, 0.000000},
        {0.860916, 0.407160, 0.000000},
        {0.865459, 0.420223, 0.000000},
        {0.869979, 0.433562, 0.000000},
        {0.874475, 0.447181, 0.000000},
        {0.878948, 0.461082, 0.000000},
        {0.883398, 0.475268, 0.000000},
        {0.887826, 0.489742, 0.000000},
        {0.892232, 0.504507, 0.000000},
        {0.896617, 0.519566, 0.000000},
        {0.900980, 0.534922, 0.000000},
        {0.905322, 0.550577, 0.000000},
        {0.909643, 0.566535, 0.000000},
        {0.913944, 0.582798, 0.000000},
        {0.916087, 0.591045, 0.000000},
        {0.922486, 0.616252, 0.000000},
        {0.926727, 0.633449, 0.000000},
        {0.930949, 0.650963, 0.000000},
        {0.935152, 0.668797, 0.000000},
        {0.939336, 0.686953, 0.000000},
        {0.943502, 0.705435, 0.000000},
        {0.947649, 0.724246, 0.000000},
        {0.951779, 0.743388, 0.000000},
        {0.955890, 0.762864, 0.000000},
        {0.959984, 0.782678, 0.000000},
        {0.964060, 0.802832, 0.000000},
        {0.968119, 0.823329, 0.000000},
        {0.972162, 0.844171, 0.000000},
        {0.976187, 0.865363, 0.000000},
        {0.980196, 0.886906, 0.000000},
        {0.982194, 0.897811, 0.000000},
        {0.988165, 0.931060, 0.000000},
        {0.992126, 0.953675, 0.000000},
        {0.996071, 0.976655, 0.000000},
        {1.000000, 1.000000, 0.000000}
};


const vector <array <double,3> > MplRainbow::rgbs = {
        {0.492157, 0.012320, 0.999981},
        {0.476471, 0.036951, 0.999829},
        {0.460784, 0.061561, 0.999526},
        {0.445098, 0.086133, 0.999070},
        {0.429412, 0.110653, 0.998464},
        {0.413725, 0.135105, 0.997705},
        {0.398039, 0.159476, 0.996795},
        {0.382353, 0.183750, 0.995734},
        {0.366667, 0.207912, 0.994522},
        {0.358824, 0.219946, 0.993859},
        {0.335294, 0.255843, 0.991645},
        {0.327451, 0.267733, 0.990831},
        {0.303922, 0.303153, 0.988165},
        {0.296078, 0.314870, 0.987202},
        {0.272549, 0.349727, 0.984086},
        {0.264706, 0.361242, 0.982973},
        {0.241176, 0.395451, 0.979410},
        {0.225490, 0.417960, 0.976848},
        {0.217647, 0.429121, 0.975512},
        {0.194118, 0.462204, 0.971281},
        {0.178431, 0.483911, 0.968276},
        {0.162745, 0.505325, 0.965124},
        {0.154902, 0.515918, 0.963493},
        {0.131373, 0.547220, 0.958381},
        {0.115686, 0.567675, 0.954791},
        {0.100000, 0.587785, 0.951057},
        {0.092157, 0.597707, 0.949135},
        {0.068627, 0.626924, 0.943154},
        {0.052941, 0.645928, 0.938988},
        {0.037255, 0.664540, 0.934680},
        {0.029412, 0.673696, 0.932472},
        {0.005882, 0.700543, 0.925638},
        {0.009804, 0.717912, 0.920906},
        {0.025490, 0.734845, 0.916034},
        {0.041176, 0.751332, 0.911023},
        {0.056863, 0.767363, 0.905873},
        {0.072549, 0.782928, 0.900587},
        {0.080392, 0.790532, 0.897892},
        {0.103922, 0.812622, 0.889604},
        {0.119608, 0.826734, 0.883910},
        {0.135294, 0.840344, 0.878081},
        {0.150980, 0.853444, 0.872120},
        {0.166667, 0.866025, 0.866025},
        {0.182353, 0.878081, 0.859800},
        {0.198039, 0.889604, 0.853444},
        {0.205882, 0.895163, 0.850217},
        {0.229412, 0.911023, 0.840344},
        {0.245098, 0.920906, 0.833602},
        {0.260784, 0.930229, 0.826734},
        {0.276471, 0.938988, 0.819740},
        {0.292157, 0.947177, 0.812622},
        {0.307843, 0.954791, 0.805381},
        {0.323529, 0.961826, 0.798017},
        {0.331373, 0.965124, 0.794290},
        {0.354902, 0.974139, 0.782928},
        {0.370588, 0.979410, 0.775204},
        {0.386275, 0.984086, 0.767363},
        {0.401961, 0.988165, 0.759405},
        {0.417647, 0.991645, 0.751332},
        {0.433333, 0.994522, 0.743145},
        {0.449020, 0.996795, 0.734845},
        {0.456863, 0.997705, 0.730653},
        {0.480392, 0.999526, 0.717912},
        {0.496078, 0.999981, 0.709281},
        {0.511765, 0.999829, 0.700543},
        {0.527451, 0.999070, 0.691698},
        {0.543137, 0.997705, 0.682749},
        {0.558824, 0.995734, 0.673696},
        {0.574510, 0.993159, 0.664540},
        {0.590196, 0.989980, 0.655284},
        {0.605882, 0.986201, 0.645928},
        {0.621569, 0.981823, 0.636474},
        {0.637255, 0.976848, 0.626924},
        {0.652941, 0.971281, 0.617278},
        {0.668627, 0.965124, 0.607539},
        {0.676471, 0.961826, 0.602635},
        {0.700000, 0.951057, 0.587785},
        {0.715686, 0.943154, 0.577774},
        {0.731373, 0.934680, 0.567675},
        {0.747059, 0.925638, 0.557489},
        {0.762745, 0.916034, 0.547220},
        {0.778431, 0.905873, 0.536867},
        {0.794118, 0.895163, 0.526432},
        {0.809804, 0.883910, 0.515918},
        {0.825490, 0.872120, 0.505325},
        {0.841176, 0.859800, 0.494656},
        {0.856863, 0.846958, 0.483911},
        {0.872549, 0.833602, 0.473094},
        {0.888235, 0.819740, 0.462204},
        {0.903922, 0.805381, 0.451244},
        {0.919608, 0.790532, 0.440216},
        {0.927451, 0.782928, 0.434676},
        {0.950980, 0.759405, 0.417960},
        {0.966667, 0.743145, 0.406737},
        {0.982353, 0.726434, 0.395451},
        {0.998039, 0.709281, 0.384106},
        {1.000000, 0.691698, 0.372702},
        {1.000000, 0.673696, 0.361242},
        {1.000000, 0.655284, 0.349727},
        {1.000000, 0.636474, 0.338158},
        {1.000000, 0.617278, 0.326539},
        {1.000000, 0.597707, 0.314870},
        {1.000000, 0.577774, 0.303153},
        {1.000000, 0.557489, 0.291390},
        {1.000000, 0.536867, 0.279583},
        {1.000000, 0.515918, 0.267733},
        {1.000000, 0.494656, 0.255843},
        {1.000000, 0.483911, 0.249883},
        {1.000000, 0.451244, 0.231948},
        {1.000000, 0.429121, 0.219946},
        {1.000000, 0.406737, 0.207912},
        {1.000000, 0.384106, 0.195845},
        {1.000000, 0.361242, 0.183750},
        {1.000000, 0.338158, 0.171626},
        {1.000000, 0.314870, 0.159476},
        {1.000000, 0.291390, 0.147302},
        {1.000000, 0.267733, 0.135105},
        {1.000000, 0.243914, 0.122888},
        {1.000000, 0.219946, 0.110653},
        {1.000000, 0.195845, 0.098400},
        {1.000000, 0.171626, 0.086133},
        {1.000000, 0.147302, 0.073853},
        {1.000000, 0.122888, 0.061561},
        {1.000000, 0.110653, 0.055411},
        {1.000000, 0.073853, 0.036951},
        {1.000000, 0.049260, 0.024637},
        {1.000000, 0.024637, 0.012320},
        {1.000000, 0.000000, 0.000000},
};


const vector <array <double,3> > Viridis::rgbs = {
        {0.267004, 0.004874, 0.329415},
        {0.268510, 0.009605, 0.335427},
        {0.269944, 0.014625, 0.341379},
        {0.271305, 0.019942, 0.347269},
        {0.272594, 0.025563, 0.353093},
        {0.273809, 0.031497, 0.358853},
        {0.274952, 0.037752, 0.364543},
        {0.276022, 0.044167, 0.370164},
        {0.277018, 0.050344, 0.375715},
        {0.277941, 0.056324, 0.381191},
        {0.278791, 0.062145, 0.386592},
        {0.279566, 0.067836, 0.391917},
        {0.280267, 0.073417, 0.397163},
        {0.280894, 0.078907, 0.402329},
        {0.281446, 0.084320, 0.407414},
        {0.281924, 0.089666, 0.412415},
        {0.282327, 0.094955, 0.417331},
        {0.282656, 0.100196, 0.422160},
        {0.282910, 0.105393, 0.426902},
        {0.283091, 0.110553, 0.431554},
        {0.283197, 0.115680, 0.436115},
        {0.283229, 0.120777, 0.440584},
        {0.283187, 0.125848, 0.444960},
        {0.283072, 0.130895, 0.449241},
        {0.282884, 0.135920, 0.453427},
        {0.282623, 0.140926, 0.457517},
        {0.282290, 0.145912, 0.461510},
        {0.281887, 0.150881, 0.465405},
        {0.281412, 0.155834, 0.469201},
        {0.280868, 0.160771, 0.472899},
        {0.280255, 0.165693, 0.476498},
        {0.279574, 0.170599, 0.479997},
        {0.278826, 0.175490, 0.483397},
        {0.278012, 0.180367, 0.486697},
        {0.277134, 0.185228, 0.489898},
        {0.276194, 0.190074, 0.493001},
        {0.275191, 0.194905, 0.496005},
        {0.274128, 0.199721, 0.498911},
        {0.273006, 0.204520, 0.501721},
        {0.271828, 0.209303, 0.504434},
        {0.270595, 0.214069, 0.507052},
        {0.269308, 0.218818, 0.509577},
        {0.267968, 0.223549, 0.512008},
        {0.266580, 0.228262, 0.514349},
        {0.265145, 0.232956, 0.516599},
        {0.263663, 0.237631, 0.518762},
        {0.262138, 0.242286, 0.520837},
        {0.260571, 0.246922, 0.522828},
        {0.258965, 0.251537, 0.524736},
        {0.257322, 0.256130, 0.526563},
        {0.255645, 0.260703, 0.528312},
        {0.253935, 0.265254, 0.529983},
        {0.252194, 0.269783, 0.531579},
        {0.250425, 0.274290, 0.533103},
        {0.248629, 0.278775, 0.534556},
        {0.246811, 0.283237, 0.535941},
        {0.244972, 0.287675, 0.537260},
        {0.243113, 0.292092, 0.538516},
        {0.241237, 0.296485, 0.539709},
        {0.239346, 0.300855, 0.540844},
        {0.237441, 0.305202, 0.541921},
        {0.235526, 0.309527, 0.542944},
        {0.233603, 0.313828, 0.543914},
        {0.231674, 0.318106, 0.544834},
        {0.229739, 0.322361, 0.545706},
        {0.227802, 0.326594, 0.546532},
        {0.225863, 0.330805, 0.547314},
        {0.223925, 0.334994, 0.548053},
        {0.221989, 0.339161, 0.548752},
        {0.220057, 0.343307, 0.549413},
        {0.218130, 0.347432, 0.550038},
        {0.216210, 0.351535, 0.550627},
        {0.214298, 0.355619, 0.551184},
        {0.212395, 0.359683, 0.551710},
        {0.210503, 0.363727, 0.552206},
        {0.208623, 0.367752, 0.552675},
        {0.206756, 0.371758, 0.553117},
        {0.204903, 0.375746, 0.553533},
        {0.203063, 0.379716, 0.553925},
        {0.201239, 0.383670, 0.554294},
        {0.199430, 0.387607, 0.554642},
        {0.197636, 0.391528, 0.554969},
        {0.195860, 0.395433, 0.555276},
        {0.194100, 0.399323, 0.555565},
        {0.192357, 0.403199, 0.555836},
        {0.190631, 0.407061, 0.556089},
        {0.188923, 0.410910, 0.556326},
        {0.187231, 0.414746, 0.556547},
        {0.185556, 0.418570, 0.556753},
        {0.183898, 0.422383, 0.556944},
        {0.182256, 0.426184, 0.557120},
        {0.180629, 0.429975, 0.557282},
        {0.179019, 0.433756, 0.557430},
        {0.177423, 0.437527, 0.557565},
        {0.175841, 0.441290, 0.557685},
        {0.174274, 0.445044, 0.557792},
        {0.172719, 0.448791, 0.557885},
        {0.171176, 0.452530, 0.557965},
        {0.169646, 0.456262, 0.558030},
        {0.168126, 0.459988, 0.558082},
        {0.166617, 0.463708, 0.558119},
        {0.165117, 0.467423, 0.558141},
        {0.163625, 0.471133, 0.558148},
        {0.162142, 0.474838, 0.558140},
        {0.160665, 0.478540, 0.558115},
        {0.159194, 0.482237, 0.558073},
        {0.157729, 0.485932, 0.558013},
        {0.156270, 0.489624, 0.557936},
        {0.154815, 0.493313, 0.557840},
        {0.153364, 0.497000, 0.557724},
        {0.151918, 0.500685, 0.557587},
        {0.150476, 0.504369, 0.557430},
        {0.149039, 0.508051, 0.557250},
        {0.147607, 0.511733, 0.557049},
        {0.146180, 0.515413, 0.556823},
        {0.144759, 0.519093, 0.556572},
        {0.143343, 0.522773, 0.556295},
        {0.141935, 0.526453, 0.555991},
        {0.140536, 0.530132, 0.555659},
        {0.139147, 0.533812, 0.555298},
        {0.137770, 0.537492, 0.554906},
        {0.136408, 0.541173, 0.554483},
        {0.135066, 0.544853, 0.554029},
        {0.133743, 0.548535, 0.553541},
        {0.132444, 0.552216, 0.553018},
        {0.131172, 0.555899, 0.552459},
        {0.129933, 0.559582, 0.551864},
        {0.128729, 0.563265, 0.551229},
        {0.127568, 0.566949, 0.550556},
        {0.126453, 0.570633, 0.549841},
        {0.125394, 0.574318, 0.549086},
        {0.124395, 0.578002, 0.548287},
        {0.123463, 0.581687, 0.547445},
        {0.122606, 0.585371, 0.546557},
        {0.121831, 0.589055, 0.545623},
        {0.121148, 0.592739, 0.544641},
        {0.120565, 0.596422, 0.543611},
        {0.120092, 0.600104, 0.542530},
        {0.119738, 0.603785, 0.541400},
        {0.119512, 0.607464, 0.540218},
        {0.119423, 0.611141, 0.538982},
        {0.119483, 0.614817, 0.537692},
        {0.119699, 0.618490, 0.536347},
        {0.120081, 0.622161, 0.534946},
        {0.120638, 0.625828, 0.533488},
        {0.121380, 0.629492, 0.531973},
        {0.122312, 0.633153, 0.530398},
        {0.123444, 0.636809, 0.528763},
        {0.124780, 0.640461, 0.527068},
        {0.126326, 0.644107, 0.525311},
        {0.128087, 0.647749, 0.523491},
        {0.130067, 0.651384, 0.521608},
        {0.132268, 0.655014, 0.519661},
        {0.134692, 0.658636, 0.517649},
        {0.137339, 0.662252, 0.515571},
        {0.140210, 0.665859, 0.513427},
        {0.143303, 0.669459, 0.511215},
        {0.146616, 0.673050, 0.508936},
        {0.150148, 0.676631, 0.506589},
        {0.153894, 0.680203, 0.504172},
        {0.157851, 0.683765, 0.501686},
        {0.162016, 0.687316, 0.499129},
        {0.166383, 0.690856, 0.496502},
        {0.170948, 0.694384, 0.493803},
        {0.175707, 0.697900, 0.491033},
        {0.180653, 0.701402, 0.488189},
        {0.185783, 0.704891, 0.485273},
        {0.191090, 0.708366, 0.482284},
        {0.196571, 0.711827, 0.479221},
        {0.202219, 0.715272, 0.476084},
        {0.208030, 0.718701, 0.472873},
        {0.214000, 0.722114, 0.469588},
        {0.220124, 0.725509, 0.466226},
        {0.226397, 0.728888, 0.462789},
        {0.232815, 0.732247, 0.459277},
        {0.239374, 0.735588, 0.455688},
        {0.246070, 0.738910, 0.452024},
        {0.252899, 0.742211, 0.448284},
        {0.259857, 0.745492, 0.444467},
        {0.266941, 0.748751, 0.440573},
        {0.274149, 0.751988, 0.436601},
        {0.281477, 0.755203, 0.432552},
        {0.288921, 0.758394, 0.428426},
        {0.296479, 0.761561, 0.424223},
        {0.304148, 0.764704, 0.419943},
        {0.311925, 0.767822, 0.415586},
        {0.319809, 0.770914, 0.411152},
        {0.327796, 0.773980, 0.406640},
        {0.335885, 0.777018, 0.402049},
        {0.344074, 0.780029, 0.397381},
        {0.352360, 0.783011, 0.392636},
        {0.360741, 0.785964, 0.387814},
        {0.369214, 0.788888, 0.382914},
        {0.377779, 0.791781, 0.377939},
        {0.386433, 0.794644, 0.372886},
        {0.395174, 0.797475, 0.367757},
        {0.404001, 0.800275, 0.362552},
        {0.412913, 0.803041, 0.357269},
        {0.421908, 0.805774, 0.351910},
        {0.430983, 0.808473, 0.346476},
        {0.440137, 0.811138, 0.340967},
        {0.449368, 0.813768, 0.335384},
        {0.458674, 0.816363, 0.329727},
        {0.468053, 0.818921, 0.323998},
        {0.477504, 0.821444, 0.318195},
        {0.487026, 0.823929, 0.312321},
        {0.496615, 0.826376, 0.306377},
        {0.506271, 0.828786, 0.300362},
        {0.515992, 0.831158, 0.294279},
        {0.525776, 0.833491, 0.288127},
        {0.535621, 0.835785, 0.281908},
        {0.545524, 0.838039, 0.275626},
        {0.555484, 0.840254, 0.269281},
        {0.565498, 0.842430, 0.262877},
        {0.575563, 0.844566, 0.256415},
        {0.585678, 0.846661, 0.249897},
        {0.595839, 0.848717, 0.243329},
        {0.606045, 0.850733, 0.236712},
        {0.616293, 0.852709, 0.230052},
        {0.626579, 0.854645, 0.223353},
        {0.636902, 0.856542, 0.216620},
        {0.647257, 0.858400, 0.209861},
        {0.657642, 0.860219, 0.203082},
        {0.668054, 0.861999, 0.196293},
        {0.678489, 0.863742, 0.189503},
        {0.688944, 0.865448, 0.182725},
        {0.699415, 0.867117, 0.175971},
        {0.709898, 0.868751, 0.169257},
        {0.720391, 0.870350, 0.162603},
        {0.730889, 0.871916, 0.156029},
        {0.741388, 0.873449, 0.149561},
        {0.751884, 0.874951, 0.143228},
        {0.762373, 0.876424, 0.137064},
        {0.772852, 0.877868, 0.131109},
        {0.783315, 0.879285, 0.125405},
        {0.793760, 0.880678, 0.120005},
        {0.804182, 0.882046, 0.114965},
        {0.814576, 0.883393, 0.110347},
        {0.824940, 0.884720, 0.106217},
        {0.835270, 0.886029, 0.102646},
        {0.845561, 0.887322, 0.099702},
        {0.855810, 0.888601, 0.097452},
        {0.866013, 0.889868, 0.095953},
        {0.876168, 0.891125, 0.095250},
        {0.886271, 0.892374, 0.095374},
        {0.896320, 0.893616, 0.096335},
        {0.906311, 0.894855, 0.098125},
        {0.916242, 0.896091, 0.100717},
        {0.926106, 0.897330, 0.104071},
        {0.935904, 0.898570, 0.108131},
        {0.945636, 0.899815, 0.112838},
        {0.955300, 0.901065, 0.118128},
        {0.964894, 0.902323, 0.123941},
        {0.974417, 0.903590, 0.130215},
        {0.983868, 0.904867, 0.136897},
        {0.993248, 0.906157, 0.143936}
};


const vector <array <double,3> > Seismic::rgbs = {
        {0.000000,0.000000,0.300000},
        {0.000000,0.000000,0.310980},
        {0.000000,0.000000,0.321961},
        {0.000000,0.000000,0.332941},
        {0.000000,0.000000,0.343922},
        {0.000000,0.000000,0.354902},
        {0.000000,0.000000,0.365882},
        {0.000000,0.000000,0.376863},
        {0.000000,0.000000,0.387843},
        {0.000000,0.000000,0.398824},
        {0.000000,0.000000,0.409804},
        {0.000000,0.000000,0.420784},
        {0.000000,0.000000,0.431765},
        {0.000000,0.000000,0.442745},
        {0.000000,0.000000,0.453725},
        {0.000000,0.000000,0.464706},
        {0.000000,0.000000,0.475686},
        {0.000000,0.000000,0.486667},
        {0.000000,0.000000,0.497647},
        {0.000000,0.000000,0.508627},
        {0.000000,0.000000,0.519608},
        {0.000000,0.000000,0.530588},
        {0.000000,0.000000,0.541569},
        {0.000000,0.000000,0.552549},
        {0.000000,0.000000,0.563529},
        {0.000000,0.000000,0.574510},
        {0.000000,0.000000,0.585490},
        {0.000000,0.000000,0.596471},
        {0.000000,0.000000,0.607451},
        {0.000000,0.000000,0.618431},
        {0.000000,0.000000,0.629412},
        {0.000000,0.000000,0.640392},
        {0.000000,0.000000,0.651373},
        {0.000000,0.000000,0.662353},
        {0.000000,0.000000,0.673333},
        {0.000000,0.000000,0.684314},
        {0.000000,0.000000,0.695294},
        {0.000000,0.000000,0.706275},
        {0.000000,0.000000,0.717255},
        {0.000000,0.000000,0.728235},
        {0.000000,0.000000,0.739216},
        {0.000000,0.000000,0.750196},
        {0.000000,0.000000,0.761176},
        {0.000000,0.000000,0.772157},
        {0.000000,0.000000,0.783137},
        {0.000000,0.000000,0.794118},
        {0.000000,0.000000,0.805098},
        {0.000000,0.000000,0.816078},
        {0.000000,0.000000,0.827059},
        {0.000000,0.000000,0.838039},
        {0.000000,0.000000,0.849020},
        {0.000000,0.000000,0.860000},
        {0.000000,0.000000,0.870980},
        {0.000000,0.000000,0.881961},
        {0.000000,0.000000,0.892941},
        {0.000000,0.000000,0.903922},
        {0.000000,0.000000,0.914902},
        {0.000000,0.000000,0.925882},
        {0.000000,0.000000,0.936863},
        {0.000000,0.000000,0.947843},
        {0.000000,0.000000,0.958824},
        {0.000000,0.000000,0.969804},
        {0.000000,0.000000,0.980784},
        {0.000000,0.000000,0.991765},
        {0.003922,0.003922,1.000000},
        {0.019608,0.019608,1.000000},
        {0.035294,0.035294,1.000000},
        {0.050980,0.050980,1.000000},
        {0.066667,0.066667,1.000000},
        {0.082353,0.082353,1.000000},
        {0.098039,0.098039,1.000000},
        {0.113725,0.113725,1.000000},
        {0.129412,0.129412,1.000000},
        {0.145098,0.145098,1.000000},
        {0.160784,0.160784,1.000000},
        {0.176471,0.176471,1.000000},
        {0.192157,0.192157,1.000000},
        {0.207843,0.207843,1.000000},
        {0.223529,0.223529,1.000000},
        {0.239216,0.239216,1.000000},
        {0.254902,0.254902,1.000000},
        {0.270588,0.270588,1.000000},
        {0.286275,0.286275,1.000000},
        {0.301961,0.301961,1.000000},
        {0.317647,0.317647,1.000000},
        {0.333333,0.333333,1.000000},
        {0.349020,0.349020,1.000000},
        {0.364706,0.364706,1.000000},
        {0.380392,0.380392,1.000000},
        {0.396078,0.396078,1.000000},
        {0.411765,0.411765,1.000000},
        {0.427451,0.427451,1.000000},
        {0.443137,0.443137,1.000000},
        {0.458824,0.458824,1.000000},
        {0.474510,0.474510,1.000000},
        {0.490196,0.490196,1.000000},
        {0.505882,0.505882,1.000000},
        {0.521569,0.521569,1.000000},
        {0.537255,0.537255,1.000000},
        {0.552941,0.552941,1.000000},
        {0.568627,0.568627,1.000000},
        {0.584314,0.584314,1.000000},
        {0.600000,0.600000,1.000000},
        {0.615686,0.615686,1.000000},
        {0.631373,0.631373,1.000000},
        {0.647059,0.647059,1.000000},
        {0.662745,0.662745,1.000000},
        {0.678431,0.678431,1.000000},
        {0.694118,0.694118,1.000000},
        {0.709804,0.709804,1.000000},
        {0.725490,0.725490,1.000000},
        {0.741176,0.741176,1.000000},
        {0.756863,0.756863,1.000000},
        {0.772549,0.772549,1.000000},
        {0.788235,0.788235,1.000000},
        {0.803922,0.803922,1.000000},
        {0.819608,0.819608,1.000000},
        {0.835294,0.835294,1.000000},
        {0.850980,0.850980,1.000000},
        {0.866667,0.866667,1.000000},
        {0.882353,0.882353,1.000000},
        {0.898039,0.898039,1.000000},
        {0.913725,0.913725,1.000000},
        {0.929412,0.929412,1.000000},
        {0.945098,0.945098,1.000000},
        {0.960784,0.960784,1.000000},
        {0.960784,0.960784,1.000000},
        {0.960784,0.960784,1.000000},
        {1.000000,0.960784,0.960784},
        {1.000000,0.960784,0.960784},
        {1.000000,0.960784,0.960784},
        {1.000000,0.945098,0.945098},
        {1.000000,0.929412,0.929412},
        {1.000000,0.913725,0.913725},
        {1.000000,0.898039,0.898039},
        {1.000000,0.882353,0.882353},
        {1.000000,0.866667,0.866667},
        {1.000000,0.850980,0.850980},
        {1.000000,0.835294,0.835294},
        {1.000000,0.819608,0.819608},
        {1.000000,0.803922,0.803922},
        {1.000000,0.788235,0.788235},
        {1.000000,0.772549,0.772549},
        {1.000000,0.756863,0.756863},
        {1.000000,0.741176,0.741176},
        {1.000000,0.725490,0.725490},
        {1.000000,0.709804,0.709804},
        {1.000000,0.694118,0.694118},
        {1.000000,0.678431,0.678431},
        {1.000000,0.662745,0.662745},
        {1.000000,0.647059,0.647059},
        {1.000000,0.631373,0.631373},
        {1.000000,0.615686,0.615686},
        {1.000000,0.600000,0.600000},
        {1.000000,0.584314,0.584314},
        {1.000000,0.568627,0.568627},
        {1.000000,0.552941,0.552941},
        {1.000000,0.537255,0.537255},
        {1.000000,0.521569,0.521569},
        {1.000000,0.505882,0.505882},
        {1.000000,0.490196,0.490196},
        {1.000000,0.474510,0.474510},
        {1.000000,0.458824,0.458824},
        {1.000000,0.443137,0.443137},
        {1.000000,0.427451,0.427451},
        {1.000000,0.411765,0.411765},
        {1.000000,0.396078,0.396078},
        {1.000000,0.380392,0.380392},
        {1.000000,0.364706,0.364706},
        {1.000000,0.349020,0.349020},
        {1.000000,0.333333,0.333333},
        {1.000000,0.317647,0.317647},
        {1.000000,0.301961,0.301961},
        {1.000000,0.286275,0.286275},
        {1.000000,0.270588,0.270588},
        {1.000000,0.254902,0.254902},
        {1.000000,0.239216,0.239216},
        {1.000000,0.223529,0.223529},
        {1.000000,0.207843,0.207843},
        {1.000000,0.192157,0.192157},
        {1.000000,0.176471,0.176471},
        {1.000000,0.160784,0.160784},
        {1.000000,0.145098,0.145098},
        {1.000000,0.129412,0.129412},
        {1.000000,0.113725,0.113725},
        {1.000000,0.098039,0.098039},
        {1.000000,0.082353,0.082353},
        {1.000000,0.066667,0.066667},
        {1.000000,0.050980,0.050980},
        {1.000000,0.035294,0.035294},
        {1.000000,0.019608,0.019608},
        {1.000000,0.003922,0.003922},
        {0.994118,0.000000,0.000000},
        {0.986275,0.000000,0.000000},
        {0.978431,0.000000,0.000000},
        {0.970588,0.000000,0.000000},
        {0.962745,0.000000,0.000000},
        {0.954902,0.000000,0.000000},
        {0.947059,0.000000,0.000000},
        {0.939216,0.000000,0.000000},
        {0.931373,0.000000,0.000000},
        {0.923529,0.000000,0.000000},
        {0.915686,0.000000,0.000000},
        {0.907843,0.000000,0.000000},
        {0.900000,0.000000,0.000000},
        {0.892157,0.000000,0.000000},
        {0.884314,0.000000,0.000000},
        {0.876471,0.000000,0.000000},
        {0.868627,0.000000,0.000000},
        {0.860784,0.000000,0.000000},
        {0.852941,0.000000,0.000000},
        {0.845098,0.000000,0.000000},
        {0.837255,0.000000,0.000000},
        {0.829412,0.000000,0.000000},
        {0.821569,0.000000,0.000000},
        {0.813725,0.000000,0.000000},
        {0.805882,0.000000,0.000000},
        {0.798039,0.000000,0.000000},
        {0.790196,0.000000,0.000000},
        {0.782353,0.000000,0.000000},
        {0.774510,0.000000,0.000000},
        {0.766667,0.000000,0.000000},
        {0.758824,0.000000,0.000000},
        {0.750980,0.000000,0.000000},
        {0.743137,0.000000,0.000000},
        {0.735294,0.000000,0.000000},
        {0.727451,0.000000,0.000000},
        {0.719608,0.000000,0.000000},
        {0.711765,0.000000,0.000000},
        {0.703922,0.000000,0.000000},
        {0.696078,0.000000,0.000000},
        {0.688235,0.000000,0.000000},
        {0.680392,0.000000,0.000000},
        {0.672549,0.000000,0.000000},
        {0.664706,0.000000,0.000000},
        {0.656863,0.000000,0.000000},
        {0.649020,0.000000,0.000000},
        {0.641176,0.000000,0.000000},
        {0.633333,0.000000,0.000000},
        {0.625490,0.000000,0.000000},
        {0.617647,0.000000,0.000000},
        {0.609804,0.000000,0.000000},
        {0.601961,0.000000,0.000000},
        {0.594118,0.000000,0.000000},
        {0.586275,0.000000,0.000000},
        {0.578431,0.000000,0.000000},
        {0.570588,0.000000,0.000000},
        {0.562745,0.000000,0.000000},
        {0.554902,0.000000,0.000000},
        {0.547059,0.000000,0.000000},
        {0.539216,0.000000,0.000000},
        {0.531373,0.000000,0.000000},
        {0.523529,0.000000,0.000000},
        {0.515686,0.000000,0.000000},
        {0.507843,0.000000,0.000000},
        {0.500000,0.000000,0.000000}
};



const vector <array <double,3> > Coolwarm::rgbs = {
        {0.229806,0.298718,0.753683},
        {0.234377,0.305542,0.759680},
        {0.238948,0.312365,0.765676},
        {0.243520,0.319189,0.771672},
        {0.248091,0.326013,0.777669},
        {0.252663,0.332837,0.783665},
        {0.257234,0.339661,0.789661},
        {0.261805,0.346484,0.795658},
        {0.266381,0.353304,0.801637},
        {0.271104,0.360011,0.807095},
        {0.275827,0.366717,0.812553},
        {0.280550,0.373423,0.818011},
        {0.285273,0.380129,0.823469},
        {0.289996,0.386836,0.828926},
        {0.294718,0.393542,0.834384},
        {0.299441,0.400248,0.839842},
        {0.304174,0.406945,0.845263},
        {0.309060,0.413498,0.850128},
        {0.313946,0.420052,0.854993},
        {0.318832,0.426605,0.859857},
        {0.323718,0.433158,0.864722},
        {0.328604,0.439712,0.869587},
        {0.333490,0.446265,0.874452},
        {0.338377,0.452819,0.879317},
        {0.343278,0.459354,0.884122},
        {0.348323,0.465711,0.888346},
        {0.353369,0.472069,0.892570},
        {0.358415,0.478426,0.896795},
        {0.363461,0.484784,0.901019},
        {0.368507,0.491141,0.905243},
        {0.373552,0.497499,0.909467},
        {0.378598,0.503856,0.913692},
        {0.383662,0.510183,0.917831},
        {0.388852,0.516298,0.921373},
        {0.394042,0.522413,0.924916},
        {0.399231,0.528528,0.928459},
        {0.404421,0.534643,0.932002},
        {0.409611,0.540759,0.935545},
        {0.414801,0.546874,0.939088},
        {0.419991,0.552989,0.942630},
        {0.425199,0.559058,0.946061},
        {0.430507,0.564883,0.948889},
        {0.435815,0.570707,0.951717},
        {0.441123,0.576532,0.954545},
        {0.446431,0.582356,0.957373},
        {0.451739,0.588181,0.960201},
        {0.457046,0.594006,0.963029},
        {0.462354,0.599830,0.965857},
        {0.467678,0.605591,0.968546},
        {0.473070,0.611077,0.970634},
        {0.478462,0.616564,0.972721},
        {0.483854,0.622050,0.974808},
        {0.489246,0.627536,0.976896},
        {0.494638,0.633022,0.978983},
        {0.500031,0.638508,0.981070},
        {0.505423,0.643995,0.983157},
        {0.510824,0.649397,0.985079},
        {0.516260,0.654498,0.986407},
        {0.521696,0.659599,0.987736},
        {0.527132,0.664700,0.989065},
        {0.532568,0.669801,0.990393},
        {0.538004,0.674902,0.991722},
        {0.543440,0.680003,0.993051},
        {0.548876,0.685104,0.994379},
        {0.554312,0.690097,0.995516},
        {0.559747,0.694768,0.996075},
        {0.565182,0.699438,0.996635},
        {0.570616,0.704109,0.997195},
        {0.576051,0.708780,0.997755},
        {0.581486,0.713451,0.998314},
        {0.586921,0.718121,0.998874},
        {0.592356,0.722792,0.999434},
        {0.597777,0.727330,0.999777},
        {0.603162,0.731527,0.999565},
        {0.608547,0.735725,0.999354},
        {0.613933,0.739923,0.999142},
        {0.619318,0.744121,0.998931},
        {0.624703,0.748318,0.998719},
        {0.630089,0.752516,0.998508},
        {0.635474,0.756714,0.998297},
        {0.640828,0.760752,0.997846},
        {0.646113,0.764436,0.996868},
        {0.651398,0.768121,0.995891},
        {0.656683,0.771806,0.994914},
        {0.661968,0.775491,0.993937},
        {0.667253,0.779176,0.992959},
        {0.672538,0.782861,0.991982},
        {0.677823,0.786546,0.991005},
        {0.683056,0.790043,0.989768},
        {0.688188,0.793178,0.988038},
        {0.693321,0.796314,0.986308},
        {0.698454,0.799450,0.984577},
        {0.703587,0.802586,0.982847},
        {0.708720,0.805721,0.981117},
        {0.713852,0.808857,0.979386},
        {0.718985,0.811993,0.977656},
        {0.724041,0.814910,0.975651},
        {0.728970,0.817464,0.973188},
        {0.733898,0.820018,0.970724},
        {0.738826,0.822572,0.968261},
        {0.743754,0.825125,0.965798},
        {0.748682,0.827679,0.963334},
        {0.753611,0.830233,0.960871},
        {0.758539,0.832787,0.958408},
        {0.763363,0.835092,0.955658},
        {0.768034,0.837035,0.952488},
        {0.772706,0.838978,0.949319},
        {0.777378,0.840921,0.946149},
        {0.782049,0.842864,0.942980},
        {0.786721,0.844807,0.939810},
        {0.791392,0.846750,0.936641},
        {0.796064,0.848693,0.933471},
        {0.800601,0.850358,0.930008},
        {0.804965,0.851666,0.926165},
        {0.809329,0.852974,0.922323},
        {0.813693,0.854282,0.918480},
        {0.818056,0.855590,0.914638},
        {0.822420,0.856898,0.910795},
        {0.826784,0.858205,0.906953},
        {0.831148,0.859513,0.903110},
        {0.835345,0.860514,0.898970},
        {0.839351,0.861167,0.894494},
        {0.843358,0.861820,0.890017},
        {0.847365,0.862472,0.885540},
        {0.851372,0.863125,0.881064},
        {0.855378,0.863778,0.876587},
        {0.859385,0.864431,0.872111},
        {0.863392,0.865084,0.867634},
        {0.867428,0.864377,0.862602},
        {0.871493,0.862309,0.857016},
        {0.875557,0.860242,0.851430},
        {0.879622,0.858175,0.845844},
        {0.883687,0.856108,0.840258},
        {0.887752,0.854040,0.834671},
        {0.891817,0.851973,0.829085},
        {0.895882,0.849906,0.823499},
        {0.899543,0.847500,0.817789},
        {0.902849,0.844796,0.811970},
        {0.906154,0.842091,0.806151},
        {0.909460,0.839386,0.800331},
        {0.912765,0.836682,0.794512},
        {0.916071,0.833977,0.788693},
        {0.919376,0.831273,0.782874},
        {0.922681,0.828568,0.777054},
        {0.925563,0.825517,0.771136},
        {0.928116,0.822197,0.765141},
        {0.930669,0.818877,0.759146},
        {0.933221,0.815557,0.753151},
        {0.935774,0.812237,0.747156},
        {0.938326,0.808917,0.741162},
        {0.940879,0.805596,0.735167},
        {0.943432,0.802276,0.729172},
        {0.945540,0.798606,0.723105},
        {0.947345,0.794696,0.716991},
        {0.949151,0.790785,0.710876},
        {0.950956,0.786875,0.704761},
        {0.952761,0.782965,0.698646},
        {0.954566,0.779055,0.692531},
        {0.956371,0.775144,0.686416},
        {0.958176,0.771234,0.680301},
        {0.959518,0.766973,0.674145},
        {0.960581,0.762501,0.667964},
        {0.961645,0.758029,0.661782},
        {0.962708,0.753557,0.655601},
        {0.963772,0.749086,0.649420},
        {0.964835,0.744614,0.643239},
        {0.965899,0.740142,0.637058},
        {0.966962,0.735670,0.630877},
        {0.967544,0.730850,0.624685},
        {0.967874,0.725847,0.618489},
        {0.968203,0.720844,0.612293},
        {0.968533,0.715841,0.606097},
        {0.968863,0.710838,0.599901},
        {0.969192,0.705836,0.593704},
        {0.969522,0.700833,0.587508},
        {0.969851,0.695830,0.581312},
        {0.969683,0.690484,0.575138},
        {0.969289,0.684982,0.568975},
        {0.968894,0.679480,0.562812},
        {0.968500,0.673977,0.556649},
        {0.968105,0.668475,0.550486},
        {0.967711,0.662973,0.544323},
        {0.967317,0.657471,0.538160},
        {0.966922,0.651969,0.531997},
        {0.966017,0.646130,0.525890},
        {0.964911,0.640159,0.519806},
        {0.963806,0.634188,0.513721},
        {0.962701,0.628218,0.507636},
        {0.961595,0.622247,0.501551},
        {0.960490,0.616276,0.495467},
        {0.959385,0.610306,0.489382},
        {0.958279,0.604335,0.483297},
        {0.956653,0.598034,0.477302},
        {0.954853,0.591622,0.471337},
        {0.953054,0.585211,0.465373},
        {0.951254,0.578799,0.459408},
        {0.949454,0.572388,0.453443},
        {0.947654,0.565976,0.447478},
        {0.945854,0.559565,0.441513},
        {0.944055,0.553153,0.435548},
        {0.941728,0.546413,0.429707},
        {0.939254,0.539581,0.423900},
        {0.936780,0.532750,0.418093},
        {0.934305,0.525918,0.412286},
        {0.931831,0.519086,0.406480},
        {0.929357,0.512254,0.400673},
        {0.926883,0.505422,0.394866},
        {0.924409,0.498590,0.389059},
        {0.921406,0.491420,0.383408},
        {0.918282,0.484173,0.377794},
        {0.915157,0.476927,0.372179},
        {0.912033,0.469680,0.366565},
        {0.908908,0.462433,0.360950},
        {0.905783,0.455186,0.355336},
        {0.902659,0.447939,0.349721},
        {0.899534,0.440692,0.344107},
        {0.895885,0.433075,0.338681},
        {0.892138,0.425389,0.333289},
        {0.888390,0.417703,0.327898},
        {0.884643,0.410017,0.322507},
        {0.880896,0.402331,0.317115},
        {0.877149,0.394645,0.311724},
        {0.873402,0.386960,0.306332},
        {0.869655,0.379274,0.300941},
        {0.865391,0.371128,0.295769},
        {0.861054,0.362916,0.290628},
        {0.856716,0.354704,0.285487},
        {0.852378,0.346492,0.280346},
        {0.848040,0.338280,0.275206},
        {0.843703,0.330068,0.270065},
        {0.839365,0.321856,0.264924},
        {0.835027,0.313644,0.259783},
        {0.830187,0.304733,0.254891},
        {0.825294,0.295749,0.250025},
        {0.820401,0.286765,0.245160},
        {0.815508,0.277781,0.240294},
        {0.810616,0.268797,0.235428},
        {0.805723,0.259813,0.230562},
        {0.800830,0.250829,0.225696},
        {0.795938,0.241845,0.220830},
        {0.790562,0.231397,0.216242},
        {0.785153,0.220851,0.211673},
        {0.779745,0.210305,0.207104},
        {0.774337,0.199759,0.202535},
        {0.768929,0.189213,0.197965},
        {0.763520,0.178667,0.193396},
        {0.758112,0.168122,0.188827},
        {0.752704,0.157576,0.184258},
        {0.746838,0.140021,0.179996},
        {0.740957,0.122240,0.175744},
        {0.735077,0.104460,0.171492},
        {0.729196,0.086679,0.167240},
        {0.723315,0.068898,0.162989},
        {0.717435,0.051118,0.158737},
        {0.711554,0.033337,0.154485},
        {0.705673,0.015556,0.150233}
};


array <double,3> Viridis::get_rgb(double x){
    if ((x < 0) or (x > 1)){
        throw runtime_error("ERROR: colormap index must be between 0 and 1");
    }

    size_t index = round(x * double(rgbs.size() - 1));

    return rgbs[index];
}


array <double,3> MplRainbow::get_rgb(double x){
    if ((x < 0) or (x > 1)){
        throw runtime_error("ERROR: colormap index must be between 0 and 1");
    }

    size_t index = round(x * double(rgbs.size() - 1));

    return rgbs[index];
}


array <double,3> MplGnuplot::get_rgb(double x){
    if ((x < 0) or (x > 1)){
        throw runtime_error("ERROR: colormap index must be between 0 and 1");
    }

    size_t index = round(x * double(rgbs.size() - 1));

    return rgbs[index];
}


array <double,3> Seismic::get_rgb(double x){
    if ((x < 0) or (x > 1)){
        throw runtime_error("ERROR: colormap index must be between 0 and 1");
    }

    size_t index = round(x * double(rgbs.size() - 1));

    return rgbs[index];
}


array <double,3> Coolwarm::get_rgb(double x){
    if ((x < 0) or (x > 1)){
        throw runtime_error("ERROR: colormap index must be between 0 and 1");
    }

    size_t index = round(x * double(rgbs.size() - 1));

    return rgbs[index];
}


string Viridis::get_svg_color(double x){
    auto rgb = get_rgb(x);
    string r = to_string(uint16_t(round(rgb[0]*255)));
    string g = to_string(uint16_t(round(rgb[1]*255)));
    string b = to_string(uint16_t(round(rgb[2]*255)));
    string s = "rgb(" + r + ',' + g + ',' + b + ")";
    return s;
}

}
