####################################
## Random Forest for Lymphocytes ##
###################################

# Modified version of Emmanuel Labaronne initial script, allowing seamless
# saving as .csv files of the training and validation sets after application of 
# the best model and this model as a .rda file.
# Paths have been also modified to facilitate the deployment on a new system.

# Configuration -------------------
pks <- list('ggplot2' , 'tidyverse' , 'randomForest', 'caret', 'doParallel', 'gridExtra', 'grid', 'SHAPforxgboost')
lapply(pks , library , character.only = T, quietly = FALSE)
theme_set(theme_bw())

#color point
color_resting = "#69b3a2"
color_activated = "#404080"

# expressed Genes 
expressedGenes <- list(unlist(read.csv(file = "src/database/filtered_genes_Lympho_Resting.csv")),
                       unlist(read.csv(file = "src/database/filtered_genes_Lympho_Activated.csv")),
                       unlist(read.csv(file = "src/database/filtered_genes_Macro_Resting.csv")),
                       unlist(read.csv(file = "src/database/filtered_genes_Macro_Activated.csv")))
names(expressedGenes) <- c("Lympho_Resting", "Lympho_Activated", "Macro_Resting", "Macro_Activated")

# funtions ----------------------------------------------------------------
plotImp <- function(model, model_name) {
  imp <- varImp(model)
  imp <- imp$importance
  colnames(imp)[1] <- "importance"
  imp$variable <- rownames(imp)
  order_var <- imp[order(imp[,1], decreasing = FALSE),"variable"]
  imp$variable <- factor(imp$variable, levels = order_var)
  
  # Determine in the state type
  if (length(grep("Activated|delta", model_name, value = TRUE)) == 1 ) {
    color <- "#404080"
  }  else if (length(grep("Resting", model_name, value = TRUE)) == 1 ) {
    color <- "#69b3a2"
  }
  
  p1 <- ggplot(data = imp, aes_string(x = "variable", y = colnames(imp)[1])) +
    geom_point(stat = "identity", color = color) + 
    coord_flip() + ggtitle(model_name) +
    theme(legend.position = "none",
          axis.text.y = element_text(size = 10)) +
  ggtitle(model_name)
  return(p1)
}

plotSHAP <- function(model, X_train) {
  shap_values <- shap.values(xgb_model = model, X_train = X_train)
  shap_scores <- shap_values$shap_score
  shap_long <- shap.prep(xgb_model = model, X_train = X_train)
  
  shap_summary <- shap.plot.summary(shap_long)
  
  features_ranked1 <- names(shap_values$mean_shap_score)[1:9]
  features_ranked2 <- names(shap_values$mean_shap_score)[10:18]
  fig_list1 <- lapply(features_ranked1, shap.plot.dependence, data_long = shap_long, alpha=0.1)
  fig_list2 <- lapply(features_ranked2, shap.plot.dependence, data_long = shap_long, alpha=0.1)
  
  return(list(shap_summary, fig_list1, fig_list2))
}
plot_dep <- function(y, data_long) {
  data0 <- data_long[variable == y, .(variable, value)]
  data0$x_feature <- data_long[variable == y, log(rfvalue)]
  size0 <- if (nrow(data0) < 1000L) {1} else {0.4}
  alpha <- if (nrow(data0) < 1000L) {1} else {0.6}
  
  plot1 <- ggplot(data = data0, aes(x = x_feature, y = value)) + 
    geom_jitter(size = size0, width = 0, 
                height = 0, alpha = alpha) +
    labs(y = paste0("SHAP value for ", y)) + 
    scale_color_gradient(low = "#FFCC33", high = "#6600CC", 
                        guide = guide_colorbar(barwidth = 10, barheight = 0.3)) +
    theme_bw() + 
    theme(legend.position = "bottom", legend.title = element_text(size = 10), 
          legend.text = element_text(size = 8)) + 
    geom_smooth(method = "loess", color = "red", size = 0.4, se = FALSE)
  
  plot1 <- plot.label(plot1, show_feature = x)
  if (add_stat_cor) {
    plot1 <- plot1 + ggpubr::stat_cor(method = "pearson")
  }
  if (add_hist) {
    plot1 <- ggExtra::ggMarginal(plot1, type = "histogram", 
                                 bins = 50, size = 10, color = "white")
  }
  plot1
}
makeRandomForest <- function(x,
                             explicit_x, 
                             filtering = TRUE, 
                             method = "regression", 
                             RBP_table = NULL, 
                             only_RBP = F) {

  message(x)
  
  # Determine the cell type
  if (length(grep("Lympho",x, value = TRUE)) == 1 ) {
    cell <- "Lympho"
    if (length(grep("delta", x, value = TRUE)) == 1 ) {
      state <- "delta"
      color <- color_activated
      spe <- index_delta_Lympho
    } else {
      spe <- index_lympho
    }
  }  else if (length(grep("Macro",x, value = TRUE)) == 1 ) {
    cell <- "Macro"
    if (length(grep("delta", x, value = TRUE)) == 1 ) {
      state <- "delta"
      color <- color_activated
      spe <- index_delta_Macro
    } else {
    spe <- index_macro
    }
  }
  # Determine in the state type
  if (length(grep("Activated", x, value = TRUE)) == 1 ) {
    state <- "Activated"
    color <- color_activated
    spe <- gsub(pattern = "Resting", replacement = "Activated", x = spe)
  }  else if (length(grep("Resting", x, value = TRUE)) == 1 ) {
    state <- "Resting"
    color <- color_resting
  } 
  
  # create sub_data for the analysis i.e. expressed genes with only column of interest
  
  if (filtering) {
    if (state == "delta") {
      sub_data <- table %>% filter(gene_id %in% expressedGenes[[paste0(cell, "_Resting")]] & 
                                   gene_id %in% expressedGenes[[paste0(cell, "_Activated")]])
    } else {
      sub_data <- table %>% filter(gene_id %in% expressedGenes[[paste0(cell, "_", state)]])
    }
  } 
  
  sub_data[is.na(sub_data$sumKozakuOFR), "sumKozakuOFR"] <- 0
  
  # Check if need to merge RBP 
  if (!only_RBP && is.data.frame(RBP_table)) {
    sub_data <- merge(sub_data, RBP_table, by = 'gene_id') %>%
      select(x, index, spe, colnames(RBP_table)) %>%
      select(-gene_id)
  } else if (only_RBP && is.data.frame(RBP_table)) {
    sub_data <- merge(sub_data %>% select(gene_id, all_of(x)), 
                      RBP_table, by = 'gene_id') %>%
      select(-gene_id)
  } else {
    sub_data <- sub_data %>% select(x, index, spe)
  }
   
  # clean the table if necessary
  sub_data <- na.exclude(sub_data) 
  sub_data <- sub_data[is.finite(rowSums(sub_data)),]
  
  if (method == "classification") {
    stats <- boxplot(sub_data[,1], plot = FALSE)$stats
    sub_data$predict <- "HighTDD"
    sub_data[sub_data[,1] < stats[4,1], "predict"] <- "mediumTDD"
    sub_data[sub_data[,1] < stats[2,1], "predict"] <- "lowTDD"
    sub_data <- sub_data[,-1]
    sub_data$predict <- as.factor(sub_data$predict)
  } else if (method == "regression") {
    colnames(sub_data)[1] <- "predict"
  } else {
    stop(paste("method must be \"regression\" or \"classification\""))
  }
  
  #DEV
  colnames(sub_data)[grep('Exp_UTR3', colnames(sub_data))] <- "Exp_UTR3"
  sub_data <- sub_data %>% mutate(
    Exp_UTR3 = if_else(Exp_UTR3 < 0.5, 0.01, Exp_UTR3),
    Exp_UTR3 = log(Exp_UTR3),
    trans_length = log(trans_length),
    Length.utr5 = log(Length.utr5)
  ) 
  
  # Divde the data in train and validation table
  set.seed(1043)
  train <- sample(nrow(sub_data), round(nrow(sub_data)*0.8))
  train_set <-  sub_data[train,]
  valid_set <- sub_data[-train,]
  
  message(paste("number of genes analysed : ", nrow(sub_data)))
  message(paste("number of genes in trainning dataset :", nrow(train_set)))
  message(paste("number of genes in valdation dataset :", nrow(valid_set)))
  
  # perform random Forest
  
  xgb_trcontrol = trainControl(method = "cv", number = 5, allowParallel = TRUE, 
                               verboseIter = FALSE, returnData = FALSE)
  xgbGrid <- expand.grid(nrounds = c(100,200),  
                         max_depth = c(3, 5, 10, 15, 20),
                         colsample_bytree = seq(0.5, 0.9, length.out = 5),
                         ## valeurs par défaut : 
                         eta = 0.1,
                         gamma=0,
                         min_child_weight = 1,
                         subsample = 1
  )
  
  mod = train(predict ~ ., data = train_set, trControl = xgb_trcontrol, tuneGrid = xgbGrid, 
                    method = "xgbTree")
  
  rf_model <- mod$finalModel
  
  pred_train_CD <- predict(rf_model, train_set %>% select(-predict) %>% as.matrix)
  data_train_CD <- as.data.frame(cbind(pred_train_CD, train_set$predict))
  colnames(data_train_CD) <- c("predicted_index", "real_index")
  
  write.csv(data_train_CD, 'results/training_set.csv')
  
  pred_valid_CD <- predict(rf_model, valid_set %>% select(-predict) %>% as.matrix)
  data_valid_CD <- as.data.frame(cbind(pred_valid_CD, valid_set$predict))
  colnames(data_valid_CD) <- c("predicted_index", "real_index")
  
  write.csv(data_valid_CD, 'results/Validation_set.csv')
  
  # saving the model
  saveRDS(rf_model, paste0('results/', explicit_x, '_model.rda'))
  
  if (method == "regression") {
    pred_train <- predict(rf_model, train_set %>% select(-predict) %>% as.matrix)
    data <- as.data.frame(cbind(pred_train, train_set$predict))
    colnames(data) <- c("predicted_index", "real_index")
    
    lm_model <- lm(predicted_index ~ real_index, data)
    r_cor <- cor(data$predicted_index, data$real_index, method = "spearman") 
    sentence <- paste0("linear regression : ", 
                       round(lm_model[["coefficients"]]["real_index"], digits = 3),
                       "x + ", 
                       round(lm_model[["coefficients"]]["(Intercept)"], digits = 3),
                       ". Spearman correlation coefficient : ",
                       round(r_cor, digits = 3))
    
    if (!state == "delta") {
      p1 <- ggplot(data = data, aes(x = real_index, y = predicted_index)) + 
        geom_point(alpha = 0.5, color = color) + 
        xlim(-0.5,2) + ylim(-0.5,2)  + geom_abline(slope = 1, alpha = 0.6, linetype = "dashed") +
        ggtitle(paste("Train set : ", x)) +
        geom_smooth(method = "lm") +
        annotate(geom = "text", x = 0, y = 1.2, label = sentence, color = "black") +
        theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
              panel.background = element_blank(), axis.line = element_line(colour = "black")) 
    } else {
      p1 <- ggplot(data = data, aes(x = real_index, y = predicted_index)) + 
        geom_point(alpha = 0.5, color = color_activated) + 
        xlim(-1,1) + ylim(-1,1) + geom_abline(slope = 1, alpha = 0.6, linetype = "dashed") +
        ggtitle(paste("Train set : ", x)) +
        geom_smooth(method = "lm") +
        annotate(geom = "text", x = -0.3, y = 0.8, label = sentence, color = "black") +
        theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
              panel.background = element_blank(), axis.line = element_line(colour = "black")) 
    }
    
    pred_valid <- predict(rf_model, valid_set %>% select(-predict) %>% as.matrix)
    # data <- as.data.frame(cbind(pred_valid$predictions, valid_set$predict))
    data <- as.data.frame(cbind(pred_valid, valid_set$predict))
    colnames(data) <- c("predicted_index", "real_index")
    lm_model <- lm(predicted_index ~ real_index, data)
    r_cor <- cor(data$predicted_index, data$real_index, method = "spearman") 
    sentence <- paste0("linear regression : ", 
                       round(lm_model[["coefficients"]]["real_index"], digits = 3),
                       "x + ", 
                       round(lm_model[["coefficients"]]["(Intercept)"], digits = 3),
                       ". Spearman correlation coefficient : ",
                       round(r_cor, digits = 3))
    
    if (!state == "delta") {
      p2 <- ggplot(data = data, aes(x = real_index, y = predicted_index)) + 
        geom_point(alpha = 0.5, color = color) + 
        ylim(-0.5,2) + xlim(-0.5,2) +
        geom_smooth(method = "lm") +
        annotate(geom = "text", x = 0, y = 1.2, label = sentence, color = "black") +
        ggtitle(paste("Validation set : ", x)) + geom_abline(slope = 1, alpha = 0.6, linetype = "dashed") +
        theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
              panel.background = element_blank(), axis.line = element_line(colour = "black"))
    } else {
     p2 <- ggplot(data = data, aes(x = real_index, y = predicted_index)) + 
        geom_point(alpha = 0.5, color = color_activated) + 
        xlim(-1,1) + ylim(-1,1) + geom_abline(slope = 1, alpha = 0.6, linetype = "dashed") +
        ggtitle(paste("Valid set : ", x)) +
        geom_smooth(method = "lm") +
        annotate(geom = "text", x = -0.3, y = 0.8, label = sentence, color = "black") +
        theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
              panel.background = element_blank(), axis.line = element_line(colour = "black")) 
    }
    
    p3 <- plotImp(mod, x)
    shap <- plotSHAP(model = rf_model, X_train = train_set %>% select(-predict) %>% as.matrix)
    plots <- list(model = mod, 
                  training_set = p1,
                  valid_set = p2,
                  ImpVar = p3, 
                  shap_summary = shap[[1]], 
                  shap_dep1 = shap[[2]],
                  shap_dep2 = shap[[3]])
    
  } else if (method == "classification") {
    pred_valid <- predict(rf_model, valid_set %>% select (- predict))
    table <- table(pred_valid$predictions, valid_set$predict)
    class.err <- c(sum(table[1,c(2,3)])/sum(table[1,]),
             sum(table[2,c(1,3)])/sum(table[2,]),
             sum(table[3,c(1,2)])/sum(table[3,]))
    table <- cbind(table, class.err)
    p3 <- plotImp(mod, x)
    plots <- list(mod, table, p3)
    names(plots) <- c("rf_model", "valid_set", "ImpVar")
  }

  return(plots)
}
# loading data ----------------------------------------------

db <- 'src/database/2023-07-17_Subset_Data_processed_Merge.csv'

table <- as_tibble(read.csv(db))

#table <- table[is.finite(table[, x1]), , drop = FALSE]
#table <-table %>% filter(!is.infinite(!!sym(x1)))
#table <- table %>% filter(!is.na(!!sym(x1)))



table$deltaTDD_Lympho <- table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h
table$deltaNonTDD_Lympho <- table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h
table$deltaRiboDens_Lympho <- table$RiboDens.Lympho_Activated - table$RiboDens.Lympho_Resting
table$deltaRiboDens_Macro <- table$RiboDens.Macro_Activated - table$RiboDens.Macro_Resting
table$deltaUTRlength_Lympho <- table$Exp_UTR3_Lympho_Activated_Mean - table$Exp_UTR3_Lympho_Resting_Mean
table$deltaUTRlength_Macro <- table$Exp_UTR3_Macro_Activated_Mean - table$Exp_UTR3_Macro_Resting_Mean

table$delta_TDD_new_Lympho <- table$Lympho_Activated.Abs_TDD_mean.3 - table$Lympho_Resting.Abs_TDD_mean.3
table$delta_TID_new_Lympho <- table$Lympho_Activated.Abs_TID_mean.3 - table$Lympho_Resting.Abs_TID_mean.3



res <- read.csv("src/database/DESeqresults_lympho_activation_untreated.csv")
res2 <- as.data.frame(res)
res2 <- res2[!is.na(res2$log2FoldChange),]
colnames(res2)[1] <- "gene_id"
res2$gene_id <- str_extract(res2$gene_id, "ENSMUSG...........")
table <- merge(table, select(res2, gene_id, log2FoldChange, padj), by = "gene_id", all.x = TRUE)

print('Columns present in the database:')
print(colnames(table))

# Set indexes -------------------------------------------------------------

index <- c("Length.utr5", 
           # "UTR5_length", 
           "UTR5_GC", 
           "Length.cds", 
           # "CDS_length",
           "CDS_GC",
           # "Length.utr3",
           # "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h",
           # "UTR3_length", 
           "UTR3_GC", 
           "trans_length",
           # "n_m6ASeq_Peaks", 
           "total_m6ASeq_score",
           "CDS_DG_MFEden",
           # "UTR5_DG", 
           # "UTR3_DG", 
           "X47UTR5.30CDS_DG_MFEden",
           # "X31CDS.End_DG",
           # "Kozack_score_frequency", 
           # "Kozack_score_efficiency", 
           # "Kozack_score_dinucleotide", 
           # "n_introns_in_UTR5", 
           # "n_CDS_fragments", 
           # "n_introns_in_UTR3", 
           "n_exon",
           # "n_intron",
           # "Total_AUUUA", 
           "N_AUUUA_UTR5",
           "N_AUUUA_CDS",
           "N_AUUUA_UTR3",
           # "Total_UUAUUUAUU", 
           # "N_UUAUUUAUU_UTR5", 
           # "N_UUAUUUAUU_CDS", 
           # "N_UUAUUUAUU_UTR3",
           "n4G_UTR5",
           "n4G_CDS",
           "n4G_UTR3",
           # "Stretches_AAA_AAG_score",
           # "Stretches_AAA_AAG_pondarated_score",
           # "Stretches_GAC_GAT_score",
           # "Stretches_GAC_GAT_pondarated_score",
           # "Stretches_GAA_GAG_score",
           # "Stretches_GAA_GAG_pondarated_score",
           # "Stretches_AAA_AAG_GAC_GAT_GAA_GAG_score",
           # "Stretches_AAA_AAG_GAC_GAT_GAA_GAG_pondarated_score",
           # "percent_ALA_A",
           # "percent_ARG_R",
           # "percent_ASN_N",
           # "percent_ASP_D",
           # "percent_CYS_C",
           # "percent_GLU_E",
           # "percent_GLN_Q",
           # "percent_GLY_G",
           # "percent_HIS_H",
           # "percent_ILE_I",
           # "percent_LEU_L",
           # "percent_LYS_K",
           # "percent_MET_M",
           # "percent_PHE_F",
           # "percent_PRO_P",
           # "percent_SER_S",
           # "percent_THR_T",
           # "percent_TRP_W",
           # "percent_TYR_Y",
           # "percent_VAL_V",
           "sumKozakuOFR",
           # "N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR5",
           # "N_.AU..AU..AU.UAUUUAU.AU..AU..AU._CDS",
           # "N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3"
           # "Hwang.transcript_half.life"
           "GC3_percent"
)

  index_lympho <- c("RiboDens.Lympho_Resting" ,
                  # "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h",
                  # "Hwang.Lympho_Resting.polyA_length"
                  "Exp_UTR3_Lympho_Resting_Mean"#,
                  # "basal_expr_Lympho_Resting_RPKM",
                  # "basal_expr_Lympho_Activated_RPKM"
                  )

index_macro <- c("RiboDens.Macro_Resting", 
                 # "DegFold.Macro_Resting.Trip.Ref_Trip_0h.3h",
                 "Exp_UTR3_Macro_Resting_Mean",
                 "basal_expr_Macro_Resting_RPKM",
                 "basal_expr_Macro_Activated_RPKM")

tddindex_resting <- c("Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", 
                      # "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h", 
                      "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h", 
                      # "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h", 
                      "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h", 
                      # "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.3h", 
                      "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.1h", 
                      # "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.1h",
                      "Abs.TDD..Macro_Resting.Trip_CHX.Ref_Trip_0h.3h", 
                      "Abs.TDD..Macro_Resting.Trip_Harr.Ref_Trip_0h.3h" ,
                      # "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", 
                      # "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h", 
                      # "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h", 
                      # "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h", 
                      # "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h", 
                      # "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.3h", 
                      # "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.1h", 
                      # "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.1h", 
                      # "Abs.NonTDD..Macro_Resting.Trip_CHX.Ref_Trip_0h.3h", 
                      # "Abs.NonTDD..Macro_Resting.Trip_Harr.Ref_Trip_0h.3h", 
                      "Rel.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h",
                      # "Rel.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h", 
                      "Rel.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h",
                      # "Rel.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h", 
                      "Rel.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h",
                      # "Rel.TDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.3h", 
                      "Rel.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.1h",
                      # "Rel.TDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.1h", 
                      "Rel.TDD..Macro_Resting.Trip_CHX.Ref_Trip_0h.3h",
                      "Rel.TDD..Macro_Resting.Trip_Harr.Ref_Trip_0h.3h"
                      # "Rel.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h",
                      # "Rel.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h", 
                      # "Rel.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h", 
                      # "Rel.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h", 
                      # "Rel.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h", 
                      # "Rel.NonTDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.3h", 
                      # "Rel.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.1h", 
                      # "Rel.NonTDD..Lympho_Resting.Trip_Harr.Ref_untreated_0h.1h", 
                      # "Rel.NonTDD..Macro_Resting.Trip_CHX.Ref_Trip_0h.3h", 
                      # "Rel.NonTDD..Macro_Resting.Trip_Harr.Ref_Trip_0h.3h",
                      # "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h", 
                      # "DegFold.Lympho_Resting.Trip.Ref_untreated_0h.3h", 
                      # "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.1h", 
                      # "DegFold.Lympho_Resting.Trip.Ref_untreated_0h.1h", 
                      # "DegFold.Macro_Resting.Trip.Ref_Trip_0h.3h",
                      # "Abs.NonDEG..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", 
                      # "Abs.NonDEG..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h", 
                      # "Abs.NonDEG..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h", 
                      # "Abs.NonDEG..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h", 
                      # "Abs.NonDEG..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h", 
                      # "Abs.NonDEG..Lympho_Resting.Trip_Harr.Ref_untreated_0h.3h", 
                      # "Abs.NonDEG..Lympho_Resting.Trip_Harr.Ref_Trip_0h.1h", 
                      # "Abs.NonDEG..Lympho_Resting.Trip_Harr.Ref_untreated_0h.1h", 
                      # "Abs.NonDEG..Macro_Resting.Trip_CHX.Ref_Trip_0h.3h", 
                      # "Abs.NonDEG..Macro_Resting.Trip_Harr.Ref_Trip_0h.3h"
)

tddindex_activated <- c("Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        # "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h", 
                        "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h", 
                        # "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h", 
                        "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h", 
                        # "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.3h", 
                        "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.1h", 
                        # "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.1h", 
                        "Abs.TDD..Macro_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        "Abs.TDD..Macro_Activated.Trip_Harr.Ref_Trip_0h.3h",
                        # "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        # "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h", 
                        # "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h", 
                        # "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h", 
                        # "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h", 
                        # "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.3h", 
                        # "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.1h", 
                        # "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.1h", 
                        # "Abs.NonTDD..Macro_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        # "Abs.NonTDD..Macro_Activated.Trip_Harr.Ref_Trip_0h.3h", 
                        "Rel.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h",
                        # "Rel.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h", 
                        "Rel.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h",
                        # "Rel.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h", 
                        "Rel.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h",
                        # "Rel.TDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.3h", 
                        "Rel.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.1h",
                        # "Rel.TDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.1h", 
                        "Rel.TDD..Macro_Activated.Trip_CHX.Ref_Trip_0h.3h",
                        "Rel.TDD..Macro_Activated.Trip_Harr.Ref_Trip_0h.3h",
                        # "Rel.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        # "Rel.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h", 
                        # "Rel.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h", 
                        # "Rel.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h", 
                        # "Rel.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h", 
                        # "Rel.NonTDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.3h", 
                        # "Rel.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.1h", 
                        # "Rel.NonTDD..Lympho_Activated.Trip_Harr.Ref_untreated_0h.1h", 
                        # "Rel.NonTDD..Macro_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        # "Rel.NonTDD..Macro_Activated.Trip_Harr.Ref_Trip_0h.3h",
                        # "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h", 
                        # "DegFold.Lympho_Activated.Trip.Ref_untreated_0h.3h", 
                        # "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.1h", 
                        # "DegFold.Lympho_Activated.Trip.Ref_untreated_0h.1h",
                        # "DegFold.Macro_Activated.Trip.Ref_Trip_0h.3h",
                        # "Abs.NonDEG..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        # "Abs.NonDEG..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h", 
                        # "Abs.NonDEG..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h", 
                        # "Abs.NonDEG..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h", 
                        # "Abs.NonDEG..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h", 
                        # "Abs.NonDEG..Lympho_Activated.Trip_Harr.Ref_untreated_0h.3h", 
                        # "Abs.NonDEG..Lympho_Activated.Trip_Harr.Ref_Trip_0h.1h", 
                        # "Abs.NonDEG..Lympho_Activated.Trip_Harr.Ref_untreated_0h.1h", 
                        # "Abs.NonDEG..Macro_Activated.Trip_CHX.Ref_Trip_0h.3h", 
                        # "Abs.NonDEG..Macro_Activated.Trip_Harr.Ref_Trip_0h.3h",
                        "Abs.TDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h", 
                        "Abs.TDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.1h", 
                        "Abs.TDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h", 
                        "Abs.TDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.1h"
                        # "Abs.NonTDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h", 
                        # "Abs.NonTDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.1h", 
                        # "Abs.NonTDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h", 
                        # "Abs.NonTDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.1h", 
                        # "Abs.NonDEG..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h", 
                        # "Abs.NonDEG..Lympho_Activated.DRB_CHX.Ref_DRB_0h.1h", 
                        # "Abs.NonDEG..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h", 
                        # "Abs.NonDEG..Lympho_Activated.DRB_Harr.Ref_DRB_0h.1h", 
                        # "Rel.TDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h", 
                        # "Rel.TDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.1h", 
                        # "Rel.TDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h", 
                        # "Rel.TDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.1h", 
                        # "Rel.NonTDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h", 
                        # "Rel.NonTDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.1h", 
                        # "Rel.NonTDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h", 
                        # "Rel.NonTDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.1h"
)

index_delta_Lympho <- c("log2FoldChange",
                        "deltaUTRlength_Lympho",
                        "Exp_UTR3_Lympho_Resting_Mean")

index_delta_Macro <- c("deltaRiboDens_Macro",
                       "deltaUTRlength_Macro",
                       "Exp_UTR3_Macro_Resting_Mean")
