## figure paper
library(ggplot2)
library(tidyverse)
library(reshape2)
theme_set(theme_bw())
library(gridExtra)

# setwd("~/RMI2/Projet_TDD/20200114_Normalisation")
table <- read.csv("data/databases/2020-03-26_07-39-50_Subset_Data_processed.csv")

color_resting = "#69b3a2"
color_activated = "#404080"
color_1h = "deepskyblue"
color_3h = "darkmagenta"

expressedGenes <- list(Lympho_Resting = unlist(read.csv(file = "results/filtred_genes_Lympho_Resting.csv")),
                       Lympho_Activated = unlist(read.csv(file = "results/filtred_genes_Lympho_Activated.csv")),
                       Macro_Resting = unlist(read.csv(file = "results/filtred_genes_Macro_Resting.csv")),
                       Macro_Activated = unlist(read.csv(file = "results/filtred_genes_Macro_Activated.csv")))

genes_RiboDens_Inf_1.5 <- table %>% filter(RiboDens.Lympho_Resting < 1.5 & RiboDens.Lympho_Activated < 1.5) %>% 
                            select(gene_id) %>% 
                            unlist() %>% 
                            as.character()
genes_RiboDens_Inf_1.5 <- genes_RiboDens_Inf_1.5[genes_RiboDens_Inf_1.5 %in% expressedGenes[["Lympho_Resting"]] & genes_RiboDens_Inf_1.5 %in% expressedGenes[["Lympho_Activated"]]]

genes_RiboDens_Sup_1.5 <- table %>% filter(RiboDens.Lympho_Resting > 1.5 & RiboDens.Lympho_Activated > 1.5) %>% 
  select(gene_id) %>% 
  unlist() %>% 
  as.character()
genes_RiboDens_Sup_1.5 <- genes_RiboDens_Sup_1.5[genes_RiboDens_Sup_1.5 %in% expressedGenes[["Lympho_Resting"]] & genes_RiboDens_Sup_1.5 %in% expressedGenes[["Lympho_Activated"]]]

load(file = "results/pval_bootstrap.RData")


# functions ---------------------------------------------------------------

summarySE <- function(data=NULL, measurevar, groupvars=NULL, na.rm=FALSE,
                      conf.interval=.95, .drop=TRUE) {
  library(plyr)
  
  # New version of length which can handle NA's: if na.rm==T, don't count them
  length2 <- function (x, na.rm=FALSE) {
    if (na.rm) sum(!is.na(x))
    else       length(x)
  }
  
  # This does the summary. For each group's data frame, return a vector with
  # N, mean, and sd
  datac <- ddply(data, groupvars, .drop=.drop,
                 .fun = function(xx, col) {
                   c(N    = length2(xx[[col]], na.rm=na.rm),
                     mean = mean   (xx[[col]], na.rm=na.rm),
                     sd   = sd     (xx[[col]], na.rm=na.rm)
                   )
                 },
                 measurevar
  )
  
  # Rename the "mean" column    
  datac <- rename(datac, c("mean" = measurevar))
  
  datac$se <- datac$sd / sqrt(datac$N)  # Calculate standard error of the mean
  
  # Confidence interval multiplier for standard error
  # Calculate t-statistic for confidence interval: 
  # e.g., if conf.interval is .95, use .975 (above/below), and use df=N-1
  ciMult <- qt(conf.interval/2 + .5, datac$N-1)
  datac$ci <- datac$se * ciMult
  
  return(datac)
}

# mRNA degradation density with Trip 0h as ref ------------------------------------------------

# pdf("final_fig/fig_degradationDensity_0hTrip.pdf", width = 8)

# Lympho Resting 1h vs 3h
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h,
              table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h))
colnames(data) <- c("at 1h", "at 3h")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=c(color_1h, color_3h)) +
  scale_color_manual(values=c(color_1h, color_3h)) + 
  ggtitle("Lymphocyte resting Triptolide") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))



# Lymphocytes Activated 1h vs 3h
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h,
                            table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h))
colnames(data) <- c("at 1h", "at 3h")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=c(color_1h, color_3h)) +
  scale_color_manual(values=c(color_1h, color_3h)) + 
  ggtitle("Lymphocyte activated Triptolide") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))


#Lymphocytes 1h Resting vs Activated
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h,
                            table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h))
colnames(data) <- c("Resting", "Activated")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=c(color_resting, color_activated)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte Triptolide at 1h") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))


#Lymphocytes 3h Resting vs Activated
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h,
                            table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h))
colnames(data) <- c("Resting", "Activated")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=alpha(c(color_resting, color_activated), 0.75)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte Triptolide at 3h") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# dev.off()

# mRNA degradation with 0h untreated as ref -------------------------------


# pdf("final_fig/fig_degradationDensity_untreated0h.pdf", width = 8)

# Lympho Resting 1h vs 3h
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h,
                            table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h))
colnames(data) <- c("at 1h", "at 3h")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=c(color_1h, color_3h)) +
  scale_color_manual(values=c(color_1h, color_3h)) + 
  ggtitle("Lymphocyte resting Triptolide") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))



# Lymphocytes Activated 1h vs 3h
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h,
                            table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h))
colnames(data) <- c("at 1h", "at 3h")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=c(color_1h, color_3h)) +
  scale_color_manual(values=c(color_1h, color_3h)) + 
  ggtitle("Lymphocyte activated Triptolide") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))


#Lymphocytes 1h Resting vs Activated
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.1h,
                            table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.1h))
colnames(data) <- c("Resting", "Activated")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=c(color_resting, color_activated)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte Triptolide at 1h") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))


#Lymphocytes 3h Resting vs Activated
data <- as.data.frame(cbind(table$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h + table$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_untreated_0h.3h,
                            table$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h + table$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_untreated_0h.3h))
colnames(data) <- c("Resting", "Activated")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1.2) + 
  xlab("fraction of mRNA degradation") + 
  scale_fill_manual(values=alpha(c(color_resting, color_activated), 0.75)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte Triptolide at 3h") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# dev.off()



# TDD index distribution --------------------------------------------------

# Lympho Resting vs Activated at 3h

data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] & gene_id %in% expressedGenes[["Lympho_Activated"]]) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
  
colnames(data) <- c("Resting", "Activated")
data <- melt(data)

# together
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-0.5,1) + 
  xlab("Absolute TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated)  +
  scale_fill_manual(values=c(color_resting, color_activated)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte : TDD index at 3h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# only resting
ggplot(data = data %>% filter(variable == "Resting"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  scale_fill_manual(values=c(color_resting)) +
  scale_color_manual(values=c(color_resting)) + 
  ggtitle("Lymphocyte : TDD index at 3h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# only activated
ggplot(data = data %>% filter(variable == "Activated"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
  scale_fill_manual(values=c(color_activated)) +
  scale_color_manual(values=c(color_activated)) + 
  ggtitle("Lymphocyte : TDD index at 3h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# Lympho Resting vs Activated at 3h
# together
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] & gene_id %in% expressedGenes[["Lympho_Activated"]]) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h, Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h)

colnames(data) <- c("Resting", "Activated")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated)  +
  scale_fill_manual(values=c(color_resting, color_activated)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte : TDD index at 1h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

# only resting
ggplot(data = data %>% filter(variable == "Resting"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  scale_fill_manual(values=c(color_resting)) +
  scale_color_manual(values=c(color_resting)) + 
  ggtitle("Lymphocyte : TDD index at 1h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# only activated
ggplot(data = data %>% filter(variable == "Activated"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
  scale_fill_manual(values=c(color_activated)) +
  scale_color_manual(values=c(color_activated)) + 
  ggtitle("Lymphocyte : TDD index at 1h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# NonTDD index distribution --------------------------------------------------

# Lympho Resting vs Activated at 3h

data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] & gene_id %in% expressedGenes[["Lympho_Activated"]]) %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("Resting", "Activated")
data <- melt(data)

# together
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute Non TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated)  +
  scale_fill_manual(values=c(color_resting, color_activated)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte : Non TDD index at 3h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# only resting
ggplot(data = data %>% filter(variable == "Resting"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute Non TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  scale_fill_manual(values=c(color_resting)) +
  scale_color_manual(values=c(color_resting)) + 
  ggtitle("Lymphocyte : Non TDD index at 3h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# only activated
ggplot(data = data %>% filter(variable == "Activated"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute Non TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
  scale_fill_manual(values=c(color_activated)) +
  scale_color_manual(values=c(color_activated)) + 
  ggtitle("Lymphocyte : Non TDD index at 3h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# Lympho Resting vs Activated at 3h
# together
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] & gene_id %in% expressedGenes[["Lympho_Activated"]]) %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h)
colnames(data) <- c("Resting", "Activated")
data <- melt(data)
ggplot(data = data, aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute Non TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated)  +
  scale_fill_manual(values=c(color_resting, color_activated)) +
  scale_color_manual(values=c(color_resting, color_activated)) + 
  ggtitle("Lymphocyte : Non TDD index at 1h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

# only resting
ggplot(data = data %>% filter(variable == "Resting"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute Non TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Resting") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_resting) + 
  scale_fill_manual(values=c(color_resting)) +
  scale_color_manual(values=c(color_resting)) + 
  ggtitle("Lymphocyte : Non TDD index at 1h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# only activated
ggplot(data = data %>% filter(variable == "Activated"), aes(x = value)) + 
  geom_histogram(aes(y=..density.., fill = variable), alpha = 0.75, binwidth = .015, position = "identity") +
  geom_density(alpha=0.6, position = "identity",aes(color = variable), size = 1)  + 
  xlim(-1,1.5) + 
  xlab("Absolute Non TDD index") + 
  geom_vline(xintercept = median(unlist(data %>% filter(variable == "Activated") %>% select(value)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
  scale_fill_manual(values=c(color_activated)) +
  scale_color_manual(values=c(color_activated)) + 
  ggtitle("Lymphocyte : Non TDD index at 1h Triptolide CHX") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# correlation replicate (row counts) --------------------------------------

all_data=read.delim("data/HTSeq_count_stats_all_libraries.csv", header=TRUE, sep=",", row.names=1, stringsAsFactors = TRUE) 
all_data <- all_data[grep(pattern = "ENS.*", rownames(all_data)),]
# pdf("final_fig/fig_replicats.pdf")


var1 <- c("R2_4_Resting_0h_Triptolide_exon", "R2_4_Resting_0h_Triptolide_exon", "R3_4_Resting_0h_Triptolide_exon",
          "R2_5_Resting_1h_Triptolide_exon", "R2_5_Resting_1h_Triptolide_exon", "R3_5_Resting_1h_Triptolide_exon",
          "R2_6_Resting_3h_Triptolide_exon", "R2_6_Resting_3h_Triptolide_exon", "R3_6_Resting_3h_Triptolide_exon",
          "R2_11_Resting_1h_Triptolide_CHX_exon", "R2_11_Resting_1h_Triptolide_CHX_exon", "R3_11_Resting_1h_Triptolide_CHX_exon",
          "R2_12_Resting_3h_Triptolide_CHX_exon", "R2_12_Resting_3h_Triptolide_CHX_exon", "R3_12_Resting_3h_Triptolide_CHX_exon", 
          "R2_13_Resting_1h_Triptolide_Harr_exon", "R2_13_Resting_1h_Triptolide_Harr_exon", "R3_13_Resting_1h_Triptolide_Harr_exon",
          "R2_14_Resting_3h_Triptolide_Harr_exon", "R2_14_Resting_3h_Triptolide_Harr_exon", "R3_14_Resting_3h_Triptolide_Harr_exon")

var2 <- c("R3_4_Resting_0h_Triptolide_exon", "R4_4_Resting_0h_Triptolide_exon", "R4_4_Resting_0h_Triptolide_exon",
          "R3_5_Resting_1h_Triptolide_exon", "R4_5_Resting_1h_Triptolide_exon", "R4_5_Resting_1h_Triptolide_exon",
          "R3_6_Resting_3h_Triptolide_exon", "R4_6_Resting_3h_Triptolide_exon", "R4_6_Resting_3h_Triptolide_exon",
          "R3_11_Resting_1h_Triptolide_CHX_exon", "R4_11_Resting_1h_Triptolide_CHX_exon", "R4_11_Resting_1h_Triptolide_CHX_exon",
          "R3_12_Resting_3h_Triptolide_CHX_exon", "R4_12_Resting_3h_Triptolide_CHX_exon", "R4_12_Resting_3h_Triptolide_CHX_exon",
          "R3_13_Resting_1h_Triptolide_Harr_exon", "R4_13_Resting_1h_Triptolide_Harr_exon", "R4_13_Resting_1h_Triptolide_Harr_exon",
          "R3_14_Resting_3h_Triptolide_Harr_exon", "R4_14_Resting_3h_Triptolide_Harr_exon", "R4_14_Resting_3h_Triptolide_Harr_exon")

legend1_1 <- rep(c("replicate 1", "replicate 1", "replicate 2" ), times = 7)
legend1_2 <- rep(c("replicate 2", "replicate 3", "replicate 3" ), times = 7)

legend2 <- c(rep("0h", times = 3), 
             rep("1h", times = 3), 
             rep("3h", times = 3), 
             rep("1h", times = 3), 
             rep("3h", times = 3), 
             rep("1h", times = 3), 
             rep("3h", times = 3) 
            )
legend3 <- c(rep("Triptolide", times = 9),
             rep("Triptolide + CHX", times = 6),
             rep("Triptolide + Harr", times = 6)
           )

for (i in 1:length(var1)) {
  data <- all_data %>% select(var1[i], var2[i]) %>% as.data.frame()
  rsq <- round(cor(x= data[,1], y = data[,2]),
               digit = 3)
  p1 <- ggplot(data = data,aes(x = log(data[,var1[i]]), log(data[,var2[i]]))) +
    geom_point(alpha = 0.5, color = color_resting  ) +
    geom_text(x = 1, y = 9, label=paste("r = ", rsq), size = 5) +
    theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
          panel.background = element_blank(), axis.line = element_line(colour = "black")) +
    xlab(paste("lymphocytes resting", legend2[i], legend3[i],  legend1_1[i], "- (log(counts))")) + 
    ylab(paste("lymphocytes resting", legend2[i], legend3[i],  legend1_2[i], "- (log(counts))")) +
    xlim(0,11) + ylim(0,11)
  
  print(p1)
}

var1 <- c("A2_7_Activated_0h_Triptolide_exon", "A2_7_Activated_0h_Triptolide_exon", "A3_7_Activated_0h_Triptolide_exon",
          "A2_8_Activated_1h_Triptolide_exon", "A2_8_Activated_1h_Triptolide_exon", "A3_8_Activated_1h_Triptolide_exon",
          "A2_9_Activated_3h_Triptolide_exon", "A2_9_Activated_3h_Triptolide_exon", "A3_9_Activated_3h_Triptolide_exon",
          "A2_18_Activated_1h_Triptolide_CHX_exon", "A2_18_Activated_1h_Triptolide_CHX_exon", "A3_18_Activated_1h_Triptolide_CHX_exon",
          "A2_19_Activated_3h_Triptolide_CHX_exon", "A2_19_Activated_3h_Triptolide_CHX_exon", "A3_19_Activated_3h_Triptolide_CHX_exon", 
          "A2_20_Activated_1h_Triptolide_Harr_exon", "A2_20_Activated_1h_Triptolide_Harr_exon", "A3_20_Activated_1h_Triptolide_Harr_exon",
          "A2_21_Activated_3h_Triptolide_Harr_exon", "A2_21_Activated_3h_Triptolide_Harr_exon", "A3_21_Activated_3h_Triptolide_Harr_exon",
          "A2_4_Activated_0h_DRB_exon", "A2_4_Activated_0h_DRB_exon", "A3_4_Activated_0h_DRB_exon",
          "A2_5_Activated_1h_DRB_exon", "A2_5_Activated_1h_DRB_exon", "A3_5_Activated_1h_DRB_exon",
          "A2_6_Activated_3h_DRB_exon", "A2_6_Activated_3h_DRB_exon", "A3_6_Activated_3h_DRB_exon",
          "A2_14_Activated_1h_DRB_CHX_exon", "A2_14_Activated_1h_DRB_CHX_exon", "A3_14_Activated_1h_DRB_CHX_exon",
          "A2_15_Activated_3h_DRB_CHX_exon", "A2_15_Activated_3h_DRB_CHX_exon", "A3_15_Activated_3h_DRB_CHX_exon",
          "A2_16_Activated_1h_DRB_Harr_exon", "A2_16_Activated_1h_DRB_Harr_exon", "A3_16_Activated_1h_DRB_Harr_exon",
          "A2_17_Activated_3h_DRB_Harr_exon", "A2_17_Activated_3h_DRB_Harr_exon", "A3_17_Activated_3h_DRB_Harr_exon")

var2 <- c("A3_7_Activated_0h_Triptolide_exon", "A4_7_Activated_0h_Triptolide_exon", "A4_7_Activated_0h_Triptolide_exon",
          "A3_8_Activated_1h_Triptolide_exon", "A4_8_Activated_1h_Triptolide_exon", "A4_8_Activated_1h_Triptolide_exon",
          "A3_9_Activated_3h_Triptolide_exon", "A4_9_Activated_3h_Triptolide_exon", "A4_9_Activated_3h_Triptolide_exon",
          "A3_18_Activated_1h_Triptolide_CHX_exon", "A4_18_Activated_1h_Triptolide_CHX_exon", "A4_18_Activated_1h_Triptolide_CHX_exon",
          "A3_19_Activated_3h_Triptolide_CHX_exon", "A4_19_Activated_3h_Triptolide_CHX_exon", "A4_19_Activated_3h_Triptolide_CHX_exon",
          "A3_20_Activated_1h_Triptolide_Harr_exon", "A4_20_Activated_1h_Triptolide_Harr_exon", "A4_20_Activated_1h_Triptolide_Harr_exon",
          "A3_21_Activated_3h_Triptolide_Harr_exon", "A4_21_Activated_3h_Triptolide_Harr_exon", "A4_21_Activated_3h_Triptolide_Harr_exon",
          "A3_4_Activated_0h_DRB_exon", "A4_4_Activated_0h_DRB_exon", "A4_4_Activated_0h_DRB_exon",
          "A3_5_Activated_1h_DRB_exon", "A4_5_Activated_1h_DRB_exon", "A4_5_Activated_1h_DRB_exon",
          "A3_6_Activated_3h_DRB_exon", "A4_6_Activated_3h_DRB_exon", "A4_6_Activated_3h_DRB_exon",
          "A3_14_Activated_1h_DRB_CHX_exon", "A4_14_Activated_1h_DRB_CHX_exon", "A4_14_Activated_1h_DRB_CHX_exon",
          "A3_15_Activated_3h_DRB_CHX_exon", "A4_15_Activated_3h_DRB_CHX_exon", "A4_15_Activated_3h_DRB_CHX_exon",
          "A3_16_Activated_1h_DRB_Harr_exon", "A4_16_Activated_1h_DRB_Harr_exon", "A4_16_Activated_1h_DRB_Harr_exon",
          "A3_17_Activated_3h_DRB_Harr_exon", "A4_17_Activated_3h_DRB_Harr_exon", "A4_17_Activated_3h_DRB_Harr_exon")

legend1_1 <- rep(c("replicate 1", "replicate 1", "replicate 2" ), times = 14)
legend1_2 <- rep(c("replicate 2", "replicate 3", "replicate 3" ), times = 14)

legend2 <- rep(c(rep("0h", times = 3), 
                 rep("1h", times = 3), 
                 rep("3h", times = 3), 
                 rep("1h", times = 3), 
                 rep("3h", times = 3), 
                 rep("1h", times = 3), 
                 rep("3h", times = 3)), 
                times = 2)
legend3 <- c(rep("Triptolide", times = 9),
             rep("Triptolide + CHX", times = 6),
             rep("Triptolide + Harr", times = 6),
             rep("DRB", times = 9),
             rep("DRB + CHX", times = 6),
             rep("DRB + Harr", times = 6)
)

for (i in 1:length(var1)) {
  data <- all_data %>% select(var1[i], var2[i]) %>% as.data.frame()
  rsq <- round(cor(x= data[,1], y = data[,2]),
               digit = 3)
  p1 <- ggplot(data = data,aes(x = log(data[,var1[i]]), log(data[,var2[i]]))) +
    geom_point(alpha = 0.5, color = color_activated  ) +
    geom_text(x = 1, y = 9, label=paste("r = ", rsq), size = 5) +
    theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
          panel.background = element_blank(), axis.line = element_line(colour = "black")) +
    xlab(paste("lymphocytes activated", legend2[i], legend3[i],  legend1_1[i], "- (log(counts))")) + 
    ylab(paste("lymphocytes activated", legend2[i], legend3[i],  legend1_2[i], "- (log(counts))")) +
    xlim(0,11) + ylim(0,11)
  
  print(p1)
}


# dev.off()

# UPF2 genes --------------------------------------------------------------

Up_UPF2ko_BMM <- read.csv("data/Upregulated_UPF2ko_mouse_BMM.csv")
Up_UPF2ko_BMM <- as.character(Up_UPF2ko_BMM[, 1])
Up_UPF2ko_thymo <- read.csv("data/Upregulated_UPF2ko_mouse_thymocytes.csv")
Up_UPF2ko_thymo <- as.character(Up_UPF2ko_thymo[, 1])

data <- table %>% select(gene_name, 
                         Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h,
                         Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, 
                         Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h,
                         Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h )

colnames(data) <- c("gene_name", "Activated - 3h", "Resting - 3h", "Activated - 1h", "Resting - 1h")
data <- data %>% pivot_longer(- gene_name , names_to = "condition", values_to = "TDD_Index", values_drop_na = TRUE)

data$condition <- factor(data$condition, levels = c("Resting - 3h", "Activated - 3h", "Resting - 1h", "Activated - 1h"))
data$type <- "pop"
data[data$gene_name %in% Up_UPF2ko_thymo | data$gene_name %in% Up_UPF2ko_BMM, "type"] <- "Up in UPF2 KO mice"

resting <- paste("p-val = ", formatC(pval[["upf2_Resting_AbsTDD3hCHX"]], format = "e", digits = 2))
activated <- paste("p-val = ", formatC(pval[["upf2_Activated_AbsTDD3hCHX"]], format = "e", digits = 2))

ggplot(data = data %>% filter(condition == "Activated - 3h" |condition == "Resting - 3h" ) , aes(x = condition, y = TDD_Index, fill = type)) + 
  geom_boxplot(outlier.size = -1, width = 0.5, alpha = 0.8) + ylim(-0.5,1) + scale_fill_manual(values = c(color_activated, color_resting)) +
  ggtitle("Lymphocytes 3h - just UPF2 KO in thymo and BMM ") + 
  annotate(geom = "text", x = 1,  y = 1, label = resting, color = "black") +
  annotate(geom = "text", x = 2,  y = 1, label = activated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
       panel.background = element_blank(), axis.line = element_line(colour = "black")) 
  
resting <- paste("p-val = ", formatC(pval[["upf2_Resting_AbsTDD1hCHX"]], format = "e", digits = 2))
activated <- paste("p-val = ", formatC(pval[["upf2_Activated_AbsTDD1hCHX"]], format = "e", digits = 2))

ggplot(data = data %>% filter (condition == "Activated - 1h" |condition == "Resting - 1h" ) , aes(x = condition, y = TDD_Index, fill = type)) + 
  geom_boxplot(outlier.size = -1, width = 0.5, alpha = 0.8) + ylim(-0.5,1) + scale_fill_manual(values = c(color_activated, color_resting)) +
  ggtitle("Lymphocytes 1h - just UPF2 KO in thymo and BMM ") +
  annotate(geom = "text", x = 1,  y = 1, label = resting, color = "black") +
  annotate(geom = "text", x = 2,  y = 1, label = activated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

# linc + UPF2 without activated cells -------------------------------------
Up_UPF2ko_BMM <- read.csv("data/Upregulated_UPF2ko_mouse_BMM.csv")
Up_UPF2ko_BMM <- as.character(Up_UPF2ko_BMM[, 1])
Up_UPF2ko_thymo <- read.csv("data/Upregulated_UPF2ko_mouse_thymocytes.csv")
Up_UPF2ko_thymo <- as.character(Up_UPF2ko_thymo[, 1])

db_linc <- "data/databases/current_subset_lincRNA.csv"
table_linc <- as_tibble(read.csv(db_linc))

to_remove <- c("X",
               "transcript_id",
               "gene_name",
               "lincRNA")

table_linc <- table_linc %>% select(- to_remove)
table_linc$type <- "linc RNA"
table_linc$x <- "linc RNA"

table_coding <- table[,colnames(table) %in% colnames(table_linc)]
table_coding$type <- "protein-coding RNA"
table_coding$x <- "linc RNA"

table_upf2 <- table[table$gene_name %in% c(Up_UPF2ko_BMM, Up_UPF2ko_thymo),colnames(table) %in% colnames(table_linc)]
table_upf2$type <- "UPF2 regulated"
table_upf2$x <- "UPF2 regulated"

table_coding2 <- table_coding
table_coding2$x  <- "UPF2 regulated"

double_table <- rbind(table_coding, table_linc, table_upf2, table_coding2)

### TDD at 3h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h,
         type,
         x)

colnames(data) <- colnames(data) <- c("TDD_index", "type", "x")#, "Resting - 1h", "Activated - 1h", "type")
# data <- data %>% pivot_longer(-type, values_to = "TDD_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA", "UPF2 regulated"))
data$x <- factor(data$x, levels = c("linc RNA", "UPF2 regulated"))

linc_RNA <- paste("p-val = ", formatC(pval[["linc_Resting_AbsTDD3hCHX"]], format = "e", digits = 2))
UPF2_regulated <- paste("p-val = ", formatC(pval[["upf2_Resting_AbsTDD3hCHX"]], format = "e", digits = 2))


ggplot(data = data , aes(x = x, y = TDD_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + scale_fill_manual(values = c(color_activated, color_resting, color_3h)) +
  ylim(-0.5,1) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1, label = linc_RNA, color = "black") +
  annotate(geom = "text", x = 2,  y = 1, label = UPF2_regulated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TDD index at 3h CHX")

### TID at 3h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h,
         type,
         x)

colnames(data) <- colnames(data) <- c("TID_index", "type", "x")#, "Resting - 1h", "Activated - 1h", "type")
# data <- data %>% pivot_longer(-type, values_to = "TDD_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA", "UPF2 regulated"))
data$x <- factor(data$x, levels = c("linc RNA", "UPF2 regulated"))

linc_RNA <- paste("p-val = ", formatC(pval[["linc_Resting_AbsNonTDD3hCHX"]], format = "e", digits = 2))
UPF2_regulated <- paste("p-val = ", formatC(pval[["upf2_Resting_AbsNonTDD3hCHX"]], format = "e", digits = 2))


ggplot(data = data , aes(x = x, y = TID_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + scale_fill_manual(values = c(color_activated, color_resting, color_3h)) +
  ylim(-0.5,1.2) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1.15, label = linc_RNA, color = "black") +
  annotate(geom = "text", x = 2,  y = 1.15, label = UPF2_regulated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TID index at 3h CHX")

### TDD at 1h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h,
         type,
         x)

colnames(data) <- colnames(data) <- c("TDD_index", "type", "x")#, "Resting - 1h", "Activated - 1h", "type")
# data <- data %>% pivot_longer(-type, values_to = "TDD_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA", "UPF2 regulated"))
data$x <- factor(data$x, levels = c("linc RNA", "UPF2 regulated"))

linc_RNA <- paste("p-val = ", formatC(pval[["linc_Resting_AbsTDD1hCHX"]], format = "e", digits = 2))
UPF2_regulated <- paste("p-val = ", formatC(pval[["upf2_Resting_AbsTDD1hCHX"]], format = "e", digits = 2))


ggplot(data = data , aes(x = x, y = TDD_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + scale_fill_manual(values = c(color_activated, color_resting, color_3h)) +
  ylim(-0.5,1) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1, label = linc_RNA, color = "black") +
  annotate(geom = "text", x = 2,  y = 1, label = UPF2_regulated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TDD index at 1h CHX")

### TID at 1h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h,
         type,
         x)

colnames(data) <- colnames(data) <- c("TID_index", "type", "x")#, "Resting - 1h", "Activated - 1h", "type")
# data <- data %>% pivot_longer(-type, values_to = "TDD_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA", "UPF2 regulated"))
data$x <- factor(data$x, levels = c("linc RNA", "UPF2 regulated"))

linc_RNA <- paste("p-val = ", formatC(pval[["linc_Resting_AbsNonTDD1hCHX"]], format = "e", digits = 2))
UPF2_regulated <- paste("p-val = ", formatC(pval[["upf2_Resting_AbsNonTDD1hCHX"]], format = "e", digits = 2))


ggplot(data = data , aes(x = x, y = TID_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + scale_fill_manual(values = c(color_activated, color_resting, color_3h)) +
  ylim(-0.5,1.2) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1.15, label = linc_RNA, color = "black") +
  annotate(geom = "text", x = 2,  y = 1.15, label = UPF2_regulated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TID index at 1h CHX")


# lincRNA -----------------------------------------------------------------

#path to the linc database
db_linc <- "data/databases/current_subset_lincRNA.csv"
table_linc <- as_tibble(read.csv(db_linc))

#create double database
to_remove <- c("X",
               "transcript_id",
               "gene_name",
               "lincRNA")

table_linc <- table_linc %>% select(- to_remove)
table_linc$type <- "linc RNA"

table_coding <- table[,colnames(table) %in% colnames(table_linc)]
table_coding$type <- "protein-coding RNA"

double_table <- rbind(table_coding, table_linc)

### TDD at 1h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                (Lympho_Activated.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h,
         Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h,
         type)

colnames(data) <- colnames(data) <- c("Resting", "Activated", "type")#, "Resting - 1h", "Activated - 1h", "type")
data <- data %>% pivot_longer(-type, values_to = "TDD_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA"))
data$condition <- factor(data$condition, levels = c("Resting", "Activated"))

resting <- paste("p-val = ", formatC(pval[["linc_Resting_AbsTDD1hCHX"]], format = "e", digits = 2))
activated <- paste("p-val = ", formatC(pval[["linc_Activated_AbsTDD1hCHX"]], format = "e", digits = 2))


ggplot(data = data , aes(x = condition, y = TDD_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + scale_fill_manual(values = c(color_activated, color_resting)) +
  ylim(-0.5,1) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1, label = resting, color = "black") +
  annotate(geom = "text", x = 2,  y = 1, label = activated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
      panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TDD index in protein-coding RNA vs in linc RNA at 1h CHX")

### TID at 1h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                  (Lympho_Activated.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.1h,
         Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.1h,
         type)

colnames(data) <- colnames(data) <- c("Resting", "Activated", "type")#, "Resting - 1h", "Activated - 1h", "type")
data <- data %>% pivot_longer(-type, values_to = "TID_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA"))
data$condition <- factor(data$condition, levels = c("Resting", "Activated"))

resting <- paste("p-val = ", formatC(pval[["linc_Resting_AbsNonTDD1hCHX"]], format = "e", digits = 2))
activated <- paste("p-val = ", formatC(pval[["linc_Activated_AbsNonTDD1hCHX"]], format = "e", digits = 2))

ggplot(data = data , aes(x = condition, y = TID_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + 
  scale_fill_manual(values = c(color_activated, color_resting)) +
  ylim(-0.5,1.2) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1.2, label = resting, color = "black") +
  annotate(geom = "text", x = 2,  y = 1.2, label = activated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TID index in protein-coding RNA vs in linc RNA at 1h CHX")

### TDD at 3h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                  (Lympho_Activated.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h,
         Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h,
         type)

colnames(data) <- colnames(data) <- c("Resting", "Activated", "type")#, "Resting - 1h", "Activated - 1h", "type")
data <- data %>% pivot_longer(-type, values_to = "TDD_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA"))
data$condition <- factor(data$condition, levels = c("Resting", "Activated"))

resting <- paste("p-val = ", formatC(pval[["linc_Resting_AbsTDD3hCHX"]], format = "e", digits = 2))
activated <- paste("p-val = ", formatC(pval[["linc_Activated_AbsTDD3hCHX"]], format = "e", digits = 2))


ggplot(data = data , aes(x = condition, y = TDD_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + scale_fill_manual(values = c(color_activated, color_resting)) +
  ylim(-0.5,1) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1, label = resting, color = "black") +
  annotate(geom = "text", x = 2,  y = 1, label = activated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TDD index in protein-coding RNA vs in linc RNA at 3h CHX")

### TID at 3h CHX
data <- double_table %>% filter((Lympho_Resting.GOLD == 1) |
                                  (Lympho_Activated.GOLD == 1) |
                                  type == "linc RNA") %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h,
         Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h,
         type)

colnames(data) <- colnames(data) <- c("Resting", "Activated", "type")#, "Resting - 1h", "Activated - 1h", "type")
data <- data %>% pivot_longer(-type, values_to = "TID_index", names_to = "condition")
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA"))
data$condition <- factor(data$condition, levels = c("Resting", "Activated"))

resting <- paste("p-val = ", formatC(pval[["linc_Resting_AbsNonTDD3hCHX"]], format = "e", digits = 2))
activated <- paste("p-val = ", formatC(pval[["linc_Activated_AbsNonTDD3hCHX"]], format = "e", digits = 2))


ggplot(data = data , aes(x = condition, y = TID_index, fill = type)) + 
  geom_boxplot(outlier.size = -1, alpha = 0.8, width = 0.5) + scale_fill_manual(values = c(color_activated, color_resting)) +
  ylim(-0.5,1.2) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 1.2, label = resting, color = "black") +
  annotate(geom = "text", x = 2,  y = 1.2, label = activated, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Absolute TID index in protein-coding RNA vs in linc RNA at 3h CHX")


#TE of linc
TE_linc <- as_tibble(x = read.csv("results/RiboDensity_lincRNA.csv")) %>%
                    select(EnsemblID, 
                           RiboDens_LymphoR,
                           RiboDens_LymphoA)
TE_linc$type <- "linc RNA"

TE_coding <- table %>% filter((Lympho_Resting.GOLD == 1) |
                              (Lympho_Activated.GOLD == 1)) %>%
                       select(gene_id, 
                              RiboDens.Lympho_Resting, 
                              RiboDens.Lympho_Activated)
TE_coding$type <- "protein-coding RNA"

colnames(TE_linc) <- colnames(TE_coding)
double_TE <- rbind(TE_coding[,-1], TE_linc[,-1]) 

colnames(double_TE) <- c("Resting", "Activated", "type")
data <- double_TE %>% pivot_longer(-type, values_to = "TE", names_to = "condition")
data <- data %>% filter(is.finite(TE))
data$condition <- factor(data$condition, levels = c("Resting", "Activated"))
data$type <- factor(data$type, levels = c("protein-coding RNA", "linc RNA"))

resting <- paste("p-val = ", formatC(pval[["linc_Resting_RiboDens"]], format = "e", digits = 2))
activated <- paste("p-val = ", formatC(pval[["linc_Activated_RiboDens"]], format = "e", digits = 2))

ggplot(data = data, aes(x = condition, y = log(1 + TE), fill = type)) + 
  geom_boxplot( outlier.alpha = 0.3) +
  ylim(-0.5,4) + #geom_hline(yintercept = 0, alpha = 0.6, linetype = "dashed") +
  annotate(geom = "text", x = 1,  y = 4, label = resting, color = "black") +
  annotate(geom = "text", x = 2,  y = 4, label = activated, color = "black") +
  scale_fill_manual(values = c(color_activated, color_resting)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  ggtitle("Translation Efficiency (TE) in protein-coding RNA vs in linc RNA")

data <- merge(table_linc, TE_linc)

ggplot(data = data) + 
  geom_point(aes(x = RiboDens.Lympho_Resting, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + xlim(0,10) + ylim(-2,1)

# dev.off()


# T-plots -----------------------------------------------------------------

res <- read.csv("results/DESeqresults_lympho_activation_untreated.csv")
TDDindexes <- dplyr::select(
  table,
  "gene_id",
  "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h",
  "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h",
  "RiboDens.Lympho_Resting",
  "RiboDens.Lympho_Activated",
  "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h",
  "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h",
  "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h",
  "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"
)
TDDindexes$RiboDens.Lympho_Resting <-
  as.numeric(as.character(TDDindexes$RiboDens.Lympho_Resting))
TDDindexes$RiboDens.Lympho_Activated <-
  as.numeric(as.character(TDDindexes$RiboDens.Lympho_Activated))


res2 <- as.data.frame(res)
res2 <- res2[!is.na(res2$log2FoldChange),]
colnames(res2)[1] <- "gene_id"
res2$gene_id <- str_extract(res2$gene_id, "ENSMUSG...........")
res2 <- merge(res2, TDDindexes, by = "gene_id")
res2 <- res2 %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] | 
                        gene_id %in% expressedGenes[["Lympho_Activated"]])

###### With CHX
res2$deltaTDD <-(res2$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
res2$deltaRiboDens <-(res2$RiboDens.Lympho_Activated - res2$RiboDens.Lympho_Resting)
res2$deltaDeg <- res2$DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h - res2$DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h
res2$deltaNonTDD <- res2$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h

#"#D55E00"
r_cor <- cor(res2$deltaTDD, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

p1 <- ggplot(data = res2, aes(x = log2FoldChange, y = deltaTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange > 0),
             aes(x = log2FoldChange, y = deltaTDD),
             alpha = 0.5,
             color = color_resting) +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange < 0),
             aes(x = log2FoldChange, y = deltaTDD),
             alpha = 0.95,
             color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1)  + xlim(-3,+3) +
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

r_cor <- cor(res2$deltaNonTDD, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ", round(r_cor, digits = 3))
p3 <- ggplot(data = res2, aes(x = log2FoldChange, y = deltaNonTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange > 0),
              aes(x = log2FoldChange, y = deltaNonTDD),
              alpha = 0.5,
              color = color_resting) +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange < 0),
              aes(x = log2FoldChange, y = deltaNonTDD),
              alpha = 0.95,
              color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1) + xlim(-3,+3) +
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))


r_cor <- cor(res2$deltaDeg, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

p5 <- ggplot(data = res2, aes(x = log2FoldChange, y = deltaDeg)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point( data = filter(res2, padj < 0.05 & log2FoldChange > 0),
              aes(x = log2FoldChange, y = deltaDeg),
              alpha = 0.5,
              color = color_resting) +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange < 0),
              aes(x = log2FoldChange, y = deltaDeg),
              alpha = 0.95,
              color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + xlim(-3,+3) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))


suppressWarnings(grid.arrange(p1,p3,p5, ncol = 1))

###### With Harr
res2$deltaTDD <- res2$Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res2$Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h
res2$deltaDeg <- res2$DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h - res2$DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h
res2$deltaNonTDD <- res2$Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res2$Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h


r_cor <- cor(res2$deltaTDD, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

p1 <- ggplot(data = res2, aes(x = log2FoldChange, y = deltaTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange > 0),
              aes(x = log2FoldChange, y = deltaTDD),
              alpha = 0.5,
              color = color_resting) +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange < 0),
              aes(x = log2FoldChange, y = deltaTDD),
              alpha = 0.95,
              color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1)  + xlim(-3,+3) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))


r_cor <- cor(res2$deltaNonTDD, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))
p3 <- ggplot(data = res2, aes(x = log2FoldChange, y = deltaNonTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res2, padj < 0.05 &log2FoldChange > 0),
              aes(x = log2FoldChange, y = deltaNonTDD),
              alpha = 0.5,
              color = color_resting) +
  geom_point(data = filter(res2, padj < 0.05 & log2FoldChange < 0),
              aes(x = log2FoldChange, y = deltaNonTDD),
              alpha = 0.95,
              color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1) + xlim(-3,+3) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))


r_cor <- cor(res2$deltaDeg, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

p5 <- ggplot(data = res2, aes(x = log2FoldChange, y = deltaDeg)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point( data = filter(res2, padj < 0.05 & log2FoldChange > 0),
              aes(x = log2FoldChange, y = deltaDeg),
              alpha = 0.5,
              color = color_resting) +
  geom_point( data = filter(res2, padj < 0.05 & log2FoldChange < 0),
              aes(x = log2FoldChange, y = deltaDeg),
              alpha = 0.95,
              color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + xlim(-3,+3) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))


suppressWarnings(grid.arrange(p1,p3,p5, ncol = 1))

### with RiboDens < 1.5

pdf(file = "final_fig/Tplots_RiboDens_Inf_1.5.pdf",
    width = 8,
    height = 7)

res2$deltaTDD <-(res2$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
res2$deltaRiboDens <-(res2$RiboDens.Lympho_Activated - res2$RiboDens.Lympho_Resting)
res2$deltaDeg <- res2$DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h - res2$DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h
res2$deltaNonTDD <- res2$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h

res3 <- res2 %>% filter(gene_id %in% genes_RiboDens_Inf_1.5)
#"#D55E00"
r_cor <- cor(res3$deltaTDD, res3$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

ggplot(data = res3, aes(x = log2FoldChange, y = deltaTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange > 0),
             aes(x = log2FoldChange, y = deltaTDD),
             alpha = 0.5,
             color = color_resting) +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange < 0),
             aes(x = log2FoldChange, y = deltaTDD),
             alpha = 0.95,
             color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1)  + xlim(-3,+3) +
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

r_cor <- cor(res3$deltaNonTDD, res3$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ", round(r_cor, digits = 3))
ggplot(data = res3, aes(x = log2FoldChange, y = deltaNonTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange > 0),
             aes(x = log2FoldChange, y = deltaNonTDD),
             alpha = 0.5,
             color = color_resting) +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange < 0),
             aes(x = log2FoldChange, y = deltaNonTDD),
             alpha = 0.95,
             color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1) + xlim(-3,+3) +
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))


r_cor <- cor(res3$deltaDeg, res3$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

ggplot(data = res3, aes(x = log2FoldChange, y = deltaDeg)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point( data = filter(res3, padj < 0.05 & log2FoldChange > 0),
              aes(x = log2FoldChange, y = deltaDeg),
              alpha = 0.5,
              color = color_resting) +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange < 0),
             aes(x = log2FoldChange, y = deltaDeg),
             alpha = 0.95,
             color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + xlim(-3,+3) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()

### with RiboDens > 1.5

pdf(file = "final_fig/Tplots_RiboDens_Sup_1.5.pdf",
    width = 8,
    height = 7)

res2$deltaTDD <-(res2$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
res2$deltaRiboDens <-(res2$RiboDens.Lympho_Activated - res2$RiboDens.Lympho_Resting)
res2$deltaDeg <- res2$DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h - res2$DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h
res2$deltaNonTDD <- res2$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h

res3 <- res2 %>% filter(gene_id %in% genes_RiboDens_Sup_1.5)
#"#D55E00"
r_cor <- cor(res3$deltaTDD, res3$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

ggplot(data = res3, aes(x = log2FoldChange, y = deltaTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange > 0),
             aes(x = log2FoldChange, y = deltaTDD),
             alpha = 0.5,
             color = color_resting) +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange < 0),
             aes(x = log2FoldChange, y = deltaTDD),
             alpha = 0.95,
             color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1)  + xlim(-3,+3) +
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

r_cor <- cor(res3$deltaNonTDD, res3$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ", round(r_cor, digits = 3))
ggplot(data = res3, aes(x = log2FoldChange, y = deltaNonTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange > 0),
             aes(x = log2FoldChange, y = deltaNonTDD),
             alpha = 0.5,
             color = color_resting) +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange < 0),
             aes(x = log2FoldChange, y = deltaNonTDD),
             alpha = 0.95,
             color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) + ylim(-1, 1) + xlim(-3,+3) +
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))


r_cor <- cor(res3$deltaDeg, res3$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))

ggplot(data = res3, aes(x = log2FoldChange, y = deltaDeg)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_point( data = filter(res3, padj < 0.05 & log2FoldChange > 0),
              aes(x = log2FoldChange, y = deltaDeg),
              alpha = 0.5,
              color = color_resting) +
  geom_point(data = filter(res3, padj < 0.05 & log2FoldChange < 0),
             aes(x = log2FoldChange, y = deltaDeg),
             alpha = 0.95,
             color = color_activated) +
  annotate(geom = "text", x = -2.5, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + xlim(-3,+3) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()


# delta TDD/TID vs GC% ----------------------------------------------------
res <- read.csv("results/DESeqresults_lympho_activation_untreated.csv")
TDDindexes <- dplyr::select(
  table,
  "gene_id",
  "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h",
  "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h",
  "RiboDens.Lympho_Resting",
  "RiboDens.Lympho_Activated",
  "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h",
  "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h",
  "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h",
  "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h",
  "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h",
  "CDS_GC",
  "GC3_percent",
  "n_exon",
  "Length.utr3"
)
TDDindexes$RiboDens.Lympho_Resting <-
  as.numeric(as.character(TDDindexes$RiboDens.Lympho_Resting))
TDDindexes$RiboDens.Lympho_Activated <-
  as.numeric(as.character(TDDindexes$RiboDens.Lympho_Activated))


res2 <- as.data.frame(res)
res2 <- res2[!is.na(res2$log2FoldChange),]
colnames(res2)[1] <- "gene_id"
res2$gene_id <- str_extract(res2$gene_id, "ENSMUSG...........")
res2 <- merge(res2, TDDindexes, by = "gene_id")
res2 <- res2 %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] | 
                          gene_id %in% expressedGenes[["Lympho_Activated"]])
res2$deltaTDD <- res2$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h
res2$deltaTID <- res2$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h
# T plot delta TDD et delta TID contre %GC et GC3 (avec et sans coloration pour le log2fold change de traduction.

pdf(file = "final_fig/delta_vs_GC.pdf")

r_cor <- cor(res2$CDS_GC, res2$deltaTDD, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))
ggplot(data = res2, aes(x = CDS_GC, y = deltaTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_smooth(method = "glm") +
  annotate(geom = "text", x = 35, y = 1, label = sentence, color = "black") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

ggplot(data = res2, aes(x = CDS_GC, y = deltaTDD, color = log2FoldChange)) +
  geom_point() +
  geom_smooth(method = "glm") +
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  geom_density_2d(color = "black", alpha = 0.2) +
  annotate(geom = "text", x = 35, y = 1, label = sentence, color = "black") +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

r_cor <- cor(res2$GC3_percent, res2$deltaTDD, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))
ggplot(data = res2, aes(x = GC3_percent, y = deltaTDD)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_smooth(method = "glm") +
  annotate(geom = "text", x = 25, y = 1, label = sentence, color = "black") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

ggplot(data = res2, aes(x = GC3_percent, y = deltaTDD, color = log2FoldChange)) +
  geom_point() +
  geom_smooth(method = "glm") +
  annotate(geom = "text", x = 25, y = 1, label = sentence, color = "black") +
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

r_cor <- cor(res2$CDS_GC, res2$deltaTID, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))
ggplot(data = res2, aes(x = CDS_GC, y = deltaTID)) +
  geom_point(alpha = 0.5, color = "grey") +
  annotate(geom = "text", x = 35, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

ggplot(data = res2, aes(x = CDS_GC, y = deltaTID, color = log2FoldChange)) +
  geom_point() +
  geom_smooth(method = "glm") +
  annotate(geom = "text", x = 35, y = 1, label = sentence, color = "black") +
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

r_cor <- cor(res2$GC3_percent, res2$deltaTID, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))
ggplot(data = res2, aes(x = GC3_percent, y = deltaTID)) +
  geom_point(alpha = 0.5, color = "grey") +
  annotate(geom = "text", x = 25, y = 1, label = sentence, color = "black") +
  geom_smooth(method = "glm") +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

ggplot(data = res2, aes(x = GC3_percent, y = deltaTID, color = log2FoldChange)) +
  geom_point() +
  geom_smooth(method = "glm") +
  annotate(geom = "text", x = 25, y = 1, label = sentence, color = "black") +
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  geom_density_2d(color = "black", alpha = 0.2) +
  ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()


# log2FoldChange vs GC ----------------------------------------------------

pdf("final_fig/log2FoldChange_vs_GC.pdf")
r_cor <- cor(res2$CDS_GC, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))
ggplot(data = res2, aes(x = CDS_GC, y = log2FoldChange)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_smooth(method = "glm") +
  annotate(geom = "text", x = 35, y = 5, label = sentence, color = "black") +
  geom_density_2d(color = "black", alpha = 0.2) +
  # ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

r_cor <- cor(res2$GC3_percent, res2$log2FoldChange, use = "complete.obs") 
sentence <- paste0("R = ",
                   round(r_cor, digits = 3))
ggplot(data = res2, aes(x = GC3_percent, y = log2FoldChange)) +
  geom_point(alpha = 0.5, color = "grey") +
  geom_smooth(method = "glm") +
  annotate(geom = "text", x = 25, y = 5, label = sentence, color = "black") +
  geom_density_2d(color = "black", alpha = 0.2) +
  # ylim(-1, 1) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()


# deltaTDD vs log2FoldChange + couleurs features --------------------------

ggplot(data = res2, aes(x = (n_exon), y = deltaTID)) + 
  geom_point(alpha = 0.1) +
  geom_density_2d(color = "black", alpha = 0.5) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_hline(yintercept = 0, linetype = "dotted") + 
  xlim(0,30)+
  geom_smooth() + 
  ggtitle("Lymphocytes, 3h CHX")

ggplot(data = res2, aes(y = (n_exon), x = log2FoldChange)) + 
  geom_point(alpha = 0.1) +
  geom_density_2d(color = "black", alpha = 0.5) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth() + 
  ggtitle("Lymphocytes, 3h CHX")

ggplot(data = res2, aes(x = log2FoldChange, y = deltaTID, color = n_exon)) + 
  geom_point() + 
  # geom_density_2d(color = "black", alpha = 0.3) + 
  scale_color_gradient2(low = "white", high = "darkblue",  mid = "#F8F8FF", midpoint = 11, limits = c(0,22)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9  mid = "#F8F8FF"
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  xlim(-2,2) + 
  ggtitle("Lymphocytes, 3h CHX")

ggplot(data = res2, aes(x = log2FoldChange, y = deltaTDD, color = Length.utr3)) + 
  geom_point() + 
  # geom_density_2d(color = "black", alpha = 0.3) + 
  scale_color_gradient2(low = "blue", high = "red",  mid = "#F8F8FF", midpoint = 1500, limits = c(0,3000)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9  mid = "#F8F8FF"
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes, 3h CHX")

ggplot(data = res2, aes(x = log2FoldChange, y = Length.utr3)) + 
  geom_point(alpha = 0.1) + 
  # geom_density_2d(color = "black", alpha = 0.3) + 
  # scale_color_gradient2(low = "blue", high = "red",  mid = "#F8F8FF", midpoint = 1500, limits = c(0,3000)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9  mid = "#F8F8FF"
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  ylim(0,7000) + 
  ggtitle("Lymphocytes, 3h CHX")


# Boxplots diff translation -----------------------------------------------
up_genes <-
  as.data.frame(res[!is.na(res$padj) &
                      res$padj < 0.05 & res$log2FoldChange > 0, ])

colnames(up_genes)[1] <- "gene_id"
up_genes$gene_id <-
  str_extract(up_genes$gene_id, "ENSMUSG...........")
TDDindexes$gene_id <-
  str_extract(TDDindexes$gene_id, "ENSMUSG...........")
up_genes <- merge(up_genes, TDDindexes, by = "gene_id")
dup <- up_genes[!duplicated(up_genes), "gene_id"]
up_genes <- up_genes[up_genes$gene_id %in% dup, ]

up_genes <- up_genes %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] | 
                                  gene_id %in% expressedGenes[["Lympho_Activated"]])

gdata_up <- melt(up_genes, measure.vars = c(8:11, 14:19))

gdata_up[grep("Activated", gdata_up$variable), "baseMean"] <-
  gdata_up[grep("Activated", gdata_up$variable), "baseMean"] * 2 ^ gdata_up[grep("Activated", gdata_up$variable), "log2FoldChange"]

# down genes

down_genes <-
  res[!is.na(res$padj) & res$padj < 0.05 & res$log2FoldChange < 0, ]

colnames(down_genes)[1] <- "gene_id"
down_genes$gene_id <-
  str_extract(down_genes$gene_id, "ENSMUSG...........")
down_genes <- merge(down_genes, TDDindexes, by = "gene_id")

down_genes <- down_genes %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] | 
                                  gene_id %in% expressedGenes[["Lympho_Activated"]])

gdata_down <- melt(down_genes, measure.vars = c(8:11, 14:19))

gdata_down[grep("Activated", gdata_down$variable), "baseMean"] <-
  gdata_down[grep("Activated", gdata_down$variable), "baseMean"] * 2 ^ gdata_down[grep("Activated", gdata_down$variable), "log2FoldChange"]

# up_genes 3h CHX
sub_gdata_1 <- gdata_up %>% filter(
  variable == "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h" |
    variable == "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"
)

sub_gdata_2 <- gdata_up %>% filter(
  variable == "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h" |
    variable == "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"
)

sub_gdata_3 <- gdata_up %>% filter(
  variable == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h" |
    variable == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"
)

levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Activated"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Activated"

levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h"] <-
  "Activated"

sub_gdata_1$variable <- factor(sub_gdata_1$variable, levels = c("Resting", "Activated"))
sub_gdata_2$variable <- factor(sub_gdata_2$variable, levels = c("Resting", "Activated"))
sub_gdata_3$variable <- factor(sub_gdata_3$variable, levels = c("Resting", "Activated"))

p1 <-
  ggplot(
    data = sub_gdata_1,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute TDD index") +
  geom_boxplot(width = 0.1) + ggtitle("IndexTDDas (3hCHX)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p2 <-
  ggplot(
    data = sub_gdata_2,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute nonTDD index") +
  geom_boxplot(width = 0.1) + ggtitle("Index nonTDD abs (3hCHX)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p3 <- ggplot(
  data = sub_gdata_1,
  aes(x = variable, y = log10(baseMean))
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("log10(Normalized Reads Counts)") +
  geom_boxplot(width = 0.1)  + ggtitle("log10 Normalized Counts ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p4 <- ggplot(
  data = sub_gdata_3,
  aes(x = variable, y = value)
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("degradation rate at 3h") +
  geom_boxplot(width = 0.1)  + ggtitle("Degradation Rate ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p5 <- ggplot(data = sub_gdata_1,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of as TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

p6 <- ggplot(data = sub_gdata_2,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of absolute  non TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

grid.arrange(p1, p2, p3,p4, p5, p6, ncol = 2)

# down genes 3h CHX
sub_gdata_1 <- gdata_down %>% filter(
  variable == "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h" |
    variable == "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"
)

sub_gdata_2 <- gdata_down %>% filter(
  variable == "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h" |
    variable == "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"
)

sub_gdata_3 <- gdata_down %>% filter(
  variable == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h" |
    variable == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"
)

levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Activated"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"] <-
  "Activated"

levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h"] <-
  "Activated"

sub_gdata_1$variable <- factor(sub_gdata_1$variable, levels = c("Resting", "Activated"))
sub_gdata_2$variable <- factor(sub_gdata_2$variable, levels = c("Resting", "Activated"))
sub_gdata_3$variable <- factor(sub_gdata_3$variable, levels = c("Resting", "Activated"))

p1 <-
  ggplot(
    data = sub_gdata_1,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute TDD index") +
  geom_boxplot(width = 0.1) + ggtitle("IndexTDDas (3hCHX)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p2 <-
  ggplot(
    data = sub_gdata_2,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute nonTDD index") +
  geom_boxplot(width = 0.1) + ggtitle("Index nonTDD abs (3hCHX)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p3 <- ggplot(
  data = sub_gdata_1,
  aes(x = variable, y = log10(baseMean))
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("log10(Normalized Reads Counts)") +
  geom_boxplot(width = 0.1)  + ggtitle("log10 Normalized Counts ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p4 <- ggplot(
  data = sub_gdata_3,
  aes(x = variable, y = value)
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("degradation rate at 3h") +
  geom_boxplot(width = 0.1)  + ggtitle("Degradation Rate ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p5 <- ggplot(data = sub_gdata_1,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of as TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

p6 <- ggplot(data = sub_gdata_2,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of absolute  non TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

grid.arrange(p1, p2, p3,p4, p5, p6, ncol = 2)

# up genes 3h Harr 
sub_gdata_1 <- gdata_up %>% filter(
  variable == "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h" |
    variable == "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"
)

sub_gdata_2 <- gdata_up %>% filter(
  variable == "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h" |
    variable == "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"
)

sub_gdata_3 <- gdata_up %>% filter(
  variable == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h" |
    variable == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"
)

levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Activated"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Activated"

levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h"] <-
  "Activated"

sub_gdata_1$variable <- factor(sub_gdata_1$variable, levels = c("Resting", "Activated"))
sub_gdata_2$variable <- factor(sub_gdata_2$variable, levels = c("Resting", "Activated"))
sub_gdata_3$variable <- factor(sub_gdata_3$variable, levels = c("Resting", "Activated"))

p1 <-
  ggplot(
    data = sub_gdata_1,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute TDD index") +
  geom_boxplot(width = 0.1) + ggtitle("IndexTDDas (3hHarr)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p2 <-
  ggplot(
    data = sub_gdata_2,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute nonTDD index") +
  geom_boxplot(width = 0.1) + ggtitle("Index nonTDD abs (3hHarr)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p3 <- ggplot(
  data = sub_gdata_1,
  aes(x = variable, y = log10(baseMean))
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("log10(Normalized Reads Counts)") +
  geom_boxplot(width = 0.1)  + ggtitle("log10 Normalized Counts ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p4 <- ggplot(
  data = sub_gdata_3,
  aes(x = variable, y = value)
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("degradation rate at 3h") +
  geom_boxplot(width = 0.1)  + ggtitle("Degradation Rate ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p5 <- ggplot(data = sub_gdata_1,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of as TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

p6 <- ggplot(data = sub_gdata_2,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of absolute  non TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

grid.arrange(p1, p2, p3,p4, p5, p6, ncol = 2)

# down genes 3h Harr
sub_gdata_1 <- gdata_down %>% filter(
  variable == "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h" |
    variable == "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"
)

sub_gdata_2 <- gdata_down %>% filter(
  variable == "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h" |
    variable == "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"
)

sub_gdata_3 <- gdata_down %>% filter(
  variable == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h" |
    variable == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"
)

levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_1$variable)[levels(sub_gdata_1$variable) == "Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Activated"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_2$variable)[levels(sub_gdata_2$variable) == "Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h"] <-
  "Activated"

levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Resting.Trip.Ref_Trip_0h.3h"] <-
  "Resting"
levels(sub_gdata_3$variable)[levels(sub_gdata_3$variable) == "DegFold.Lympho_Activated.Trip.Ref_Trip_0h.3h"] <-
  "Activated"

sub_gdata_1$variable <- factor(sub_gdata_1$variable, levels = c("Resting", "Activated"))
sub_gdata_2$variable <- factor(sub_gdata_2$variable, levels = c("Resting", "Activated"))
sub_gdata_3$variable <- factor(sub_gdata_3$variable, levels = c("Resting", "Activated"))


p1 <-
  ggplot(
    data = sub_gdata_1,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute TDD index") +
  geom_boxplot(width = 0.1) + ggtitle("IndexTDDas (3hHarr)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p2 <-
  ggplot(
    data = sub_gdata_2,
    aes(x = variable, y = value)
  ) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") + ylab("absolute nonTDD index") +
  geom_boxplot(width = 0.1) + ggtitle("Index nonTDD abs (3hHarr)") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p3 <- ggplot(
  data = sub_gdata_1,
  aes(x = variable, y = log10(baseMean))
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("log10(Normalized Reads Counts)") +
  geom_boxplot(width = 0.1)  + ggtitle("log10 Normalized Counts ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p4 <- ggplot(
  data = sub_gdata_3,
  aes(x = variable, y = value)
) +
  geom_violin(aes(fill = variable), alpha = 0.8) +
  geom_jitter(width = 0.1, alpha = 0.5) + theme(legend.position = "none") +  ylab("degradation rate at 3h") +
  geom_boxplot(width = 0.1)  + ggtitle("Degradation Rate ") + scale_fill_manual(values = c(color_resting, color_activated)) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), panel.background = element_blank(), axis.line = element_line(colour = "black"))

p5 <- ggplot(data = sub_gdata_1,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of as TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

p6 <- ggplot(data = sub_gdata_2,
             aes(fill = variable, x = value)) +
  geom_histogram(
    aes(y = ..density.., fill = variable, ),
    alpha = 0.75,
    binwidth = .015,
    position = "identity"
  ) +
  geom_density(
    alpha = 0.6,
    position = "identity",
    aes(color = variable),
    size = 1
  )  +
  xlim(-0.5, 1.2) +
  xlab("as TDD index") +
  scale_fill_manual(values = c("#69b3a2", "#404080")) +
  scale_color_manual(values = c("#69b3a2", "#404080")) +
  ggtitle("Density of absolute  non TDD index") +
  theme(
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    panel.background = element_blank(),
    axis.line = element_line(colour = "black")
  )

grid.arrange(p1, p2, p3,p4, p5, p6, ncol = 2)


# delta TDD vs delta non TDD ----------------------------------------------

###### With CHX
# res2 contain only expressed genes 
res2$deltaTDD <-
  (
    res2$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h
  )
res2$deltaNonTDD <- res2$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res2$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h

r <- round(cor(x = res2$deltaTDD, y = res2$deltaNonTDD),
           digit = 3)
pdf(file = "final_fig/deltaTDD_vs_deltaNonTDD.pdf",
    width = 8,
    height = 6)

ggplot(data = res2, aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h CHX")

r <- round(cor(x = res2 %>% filter(log2FoldChange > 0) %>% select(deltaTDD), 
               y = res2 %>% filter(log2FoldChange > 0)  %>% select(deltaNonTDD)),
           digit = 3)

ggplot(data = res2 %>% filter(log2FoldChange > 0), aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "grey", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h CHX")

r <- round(cor(x = res2 %>% filter(log2FoldChange < 0) %>% select(deltaTDD), 
               y = res2 %>% filter(log2FoldChange < 0)  %>% select(deltaNonTDD)),
           digit = 3)

ggplot(data = res2 %>% filter(log2FoldChange < 0), aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "grey", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h CHX")


###### With Harr
res2$deltaTDD <- (res2$Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res2$Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h)
res2$deltaNonTDD <- res2$Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res2$Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h

r <- round(cor(x = res2$deltaTDD, y = res2$deltaNonTDD),
           digit = 3)


ggplot(data = res2, aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h Harr")

r <- round(cor(x = res2 %>% filter(log2FoldChange > 0) %>% select(deltaTDD), 
               y = res2 %>% filter(log2FoldChange > 0)  %>% select(deltaNonTDD)),
           digit = 3)

ggplot(data = res2 %>% filter(log2FoldChange > 0), aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "grey", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h Harr")

r <- round(cor(x = res2 %>% filter(log2FoldChange < 0) %>% select(deltaTDD), 
               y = res2 %>% filter(log2FoldChange < 0)  %>% select(deltaNonTDD)),
           digit = 3)

ggplot(data = res2 %>% filter(log2FoldChange < 0), aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "grey", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + 
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h Harr")

### genes with riboDens <1.5
res3 <- res2[res2$gene_id %in% genes_RiboDens_Inf_1.5,]

res3$deltaTDD <-
  (
    res3$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res3$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h
  )
res3$deltaNonTDD <- res3$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res3$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h

r <- round(cor(x = res3$deltaTDD, y = res3$deltaNonTDD),
           digit = 3)

ggplot(data = res3, aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h CHX, genes with RiboDens < 1.5 in activated and resting cells")

res3$deltaTDD <-
  (
    res3$Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res3$Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h
  )
res3$deltaNonTDD <- res3$Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res3$Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h

r <- round(cor(x = res3$deltaTDD, y = res3$deltaNonTDD),
           digit = 3)

ggplot(data = res3, aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h Harr, genes with RiboDens < 1.5 in activated and resting cells")

### genes with riboDens > 1.5
res3 <- res2[res2$gene_id %in% genes_RiboDens_Sup_1.5,]

res3$deltaTDD <-
  (
    res3$Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res3$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h
  )
res3$deltaNonTDD <- res3$Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h - res3$Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h

r <- round(cor(x = res3$deltaTDD, y = res3$deltaNonTDD),
           digit = 3)

ggplot(data = res3, aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h CHX, genes with RiboDens > 1.5 in activated and resting cells")

res3$deltaTDD <-
  (
    res3$Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res3$Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h
  )
res3$deltaNonTDD <- res3$Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h - res3$Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h

r <- round(cor(x = res3$deltaTDD, y = res3$deltaNonTDD),
           digit = 3)

ggplot(data = res3, aes(x = deltaNonTDD, y = deltaTDD, color = log2FoldChange)) + 
  geom_point() + 
  geom_density_2d(color = "black", alpha = 0.2) + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) + #56B4E9
  geom_vline(xintercept = 0, linetype = "dotted") + 
  geom_hline(yintercept = 0, linetype = "dotted") + 
  geom_smooth(method = "lm") +
  geom_text(x = -0.7, y = 0.8, label=paste("pearson correlation : ", r), size = 4, check_overlap = TRUE, color = "black") +
  xlim(-1,1) + ylim (-1,1) +
  ggtitle("Lymphocytes, 3h Harr, genes with RiboDens < 1.5 in activated and resting cells")

dev.off()

# plot exemple calcul index -----------------------------------------------

load("results/dbNormCountsAll_exon.RData")

### Non TDD gene in activated lympho
data_activated <- dball %>% filter(cell == "Lympho" & 
                                  `Activated/Resting` == "Activated" &
                                   ensemblID == "ENSMUSG00000000628.9" &
                                   (treatment == "Trip" | treatment == "TripCHX") )
data_activated <- group_by(data_activated, ensemblID, time, treatment)
data_activated <- summarise(data_activated, mean = mean(normReadsCounts), sem = sd(normReadsCounts)/sqrt(length(unique(replicate))))
data_activated$time <- as.numeric(str_extract(data_activated$time, "."))
data_activated <- rbind(data_activated, data_activated[1,])
data_activated[nrow(data_activated),"treatment"] <- "TripCHX"

tddIndex_activated <- round(unlist(table %>% filter(gene_id == gsub("(^[^.]*)(.*$)", "\\1",  gene)) %>% select(Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)), digits = 2)

library(grid)
grob <- grobTree(textGrob(paste("Absolute TDD index : ", tddIndex_activated), x = 0.1,  y = 0.95, hjust = 0,
                          gp = gpar(col = "black", fontsize = 13, fontface = "italic")))

ggplot() + 
  geom_point(data = data_activated, aes(x = time, y = mean, colour = treatment), alpha = 1, shape = 3) + 
  geom_line(data = data_activated, aes(x = time, y = mean, colour = treatment)) + 
  scale_color_manual(values = c(color_resting, color_activated)) +
  geom_linerange(data = data_activated, aes(x = time, ymax = (mean + sem), ymin = (mean-sem), colour = treatment)) + 
  xlab("Time (h)") + 
  ylab("Relative abundance") + 
  ggtitle("ENSMUSG00000000628 in activated lymphocytes") + 
  annotation_custom(grob) + ylim(0, 70) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

### TDD gene in activated lympho
data_activated <- dball %>% filter(cell == "Lympho" & 
                                     `Activated/Resting` == "Activated" &
                                     ensemblID == "ENSMUSG00000063884.5" &
                                     (treatment == "Trip" | treatment == "TripCHX") )
data_activated <- group_by(data_activated, ensemblID, time, treatment)
data_activated <- summarise(data_activated, mean = mean(normReadsCounts), sem = sd(normReadsCounts)/sqrt(length(unique(replicate))))
data_activated$time <- as.numeric(str_extract(data_activated$time, "."))
data_activated <- rbind(data_activated, data_activated[1,])
data_activated[nrow(data_activated),"treatment"] <- "TripCHX"

tddIndex_activated <- round(unlist(table %>% filter(gene_id == "ENSMUSG00000063884") %>% select(Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)), digits = 2)
grob <- grobTree(textGrob(paste("Absolute TDD index : ", tddIndex_activated), x = 0.1,  y = 0.95, hjust = 0,
                          gp = gpar(col = "black", fontsize = 13, fontface = "italic")))

ggplot() + 
  geom_point(data = data_activated, aes(x = time, y = mean, colour = treatment), alpha = 1, shape = 3) + 
  geom_line(data = data_activated, aes(x = time, y = mean, colour = treatment)) + 
  scale_color_manual(values = c(color_resting, color_activated)) +
  geom_linerange(data = data_activated, aes(x = time, ymax = (mean + sem), ymin = (mean-sem), colour = treatment)) + 
  xlab("Time (h)") + 
  ylab("Relative abundance") + 
  ggtitle("ENSMUSG00000063884 in activated lymphocytes") + 
  annotation_custom(grob) + ylim(0, 300) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 
             

### middle TDD gene in activated lympho
data_activated <- dball %>% filter(cell == "Lympho" & 
                                     `Activated/Resting` == "Activated" &
                                     ensemblID == "ENSMUSG00000001416.12" &
                                     (treatment == "Trip" | treatment == "TripCHX") )
data_activated <- group_by(data_activated, ensemblID, time, treatment)
data_activated <- summarise(data_activated, mean = mean(normReadsCounts), sem = sd(normReadsCounts)/sqrt(length(unique(replicate))))
data_activated$time <- as.numeric(str_extract(data_activated$time, "."))
data_activated <- rbind(data_activated, data_activated[1,])
data_activated[nrow(data_activated),"treatment"] <- "TripCHX"

tddIndex_activated <- round(unlist(table %>% filter(gene_id == "ENSMUSG00000001416") %>% select(Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)), digits = 2)

nontddIndex_activated <- round(unlist(table %>% filter(gene_id == "ENSMUSG00000001416") %>% select(Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)), digits = 2)


library(grid)
grob <- grobTree(textGrob(paste("Absolute TDD index : ", tddIndex_activated, "\nAbsolutde Non TDD index : ",nontddIndex_activated), x = 0.1,  y = 0.95, hjust = 0,
                          gp = gpar(col = "black", fontsize = 13, fontface = "italic")))

ggplot() + 
  geom_point(data = data_activated, aes(x = time, y = mean, colour = treatment), alpha = 1, shape = 3) + 
  geom_line(data = data_activated, aes(x = time, y = mean, colour = treatment)) + 
  scale_color_manual(values = c(color_resting, color_activated)) +
  geom_linerange(data = data_activated, aes(x = time, ymax = (mean + sem), ymin = (mean-sem), colour = treatment)) + 
  xlab("Time (h)") + 
  ylab("Relative abundance") + 
  ggtitle("ENSMUSG00000001416 in activated lymphocytes") + 
  annotation_custom(grob) + ylim(0,1300) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 



# plot for Diff Exp ----------------------------------------------------

### Expression 3h post-activation
res3h <- read.csv("~/RMI2/gitlab/tdd/results/DiffExpr_lympho_activation3h.csv")
data <- as.data.frame(res3h)
data$sign <- "n.s"
data[data$padj < 0.05 & !is.na(data$padj),"sign"] <- "sign"
data$ActivatedMean <- data$baseMean * 2^data$log2FoldChange

p1 <- ggplot(data = data, aes(x = log10(baseMean), y = log2FoldChange, color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

p2 <- ggplot(data = data, aes(x = log2FoldChange, y = -log(padj), color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_hline(yintercept = -log(0.05), linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

p3 <- ggplot(data = data, aes(x = log10(baseMean), y = log10(ActivatedMean), color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

grid.arrange(p1, p2, p3, ncol = 3)

### Expression 6h post-activation
res6h <- read.csv("~/RMI2/gitlab/tdd/results/DiffExpr_lympho_activation6h.csv")
data <- as.data.frame(res6h)
data$sign <- "n.s"
data[data$padj < 0.05 & !is.na(data$padj),"sign"] <- "sign"
data$ActivatedMean <- data$baseMean * 2^data$log2FoldChange

p1 <- ggplot(data = data, aes(x = log10(baseMean), y = log2FoldChange, color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

p2 <- ggplot(data = data, aes(x = log2FoldChange, y = -log(padj), color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_hline(yintercept = -log(0.05), linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

p3 <- ggplot(data = data, aes(x = log10(baseMean), y = log10(ActivatedMean), color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

grid.arrange(p1, p2, p3, ncol = 3)

res <- read.csv("~/RMI2/gitlab/tdd/results/DESeqresults_lympho_activation_untreated.csv")
data <- as.data.frame(res)
data$sign <- "n.s"
data[data$padj < 0.05 & !is.na(data$padj),"sign"] <- "sign"
data$ActivatedMean <- data$baseMean * 2^data$log2FoldChange

p1 <- ggplot(data = data, aes(x = log10(baseMean), y = log2FoldChange, color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_hline(yintercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

p2 <- ggplot(data = data, aes(x = log2FoldChange, y = -log(padj), color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_hline(yintercept = -log(0.05), linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

p3 <- ggplot(data = data, aes(x = log10(baseMean), y = log10(ActivatedMean), color = sign)) + 
  geom_point(size = 0.9) + 
  scale_color_manual(values = c("grey", color_activated)) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) 

grid.arrange(p1, p2, p3, ncol = 3)

### plot counts
source("src/DEseq_analysis.R")

data <- as.data.frame(counts(dds3h, normalized = TRUE))
data$gene_id <- rownames(data)
data <- pivot_longer(data, cols = c(1:6), names_to = "condition")
data[grep("Resting", data$condition), "condition"] <- "Resting"
data[grep("Activated", data$condition), "condition"] <- "Activated"
data$condition <- factor(data$condition, levels = c("Resting", "Activated"))

pdf(file = "final_fig/plotCounts.pdf")
ymax = max(data[data$gene_id == "ENSMUSG00000032097.9","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000032097.9"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0, w = 0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("DDX6 - RNAseq") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

ymax = max(data[data$gene_id == "ENSMUSG00000000838.16","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000000838.16"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0, w = 0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("FMR1 - RNAseq") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

ymax = max(data[data$gene_id == "ENSMUSG00000001016.11","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000001016.11"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0, w = 0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("ILF2 - RNAseq") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

ymax = max(data[data$gene_id == "ENSMUSG00000032178.12","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000032178.12"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0,w=0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("ILF3 - RNAseq") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

## Ribosome Profiling
data <- as.data.frame(counts(dds, normalized = TRUE))
data <- data %>% select(1:6)
data$gene_id <- rownames(data)
data <- pivot_longer(data, cols = c(1:6), names_to = "condition")
data[grep("r", data$condition), "condition"] <- "Resting"
data[grep("a", data$condition), "condition"] <- "Activated"
data$condition <- factor(data$condition, levels = c("Resting", "Activated"))

ymax = max(data[data$gene_id == "ENSMUSG00000032097.9","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000032097.9"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0, w = 0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("DDX6 - Ribosome Profiling") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

ymax = max(data[data$gene_id == "ENSMUSG00000000838.16","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000000838.16"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0.01, w = 0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("FMR1 - Ribosome Profiling") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

ymax = max(data[data$gene_id == "ENSMUSG00000001016.11","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000001016.11"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0, w = 0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("ILF2 - Ribosome Profiling") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

ymax = max(data[data$gene_id == "ENSMUSG00000032178.12","value"])
ymax <- ymax + ymax * 0.05
ggplot(data = data %>% filter(gene_id == "ENSMUSG00000032178.12"), aes(x = condition, y = value, color = condition)) + 
  geom_point(position=position_jitter(h=0,w=0.05)) + ylim(c(0,ymax)) +
  scale_color_manual(values = c(color_resting, color_activated)) + 
  ggtitle("ILF3 - Ribosome Profiling") + ylab("Normalized Counts") + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()


# Scatter Plot TDD index --------------------------------------------------
pdf(file = "final_fig/correlation_TDD_index.pdf",
    width = 8, 
    height = 6)

cap <- function(x){
  x[x < -1] <- -1
  x[x > 1.5] <- 1.5
  return(x)
}
  
  ## Lympho Resting
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] ) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("CHX", "Harr")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = CHX, y = Harr)) + 
  geom_point(color = color_resting, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Resting - 3h - Triptolide") + 
  xlab("Triptolide - CHX - 3h") + 
  ylab("Triptolide - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

## Lympho Activated
### Triptolide CHX vs Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h, Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("CHX", "Harr")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
               digit = 3)
  
ggplot(data = data, aes(x = CHX, y = Harr)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - Triptolide") + 
  xlab("Triptolide - CHX - 3h") + 
  ylab("Triptolide - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### DRB CHX vs Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.TDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h, Abs.TDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h)
colnames(data) <- c("CHX", "Harr")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = CHX, y = Harr)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - DRB") + 
  xlab("DRB - CHX - 3h") + 
  ylab("DRB - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### DRB CHX vs Trip CHX
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.TDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h, Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h) 
colnames(data) <- c("DRB", "Trip")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = DRB, y = Trip)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - CHX") + 
  xlab("DRB - CHX - 3h") + 
  ylab("Triptolide - CHX - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### DRB Harr vs Trip Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.TDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h, Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("DRB", "Trip")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = DRB, y = Trip)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - Harr") + 
  xlab("DRB - Harr - 3h") + 
  ylab("Triptolide - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### Resting vs Activated Trip CHX
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] &  gene_id %in% expressedGenes[["Lympho_Resting"]]) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("Resting", "Activated")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = Resting, y = Activated)) + 
  geom_point(color = color_3h, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Triptolide - CHX - 3h") + 
  xlab("Resting - Triptolide - CHX") + 
  ylab("Activated - Triptolide - CHX") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### Resting vs Activated Trip Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] & gene_id %in% expressedGenes[["Lympho_Resting"]]) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h, Abs.TDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("Resting", "Activated")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = Resting, y = Activated)) + 
  geom_point(color = color_3h, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Triptolide - Harr - 3h") + 
  xlab("Resting - Triptolide - Harr") + 
  ylab("Activated - Triptolide - Harr") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()

### TID correlation 
pdf(file = "final_fig/correlation_TID_index.pdf",
    width = 8, 
    height = 6)

cap <- function(x){
  x[x < -1] <- -1
  x[x > 1.5] <- 1.5
  return(x)
}

## Lympho Resting
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]] ) %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("CHX", "Harr")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = CHX, y = Harr)) + 
  geom_point(color = color_resting, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Resting - 3h - Triptolide") + 
  xlab("Triptolide - CHX - 3h") + 
  ylab("Triptolide - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

## Lympho Activated
### Triptolide CHX vs Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("CHX", "Harr")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = CHX, y = Harr)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - Triptolide") + 
  xlab("Triptolide - CHX - 3h") + 
  ylab("Triptolide - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### DRB CHX vs Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.NonTDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h, Abs.NonTDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h)
colnames(data) <- c("CHX", "Harr")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = CHX, y = Harr)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - DRB") + 
  xlab("DRB - CHX - 3h") + 
  ylab("DRB - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### DRB CHX vs Trip CHX
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.NonTDD..Lympho_Activated.DRB_CHX.Ref_DRB_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h) 
colnames(data) <- c("DRB", "Trip")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = DRB, y = Trip)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - CHX") + 
  xlab("DRB - CHX - 3h") + 
  ylab("Triptolide - CHX - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### DRB Harr vs Trip Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] ) %>% 
  select(Abs.NonTDD..Lympho_Activated.DRB_Harr.Ref_DRB_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("DRB", "Trip")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = DRB, y = Trip)) + 
  geom_point(color = color_activated, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Activated - 3h - Harr") + 
  xlab("DRB - Harr - 3h") + 
  ylab("Triptolide - Harr - 3h") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### Resting vs Activated Trip CHX
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] &  gene_id %in% expressedGenes[["Lympho_Resting"]]) %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("Resting", "Activated")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = Resting, y = Activated)) + 
  geom_point(color = color_3h, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Triptolide - CHX - 3h") + 
  xlab("Resting - Triptolide - CHX") + 
  ylab("Activated - Triptolide - CHX") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

### Resting vs Activated Trip Harr
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]] & gene_id %in% expressedGenes[["Lympho_Resting"]]) %>% 
  select(Abs.NonTDD..Lympho_Resting.Trip_Harr.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_Harr.Ref_Trip_0h.3h)
colnames(data) <- c("Resting", "Activated")
data <- as.data.frame(apply(data, 2, cap))
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = Resting, y = Activated)) + 
  geom_point(color = color_3h, alpha = 0.5) + 
  geom_abline(slope = 1, intercept = 0, linetype = "dotted") + 
  ggtitle("Lymphocytes - Triptolide - Harr - 3h") + 
  xlab("Resting - Triptolide - Harr") + 
  ylab("Activated - Triptolide - Harr") + 
  xlim(-0.5,1.5) + 
  ylim(-0.5,1.5) + 
  geom_text(x = -0.2, y = 1.3, label=paste("pearson correlation : ", rsq), size = 5, check_overlap = TRUE) +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()

### TDD vs Non TDD in Resting
library(viridis)
get_density <- function(x, y, ...) {
  dens <- MASS::kde2d(x, y, ...)
  ix <- findInterval(x, dens$x)
  iy <- findInterval(y, dens$y)
  ii <- cbind(ix, iy)
  return(dens$z[ii])
}
  
pdf(file = "final_fig/TDD_vs_NonTDD.pdf",
    width = 8,
    height = 7.4)
  
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("TDD", "NonTDD")
data <- as.data.frame(apply(data, 2, cap))
data$density <- get_density(data$NonTDD, data$TDD, n = 100)
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = NonTDD, y = TDD, color = density)) + 
  geom_point(alpha = 0.5) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  ggtitle("Lymphocytes - Resting - Triptolide - CHX - 3h") + 
  xlab("Absolute Non TDD index") + 
  ylab("Absolute TDD index") + 
  scale_color_viridis() +
  geom_density_2d(color = "black", alpha = 0.5) +
  xlim(-0.5,1.3) +
  ylim(-0.5,1.3) +
  geom_text(x = 0.5, y = 1.3, label=paste("pearson correlation : ", rsq), size = 4, check_overlap = TRUE, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))
  
### TDD vs Non TDD in Activated
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]]) %>% 
  select(Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("TDD", "NonTDD")
data <- as.data.frame(apply(data, 2, cap))
data$density <- get_density(data$NonTDD, data$TDD, n = 100)
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = NonTDD, y = TDD, color = density)) + 
  geom_point(alpha = 0.5) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  ggtitle("Lymphocytes - Activated - Triptolide - CHX - 3h") + 
  xlab("Absolute Non TDD index") + 
  ylab("Absolute TDD index") + 
  scale_color_viridis() +
  geom_density_2d(color = "black", alpha = 0.5) +
  xlim(-0.5,1.3) + 
  ylim(-0.5,1.3) + 
  geom_text(x = 0.5, y = 1.3, label=paste("pearson correlation : ", rsq), size = 4, check_overlap = TRUE, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

# RiboDens inf 1.5 
data <- table %>% filter(gene_id %in% genes_RiboDens_Inf_1.5) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("TDD", "NonTDD")
data <- as.data.frame(apply(data, 2, cap))
data$density <- get_density(data$NonTDD, data$TDD, n = 100)
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = NonTDD, y = TDD, color = density)) + 
  geom_point(alpha = 0.5) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  ggtitle("Lymphocytes - Resting - Triptolide - CHX - 3h - Genes RiboDens Inf 1.5") + 
  xlab("Absolute Non TDD index") + 
  ylab("Absolute TDD index") + 
  scale_color_viridis() +
  geom_density_2d(color = "black", alpha = 0.5) +
  xlim(-0.5,1.3) +
  ylim(-0.5,1.3) +
  geom_text(x = 0.5, y = 1.3, label=paste("pearson correlation : ", rsq), size = 4, check_overlap = TRUE, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

data <- table %>% filter(gene_id %in% genes_RiboDens_Inf_1.5) %>% 
  select(Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("TDD", "NonTDD")
data <- as.data.frame(apply(data, 2, cap))
data$density <- get_density(data$NonTDD, data$TDD, n = 100)
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = NonTDD, y = TDD, color = density)) + 
  geom_point(alpha = 0.5) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  ggtitle("Lymphocytes - Activated - Triptolide - CHX - 3h - Genes RiboDens Inf 1.5") + 
  xlab("Absolute Non TDD index") + 
  ylab("Absolute TDD index") + 
  scale_color_viridis() +
  geom_density_2d(color = "black", alpha = 0.5) +
  xlim(-0.5,1.3) + 
  ylim(-0.5,1.3) + 
  geom_text(x = 0.5, y = 1.3, label=paste("pearson correlation : ", rsq), size = 4, check_overlap = TRUE, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

## RiboDens Sup 1.5 
data <- table %>% filter(gene_id %in% genes_RiboDens_Sup_1.5) %>% 
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("TDD", "NonTDD")
data <- as.data.frame(apply(data, 2, cap))
data$density <- get_density(data$NonTDD, data$TDD, n = 100)
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = NonTDD, y = TDD, color = density)) + 
  geom_point(alpha = 0.5) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  ggtitle("Lymphocytes - Resting - Triptolide - CHX - 3h - Genes RiboDens Sup 1.5") + 
  xlab("Absolute Non TDD index") + 
  ylab("Absolute TDD index") + 
  scale_color_viridis() +
  geom_density_2d(color = "black", alpha = 0.5) +
  xlim(-0.5,1.3) +
  ylim(-0.5,1.3) +
  geom_text(x = 0.5, y = 1.3, label=paste("pearson correlation : ", rsq), size = 4, check_overlap = TRUE, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

data <- table %>% filter(gene_id %in% genes_RiboDens_Sup_1.5) %>% 
  select(Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
colnames(data) <- c("TDD", "NonTDD")
data <- as.data.frame(apply(data, 2, cap))
data$density <- get_density(data$NonTDD, data$TDD, n = 100)
rsq <- round(cor(x= data[,1], y = data[,2]),
             digit = 3)

ggplot(data = data, aes(x = NonTDD, y = TDD, color = density)) + 
  geom_point(alpha = 0.5) + 
  geom_vline(xintercept = 0, linetype = "dotted") +
  geom_hline(yintercept = 0, linetype = "dotted") +
  ggtitle("Lymphocytes - Activated - Triptolide - CHX - 3h - Genes RiboDens Sup 1.5") + 
  xlab("Absolute Non TDD index") + 
  ylab("Absolute TDD index") + 
  scale_color_viridis() +
  geom_density_2d(color = "black", alpha = 0.5) +
  xlim(-0.5,1.3) + 
  ylim(-0.5,1.3) + 
  geom_text(x = 0.5, y = 1.3, label=paste("pearson correlation : ", rsq), size = 4, check_overlap = TRUE, color = "black") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))
dev.off()
  

# Density GO --------------------------------------------------------------

Transcripts_TDD_CHX_GO <- read_csv("data/Transcripts_TDD_CHX_GO.csv")
Transcripts_NonTDD_CHX_GO <- read_csv("data/Transcripts_NonTDD_CHX_GO.csv")

go_number <- c("GO:0000502",
               "GO:0004386",
              "GO:0042254",
              "GO:0000398",
              "GO:0006281",
              "GO:0005743",
              "GO:0000122",
              "GO:0000977",
              "GO:0010468")

go_names <- c("proteasome complex",
              "helicase activity",
              "ribosome biogenesis",
              "mRNA splicing; via spliceosome",
              "DNA repair",
              "mitochondrial inner membrane",
              "negative regulation of transcription by RNA polymerase II",
              "RNA polymerase II regulatory region DNA binding",
              "regulation of gene expression")
pdf('final_fig/distribution_GO.pdf',
    width = 8,
    height = 7)
for (i in 1:length(go_number)){
  genes_go <- Transcripts_TDD_CHX_GO %>% filter(GO_Term == go_number[i] & condition == 'Abs(TDD)>Lympho_Resting>Trip_CHX>Ref_Trip_0h>3h') %>% 
    select(gene_id) %>% 
    unlist()
  genes_go <- gsub("\\.[[:digit:]]*", "", genes_go)
  
  data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>% 
    select(gene_id, Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
  
  data$legend <- "population"
  data[data$gene_id %in% genes_go, "legend"] <- "GO_genes"
  data$legend <- factor(data$legend, levels = c("population", "GO_genes"))
  colnames(data) <- c("gene_id", "TDD_index", "legend")
  # data <- melt(data)
  
  # together
  print(ggplot(data = data, aes(x = TDD_index)) + 
    # geom_histogram(aes(y=..density.., fill = legend), alpha = 0.75, binwidth = .015, position = "identity") +
    geom_density(alpha=0.8, position = "identity",aes(color = legend, fill = legend), size = 1)  + 
    xlim(-0.5,1) + 
    xlab("Absolute TDD index") + 
    geom_vline(xintercept = median(unlist(data %>% filter(legend == "GO_genes") %>% select(TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
    geom_vline(xintercept = median(unlist(data %>% filter(legend == "population") %>% select(TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_resting)  +
    scale_fill_manual(values=c( color_resting, color_activated)) +
    scale_color_manual(values=c( color_resting, color_activated)) + 
    ggtitle(paste(go_number[i], go_names[i], "in resting lymphocytes")) +
    theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
          panel.background = element_blank(), axis.line = element_line(colour = "black")))
  
  ### activated
  genes_go <- Transcripts_TDD_CHX_GO %>% filter(GO_Term == go_number[i] & condition == 'Abs(TDD)>Lympho_Activated>Trip_CHX>Ref_Trip_0h>3h') %>% 
    select(gene_id) %>% 
    unlist()
  genes_go <- gsub("\\.[[:digit:]]*", "", genes_go)
  
  data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]]) %>% 
    select(gene_id, Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
  
  data$legend <- "population"
  data[data$gene_id %in% genes_go, "legend"] <- "GO_genes"
  data$legend <- factor(data$legend, levels = c("population", "GO_genes"))
  colnames(data) <- c("gene_id", "TDD_index", "legend")
  # data <- melt(data)
  
  # together
  print(ggplot(data = data, aes(x = TDD_index)) + 
    # geom_histogram(aes(y=..density.., fill = legend), alpha = 0.75, binwidth = .015, position = "identity") +
    geom_density(alpha=0.8, position = "identity",aes(color = legend, fill = legend), size = 1)  + 
    xlim(-0.5,1) + 
    xlab("Absolute TDD index") + 
    geom_vline(xintercept = median(unlist(data %>% filter(legend == "GO_genes") %>% select(TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
    geom_vline(xintercept = median(unlist(data %>% filter(legend == "population") %>% select(TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_resting)  +
    scale_fill_manual(values=c( color_resting, color_activated)) +
    scale_color_manual(values=c( color_resting, color_activated)) + 
    ggtitle(paste(go_number[i], go_names[i], "in activated lymphocytes")) +
    theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
          panel.background = element_blank(), axis.line = element_line(colour = "black")))
}
dev.off()

## Index Non TDD
go_number <- c("GO:0000981",
               "GO:0000977",
               "GO:0045087",
               "GO:0000381",
               "GO:0016874",
               "GO:0031966")

go_names <- c("DNA-binding transcription factor activity",
              "RNA polymerase II regulatory region DNA binding",
              "innate immune response",
              "regulation of alternative mRNA splicing; via spliceosome",
              "ligase activity",
              "mitochondrial membrane")

pdf('final_fig/distribution_GO_NonTDDindex.pdf',
    width = 8,
    height = 7)
for (i in 1:length(go_number)){
  genes_go <- Transcripts_NonTDD_CHX_GO %>% filter(GO_Term == go_number[i] & condition == 'Abs(NonTDD)>Lympho_Resting>Trip_CHX>Ref_Trip_0h>3h') %>% 
    select(gene_id) %>% 
    unlist()
  genes_go <- gsub("\\.[[:digit:]]*", "", genes_go)
  
  data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>% 
    select(gene_id, Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)
  
  data$legend <- "population"
  data[data$gene_id %in% genes_go, "legend"] <- "GO_genes"
  data$legend <- factor(data$legend, levels = c("population", "GO_genes"))
  colnames(data) <- c("gene_id", "Non_TDD_index", "legend")
  # data <- melt(data)
  
  # together
  print(ggplot(data = data, aes(x = Non_TDD_index)) + 
          # geom_histogram(aes(y=..density.., fill = legend), alpha = 0.75, binwidth = .015, position = "identity") +
          geom_density(alpha=0.8, position = "identity",aes(color = legend, fill = legend), size = 1)  + 
          xlim(-0.5,1.2) + 
          xlab("Absolute Non TDD index") + 
          geom_vline(xintercept = median(unlist(data %>% filter(legend == "GO_genes") %>% select(Non_TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
          geom_vline(xintercept = median(unlist(data %>% filter(legend == "population") %>% select(Non_TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_resting)  +
          scale_fill_manual(values=c( color_resting, color_activated)) +
          scale_color_manual(values=c( color_resting, color_activated)) + 
          ggtitle(paste(go_number[i], go_names[i], "in resting lymphocytes")) +
          theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
                panel.background = element_blank(), axis.line = element_line(colour = "black")))
  
  ### activated
  genes_go <- Transcripts_NonTDD_CHX_GO %>% filter(GO_Term == go_number[i] & condition == 'Abs(NonTDD)>Lympho_Activated>Trip_CHX>Ref_Trip_0h>3h') %>% 
    select(gene_id) %>% 
    unlist()
  genes_go <- gsub("\\.[[:digit:]]*", "", genes_go)
  
  data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Activated"]]) %>% 
    select(gene_id, Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h)
  
  data$legend <- "population"
  data[data$gene_id %in% genes_go, "legend"] <- "GO_genes"
  data$legend <- factor(data$legend, levels = c("population", "GO_genes"))
  colnames(data) <- c("gene_id", "Non_TDD_index", "legend")
  # data <- melt(data)
  
  # together
  print(ggplot(data = data, aes(x = Non_TDD_index)) + 
          # geom_histogram(aes(y=..density.., fill = legend), alpha = 0.75, binwidth = .015, position = "identity") +
          geom_density(alpha=0.8, position = "identity",aes(color = legend, fill = legend), size = 1)  + 
          xlim(-0.5,1.2) + 
          xlab("Absolute Non TDD index") + 
          geom_vline(xintercept = median(unlist(data %>% filter(legend == "GO_genes") %>% select(Non_TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_activated) + 
          geom_vline(xintercept = median(unlist(data %>% filter(legend == "population") %>% select(Non_TDD_index)), na.rm = TRUE), linetype = "dashed", color = color_resting)  +
          scale_fill_manual(values=c( color_resting, color_activated)) +
          scale_color_manual(values=c( color_resting, color_activated)) + 
          ggtitle(paste(go_number[i], go_names[i], "in activated lymphocytes")) +
          theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
                panel.background = element_blank(), axis.line = element_line(colour = "black")))
}
dev.off()


#  plot Delta Non-TDD Vs Delta Ribodens en coloriant par le %GC de --------

CDS_GC <- table %>% filter(gene_id %in% res2$gene_id) %>%
                    select(gene_id, CDS_GC)
data <- merge(res2, CDS_GC, by = "gene_id")

ggplot(data = data, aes(color = deltaNonTDD, y = log2FoldChange, x = CDS_GC)) + 
  geom_point() + 
  scale_color_gradient2(low="darkblue", high="red", mid = "#F8F8FF", midpoint = 0) + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) +
  geom_density_2d(color = "black", alpha = 0.2) 
  

# Random Forest -----------------------------------------------------------

source("src/randomForest.R")

pdf(file = "final_fig/randomForest.pdf",
    width = 12, 
    height = 12)
# Resting
x <- "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])

x <- "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])

# Activated
x <- "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])

x <- "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])

# Delta
x <- "deltaTDD_Lympho"
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])

x <- "deltaNonTDD_Lympho"
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])

dev.off()

pdf(file = "final_fig/randomForest_delta.pdf",
    width = 12, 
    height = 12)
x <-  "deltaTDD_Lympho" 
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])

x <-  "deltaNonTDD_Lympho" 
model <- makeRandomForest(x = x)
model[["rf_model"]]
model[["rf_model"]]$finalModel
suppressWarnings(grid.arrange(model[["training_set"]], model[["valid_set"]], ncol = 1))
suppressWarnings(model[["ImpVar"]])
dev.off()

# boxplots RandomForest ---------------------------------------------------
library(ggpubr)

pdf(file = "final_fig/boxplots_features.pdf")

# number uORF
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>%
                  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, number_uORF) 
data[data$number_uORF >= 8 , "number_uORF"]  <- "8 or more"

dataSE <- summarySE(data, measurevar = "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", groupvars = "number_uORF")
dataSE$number_uORF <- factor(dataSE$number_uORF)
dataSE$N <- paste("n =", dataSE$N)
# ymin <- min(dataSE$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h - dataSE$se, na.rm = T)
# ymax <- max(dataSE$Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h + dataSE$se, na.rm = T)
# 
# ggplot(data = dataSE , aes(x = number_uORF, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + 
#   geom_point(aes(color = number_uORF)) + 
#   geom_errorbar(aes(ymin = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h - se, 
#                     ymax = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h + se, 
#                     color = number_uORF), 
#                 width = .1) +
#   geom_text(aes(label = N), y = ymax + 0.1*ymax, size = 3.3) + 
#   ylim(ymin, ymax + 0.15*ymax)

my_comparisons <- list( c("0", "1"), c("0", "2"), c("0", "3"), c("0","4"),c("0", "5"),c("0", "6"),c("0", "7"),c("0", "8 or more")  )
data$number_uORF <- factor(data$number_uORF)
ggplot(data = data, aes(x = number_uORF, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + 
  geom_violin(aes(fill = number_uORF)) + geom_boxplot(width=0.2) + 
  geom_text(data = dataSE, aes(label = N), y = 1.4, size = 3.3) +
  # stat_compare_means(comparisons = my_comparisons, label = "p.signif", method = "t.test") +
  theme(legend.position="none")

#uORF v2
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>%
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, number_uORF) 
data[data$number_uORF >= 1 , "number_uORF"]  <- "1 or more"

dataSE <- summarySE(data, measurevar = "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", groupvars = "number_uORF")
dataSE$number_uORF <- factor(dataSE$number_uORF)
dataSE$N <- paste("n =", dataSE$N)

data$number_uORF <- factor(data$number_uORF)
ggplot(data = data, aes(x = number_uORF, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + 
  geom_violin(aes(fill = number_uORF)) + geom_boxplot(width=0.2) + 
  geom_text(data = dataSE, aes(label = N), y = 1.4, size = 3.3) +
  stat_compare_means() +
  theme(legend.position="none")

# n AU-rich motif
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>%
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3) 

data[data$N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3 > 0, "N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3"] <- "1 or more"
dataSE <- summarySE(data, measurevar = "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", groupvars = "N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3")
dataSE$N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3 <- factor(dataSE$N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3)
dataSE$N <- paste("n =", dataSE$N)

ggplot(data = data, aes(x = N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + 
  geom_violin(aes(fill = N_.AU..AU..AU.UAUUUAU.AU..AU..AU._UTR3)) + ylim(-1,2) + geom_boxplot(width=0.2) + 
  stat_compare_means() + 
  geom_text(data = dataSE, aes(label = N), y = 1.6, size = 3.3) + 
  theme(legend.position="none")

# n4G_UTR3
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>%
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, n4G_UTR3) 

data[data$n4G_UTR3 > 0, "n4G_UTR3"] <- "1 or more"
dataSE <- summarySE(data, measurevar = "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", groupvars = "n4G_UTR3")
dataSE$n4G_UTR3 <- factor(dataSE$n4G_UTR3)
dataSE$N <- paste("n =", dataSE$N)

ggplot(data = data, aes(x = n4G_UTR3, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + 
  geom_violin(aes(fill = n4G_UTR3)) + ylim(-1,2) + geom_boxplot(width=0.2) + 
  stat_compare_means() + 
  geom_text(data = dataSE, aes(label = N), y = 1.6, size = 3.3) + 
  theme(legend.position="none")

# n4G_UTR5
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>%
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, n4G_UTR5) 

data[data$n4G_UTR5 > 0, "n4G_UTR5"] <- "1 or more"
dataSE <- summarySE(data, measurevar = "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", groupvars = "n4G_UTR5")
dataSE$n4G_UTR5 <- factor(dataSE$n4G_UTR5)
dataSE$N <- paste("n =", dataSE$N)

ggplot(data = data, aes(x = n4G_UTR5, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + 
  geom_violin(aes(fill = n4G_UTR5)) + ylim(-1,2) + geom_boxplot(width=0.2) + 
  stat_compare_means() + 
  geom_text(data = dataSE, aes(label = N), y = 1.6, size = 3.3) + 
  theme(legend.position="none")

# n4G_CDS
data <- table %>% filter(gene_id %in% expressedGenes[["Lympho_Resting"]]) %>%
  select(Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h, n4G_CDS) 

data[data$n4G_CDS > 0, "n4G_CDS"] <- "1 or more"
dataSE <- summarySE(data, measurevar = "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h", groupvars = "n4G_CDS")
dataSE$n4G_CDS <- factor(dataSE$n4G_CDS)
dataSE$N <- paste("n =", dataSE$N)

ggplot(data = data, aes(x = n4G_CDS, y = Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h)) + 
  geom_violin(aes(fill = n4G_CDS)) + ylim(-1,2) + geom_boxplot(width=0.2) + 
  stat_compare_means() + 
  geom_text(data = dataSE, aes(label = N), y = 1.6, size = 3.3) + 
  theme(legend.position="none")
dev.off()


# CSC figures -------------------------------------------------------------

  source(file = "~/RMI2/gitlab/tdd/src/CSC_functions.R")
  
  color_resting = "#69b3a2"
  color_activated = "#404080"
  names_aa <- c("Ala", "Arg", "Asn", "Asp", "Cys", "Gln", "Glu", "Gly", "His", "Ile", "Leu", "Lys", "Met", "Phe", "Pro", "Ser", "Thr", "Trp", "Tyr", "Val")
  fastaFile <- readDNAStringSet("~/RMI2/gitlab/tdd/data/cds_seq_mm10.txt")
  codon_counts <- count_codon(fastaFile)
  codon_freq <- freq_codon(codon_counts)
  df_codon_freq <- list_to_df(codon_freq)
  df_codon_count <- list_to_df(codon_counts)

## Absolute TDD index

pdf(file = "final_fig/CSC.pdf",
    width = 9,
    height = 9)
index_resting <- "Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"
index_activated <- "Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"

suppressWarnings(rCSC_resting <- get_rCSC(df_codon_freq = df_codon_freq, index = index_resting))
names(rCSC_resting) <- names_codon_aa
suppressWarnings(rCSC_activated <- get_rCSC(df_codon_freq = df_codon_freq, index = index_activated))
names(rCSC_activated) <- names_codon_aa

load("results/pvalCSC_Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h.RData")
load("results/pvalCSC_Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h.RData")

# pval_activated <- pval_resting
# pval_activated$FDR <- 1
# pval_activated[rownames(pval_activated) %in% c("AGC","AAC","CUU","AUU","GAU"), "FDR"] <- 0.005
# pval_activated[rownames(pval_activated) %in% c("UGC","GCU","GCA","AAA","GAA"), "FDR"] <- 0.04
barplot_resting <- barplot_rCSC(rCSC_resting, pval_resting$FDR)
barplot_resting <- barplot_resting + scale_fill_manual(values = color_resting) + guides(fill=FALSE)

barplot_activated <- barplot_rCSC(rCSC_activated, pval_activated$FDR)
barplot_activated <- barplot_activated + scale_fill_manual(values = color_activated) + guides(fill=FALSE)

grid.arrange(barplot_resting, barplot_activated, ncol = 1,
             top = "Absolute TDD ")
dev.off()
pdf(file = "final_fig/CSC_sup.pdf",
    width = 9,
    height = 5)
barplot_sup_rCSC(rCSC_resting, rCSC_activated, pval_resting, pval_activated)
dev.off()

load("results/rCSC_perm_df_Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
perm_CSC_resting <- rCSC_df
perm_CSC_resting$codon <- names_codon_aa
ggrandomResting <- barplot_sup_perm(rCSC = rCSC_resting, perm_rCSC = perm_CSC_resting, pval = pval_resting) 
ggrandomResting <- ggrandomResting + 
                    scale_fill_manual("legend", values = c("real_sequence" = color_resting, "permutated_sequence" = "#6d6d6d")) + 
                    guides(fill=FALSE)

load("results/rCSC_perm_df_Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
perm_CSC_activated <- rCSC_df
perm_CSC_activated$codon <- names_codon_aa
ggrandomActivated <- barplot_sup_perm(rCSC = rCSC_activated, perm_rCSC = perm_CSC_activated, pval = pval_activated) 
ggrandomActivated <- ggrandomActivated + 
                      scale_fill_manual("legend", values = c("real_sequence" = color_activated, "permutated_sequence" = "#6d6d6d")) +
                      guides(fill=FALSE)

pdf(file = "final_fig/CSC_perm.pdf",
    width = 9,
    height = 9)
grid.arrange(ggrandomResting, ggrandomActivated, ncol = 1,
             top = "Absolute TDD")
dev.off()
## Absolute TID index
index_resting <- "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"
index_activated <- "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"

suppressWarnings(rCSC_resting <- get_rCSC(df_codon_freq = df_codon_freq, index = index_resting))
names(rCSC_resting) <- names_codon_aa
suppressWarnings(rCSC_activated <- get_rCSC(df_codon_freq = df_codon_freq, index = index_activated))
names(rCSC_activated) <- names_codon_aa

load("results/pvalCSC_Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h.RData")
load("results/pvalCSC_Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h.RData")

# pval_activated <- pval_resting
# pval_activated$FDR <- 1
# pval_activated[rownames(pval_activated) %in% c("AGC","AAC","CUU","AUU","GAU"), "FDR"] <- 0.005
# pval_activated[rownames(pval_activated) %in% c("UGC","GCU","GCA","AAA","GAA"), "FDR"] <- 0.04
barplot_resting <- barplot_rCSC(rCSC_resting, pval_resting$FDR)
barplot_resting <- barplot_resting + scale_fill_manual(values = color_resting) + guides(fill=FALSE)

barplot_activated <- barplot_rCSC(rCSC_activated, pval_activated$FDR)
barplot_activated <- barplot_activated + scale_fill_manual(values = color_activated) + guides(fill=FALSE)

pdf(file = "final_fig/CSC_TID.pdf",
    width = 9,
    height = 9)
grid.arrange(barplot_resting, barplot_activated, ncol = 1,
             top = "Absolute TID ")
dev.off()

pdf(file = "final_fig/CSC_sup_TID.pdf",
    width = 9,
    height = 5)
barplot_sup_rCSC(rCSC_resting, rCSC_activated, pval_resting, pval_activated) + ggtitle ("Absolute TID index")
dev.off()

load("results/rCSC_perm_df_Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
perm_CSC_resting <- rCSC_df
perm_CSC_resting$codon <- names_codon_aa
ggrandomResting <- barplot_sup_perm(rCSC = rCSC_resting, perm_rCSC = perm_CSC_resting, pval = pval_resting) 
ggrandomResting <- ggrandomResting + 
  scale_fill_manual("legend", values = c("real_sequence" = color_resting, "permutated_sequence" = "#6d6d6d")) + 
  guides(fill=FALSE)

load("results/rCSC_perm_df_Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
perm_CSC_activated <- rCSC_df
perm_CSC_activated$codon <- names_codon_aa
ggrandomActivated <- barplot_sup_perm(rCSC = rCSC_activated, perm_rCSC = perm_CSC_activated, pval = pval_activated) 
ggrandomActivated <- ggrandomActivated + 
  scale_fill_manual("legend", values = c("real_sequence" = color_activated, "permutated_sequence" = "#6d6d6d")) +
  guides(fill=FALSE)

pdf(file = "final_fig/CSC_perm_TID.pdf",
    width = 9,
    height = 9)
grid.arrange(ggrandomResting, ggrandomActivated, ncol = 1,
             top = "Absolute TID")
dev.off()

data_res <- stats_rCSC(rCSC_resting)
data_act <- stats_rCSC(rCSC_activated)
data_res$state <- "resting"
data_act$state <- "activated"
data <- rbind(data_res, data_act)
data$state <- factor(data$state, levels = c("resting", "activated"))
data_miror <- data
data_miror$variable <- gsub("GC", "AT", data_miror$variable)
data_miror$value <- 100 - data$value

data <- rbind(data, data_miror)

neg_res <- ggplot(data = filter(data, 
                     (variable == "GC3_percent" | variable == "AT3_percent") &
                     state == "resting" & rCSC == "negative" ), 
       aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Resting")

neg_act <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "activated" & rCSC == "negative" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Activated")

pos_res <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "resting" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Resting")

pos_act <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "activated" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Activated")

## GC % 
neg_res_GC <- ggplot(data = filter(data, 
                                (variable == "GC_percent" | variable == "AT_percent") &
                                  state == "resting" & rCSC == "negative" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Resting")

neg_act_GC <- ggplot(data = filter(data, 
                                (variable == "GC_percent" | variable == "AT_percent") &
                                  state == "activated" & rCSC == "negative" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Activated")

pos_res_GC <- ggplot(data = filter(data, 
                                (variable == "GC_percent" | variable == "AT_percent") &
                                  state == "resting" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Resting")

pos_act_GC <- ggplot(data = filter(data, 
                                (variable == "GC_percent" | variable == "AT_percent") &
                                  state == "activated" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Activated")

## TID
index_resting <- "Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h"
index_activated <- "Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h"
suppressWarnings(rCSC_resting <- get_rCSC(df_codon_freq = df_codon_freq, index = index_resting))
names(rCSC_resting) <- names_codon_aa
suppressWarnings(rCSC_activated <- get_rCSC(df_codon_freq = df_codon_freq, index = index_activated))
names(rCSC_activated) <- names_codon_aa

data_res <- stats_rCSC(rCSC_resting)
data_act <- stats_rCSC(rCSC_activated)
data_res$state <- "resting"
data_act$state <- "activated"
data <- rbind(data_res, data_act)
data$state <- factor(data$state, levels = c("resting", "activated"))
data_miror <- data
data_miror$variable <- gsub("GC", "AT", data_miror$variable)
data_miror$value <- 100 - data$value

data <- rbind(data, data_miror)

neg_res_TID <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "resting" & rCSC == "negative" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Resting")

neg_act_TID <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "activated" & rCSC == "negative" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Activated")

pos_res_TID <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "resting" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Resting")

pos_act_TID <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "activated" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Activated")

## GC % 
neg_res_TID_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "resting" & rCSC == "negative" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Resting")

neg_act_TID_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "activated" & rCSC == "negative" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Activated")

pos_res_TID_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "resting" & rCSC == "positive" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Resting")

pos_act_TID_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "activated" & rCSC == "positive" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Activated")

pdf(file = "piechart_rCSC_TDD_TID.pdf")
grid.arrange(neg_res, neg_act, neg_res_TID, neg_act_TID, ncol = 2, top = "GC3 percent with negative rCSC codon")
grid.arrange(pos_res, pos_act, pos_res_TID, pos_act_TID, ncol = 2, top = "GC3 percent with positive rCSC codon")
grid.arrange(neg_res_GC, neg_act_GC, neg_res_TID_GC, neg_act_TID_GC, ncol = 2, top = "GC percent with negative rCSC codon")
grid.arrange(pos_res_GC, pos_act_GC, pos_res_TID_GC, pos_act_TID_GC, ncol = 2, top = "GC percent with positive rCSC codon")
dev.off()

###### pieChart from random sequences

load("results/rCSC_perm_df_Abs.TDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
rCSC_resting <- rCSC_df$median
names(rCSC_resting) <- rCSC_df$codon

load("results/rCSC_perm_df_Abs.TDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
rCSC_activated <- rCSC_df$median
names(rCSC_activated) <- rCSC_df$codon

data_res <- stats_rCSC(rCSC_resting)
data_act <- stats_rCSC(rCSC_activated)
data_res$state <- "resting"
data_act$state <- "activated"
data <- rbind(data_res, data_act)
data$state <- factor(data$state, levels = c("resting", "activated"))
data_miror <- data
data_miror$variable <- gsub("GC", "AT", data_miror$variable)
data_miror$value <- 100 - data$value

data <- rbind(data, data_miror)

neg_res <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "resting" & rCSC == "negative" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Resting")

neg_act <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "activated" & rCSC == "negative" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Activated")

pos_res <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "resting" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Resting")

pos_act <- ggplot(data = filter(data, 
                                (variable == "GC3_percent" | variable == "AT3_percent") &
                                  state == "activated" & rCSC == "positive" ), 
                  aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Activated")

## GC % 
neg_res_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "resting" & rCSC == "negative" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Resting")

neg_act_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "activated" & rCSC == "negative" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC negative - Activated")

pos_res_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "resting" & rCSC == "positive" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Resting")

pos_act_GC <- ggplot(data = filter(data, 
                                   (variable == "GC_percent" | variable == "AT_percent") &
                                     state == "activated" & rCSC == "positive" ), 
                     aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TDD - rCSC positive - Activated")

## TID

load("results/rCSC_perm_df_Abs.NonTDD..Lympho_Resting.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
rCSC_resting <- rCSC_df$median
names(rCSC_resting) <- rCSC_df$codon

load("results/rCSC_perm_df_Abs.NonTDD..Lympho_Activated.Trip_CHX.Ref_Trip_0h.3h_10000perm.RData")
rCSC_activated <- rCSC_df$median
names(rCSC_activated) <- rCSC_df$codon

data_res <- stats_rCSC(rCSC_resting)
data_act <- stats_rCSC(rCSC_activated)
data_res$state <- "resting"
data_act$state <- "activated"
data <- rbind(data_res, data_act)
data$state <- factor(data$state, levels = c("resting", "activated"))
data_miror <- data
data_miror$variable <- gsub("GC", "AT", data_miror$variable)
data_miror$value <- 100 - data$value

data <- rbind(data, data_miror)

neg_res_TID <- ggplot(data = filter(data, 
                                    (variable == "GC3_percent" | variable == "AT3_percent") &
                                      state == "resting" & rCSC == "negative" ), 
                      aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Resting")

neg_act_TID <- ggplot(data = filter(data, 
                                    (variable == "GC3_percent" | variable == "AT3_percent") &
                                      state == "activated" & rCSC == "negative" ), 
                      aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Activated")

pos_res_TID <- ggplot(data = filter(data, 
                                    (variable == "GC3_percent" | variable == "AT3_percent") &
                                      state == "resting" & rCSC == "positive" ), 
                      aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Resting")

pos_act_TID <- ggplot(data = filter(data, 
                                    (variable == "GC3_percent" | variable == "AT3_percent") &
                                      state == "activated" & rCSC == "positive" ), 
                      aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Activated")

## GC % 
neg_res_TID_GC <- ggplot(data = filter(data, 
                                       (variable == "GC_percent" | variable == "AT_percent") &
                                         state == "resting" & rCSC == "negative" ), 
                         aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Resting")

neg_act_TID_GC <- ggplot(data = filter(data, 
                                       (variable == "GC_percent" | variable == "AT_percent") &
                                         state == "activated" & rCSC == "negative" ), 
                         aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC negative - Activated")

pos_res_TID_GC <- ggplot(data = filter(data, 
                                       (variable == "GC_percent" | variable == "AT_percent") &
                                         state == "resting" & rCSC == "positive" ), 
                         aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Resting")

pos_act_TID_GC <- ggplot(data = filter(data, 
                                       (variable == "GC_percent" | variable == "AT_percent") &
                                         state == "activated" & rCSC == "positive" ), 
                         aes(x="", y = value, fill = variable)) + 
  geom_bar(stat = "identity") + 
  coord_polar("y", start=0) +
  scale_fill_manual(values = c(color_resting, color_activated)) + 
  ggtitle("TID - rCSC positive - Activated")

pdf(file = "piechart_rCSC_TDD_TID_random_sequences.pdf")
grid.arrange(neg_res, neg_act, neg_res_TID, neg_act_TID, ncol = 2, top = "GC3 percent with negative rCSC codon - Random Sequences")
grid.arrange(pos_res, pos_act, pos_res_TID, pos_act_TID, ncol = 2, top = "GC3 percent with positive rCSC codon - Random Sequences")
grid.arrange(neg_res_GC, neg_act_GC, neg_res_TID_GC, neg_act_TID_GC, ncol = 2, top = "GC percent with negative rCSC codon - Random Sequences")
grid.arrange(pos_res_GC, pos_act_GC, pos_res_TID_GC, pos_act_TID_GC, ncol = 2, top = "GC percent with positive rCSC codon - Random Sequences")
dev.off()


# ribosomal proteins ------------------------------------------------------


