

#' Annotate peaks' associated PAS and motifs
#'
#' @param peak.sites.file Peak sites table generated by function peak_calling
#' @param pas.reference.file Known PAS table
#' @param genome Full genome sequence. Required to be a S4 object.
#' @param pas.search.cut.1 Distance (nt) from the peak mode to upstream boundary of the searching interval for PAS
#' @param pas.search.cut.2 Distance (nt) from the peak mode to downstream boundary of the searching interval for PAS
#' @param polystretch_length Length of consecutive A sequence
#' @param max_mismatch Maximal tolerance of mismatch
#' @param motif.search.cut Window width for searching specified motifs
#' @param invert_strand default FALSE
#'
#' @return A peak annotation table.
#' @export
#'
#' @examples
#' ls()
#'
#'
peak_annotation <- function(peak.sites.file,
                            pas.reference.file,
                            genome,
                            pas.search.cut.1 = 0,
                            pas.search.cut.2 = 300,
                            polystretch_length = 13,
                            max_mismatch = 1,
                            motif.search.cut = 300,
                            invert_strand = FALSE
) {

  # Read in the peak sites table
  mode.pos = gene = seq = strand = gene.start = gene.end = mu = sigma = pi = from = to = width.fit = polyA_ID = cluster_ID = UTR3_hr = NULL
  peak.table <- utils::read.table(peak.sites.file, header = TRUE, sep=",", stringsAsFactors = FALSE)
  peak.table <- peak.table %>% dplyr::select(mode.pos, gene, seq, strand, gene.start, gene.end, UTR3_hr, mu, sigma, pi, from, to, width.fit, polyA_ID, cluster_ID)
  abnormal.id = which(peak.table$width.fit <= 0)
  peak.table <- peak.table[!peak.table$cluster_ID %in% peak.table$cluster_ID[abnormal.id],]
  peak.table = peak.table %>% tidyr::drop_na()
  peak.table = peak.table %>% dplyr::filter(UTR3_hr == 'YES')
  peak.table = peak.table[peak.table$sigma < 200 & peak.table$sigma > 10, ]
  peak.table <- peak.table %>% dplyr::select(mode.pos, gene, seq, strand, gene.start, gene.end, mu, sigma, pi, from, to, width.fit, polyA_ID, cluster_ID)

  message('Number of peaks after pre-filtering: ', nrow(peak.table))

  # read in pas.reference.file
  Chrom = Positio = Strand = mgi_symbol = NULL
  pas.ref.df = utils::read.csv(pas.reference.file)
  pas.ref.df = pas.ref.df %>% dplyr::select(Chrom, Position, Strand, mgi_symbol)

  # build gr for pas reference and peak regions
  pas.ref.df.gr = GenomicRanges::GRanges(paste0(pas.ref.df$Chrom,":", pas.ref.df$Position, ":", pas.ref.df$Strand))
  peak.table$lb.center = peak.table$mode.pos + ifelse(peak.table$strand == '-', -pas.search.cut.2, pas.search.cut.1)
  peak.table$ub.center = peak.table$mode.pos + ifelse(peak.table$strand == '-', -pas.search.cut.1, pas.search.cut.2)
  peak.table$lb.center = pmin(pmax(peak.table$lb.center, peak.table$gene.start), peak.table$gene.end)
  peak.table$ub.center = pmax(pmin(peak.table$ub.center, peak.table$gene.end), peak.table$gene.start)
  pk_gr = GenomicRanges::GRanges(paste0(peak.table$seq, ":", peak.table$lb.center, "-", peak.table$ub.center, ":", peak.table$strand))

  # find PAS
  message('Annotating known PAS around peaks.')
  idx = pas.dist = NULL
  peak.table$n_PAS = peak.table$PAS_full = ""
  all_hits <-  GenomicAlignments::findOverlaps(pk_gr, pas.ref.df.gr, type = 'any')
  idx_to_annotate_known_PAS <- S4Vectors::queryHits(all_hits)
  found.id = unique(idx_to_annotate_known_PAS)
  hits.pas.df.0 = data.frame(pas.ref.df.gr[S4Vectors::subjectHits(all_hits)])
  hits.pas.df = data.frame(idx = idx_to_annotate_known_PAS, pas = hits.pas.df.0$start)
  hits.pas.df$pas.dist = (peak.table$mode.pos[hits.pas.df$idx] - hits.pas.df$pas)*((peak.table$strand[hits.pas.df$idx] == '-')*2 -1)
  hits.pas.df = hits.pas.df %>% dplyr::group_by(idx) %>% dplyr::summarise(nn = dplyr::n(), full_list = paste0(pas.dist, collapse = ','))
  peak.table$n_PAS[found.id] = hits.pas.df$nn
  peak.table$PAS_full[found.id] = hits.pas.df$full_list

  # find motif
  message('Annotating motifs around peaks.')

  # generate gr object
  all.peaks <- peak.table$polyA_ID
  # strand = sub(".*:.*:.*-.*:(.*)", "\\1", all.peaks)
  # strand = plyr::mapvalues(x = strand, from = c("1", "-1"), to = c("+", "-"))
  # peak.remainder = sub(".*:(.*:.*-.*):.*", "\\1", all.peaks)
  # peaks.use = paste0(peak.remainder, ":", strand)
  peaks.use = sub(".*:(.*:.*-.*:.*)", "\\1", all.peaks)
  gr <- GenomicRanges::GRanges(peaks.use)

  # add meta data
  peak.table$crude.start = peak.table$mode.pos - motif.search.cut
  peak.table$crude.end = peak.table$mode.pos + motif.search.cut
  GenomicRanges::mcols(gr)$crude.start = pmin(pmax(peak.table$crude.start, peak.table$gene.start), peak.table$gene.end)
  GenomicRanges::mcols(gr)$crude.end = pmin(pmax(peak.table$crude.end, peak.table$gene.start), peak.table$gene.end)
  GenomicRanges::mcols(gr)$mode.pos = peak.table$mode.pos
  GenomicRanges::mcols(gr)$polyA_ID = peak.table$polyA_ID

  # annotate motifs
  anno.df <- annotate_gr(gr = gr,
                          genome = genome,
                          invert_strand = invert_strand,
                          annotationType = "any",
                          polystretch_length = polystretch_length,
                          max_mismatch = max_mismatch
  )

  polyA_ID = pA_stretch = NULL
  anno.df = anno.df %>% dplyr::select(polyA_ID:pA_stretch)
  peak.table = peak.table %>% dplyr::left_join(anno.df, by = 'polyA_ID')

  return(peak.table)

}


################################################################################

annotate_gr <- function(gr,
                        genome,
                        invert_strand = FALSE,
                        annotationType ="any",
                        annotation_correction = TRUE,
                        polystretch_length = 13,
                        max_mismatch = 1
) {

  if (invert_strand) gr <- BiocGenerics::invertStrand(gr)
  df <- as.data.frame(gr)
  if (is.null(genome) | !isS4(genome)) warning("Genome is Null. Motif annotation cannot be proceeded.")
  if (isS4(genome)) {

    # Set up progress bar
    pbar <- utils::txtProgressBar(min = 0, max = length(gr), style = 3)

    motif_details <- lapply(1:length(gr), function(i, gr, pbar) {
      utils::setTxtProgressBar(pbar, i)
      pk <- gr[i]
      base_seq(genome, coord = pk, mismatch = max_mismatch, polystretch_length = polystretch_length)
      },
      gr,
      pbar
    )

    cat("\n")  # Keep output neat for pbar

    df$pA_motif_TTTAAA = get_motif_pos('pA_motif_pos_TTTAAA', motif_details)
    df$pA_motif_TATAAA = get_motif_pos('pA_motif_pos_TATAAA', motif_details)
    df$pA_motif_GATAAA = get_motif_pos('pA_motif_pos_GATAAA', motif_details)
    df$pA_motif_CATAAA = get_motif_pos('pA_motif_pos_CATAAA', motif_details)
    df$pA_motif_ATTAAA = get_motif_pos('pA_motif_pos_ATTAAA', motif_details)
    df$pA_motif_AGTAAA = get_motif_pos('pA_motif_pos_AGTAAA', motif_details)
    df$pA_motif_ACTAAA = get_motif_pos('pA_motif_pos_ACTAAA', motif_details)
    df$pA_motif_AATGAA = get_motif_pos('pA_motif_pos_AATGAA', motif_details)
    df$pA_motif_AATATA = get_motif_pos('pA_motif_pos_AATATA', motif_details)
    df$pA_motif_AATAAA = get_motif_pos('pA_motif_pos_AATAAA', motif_details)
    df$pA_motif_AAGAAA = get_motif_pos('pA_motif_pos_AAGAAA', motif_details)
    df$pA_motif_AACAAA = get_motif_pos('pA_motif_pos_AACAAA', motif_details)
    df$pA_stretch = get_motif_pos('pA_stretch_pos', motif_details)

    return(df)

  }
}

################################################################################

get_motif_pos <- function(ele_name, motif_details) {

  tmp = unlist(lapply(motif_details, function(x) {
    pA_motif_position <- NA
    motif_pos <- unlist(x[ele_name])
    if (length(motif_pos) > 0) {
      if (all(!is.na(motif_pos)) & length(motif_pos) > 0)
        pA_motif_position = paste0(motif_pos, collapse = ',')
    }
    return (pA_motif_position)
  } ))

  return(tmp)

}

################################################################################

base_seq <- function(genome,
                     coord = NULL,
                     offset = -100,
                     length = 300,
                     mismatch = 1,
                     polystretch_length = 13
) {

  coord.meta = as.integer(data.frame(GenomicRanges::mcols(coord)[1:3]))
  coord <- as.character(coord)
  coord_split <- strsplit(coord, split = ":")[[1]]
  chrom  <- coord_split[1]
  strand <- coord_split[3]
  coords <- strsplit(coord_split[2],"-")[[1]]
  start  <- as.numeric(coords[1])
  stop   <- as.numeric(coords[2])

  if ( start > stop) { tmp = start; start = stop; stop = tmp } # swap start and stop if needed

  # Extract downstream sequence
  seq_start_position = coord.meta[3]
  seq_end_position = coord.meta[2]

  # check seq length
  chromLengths = GenomeInfoDb::seqlengths(genome)
  chromLength <- chromLengths[chrom]

  if (is.na(chromLength)){
    warning(paste0("No chromosome entry in genome for ", chrom))
    stop
  }

  if (seq_end_position > chromLength){
    seq_end_position <- chromLength
    warning(paste0("Peak ", chrom, ":", start, "-", stop, ":", strand, " is at end of chromosome."))
    stop
  }

  sequ <- BSgenome::getSeq(genome, chrom, seq_start_position, seq_end_position)
  sequ_upstream <- {}
  if (strand == '-')  sequ_upstream <- sequ

  # Extract upstream sequence
  seq_start_position = coord.meta[1]
  seq_end_position = coord.meta[3]

  if (seq_start_position <= 0) {
    warning(paste0("Peak ", chrom, ":", start, "-", stop, ":", strand, " is at end of chromosome."))
    stop
  }

  sequ_tmp <- BSgenome::getSeq(genome, chrom, seq_start_position, seq_end_position)

  if (strand == '-') {
    sequ <- Biostrings::reverseComplement(sequ_tmp)
    sequ_upstream <- Biostrings::reverseComplement(sequ_upstream)
  } else {
    sequ_upstream <- sequ_tmp
  }

  # 12 pA_motifs
  pA_motif_TTTAAA  <-  Biostrings::matchPattern(pattern = "TTTAAA", subject = sequ)
  pA_motif_TATAAA  <-  Biostrings::matchPattern(pattern = "TATAAA", subject = sequ)
  pA_motif_GATAAA  <-  Biostrings::matchPattern(pattern = "GATAAA", subject = sequ)
  pA_motif_CATAAA  <-  Biostrings::matchPattern(pattern = "CATAAA", subject = sequ)
  pA_motif_ATTAAA  <-  Biostrings::matchPattern(pattern = "ATTAAA", subject = sequ)
  pA_motif_AGTAAA  <-  Biostrings::matchPattern(pattern = "AGTAAA", subject = sequ)
  pA_motif_ACTAAA  <-  Biostrings::matchPattern(pattern = "ACTAAA", subject = sequ)
  pA_motif_AATGAA  <-  Biostrings::matchPattern(pattern = "AATGAA", subject = sequ)
  pA_motif_AATATA  <-  Biostrings::matchPattern(pattern = "AATATA", subject = sequ)
  pA_motif_AATAAA  <-  Biostrings::matchPattern(pattern = "AATAAA", subject = sequ)
  pA_motif_AAGAAA  <-  Biostrings::matchPattern(pattern = "AAGAAA", subject = sequ)
  pA_motif_AACAAA  <-  Biostrings::matchPattern(pattern = "AACAAA", subject = sequ)

  A_pattern <- paste(rep("A", polystretch_length), collapse = '')
  pA_stretch <- Biostrings::matchPattern(pattern = A_pattern, subject = sequ, max.mismatch = mismatch)

  return(list(pA_motif_pos_TTTAAA = Biostrings::start(pA_motif_TTTAAA),
              pA_motif_pos_TATAAA = Biostrings::start(pA_motif_TATAAA),
              pA_motif_pos_GATAAA = Biostrings::start(pA_motif_GATAAA),
              pA_motif_pos_CATAAA = Biostrings::start(pA_motif_CATAAA),
              pA_motif_pos_ATTAAA = Biostrings::start(pA_motif_ATTAAA),
              pA_motif_pos_AGTAAA = Biostrings::start(pA_motif_AGTAAA),
              pA_motif_pos_ACTAAA = Biostrings::start(pA_motif_ACTAAA),
              pA_motif_pos_AATGAA = Biostrings::start(pA_motif_AATGAA),
              pA_motif_pos_AATATA = Biostrings::start(pA_motif_AATATA),
              pA_motif_pos_AATAAA = Biostrings::start(pA_motif_AATAAA),
              pA_motif_pos_AAGAAA = Biostrings::start(pA_motif_AAGAAA),
              pA_motif_pos_AACAAA = Biostrings::start(pA_motif_AACAAA),
              pA_stretch_pos = Biostrings::start(pA_stretch)
              )
         )

}


### END ###


