from Bio import SeqIO
from Bio.Seq import Seq
from Bio.SeqRecord import SeqRecord
import argparse
import statistics
import re

def get_options():
    description = "Determines accuracy for gene calls."
    parser = argparse.ArgumentParser(description=description,
                                     prog='python gene_recall.py')

    IO = parser.add_argument_group('Input/options.out')
    IO.add_argument('--seq',
                    help='Reference sequences genes called from (FASTA format)')
    IO.add_argument('--CDS',
                    help='Reference gene panel (FASTA format)')
    IO.add_argument('--query',
                    help='Genes generated by gene caller to query (FASTA format)')
    IO.add_argument('--query2',
                    default=None,
                    help='Genes generated by second tools ')
    IO.add_argument('--caller',
                    type=str,
                    choices=['prod', 'ggc', 'pan'],
                    help='Gene caller used. Can be either \'prod\' for Prodigal, \'ggc\' for ggCaller or \'pan\' for Panaroo.')
    IO.add_argument('--caller2',
                    type=str,
                    default=None,
                    choices=['prod', 'ggc', 'pan', None],
                    help='Gene caller used for query2. Can be either \'prod\' for Prodigal, \'ggc\' for ggCaller or \'pan\' for Panaroo.')
    IO.add_argument('--min-size',
                    type=int,
                    default=90,
                    help='Minimum gene length to include. '
                    '[Default = 90] ')
    IO.add_argument('--exact',
                    default=False,
                    action="store_true",
                    help='Rather than 3-prime accuracy, conduct exact matching. ')
    IO.add_argument('--outpref',
                    default="compare_fasta_out",
                    help='Out prefix. Default = "compare_fasta_out". ')
    return parser.parse_args()

def remove_invalid(query_seq):
    import re
    start_codons = ["ATG", "GTG", "TTG"]
    stop_codons = ["TAA", "TGA", "TAG"]
    # check if codons in right place, and in correct frame
    if query_seq[0:3] not in start_codons or query_seq[-3:] not in stop_codons or len(query_seq) % 3 != 0:
        return False

    # check if sequence contains a premature stop codon
    for stop in stop_codons:
        stop_indices = [m.start() for m in re.finditer(stop, query_seq)]
        # remove last stop as this is valid
        if query_seq[-3:] == stop:
            stop_indices = stop_indices[:-1]
        if any([i % 3 == 0 for i in stop_indices]):
            return False

    # If all tests come back fine, return true
    return True

def compare_3prime(genome_fasta, ref_fasta, query_fasta, caller_type, min_size, outpref):
    # generate lists to hold gene coordinates
    genome_list = []
    genome_rec = {}
    ref_rec = {}
    query_rec = {}

    total_correct_query_records = 0
    total_ref_records = 0
    total_query_records = 0
    total_unmatched_query_records = 0

    unmatched_query_list = []
    unmatched_ref_list = []
    incorrect_query_list = []
    duplicated_query_list = []

    # parse genome_fasta
    id_dict = {}
    id_num = 0
    for rec in SeqIO.parse(genome_fasta, "fasta"):
        description = (rec.description).split("_")
        id = description[0]
        # if genome not found before, add to list.
        if id not in genome_list:
            genome_list.append(id)
            genome_rec[id] = str(rec.seq)
            ref_rec[id] = {}
            query_rec[id] = {}
            id_dict[id_num] = id
            id_num += 1
        # if genome made up of multiple contigs, just concatenate the string together to enable searching.
        else:
            genome_rec[id] += str(rec.seq)

    # parse ref_fasta
    for rec in SeqIO.parse(ref_fasta, "fasta"):
        description = (rec.description).split("_")
        id = description[0]

        if len(str(rec.seq)) < min_size:
           continue


        # look for the 3prime index of the string
        start_index = genome_rec[id].find(str(rec.seq))
        if start_index == -1:
            prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
        else:
            prime3 = start_index + (len(str(rec.seq)) - 1)
        #check that sequence is present in genome is says, and that the gene sequence is valid and no Ns present. Also check that no duplicate entries, as ggc can't call duplicate genes
        if prime3 != -1 and remove_invalid(str(rec.seq)) and "N" not in str(rec.seq) and prime3 not in ref_rec[id]:
            ref_rec[id][prime3] = str(rec.seq)
            total_ref_records += 1
            unmatched_ref_list.append((id, str(rec.seq)))

    if caller_type == "ggc":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta, "fasta"):
            id = rec.id.split('_')[0]
            rev_comp = False

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
                rev_comp = True
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)

            # determine if gene sequence is real, if not add to duplicated_query_list
            if prime3 != -1:
                # check if already present, update with longer sequence, add shorter sequence to incorrect sequences
                if prime3 in query_rec[id]:
                    if len(query_rec[id][prime3]) < len(str(rec.seq)):
                        duplicated_query_list.append((id, query_rec[id][prime3]))
                        query_rec[id][prime3] = str(rec.seq)
                    else:
                        duplicated_query_list.append((id, str(rec.seq)))
                else:
                    query_rec[id][prime3] = str(rec.seq)
            else:
                incorrect_query_list.append((id, str(rec.seq)))

    elif caller_type == "prod":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta, "fasta"):
            description = (rec.description).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec[id]:
                query_rec[id][prime3] = str(rec.seq)
            elif prime3 in query_rec[id]:
                if len(query_rec[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list.append((id, query_rec[id][prime3]))
                    query_rec[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list.append((id, str(rec.seq)))
            else:
                incorrect_query_list.append((id, str(rec.seq)))

    elif caller_type == "pan":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta, "fasta"):
            description = (rec.id).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec[id]:
                query_rec[id][prime3] = str(rec.seq)
            elif prime3 in query_rec[id]:
                if len(query_rec[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list.append((id, query_rec[id][prime3]))
                    query_rec[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list.append((id, str(rec.seq)))
            else:
                incorrect_query_list.append((id, str(rec.seq)))

    # iterate over query_rec, count number of times each 3prime match found
    for colour, prime3_dict in query_rec.items():
        for prime3_key in prime3_dict.keys():
            if prime3_key in ref_rec[colour]:
                total_correct_query_records += 1
                unmatched_ref_list.remove((colour, ref_rec[colour][prime3_key]))
            else:
                total_unmatched_query_records += 1
                unmatched = (colour, prime3_dict[prime3_key])
                unmatched_query_list.append(unmatched)
            total_query_records += 1

    total_query_records += len(incorrect_query_list)
    total_query_records += len(duplicated_query_list)

    recall = total_correct_query_records / total_ref_records
    precision = total_correct_query_records / total_query_records

    with open(outpref + "_summary.txt", "w") as f:
        f.write("Total_true\t{}".format(total_ref_records) + "\n")
        f.write("Total_called\t{}".format(total_query_records) + "\n")
        f.write("TP\t{}".format(total_correct_query_records) + "\n")
        f.write("FP\t{}".format(len(unmatched_query_list)) + "\n")
        f.write("FN\t{}".format(total_ref_records - total_correct_query_records) + "\n")
        f.write("Artifical\t{}".format(len(incorrect_query_list)) + "\n")
        f.write("Duplicated\t{}".format(len(duplicated_query_list)) + "\n")
        f.write("Recall\t{}".format(recall) + "\n")
        f.write("Precision\t{}".format(precision) + "\n")

    # print false negatives
    DNA_records = []
    for entry in unmatched_ref_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0], description=""))
    SeqIO.write(DNA_records, outpref + "_FN.fasta", "fasta")

    # print false positives
    DNA_records = []
    for entry in unmatched_query_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0], description=""))
    SeqIO.write(DNA_records, outpref + "_FP.fasta", "fasta")

    # print artificial
    DNA_records = []
    for entry in incorrect_query_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0], description=""))
    SeqIO.write(DNA_records, outpref + "_ART.fasta", "fasta")

    # print duplicated
    DNA_records = []
    for entry in duplicated_query_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0], description=""))
    SeqIO.write(DNA_records, outpref + "_DUP.fasta", "fasta")

def compare_exact(genome_fasta, ref_fasta, query_fasta, caller_type, min_size, outpref):
    # generate lists to hold gene coordinates
    genome_list = []
    genome_rec = {}
    ref_rec = {}
    query_rec = {}

    total_correct_query_records = 0
    total_ref_records = 0
    total_query_records = 0
    total_unmatched_query_records = 0

    unmatched_query_list = []
    unmatched_ref_list = []
    incorrect_query_list = []
    duplicated_query_list = []

    # parse genome_fasta
    id_dict = {}
    id_num = 0
    for rec in SeqIO.parse(genome_fasta, "fasta"):
        description = (rec.description).split("_")
        id = description[0]
        # if genome not found before, add to list.
        if id not in genome_list:
            genome_list.append(id)
            genome_rec[id] = str(rec.seq)
            ref_rec[id] = {}
            query_rec[id] = {}
            id_dict[id_num] = id
            id_num += 1
        # if genome made up of multiple contigs, just concatenate the string together to enable searching.
        else:
            genome_rec[id] += str(rec.seq)

    # parse ref_fasta
    for rec in SeqIO.parse(ref_fasta, "fasta"):
        description = (rec.description).split("_")
        id = description[0]

        if len(str(rec.seq)) < min_size:
           continue

        # look for the 3prime index of the string
        start_index = genome_rec[id].find(str(rec.seq))
        if start_index == -1:
            prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
        else:
            prime3 = start_index + (len(str(rec.seq)) - 1)
        # check that sequence is present in genome is says, and that the gene sequence is valid and no Ns present. Also check that no duplicate entries, as ggc can't call duplicate genes
        if prime3 != -1 and remove_invalid(str(rec.seq)) and "N" not in str(rec.seq) and prime3 not in ref_rec[id]:
            ref_rec[id][prime3] = str(rec.seq)
            total_ref_records += 1
            unmatched_ref_list.append((id, str(rec.seq)))

    if caller_type == "ggc":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta, "fasta"):
            id = rec.id.split('_')[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)

            # determine if gene sequence is real, if not add to duplicated_query_list
            if prime3 != -1:
                # check if already present, update with longer sequence, add shorter sequence to incorrect sequences
                if prime3 in query_rec[id]:
                    if len(query_rec[id][prime3]) < len(str(rec.seq)):
                        duplicated_query_list.append((id, query_rec[id][prime3]))
                        query_rec[id][prime3] = str(rec.seq)
                    else:
                        duplicated_query_list.append((id, str(rec.seq)))
                else:
                    query_rec[id][prime3] = str(rec.seq)
            else:
                incorrect_query_list.append((id, str(rec.seq)))

    elif caller_type == "prod":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta, "fasta"):
            description = (rec.id).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec[id]:
                query_rec[id][prime3] = str(rec.seq)
            elif prime3 in query_rec[id]:
                if len(query_rec[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list.append((id, query_rec[id][prime3]))
                    query_rec[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list.append((id, str(rec.seq)))
            else:
                incorrect_query_list.append((id, str(rec.seq)))

    elif caller_type == "pan":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta, "fasta"):
            description = (rec.id).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec[id]:
                query_rec[id][prime3] = str(rec.seq)
            elif prime3 in query_rec[id]:
                if len(query_rec[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list.append((id, query_rec[id][prime3]))
                    query_rec[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list.append((id, str(rec.seq)))
            else:
                incorrect_query_list.append((id, str(rec.seq)))

    prop_length = []
    actual_length = []
    # iterate over query_rec, count number of times each 3prime match found
    for colour, prime3_dict in query_rec.items():
        for prime3_key, seq in prime3_dict.items():
            if prime3_key in ref_rec[colour]:
                ref_seq = ref_rec[colour][prime3_key]
                if ref_seq == seq:
                    total_correct_query_records += 1
                    unmatched_ref_list.remove((colour, ref_rec[colour][prime3_key]))
                else:
                    length_diff = len(seq) / len(ref_seq)
                    total_unmatched_query_records += 1
                    unmatched = (colour, prime3_dict[prime3_key], length_diff)
                    unmatched_query_list.append(unmatched)
                prop_length.append(len(seq) / len(ref_seq))
            else:
                total_unmatched_query_records += 1
                unmatched = (colour, prime3_dict[prime3_key], "NA")
                unmatched_query_list.append(unmatched)
            total_query_records += 1

    total_query_records += len(incorrect_query_list)
    total_query_records += len(duplicated_query_list)

    recall = total_correct_query_records / total_ref_records
    precision = total_correct_query_records / total_query_records

    mean_prop_length = statistics.mean(prop_length)
    stdev_prop_length = statistics.stdev(prop_length)

    with open(outpref + "_summary.txt", "w") as f:
        f.write("Total_true\t{}".format(total_ref_records) + "\n")
        f.write("Total_called\t{}".format(total_query_records) + "\n")
        f.write("TP\t{}".format(total_correct_query_records) + "\n")
        f.write("FP\t{}".format(len(unmatched_query_list)) + "\n")
        f.write("FN\t{}".format(total_ref_records - total_correct_query_records) + "\n")
        f.write("Artifical\t{}".format(len(incorrect_query_list)) + "\n")
        f.write("Duplicated\t{}".format(len(duplicated_query_list)) + "\n")
        f.write("Recall\t{}".format(recall) + "\n")
        f.write("Precision\t{}".format(precision) + "\n")
        f.write("Mean_len_prop\t{}".format(mean_prop_length) + "\n")
        f.write("Stdev_len_prop\t{}".format(stdev_prop_length) + "\n")

    # print false negatives
    DNA_records = []
    for entry in unmatched_ref_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0], description=""))
    SeqIO.write(DNA_records, outpref + "_FN.fasta", "fasta")

    # print false positives
    DNA_records = []
    for entry in unmatched_query_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0] + "_Lendiff={}".format(entry[2]), description=""))
    SeqIO.write(DNA_records, outpref + "_FP.fasta", "fasta")

    # print artificial
    DNA_records = []
    for entry in incorrect_query_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0], description=""))
    SeqIO.write(DNA_records, outpref + "_ART.fasta", "fasta")

    # print duplicated
    DNA_records = []
    for entry in duplicated_query_list:
        DNA_records.append(SeqRecord(Seq(entry[1]), id=entry[0], description=""))
    SeqIO.write(DNA_records, outpref + "_DUP.fasta", "fasta")

    with open(outpref + "_len_prop.txt", "w") as f:
        for entry in prop_length:
            f.write(str(entry) + "\n")

def compare_exact2(genome_fasta, ref_fasta, query_fasta1, caller_type1, min_size,
                   query_fasta2, caller_type2, outpref):
    # generate lists to hold gene coordinates
    genome_list = []
    genome_rec = {}
    ref_rec = {}
    query_rec1 = {}
    query_rec2 = {}

    # data for query1
    total_correct_query_records1 = 0
    total_ref_records1 = 0
    total_query_records1 = 0
    total_unmatched_query_records1 = 0

    unmatched_query_list1 = []
    unmatched_ref_list1 = []
    incorrect_query_list1 = []
    duplicated_query_list1 = []

    # data for query2
    total_correct_query_records2 = 0
    total_ref_records2 = 0
    total_query_records2 = 0
    total_unmatched_query_records2 = 0

    unmatched_query_list2 = []
    unmatched_ref_list2 = []
    incorrect_query_list2 = []
    duplicated_query_list2 = []

    # parse genome_fasta
    id_dict = {}
    id_num = 0
    for rec in SeqIO.parse(genome_fasta, "fasta"):
        description = (rec.description).split("_")
        id = description[0]
        # if genome not found before, add to list.
        if id not in genome_list:
            genome_list.append(id)
            genome_rec[id] = str(rec.seq)
            ref_rec[id] = {}
            query_rec1[id] = {}
            query_rec2[id] = {}
            id_dict[id_num] = id
            id_num += 1
        # if genome made up of multiple contigs, just concatenate the string together to enable searching.
        else:
            genome_rec[id] += str(rec.seq)

    # parse ref_fasta
    for rec in SeqIO.parse(ref_fasta, "fasta"):
        description = (rec.description).split("_")
        id = description[0]

        if len(str(rec.seq)) < min_size:
           continue

        # look for the 3prime index of the string
        start_index = genome_rec[id].find(str(rec.seq))
        if start_index == -1:
            prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
        else:
            prime3 = start_index + (len(str(rec.seq)) - 1)
        # check that sequence is present in genome is says, and that the gene sequence is valid and no Ns present. Also check that no duplicate entries, as ggc can't call duplicate genes
        if prime3 != -1 and remove_invalid(str(rec.seq)) and "N" not in str(rec.seq) and prime3 not in ref_rec[id]:
            ref_rec[id][prime3] = str(rec.seq)
            total_ref_records1 += 1
            unmatched_ref_list1.append((id, str(rec.seq), prime3))
            total_ref_records2 += 1
            unmatched_ref_list2.append((id, str(rec.seq), prime3))

    # run for caller 1
    if caller_type1 == "ggc":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta1, "fasta"):
            id = rec.id.split('_')[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)

            # determine if gene sequence is real, if not add to duplicated_query_list1
            if prime3 != -1:
                # check if already present, update with longer sequence, add shorter sequence to incorrect sequences
                if prime3 in query_rec1[id]:
                    if len(query_rec1[id][prime3]) < len(str(rec.seq)):
                        duplicated_query_list1.append((id, query_rec1[id][prime3]))
                        query_rec1[id][prime3] = str(rec.seq)
                    else:
                        duplicated_query_list1.append((id, str(rec.seq)))
                else:
                    query_rec1[id][prime3] = str(rec.seq)
            else:
                incorrect_query_list1.append((id, str(rec.seq)))

    elif caller_type1 == "prod":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta1, "fasta"):
            description = (rec.id).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec1[id]:
                query_rec1[id][prime3] = str(rec.seq)
            elif prime3 in query_rec1[id]:
                if len(query_rec1[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list1.append((id, query_rec1[id][prime3]))
                    query_rec1[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list1.append((id, str(rec.seq)))
            else:
                incorrect_query_list1.append((id, str(rec.seq)))

    elif caller_type1 == "pan":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta1, "fasta"):
            description = (rec.id).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec1[id]:
                query_rec1[id][prime3] = str(rec.seq)
            elif prime3 in query_rec1[id]:
                if len(query_rec1[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list1.append((id, query_rec1[id][prime3]))
                    query_rec1[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list1.append((id, str(rec.seq)))
            else:
                incorrect_query_list1.append((id, str(rec.seq)))

    prop_length1 = []
    unmatched_query_dict1 = {}
    # iterate over query_rec, count number of times each 3prime match found
    for colour, prime3_dict in query_rec1.items():
        unmatched_query_dict1[colour] = {}
        for prime3_key, seq in prime3_dict.items():
            if prime3_key in ref_rec[colour]:
                ref_seq = ref_rec[colour][prime3_key]
                if ref_seq == seq:
                    total_correct_query_records1 += 1
                else:
                    length_diff = len(seq) / len(ref_seq)
                    total_unmatched_query_records1 += 1
                    unmatched = (colour, prime3_dict[prime3_key], length_diff)
                    unmatched_query_list1.append(unmatched)
                    unmatched_query_dict1[colour][prime3_key] = (prime3_dict[prime3_key], length_diff, ref_seq)
                unmatched_ref_list1.remove((colour, ref_rec[colour][prime3_key], prime3_key))
                prop_length1.append(len(seq) / len(ref_seq))
            else:
                total_unmatched_query_records1 += 1
                unmatched = (colour, prime3_dict[prime3_key], "NA")
                unmatched_query_list1.append(unmatched)
                unmatched_query_dict1[colour][prime3_key] = (prime3_dict[prime3_key], "NA", "NA")
            total_query_records1 += 1

    total_query_records1 += len(incorrect_query_list1)
    total_query_records1 += len(duplicated_query_list1)

    # run for caller 2
    if caller_type2 == "ggc":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta2, "fasta"):
            id = rec.id.split('_')[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)

            # determine if gene sequence is real, if not add to duplicated_query_list2
            if prime3 != -1:
                # check if already present, update with longer sequence, add shorter sequence to incorrect sequences
                if prime3 in query_rec2[id]:
                    if len(query_rec2[id][prime3]) < len(str(rec.seq)):
                        duplicated_query_list2.append((id, query_rec2[id][prime3]))
                        query_rec2[id][prime3] = str(rec.seq)
                    else:
                        duplicated_query_list2.append((id, str(rec.seq)))
                else:
                    query_rec2[id][prime3] = str(rec.seq)
            else:
                incorrect_query_list2.append((id, str(rec.seq)))

    elif caller_type2 == "prod":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta2, "fasta"):
            description = (rec.id).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec2[id]:
                query_rec2[id][prime3] = str(rec.seq)
            elif prime3 in query_rec2[id]:
                if len(query_rec2[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list2.append((id, query_rec2[id][prime3]))
                    query_rec2[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list2.append((id, str(rec.seq)))
            else:
                incorrect_query_list2.append((id, str(rec.seq)))

    elif caller_type2 == "pan":
        # parse query_fasta
        for rec in SeqIO.parse(query_fasta2, "fasta"):
            description = (rec.id).split("_")
            id = description[0]

            # look for the 3prime index of the string
            start_index = genome_rec[id].find(str(rec.seq))
            if start_index == -1:
                prime3 = genome_rec[id].find(str(rec.seq.reverse_complement()))
            else:
                prime3 = start_index + (len(str(rec.seq)) - 1)
            if "N" not in str(rec.seq) and prime3 not in query_rec2[id]:
                query_rec2[id][prime3] = str(rec.seq)
            elif prime3 in query_rec2[id]:
                if len(query_rec2[id][prime3]) < len(str(rec.seq)):
                    duplicated_query_list2.append((id, query_rec2[id][prime3]))
                    query_rec2[id][prime3] = str(rec.seq)
                else:
                    duplicated_query_list2.append((id, str(rec.seq)))
            else:
                incorrect_query_list2.append((id, str(rec.seq)))

    prop_length2 = []
    unmatched_query_dict2 = {}
    # iterate over query_rec, count number of times each 3prime match found
    for colour, prime3_dict in query_rec2.items():
        unmatched_query_dict2[colour] = {}
        for prime3_key, seq in prime3_dict.items():
            if prime3_key in ref_rec[colour]:
                ref_seq = ref_rec[colour][prime3_key]
                if ref_seq == seq:
                    total_correct_query_records2 += 1
                else:
                    length_diff = len(seq) / len(ref_seq)
                    total_unmatched_query_records2 += 1
                    unmatched = (colour, prime3_dict[prime3_key], length_diff)
                    unmatched_query_list2.append(unmatched)
                    unmatched_query_dict2[colour][prime3_key] = (prime3_dict[prime3_key], length_diff, ref_seq)
                unmatched_ref_list2.remove((colour, ref_rec[colour][prime3_key], prime3_key))
                prop_length2.append(len(seq) / len(ref_seq))
            else:
                total_unmatched_query_records2 += 1
                unmatched = (colour, prime3_dict[prime3_key], "NA", "NA")
                unmatched_query_list2.append(unmatched)
                unmatched_query_dict2[colour][prime3_key] = (prime3_dict[prime3_key], "NA", "NA")
            total_query_records2 += 1

    total_query_records2 += len(incorrect_query_list2)
    total_query_records2 += len(duplicated_query_list2)

    # compare false positives between the two callers (use first caller as reference)
    inter_caller_comp_FP = []
    inter_caller_comp_FN = []
    for colour, prime3_dict in unmatched_query_dict1.items():
        for prime3_key, seq_tup in prime3_dict.items():
            seq1 = seq_tup[0]
            len_diff1 = seq_tup[1]
            ref_seq = seq_tup[2]
            dist_end = min([prime3_key, len(genome_rec[colour]) - prime3_key])
            if prime3_key in unmatched_query_dict2[colour]:
                seq2 = unmatched_query_dict2[colour][prime3_key][0]
                len_diff2 = unmatched_query_dict2[colour][prime3_key][1]
                len_diff = len(seq2) / len(seq1)
                inter_caller_comp_FP.append((colour, ref_seq, seq1, len_diff1, seq2, len_diff2, len_diff, dist_end))
                if len_diff1 != "NA":
                    inter_caller_comp_FN.append((colour, ref_seq, seq1, len_diff1, seq2, len_diff2, len_diff, dist_end))
            else:
                inter_caller_comp_FP.append((colour, ref_seq, seq1, len_diff1, "NA", "NA", "NA", dist_end))
                if len_diff1 != "NA":
                    inter_caller_comp_FN.append((colour, ref_seq, seq1, len_diff1, "NA", "NA", "NA", dist_end))

    for colour, prime3_dict in unmatched_query_dict2.items():
        for prime3_key, seq_tup in prime3_dict.items():
            seq2 = seq_tup[0]
            len_diff2 = seq_tup[1]
            ref_seq = seq_tup[2]
            dist_end = min([prime3_key, len(genome_rec[colour]) - prime3_key])
            if prime3_key not in unmatched_query_dict1[colour]:
                inter_caller_comp_FP.append((colour, ref_seq, "NA", "NA", seq2, len_diff2, "NA", dist_end))
                if len_diff2 != "NA":
                    inter_caller_comp_FN.append((colour, ref_seq, "NA", "NA", seq2, len_diff2, "NA", dist_end))

    #compare false negatives
    for entry in unmatched_ref_list1:
        dist_end = min([entry[2], len(genome_rec[entry[0]]) - entry[2]])
        if entry in unmatched_ref_list2:
            inter_caller_comp_FN.append((entry[0], entry[1], entry[1], "NA", entry[1], "NA", "NA", dist_end))
        else:
            inter_caller_comp_FN.append((entry[0], entry[1], entry[1], "NA", "NA", "NA", "NA", dist_end))

    for entry in unmatched_ref_list2:
        dist_end = min([entry[2], len(genome_rec[entry[0]]) - entry[2]])
        if entry not in unmatched_ref_list1:
            inter_caller_comp_FN.append((entry[0], entry[1], "NA", "NA", entry[1], "NA", "NA", dist_end))

    # compare artifical calls
    inter_caller_comp_ART = []
    for entry in incorrect_query_list1:
        if entry in incorrect_query_list2:
            inter_caller_comp_ART.append((entry[0], entry[1], entry[1]))
        else:
            inter_caller_comp_ART.append((entry[0], entry[1], "NA"))

    for entry in incorrect_query_list2:
        if entry not in incorrect_query_list1:
            inter_caller_comp_ART.append((entry[0], "NA", entry[1]))

    # compare duplicated calls
    inter_caller_comp_DUP = []
    for entry in duplicated_query_list1:
        if entry in duplicated_query_list2:
            inter_caller_comp_DUP.append((entry[0], entry[1], entry[1]))
        else:
            inter_caller_comp_DUP.append((entry[0], entry[1], "NA"))

    for entry in duplicated_query_list2:
        if entry not in duplicated_query_list1:
            inter_caller_comp_DUP.append((entry[0], "NA", entry[1]))

    with open(outpref + "_FP.txt", "w") as f:
        f.write("Colour\tRef_seq\tCaller1_seq\tCaller1_ref_lendiff\tCaller2_seq\tCaller2_ref_lendiff\tInter_lendiff\tprime3_dist_end\n")
        for entry in inter_caller_comp_FP:
            f.write("{}\t{}\t{}\t{}\t{}\t{}\t{}\t{}\n".format(entry[0], entry[1], entry[2], entry[3], entry[4],entry[5], entry[6], entry[7]))

    with open(outpref + "_FN.txt", "w") as f:
        f.write("Colour\tRef_seq\tCaller1_seq\tCaller1_ref_lendiff\tCaller2_seq\tCaller2_ref_lendiff\tInter_lendiff\tprime3_dist_end\n")
        for entry in inter_caller_comp_FN:
            f.write("{}\t{}\t{}\t{}\t{}\t{}\t{}\t{}\n".format(entry[0], entry[1], entry[2], entry[3], entry[4], entry[5], entry[6], entry[7]))

    with open(outpref + "_ART.txt", "w") as f:
        f.write("Colour\tCaller1_seq\tCaller2_seq\n")
        for entry in inter_caller_comp_ART:
            f.write("{}\t{}\t{}\n".format(entry[0], entry[1], entry[2]))

    with open(outpref + "_DUP.txt", "w") as f:
        f.write("Colour\tCaller1_seq\tCaller2_seq\n")
        for entry in inter_caller_comp_DUP:
            f.write("{}\t{}\t{}\n".format(entry[0], entry[1], entry[2]))

def main():
    options = get_options()
    query2 = options.query2
    if query2 is None:
        if options.exact:
            compare_exact(options.seq, options.CDS, options.query, options.caller,
                                          options.min_size, options.outpref)
        else:
            compare_3prime(options.seq, options.CDS, options.query, options.caller, options.min_size, options.outpref)
    else:
        compare_exact2(options.seq, options.CDS, options.query, options.caller, options.min_size,
                       options.query2, options.caller2, options.outpref)

    return 0

if __name__ == '__main__':
    main()



