import pandas as pd
import argparse
from Bio import SeqIO
import seaborn as sns
import matplotlib.pyplot as plt
from matplotlib import rcParams
rcParams.update({'figure.autolayout': True})

def get_options():
	description = 'Determines coverage of annotated genes in ggCaller graph by significant unitig hits'
	parser = argparse.ArgumentParser(description=description,
									 prog='python count_annotations.py')
	IO = parser.add_argument_group('Input/Output options')
	IO.add_argument('--fasta',
					required=True,
					help='Fasta file generated by ggCaller query mode ')
	IO.add_argument('--annotations',
					required=True,
					help='Excel file containing gene annotations. Format should match '
						 'https://www.ncbi.nlm.nih.gov/pmc/articles/PMC5708525/#SD1')
	IO.add_argument('--outpref',
					default="hits",
					help='Output prefix ')
	return parser.parse_args()

def parse_annotations(annotation_file):
	annotation_dict = {}
	df = pd.read_excel(annotation_file)
	for index, row in df.iterrows():
		# determine whether protein is MGE
		MGE_rows = row[4:11]
		# if protein annotation suggests MGE annotate as MGE
		if MGE_rows.str.count("Y").sum() == 1 and row[4:].str.count("Y").sum() == 1:
			annotation_dict[row[0]] = "MGE"
		# for MGE pseudogene
		elif MGE_rows.str.count("Y").sum() == 1 and row[4:].str.count("Y").sum() == 2 and row[24] == "Y":
			annotation_dict[row[0]] = "MGE"
		elif row[21:24].str.count("Y").sum() > 0:
			annotation_dict[row[0]] = "hypothetical"
		else:
			if pd.isnull(row[2]):
				annotation_dict[row[0]] = row[3]
			else:
				annotation_dict[row[0]] = row[2]
			annotation_dict[row[0]] = annotation_dict[row[0]].replace(" (fragment)", "")
	
	return annotation_dict

def parse_query(fasta, delim="_", position=-1):
	fasta_list = []
	count_list = []
	query_size_list = []
	hit_size_list = []
	mapped_set = set()
	for seq_record in SeqIO.parse(fasta, "fasta"):
		rec_set = set()
		hit_size_list.append(len(seq_record.seq))
		# parse different annotations for gene
		split_desc = seq_record.description.split("annotation=")
		desc = split_desc[-1]
		desc = desc.split(";")
		for entry in desc:
			#print(entry)
			if entry != "":
				split_entry = entry.split(delim)
				#print(split_entry[position])
				rec_set.add(split_entry[position])
		fasta_list.append(rec_set)
		# parse number of significant hits
		desc = split_desc[0]
		desc = desc.split("QUERY=")[-1]
		desc = desc.split(";")
		desc = [x.strip() for x in desc]
		#print(desc)
		#print(str(len(desc)))
		count_list.append(len(desc))
		query_size_list.append(sum(len(s) for s in desc))
		mapped_set.update(desc)

	print("No. mapped queries: {}".format(len(mapped_set)))
	return fasta_list, count_list, query_size_list, hit_size_list

def main():
	options = get_options()
	annotation_file = options.annotations
	fasta = options.fasta

	out_pref = options.outpref

	annotation_dict = parse_annotations(annotation_file)
	fasta_list, count_list, query_size_list, hit_size_list = parse_query(fasta)

	# iterate over fasta_list, determine number of each annotation
	annotation_count_query = {}
	annotation_count_ORF = {}
	annotation_count_size = {}
	annotation_hit_size = {}

	for index, entry in enumerate(fasta_list):
		annotation = "hypothetical"
		# print(entry)
		for ID in entry:
			if ID in annotation_dict:
				# match the first non-hypothetical annotation
				annotation_temp = annotation_dict[ID]
				if "hypothetical" not in annotation_temp:
					annotation = annotation_temp
					# print(annotation)
					break
		# print(annotation)
		if annotation not in annotation_count_query:
			annotation_count_query[annotation] = count_list[index]
			annotation_count_ORF[annotation] = 1
			annotation_count_size[annotation] = query_size_list[index]
			annotation_hit_size[annotation] = hit_size_list[index]
		else:
			annotation_count_query[annotation] += count_list[index]
			annotation_count_size[annotation] += query_size_list[index]
			annotation_hit_size[annotation] += hit_size_list[index]
			annotation_count_ORF[annotation] += 1

	annotation_ratio = {}
	annotation_size_ratio = {}
	annotation_RPG = {}
	for annotation in annotation_count_query:
		annotation_ratio[annotation] = annotation_count_query[annotation] / annotation_count_ORF[annotation]
		annotation_size_ratio[annotation] = annotation_count_size[annotation] / annotation_count_ORF[annotation]
		annotation_RPG[annotation] = annotation_count_query[annotation] / (annotation_hit_size[annotation] / 1000)

	# print(annotation_count_query)
	# print(annotation_count_ORF)

	df_query = pd.DataFrame(annotation_count_query.items(), columns=["Gene", "Count"])
	df_ORF = pd.DataFrame(annotation_count_ORF.items(), columns=["Gene", "Count"])
	df_query = df_query.sort_values(['Count'], ascending=False).reset_index(drop=True)
	df_ORF = df_ORF.sort_values(['Count'], ascending=False).reset_index(drop=True)
	df_ratio = pd.DataFrame(annotation_ratio.items(), columns=["Gene", "Count"])
	df_ratio = df_ratio.sort_values(['Count'], ascending=False).reset_index(drop=True)
	df_size_ratio = pd.DataFrame(annotation_size_ratio.items(), columns=["Gene", "Count"])
	df_size_ratio = df_size_ratio.sort_values(['Count'], ascending=False).reset_index(drop=True)
	df_RPG = pd.DataFrame(annotation_RPG.items(), columns=["Gene", "Count"])
	df_RPG = df_RPG.sort_values(['Count'], ascending=False).reset_index(drop=True)

	ax = sns.barplot(data=df_query, y="Gene", x="Count")
	ax.set(xscale="log")
	ax.set(ylabel='', xlabel='Number of unitig hits')
	plt.savefig(out_pref + "_query.svg")

	plt.clf()

	ax = sns.barplot(data=df_ORF, y="Gene", x="Count")
	ax.set(xscale="log")
	ax.set(ylabel='', xlabel='Number of ORFs')
	plt.savefig(out_pref + "_ORF.svg")

	plt.clf()

	ax = sns.barplot(data=df_ratio, y="Gene", x="Count")
	# ax.set(xscale="log")
	ax.set(ylabel='', xlabel='No. Mapped Queries per Gene')
	plt.savefig(out_pref + "_count_ratio.svg")

	plt.clf()

	ax = sns.barplot(data=df_size_ratio, y="Gene", x="Count")
	# ax.set(xscale="log")
	ax.set(ylabel='', xlabel='Mapped Query Length per Gene')
	plt.savefig(out_pref + "_size_ratio.svg")

	plt.clf()

	ax = sns.barplot(data=df_RPG, y="Gene", x="Count")
	# ax.set(xscale="log")
	ax.set(ylabel='', xlabel='No. mapped queries per kilo-basepair')
	plt.savefig(out_pref + "_RPG.svg")


if __name__ == "__main__":
	main()