#ifndef SKETCH_BB_MINHASH_H__
#define SKETCH_BB_MINHASH_H__
#include "sketch/count_eq.h"
#include "hll.h"
#include "aesctr/wy.h"
#if !NDEBUG
#include <cstdarg>
#endif
#include <mutex>

namespace sketch {
inline namespace minhash {

using hash::WangHash;


namespace detail {

// Based on content from BinDash https://github.com/zhaoxiaofei/bindash
static inline uint64_t twounivhash(uint64_t s, uint64_t t) {
    static constexpr uint64_t LARGE_PRIME = 9223372036854775783ull;
    // Constants generated by `python -c 'import random;print("UINT64_C(%s)" % hex(random.randint(0, 1 << 64)))'`
    return (UINT64_C(0x1e68e69958ce15c1) * (UINT64_C(0x84e09756b31589c9) * s + UINT64_C(0xd89576eb901ab7d3) * t) + UINT64_C(0x2f28f2976668b622)) % LARGE_PRIME;
}

template<typename T>
static constexpr T default_val() {
    return std::numeric_limits<T>::max();
}

template<typename Container>
inline int densifybin(Container &hashes) {
    using vtype = typename std::decay<decltype(hashes[0])>::type;
    const auto empty_val = default_val<vtype>();
    auto it = hashes.cbegin();
    auto min = *it, max = min;
    while(++it != hashes.end()) {
        auto v = *it;
        min = std::min(min, v);
        max = std::max(max, v);
    }
    if (max != empty_val) {
        return 0; // Full sketch
    }
    if (min == empty_val) {
        std::fill(hashes.begin(), hashes.end(), empty_val);
        return -1; // Empty sketch
    }
    for (uint64_t i = 0; i < hashes.size(); i++) {
        uint64_t j = i, nattempts = 0;
        while(hashes[j] == empty_val)
            j = twounivhash(i, ++nattempts) % hashes.size();
        hashes[i] = hashes[j];
    }
    return 1;
}


template<typename Cont>
static inline double harmonic_cardinality_estimate_diffmax_impl(const Cont &minvec, const long double num) {
    const long double s = std::accumulate(minvec.begin(), minvec.end(), static_cast<long double>(0), [num](auto sum, const auto v) {return sum + v / num;});
    return s ? std::pow(minvec.size(), 2) / s: 0.;
}

template<typename Cont>
static inline double harmonic_cardinality_estimate_impl(const Cont &minvec) {
    using VT = std::decay_t<decltype(*minvec.begin())>;
    size_t nz = 0;
    for(const auto v: minvec) nz += v == detail::default_val<VT>();
    if(nz == minvec.size()) return 0.;
    const long double num = is_pow2(minvec.size()) ? std::ldexp(static_cast<long double>(1.), sizeof(VT) * CHAR_BIT - ilog2(minvec.size()))
                                                   : ((long double)UINT64_C(-1)) / minvec.size();
    return harmonic_cardinality_estimate_diffmax_impl(minvec, num);
}

template<typename T, typename Allocator>
static inline double harmonic_cardinality_estimate(std::vector<T, Allocator> &minvec, bool) {
    return harmonic_cardinality_estimate_impl(minvec);
}

template<typename T, typename Allocator>
static inline double harmonic_cardinality_estimate(const std::vector<T, Allocator> &minvec) {
    return harmonic_cardinality_estimate_impl(minvec);
}

INLINE void setnthbit(uint64_t *ptr, size_t index, bool val) {
    ptr[index / 64] |= uint64_t(val) << (index % 64);
}
template<typename T> INLINE void setnthbit(T *ptr, size_t index, bool val) {
    return setnthbit(reinterpret_cast<uint64_t *>(ptr), index, val);
}

static INLINE uint64_t getnthbit(const uint64_t *ptr, size_t index) {
    return (ptr[index / 64] >> (index % 64)) & 1u;
}

template<typename T> INLINE T getnthbit(const T *ptr, size_t index) {
    return T(getnthbit(reinterpret_cast<const uint8_t *>(ptr), index));
}
INLINE uint64_t getnthbit(uint64_t val, size_t index) {
    return getnthbit(&val, index);
}

#if __SSE2__
INLINE auto matching_bits(const __m128i *s1, const __m128i *s2, uint16_t b) {
     __m128i match = ~(*s1++ ^ *s2++);
     while(--b)
         match &= ~(*s1++ ^ *s2++);
     return popcount(common::vatpos(match, 0)) + popcount(common::vatpos(match, 1));
}
#endif

#if __AVX2__
INLINE auto sbit_accum(const __m256i *s1, const __m256i *s2, uint16_t b) {
    __m256i match = ~(*s1++ ^ *s2++);
    while(--b)
        match &= ~(*s1++ ^ *s2++);
    return match;
}
INLINE auto matching_bits(const __m256i *s1, const __m256i *s2, uint16_t b) {
    return popcnt256(sbit_accum(s1, s2, b));
}
#endif


#if HAS_AVX_512
INLINE auto sbit_accum(const __m512i *s1, const __m512i *s2, uint16_t b) {
    __m512i match = ~(*s1++ ^ *s2++);
    while(--b)
        match &= ~(*s1++ ^ *s2++);
    return match;
}
INLINE auto matching_bits(const __m512i *s1, const __m512i *s2, uint16_t b) {
    return popcnt512(sbit_accum(s1, s2, b));
}
#endif

} // namespace detail

struct phll_t {
    std::vector<uint8_t, Allocator<uint8_t>> core_;
    volatile mutable double                   est_ = -1.;
    //static auto make_base(size_t coresize) {return coresize > 128 ? 16.: std::pow(std::ldexp(1., 64), 1. / 15.);}
    phll_t(std::vector<uint8_t, Allocator<uint8_t>> &&v): core_(std::move(v))  {}
    phll_t(std::vector<uint8_t, Allocator<uint8_t>> &v): core_(std::move(v)) {}
    phll_t(const std::vector<uint8_t, Allocator<uint8_t>> &v): core_(v) {}
    phll_t(const phll_t &o) = default;
    phll_t(phll_t &&o)      = default;
    size_t size() const {return core_.size() << 1;}
    std::array<uint32_t, 16> get_counts() const {
        std::array<uint32_t, 16> counts{0};
        for(const auto v: core_) {
            ++counts[v >> 4];
            ++counts[v&0xFu];
        }
        return counts;
    }
    INLINE double cardinality_estimate() const {
        return register_estimate(get_counts());
    }
    std::array<double, 3> full_set_comparison(const phll_t &o) const {
        if(est_ < 0.) est_ = cardinality_estimate();
        if(o.est_ < 0.) o.est_ = o.cardinality_estimate();
        auto us = union_size(o);
        auto is = est_ + o.est_ - us;
        double me_only = est_ > is ? est_ - is: 0.,
               o_only  = o.est_ > is ? o.est_ - is: 0.;
        return std::array<double, 3>{{me_only, o_only, is}};
    }
    INLINE double jaccard_index(const phll_t &o) const {
        auto fs = full_set_comparison(o);
        return fs[2] / (fs[0] + fs[1] + fs[2]);
    }
    INLINE double union_size(const phll_t &o) const {
        std::array<uint32_t, 16> counts{0};
        if(size() < sizeof(Space::COUNT)) {
            for(size_t i = 0; i < core_.size(); ++i) {
                ++counts[std::max(core_[i] >> 4, o.core_[i] >> 4)];
                ++counts[std::max(core_[i] & 0xFu, o.core_[i] & 0xFu)];
            }
        } else {
            const Space::Type *ptr = static_cast<const Space::Type *>(static_cast<const void *>(core_.data())),
                               *optr = static_cast<const Space::Type *>(static_cast<const void *>(o.core_.data()));
            const auto lmask = Space::set1(0x0F0F0F0F0F0F0F0F), umask = Space::set1(0xF0F0F0F0F0F0F0F0);
            for(size_t i = 0; i < core_.size() / Space::COUNT; ++i) {
                Space::VType lhs = Space::load(ptr + i), rhs = Space::load(optr + i);
                auto getbits = [&](auto mask) {return hll::detail::SIMDHolder::max_fn(Space::and_fn(lhs.simd_, mask), Space::and_fn(rhs.simd_, mask));};
#if __AVX512F__ && !__AVX512BW__
                auto lv = getbits(lmask);
                ((Space::VType *)(&lv))->for_each([&counts](auto x) {++counts[x];});
                lv = Space::srli(getbits(umask), 4);
                ((Space::VType *)(&lv))->for_each([&](auto x) {++counts[x];});
#else
                Space::VType lv = getbits(lmask);
                lv.for_each([&](auto x) {++counts[x];});
                lv = Space::srli(getbits(umask), 4);
                lv.for_each([&](auto x) {++counts[x];});
#endif
            }
        }
        return register_estimate(counts);

    }
    double harmonic_mean_estimate() const {return harmonic_mean_estimate(get_counts());}
    template<typename C>
    static double harmonic_mean_estimate(const C &counts) {
        //for(unsigned i = 0 ; i < 16; ++i) std::fprintf(stderr, "count %u %u times\n", i, counts[i]);
        //assert(std::accumulate(counts.begin(), counts.end(), size_t(0)) == size());;
        long double sum = counts[0];
        for(int i = 1; i < 16; ++i) {
            sum += std::ldexp(static_cast<long double>(counts[i]), -4 * i);
        }
        return std::pow(counts.size(), 2) / sum;
    }
    template<typename C>
    double register_estimate(const C &counts) const {
#if 0
        return harmonic_mean_estimate(counts);
#else
        double est = harmonic_mean_estimate(counts);
        est = est / (0.2 * std::log(est) - 0.5);
        return est;
#endif
    }
    phll_t &operator+=(const phll_t &o) {
        using hll::detail::SIMDHolder;
        PREC_REQ(size() == o.size(), "must have matching sizes");
        if(size() < sizeof(Space::COUNT)) {
            for(size_t i = 0; i < core_.size(); ++i) {
                core_[i] = (std::max(core_[i] >> 4, o.core_[i] >> 4) << 4) |
                            std::max(core_[i] & 0xFu, o.core_[i] & 0xFu);
            }
        } else {
            const Space::Type *ptr = static_cast<const Space::Type *>(static_cast<const void *>(core_.data())),
                               *optr = static_cast<const Space::Type *>(static_cast<const void *>(o.core_.data()));
            const auto lmask = Space::set1(0x0F0F0F0F0F0F0F0FULL), umask = Space::set1(0xF0F0F0F0F0F0F0F0ULL);
            for(size_t i = 0; i < core_.size() / Space::COUNT; ++i) {
                Space::VType lhs = Space::load(ptr + i), rhs = Space::load(optr + i);
                auto getbits = [&](auto mask) {
                    Space::VType ret = Space::max(Space::and_fn(lhs.simd_, mask), Space::and_fn(rhs.simd_, mask));
                    return ret.simd_;
                    //static_assert(sizeof(ret) == sizeof(v));
                };
                Space::store(reinterpret_cast<Space::Type *>(core_.data() + i * Space::COUNT),
                             getbits(lmask) | getbits(umask));
            }
        }
        return *this;
    }
    phll_t operator+(const phll_t &o) const {
        phll_t ret(*this);
        ret += o;
        return ret;
    }
};


template<template<typename> class Policy, typename RNGType, typename CountType>
struct SuperMinHash;
struct FinalBBitMinHash;
template<typename T, typename Hasher>
class DivBBitMinHasher;
struct FinalDivBBitMinHash {
private:
    FinalDivBBitMinHash() {}
public:
    using value_type = uint64_t;
    double est_cardinality_;
    uint64_t nbuckets_;
    uint32_t b_;
    std::vector<value_type, Allocator<value_type>> core_;
    FinalDivBBitMinHash(unsigned nbuckets, unsigned b, double est):
        est_cardinality_(est), nbuckets_(nbuckets), b_(b),
        core_(value_type(b) * nbuckets / 64)
    {
        PREC_REQ(b * nbuckets % 64 == 0, "b * nbuckets must be divisible by 64");
    }
    void free() {
        decltype(core_) tmp;
        std::swap(tmp, core_);
    }
    bool operator==(const FinalDivBBitMinHash &o) const {
#define H__(x) (this->x == o.x)
        return H__(nbuckets_) && H__(core_);// Removed b_ and est_cardinality_, since the sets are the same, and that's what really matters.
#undef H__
    }
    bool operator!=(const FinalDivBBitMinHash &o) const {
        return !this->operator==(o);
    }
    FinalDivBBitMinHash(const std::string &path): FinalDivBBitMinHash(path.data()) {}
    FinalDivBBitMinHash(const char *path): est_cardinality_(0), nbuckets_(0), b_(0) {
        read(path);
    }
    FinalDivBBitMinHash(gzFile ifp): est_cardinality_(0), nbuckets_(0), b_(0) {
        read(ifp);
    }
    FinalDivBBitMinHash(FinalDivBBitMinHash &&o) = default;
    FinalDivBBitMinHash(const FinalDivBBitMinHash &o) = default;
    template<template<typename> class Policy, typename RNG, typename CountType>
    FinalDivBBitMinHash(SuperMinHash<Policy, RNG, CountType> &&o): FinalDivBBitMinHash(o.finalize()) {}
    template<template<typename> class Policy, typename RNG, typename CountType>
    FinalDivBBitMinHash(const SuperMinHash<Policy, RNG, CountType> &o): FinalDivBBitMinHash(o.finalize()) {}
    template<typename T, typename Hasher=WangHash>
    FinalDivBBitMinHash(DivBBitMinHasher<T, Hasher> &&o): FinalDivBBitMinHash(std::move(o.finalize())) {
        o.free();
    }
    template<typename T, typename Hasher=WangHash>
    FinalDivBBitMinHash(const DivBBitMinHasher<T, Hasher> &o): FinalDivBBitMinHash(std::move(o.finalize())) {}
    double cardinality_estimate() const {return est_cardinality_;}
    ssize_t read(gzFile fp) {
        uint64_t arr[2];
        if(gzread(fp, arr, sizeof(arr)) != sizeof(arr)) throw ZlibError("Could not read from file.");
        ssize_t ret = sizeof(arr);
        b_ = arr[0];
        nbuckets_ = arr[1];
        PREC_REQ(b_ * nbuckets_ % 64 == 0, "b * nbuckets must be divisible by 64");
        if(gzread(fp, &est_cardinality_, sizeof(est_cardinality_)) != sizeof(est_cardinality_)) throw ZlibError("Could not read from file.");
        ret += sizeof(est_cardinality_);
        core_.resize(b_ * nbuckets_ / 64 + (b_ * nbuckets_ % 64 != 0));
        const size_t expected = sizeof(core_[0]) * core_.size();
        if(gzread(fp, core_.data(), expected) != ssize_t(expected))
            throw ZlibError(std::string("Failed to read core from file"));
        ret += expected;
        return ret;
    }
    DBSKETCH_WRITE_STRING_MACROS
    DBSKETCH_READ_STRING_MACROS
    ssize_t write(gzFile fp) const {
        uint64_t arr[] {b_, nbuckets_};
        ssize_t ret;
        if(HEDLEY_UNLIKELY((ret = gzwrite(fp, arr, sizeof(arr))) != sizeof(arr))) throw std::runtime_error("Could not write to file");
        ret += sizeof(arr);
        if(HEDLEY_UNLIKELY(gzwrite(fp, &est_cardinality_, sizeof(est_cardinality_)) != sizeof(est_cardinality_))) throw std::runtime_error("Could not write to file");
        ret += sizeof(est_cardinality_);
        if(HEDLEY_UNLIKELY(gzwrite(fp, core_.data(), core_.size() * sizeof(core_[0])) != ssize_t(core_.size() * sizeof(core_[0])))) throw std::runtime_error("Could not write to file");
        ret += sizeof(core_[0]) * core_.size();
        return ret;
    }
    size_t nmatches(const FinalDivBBitMinHash &o) const {
        return equal_bblocks(o);
    }
    double frac_equal(const FinalDivBBitMinHash &o) const {
        auto neq = equal_bblocks(o);
        return double(neq) / (nbuckets_);
    }
    uint64_t nmin() const {
        return nbuckets_;
    }
    size_t nblocks() const {return nmin();}
    double jaccard_index(const FinalDivBBitMinHash &o) const {
        /*
         * reference: https://arxiv.org/abs/1802.03914.
        */
        const double b2pow = std::ldexp(1., -b_);
        double frac = frac_equal(o);
        frac -= b2pow;
        return std::max(0., frac / (1. - b2pow));
    }
    double containment_index(const FinalDivBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        return is / est_cardinality_;
    }
    double intersection_size(const FinalDivBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        return is;
    }
    double union_size(const FinalDivBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        return est_cardinality_ + o.est_cardinality_ - is;
    }
    std::array<double, 3> full_set_comparison(const FinalDivBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        double me_only = est_cardinality_ > is ? est_cardinality_ - is: 0.,
               o_only  = o.est_cardinality_ > is ? o.est_cardinality_ - is: 0.;
        return std::array<double, 3>{{me_only, o_only, is}};
    }
    uint64_t equal_bblocks(const FinalDivBBitMinHash &o) const {
        assert(o.core_.size() == core_.size());
        assert(b_ <= 64); // b_ > 64 not yet supported, though it could be done with a larger hash
#if 0
        switch(b_) {
            case 4: return eq::count_eq_nibbles((uint8_t *)core_.data(), (uint8_t *)o.core_.data(), core_.size() << 3);
            case 8: return eq::count_eq_bytes((uint8_t *)core_.data(), (uint8_t *)o.core_.data(), core_.size() << 3);
            case 16: return eq::count_eq_shorts((uint16_t *)core_.data(), (uint16_t *)o.core_.data(), core_.size() << 2);
            case 32: return eq::count_eq_words((uint32_t *)core_.data(), (uint32_t *)o.core_.data(), core_.size() << 1);
            case 64: return eq::count_eq_longs(core_.data(), o.core_.data(), core_.size());
        }
#endif
        // p_ already guaranteed to be greater than 6
        auto l2szfloor = ilog2(nbuckets_);
        const value_type *p1 = core_.data(), *pe = core_.data() + b_ * (1ull << l2szfloor) / 64, *p2 = o.core_.data();
        uint64_t sum;
        switch(l2szfloor) {
            case 6: {
                auto match = ~(*p1++ ^ *p2++);
                while(p1 != pe) match &= ~(*p1++ ^ *p2++);
                sum = popcount(match);
                break;
            }
            case 7:
                sum = detail::matching_bits(reinterpret_cast<const __m128i *>(p1), reinterpret_cast<const __m128i *>(p2), b_);
                break;
#if __AVX2__
            case 8: {
                sum = common::sum_of_u64s(detail::matching_bits(reinterpret_cast<const __m256i *>(p1), reinterpret_cast<const __m256i *>(p2), b_));
                break;
            }
#  if HAS_AVX_512
            case 9: {
                sum = common::sum_of_u64s(detail::matching_bits(reinterpret_cast<const __m512i *>(p1), reinterpret_cast<const __m512i *>(p2), b_));
                break;
            }
            default: {
                // Process each 'b' remainder block in
                const __m512i *vp1 = reinterpret_cast<const __m512i *>(p1), *vp2 = reinterpret_cast<const __m512i *>(p2);
                auto lsum = detail::matching_bits(vp1, vp2, b_);
                for(size_t i = 1; i < (size_t(1) << (l2szfloor - 9u)); ++i) {
                    vp1 += b_;
                    vp2 += b_;
                    lsum = _mm512_add_epi64(detail::matching_bits(vp1, vp2, b_), lsum);
                }
                assert((value_type*)vp1 == &core_[core_.size()]);
                sum = common::sum_of_u64s(lsum);
                break;
            }
#    else /* has avx2 not not 512 */
            default: {
                const __m256i *vp1 = reinterpret_cast<const __m256i *>(p1), *vp2 = reinterpret_cast<const __m256i *>(p2);
                auto lsum = detail::matching_bits(vp1, vp2, b_);
                for(size_t i = 1; i < 1ull << (l2szfloor - 8u); ++i) {
                    vp1 += b_;
                    vp2 += b_;
                    lsum = _mm256_add_epi64(detail::matching_bits(vp1, vp2, b_), lsum);
                    assert(vp1 <= reinterpret_cast<const __m256i *>(pe));
                }
                sum = common::sum_of_u64s(lsum);
                break;
            }
#  endif /* avx512 or avx2 */
#else /* assume SSE2 */
            default: {
                // Process each 'b' remainder block in
                const __m128i *vp1 = reinterpret_cast<const __m128i *>(p1), *vp2 = reinterpret_cast<const __m128i *>(p2), *vpe = reinterpret_cast<const __m128i *>(pe);
                __m128i match = ~(*vp1++ ^ *vp2++);
                for(unsigned b = b_; --b;match &= ~(*vp1++ ^ *vp2++));
                auto lsum = common::sum_of_u64s(match);
                while((uint64_t *)vp1 + 2 <= (uint64_t *)vpe) {
                    match = ~(*vp1++ ^ *vp2++);
                    for(unsigned b = b_; --b; match &= ~(*vp1++ ^ *vp2++));
                    lsum += common::sum_of_u64s(match);
                }
                sum = lsum;
                break;
            }
#endif
        } // switch(l2szfloor)
        /* Now handle the rest of the bits.
         */
        const value_type *const pf = &core_[core_.size()];
        if(pe == pf)
            return sum; // If there is no remainder, we're done
#if HAS_AVX_512
        {
            __m512i lsum = _mm512_set1_epi64(0);
            const __m512i *vp1 = reinterpret_cast<const __m512i *>(pe), *vp2 = reinterpret_cast<const __m512i *>(o.core_.data() +  b_ * (1ull << l2szfloor) / 64);
            while(vp1 + b_ <= reinterpret_cast<const __m512i *>(pf)) {
                __m512i match = ~(*vp1++ ^ *vp2++);
                for(unsigned b = b_; --b; match &= ~(*vp1++ ^ *vp2++));
#if __AVX512VPOPCNTDQ__
                lsum = _mm512_add_epi64(_mm512_popcnt_epi64(match), lsum);
#else
                lsum = _mm512_add_epi64(popcnt512(match), lsum);
#endif
            }
            sum += common::sum_of_u64s(lsum);
            p1 = reinterpret_cast<const uint64_t *>(vp1);
            p2 = reinterpret_cast<const uint64_t *>(vp2);
        }
#elif __AVX2__
        {
            const __m256i *vp1 = reinterpret_cast<const __m256i *>(pe), *vp2 = reinterpret_cast<const __m256i *>(o.core_.data() + b_ * (1ull << l2szfloor) / 64);
            __m256i lsum = _mm256_set1_epi64x(0);
            while(vp1 + b_ <= reinterpret_cast<const __m256i *>(pf)) {
                __m256i match = ~(*vp1++ ^ *vp2++);
                for(unsigned b = b_; --b;match &= ~(*vp1++ ^ *vp2++));
                lsum = _mm256_add_epi64(lsum, popcnt256(match));
            }
            sum += common::sum_of_u64s(lsum);
            p1 = reinterpret_cast<const uint64_t *>(vp1);
            p2 = reinterpret_cast<const uint64_t *>(vp2);
        }
#else
        const __m128i *vp1 = reinterpret_cast<const __m128i *>(pe);
        const __m128i *vp2 = reinterpret_cast<const __m128i *>(o.core_.data() + b_ * (1ull << l2szfloor) / 64);
        while(vp1 + b_ <= reinterpret_cast<const __m128i *>(pf)) {
            __m128i match = ~(*vp1++ ^ *vp2++);
            for(unsigned b = b_; --b; match &= ~(*vp1++ ^ *vp2++));
            sum += popcount(vatpos(match, 0)) + popcount(vatpos(match, 1));
            // Since there's no faster popcount for __m128i currently.
        }
        p1 = reinterpret_cast<const uint64_t *>(vp1);
        p2 = reinterpret_cast<const uint64_t *>(vp2);
#endif
        while(p1 < pf) {
            uint64_t match = ~(*p1++ * *p2++);
            for(unsigned b = b_; --b; match &= ~(*p1++ ^ *p2++));
            sum += popcount(match);
        }
        return sum;
    }
};

template<typename T, typename Allocator>
FinalDivBBitMinHash div_bbit_finalize(uint32_t b, const std::vector<T, Allocator> &core_ref, double est_v=0.);




template<template<typename> class Policy=policy::SizePow2Policy, typename RNGType=wy::WyHash<uint32_t, 1>, typename CountType=uint32_t>
struct SuperMinHash {
    // Note:
    // Instead of maintaining real and integral portions of a hash in floating point,
    // we store these in 32-bit portions of a 64-bit value, keeping the index information
    // in the higher register and the fractional portion in a lower one.
    // This is reduced entropy per 64-bit value (when finalized), so we're quite justified
    // in b-bit minimizing.
    // The number of bits needed for full minimizer encoding in this scheme
    // is 32 + log2(m_).
    // Best of all, because our sketch only needs mod in CountType space,
    // our mods are 5 instructions vs 16-19 on x86 for fastmod.
    Policy<CountType> pol_;
    using BType = typename std::make_signed<CountType>::type;
    uint64_t a_, i_;
    uint32_t m_;
#if NOT_THREADSAFE
    uint64_t count_;
#else
    std::atomic<uint64_t> count_;
#endif
#if VERBOSE_AF
    std::atomic<size_t> inner_loop_count_;
#endif
    uint64_t seed_;

    static_assert(sizeof(typename RNGType::result_type) == sizeof(uint32_t), "must generate 32-bit values");


    std::vector<CountType>  p_;
    std::vector<uint64_t>   h_;
    std::vector<CountType>  q_;
    std::vector<BType>      b_;
    unsigned            bbits_;
    SuperMinHash(size_t arg, unsigned bbits=0, uint64_t seed=0): pol_(arg), a_(pol_.arg2vecsize(arg) - 1), i_(0), m_(pol_.arg2vecsize(arg)),
        count_(0), seed_(seed),
        p_(m_), h_(pol_.arg2vecsize(arg), uint64_t(-1)), q_(pol_.arg2vecsize(arg), -1), b_(pol_.arg2vecsize(arg))
#if VERBOSE_AF
    , inner_loop_count_(0)
#endif
    , bbits_(bbits ? bbits: unsigned(needed_bits()))
    {
#if VERBOSE_AF
        std::fprintf(stderr, "[%s:%d:%s] size of a %zu, q: %zu, p: %zu, b: %zu. h %zu\n", __FILE__, __LINE__, __PRETTY_FUNCTION__, size_t(a_), q_.size(), p_.size(), b_.size(), h_.size());
#endif
        b_.back() = m_;
        assert(m_ <= std::numeric_limits<CountType>::max());
    }
    void free() {
        auto p(std::move(p_));
        auto h(std::move(h_));
        auto q(std::move(q_));
        auto b(std::move(b_));
    }
    SuperMinHash(std::string s) {throw NotImplementedError("SuperMinHash can't be made from std::string");}
    void reset() {
        std::fill(p_.begin(), p_.end(), 0);
        std::fill(h_.begin(), h_.end(), uint64_t(-1));
        std::fill(q_.begin(), q_.end(), -1);
        std::fill(b_.begin(), b_.end(), 0);
        i_ = 0;
        a_ = h_.size() - 1;
        m_ = h_.size();
        count_ = 0;
    }
    SuperMinHash(SuperMinHash &&o):
        pol_(o.pol_), a_(o.a_), i_(o.i_), m_(o.m_),
#if NOT_THREADSAFE
    count_(o.count_),
#  if VERBOSE_AF
    inner_loop_count_(o.inner_loop_count_),
#  endif
#else
    count_{o.count_.load()},
#  if VERBOSE_AF
    inner_loop_count_(o.inner_loop_count_.load()),
#  endif
#endif
        seed_(o.seed_), p_(std::move(o.p_)), h_(std::move(o.h_)), q_(std::move(o.q_)), b_(std::move(o.b_)),bbits_(o.bbits_)
    {
    }
    SuperMinHash(const SuperMinHash &o):
        pol_(o.pol_), a_(o.a_), i_(o.i_), m_(o.m_),
#if NOT_THREADSAFE
    count_(o.count_),
#  if VERBOSE_AF
    inner_loop_count_(o.inner_loop_count_),
#  endif
#else
    count_{o.count_.load()},
#  if VERBOSE_AF
    inner_loop_count_(o.inner_loop_count_.load()),
#  endif
#endif
        seed_(o.seed_), p_(o.p_), h_(o.h_), q_(o.q_), b_(o.b_), bbits_(o.bbits_)
    {
    }
    static constexpr uint64_t join_cmp(uint32_t i, uint32_t r) {
        return (uint64_t(i) << 32) | r;
    }
    size_t needed_bits() const {
        return std::ceil(32 + std::log2(m_));
    }
#if VERBOSE_AF
    ~SuperMinHash() {
        std::fprintf(stderr, "%zu for total count, %zu for inner loop count for a ratio of %lf currently\n", size_t(count_), inner_loop_count_.load(), count_ ? float(inner_loop_count_) / count_: -1.);
    }
#endif
    void add(uint64_t item) {addh(item);}
    void addh(uint64_t item) {
        ++count_;
        RNGType gen(item ^ seed_);
        uint64_t j = 0;
        while(j <= a_) {
#if VERBOSE_AF
            ++inner_loop_count_;
#endif
            uint32_t k = pol_.mod(gen());
            assert(k < m_);
            auto qfunc = [&](auto x) {
                if(q_[x] != i_)
                    q_[x] = i_, p_[x] = x;
            };
            qfunc(j);
            qfunc(k);
            std::swap(p_[k], p_[j]);
            auto crj = (uint64_t(j) << 32) | gen();
            if(crj < h_[p_[j]]) {
                auto jprime = std::min(m_ - 1, uint32_t(h_[p_[j]] >> 32));
                h_[p_[j]] = crj;
                if(j < jprime) {
                    --b_[jprime];
                    ++b_[j];
                    while(b_[a_] == 0)
                        --a_;
                }
            }
            ++j;
        }
        ++i_;
    }
    size_t write(gzFile fp) const {
        return this->finalize().write(fp);
    }
    size_t write_unfinalized(gzFile fp) const {
        size_t ret = h_.size();
        ret = gzwrite(fp, &ret, sizeof(ret));
        char buf[sizeof(*this)];
        std::memcpy(buf, this, sizeof(*this));
#define CLEAR_CON(x) std::memset(buf + offsetof(SuperMinHash, x), 0, sizeof(x))
        CLEAR_CON(p_);
        CLEAR_CON(q_);
        CLEAR_CON(h_);
        CLEAR_CON(b_);
#undef CLEAR_CON
        ret += gzwrite(fp, buf, sizeof(buf));
        ret += gzwrite(fp, p_.data(), sizeof(p_[0]) * p_.size());
        ret += gzwrite(fp, h_.data(), sizeof(h_[0]) * h_.size());
        ret += gzwrite(fp, q_.data(), sizeof(q_[0]) * q_.size());
        ret += gzwrite(fp, b_.data(), sizeof(b_[0]) * b_.size());
        return ret;
    }
    size_t read(gzFile fp) {
        size_t nelem;
        size_t ret = gzread(fp, &nelem, sizeof(nelem));
        ret += gzread(fp, this, sizeof(*this));
        pol_ = Policy<CountType>(nelem);
        p_.resize(nelem);
        ret += gzread(fp, p_.data(), sizeof(p_[0]) * p_.size());
        h_.resize(nelem);
        ret += gzread(fp, h_.data(), sizeof(h_[0]) * h_.size());
        q_.resize(nelem);
        ret += gzread(fp, q_.data(), sizeof(q_[0]) * q_.size());
        b_.resize(nelem);
        ret += gzread(fp, b_.data(), sizeof(b_[0]) * b_.size());
        return ret;
    }
    size_t size() const {return h_.size();}
    double cardinality_estimate() const {
        const auto *ptr = &h_;
        decltype(h_) tmp;
        if(std::find(h_.begin(), h_.end(), UINT64_C(-1)) != h_.end()) {
            tmp = h_;
            detail::densifybin(tmp);
            ptr = &tmp;
        }
        double cest = detail::harmonic_cardinality_estimate_diffmax_impl(*ptr, h_.size() << 32);
        return cest;
    }
    FinalDivBBitMinHash finalize(uint32_t b=0) const {
        if(b == 0) b = bbits_;
        const auto *ptr = &h_;
        decltype(h_) tmp;
        double cest = -1;
        if(std::find(h_.begin(), h_.end(), UINT64_C(-1)) != h_.end()) {
            tmp = h_;
            decltype(tmp) t2(tmp);
            for(auto &e: t2) e &= 0xFFFFFFFFu;
            cest = detail::harmonic_cardinality_estimate_diffmax_impl(t2, 1ull<<32);
            detail::densifybin(tmp);
            ptr = &tmp;
        } else {
            std::vector<uint64_t> t2; t2.reserve(h_.size());
            for(const auto e: h_) t2.push_back(e & 0xFFFFFFFFu);
            cest = detail::harmonic_cardinality_estimate_diffmax_impl(t2, 1ull<<32);
        }
        if(b > needed_bits()) {
            if(ptr == &h_) {
                tmp = h_;
                ptr = &tmp;
            }
            for(auto &e: tmp) {
                RNGType gen(e);
                e = gen();
            }
        }
        return div_bbit_finalize(b, *ptr, cest);
    }
    auto cfinalize(uint32_t b=0) const {return finalize(b ? b: bbits_);}
    void clear() {
        SuperMinHash tmp(std::move(*this));
    }
    DBSKETCH_WRITE_STRING_MACROS
    DBSKETCH_READ_STRING_MACROS
    using final_type = FinalDivBBitMinHash;
};



template<typename CountingType, typename=typename std::enable_if<
            std::is_arithmetic<CountingType>::value
        >::type>
struct FinalCountingBBitMinHash;


template<typename T, typename Hasher=WangHash>
class DivBBitMinHasher {
    std::vector<T, Allocator<T>> core_;
    uint32_t b_;
    schism::Schismatic<T> div_;
    Hasher hf_;
public:
    using final_type = FinalDivBBitMinHash;
    template<typename... Args>
    DivBBitMinHasher(unsigned nbuckets, unsigned b, Args &&... args):
        core_(roundupdiv(nbuckets, 64), detail::default_val<T>()),
        b_(b), div_(core_.size()), hf_(std::forward<Args>(args)...)
    {
#if VERBOSE_AF
        if(nbuckets % 64) {
            std::fprintf(stderr, "Warning: rounded n buckets from %u for %u so that it's convenient for our comparison strategy.\n", nbuckets, unsigned(div_.d_));
        }
#endif
        assert(core_.size() % 64 == 0);
        if(b_ < 1 || b_ > 64) throw std::invalid_argument("b must be >= 1 and <= 64");
    }
    bool operator==(const DivBBitMinHasher &o) const {
        return b_ == o.b_ && std::equal(core_.begin(), core_.end(), o.core_.begin());
    }
    bool operator!=(const DivBBitMinHasher &o) const {
        return !operator==(o);
    }
    void show() const {for(const auto v: core_) std::fprintf(stderr, "%zu\t", size_t(v)); std::fputc('\n', stderr);}
    void addh(T val) {val = hf_(val);add(val);}
    void clear() {
        std::fill(core_.begin(), core_.end(), detail::default_val<T>());
    }
    T nbuckets() const {return div_.d_;} // also core_.size()
    INLINE void add(T hv) {
        const T bucket = div_.mod(hv);
        const T quot = div_.div(hv);
        auto &ref = core_[bucket];
#ifdef NOT_THREADSAFE
        ref = std::min(quot, ref);
#else
        while(quot < ref)
            __sync_bool_compare_and_swap(std::addressof(ref), ref, quot);
#endif
    }
    void write(const char *fn, int compression=6) const {
        finalize().write(fn, compression);
    }
    void write(const std::string &fn, int compression=6) const {write(fn.data(), compression);}
    int densify() {
        const int rc = detail::densifybin(core_);
#if VERBOSE_AF
        switch(rc) {
            case -1: std::fprintf(stderr, "[W] Can't densify empty thing\n"); break;
            case 0: std::fprintf(stderr, "The densification, it does nothing\n"); break;
            case 1: std::fprintf(stderr, "Densifying something that needs it\n");
        }
#endif
        return rc;
    }
    double cardinality_estimate() const {
        return detail::harmonic_cardinality_estimate(core_);
    }
    FinalDivBBitMinHash finalize(unsigned b=0) const;
    FinalDivBBitMinHash cfinalize(unsigned b=0) const {return finalize(b);}
};


template<typename FT=float, typename KT=uint64_t, typename CT=uint32_t>
class ICWSampler {
    // https://static.googleusercontent.com/media/research.google.com/en//pubs/archive/36928.pdf
    // Improved Consistent Sampling, Weighted Minhash and L1 Sketching, Sergey Ioffe
    static_assert(std::is_floating_point<FT>::value, "FT must be floating point");
    static_assert(std::is_integral<KT>::value, "KT must be integral");
    std::vector<KT> keys_;
    std::vector<FT> vals_;
    std::vector<CT> t_;
    std::mutex mut_;
    FT l1sum_ = 0.;
    uint64_t seed_;
    // TODO: reduce comparisons for low-count items (See 3.4)
    // TODO: consider HIP/CUDA port
public:
    ICWSampler(size_t n, uint64_t seed=0): seed_(seed) {
        keys_.resize(n, std::numeric_limits<KT>::max());
        vals_.resize(n, std::numeric_limits<FT>::max());
        t_.resize(n, std::numeric_limits<CT>::max());
    }
    void addh(KT key, FT count) {
        if(count <= static_cast<FT>(0)) return;
        wy::WyRand<uint32_t, 2> rng;
        Gamma21<FT> gamgen;
        std::uniform_real_distribution<FT> urd;
        l1sum_ += std::abs(count);
        // Don't add a key twice, it's bad.
        auto lc = std::log(count);
        for(size_t i = 0; i < size(); ++i) {
            auto r = gamgen(rng), c = gamgen(rng), b = urd(rng);
            const auto t = std::floor(lc / r + b);
            const auto y = std::exp(r * (t - b));
            const auto a = c / (y * std::exp(r));
            if(a < vals_[i]) {
                std::lock_guard<decltype(mut_)> guard(mut_);
                if(a < vals_[i]) { // Second check, in case this was changed while we waited for the lock
                    vals_[i] = a;
                    keys_[i] = key;
                    t_[i] = t;
                }
            }
        }
    }
    std::vector<KT> to_vector() const {
        std::vector<KT> ret(size());
        XXH3PairHasher xh;
        for(size_t i = 0; i < size(); ++i)
            ret[i] = xh(keys_[i], t_[i]);
        return ret;
    }
    using final_type = FinalDivBBitMinHash;
    final_type finalize(unsigned b=64u) const {
        b = std::min(b, 64u);
        auto vec = to_vector();
        return div_bbit_finalize(b, vec, l1sum_);
    }
    size_t size() const {return vals_.size();}
};

struct phll_t;

template<typename T, typename Hasher=WangHash>
class BBitMinHasher {
    std::vector<T, common::Allocator<T>> core_;
    uint32_t b_, p_;
    Hasher hf_;
public:
    void free() {
        std::vector<T, common::Allocator<T>>().swap(core_);
    }
    using final_type = FinalBBitMinHash;
    static constexpr size_t NBITS = sizeof(T) * CHAR_BIT;
    BBitMinHasher(unsigned p): BBitMinHasher(p, NBITS - p) {}
    template<typename... Args>
    BBitMinHasher(unsigned p, unsigned b, Args &&... args):
        core_(size_t(1) << p, detail::default_val<T>()), b_(b), p_(p), hf_(std::forward<Args>(args)...)
    {
        if(b_ + p_ > sizeof(T) * CHAR_BIT) {
            char buf[512];
            std::sprintf(buf, "[E:%s:%s:%d] Width of type (%zu) is insufficient for selected p/b parameters (%d/%d)",
                         __FILE__, __PRETTY_FUNCTION__, __LINE__, sizeof(T) * CHAR_BIT, int(b_), int(p_));
            throw std::runtime_error(buf);
        }
        postcondition_require(is_pow2(core_.size()), "should be a power of two");
    }
    bool operator==(const BBitMinHasher &o) const {
        return b_ == o.b_ && p_ == o.p_ && std::equal(core_.begin(), core_.end(), o.core_.begin());
    }
    bool operator!=(const BBitMinHasher &o) const {
        return !operator==(o);
    }
    void reset() {
        std::fill(core_.begin(), core_.end(), detail::default_val<T>());
    }
    DBSKETCH_READ_STRING_MACROS
    ssize_t read(gzFile fp) {
        throw NotImplementedError("NotImplemented function. This is likely an error, as you probabyl don't mean to call this.");
        return -1;
    }
    auto getp() const {return p_;}
    auto getb() const {return b_;}
    BBitMinHasher compress(unsigned p, unsigned b=0) const {
        if(!b) b = b_;
        if(p == p_) return *this;
        if(p > p_) throw std::invalid_argument("Can't compress a sketch to a larger size");
        BBitMinHasher ret(p, b);
        const T increment = (std::numeric_limits<T>::max() >> p_) + 1;
        const unsigned shift = p_ - p, ratio = 1 << shift;
        auto start = core_.data();
        for(size_t i = 0; i < ret.core_.size(); ++i, start += ratio) {
            unsigned j = 0;
            do {
                if(start[j] != detail::default_val<T>()) {
                    ret.core_[i] = start[j] + j * increment;
                    break;
                }
            } while(++j != ratio);
        }
        return ret;
    }
    void addh(T val) {val = hf_(val);add(val);}
    void clear() {
        std::fill(core_.begin(), core_.end(), detail::default_val<T>());
    }
    void swap(BBitMinHasher &o) {
        std::swap_ranges((uint8_t *)this, (uint8_t *)this + sizeof(*this), (uint8_t *)std::addressof(o));
    }
    void add(T hv) {
        auto &ref = core_[hv>>(sizeof(T) * CHAR_BIT - p_)];
        hv <<= p_; hv >>= p_; // Clear top values
#ifdef NOT_THREADSAFE
        ref = std::min(ref, hv);
        assert(ref <= hv);
#else
        while(hv < ref)
            __sync_bool_compare_and_swap(std::addressof(ref), ref, hv);
#endif
    }
    void write(const char *fn, int compression=6, uint32_t b=0) const {
        finalize(b ? b: b_).write(fn, compression);
    }
    void write(const std::string &fn, int compression=6, uint32_t b=0) const {write(fn.data(), compression, b);}
    void write(gzFile fp, uint32_t b=0) const {
        finalize(b?b:b_).write(fp);
    }
    void show() const {
        std::fprintf(stderr, "b_: %u. size: %zu\n", b_, size());
        for(size_t i = 0; i < size(); ++i) {
            if(core_[i] != detail::default_val<T>())
                std::fprintf(stderr, "I: %zu. V: %zu\n", i, size_t(core_[i]));
        }
    }
    int densify() {
        auto rc = detail::densifybin(core_);
#if VERBOSE_AF
        switch(rc) {
            case -1: std::fprintf(stderr, "[W] Can't densify empty thing\n"); break;
            case 0: std::fprintf(stderr, "The densification, it does nothing\n"); break;
            case 1: std::fprintf(stderr, "Densifying something that needs it\n");
        }
#endif
        return rc;
    }
    double make_alpha() const {
        auto m = 1u << p_;
        switch(1u << p_) {
            case 16: return .673;
            case 32: return .697;
            case 64: return .709;
            default: return 0.7213 / (1 + 1.079/m);
        }
    }
    size_t nmatches(const BBitMinHasher &o) const {
        size_t ret = core_[0] == o.core_[0];
        for(size_t i = 1; i < core_.size(); ++i)
            ret += core_[i] == o.core_[i];
        return ret;
    }
    double jaccard_index(const BBitMinHasher &o) const {
        auto it = core_.begin(), oit = o.core_.begin();
        size_t ret = *it++ == *oit++;
        do {
            ret += *it++ == *oit++;
        } while(it != core_.end());
        return double(ret) / core_.size();
    }
    double intersection_size(const BBitMinHasher &o) const {
        const long double numinv = 1. / std::ldexp(static_cast<long double>(1.), NBITS - p_);
        auto f = [numinv](const auto v) {return v * numinv;};
        long double tmp = 0.;
        size_t nshared = 0;
        for(size_t i = 0; i < core_.size(); ++i) {
            auto v = std::min(core_[i], o.core_[i]);
            tmp += f(v != detail::default_val<T>() ? v: (std::numeric_limits<T>::max() >> p_));
            nshared += core_[i] == o.core_[i];
        }
        return nshared * std::pow(core_.size(), 2) / (tmp * core_.size());
        // Returns estimate of cardinality of union, multiplied by the est jaccard index
    }
    double union_size(const BBitMinHasher &o) const {
        //auto it = core_.begin(), oit = o.core_.begin();
        const long double numinv = 1. / std::ldexp(static_cast<long double>(1.), NBITS - p_);
        auto f = [numinv](const auto v) {return v * numinv;};
        long double tmp = 0.;
        for(size_t i = 0; i < core_.size(); ++i) {
            auto v = std::min(core_[i], o.core_[i]);
            tmp += f(v != detail::default_val<T>() ? v: (std::numeric_limits<T>::max() >> p_));
        }
        return std::pow(core_.size(), 2) / tmp;
    }
    double cardinality_estimate(MHCardinalityMode mode=HARMONIC_MEAN) const {
        if(std::find_if(core_.begin(), core_.end(), [](auto x) {return x != detail::default_val<T>();}) == core_.end())
            return 0.; // Empty sketch
        if(likely(mode == HARMONIC_MEAN)) {
            const long double numinv = 1. / std::ldexp(static_cast<long double>(1.), NBITS - p_);
            auto f = [numinv](const auto v) {return v * numinv;};
            long double tmp = 0.;
            for(const auto v: core_)
                tmp += f(v != detail::default_val<T>() ? v: (std::numeric_limits<T>::max() >> p_));
            return std::pow(core_.size(), 2) / tmp;
        }
        const long double num = std::ldexp(static_cast<long double>(1.), sizeof(T) * CHAR_BIT - p_);
        double sum;
        using CT = std::decay_t<decltype(core_)>;
        CT tmp;
        const auto *ptr = &core_;
        switch(mode) {
        case ARITHMETIC_MEAN: // better? Still not great.
            return arithmean(ptr->begin(), ptr->end(), [num](auto x) {return num / x;}) * core_.size();
        case MEDIAN: {
            std::vector<double> ests(ptr->size());
            for(size_t i = 0; i < ests.size(); ++i)
                ests[i] = num / core_[i];
            std::sort(ests.begin(), ests.end());
            return (ests[ests.size() / 2] + (ests[ests.size() / 2 - 1])) * .5 * core_.size() * make_alpha();
        }
        case GEOMETRIC_MEAN: // better? Still not great.
            // This can be accelerated with vector class library's fast log routines and conversion operations
            return
                geomean_invprod(ptr->begin(), ptr->end(), num) * core_.size() * make_alpha();
            // pth root of product of all estimates, where p = core_.size()
            // exp(1/p * [log(num) * len(minimizers) - sum(log(x) for x in minimizers)])
            // Then, times the number of minimizers, because we've partitioned the data into that many streams.
        case HLL_METHOD: {
            std::array<uint32_t, 64> arr{0};
            auto diff = p_ - 1;
            for(const auto v: (*ptr))
                ++arr[v == detail::default_val<T>() ? 0: integral::clz(v) - diff];
            return hll::detail::ertl_ml_estimate(arr, p_, sizeof(T) * CHAR_BIT - p_, 0);
        }
        default: HEDLEY_UNREACHABLE(); // IMPOCEROUS
        }
        return sum;
    }
    size_t size() const {return core_.size();}
    BBitMinHasher &operator+=(const BBitMinHasher &o) {
        if(size() != o.size()) throw std::runtime_error("Wrong sizes");
        if(size() == 0) throw std::runtime_error("Empty sketches");
        std::fprintf(stderr, "Size: %zu\n", size());
        size_t i = 0;
#if __AVX512F__
        __m512i *p1 = reinterpret_cast<__m512i *>(core_.data());
        const __m512i *p2 = reinterpret_cast<const __m512i *>(o.core_.data());
        CONST_IF(sizeof(T) == 8) {
            for(; i < core_.size() / (sizeof(__m512i) / sizeof(T)); ++i) {
                _mm512_store_si512(p1 + i, _mm512_min_epu64(_mm512_load_si512(p1 + i), _mm512_load_si512(p2 + i)));
            }
        } else CONST_IF(sizeof(T) == 4) {
            for(; i < core_.size() / (sizeof(__m512i) / sizeof(T)); ++i) {
                _mm512_store_si512(p1 + i, _mm512_min_epu32(_mm512_load_si512(p1 + i), _mm512_load_si512(p2 + i)));
            }
        }
#else /* no avx512f */
#    if __AVX2__
        CONST_IF(sizeof(T) == 4) {
            __m256i *p1 = reinterpret_cast<__m256i *>(core_.data());
            const __m256i *p2 = reinterpret_cast<const __m256i *>(o.core_.data());
            for(i = 0; i < core_.size() / (sizeof(__m256i) / sizeof(T)); ++i) {
                _mm256_store_si256(p1 + i, _mm256_min_epu32(_mm256_loadu_si256(p1 + i), _mm256_loadu_si256(p2 + i)));
            }
            i *= (sizeof(__m256i) / sizeof(T));
        }
#    elif __SSE2__
        CONST_IF(sizeof(T) == 4) {
            __m128i *p1 = reinterpret_cast<__m128i *>(core_.data());
            const __m128i *p2 = reinterpret_cast<const __m128i *>(o.core_.data());
            for(i = 0; i < core_.size() / (sizeof(__m128i) / sizeof(uint64_t)); ++i) {
                _mm_storeu_si128(p1 + i, _mm_min_epu32(_mm_loadu_si128(p1 + i), _mm_loadu_si128(p2 + i)));
            }
            i *= (sizeof(__m128i) / sizeof(T));
        }
#    endif
#endif
        if(i < core_.size()) {
            std::transform(core_.begin() + i, core_.end(),
                           o.core_.begin() + i,
                           core_.begin() + i,
                           [](auto x, auto y) {return std::min(x, y);});
        }
        return *this;
    }
    BBitMinHasher operator+(const BBitMinHasher &o) const {
        auto ret(*this);
        ret += o;
        return ret;
    }
    std::vector<uint32_t> cudapack32(uint64_t b=0) const {
        // Lower simd, higher parallelism
        b = b ? b: b_;
        PREC_REQ(b <= 32, "b can't be > 64");
        PREC_REQ(p_ >= 5, "p must be >= 5 for this");
        std::vector<uint32_t> ret(b * (core_.size() >> 5));
        for(size_t i = 0; i < core_.size(); ++i) {
            for(unsigned bi = 0; bi < b; ++bi) {
                detail::setnthbit(&ret[(i >> 5) * b + bi], (core_[i] >> bi) & 1);
            }
        }
        return ret;
    }
    std::vector<uint64_t> cudapack64(uint64_t b=0) const {
        // Lower simd, higher parallelism
        b = b ? b: b_;
        PREC_REQ(b <= 64, "b can't be > 64");
        PREC_REQ(p_ >= 6, "p must be >= 5 for this");
        std::vector<uint64_t> ret(b * (core_.size() >> 6));
        for(size_t i = 0; i < core_.size(); ++i) {
            for(unsigned bi = 0; bi < b; ++bi) {
                detail::setnthbit(&ret[(i >> 6) * b + bi], (core_[i] >> bi) & 1);
            }
        }
        return ret;
    }
    FinalBBitMinHash finalize(uint32_t b=0) const;
    FinalBBitMinHash cfinalize(uint32_t b=0) const;
    double wh_base() const {
        return std::pow((long double)(1uL << (64 - p_)), 1.L/254);
    }
    whll::wh119_t make_whll() const {
        long double base = wh_base();
        std::vector<uint8_t, Allocator<uint8_t>> retvec(core_.size());
        long double d = 1.L/ std::log(base);
        const uint8_t maxv = 255;
        for(size_t i = 0; i < core_.size(); ++i) {
            if(core_[i] != detail::default_val<T>()) {
                long double v = core_[i];
                retvec[i] = v ? uint8_t(maxv - std::ceil(std::log(v) * d)) /*- 1*/: maxv;
            }
        }
        return whll::wh119_t(retvec, base);
    }
    auto make_hll() const {
        hll::hllbase_t<Hasher> ret(p_);
        const auto ptr = ret.mutable_core().data();
        for(size_t i = 0; i < ret.core().size(); ++i) {
            ptr[i] = core_[i] == detail::default_val<T>()
                   ? 0: clz(((core_[i] << 1)|1) << (p_ - 1)) + 1;
        }
        ret.sum();
        return ret;
    }
    auto make_packed16hll() const {
        //std::fprintf(stderr, "TODO [%s]: update estimation to account for lowering the radix for p_ >= 8\n", __PRETTY_FUNCTION__);
        std::vector<uint8_t, Allocator<uint8_t>> retvec(core_.size() >> 1);
        //static constexpr long double base = 16;
        for(size_t i = 0; i < retvec.size(); ++i) {
            auto reg2val = [dv=detail::default_val<T>()] (auto x) {
                return x == dv ? uint8_t(0): ((uint8_t(clz(x)) >> 2) + 1);
            };
            retvec[i] = (reg2val(core_[2*i]) << 4) | (reg2val(core_[2*i+1]) & 0xF);
        }
        return phll_t(retvec);
    }
    const auto &core() const {return core_;}
};


template<typename HashStruct=WangHash>
class WideHyperLogLogHasher: public BBitMinHasher<uint64_t, HashStruct> {
public:
    using super = BBitMinHasher<uint64_t, HashStruct>;
    using final_type = whll::wh119_t;
    WideHyperLogLogHasher(unsigned p): BBitMinHasher<uint64_t, HashStruct>(p, 64 - p) {
    }
    whll::wh119_t finalize() const {return super::make_whll();}
    whll::wh119_t cfinalize() const {return finalize();}
    double cardinality_estimate() const {
        return finalize().cardinality_estimate();
    }
};
template<typename HashStruct=WangHash>
class Packed16HyperLogLogHasher: public BBitMinHasher<uint64_t, HashStruct> {
public:
    using super = BBitMinHasher<uint64_t, HashStruct>;
    using final_type = phll_t;
    template<typename... Args>
    Packed16HyperLogLogHasher(Args &&...args): BBitMinHasher<uint64_t, HashStruct>(std::forward<Args>(args)...) {
    }
    phll_t finalize() const {return super::make_packed16hll();}
    phll_t cfinalize() const {return finalize();}
    double cardinality_estimate() const {
        return finalize().cardinality_estimate();
    }
};

template<typename HashStruct=WangHash>
class HyperLogLogHasher: public BBitMinHasher<uint64_t, HashStruct> {
public:
    using super = BBitMinHasher<uint64_t, HashStruct>;
    using final_type = hll::hll_t;
    template<typename... Args>
    HyperLogLogHasher(Args &&...args): BBitMinHasher<uint64_t, HashStruct>(std::forward<Args>(args)...) {
    }
    operator hll::hllbase_t<HashStruct>() const {return this->finalize();}
    hllbase_t<HashStruct> finalize() const {return super::make_hll();}
    hllbase_t<HashStruct> cfinalize() const {return finalize();}
    double cardinality_estimate() const {
        return finalize().cardinality_estimate();
    }
};


template<typename T, typename Hasher=WangHash>
void swap(BBitMinHasher<T, Hasher> &a, BBitMinHasher<T, Hasher> &b) {
    a.swap(b);
}


template<typename T, typename CountingType, typename Hasher=WangHash>
class CountingBBitMinHasher: public BBitMinHasher<T, Hasher> {
    using super = BBitMinHasher<T, Hasher>;
    std::vector<CountingType, Allocator<CountingType>> counters_;
    // Not threadsafe currently
    // Consider bitpacking with p_ bits for counting
public:
    using final_type = FinalCountingBBitMinHash<CountingType>;
    template<typename... Args>
    CountingBBitMinHasher(unsigned p, unsigned b, Args &&... args): super(p, b, std::forward<Args>(args)...), counters_(1ull << p) {}
    void add(T hv) {
        auto ind = hv>>(sizeof(T) * CHAR_BIT - this->p_);
        auto &ref = this->core_[ind];
        hv <<= this->p_; hv >>= this->p_; // Clear top values. We could also shift/mask, but that requires two other constants vs 1.
        if(ref < hv)
            ref = hv, counters_[ind] = 1;
        else if(ref == hv) ++ref;
    }
    FinalCountingBBitMinHash<CountingType> finalize(uint32_t b=0) const;
    std::pair<CountingType, CountingType> multiset_comparison(const CountingBBitMinHasher &o) const {
        PREC_REQ(this->size() == o.size(), "Can't compare different-sized sketches");
        CountingType shared_sum = 0, total_sum = 0;
        for(size_t i = 0; i < this->size(); ++i) {
            if(this->core_[i] == o.core_[i]) {
                shared_sum += std::min(this->counters_[i], o.counters_[i]);
                total_sum += std::max(this->counters_[i], o.counters_[i]);
            } else total_sum += this->counters_[i] + o.counters_[i];
        }
        return std::make_pair(shared_sum, total_sum);
    }
    auto histogram_intersection(const CountingType &o) const {
        auto p = multiset_comparison(o);
        return double(p.first) / double(p.second);
    }
};


struct FinalBBitMinHash {
protected:
    FinalBBitMinHash() {}
public:
    using value_type = uint64_t;
    double est_cardinality_;
    uint32_t b_, p_;
    std::vector<value_type, Allocator<value_type>> core_;
    FinalBBitMinHash(unsigned p, unsigned b, double est): est_cardinality_(est), b_(b), p_(p),
        core_((value_type(b) << p) >> 6)
    {
#if VERBOSE_AF
        std::fprintf(stderr, "Initializing finalbb with %u for b and %u for p. Number of u64s: %zu. Total nbits: %zu\n", b, p, core_.size(), core_.size() * 64);
#endif
    }
    void free() {
        decltype(core_) tmp;
        std::swap(tmp, core_);
    }
    bool operator==(const FinalBBitMinHash &o) const {
        return b_ == o.b_ && p_ == o.p_ && std::equal(core_.data(), core_.data() + core_.size(), o.core_.data());
    }
    bool operator!=(const FinalBBitMinHash &o) const {
        return !operator==(o);
    }
    FinalBBitMinHash(const std::string &path): FinalBBitMinHash(path.data()) {}
    FinalBBitMinHash(const char *path): est_cardinality_(0), b_(0), p_(0) {
        read(path);
    }
    FinalBBitMinHash(gzFile fp): est_cardinality_(0), b_(0), p_(0) {
        read(fp);
    }
    double cardinality_estimate() const {return est_cardinality_;}
    size_t nblocks() const {
        return size_t(1) << p_;
    }
    FinalBBitMinHash(FinalBBitMinHash &&o) = default;
    FinalBBitMinHash(const FinalBBitMinHash &o) = default;
    template<typename T, typename Hasher=WangHash>
    FinalBBitMinHash(BBitMinHasher<T, Hasher> &&o): FinalBBitMinHash(std::move(o.finalize())) {
        o.free();
    }
    template<typename T, typename Hasher=WangHash>
    FinalBBitMinHash(const BBitMinHasher<T, Hasher> &o): FinalBBitMinHash(std::move(o.finalize())) {}
    double r() const {
        return std::ldexp(est_cardinality_, -int(sizeof(value_type) * CHAR_BIT - p_));
    }
    double ab() const {
        const auto _r = r();
        auto rm1 = 1. - _r;
        auto rm1p = std::pow(rm1, std::ldexp(1., b_) - 1);
        return _r * rm1p / (1. - (rm1p * rm1));
    }
    std::array<double, 3> full_set_comparison(const FinalBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        double me_only = est_cardinality_ > is ? est_cardinality_ - is: 0.,
               o_only  = o.est_cardinality_ > is ? o.est_cardinality_ - is: 0.;
        return std::array<double, 3>{{me_only, o_only, is}};
    }
    ssize_t read(gzFile fp) {
        uint32_t arr[2];
        if(gzread(fp, arr, sizeof(arr)) != sizeof(arr)) throw ZlibError("Could not read from file.");
        b_ = arr[0];
        p_ = arr[1];
        if(gzread(fp, &est_cardinality_, sizeof(est_cardinality_)) != sizeof(est_cardinality_)) throw ZlibError("Could not read from file.");
        core_.resize((value_type(b_) << p_) >> 6);
        const size_t nb = sizeof(core_[0]) * core_.size();
        if(gzread(fp, core_.data(), nb) != ssize_t(nb)) throw ZlibError("Could not read data vector from file");
        return nb + sizeof(arr) + sizeof(est_cardinality_);
    }
    DBSKETCH_READ_STRING_MACROS
    DBSKETCH_WRITE_STRING_MACROS
    ssize_t write(gzFile fp) const {
        uint32_t arr[] {b_, p_};
        if(HEDLEY_UNLIKELY(gzwrite(fp, arr, sizeof(arr)) != sizeof(arr))) throw std::runtime_error("Could not write to file");
        if(HEDLEY_UNLIKELY(gzwrite(fp, &est_cardinality_, sizeof(est_cardinality_)) != sizeof(est_cardinality_))) throw std::runtime_error("Could not write to file");
        const size_t nb = sizeof(core_[0]) * core_.size();
        if(HEDLEY_UNLIKELY(gzwrite(fp, core_.data(), nb) != ssize_t(nb))) throw std::runtime_error("Could not write to file");
        ssize_t ret = sizeof(arr) + sizeof(est_cardinality_) + nb;
        return ret;
    }
    uint64_t popcnt() const {
        Space::VType tmp;
        const Type *op(reinterpret_cast<const Type *>(&core_[0])),
                   *ep(reinterpret_cast<const Type *>(&core_[core_.size()]));
        auto sum = popcnt_fn(*op++);
        while(op < ep)
            sum = Space::add(sum, popcnt_fn(*op++));
        return sum_of_u64s(sum);
    }
#if HAS_AVX_512
    template<typename Func1, typename Func2>
    uint64_t equal_bblocks_sub(const uint64_t *p1, const uint64_t *pe, const uint64_t *p2, const Func1 &f1, const Func2 &f2) const {
        using VT = __m512i;
        if(core_.size() * sizeof(core_[0]) < sizeof(__m512i)) {
            uint64_t sum = f2(*p1++, *p2++);
            while(p1 != pe)
                sum += f2(*p1++, *p2++);
            return sum;
        }
        const VT *vp1 = reinterpret_cast<const VT *>(p1);
        const VT *vpe = reinterpret_cast<const VT *>(pe);
        const VT *vp2 = reinterpret_cast<const VT *>(p2);
        auto sum = f1(*vp1++, *vp2++);
        while(vp1 != vpe) sum = _mm512_add_epi64(f1(*vp1++, *vp2++), sum);
        return sum_of_u64s(sum);
    }
#elif __AVX2__
    template<typename Func1, typename Func2>
    uint64_t equal_bblocks_sub(const uint64_t *p1, const uint64_t *pe, const uint64_t *p2, const Func1 &f1, const Func2 &f2) const {
        using VT = __m256i;
        if(core_.size() * sizeof(core_[0]) < sizeof(__m256i)) {
            uint64_t sum = f2(*p1++, *p2++);
            while(p1 != pe)
                sum += f2(*p1++, *p2++);
            return sum;
        }
        const VT *vp1 = reinterpret_cast<const VT *>(p1);
        const VT *vpe = reinterpret_cast<const VT *>(pe);
        const VT *vp2 = reinterpret_cast<const VT *>(p2);
        auto sum = f1(*vp1++, *vp2++);
        while(vp1 != vpe) sum = _mm256_add_epi64(f1(*vp1++, *vp2++), sum);
        return sum_of_u64s(sum);
    }
#else
    template<typename Func1, typename Func2>
    uint64_t equal_bblocks_sub(const uint64_t *p1, const uint64_t *pe, const uint64_t *p2, const Func1 &f1, const Func2 &f2) const {
        using VT = __m128i;
        uint64_t sum = 0;
        if(core_.size() * sizeof(core_[0]) >= sizeof(Space::VType)) {
            const VT *vp1 = reinterpret_cast<const VT *>(p1);
            const VT *vpe = reinterpret_cast<const VT *>(pe);
            const VT *vp2 = reinterpret_cast<const VT *>(p2);
            do {sum += f1(*vp1++, *vp2++);} while(vp1 != vpe);
            p1 = reinterpret_cast<const uint64_t *>(vp1), p2 = reinterpret_cast<const uint64_t *>(vp2);
        }
        while(p1 != pe)
            sum += f2(*p1++, *p2++);
        return sum;
    }
#endif
    uint64_t equal_bblocks(const FinalBBitMinHash &o) const {
        switch(b_) {
            case 4: return eq::count_eq_nibbles((uint8_t *)core_.data(), (uint8_t *)o.core_.data(), core_.size() * 16);
            case 8: return eq::count_eq_bytes((uint8_t *)core_.data(), (uint8_t *)o.core_.data(), core_.size() * 8);
            case 16: return eq::count_eq_shorts((uint16_t *)core_.data(), (uint16_t *)o.core_.data(), core_.size() * 4);
            case 32: return eq::count_eq_words((uint32_t *)core_.data(), (uint32_t *)o.core_.data(), core_.size() * 2);
            case 64: return eq::count_eq_longs(core_.data(), o.core_.data(), core_.size());
            default: ;
        }
        assert(o.core_.size() == core_.size());
        const value_type *p1 = core_.data(), *pe = core_.data() + core_.size(), *p2 = o.core_.data();
        assert(b_ <= 64); // b_ > 64 not yet supported, though it could be done with a larger hash
        // p_ already guaranteed to be greater than 6
        switch(p_) {
            case 6: {
                auto match = ~(*p1++ ^ *p2++);
                while(p1 != pe) match &= ~(*p1++ ^ *p2++);
                return popcount(match);
            }
            case 7: {
                const __m128i *vp1 = reinterpret_cast<const __m128i *>(p1), *vp2 = reinterpret_cast<const __m128i *>(p2), *vpe = reinterpret_cast<const __m128i *>(pe);
                __m128i match = ~(*vp1++ ^ *vp2++);
                while(vp1 != vpe)
                    match &= ~(*vp1++ ^ *vp2++);
                return popcount(common::vatpos(match, 0)) + popcount(common::vatpos(match, 1));
            }
#if __AVX2__
            case 8: return common::sum_of_u64s(detail::matching_bits(reinterpret_cast<const __m256i *>(p1), reinterpret_cast<const __m256i *>(p2), b_));
#  if HAS_AVX_512
            case 9: return common::sum_of_u64s(detail::matching_bits(reinterpret_cast<const __m512i *>(p1), reinterpret_cast<const __m512i *>(p2), b_));
            default: {
                // Process each 'b' remainder block in
                const __m512i *vp1 = reinterpret_cast<const __m512i *>(p1), *vp2 = reinterpret_cast<const __m512i *>(p2);
                auto sum = detail::matching_bits(vp1, vp2, b_);
                for(size_t i = 1; i < (size_t(1) << (p_ - 9u)); ++i) {
                    vp1 += b_;
                    vp2 += b_;
                    sum = _mm512_add_epi64(detail::matching_bits(vp1, vp2, b_), sum);
                }
                assert((value_type*)vp1 == &core_[core_.size()]);
                return common::sum_of_u64s(sum);
            }
#    else /* has avx2 not not 512 */
            default: {
                const __m256i *vp1 = reinterpret_cast<const __m256i *>(p1), *vp2 = reinterpret_cast<const __m256i *>(p2);
                auto sum = detail::matching_bits(vp1, vp2, b_);
                for(size_t i = 1; i < 1ull << (p_ - 8u); ++i) {
                    vp1 += b_;
                    vp2 += b_;
                    sum = _mm256_add_epi64(detail::matching_bits(vp1, vp2, b_), sum);
                    assert(vp1 <= reinterpret_cast<const __m256i *>(pe));
                }
#ifndef NDEBUG
                auto fptr = (value_type*)(reinterpret_cast<const __m256i *>(p1) + (size_t(b_) << (p_ - 8u)));
                assert(fptr == (p1 + core_.size()) || !std::fprintf(stderr, "fptr: %p. optr: %p\n", static_cast<const void *>(fptr), static_cast<const void *>(p1 + core_.size())));
#endif
                return common::sum_of_u64s(sum);
            }
#  endif /* avx512 or avx2 */
#else /* assume SSE2 */
            default: {
                // Process each 'b' remainder block in
                const __m128i *vp1 = reinterpret_cast<const __m128i *>(p1), *vp2 = reinterpret_cast<const __m128i *>(p2), *vpe = reinterpret_cast<const __m128i *>(pe);

                auto notincxor = [&](){return ~(*vp1++ ^ *vp2++);};
                __m128i match = notincxor();
                for(unsigned b = b_; --b;match &= notincxor());
                auto sum = popcount(*(const uint64_t *)&match) + popcount(((const uint64_t *)&match)[1]);
                while(vp1 != vpe) {
                    match = ~(*vp1++ ^ *vp2++);
                    for(unsigned b = b_; --b; match &= notincxor());
                    sum += popcount(*(const uint64_t *)&match) + popcount(((const uint64_t *)&match)[1]);
                }
                return sum;
            }
#endif
        }
    }
    double frac_equal(const FinalBBitMinHash &o) const {
        auto num = equal_bblocks(o);
        std::fprintf(stderr, "numeq: %zu. total blocks: %zu\n", size_t(num), core_.size() * (64 / b_));
        return std::ldexp(num, -int(p_));
    }
    uint64_t nmin() const {
        return uint64_t(1) << p_;
    }
    size_t nmatches(const FinalBBitMinHash &o) const {
        return equal_bblocks(o);
    }
    double jaccard_index(const FinalBBitMinHash &o) const {
        /*
         * reference: https://arxiv.org/abs/1802.03914.
        */
        const double b2pow = std::ldexp(1., -b_);
        double frac = frac_equal(o);
        frac -= b2pow;
        return std::max(0., frac / (1. - b2pow));
    }
    double containment_index(const FinalBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        return is / est_cardinality_;
    }
    double intersection_size(const FinalBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        return is;
    }
    double union_size(const FinalBBitMinHash &o) const {
        double ji = jaccard_index(o);
        double is = (est_cardinality_ + o.est_cardinality_) * ji / (1. + ji);
        return est_cardinality_ + o.est_cardinality_ - is;
    }
}; // FinalBBitMinHash

INLINE double jaccard_index(const FinalBBitMinHash &a, const FinalBBitMinHash &b) {
    return a.jaccard_index(b);
}


#define DEFAULT_SET_CASE(num, type, p_) \
        default:\
            for(size_t ov = 0, ev = 1 << (p_ - num); ov != ev; ++ov) {\
                static constexpr size_t ratio = sizeof(type) / sizeof(FinalType);\
                static constexpr size_t bits_per_vector = sizeof(type) * CHAR_BIT;\
                auto main_ptr = ret.core_.data() + ov * ratio * b;\
                auto core_ptr = core_ref.data() + ov * bits_per_vector;\
                for(size_t _b = 0; _b < b; ++_b) {\
                    auto ptr = main_ptr + (_b * ratio);\
                    SK_UNROLL_8\
                    for(size_t i = 0; i < bits_per_vector; ++i)\
                        setnthbit(ptr, i, getnthbit(core_ptr[i], _b));\
                }\
            }\
            break
#define SET_CASE(num, type, p_) \
        case num:\
            for(size_t _b = 0; _b < b; ++_b) {\
                auto ptr = ret.core_.data() + (_b * sizeof(type)/sizeof(FinalType));\
                for(size_t i = 0; i < (sizeof(type) * CHAR_BIT); ++i)\
                    setnthbit(ptr, i, getnthbit(core_ref[i], _b));\
            }\
            break

template<typename T, typename Hasher>
FinalBBitMinHash BBitMinHasher<T, Hasher>::finalize(uint32_t b) const {
    b = b ? b: b_; // Use the b_ of BBitMinHasher if not specified; this is because we can make multiple kinds of bbit minhashes from the same hasher.
    assert(b);
    assert(core_.size() % 64 == 0);
    const auto *ptr = &core_;
    std::decay_t<decltype(core_)> tmp;
    size_t ndef;
    double cest = -1.;
    if((ndef = std::count_if(core_.begin(), core_.end(), [](auto x) {return x == detail::default_val<T>();}))) {
#ifndef NDEBUG
        std::fprintf(stderr, "requires densification: %zu/%zu need to be densified\n", ndef, core_.size());
#endif
        tmp = core_;
        cest = detail::harmonic_cardinality_estimate_impl(tmp);
        std::replace(tmp.begin(), tmp.end(), std::numeric_limits<T>::max() >> p_, detail::default_val<T>());
        int ret = detail::densifybin(tmp);
        if(ret < 0) {
            std::fprintf(stderr, "Could not densify empty sketch; setting all minimizers to empty value. It will compare completely equal to all empty sketches and full dissimilar to all others.\n");
        }
        ptr = &tmp;
    }
    const auto &core_ref = *ptr;
    if(cest < 0) cest = detail::harmonic_cardinality_estimate_impl(core_ref);
    using detail::getnthbit;
    using detail::setnthbit;
    FinalBBitMinHash ret(p_, b, cest);
    using FinalType = typename FinalBBitMinHash::value_type;
#if !NDEBUG
#define CASE_6_TEST\
                for(size_t i = 0; i < core_ref.size(); ++i) {\
                    for(size_t _b = 0; _b < b; ++_b) {\
                        assert(getnthbit(ret.core_.data() + _b, i) == getnthbit(core_ref[i], _b));\
                    }\
                }
#else
#define CASE_6_TEST
#endif
    if(b_ == 64) {
        // We've already failed for the case of b_ + p_ being greater than the width of T
        std::copy(core_ref.begin(), core_ref.end(), (uint64_t *)ret.core_.data());
    } else if(b_ == 32) {
        std::copy(core_ref.begin(), core_ref.end(), (uint32_t *)ret.core_.data());
    } else if(b_ == 16) {
        std::copy(core_ref.begin(), core_ref.end(), (uint16_t *)ret.core_.data());
    } else if(b_ == 8) {
        std::copy(core_ref.begin(), core_ref.end(), (uint8_t *)ret.core_.data());
    } else if(b_ == 4) {
        auto rp = (uint8_t *)ret.core_.data();
        const size_t end = core_ref.size() >> 1;
        for(size_t i = 0; i < end; ++i) {
            *rp++ = ((core_ref[2 * i] & 0xFu) << 4) | (core_ref[2 * i + 1] & 0xFu);
        }
    } else {
        if(HEDLEY_UNLIKELY(p_ < 6))
            throw std::runtime_error("BBit minhashing requires at least p = 6 for non-power of two b currently. We could reduce this requirement using 32-bit integers.");
        // Pack an SSE2 element for each 1 << (p_ - 7)
        switch(p_) {
        case 6:
            for(size_t _b = 0; _b < b; ++_b) {
                auto ptr = ret.core_.data() + (_b * sizeof(uint64_t)/sizeof(FinalType));
                for(size_t i = 0; i < (sizeof(uint64_t) * CHAR_BIT); ++i)
                    setnthbit(ptr, i, getnthbit(core_ref[i], _b));
            }
            CASE_6_TEST
            break;
        SET_CASE(7, __m128i, p_);
#if __AVX2__
        SET_CASE(8, __m256i, p_);
#if HAS_AVX_512
        SET_CASE(9, __m512i, p_);

        DEFAULT_SET_CASE(9u, __m512i, p_);
#else
        DEFAULT_SET_CASE(8u, __m256i, p_);
#endif
#else /* no avx2 or 512 */
        DEFAULT_SET_CASE(7u, __m128i, p_);
#endif
        }
    }
    return ret;
}

template<typename T, typename Hasher>
FinalBBitMinHash BBitMinHasher<T, Hasher>::cfinalize(uint32_t b) const {
    return finalize(b);
}

template<typename T, typename Allocator>
FinalDivBBitMinHash div_bbit_finalize(uint32_t b, const std::vector<T, Allocator> &core_ref, double est_v) {
    using detail::getnthbit;
    using detail::setnthbit;
    if(est_v == 0.) est_v = detail::harmonic_cardinality_estimate_impl(core_ref);
    //std::fprintf(stderr, "Calling with core_ref size of %zu and b as %d\n", core_ref.size(), b);
    FinalDivBBitMinHash ret(core_ref.size(), b, est_v);
    using FinalType = typename FinalDivBBitMinHash::value_type;
    assert(ret.core_.size() % b == 0 || !(b & (b - 1)));
    assert(core_ref.size() % 64 == 0 || !(b & (b - 1)));
    //std::fprintf(stderr, "core size: %zu being collapsed into b (%d)-bit samples in core of size %zu\n", core_ref.size(), int(b), ret.core_.size());
    // TODO: consider supporting non-power of 2 numbers of minimizers by subsetting to the first k <= (1<<p) minimizers.
    if(b == 64) {
        std::memcpy(ret.core_.data(), core_ref.data(), sizeof(core_ref[0]) * core_ref.size());
#if 0
    } else if(b == 32) {
        std::copy(core_ref.begin(), core_ref.end(), (uint32_t *)ret.core_.data());
    } else if(b == 16) {
        std::copy(core_ref.begin(), core_ref.end(), (uint16_t *)ret.core_.data());
    } else if(b == 8) {
        std::copy(core_ref.begin(), core_ref.end(), (uint8_t *)ret.core_.data());
#endif
    } else {
        const auto l2szfloor = ilog2(core_ref.size());
        const auto pow2 = 1ull << l2szfloor;
        if(l2szfloor < 6) {
            std::fprintf(stderr, "l2sz: %u\n", l2szfloor);
            throw std::runtime_error("FinalDivBBitMinHash currently requires at least 64 minimizers.");
        }
        switch(l2szfloor) {
        case 6:
                for(size_t _b = 0; _b < b; ++_b)
                    for(size_t i = 0; i < 64u; ++i)
                        ret.core_.operator[](i / (sizeof(T) * CHAR_BIT) * b + _b) |= (core_ref[i] & (FinalType(1) << _b)) << (i % (sizeof(FinalType) * CHAR_BIT));
                CASE_6_TEST
            break;
        SET_CASE(7, __m128i, l2szfloor);
#if __AVX2__
        SET_CASE(8, __m256i, l2szfloor);
#  if HAS_AVX_512
        SET_CASE(9, __m512i, l2szfloor);

        DEFAULT_SET_CASE(9u, __m512i, l2szfloor);
#  else
        DEFAULT_SET_CASE(8u, __m256i, l2szfloor);
#  endif
#else /* no avx2 or 512 */
        DEFAULT_SET_CASE(7u, __m128i, l2szfloor);
#endif
        }
        if(pow2 != core_ref.size()) {
            assert(is_pow2(core_ref.size() - (core_ref.size() & ((1ull << l2szfloor) - 1))));
#define LEFTOVERS(type)\
            for(size_t ind = pow2 / (sizeof(type) * CHAR_BIT);ind < core_ref.size() / (sizeof(type) * CHAR_BIT);++ind) {\
                auto main_ptr = ret.core_.data() + ind * sizeof(type) / sizeof(FinalType) * b;\
                auto core_ptr = core_ref.data() + ind * sizeof(type) * CHAR_BIT;\
                /* std::fprintf(stderr, "main ptr is %zd away from the end of the main thing\n", (ret.core_.data() + ret.core_.size()) - main_ptr); */\
                /* std::fprintf(stderr, "core ptr is %zd away from the end of the core thing\n", (core_ref.data() + core_ref.size()) - core_ptr); */\
                for(auto _b = 0u; _b < b; ++_b) {\
                    auto ptr = main_ptr + (_b * sizeof(type)/sizeof(FinalType));\
                    for(size_t i = 0u; i < sizeof(type) * CHAR_BIT; ++i) {\
                        setnthbit(ptr, i, getnthbit(core_ptr[i], _b));\
                    }\
                }\
            }\
            size_t ind = core_ref.size() / (sizeof(type) * CHAR_BIT) * (sizeof(type) * CHAR_BIT); // Get the last n
#if HAS_AVX_512
            LEFTOVERS(__m512i)
#elif __AVX2__
            LEFTOVERS(__m256i)
#else
            LEFTOVERS(__m128i)
#endif
            while(ind < core_ref.size()) {
                //std::fprintf(stderr, "Rem: %zd. Mod of rem: %zd\n", core_ref.size() - ind, (core_ref.size() - ind) % 64);
                auto core_ptr = core_ref.data() + ind;
                auto ref_ptr  = ret.core_.data() + (ind / (sizeof(FinalType) * CHAR_BIT) * b);
                //std::fprintf(stderr, "ret vore Rem: %zd. Mod of rem: %zd\n", &*ret.core_.end() - ref_ptr, (&*ret.core_.end() - ref_ptr) % 64);
                for(size_t _b = 0; _b < b; ++_b)
                    for(size_t i = 0; i < 64u; ++i)
                        setnthbit(ref_ptr, i, getnthbit(core_ptr, _b));
                ind += sizeof(FinalType) * CHAR_BIT;
            }
        }
    }
    return ret;
}

#undef DEFAULT_SET_CASE
#undef SET_CASE
template<typename T, typename Hasher>
FinalDivBBitMinHash DivBBitMinHasher<T, Hasher>::finalize(uint32_t b) const {
    b = b ? b: b_; // Use the b_ of DivBBitMinHasher if not specified; this is because we can make multiple kinds of bbit minhashes from the same hasher.
    const auto *ptr = &core_;
    std::remove_const_t<decltype(core_)> tmp;
    if(std::find(core_.begin(), core_.end(), detail::default_val<T>()) != core_.end()) {
        tmp = core_;
        detail::densifybin(tmp);
        ptr = &tmp;
    }
    const auto &core_ref = *ptr;
    return div_bbit_finalize<T>(b, core_ref);
}


template<typename CountingType, typename>
struct FinalCountingBBitMinHash: public FinalBBitMinHash {
    std::vector<CountingType, Allocator<CountingType>> counters_;
    FinalCountingBBitMinHash(FinalBBitMinHash &&tmp, const std::vector<CountingType, Allocator<CountingType>> &counts): FinalBBitMinHash(std::move(tmp)), counters_(counts), cached_sum_(std::accumulate(counters_.begin(), counters_.end(), CountingType(0))) {}
    CountingType cached_sum_;
    FinalCountingBBitMinHash(unsigned p, unsigned b, double est): FinalBBitMinHash(p, b, est), counters_(size_t(1) << this->p_) {}
    FinalCountingBBitMinHash(std::string path) {
        this->read(path);
    }
    ssize_t write(gzFile fp) const {
        ssize_t ret = FinalBBitMinHash::write(fp);
        ssize_t nb = counters_.size() * sizeof(counters_[0]);
        if(gzwrite(fp, counters_.data(), nb) != nb) throw std::runtime_error("reading from file failed");
        ret += nb;
        return ret;
    }
    ssize_t read(gzFile fp) {
        ssize_t ret = FinalBBitMinHash::read(fp);
        counters_.resize(size_t(1) << this->p_);
        ssize_t nb = counters_.size() * sizeof(counters_[0]);
        if(gzread(fp, counters_.data(), nb) != nb) throw std::runtime_error("reading from file failed");
        ret += nb;
        cached_sum_ = std::accumulate(counters_.begin(), counters_.end(), CountingType(0));
        return ret;
    }
    bool operator==(const FinalCountingBBitMinHash &o) {
        return std::equal(this->core_.begin(), this->core_.end(), o.core_.begin()) &&
               std::equal(this->counters_.begin(), this->counters_.end(), o.counters_.begin());
    }
    DBSKETCH_WRITE_STRING_MACROS
    DBSKETCH_READ_STRING_MACROS

    struct HistResult {
        uint64_t matched_sum_,
                 total_sum_,
                 matched_bits_,
                 total_bits_;
        double jaccard_index() const {
            return static_cast<double>(matched_bits_) / total_bits_;
        }
        double weighted_jaccard_index() const {
            return static_cast<double>(matched_sum_) / total_sum_;
        }
        void print(std::FILE *fp=stderr) const {
            std::fprintf(fp, "Matched sum: %zu, total sum: %zu. matched bits: %zu. total bits: %zu.\n", matched_sum_, total_sum_, matched_bits_, total_bits_);
            std::fprintf(fp, "ji: %lf. wji: %lf\n",  jaccard_index(), weighted_jaccard_index());
        }
    };
    template<typename TT>
    void twobit_range(TT v, unsigned offset) {
        for(size_t i = 0; i < sizeof(v) * CHAR_BIT; i += 2) {

        }
    }
    template<typename=typename std::enable_if<std::is_same<CountingType, uint32_t>::value>::type> // Only finished for uint32_t currently
    HistResult histogram_sums(const FinalCountingBBitMinHash &o) const {
        if(!std::is_same<CountingType, uint32_t>::value) throw NotImplementedError("histogram_sums only available for uint32_t");
        PREC_REQ(o.core_.size() == core_.size(), "mismatched sizes");
        const uint64_t *p1 = core_.data(), *pe = core_.data() + core_.size(), *p2 = o.core_.data();
        assert(b_ <= 64 || !std::fprintf(stderr, "b_: %u\n", b_) || (7 < 64)); // b_ > 64 not yet supported, though it could be done with a larger hash
        assert(o.core_.size() == core_.size());
        assert(b_ <= 64); // b_ > 64 not yet supported, though it could be done with a larger hash
        auto l2szfloor = p_;
        uint64_t sum, total_sum = 0;
        switch(l2szfloor) {
            case 6: {
                auto match = ~(*p1++ ^ *p2++);
                while(p1 != pe) match &= ~(*p1++ ^ *p2++);
                if(match) {
                    sum = popcount(match);
                    do {
                        auto ind = ctz(match);
                        total_sum += std::min(counters_[ind], o.counters_[ind]);
                        auto t = match & -match;
                        match &= t;
                    } while(match);
                } else {
                    sum = 0;
                }
                break;
            }
            case 7: {
                const __m128i *vp1 = reinterpret_cast<const __m128i *>(p1), *vp2 = reinterpret_cast<const __m128i *>(p2), *vpe = reinterpret_cast<const __m128i *>(pe);
                __m128i match = ~(*vp1++ ^ *vp2++);
                while(vp1 != vpe)
                    match &= ~(*vp1++ ^ *vp2++);
                uint64_t v1 = common::vatpos(match, 0), v2 = common::vatpos(match, 1);
                sum = popcount(v1) + popcount(v2);
                if(v1) {
                    do {
                        auto ind = ctz(v1);
                        total_sum += std::min(counters_[ind], o.counters_[ind]);
                        auto t = v1 & -v1;
                        v1 &= t;
                    }  while(v1);
                }
                if(v2) {
                    do {
                        auto ind = ctz(v2);
                        total_sum += std::min(counters_[ind + 64], o.counters_[ind + 64]);
                        auto t = v2 & -v2;
                        v2 &= t;
                    }  while(v2);
                }
                break;
            }
#if __AVX2__
            case 8: {
                auto bv = detail::sbit_accum(reinterpret_cast<const __m256i *>(p1), reinterpret_cast<const __m256i *>(p2), b_);
                sum = common::sum_of_u64s(popcnt256(bv));
                if(sum) {
                    for(size_t i = 0; i < 4; ++i) {
                        auto v = common::vatpos(bv, i);
                        if(v) {
                            auto offset = i * 64;
                            do {
                                auto ind = ctz(v);
                                total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                                auto t = v & -v;
                                v &= t;
                            } while(v);
                        }
                    }
                }
                break;
            }
#  if HAS_AVX_512
            case 9: {
                auto bv = detail::sbit_accum(reinterpret_cast<const __m512i *>(p1), reinterpret_cast<const __m512i *>(p2), b_);
                sum = common::sum_of_u64s(popcnt512(bv));
                if(sum) {
                    for(size_t i = 0; i < 8; ++i) {
                        auto v = common::vatpos(bv, i);
                        if(v) {
                            do {
                            auto offset = i * 64;
                            auto ind = ctz(v);
                            total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                            auto t = v & -v;
                            v &= t;
                            } while(v);
                        }
                    }
                }
                break;
            }
            default: {
                // Process each 'b' remainder block in
                uint64_t offset = 0;
                const __m512i *vp1 = reinterpret_cast<const __m512i *>(p1), *vp2 = reinterpret_cast<const __m512i *>(p2), *vpe = reinterpret_cast<const __m512i *>(pe);
                auto bv = detail::sbit_accum(vp1, vp2, b_);
                auto lsum = popcnt512(bv);
                for(size_t i = 0; i < 8; ++i) {
                    auto v = common::vatpos(bv, i);
                    if(v) {
                        do {
                            auto ind = ctz(v);
                            total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                            auto t = v & -v;
                            v &= t;
                        } while(v);
                    }
                    offset += 64;
                }
                for(size_t i = 1; i < (size_t(1) << (l2szfloor - 9u)); ++i) {
                    vp1 += b_;
                    vp2 += b_;
                    bv = detail::sbit_accum(vp1, vp2, b_);
                    lsum = _mm512_add_epi64(popcnt512(bv), lsum);
                    for(size_t i = 0; i < 8; ++i) {
                        auto v = common::vatpos(bv, i);
                        if(v) {
                            do {
                                auto ind = ctz(v);
                                total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                                auto t = v & -v;
                                v &= t;
                            } while(v);
                        }
                        offset += 64;
                    }
                }
                assert((value_type*)vp1 == &core_[core_.size()]);
                sum = common::sum_of_u64s(lsum);
                break;
            }
#    else /* has avx2 not not 512 */
            default: {
                const __m256i *vp1 = reinterpret_cast<const __m256i *>(p1), *vp2 = reinterpret_cast<const __m256i *>(p2);
                auto bv = detail::sbit_accum(vp1, vp2, b_);
                auto lsum = popcnt256(bv);
                uint64_t offset = 0;
                for(size_t i = 0; i < 4; ++i) {
                    auto v = common::vatpos(bv, i);
                    if(v) {
                        do {
                            auto ind = ctz(v);
                            total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                            auto t = v & -v;
                            v &= t;
                        } while(v);
                    }
                    offset += 64;
                }
                for(size_t i = 1; i < 1ull << (l2szfloor - 8u); ++i) {
                    vp1 += b_;
                    vp2 += b_;
                    bv = detail::sbit_accum(vp1, vp2, b_);
                    lsum = _mm256_add_epi64(popcnt256(bv), lsum);
                    assert(vp1 <= reinterpret_cast<const __m256i *>(pe));
                    for(size_t i = 0; i < 4; ++i) {
                        auto v = common::vatpos(bv, i);
                        if(v) {
                            do {
                                auto ind = ctz(v);
                                total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                                auto t = v & -v;
                                v &= t;
                            } while(v);
                        }
                        offset += 64;
                    }
                }
                sum = common::sum_of_u64s(lsum);
                break;
            }
#  endif /* avx512 or avx2 */
#else /* assume SSE2 */
            default: {
                // Process each 'b' remainder block in
                const __m128i *vp1 = reinterpret_cast<const __m128i *>(p1), *vp2 = reinterpret_cast<const __m128i *>(p2), *vpe = reinterpret_cast<const __m128i *>(pe);
                __m128i match = ~(*vp1++ ^ *vp2++);
                for(unsigned b = b_; --b;match &= ~(*vp1++ ^ *vp2++));
                auto v1 = vatpos(match, 0), v2 = vatpos(match, 1);
                auto lsum = popcount(v1) + popcount(v2);
                if(v1) {
                    do {
                        auto ind = ctz(v1);
                        total_sum += std::min(counters_[ind], o.counters_[ind]);
                        v1 &= (v1 & -v1);
                    } while(v1);
                }
                if(v2) {
                    do {
                        auto ind = ctz(v2);
                        total_sum += std::min(counters_[ind + 64], o.counters_[ind + 64]);
                        v2 &= (v2 & -v2);
                    } while(v2);
                }
                uint64_t offset = 128;
                while((uint64_t *)vp1 + 2 <= (uint64_t *)vpe) {
                    match = ~(*vp1++ ^ *vp2++);
                    for(unsigned b = b_; --b; match &= ~(*vp1++ ^ *vp2++));
                    v1 = vatpos(match, 0), v2 = vatpos(match, 1);
                    lsum += popcount(v1) + popcount(v2);
                    if(v1) {
                        do {
                            auto ind = ctz(v1);
                            total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                            v1 &= (v1 & -v1);
                        } while(v1);
                    }
                    offset += 64;
                    if(v2) {
                        do {
                            auto ind = ctz(v2);
                            total_sum += std::min(counters_[ind + offset], o.counters_[ind + offset]);
                            v2 &= (v2 & -v2);
                        } while(v2);
                    }
                    offset += 64;
                }
                sum = lsum;
                break;
            }
#endif
        } // switch(l2szfloor)
        HistResult ret{total_sum, cached_sum_ + o.cached_sum_ - total_sum, sum, core_.size() * sizeof(core_[0]) * sizeof(char)};
        return ret;
    }
};

template<typename T, typename CountingType, typename Hasher>
FinalCountingBBitMinHash<CountingType> CountingBBitMinHasher<T, CountingType, Hasher>::finalize(uint32_t b) const {
    auto bbm = BBitMinHasher<T, Hasher>::finalize(b);
    return FinalCountingBBitMinHash<CountingType>(std::move(bbm), this->counters_);
}

template<typename T>
decltype(auto) intersection_size(const T &x, const T &y) {
    return x.intersection_size(y);
}

} // minhash
namespace mh = minhash;
} // namespace sketch


#endif /* #ifndef SKETCH_BB_MINHASH_H__*/
