### Comparison of lncRNA dynamics with protein-coding gene pre-mRNA dynamics

# Load required R packages
library(tidyverse)
library(ggsci)
library(RColorBrewer)
library(gplots)
library(ggsci)

# Load required data (note the required clustering info was generated by the script 'get_data_for_Fig_1.R')
counts <- readRDS("data/t98g_filtered_coords_clust.rds")

# Check to see if figure 4 directory already exists and if not create it
if(!dir.exists("figures/Fig_4")){dir.create("figures/Fig_4")}



## Heatmap of coding gene pre-mRNA (Fig. 4a)

# Get just coding data
coding <-  filter(counts, transcript_type=="protein_coding")  %>%
  drop_na(premrna_cluster)

# Define function to find the approximate time when mRNA expression is at its peak 
get_mRNA_peak_time <- function(input_mRNA){
  # Get mRNA expression
  mRNA_expression <- data.frame(time = seq(0,400,10), mRNA=unlist(input_mRNA))
  # Fit loess
  lw1 <- loess(mRNA ~ time, data = mRNA_expression, span = 0.5)
  # Return the time when mRNA expression is ~ maximal
  return(seq(0,400,10)[which.max(lw1$fitted)])
}

# Quickly create a small figure to illustrate what `get_mRNA_peak_time()` does
# Create some noisy gene expression data
set.seed(123)
test_data <- data.frame(time = seq(0,400,10),
                        true_gene_expression=sin(seq(0,4,0.1)),
                        gene_expression=sin(seq(0,4,0.1)) + rnorm(41, mean = 0, sd=0.1))
# Plot and add vertical line for our estimate of when gene expression peaks
ggsave(filename = "figures_track/get_mRNA_peak_time.png", 
       plot = ggplot(test_data, aes(x=time, y=gene_expression)) + 
         geom_point() +
         geom_line(aes(y = true_gene_expression)) +
         geom_smooth(span = 0.5, se = FALSE) +
         geom_vline(xintercept=get_mRNA_peak_time(test_data$gene_expression)),
       device = "png", width = 15, height = 9, units = "cm")

# order rows by peak in mRNA expression
coding <- coding[order(sapply(X = coding$exon, FUN = get_mRNA_peak_time)),]

# Order rows by pre-mRNA cluster number
coding <- arrange(coding, premrna_cluster)

# Get colours for sidebars which indicate cluster membership
cols <- pal_npg("nrc", alpha = 1)(length(unique(coding$premrna_cluster)))
cols <- cols[coding$premrna_cluster]

# Restrict to just expression data and scale rows to have mean zero and standard deviation one
plot_data <- coding %>%
  dplyr::select(first_10Kb) %>%
  unnest(cols = c(first_10Kb))
# Scale the data
plot_data <- t(apply(plot_data, 1, scale))

# Note that we again scale the input expression matrix on rows (genes) to Z-scores to facilitate meaningful visual comparison. What percentage of values are outside the z-score range of -2.5 to 2.5?
paste0(round(100*sum(as.vector(plot_data)>2.5 |  as.vector(plot_data)< -2.5)/length(as.vector(plot_data)),2),"%")
# We'll use the breaks argument
breaks <- seq(-2.5, 2.5, by=0.2)
# add outliers
breaks <- append(breaks, max(as.vector(plot_data)))
breaks <- append(breaks, min(as.vector(plot_data)),0)

# Get colours for the heatmap
heatmap_cols <- colorRampPalette(brewer.pal(9,"PuBuGn"))(length(breaks)-1)

# Define plotting function
fig4a <- function(){
  require(gplots)
  heatmap.2(plot_data,
            Rowv=NA, Colv=NA,
            col=heatmap_cols,
            scale="none", # already done
            density.info="none",
            trace="none",
            main="",
            RowSideColors=cols,
            dendrogram="none",
            key=FALSE,
            labRow="",
            labCol=c("0","","","","","50","","","","","100","","","","","150","","","","","200","","","","","250","","","","","300","","","","","350","","","","","400"), srtCol = 0, cexCol=1.1,
            adjCol=c(NA,0.2), # shift x-axis labels slightly left or right
            offsetCol=0.5, # move x axis labels up and down
            breaks=breaks,
            lmat=rbind(c(9,5,6), c(4,1,2),c(7,8,3)),
            lwid = c(1,1,9), # 1st column - no dendrogram, 2nd column - coloured bars, 3rd column -heatmap
            lhei = c(1,10,1), # 1st row - key, 2nd row heatmap
            margins = c(0.5, 5),
            useRaster = TRUE)
}

# Plot heatmap and save out as svg - later to be assembled as multipanel pdf
pdf("figures/Fig_4/Fig4a1.pdf", width=8, height=10)
fig4a()
dev.off()

# Define a function to plot row z-score key
color.bar <- function(lut=heatmap_cols, min=-3, max=3, nticks=7, ticks=seq(min, max, len=nticks), title='', rectangle_edges=breaks) {
  rectangle_edges[1] <- min
  rectangle_edges[length(rectangle_edges)] <- max
  
  plot(c(min,max),c(0,10), type='n', bty='n', xaxt='n', xlab='', yaxt='n', ylab='', main=title)
  axis(1, ticks, las=1, labels=FALSE, lwd=2)
  for (i in 1:(length(lut))) {
    x1=rectangle_edges[i]
    x2=rectangle_edges[i+1]
    rect(x1,0,x2,10, col=lut[i], border=NA)
  }
}

# Save out the row z-score key
pdf("figures/Fig_4/Fig4a_zscore_key.pdf", width=3, height=0.5)
par(mar=c(1,0.2,0.2,0.2)) # bottom, left, top and right margins
color.bar()
dev.off()



### Heatmap of coding genes (mRNA) ordered by induction dynamics (Fig. 4b)

# Get mRNA expression data and scale rows to have mean zero and standard deviation one
plot_data <- coding %>%
  dplyr::select(exon) %>%
  unnest(cols = c(exon))
# Scale the data
plot_data <- t(apply(plot_data, 1, scale))

# Note that we again scale the input expression matrix on rows (genes) to Z-scores to facilitate meaningful visual comparison.
# What percentage of values are outside the z-score range of -2 to 2?
paste0(round(100*sum(as.vector(plot_data)>2 |  as.vector(plot_data)< -2)/length(as.vector(plot_data)),2),"%")
# use the breaks argument
breaks <- seq(-2, 2, by=0.2)
# add outliers
breaks <- append(breaks, max(as.vector(plot_data)))
breaks <- append(breaks, min(as.vector(plot_data)),0)

# Get colours for the heatmap
heatmap_cols <- colorRampPalette(brewer.pal(9,"PuBuGn"))(length(breaks)-1)

fig4b <- function(){
  library(gplots)
  heatmap.2(plot_data,
            Rowv=NA, Colv=NA,
            col=heatmap_cols,
            scale="none", # already done
            density.info="none",
            trace="none",
            main="",
            RowSideColors=cols,
            dendrogram="none",
            key=FALSE,
            labRow="",
            labCol=c("0","","","","","50","","","","","100","","","","","150","","","","","200","","","","","250","","","","","300","","","","","350","","","","","400"), srtCol = 0, cexCol=1.1,
            adjCol=c(NA,0.2), # shift x-axis labels slightly left or right
            offsetCol=0.5, # move x axis labels up and down
            breaks=breaks,
            lmat=rbind(c(9,5,6), c(4,1,2),c(7,8,3)),
            lwid = c(1,1,9), # 1st column - no dendrogram, 2nd column - coloured bars, 3rd column -heatmap
            lhei = c(1,10,1), # 1st row - key, 2nd row heatmap
            margins = c(0.5, 5),
            useRaster = TRUE)
}

# Plot heatmap and save out as svg - later to be assembled as multipanel pdf
pdf("figures/Fig_4/Fig4b1.pdf", width=8, height=10)
fig4b()
dev.off()

# Function to plot row z-score key
color.bar <- function(lut=heatmap_cols, min=-3, max=3, nticks=7, ticks=seq(min, max, len=nticks), title='', rectangle_edges=breaks) {
  rectangle_edges[1] <- min
  rectangle_edges[length(rectangle_edges)] <- max
  
  plot(c(min,max),c(0,10), type='n', bty='n', xaxt='n', xlab='', yaxt='n', ylab='', main=title)
  axis(1, ticks, las=1, labels=FALSE, lwd=2)
  for (i in 1:(length(lut))) {
    x1=rectangle_edges[i]
    x2=rectangle_edges[i+1]
    rect(x1,0,x2,10, col=lut[i], border=NA)
  }
}

# Save out the row z-score key
pdf("figures/Fig_4/Fig4b_zscore_key.pdf", width=3, height=0.5)
par(mar=c(1,0.2,0.2,0.2)) # bottom, left, top and right margins
color.bar()
dev.off()



### Plot cluster centroids comparing lncRNA and coding pre-mRNA expression (Fig. 4c)
#We plot the centroids of the lncRNA and coding pre-mRNA (first 10Kb of intron). We also look at the Pearson correlation coefficients between each centroid.

# Define function to find the cluster centroids, 5th and 95th percentiles of expression values
get_centroids <- function(input_df, cluster_col){
  # Get number of clusters
  num_clusters <- length(unique(input_df[[cluster_col]]))
  # Restrict to just expression data and scale rows to have mean zero and standard deviation one
  input_df <- input_df[!sapply(input_df$first_10Kb, is.null),]
  expr_only <- input_df %>%
    dplyr::select(first_10Kb) %>%
    unnest(cols = c(first_10Kb))
  # Scale the data
  expr_only <- t(apply(expr_only, 1, scale))
  # Remove NA rows that can be introduced if alll zero
  input_df <- input_df[!is.na(rowSums(expr_only)),]
  expr_only <- expr_only[!is.na(rowSums(expr_only)),]
  
  # define data frame to hold centroids
  centroids <- data.frame(matrix(nrow=num_clusters, ncol=41))
  # define data frame to hold 5th percentile
  cent5 <- data.frame(matrix(nrow=num_clusters, ncol=41))
  # define data frame to hold 95th percentile
  cent95 <- data.frame(matrix(nrow=num_clusters, ncol=41))
  for(n in 1:num_clusters){
    # Select one of the clusters
    cluster_x <- as.matrix(expr_only[input_df[[cluster_col]]==n,])
    centroid<-colMeans(cluster_x)
    # Save cluster centroid
    centroids[n,] <- centroid
    # Save 5th percentild of the cluster
    cent5[n,]<-apply(cluster_x , 2 , quantile , probs = c(0.05) , na.rm = TRUE )
    # Save 95th percentild of the cluster
    cent95[n,]<-apply(cluster_x , 2 , quantile , probs = c(0.95) , na.rm = TRUE )
  }
  return(list(centroids, cent5, cent95))
}

# Define function to plot centroids plus 95th and 5th percentiles
plot_centroid <- function(centroid_num, centroids_df){
  # define time
  t <- seq(0,400,10)
  # Get cumber of clusters
  num_clusters <- nrow(centroids_df[[1]])
  # Get colours for plotting:
  # light coloured-background representing 5% and 95% precentiles
  percentile_cols <- ggsci::pal_npg("nrc", alpha = 0.3)(num_clusters)
  # Dark coloured centroids
  centroid_cols <- ggsci::pal_npg("nrc", alpha = 1)(num_clusters)
  
  plot(t,
       unlist(centroids_df[[1]][centroid_num,]), 
       ylim=range(c(centroids_df[[1]][centroid_num,],
                    centroids_df[[2]][centroid_num,],
                    centroids_df[[3]][centroid_num,])),
       type="l",
       col=centroid_cols[centroid_num],
       lwd=2,
       xlab="", ylab="", xaxt="n", yaxt="n")
  polygon(c(t, rev(t)), c(centroids_df[[2]][centroid_num,], rev(centroids_df[[3]][centroid_num,])),
          col = percentile_cols[centroid_num], border = NA)
}

# Get centroid info
coding_centroids <- get_centroids(filter(counts, transcript_type=="protein_coding") %>% drop_na(premrna_cluster), "premrna_cluster")
noncoding_centroids <- get_centroids(filter(counts, transcript_type!="protein_coding"), "lncrna_cluster")

fig4c <- function(){
  # Set up a grid
  par(mfcol=c(7,7))
  # Set thin plot margins
  par(mar=c(0.2,0.2,0.2,0.2)) # bottom, left, top and right margins
  
  # Create a blank plot for the top left corner
  plot(0,type='n',axes=FALSE,ann=FALSE)
  
  # Plot all of the non-coding centroids in the left-most column of the grid
  plot_centroid(centroid_num = 1,centroids_df = noncoding_centroids)
  plot_centroid(centroid_num = 2,centroids_df = noncoding_centroids)
  plot_centroid(centroid_num = 3,centroids_df = noncoding_centroids)
  plot_centroid(centroid_num = 4,centroids_df = noncoding_centroids)
  plot_centroid(centroid_num = 5,centroids_df = noncoding_centroids)
  plot_centroid(centroid_num = 6,centroids_df = noncoding_centroids)
  
  # Define colours for background of boxes with correlation values
  #bg_cols <- scales::alpha(colorRampPalette(brewer.pal(9,"RdBu"))(21), 0.7)
  bg_cols <- scales::alpha(colorRampPalette(brewer.pal(9,"Blues"))(11), 0.7)
  bg_cols <- c(rep(bg_cols[1],10), bg_cols)
  # Plot coding centroids
  for(i in 1:nrow(coding_centroids[[1]])){
    # Plot coding gene centroid in the top row of the grid
    plot_centroid(centroid_num = i, centroids_df = coding_centroids)
    
    # Plot coding - non-coding gene centroid correlation value
    for(n in 1:nrow(noncoding_centroids[[1]])){
      # Determine the Pearson correlation between the coding and non-coding centroids
      cor_value <-round(cor(unlist(coding_centroids[[1]][i,]), unlist(noncoding_centroids[[1]][n,]), method = "pearson"),2)
      # Determine the colour of the box
      box_col <-  bg_cols[which(cor_value < seq(-0.9,1,0.1))[1]]
      
      plot(0,axes=TRUE, xlab="", ylab="", xaxt="n", yaxt="n", col="white")
      rect(par("usr")[1],par("usr")[3],par("usr")[2],par("usr")[4],col = box_col)
      legend(x="center", legend=paste0(cor_value,"     "), bty="n", cex=2.5)  
    }
  }
  par(mfcol=c(1,1))
}

# Function to plot color bar for Pearson's correlation
color.bar <- function(lut, min, max=-min, nticks=11, ticks=seq(min, max, len=nticks), title='') {
  scale = (length(lut)-1)/(max-min)
  plot(c(0,10), c(min,max), type='n', bty='n', xaxt='n', xlab='', yaxt='n', ylab='', main=title)
  axis(4, ticks, las=1, labels=FALSE)
  for (i in 1:(length(lut)-1)) {
    y = (i-1)/scale + min
    rect(0,y,10,y+1/scale, col=lut[i], border=NA)
  }
}

# Plot figure 4c, saving out as svg - later to be assembled in multi-panel pdf
pdf("figures/Fig_4/Fig4c.pdf", width=12, height=8)
fig4c()
dev.off()

# Get colours for bar
bg_cols <- scales::alpha(colorRampPalette(brewer.pal(9,"Blues"))(11), 0.7)
bg_cols <- c(rep(bg_cols[1],10), bg_cols)

# Save out the colour bar
pdf("figures/Fig_4/Pearson_correlation_key.pdf", width=1.75, height=5)
color.bar(bg_cols, -1)
dev.off()

print("Finished plotting Fig. 4")
