#include <iostream>
#include <cstring>
#include <math.h>

#include "CommonTypes.H"
#include "Error.H"
#include "Variable.H"
#include "Potential.H"
#include "Evidence.H"
#include "EvidenceManager.H"
#include "SlimFactor.H"
#include "PotentialManager.H"

PotentialManager::PotentialManager()
{
	ludecomp=NULL;
	perm=NULL;
}

PotentialManager::~PotentialManager()
{
	if(ludecomp!=NULL)
	{
		gsl_matrix_free(ludecomp);
	}
	if(perm!=NULL)
	{
		gsl_permutation_free(perm);
	}
	for(map<int,Potential*>::iterator pIter=potBuffer.begin();pIter!=potBuffer.end();pIter++)
	{
		delete pIter->second;
	}
	potBuffer.clear();
}

int 
PotentialManager::setEvidenceManager(EvidenceManager* aPtr)
{
	evMgr=aPtr;
	return 0;
}

int 
PotentialManager::setRestrictedNeighborSet(map<int,Variable*>& rSet)
{
	for(map<int,Variable*>::iterator vIter=rSet.begin();vIter!=rSet.end();vIter++)
	{
		restrictedNeighborSet[vIter->first]=vIter->second;
	}
	return 0;
}

int
PotentialManager::setOutputDir(const char* aDirName)
{
	strcpy(outputDir,aDirName);
	return 0;
}

int
PotentialManager::initPooled()
{
	//INTINTMAP& trainEvidSet=evMgr->getTrainingSet();
	INTINTMAP& testEvidSet=evMgr->getTestSet();
	//estimateAllMeanCov(false,globalMean,globalCovar,testEvidSet,NULL,NULL);
	estimateAllMeanCov(false,globalMean,globalCovar,testEvidSet);
	ludecomp=gsl_matrix_alloc(MAXFACTORSIZE_ALLOC,MAXFACTORSIZE_ALLOC);
	perm=gsl_permutation_alloc(MAXFACTORSIZE_ALLOC);
	return 0;
}

int
PotentialManager::init()
{
	char mFName[1024];
	char sdFName[1024];
	sprintf(mFName,"%s/gauss_mean.txt",outputDir);
	sprintf(sdFName,"%s/gauss_std.txt",outputDir);
	ifstream inFile(mFName);
	if(inFile.good())
	{
		readAllMeanCov(mFName,sdFName);
	}
	else
	{
		INTINTMAP& trainEvidSet=evMgr->getTrainingSet();
		estimateAllMeanCov(false,globalMean,globalCovar,trainEvidSet,mFName,sdFName);
	}
	ludecomp=gsl_matrix_alloc(MAXFACTORSIZE_ALLOC,MAXFACTORSIZE_ALLOC);
	perm=gsl_permutation_alloc(MAXFACTORSIZE_ALLOC);
	return 0;
}


int
PotentialManager::init(int f)
{
	char mFName[1024];
	char sdFName[1024];
	sprintf(mFName,"%s/gauss_mean_%d.txt",outputDir,f);
	sprintf(sdFName,"%s/gauss_std_%d.txt",outputDir,f);
	ifstream inFile(mFName);
	if(inFile.good())
	{
		readAllMeanCov(mFName,sdFName);
	}
	else
	{
		//INTINTMAP& trainEvidSet=evMgr->getTrainingSet();
		INTINTMAP& trainEvidSet=evMgr->getTestSet();
		//estimateAllMeanCov(false,globalMean,globalCovar,trainEvidSet,mFName,sdFName);
		estimateAllMeanCov(false,globalMean,globalCovar,trainEvidSet);
	}
	ludecomp=gsl_matrix_alloc(MAXFACTORSIZE_ALLOC,MAXFACTORSIZE_ALLOC);
	perm=gsl_permutation_alloc(MAXFACTORSIZE_ALLOC);
	return 0;
}


int
PotentialManager::initValidationSet(int validSetSize)
{
	char mFName[1024];
	char sdFName[1024];
	sprintf(mFName,"%s/gauss_mean_v%d.txt",outputDir,validSetSize);
	sprintf(sdFName,"%s/gauss_std_v%d.txt",outputDir,validSetSize);
	ifstream inFile(mFName);
	if(inFile.good())
	{
		readAllMeanCov(mFName,sdFName);
	}
	else
	{
		INTINTMAP& validationSet=evMgr->getValidationSet();
		estimateAllMeanCov(false,globalMean,globalCovar,validationSet,mFName,sdFName);
	}
	ludecomp=gsl_matrix_alloc(MAXFACTORSIZE_ALLOC,MAXFACTORSIZE_ALLOC);
	perm=gsl_permutation_alloc(MAXFACTORSIZE_ALLOC);
	return 0;
}


int
PotentialManager::initValidationSet(int validSetSize,int dId)
{
	char mFName[1024];
	char sdFName[1024];
	sprintf(mFName,"%s/gauss_mean_v%d.txt",outputDir,validSetSize);
	sprintf(sdFName,"%s/gauss_std_v%d.txt",outputDir,validSetSize);
	INTINTMAP& validationSet=evMgr->getValidationSet();
	estimateAllMeanCov(false,globalMean,globalCovar,validationSet,mFName,sdFName,dId);
	ludecomp=gsl_matrix_alloc(MAXFACTORSIZE_ALLOC,MAXFACTORSIZE_ALLOC);
	perm=gsl_permutation_alloc(MAXFACTORSIZE_ALLOC);
	return 0;
}

int
PotentialManager::reset()
{
	globalMean.clear();
	for(map<int,INTDBLMAP*>::iterator cIter=globalCovar.begin();cIter!=globalCovar.end();cIter++)
	{
		cIter->second->clear();
		delete cIter->second;
	}
	globalCovar.clear();
	if(ludecomp!=NULL)
	{
		gsl_matrix_free(ludecomp);
	}
	if(perm!=NULL)
	{
		gsl_permutation_free(perm);
	}
	for(map<int,Potential*>::iterator pIter=potBuffer.begin();pIter!=potBuffer.end();pIter++)
	{
		delete pIter->second;
	}
	potBuffer.clear();
	return 0;
}

int
PotentialManager::initRandom()
{
	INTINTMAP& trainEvidSet=evMgr->getTestSet();
	estimateAllMeanCov(true,globalMean_Rand,globalCovar_Rand,trainEvidSet,NULL,NULL);
	return 0;
}

int
PotentialManager::estimateAllMeanCov(bool random, INTDBLMAP& gMean, map<int,INTDBLMAP*>& gCovar,INTINTMAP& trainEvidSet, const char* mFName, const char* sdFName,int leaveOutData)
{
	ofstream mFile;
	ofstream sdFile;

	if(!random)
	{
		if((mFName!=NULL) && (sdFName!=NULL))
		{
			mFile.open(mFName);
			sdFile.open(sdFName);
		}
	}

	int evidCnt=trainEvidSet.size();
	if(leaveOutData!=-1)
	{
		evidCnt=evidCnt-1;
	}
	//First get the mean and then the variance
	int dId=0;
	for(INTINTMAP_ITER eIter=trainEvidSet.begin();eIter!=trainEvidSet.end();eIter++)
	{
		if(dId==leaveOutData)
		{	
			dId++;
			continue;
		}
		EMAP* evidMap=NULL;
		if(random)
		{
			evidMap=evMgr->getRandomEvidenceAt(eIter->first);
		}
		else
		{
			evidMap=evMgr->getEvidenceAt(eIter->first);
		}
		for(EMAP_ITER vIter=evidMap->begin();vIter!=evidMap->end(); vIter++)
		{
			int vId=vIter->first;
			Evidence* evid=vIter->second;
			double val=evid->getEvidVal();
			if(gMean.find(vId)==gMean.end())
			{
				gMean[vId]=val;
			}
			else
			{
				gMean[vId]=gMean[vId]+val;
			}
		}
		dId++;	
	}
	//Now estimate the mean
	for(INTDBLMAP_ITER idIter=gMean.begin();idIter!=gMean.end();idIter++)
	{
		if(idIter->first==176)
		{
			//cout <<"Stop here: Variable " << idIter->first << " mean " << idIter->second << endl;
		}
		idIter->second=idIter->second/(double) evidCnt;
		if(!random)
		{
			if(mFile.good())
			{
				mFile<<idIter->first<<"\t" << idIter->second<< endl;
			}
		}
	}
	int covPair=0;
	//Now the variance
	for(INTINTMAP_ITER eIter=trainEvidSet.begin();eIter!=trainEvidSet.end();eIter++)
	{
		EMAP* evidMap=NULL;
		if(random)
		{
			evidMap=evMgr->getRandomEvidenceAt(eIter->first);
		}
		else
		{
			evidMap=evMgr->getEvidenceAt(eIter->first);
		}
		for(EMAP_ITER vIter=evidMap->begin();vIter!=evidMap->end(); vIter++)
		{
			int vId=vIter->first;
			Evidence* evid=vIter->second;
			double vval=evid->getEvidVal();
			double vmean=gMean[vId];
			INTDBLMAP* vcov=NULL;
			if(gCovar.find(vId)==gCovar.end())
			{
				vcov=new INTDBLMAP;
				gCovar[vId]=vcov;
			}
			else
			{
				vcov=gCovar[vId];
			}
			for(EMAP_ITER uIter=vIter;uIter!=evidMap->end();uIter++)
			{
				int uId=uIter->first;
				//Don't compute covariance of vId uId pairs that both are not in the restrictedNeighborSet, when
				//the restrictedNeighborSet is empty
			/*	if((!random) && (vId!=uId) && (restrictedNeighborSet.size()>0))
				{
					if((restrictedNeighborSet.find(vId)==restrictedNeighborSet.end()) && (restrictedNeighborSet.find(uId)==restrictedNeighborSet.end()))
					{
						continue;
					}
				}*/
				Evidence* evid1=uIter->second;
				double uval=evid1->getEvidVal();
				double umean=gMean[uId];
				double diffprod=(vval-vmean)*(uval-umean);
				INTDBLMAP* ucov=NULL;
				if(gCovar.find(uId)==gCovar.end())
				{
					ucov=new INTDBLMAP;
					gCovar[uId]=ucov;
				}
				else
				{
					ucov=gCovar[uId];
				}
				if(vcov->find(uId)==vcov->end())
				{
					covPair++;
					(*vcov)[uId]=diffprod;
				}
				else
				{
					(*vcov)[uId]=(*vcov)[uId]+diffprod;
				}
				if(uId!=vId)
				{
					if(ucov->find(vId)==ucov->end())
					{
						(*ucov)[vId]=diffprod;
					}
					else
					{
						(*ucov)[vId]=(*ucov)[vId]+diffprod;
					}
				}
			}
		}

	}
	cout <<"Total covariance pairs estimated " << covPair << endl;
	//Now estimate the variance
	for(map<int,INTDBLMAP*>::iterator idIter=gCovar.begin();idIter!=gCovar.end();idIter++)
	{
		INTDBLMAP* var=idIter->second;
		for(INTDBLMAP_ITER vIter=var->begin();vIter!=var->end();vIter++)
		{
			if(vIter->first==idIter->first)
			{
				//vIter->second=2*vIter->second/((double)(gCovar.size()-1));
				//vIter->second=2*vIter->second/((double)(evidCnt-1));
				//vIter->second=(0.001+vIter->second)/((double)(evidCnt-1));
				vIter->second=(vIter->second)/((double)(evidCnt-1));
				double variance=vIter->second;
				if(idIter->first==176)
				{
				//	cout <<"Stop here: Variable " << idIter->first << " variance " << idIter->second << endl;
				}
			}
			else
			{
				vIter->second=vIter->second/((double)(evidCnt-1));
				//vIter->second=vIter->second/((double)(gCovar.size()-1));
				//vIter->second=0;
			}
			if(!random)
			{
				if(sdFile.good())
				{
					sdFile<<idIter->first<<"\t" << vIter->first <<"\t" << vIter->second << endl;
				}
			}
		}
	}
	if(!random)
	{
		if(mFile.good())
		{
			mFile.close();
		}
		if(sdFile.good())
		{
			sdFile.close();
		}
	}	
	return 0;
}

int
PotentialManager::estimateAllMeanCov(bool random, INTDBLMAP& gMean, map<int,INTDBLMAP*>& gCovar,INTINTMAP& trainEvidSet)
{
	int evidCnt=trainEvidSet.size();
	//First get the mean and then the variance
	int dId=0;
	for(INTINTMAP_ITER eIter=trainEvidSet.begin();eIter!=trainEvidSet.end();eIter++)
	{
		EMAP* evidMap=NULL;
		if(random)
		{
			evidMap=evMgr->getRandomEvidenceAt(eIter->first);
		}
		else
		{
			evidMap=evMgr->getEvidenceAt(eIter->first);
		}
		for(EMAP_ITER vIter=evidMap->begin();vIter!=evidMap->end(); vIter++)
		{
			int vId=vIter->first;
			Evidence* evid=vIter->second;
			double val=evid->getEvidVal();
			if(gMean.find(vId)==gMean.end())
			{
				gMean[vId]=val;
			}
			else
			{
				gMean[vId]=gMean[vId]+val;
			}
		}
		dId++;	
	}
	//Now estimate the mean
	for(INTDBLMAP_ITER idIter=gMean.begin();idIter!=gMean.end();idIter++)
	{
		idIter->second=idIter->second/(double) evidCnt;
		INTDBLMAP* vcov=new INTDBLMAP;
		gCovar[idIter->first]=vcov;
	}
	return 0;
}


int
PotentialManager::estimateCovariance(bool random,INTDBLMAP* vcov, int vId, int uId)
{
	if((uId==4 && vId==71) || (uId==71 && uId==4))
	{
		//cout << "Stop here " << endl;
	}
	INTINTMAP& trainEvidSet=evMgr->getTestSet();
	int evidCnt=trainEvidSet.size();
	INTDBLMAP* ucov=globalCovar[uId];
	for(INTINTMAP_ITER eIter=trainEvidSet.begin();eIter!=trainEvidSet.end();eIter++)
	{
		EMAP* evidMap=NULL;
		if(random)
		{
			evidMap=evMgr->getRandomEvidenceAt(eIter->first);
		}
		else
		{
			evidMap=evMgr->getEvidenceAt(eIter->first);
		}
		Evidence* evid=(*evidMap)[vId];
		double vval=evid->getEvidVal();
		double vmean=globalMean[vId];
		Evidence* evid1=(*evidMap)[uId];
		double uval=evid1->getEvidVal();
		double umean=globalMean[uId];
		double diffprod=(vval-vmean)*(uval-umean);
		if(vcov->find(uId)==vcov->end())
		{
			(*vcov)[uId]=diffprod;
		}
		else
		{
			(*vcov)[uId]=(*vcov)[uId]+diffprod;
		}
		if(uId!=vId)
		{
			if(ucov->find(vId)==ucov->end())
			{
				(*ucov)[vId]=diffprod;
			}
			else
			{
				(*ucov)[vId]=(*ucov)[vId]+diffprod;
			}
		}
	}
	//cout <<"Total covariance pairs estimated " << covPair << endl;
	//Now estimate the variance
	if(uId==vId)
	{
		double ssd=(*vcov)[uId];
		(*vcov)[uId]=(0.001+ssd)/((double)(evidCnt-1));
	}
	else
	{
		double ssduv=(*ucov)[vId];
		(*ucov)[vId]=ssduv/((double)(evidCnt-1));
		(*vcov)[uId]=ssduv/((double)(evidCnt-1));
	}
	return 0;
}

int
PotentialManager::readAllMeanCov(const char* mFName, const char* sdFName)
{

	ifstream mFile(mFName);
	ifstream sdFile(sdFName);
	char buffer[1024];
	while(mFile.good())
	{
		mFile.getline(buffer,1023);
		if(strlen(buffer)<=0)
		{
			continue;
		}
		char* tok=strtok(buffer,"\t");
		int tokCnt=0;
		int vId;
		double mean=0;
		while(tok!=NULL)
		{	
			if(tokCnt==0)
			{
				vId=atoi(tok);	
			}
			else if(tokCnt==1)
			{
				mean=atof(tok);
			}
			tok=strtok(NULL,"\t");
			tokCnt++;
		}
		globalMean[vId]=mean;
	}
	mFile.close();
	int lineNo=0;
	while(sdFile.good())
	{
		sdFile.getline(buffer,1023);
		if(strlen(buffer)<=0)
		{
			continue;
		}
		char* tok=strtok(buffer,"\t");
		int tokCnt=0;
		int vId=0;
		int uId=0;
		double covariance=0;
		while(tok!=NULL)
		{
			if(tokCnt==0)
			{
				uId=atoi(tok);
			}
			else if(tokCnt==1)
			{
				vId=atoi(tok);
			}
			else if(tokCnt==2)
			{
				covariance=atof(tok);
			}
			tok=strtok(NULL,"\t");
			tokCnt++;
		}
		INTDBLMAP* ucov=NULL;
		if(globalCovar.find(uId)==globalCovar.end())
		{
			ucov=new INTDBLMAP;
			globalCovar[uId]=ucov;
		}
		else
		{
			ucov=globalCovar[uId];
		}
		(*ucov)[vId]=covariance;
		if(uId==0 && vId==0)
		{
			cout << "Found uId=0 vId=0 covar="<< covariance << " at lineno " << lineNo  << endl;
		}
		lineNo++;
	}
	sdFile.close();
	return 0;
}


Error::ErrorCode
PotentialManager::populatePotentialsSlimFactors(map<int,SlimFactor*>& factorSet,VSET& varSet)
{
	//The set of flags to keep status of the potentials that have been calculated
	map<int,bool> doneFlag;
	for(map<int,SlimFactor*>::iterator fIter=factorSet.begin();fIter!=factorSet.end();fIter++)
	{
		doneFlag[fIter->first]=false;
	}
	int popFId=0;
	for(map<int,SlimFactor*>::reverse_iterator rIter=factorSet.rbegin();rIter!=factorSet.rend();rIter++)
	{
		//If we have computed the potential for this flag move one
		if(doneFlag[rIter->first])
		{
			popFId++;
			continue;
		}
		SlimFactor* sFactor=rIter->second;
		if(sFactor->fId==176)
		{
			cout <<"Stop here " << endl;
		}
		//Otherwise create the potential
		Potential* aPotFunc=new Potential;
		for(int j=0;j<sFactor->vCnt;j++)
		{
			Variable* aVar=varSet[sFactor->vIds[j]];
			if(j==sFactor->vCnt-1)
			{
				aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
			}
			else
			{
				aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
			}
		}
		aPotFunc->potZeroInit();
		populatePotential(aPotFunc,false);
		aPotFunc->calculateJointEntropy();
		sFactor->jointEntropy=aPotFunc->getJointEntropy();
		if(sFactor->jointEntropy<0)
		{
		//	sFactor->jointEntropy=0;
		//	cout <<"Negative entropy for " << sFactor->fId << endl;
		}
		doneFlag[rIter->first]=true;
		delete aPotFunc;
		if(popFId%100000==0)
		{
			cout <<"Done with " << factorSet.size()-popFId << " factors " << endl;
		}
		popFId++;
	}
	return Error::SUCCESS;
}


int 
PotentialManager::estimateMarginalEntropies(map<int,SlimFactor*>& slimFactors,VSET& varSet,bool random)
{	
	for(map<int,SlimFactor*>::iterator aIter=slimFactors.begin();aIter!=slimFactors.end();aIter++)
	{
		SlimFactor* sFactor=aIter->second;
		if(sFactor->vCnt>1)
		{
			break;
		}
		Potential* aPotFunc=new Potential;
		Variable* aVar=varSet[sFactor->vIds[0]];
		aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
		aPotFunc->potZeroInit();
		populatePotential(aPotFunc,random);
		aPotFunc->calculateJointEntropy();
		sFactor->jointEntropy=aPotFunc->getJointEntropy();
		delete aPotFunc;
	}
	return 0;
}

//Estimate the random information for all factors of a particular size. Assume that
//randInfo is already allocated
Error::ErrorCode
PotentialManager::estimateRandomInfo(map<int,SlimFactor*>& factorSet, 
		VSET& varSet, vector<double>& randInfo, int fSize)
{
	int rInd=0;
	for(map<int,SlimFactor*>::iterator rIter=factorSet.begin();rIter!=factorSet.end();rIter++)
	{
		SlimFactor* sFactor=rIter->second;
		if(sFactor->vCnt<fSize)
		{
			continue;
		}
		else if(sFactor->vCnt>fSize)
		{
			break;
		}
		//Otherwise create a potential
		Potential* aPotFunc=new Potential;
		for(int j=0;j<sFactor->vCnt;j++)
		{
			Variable* aVar=varSet[sFactor->vIds[j]];
			if(j==sFactor->vCnt-1)
			{
				aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
			}
			else
			{
				aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
			}
		}
		aPotFunc->potZeroInit();
		populatePotential(aPotFunc,true);
		aPotFunc->calculateJointEntropy();
		double rInfo=(-1)*aPotFunc->getJointEntropy();
		for(int j=0;j<sFactor->vCnt;j++)
		{
			SlimFactor* subFactor=factorSet[sFactor->vIds[j]];
			rInfo=rInfo+subFactor->jointEntropy;
		}
		randInfo.push_back(rInfo);
		rInd++;
		delete aPotFunc;
	}
	return Error::SUCCESS;

}


int 
PotentialManager::populateFactor(map<int,SlimFactor*>& factorSet,VSET& varSet,SlimFactor* sFactor,bool random)
{
	Potential* aPotFunc=new Potential;
	//string fullConfStr;
	char confStr[CONSTR_LEN];
	for(int j=0;j<sFactor->vCnt;j++)
	{
		Variable* aVar=varSet[sFactor->vIds[j]];
		if(j==sFactor->vCnt-1)
		{
			
			aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
		}
		else
		{
			aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
		}
		//sprintf(confStr,"-%d",sFactor->vIds[j]);
		//fullConfStr.append(confStr);
	}
	aPotFunc->potZeroInit();
	populatePotential(aPotFunc,random);
	aPotFunc->calculateJointEntropy();
	//jointEntropies[fullConfStr]=aPotFunc->getJointEntropy();
	double rInfo=(-1)*aPotFunc->getJointEntropy();
	for(int j=0;j<sFactor->vCnt;j++)
	{
		SlimFactor* subFactor=factorSet[sFactor->vIds[j]];
		rInfo=rInfo+subFactor->jointEntropy;
	}
	sFactor->mutualInfo=rInfo;
	
	delete aPotFunc;
	return 0;
}


int 
PotentialManager::populateFactor_Buffer(map<int,SlimFactor*>& factorSet,VSET& varSet,SlimFactor* sFactor,bool random)
{
	Potential* aPotFunc=NULL;
	if(potBuffer.find(sFactor->vCnt)==potBuffer.end())
	{
		aPotFunc=new Potential;
		aPotFunc->initMatrix(sFactor->vCnt);
		potBuffer[sFactor->vCnt]=aPotFunc;
	}
	else
	{
		aPotFunc=potBuffer[sFactor->vCnt];
		aPotFunc->resetVarSet();
	}
	//string fullConfStr;
	char confStr[CONSTR_LEN];
	for(int j=0;j<sFactor->vCnt;j++)
	{
		Variable* aVar=varSet[sFactor->vIds[j]];
		if(j==sFactor->vCnt-1)
		{
			
			aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
		}
		else
		{
			aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
		}
		//sprintf(confStr,"-%d",sFactor->vIds[j]);
		//fullConfStr.append(confStr);
	}
	aPotFunc->potZeroInit_MeanOnly();
	populatePotential(aPotFunc,random);
	aPotFunc->calculateJointEntropy();
	//jointEntropies[fullConfStr]=aPotFunc->getJointEntropy();
	double rInfo=(-1)*aPotFunc->getJointEntropy();
	for(int j=0;j<sFactor->vCnt;j++)
	{
		SlimFactor* subFactor=factorSet[sFactor->vIds[j]];
		rInfo=rInfo+subFactor->jointEntropy;
	}
	sFactor->mutualInfo=rInfo;
	
	//delete aPotFunc;
	return 0;
}

int
PotentialManager::populatePotential(Potential* aPot, bool random)
{
	VSET& potVars=aPot->getAssocVariables();
	for(VSET_ITER vIter=potVars.begin();vIter!=potVars.end(); vIter++)
	{
		double mean=0;
		double cov=0;
		INTDBLMAP* covar=NULL;
		if(random)
		{
			if(globalMean_Rand.find(vIter->first)==globalMean_Rand.end())
			{
				cout <<"No var with id " << vIter->first << endl;
				exit(0);
			}
			mean=globalMean_Rand[vIter->first];
			covar=globalCovar_Rand[vIter->first];
		}
		else
		{
			if(globalMean.find(vIter->first)==globalMean.end())
			{
				cout <<"No var with id " << vIter->first << endl;
				exit(0);
			}
			mean=globalMean[vIter->first];
			covar=globalCovar[vIter->first];
		}
		aPot->updateMean(vIter->first,mean);
		for(VSET_ITER uIter=vIter;uIter!=potVars.end();uIter++)
		{
			if(covar->find(uIter->first)==covar->end())
			{
				estimateCovariance(random,covar,vIter->first,uIter->first);
				//cout <<"No var " << uIter->first << " in covariance of " << vIter->first << endl;
				//exit(0);
			}
			double cval=(*covar)[uIter->first];
			aPot->updateCovariance(vIter->first,uIter->first,cval);
			aPot->updateCovariance(uIter->first,vIter->first,cval);
		}
	}

	//aPot->makeValidJPD();
	aPot->makeValidJPD(ludecomp,perm);
	return 0;
}

double
PotentialManager::getPseudoLikelihood(SlimFactor* sFactor,VSET& varSet, bool train)
{
	Potential* aPotFunc=new Potential;
	Variable* aVar=varSet[sFactor->fId];
	aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
	for(INTINTMAP_ITER aIter=sFactor->mergedMB.begin();aIter!=sFactor->mergedMB.end();aIter++)
	{
		Variable* aVar=varSet[aIter->first];
		aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
	}
	aPotFunc->potZeroInit();
	populatePotential(aPotFunc,false);
	//This function creates a submatrix of the covariance matrix and inverts it
	aPotFunc->initMBCovMean();
	INTINTMAP* dataSet=NULL;
	if(train)
	{
		dataSet=&(evMgr->getTrainingSet());
	}
	else
	{
		dataSet=&(evMgr->getTestSet());
	}
	INTDBLMAP subData;
	double pll=0;
	int thresholded=0;
	for(INTINTMAP_ITER dIter=dataSet->begin();dIter!=dataSet->end();dIter++)
	{
		EMAP* evidMap=NULL;
		evidMap=evMgr->getEvidenceAt(dIter->first);
		Evidence* evid=(*evidMap)[sFactor->fId];
		double val=evid->getEvidVal();
		subData[sFactor->fId]=val;
		for(INTINTMAP_ITER vIter=sFactor->mergedMB.begin();vIter!=sFactor->mergedMB.end(); vIter++)
		{
			int vId=vIter->first;
			Evidence* evid=(*evidMap)[vIter->first];
			double val=evid->getEvidVal();
			subData[vId]=val;
		}
		double cll=aPotFunc->getCondPotValueFor(subData);

		if(cll<1e-50)
		{
			cll=1e-50;
			thresholded++;
		}
		pll=pll+log(cll);
	}
	subData.clear();
	if(thresholded>0)
	{
	//	cout <<"Thresholded " << thresholded << " datapoints to 1e-50" << endl;
	}
	delete aPotFunc;
	return pll;
}


double
PotentialManager::getPseudoLikelihood(SlimFactor* sFactor,VSET& varSet, bool train,int& status)
{
	status=0;
	Potential* aPotFunc=new Potential;
	Variable* aVar=varSet[sFactor->fId];
	aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
	for(INTINTMAP_ITER aIter=sFactor->mergedMB.begin();aIter!=sFactor->mergedMB.end();aIter++)
	{
		Variable* aVar=varSet[aIter->first];
		aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
	}
	aPotFunc->potZeroInit();
	populatePotential(aPotFunc,false);
	//This function creates a submatrix of the covariance matrix and inverts it
	aPotFunc->initMBCovMean();
	if(aPotFunc->getCondVariance()<0)
	{
		status=-1;
		delete aPotFunc;
		return 0;
	}
	INTINTMAP* dataSet=NULL;
	if(train)
	{
		dataSet=&(evMgr->getTrainingSet());
	}
	else
	{
		dataSet=&(evMgr->getTestSet());
	}
	INTDBLMAP subData;
	double pll=0;
	int thresholded=0;
	for(INTINTMAP_ITER dIter=dataSet->begin();dIter!=dataSet->end();dIter++)
	{
		EMAP* evidMap=NULL;
		evidMap=evMgr->getEvidenceAt(dIter->first);
		Evidence* evid=(*evidMap)[sFactor->fId];
		double val=evid->getEvidVal();
		subData[sFactor->fId]=val;
		for(INTINTMAP_ITER vIter=sFactor->mergedMB.begin();vIter!=sFactor->mergedMB.end(); vIter++)
		{
			int vId=vIter->first;
			Evidence* evid=(*evidMap)[vIter->first];
			double val=evid->getEvidVal();
			subData[vId]=val;
		}
		double cll=aPotFunc->getCondPotValueFor(subData);

		if(cll<1e-50)
		{
			cll=1e-50;
			thresholded++;
		}
		pll=pll+log(cll);
	}
	subData.clear();
	if(thresholded>0)
	{
	//	cout <<"Thresholded " << thresholded << " datapoints to 1e-50" << endl;
	}
	delete aPotFunc;
	return pll;
}


double
PotentialManager::getPseudoLikelihood(SlimFactor* sFactor,VSET& varSet, bool train,int& status, Potential* aPotFunc)
{
	status=0;
	Variable* aVar=varSet[sFactor->fId];
	aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
	for(INTINTMAP_ITER aIter=sFactor->mergedMB.begin();aIter!=sFactor->mergedMB.end();aIter++)
	{
		Variable* aVar=varSet[aIter->first];
		aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
	}
	aPotFunc->potZeroInit();
	populatePotential(aPotFunc,false);
	//This function creates a submatrix of the covariance matrix and inverts it
	aPotFunc->initMBCovMean();
	if(aPotFunc->getCondVariance()<0)
	{
		status=-1;
		return 0;
	}
	INTINTMAP* dataSet=NULL;
	if(train)
	{
		dataSet=&(evMgr->getTrainingSet());
	}
	else
	{
		dataSet=&(evMgr->getTestSet());
	}
	INTDBLMAP subData;
	double pll=0;
	int thresholded=0;
	for(INTINTMAP_ITER dIter=dataSet->begin();dIter!=dataSet->end();dIter++)
	{
		EMAP* evidMap=NULL;
		evidMap=evMgr->getEvidenceAt(dIter->first);
		Evidence* evid=(*evidMap)[sFactor->fId];
		double val=evid->getEvidVal();
		subData[sFactor->fId]=val;
		for(INTINTMAP_ITER vIter=sFactor->mergedMB.begin();vIter!=sFactor->mergedMB.end(); vIter++)
		{
			int vId=vIter->first;
			Evidence* evid=(*evidMap)[vIter->first];
			double val=evid->getEvidVal();
			subData[vId]=val;
		}
		double cll=aPotFunc->getCondPotValueFor(subData);

		if(cll<1e-50)
		{
			cll=1e-50;
			thresholded++;
		}
		pll=pll+log(cll);
	}
	subData.clear();
	return pll;
}


double 
PotentialManager::getGaussianLikelihood(map<int,SlimFactor*>& factorSet,VSET& varSet, bool train)
{
	Potential* aPotFunc=new Potential;
	for(map<int,SlimFactor*>::iterator fIter=factorSet.begin();fIter!=factorSet.end();fIter++)
	{
		Variable* aVar=varSet[fIter->first];
		if(fIter==factorSet.begin())
		{
			aPotFunc->setAssocVariable(aVar,Potential::FACTOR);
		}
		else
		{
			aPotFunc->setAssocVariable(aVar,Potential::MARKOV_BNKT);
		}
	}
	aPotFunc->potZeroInit();
	//Use the graph structure to update the particular elements of the covariance and mean of this potential
	for(map<int,SlimFactor*>::iterator fIter=factorSet.begin();fIter!=factorSet.end();fIter++)
	{
		SlimFactor* sFactor=fIter->second;

		double mean=0;
		double cov=0;
		INTDBLMAP* covar=NULL;
		if(globalMean.find(fIter->first)==globalMean.end())
		{
			cout <<"No var with id " << fIter->first << endl;
			exit(0);
		}
		mean=globalMean[fIter->first];
		covar=globalCovar[fIter->first];
		aPotFunc->updateMean(fIter->first,mean);
		double vval=(*covar)[fIter->first];
		aPotFunc->updateCovariance(fIter->first,fIter->first,vval);
		for(INTINTMAP_ITER mIter=sFactor->mergedMB.begin();mIter!=sFactor->mergedMB.end();mIter++)
		{
			if(covar->find(mIter->first)==covar->end())
			{
				cout <<"No var " << mIter->first << " in covariance of " << fIter->first << endl;
				exit(0);
			}
			double cval=(*covar)[mIter->first];
			aPotFunc->updateCovariance(fIter->first,mIter->first,cval);
			aPotFunc->updateCovariance(mIter->first,fIter->first,cval);
		}
	}

	aPotFunc->makeValidJPD();
	INTDBLMAP datapt;
	double gll=0;
	int thresholded=0;
	INTINTMAP* dataSet;
	if(train)
	{
		dataSet=(&evMgr->getTrainingSet());
	}
	else
	{
		dataSet=(&evMgr->getTestSet());
	}
	for(INTINTMAP_ITER dIter=dataSet->begin();dIter!=dataSet->end();dIter++)
	{
		EMAP* evidMap=NULL;
		evidMap=evMgr->getEvidenceAt(dIter->first);
		for(map<int,SlimFactor*>::iterator fIter=factorSet.begin();fIter!=factorSet.end();fIter++)
		{
			int vId=fIter->first;
			Evidence* evid=(*evidMap)[vId];
			double val=evid->getEvidVal();
			datapt[vId]=val;
		}
		double jll=aPotFunc->getJointPotValueFor(datapt);

		if(jll<1e-50)
		{
			jll=1e-50;
			thresholded++;
		}
		gll=gll+log(jll);
	}
	delete aPotFunc;
	return gll;
}

double
PotentialManager::getLikelihood(SlimFactor* sFactor,VSET& varSet)
{
	cout <<"Not implemented" << endl;
	return 0;
}


double
PotentialManager::getLikelihood(SlimFactor* sFactor,VSET& varSet,map<int,int>& visitedVertices )
{
	double dll=0;
	cout <<"Not implemented" << endl;
	return dll;
}


int
PotentialManager::estimateConditionalPotential(SlimFactor* sFactor,VSET& varSet,Potential** pot, STRDBLMAP& counts)
{
	cout <<"Not implemented" << endl;
	return 0;
}

int
PotentialManager::populatePotential(Potential* pot,STRDBLMAP& counts)
{
	cout <<"Not implemented" << endl;
	return 0;
}

int 
PotentialManager::estimateCanonicalPotential(SlimFactor* sFactor, VSET& variableSet,INTINTMAP& defInst,INTINTMAP& factorSubsets,map<int,SlimFactor*>& canonicalFactorSet)
{
	cout <<"Not implemented" << endl;
	return 0;
}


int 
PotentialManager::estimateCanonicalPotential_Abbeel(SlimFactor* sFactor, VSET& variableSet,INTINTMAP& defInst,INTINTMAP& factorSubsets,map<int,SlimFactor*>& canonicalFactorSet)
{
	cout <<"Not implemented" << endl;
	return 0;
}



int 
PotentialManager::estimateCanonicalPotential_Approximate(SlimFactor* sFactor, VSET& variableSet,INTINTMAP& defInst,INTINTMAP& factorSubsets,map<int,SlimFactor*>& canonicalFactorSet)
{
	cout <<"Not implemented" << endl;
	return 0;
}

int
PotentialManager::resetPotFuncs()
{
	for(map<int,Potential*>::iterator pIter=potFuncs.begin();pIter!=potFuncs.end();pIter++)
	{
		delete pIter->second;
	}
	potFuncs.clear();
	return 0;
}



int 
PotentialManager::estimateCanonicalPotential_Joint(SlimFactor* sFactor, VSET& variableSet,INTINTMAP& defInst,INTINTMAP& factorSubsets,map<int,SlimFactor*>& canonicalFactorSet)
{
	cout <<"Not implemented" << endl;
	return 0;
}

Potential*
PotentialManager::getPotential(int fId)
{
	if(potFuncs.find(fId)==potFuncs.end())
	{
		return NULL;
	}
	return potFuncs[fId];
}


double 
PotentialManager::getConditionalEntropy(int vId,INTINTMAP& fVars,VSET& varSet)
{
	double condEntropy=0;
	//string fullConfStr;
	//string partConfStr;
	char confStr[CONSTR_LEN];
	/*int varCnt=0;
	for(INTINTMAP_ITER aIter=fVars.begin();aIter!=fVars.end();aIter++)
	{
		sprintf(confStr,"-%d",aIter->first);
		fullConfStr.append(confStr);
		if(aIter->first!=vId)
		{
			partConfStr.append(confStr);
			varCnt++;
		}
	}*/
	double fullJointEntropy=0;
	/*if(jointEntropies.find(fullConfStr)!=jointEntropies.end())
	{
		fullJointEntropy=jointEntropies[fullConfStr];
	}
	else
	{*/
		Potential* potFunc=new Potential;
		for(INTINTMAP_ITER aIter=fVars.begin();aIter!=fVars.end();aIter++)
		{
			if(aIter==fVars.begin())
			{
				potFunc->setAssocVariable(varSet[aIter->first],Potential::FACTOR);
			}
			else
			{
				potFunc->setAssocVariable(varSet[aIter->first],Potential::MARKOV_BNKT);
			}
		}
		potFunc->potZeroInit();
		populatePotential(potFunc,false);
		potFunc->calculateJointEntropy();
		fullJointEntropy=potFunc->getJointEntropy();
		//jointEntropies[fullConfStr]=fullJointEntropy;
		delete potFunc;
	//}
	if(fVars.size()==1)
	{
		/*if(jointEntropies.size()>=20000)
		{
			jointEntropies.clear();
		}*/
		return fullJointEntropy;
	}
	double partJointEntropy=0;
	/*if(jointEntropies.find(partConfStr)!=jointEntropies.end())
	{
		partJointEntropy=jointEntropies[partConfStr];
	}
	else
	{
		Potential* potFunc=new Potential;*/
		potFunc=new Potential;
		bool setFactorVar=false;
		for(INTINTMAP_ITER aIter=fVars.begin();aIter!=fVars.end();aIter++)
		{
			if(aIter->first==vId)
			{
				continue;
			}
			if(!setFactorVar)
			{
				setFactorVar=true;
				potFunc->setAssocVariable(varSet[aIter->first],Potential::FACTOR);
			}
			else
			{
				potFunc->setAssocVariable(varSet[aIter->first],Potential::MARKOV_BNKT);
			}
		}
		STRDBLMAP counts;
		potFunc->potZeroInit();
		populatePotential(potFunc,false);
		potFunc->calculateJointEntropy();
		partJointEntropy=potFunc->getJointEntropy();
		//jointEntropies[partConfStr]=partJointEntropy;
		delete potFunc;
	//}
	condEntropy=fullJointEntropy-partJointEntropy;
	//fullConfStr.clear();
	//partConfStr.clear();
	/*if(jointEntropies.size()>=20000)
	{
		jointEntropies.clear();
	}*/
	return condEntropy;
}


double 
PotentialManager::getSampleLikelihood(map<int,SlimFactor*>& factorSet, VSET& varSet, INTINTMAP* sample)
{
	double sampleLL=0;
	cout <<"Not implemented " <<endl;
	return sampleLL;
}


int 
PotentialManager::getVariableSample(INTINTMAP& jointConf,VSET& varSet,int vId,SlimFactor* sFactor, gsl_rng* r)
{
	Potential* pot=NULL;
	if(potFuncs.find(vId)==potFuncs.end())
	{
		pot=new Potential;
		STRDBLMAP counts;
		estimateConditionalPotential(sFactor,varSet,&pot,counts);
		potFuncs[vId]=pot;
	}
	else
	{
		pot=potFuncs[vId];
	}
	//int sample=pot->generateSample(jointConf,vId,r);
	int sample=-1;
	return sample;
}

int
PotentialManager::clearJointEntropies()
{
	jointEntropies.clear();
	return 0;
}

double
PotentialManager::estimateCanonicalValue(INTINTMAP& reqInst,INTINTMAP& defInst,INTINTMAP& allSubsets,map<int,SlimFactor*>& canFactors,Potential* condPot)
{
	double pVal=0;
	cout <<"Not implemented " << endl;
	return 0;
}


double
PotentialManager::estimateCanonicalValue_Joint(INTINTMAP& reqInst,INTINTMAP& defInst,INTINTMAP& allSubsets,map<int,SlimFactor*>& canFactors,Potential* jointPot)
{
	cout <<"Not implemented " << endl;
	return 0;
}
