#!/bin/bash
# 2.make_gene_and_TE_counts.sh

### Prepare TE annotations
# Get the RepeatMaker output file from Supplementary Data 1 and store it ./data
# Use rmsk2bed script from BEDOPS (version 2.4.27) to convert .fa.out to BED file
rmsk2bed < ./data/danRer11.nonalt.fa.out > ./data/danRer11.nonalt.fa.out.bed
# Convert to GTF format
time python ./scripts/RM_bed2GTF.py -i ./data/danRer11.nonalt.fa.out.bed -o ./data/danRer11.nonalt.fa.out.gtf
# Adapt GTF to TEtranscript requested format
# Use ./data/danRer11_classes_wredundant.txt file to solve some TE superfamily/family name redundancy.
# Subset TE classes: DNA, LINE, LTR, RC, SINE
sed 's/?//g' ./data/danRer11_classes_wredundant.txt \
  | awk '{if ($2=="DNA" || $2=="LINE" || $2=="LTR" || $2=="RC" || $2=="SINE") print }' \
  > ./data/danRer11_classes_wredundant.subClass.txt
python ./scripts/adapt_GTF_TEtranscript.py \
  -i ./data/danRer11.nonalt.fa.out.gtf \
  -c ./data/danRer11_classes_wredundant.subClass.txt \
  -o ./data/danRer11.TEtrans_uID.gtf


### Get gene annotations
# Download GRCz11.98 gene annotations from Ensembl
wget -O ./data/Danio_rerio.GRCz11.98.chr.gtf.gz http://ftp.ensembl.org/pub/release-98/gtf/danio_rerio/Danio_rerio.GRCz11.98.chr.gtf.gz
# Download ERCC
cd data
wget https://assets.thermofisher.com/TFS-Assets/LSG/manuals/ERCC92.zip
unzip ERCC92.zip
rm ERCC92.zip ERCC92.fa
cd ..
# Add ERCC gtf to GRCz11.98 annotations
cat <(zcat ./data/Danio_rerio.GRCz11.98.chr.gtf.gz) \
    ./data/ERCC92.gtf \
    | gzip > ./data/Danio_rerio.GRCz11.98.chr.ERCC.gtf.gz
# BAM files chromosomes have the "chr" on it while the annotations don't.
# Add "chr" to each entry of the GTF but the ERCCs. Take special care of MT chromosome.
zcat ./data/Danio_rerio.GRCz11.98.chr.ERCC.gtf.gz \
  | sed -e 's/^/chr/' \
  | sed -e 's/^chr#/#/;s/^chrERCC/ERCC/;s/^chrMT/chrM/' \
  | gzip > ./data/Danio_rerio.GRCz11.98.chr.ERCC.addChr.gtf.gz


### Run TEtranscripts
# Run samples in pairs
BAM_DIR="./data/BAMs"
GE_GTF="./data/Danio_rerio.GRCz11.98.chr.ERCC.addChr.gtf.gz"
TE_GTF="./data/danRer11.TEtrans_uID.gtf"
TETRANS_OUT_DIR="./data/TEtranscripts"
mkdir $TETRANS_OUT_DIR
# 1_cell        vs      2_cell
S1="1_cell"; S2="2_cell"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# 128_cell      vs      1k_cell
S1="128_cell"; S2="1k_cell"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# dome          vs      50pc_epiboly
S1="dome"; S2="50pc_epiboly"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# shield        vs      75pc_epiboly
S1="shield"; S2="75pc_epiboly"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# 1_4_somites   vs      14_19_somites
S1="1_4_somites"; S2="14_19_somites"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# 20_25_somites vs      prim_5
S1="20_25_somites"; S2="prim_5"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# prim_15       vs      prim_25
S1="prim_15"; S2="prim_25"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# long_pec      vs      protruding_mouth
S1="long_pec"; S2="protruding_mouth"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# day_4         vs      day_5
S1="day_4"; S2="day_5"
TEtranscripts --format BAM --stranded reverse --mode multi --sortByPos --control $BAM_DIR/${S1}_rep_1.bam $BAM_DIR/${S1}_rep_2.bam $BAM_DIR/${S1}_rep_3.bam $BAM_DIR/${S1}_rep_4.bam $BAM_DIR/${S1}_rep_5.bam --treatment $BAM_DIR/${S2}_rep_1.bam $BAM_DIR/${S2}_rep_2.bam $BAM_DIR/${S2}_rep_3.bam $BAM_DIR/${S2}_rep_4.bam $BAM_DIR/${S2}_rep_5.bam --GTF $GE_GTF --TE $TE_GTF --project ${TETRANS_OUT_DIR}/${S1}_vs_${S2}
# Join the *.cntTable files
DIR_PATH="\.\/data\/BAMs\/"
paste <(cut -f1 ./data/TEtranscripts/1_cell_vs_2_cell.cntTable | sed 's/gene\/TE/Feature/') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/1_cell_vs_2_cell.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/128_cell_vs_1k_cell.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/dome_vs_50pc_epiboly.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/shield_vs_75pc_epiboly.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/1_4_somites_vs_14_19_somites.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/20_25_somites_vs_prim_5.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/prim_15_vs_prim_25.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/long_pec_vs_protruding_mouth.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
      <(sed "s/${DIR_PATH}//g" ./data/TEtranscripts/day_4_vs_day_5.cntTable | sed -e 's/.bam.[TC]//g' | sed 's/gene\/TE/Feature/'| awk '{print $7"\t"$8"\t"$9"\t"$10"\t"$11"\t"$2"\t"$3"\t"$4"\t"$5"\t"$6}') \
  | sed 's/"//g' \
  > ./data/TEtranscripts/join.cntTable

# Load join.cntTable into R as DESeq object and save it in RDS format
Rscript ./scripts/load_TEtrans_cntTable.R


### Run Telescope
# Since telescope does not consider stranded RNA-seq data, alignment files were split between 
# forward and reverse mapping strand. Here we used the script to split BAM files by strand from 
# Joseph Chao-Chung Kuo in his blog: https://josephcckuo.wordpress.com/2016/11/18/splitting-reads-in-a-bam-file-by-strands/

## Split BAMs
OUT_BAM_SPLIT="./data/BAMs/strand_split"
mkdir $OUT_BAM_SPLIT
BAM_FILES=( $BAM_DIR/*.bam )
THREADS=6
for BAM in ${BAM_FILES[*]}; do
    bash ./scripts/BAM_split_strand_PE.sh $BAM $OUT_BAM_SPLIT $THREADS
done

## Run telescope on split BAMs
# 20_04_23_White2017_Telescope_stranded/run_White2017_Telescope_stranded.sh
OUT_TELES="./data/telescope"
mkdir $OUT_TELES
mkdir ${OUT_TELES}/fwd_fwd
mkdir ${OUT_TELES}/rev_rev
#
BAM_FILES_FWD=( $OUT_BAM_SPLIT/*fwd.bam )
BAM_FILES_REV=( $OUT_BAM_SPLIT/*rev.bam )
TE_GTF="./data/danRer11.TEtrans_uID.gtf"
TE_GTF_FWD="./data/danRer11.TEtrans_uID.fwd.gtf"
TE_GTF_REV="./data/danRer11.TEtrans_uID.rev.gtf"
awk '$7=="+"' $TE_GTF > $TE_GTF_FWD
awk '$7=="-"' $TE_GTF > $TE_GTF_REV
# Run for FWD - FWD
for BAM in ${BAM_FILES_FWD[*]}; do
    SAMPLE_NAME=$(basename ${BAM} | sed -e 's/_fwd.bam//')
    bash ./scripts/run_telescope.sh ${OUT_TELES}/fwd_fwd $SAMPLE_NAME $BAM $TE_GTF_FWD
done
# Run for REV - REV
for BAM in ${BAM_FILES_REV[*]}; do
    SAMPLE_NAME=$(basename ${BAM} | sed -e 's/_rev.bam//')
    bash ./scripts/run_telescope.sh ${OUT_TELES}/rev_rev $SAMPLE_NAME $BAM $TE_GTF_REV
done

## Sort all files on the same way (telescope output results don't contain the same number of rows and order)
OUT_TELES="./data/telescope"
TE_GTF="./data/danRer11.TEtrans_uID.gtf"
TELES_REPORT_FILES=( $OUT_TELES/*/*telescope_report.tsv )
for TELES_REPORT in ${TELES_REPORT_FILES[*]}; do
    SAMPLE=$(basename $TELES_REPORT | sed -e 's/^telescope_//;s/-telescope_report.tsv$//')
    TELES_REPORT_OUT="$(dirname $TELES_REPORT)/Edited_telescope_report/${SAMPLE}_telesReport.tab"
    TELES_REPORT_OUT_DIR=$(dirname $TELES_REPORT_OUT)
    if [[ ! -d "$TELES_REPORT_OUT_DIR" ]]; then
        echo "mkdir -p $TELES_REPORT_OUT_DIR"
        mkdir -p $TELES_REPORT_OUT_DIR
    fi
    python ./scripts/sort_telescope_report.py -g $TE_GTF -r $TELES_REPORT -o $TELES_REPORT_OUT
done

## Join results into a single table
HOME_DIR=$(pwd)
#reads_FWD_tes_FWD
cd ./data/telescope/fwd_fwd/Edited_telescope_report
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 2 -o join_telesRep.final_count.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 3 -o join_telesRep.final_conf.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 4 -o join_telesRep.final_prop.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 5 -o join_telesRep.init_aligned.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 6 -o join_telesRep.unique_count.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 7 -o join_telesRep.init_best.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 8 -o join_telesRep.init_best_random.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 9 -o join_telesRep.init_best_avg.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 10 -o join_telesRep.init_prop.tab
cd $HOME_DIR
#reads_REV_tes_REV
cd ./data/telescope/rev_rev/Edited_telescope_report
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 2 -o join_telesRep.final_count.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 3 -o join_telesRep.final_conf.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 4 -o join_telesRep.final_prop.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 5 -o join_telesRep.init_aligned.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 6 -o join_telesRep.unique_count.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 7 -o join_telesRep.init_best.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 8 -o join_telesRep.init_best_random.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 9 -o join_telesRep.init_best_avg.tab
python $HOME_DIR/scripts/join_telesRep_column.py -f "1_cell_rep_1_telesRep.tab,1_cell_rep_2_telesRep.tab,1_cell_rep_3_telesRep.tab,1_cell_rep_4_telesRep.tab,1_cell_rep_5_telesRep.tab,2_cell_rep_1_telesRep.tab,2_cell_rep_2_telesRep.tab,2_cell_rep_3_telesRep.tab,2_cell_rep_4_telesRep.tab,2_cell_rep_5_telesRep.tab,128_cell_rep_1_telesRep.tab,128_cell_rep_2_telesRep.tab,128_cell_rep_3_telesRep.tab,128_cell_rep_4_telesRep.tab,128_cell_rep_5_telesRep.tab,1k_cell_rep_1_telesRep.tab,1k_cell_rep_2_telesRep.tab,1k_cell_rep_3_telesRep.tab,1k_cell_rep_4_telesRep.tab,1k_cell_rep_5_telesRep.tab,dome_rep_1_telesRep.tab,dome_rep_2_telesRep.tab,dome_rep_3_telesRep.tab,dome_rep_4_telesRep.tab,dome_rep_5_telesRep.tab,50pc_epiboly_rep_1_telesRep.tab,50pc_epiboly_rep_2_telesRep.tab,50pc_epiboly_rep_3_telesRep.tab,50pc_epiboly_rep_4_telesRep.tab,50pc_epiboly_rep_5_telesRep.tab,shield_rep_1_telesRep.tab,shield_rep_2_telesRep.tab,shield_rep_3_telesRep.tab,shield_rep_4_telesRep.tab,shield_rep_5_telesRep.tab,75pc_epiboly_rep_1_telesRep.tab,75pc_epiboly_rep_2_telesRep.tab,75pc_epiboly_rep_3_telesRep.tab,75pc_epiboly_rep_4_telesRep.tab,75pc_epiboly_rep_5_telesRep.tab,1_4_somites_rep_1_telesRep.tab,1_4_somites_rep_2_telesRep.tab,1_4_somites_rep_3_telesRep.tab,1_4_somites_rep_4_telesRep.tab,1_4_somites_rep_5_telesRep.tab,14_19_somites_rep_1_telesRep.tab,14_19_somites_rep_2_telesRep.tab,14_19_somites_rep_3_telesRep.tab,14_19_somites_rep_4_telesRep.tab,14_19_somites_rep_5_telesRep.tab,20_25_somites_rep_1_telesRep.tab,20_25_somites_rep_2_telesRep.tab,20_25_somites_rep_3_telesRep.tab,20_25_somites_rep_4_telesRep.tab,20_25_somites_rep_5_telesRep.tab,prim_5_rep_1_telesRep.tab,prim_5_rep_2_telesRep.tab,prim_5_rep_3_telesRep.tab,prim_5_rep_4_telesRep.tab,prim_5_rep_5_telesRep.tab,prim_15_rep_1_telesRep.tab,prim_15_rep_2_telesRep.tab,prim_15_rep_3_telesRep.tab,prim_15_rep_4_telesRep.tab,prim_15_rep_5_telesRep.tab,prim_25_rep_1_telesRep.tab,prim_25_rep_2_telesRep.tab,prim_25_rep_3_telesRep.tab,prim_25_rep_4_telesRep.tab,prim_25_rep_5_telesRep.tab,long_pec_rep_1_telesRep.tab,long_pec_rep_2_telesRep.tab,long_pec_rep_3_telesRep.tab,long_pec_rep_4_telesRep.tab,long_pec_rep_5_telesRep.tab,protruding_mouth_rep_1_telesRep.tab,protruding_mouth_rep_2_telesRep.tab,protruding_mouth_rep_3_telesRep.tab,protruding_mouth_rep_4_telesRep.tab,protruding_mouth_rep_5_telesRep.tab,day_4_rep_1_telesRep.tab,day_4_rep_2_telesRep.tab,day_4_rep_3_telesRep.tab,day_4_rep_4_telesRep.tab,day_4_rep_5_telesRep.tab,day_5_rep_1_telesRep.tab,day_5_rep_2_telesRep.tab,day_5_rep_3_telesRep.tab,day_5_rep_4_telesRep.tab,day_5_rep_5_telesRep.tab" -n "1_cell_rep_1,1_cell_rep_2,1_cell_rep_3,1_cell_rep_4,1_cell_rep_5,2_cell_rep_1,2_cell_rep_2,2_cell_rep_3,2_cell_rep_4,2_cell_rep_5,128_cell_rep_1,128_cell_rep_2,128_cell_rep_3,128_cell_rep_4,128_cell_rep_5,1k_cell_rep_1,1k_cell_rep_2,1k_cell_rep_3,1k_cell_rep_4,1k_cell_rep_5,dome_rep_1,dome_rep_2,dome_rep_3,dome_rep_4,dome_rep_5,50pc_epiboly_rep_1,50pc_epiboly_rep_2,50pc_epiboly_rep_3,50pc_epiboly_rep_4,50pc_epiboly_rep_5,shield_rep_1,shield_rep_2,shield_rep_3,shield_rep_4,shield_rep_5,75pc_epiboly_rep_1,75pc_epiboly_rep_2,75pc_epiboly_rep_3,75pc_epiboly_rep_4,75pc_epiboly_rep_5,1_4_somites_rep_1,1_4_somites_rep_2,1_4_somites_rep_3,1_4_somites_rep_4,1_4_somites_rep_5,14_19_somites_rep_1,14_19_somites_rep_2,14_19_somites_rep_3,14_19_somites_rep_4,14_19_somites_rep_5,20_25_somites_rep_1,20_25_somites_rep_2,20_25_somites_rep_3,20_25_somites_rep_4,20_25_somites_rep_5,prim_5_rep_1,prim_5_rep_2,prim_5_rep_3,prim_5_rep_4,prim_5_rep_5,prim_15_rep_1,prim_15_rep_2,prim_15_rep_3,prim_15_rep_4,prim_15_rep_5,prim_25_rep_1,prim_25_rep_2,prim_25_rep_3,prim_25_rep_4,prim_25_rep_5,long_pec_rep_1,long_pec_rep_2,long_pec_rep_3,long_pec_rep_4,long_pec_rep_5,protruding_mouth_rep_1,protruding_mouth_rep_2,protruding_mouth_rep_3,protruding_mouth_rep_4,protruding_mouth_rep_5,day_4_rep_1,day_4_rep_2,day_4_rep_3,day_4_rep_4,day_4_rep_5,day_5_rep_1,day_5_rep_2,day_5_rep_3,day_5_rep_4,day_5_rep_5" -c 10 -o join_telesRep.init_prop.tab
cd $HOME_DIR

###########