#!/bin/bash

#https://hicexplorer.readthedocs.io/en/latest/content/mES-HiC_analysis.html#build-visualize-and-correct-hi-c-matrix

#create the directories
cd ~
mkdir -p BG3_HiC_KD
cd BG3_HiC_KD/
mkdir -p HiCExplorer
cd HiCExplorer

mkdir -p mapped_files
mkdir -p fastq
mkdir -p hiCmatrix
mkdir -p plots
mkdir -p TADs
mkdir -p compartments


################################################################################
#  download files check that the correct identifiers have been used.
################################################################################
cd fastq

fastq-dump --split-files --outdir fastq SRX5014527 #Kc167 rep1
fastq-dump --split-files --outdir fastq SRX5014528 #Kc167 rep2
fastq-dump --split-files --outdir fastq SRX5014529 #BG3 rep1
fastq-dump --split-files --outdir fastq SRX5014530 #BG3 rep2

fastq-dump --split-files --outdir fastq SRX7918778 #BG3 BEAF-32 KD rep1
fastq-dump --split-files --outdir fastq SRX7918779 #BG3 BEAF-32 KD rep2
fastq-dump --split-files --outdir fastq SRX7918780 #BG3 Cp190 Chro KD rep1
fastq-dump --split-files --outdir fastq SRX7918781 #BG3 Cp190 Chro KD rep2
fastq-dump --split-files --outdir fastq SRX7918782 #BG3 BEAF-32 Dref KD rep1
fastq-dump --split-files --outdir fastq SRX7918783 #BG3 BEAF-32 Dref KD rep2

cd ..
################################################################################
#  Subsample the Kc167 files
################################################################################
python subsample.py 0.82 fastq/SRX5014527_1.fastq fastq/SRX5014527_2.fastq fastq/SRX5014527_sub_1.fastq fastq/SRX5014527_sub_2.fastq
python subsample.py 0.65 fastq/SRX5014528_1.fastq fastq/SRX5014528_2.fastq fastq/SRX5014528_sub_1.fastq fastq/SRX5014528_sub_2.fastq


################################################################################
#  align the reads
################################################################################

#download the genome and prepare it
#bwa index -a bwtsw ~/genomes/DM6/dm6.fa

#Kc167 rep1
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa  fastq/SRX5014527_1.fastq | samtools view -Shb - >  mapped_files/SRX5014527_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa  fastq/SRX5014527_2.fastq | samtools view -Shb - >  mapped_files/SRX5014527_2.sam

#Kc167 rep2
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa  fastq/SRX5014528_1.fastq | samtools view -Shb - >  mapped_files/SRX5014528_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa  fastq/SRX5014528_2.fastq | samtools view -Shb - >  mapped_files/SRX5014528_2.sam

#BG3 rep1
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX5014529_1.fastq | samtools view -Shb - > mapped_files/SRX5014529_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX5014529_2.fastq | samtools view -Shb - > mapped_files/SRX5014529_2.sam


#BG3 rep2
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX5014530_1.fastq | samtools view -Shb - > mapped_files/SRX5014530_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX5014530_2.fastq | samtools view -Shb - > mapped_files/SRX5014530_2.sam


#BG3 BEAF-32 KD rep1
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918778_1.fastq | samtools view -Shb - > mapped_files/SRX7918778_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918778_2.fastq | samtools view -Shb - > mapped_files/SRX7918778_2.sam


#BG3 BEAF-32 KD rep2
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918779_1.fastq | samtools view -Shb - > mapped_files/SRX7918779_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918779_2.fastq | samtools view -Shb - > mapped_files/SRX7918779_2.sam


#BG3 Cp190 Chro KD rep1
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918780_1.fastq | samtools view -Shb - > mapped_files/SRX7918780_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918780_2.fastq | samtools view -Shb - > mapped_files/SRX7918780_2.sam


#BG3 Cp190 Chro KD rep2
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918781_1.fastq | samtools view -Shb - > mapped_files/SRX7918781_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918781_2.fastq | samtools view -Shb - > mapped_files/SRX7918781_2.sam

#BG3 BEAF-32 Dref KD rep1
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918782_1.fastq | samtools view -Shb - > mapped_files/SRX7918782_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918782_2.fastq | samtools view -Shb - > mapped_files/SRX7918782_2.sam


#BG3 BEAF-32 Dref KD rep2
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918783_1.fastq | samtools view -Shb - > mapped_files/SRX7918783_1.sam
bwa mem -t 20 -A1 -B4 -E50 -L0 ~/genomes/DM6/dm6.fa fastq/SRX7918783_2.fastq | samtools view -Shb - > mapped_files/SRX7918783_2.sam



################################################################################
# Build Matrices  Kc167 WT DPNII for TADs
################################################################################

#WT rep1 150bp
hicBuildMatrix --samFiles mapped_files/SRX5014527_1.sam mapped_files/SRX5014527_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam Kc167_WT_rep1_hic_dpnII.bam \
                -o hiCmatrix/Kc167_WT_rep1_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/Kc167_WT_rep1_hic_matrix_dpnII_hicQC

#WT rep2 150bp
hicBuildMatrix --samFiles mapped_files/SRX5014528_1.sam mapped_files/SRX5014528_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam Kc167_WT_rep2_hic_dpnII.bam \
                -o hiCmatrix/Kc167_WT_rep2_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/Kc167_WT_rep2_hic_matrix_dpnII_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/Kc167_WT_rep1_hic_matrix_dpnII.h5 hiCmatrix/Kc167_WT_rep2_hic_matrix_dpnII.h5 \
                -o hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII.h5 -o plots/Kc167_WT_merged_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/Kc167_WT_rep1_hic_matrix_dpnII.h5 -o plots/Kc167_WT_rep1_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/Kc167_WT_rep2_hic_matrix_dpnII.h5 -o plots/Kc167_WT_rep2_hic_matrix_dpnII_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII.h5 -o hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/Kc167_WT_rep1_hic_matrix_dpnII.h5 -o hiCmatrix/Kc167_WT_rep1_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/Kc167_WT_rep2_hic_matrix_dpnII.h5 -o hiCmatrix/Kc167_WT_rep2_hic_matrix_dpnII_corrected.h5


################################################################################
# Build Matrices  BG3 WT DPNII for TADs
################################################################################

#WT rep1 150bp
hicBuildMatrix --samFiles mapped_files/SRX5014529_1.sam mapped_files/SRX5014529_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_WT_rep1_hic_dpnII.bam \
                -o hiCmatrix/BG3_WT_rep1_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_WT_rep1_hic_matrix_dpnII_hicQC

#WT rep2 150bp
hicBuildMatrix --samFiles mapped_files/SRX5014530_1.sam mapped_files/SRX5014530_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_WT_rep2_hic_dpnII.bam \
                -o hiCmatrix/BG3_WT_rep2_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_WT_rep2_hic_matrix_dpnII_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_WT_rep1_hic_matrix_dpnII.h5 hiCmatrix/BG3_WT_rep2_hic_matrix_dpnII.h5 \
                -o hiCmatrix/BG3_WT_merged_hic_matrix_dpnII.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_WT_merged_hic_matrix_dpnII.h5 -o plots/BG3_WT_merged_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_WT_rep1_hic_matrix_dpnII.h5 -o plots/BG3_WT_rep1_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_WT_rep2_hic_matrix_dpnII.h5 -o plots/BG3_WT_rep2_hic_matrix_dpnII_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_WT_merged_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_WT_merged_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_WT_rep1_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_WT_rep1_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_WT_rep2_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_WT_rep2_hic_matrix_dpnII_corrected.h5


################################################################################
# Build Matrices BG3 100Kb for plots
################################################################################
#WT rep1 100Kb
hicBuildMatrix --samFiles mapped_files/SRX5014529_1.sam mapped_files/SRX5014529_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_WT_rep1_hic_100Kb.bam \
                -o hiCmatrix/BG3_WT_rep1_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_WT_rep1_hic_matrix_100Kb_hicQC

#WT rep2 100Kb
hicBuildMatrix --samFiles mapped_files/SRX5014530_1.sam mapped_files/SRX5014530_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_WT_rep2_hic_100Kb.bam \
                -o hiCmatrix/BG3_WT_rep2_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_WT_rep2_hic_matrix_100Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_WT_rep1_hic_matrix_100Kb.h5 hiCmatrix/BG3_WT_rep2_hic_matrix_100Kb.h5 \
                -o hiCmatrix/BG3_WT_merged_hic_matrix_100Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_WT_merged_hic_matrix_100Kb.h5 -o plots/BG3_WT_merged_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_WT_rep1_hic_matrix_100Kb.h5 -o plots/BG3_WT_rep1_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_WT_rep2_hic_matrix_100Kb.h5 -o plots/BG3_WT_rep2_hic_matrix_100Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_WT_merged_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_WT_merged_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_WT_rep1_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_WT_rep1_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_WT_rep2_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_WT_rep2_hic_matrix_100Kb_corrected.h5


################################################################################
# Build Matrices BG3 10Kb for compartments
################################################################################

#WT rep1 10Kb
hicBuildMatrix --samFiles mapped_files/SRX5014529_1.sam mapped_files/SRX5014529_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_WT_rep1_hic_10Kb.bam \
                -o hiCmatrix/BG3_WT_rep1_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_WT_rep1_hic_matrix_10Kb_hicQC

#WT rep2 10Kb
hicBuildMatrix --samFiles mapped_files/SRX5014530_1.sam mapped_files/SRX5014530_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_WT_rep2_hic_10Kb.bam \
                -o hiCmatrix/BG3_WT_rep2_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_WT_rep2_hic_matrix_10Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_WT_rep1_hic_matrix_10Kb.h5 hiCmatrix/BG3_WT_rep2_hic_matrix_10Kb.h5 \
                -o hiCmatrix/BG3_WT_merged_hic_matrix_10Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_WT_merged_hic_matrix_10Kb.h5 -o plots/BG3_WT_merged_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_WT_rep1_hic_matrix_10Kb.h5 -o plots/BG3_WT_rep1_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_WT_rep2_hic_matrix_10Kb.h5 -o plots/BG3_WT_rep2_hic_matrix_10Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_WT_merged_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_WT_merged_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_WT_rep1_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_WT_rep1_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_WT_rep2_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_WT_rep2_hic_matrix_10Kb_corrected.h5

################################################################################
# Build Matrices  BG3 BEAF_KD DPNII for TADs
################################################################################

#BEAF_KD rep1 150bp
hicBuildMatrix --samFiles mapped_files/SRX7918778_1.sam mapped_files/SRX7918778_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_BEAF_KD_rep1_hic_dpnII.bam \
                -o hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_dpnII_hicQC

#BEAF_KD rep2 150bp
hicBuildMatrix --samFiles mapped_files/SRX7918779_1.sam mapped_files/SRX7918779_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_BEAF_KD_rep2_hic_dpnII.bam \
                -o hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_dpnII_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_dpnII.h5 hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_dpnII.h5 \
                -o hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII.h5 -o plots/BG3_BEAF_KD_merged_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_dpnII.h5 -o plots/BG3_BEAF_KD_rep1_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_dpnII.h5 -o plots/BG3_BEAF_KD_rep2_hic_matrix_dpnII_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.0 5 -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_dpnII_corrected.h5

################################################################################
# Build Matrices BG3 BEAF_KD 100Kb for plots
################################################################################
#BEAF_KD rep1 100Kb
hicBuildMatrix --samFiles mapped_files/SRX7918778_1.sam mapped_files/SRX7918778_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_KD_rep1_hic_100Kb.bam \
                -o hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_100Kb_hicQC

#BEAF_KD rep2 100Kb
hicBuildMatrix --samFiles mapped_files/SRX7918779_1.sam mapped_files/SRX7918779_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_KD_rep2_hic_100Kb.bam \
                -o hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_100Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_100Kb.h5 hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_100Kb.h5 \
                -o hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_100Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_100Kb.h5 -o plots/BG3_BEAF_KD_merged_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_100Kb.h5 -o plots/BG3_BEAF_KD_rep1_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_100Kb.h5 -o plots/BG3_BEAF_KD_rep2_hic_matrix_100Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.2 5 -m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.9 5 -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.2 5 -m hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_100Kb_corrected.h5


################################################################################
# Build Matrices BG3 BEAF_KD 10Kb for compartments
################################################################################

#BEAF_KD rep1 10Kb
hicBuildMatrix --samFiles mapped_files/SRX7918778_1.sam mapped_files/SRX7918778_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_KD_rep1_hic_10Kb.bam \
                -o hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_10Kb_hicQC

#BEAF_KD rep2 10Kb
hicBuildMatrix --samFiles mapped_files/SRX7918779_1.sam mapped_files/SRX7918779_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_KD_rep2_hic_10Kb.bam \
                -o hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_10Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_10Kb.h5 hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_10Kb.h5 \
                -o hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_10Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_10Kb.h5 -o plots/BG3_BEAF_KD_merged_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_10Kb.h5 -o plots/BG3_BEAF_KD_rep1_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_10Kb.h5 -o plots/BG3_BEAF_KD_rep2_hic_matrix_10Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -0.8 5 -m hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_10Kb_corrected.h5



################################################################################
# Build Matrices  BG3 Cp190_Chrom_KD DPNII for TADs
################################################################################

#Cp190_Chrom_KD rep1 150bp
hicBuildMatrix --samFiles mapped_files/SRX7918780_1.sam mapped_files/SRX7918780_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_Cp190_Chrom_KD_rep1_hic_dpnII.bam \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_dpnII_hicQC

#Cp190_Chrom_KD rep2 150bp
hicBuildMatrix --samFiles mapped_files/SRX7918781_1.sam mapped_files/SRX7918781_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_Cp190_Chrom_KD_rep2_hic_dpnII.bam \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_dpnII_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_dpnII.h5 hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_dpnII.h5 \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII.h5 -o plots/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_dpnII.h5 -o plots/BG3_Cp190_Chrom_KD_rep1_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_dpnII.h5 -o plots/BG3_Cp190_Chrom_KD_rep2_hic_matrix_dpnII_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_dpnII_corrected.h5

################################################################################
# Build Matrices BG3 Cp190_Chrom_KD 100Kb for plots
################################################################################
#Cp190_Chrom_KD rep1 100Kb
hicBuildMatrix --samFiles mapped_files/SRX7918780_1.sam mapped_files/SRX7918780_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_Cp190_Chrom_KD_rep1_hic_100Kb.bam \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_100Kb_hicQC

#Cp190_Chrom_KD rep2 100Kb
hicBuildMatrix --samFiles mapped_files/SRX7918781_1.sam mapped_files/SRX7918781_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_Cp190_Chrom_KD_rep2_hic_100Kb.bam \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_100Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_100Kb.h5 hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_100Kb.h5 \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb.h5 -o plots/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_100Kb.h5 -o plots/BG3_Cp190_Chrom_KD_rep1_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_100Kb.h5 -o plots/BG3_Cp190_Chrom_KD_rep2_hic_matrix_100Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_100Kb_corrected.h5


################################################################################
# Build Matrices BG3 Cp190_Chrom_KD 10Kb for compartments
################################################################################

#Cp190_Chrom_KD rep1 10Kb
hicBuildMatrix --samFiles mapped_files/SRX7918780_1.sam mapped_files/SRX7918780_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_Cp190_Chrom_KD_rep1_hic_10Kb.bam \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb_hicQC

#Cp190_Chrom_KD rep2 10Kb
hicBuildMatrix --samFiles mapped_files/SRX7918781_1.sam mapped_files/SRX7918781_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_Cp190_Chrom_KD_rep2_hic_10Kb.bam \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb.h5 hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb.h5 \
                -o hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_10Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_10Kb.h5 -o plots/BG3_Cp190_Chrom_KD_merged_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb.h5 -o plots/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb.h5 -o plots/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb_corrected.h5


################################################################################
# Build Matrices  BG3 BEAF_DREF_KD DPNII for TADs
################################################################################

#BEAF_DREF_KD rep1 150bp
hicBuildMatrix --samFiles mapped_files/SRX7918782_1.sam mapped_files/SRX7918782_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_BEAF_DREF_KD_rep1_hic_dpnII.bam \
                -o hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_dpnII_hicQC

#BEAF_DREF_KD rep2 150bp
hicBuildMatrix --samFiles mapped_files/SRX7918783_1.sam mapped_files/SRX7918783_2.sam \
               --restrictionCutFile dm6_DPNII_positions.bed \
               --minDistance 150 \
               --threads 30 \
                --outBam BG3_BEAF_DREF_KD_rep2_hic_dpnII.bam \
                -o hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_dpnII.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_dpnII_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_dpnII.h5 hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_dpnII.h5 \
                -o hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII.h5 -o plots/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_dpnII.h5 -o plots/BG3_BEAF_DREF_KD_rep1_hic_matrix_dpnII_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_dpnII.h5 -o plots/BG3_BEAF_DREF_KD_rep2_hic_matrix_dpnII_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_dpnII_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.2 5 -m hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_dpnII.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_dpnII_corrected.h5

################################################################################
# Build Matrices BG3 BEAF_DREF_KD 100Kb for plots
################################################################################
#BEAF_DREF_KD rep1 100Kb
hicBuildMatrix --samFiles mapped_files/SRX7918782_1.sam mapped_files/SRX7918782_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_DREF_KD_rep1_hic_100Kb.bam \
                -o hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_100Kb_hicQC

#BEAF_DREF_KD rep2 100Kb
hicBuildMatrix --samFiles mapped_files/SRX7918783_1.sam mapped_files/SRX7918783_2.sam \
                --binSize 100000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_DREF_KD_rep2_hic_100Kb.bam \
                -o hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_100Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_100Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_100Kb.h5 hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_100Kb.h5 \
                -o hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb.h5 -o plots/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_100Kb.h5 -o plots/BG3_BEAF_DREF_KD_rep1_hic_matrix_100Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_100Kb.h5 -o plots/BG3_BEAF_DREF_KD_rep2_hic_matrix_100Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_100Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -2.4 5 -m hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_100Kb.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_100Kb_corrected.h5


################################################################################
# Build Matrices BG3 BEAF_DREF_KD 10Kb for compartments
################################################################################

#BEAF_DREF_KD rep1 10Kb
hicBuildMatrix --samFiles mapped_files/SRX7918782_1.sam mapped_files/SRX7918782_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_DREF_KD_rep1_hic_10Kb.bam \
                -o hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb_hicQC

#BEAF_DREF_KD rep2 10Kb
hicBuildMatrix --samFiles mapped_files/SRX7918783_1.sam mapped_files/SRX7918783_2.sam \
                --binSize 10000 \
                --restrictionSequence GATC \
                --threads 30 \
                --outBam BG3_BEAF_DREF_KD_rep2_hic_10Kb.bam \
                -o hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb.h5 \
                --QCfolder hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb_hicQC


#Merge (sum) matrices from replicates
hicSumMatrices -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb.h5 hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb.h5 \
                -o hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_10Kb.h5

hicCorrectMatrix diagnostic_plot \
-m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_10Kb.h5 -o plots/BG3_BEAF_DREF_KD_merged_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb.h5 -o plots/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb_diagnostic_plot.png

# generate diagnostic plots
mkdir -p plots
hicCorrectMatrix diagnostic_plot \
            -m hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb.h5 -o plots/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb_diagnostic_plot.png



#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb_corrected.h5

#Correction of Hi-C matrix
hicCorrectMatrix correct --filterThreshold -1.4 5 -m hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb.h5 -o hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb_corrected.h5



################################################################################
# Dist Vs Counts all 100Kb
################################################################################
hicPlotDistVsCounts -m \
hiCmatrix/BG3_WT_rep1_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_WT_rep2_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_WT_merged_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/Kc167_WT_rep1_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/Kc167_WT_rep2_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/Kc167_WT_merged_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_KD_rep1_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_KD_rep2_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_DREF_KD_rep1_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_DREF_KD_rep2_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_Cp190_Chrom_KD_rep1_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_Cp190_Chrom_KD_rep2_hic_matrix_10Kb_corrected.h5 \
hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_10Kb_corrected.h5 \
-o plots/counts_vs_dist_10Kb_all.png \
--outFileData plots/counts_vs_dist_10Kb_all.dat \
--labels 'BG3 rep1' 'BG3 rep2' 'BG3 merged' 'Kc167 rep1' 'Kc167 rep2' 'Kc167 merged' 'BEAF KD rep1' 'BEAF KD rep2' 'BEAF KD merged' 'BEAF DREF KD rep1' 'BEAF DREF KD rep2' 'BEAF DREF KD merged' 'Cp190 Chrom KD rep1' 'Cp190 Chrom KD rep2' 'Cp190 Chrom KD merged' \
--maxdepth 20000000 \
--plotsize 8 4.2



################################################################################
# export for plots matrix at 100Kb
################################################################################
hicCompareMatrices --matrices hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_100Kb_corrected.h5 \
hiCmatrix/BG3_WT_merged_hic_matrix_100Kb_corrected.h5 \
-o hiCmatrix/BG3_BEAF_KD_WT_merged_hic_matrix_100Kb_corrected.h5 --operation log2ratio

hicConvertFormat -m hiCmatrix/BG3_BEAF_KD_WT_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_BEAF_KD_WT_merged_hic_matrix_100Kb_corrected.mat

hicCompareMatrices --matrices hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb_corrected.h5 \
hiCmatrix/BG3_WT_merged_hic_matrix_100Kb_corrected.h5 \
-o hiCmatrix/BG3_BEAF_DREF_KD_WT_merged_hic_matrix_100Kb_corrected.h5 --operation log2ratio

hicConvertFormat -m hiCmatrix/BG3_BEAF_DREF_KD_WT_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_BEAF_DREF_KD_WT_merged_hic_matrix_100Kb_corrected.mat.gz

hicCompareMatrices --matrices hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb_corrected.h5 \
hiCmatrix/BG3_WT_merged_hic_matrix_100Kb_corrected.h5 \
-o hiCmatrix/BG3_Cp190_Chrom_KD_WT_merged_hic_matrix_100Kb_corrected.h5 --operation log2ratio

hicConvertFormat -m hiCmatrix/BG3_Cp190_Chrom_KD_WT_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_Cp190_Chrom_KD_WT_merged_hic_matrix_100Kb_corrected.mat.gz

hicCompareMatrices --matrices hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_KD_WT_merged_hic_matrix_100Kb_corrected.h5 \
-o hiCmatrix/BG3_Cp190_Chrom_KD_BEAF_KD_merged_hic_matrix_100Kb_corrected.h5 --operation log2ratio

hicConvertFormat -m hiCmatrix/BG3_Cp190_Chrom_KD_BEAF_KD_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_Cp190_Chrom_KD_BEAF_KD_merged_hic_matrix_100Kb_corrected.mat.gz


hicCompareMatrices --matrices hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb_corrected.h5 \
hiCmatrix/BG3_BEAF_KD_WT_merged_hic_matrix_100Kb_corrected.h5 \
-o hiCmatrix/BG3_BEAF_DREF_KD_BEAF_KD_merged_hic_matrix_100Kb_corrected.h5 --operation log2ratio

hicConvertFormat -m hiCmatrix/BG3_BEAF_DREF_KD_BEAF_KD_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_BEAF_DREF_KD_BEAF_KD_merged_hic_matrix_100Kb_corrected.mat.gz


hicConvertFormat -m hiCmatrix/BG3_WT_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_WT_merged_hic_matrix_100Kb_corrected.mat.gz

hicConvertFormat -m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_100Kb_corrected.mat.gz

hicConvertFormat -m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_100Kb_corrected.mat.gz

hicConvertFormat -m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_100Kb_corrected.mat.gz

hicConvertFormat -m hiCmatrix/Kc167_WT_merged_hic_matrix_100Kb_corrected.h5 --inputFormat h5 \
--outputFormat homer -o hiCmatrix/Kc167_WT_merged_hic_matrix_100Kb_corrected.mat.gz

################################################################################
# call TADs cells
################################################################################

hicFindTADs -m hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/Kc167_WT_merged_dpnII \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.04 \
--step 2000


hicFindTADs -m hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/Kc167_WT_merged_dpnII_strong \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.08 \
--step 2000


hicFindTADs -m hiCmatrix/BG3_WT_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_WT_merged_dpnII \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.04 \
--step 2000


hicFindTADs -m hiCmatrix/BG3_WT_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_WT_merged_dpnII_strong \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.08 \
--step 2000


hicFindTADs -m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_BEAF_DREF_KD_merged_dpnII \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.04 \
--step 2000


hicFindTADs -m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_BEAF_DREF_KD_merged_dpnII_strong \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.08 \
--step 2000


hicFindTADs -m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_BEAF_KD_merged_dpnII \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.04 \
--step 2000


hicFindTADs -m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_BEAF_KD_merged_dpnII_strong \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.08 \
--step 2000



hicFindTADs -m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_Cp190_Chrom_KD_merged_dpnII \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.04 \
--step 2000


hicFindTADs -m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outPrefix TADs/BG3_Cp190_Chrom_KD_merged_dpnII_strong \
--correctForMultipleTesting fdr \
--numberOfProcessors 30 \
--minBoundaryDistance 5000 --thresholdComparisons 0.01 --delta 0.08 \
--step 2000


################################################################################
# enriched contacts
################################################################################
hicFindEnrichedContacts --matrix hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII_corrected.h5 \
--outFileName hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII_corrected_enriched.h5 --method obs/exp

hicFindEnrichedContacts --matrix hiCmatrix/BG3_WT_merged_hic_matrix_dpnII_corrected.h5 \
--outFileName hiCmatrix/BG3_WT_merged_hic_matrix_dpnII_corrected_enriched.h5 --method obs/exp

hicFindEnrichedContacts --matrix hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outFileName hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII_corrected_enriched.h5 --method obs/exp

hicFindEnrichedContacts --matrix hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outFileName hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_corrected_enriched.h5 --method obs/exp

hicFindEnrichedContacts --matrix hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_corrected.h5 \
--outFileName hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_corrected_enriched.h5 --method obs/exp



hicConvertFormat -m hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII_corrected_enriched.h5 --inputFormat h5 \
--outputFormat ginteractions -o  hiCmatrix/Kc167_WT_merged_hic_matrix_dpnII_corrected_enriched_GInteractions

hicConvertFormat -m hiCmatrix/BG3_WT_merged_hic_matrix_dpnII_corrected_enriched.h5 --inputFormat h5 \
--outputFormat ginteractions -o  hiCmatrix/BG3_WT_merged_hic_matrix_dpnII_corrected_enriched_GInteractions

hicConvertFormat -m hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII_corrected_enriched.h5 --inputFormat h5 \
--outputFormat ginteractions -o  hiCmatrix/BG3_BEAF_KD_merged_hic_matrix_dpnII_corrected_enriched_GInteractions

hicConvertFormat -m hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_corrected_enriched.h5 --inputFormat h5 \
--outputFormat ginteractions -o  hiCmatrix/BG3_Cp190_Chrom_KD_merged_hic_matrix_dpnII_corrected_enriched_GInteractions

hicConvertFormat -m hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_corrected_enriched.h5 --inputFormat h5 \
--outputFormat ginteractions -o  hiCmatrix/BG3_BEAF_DREF_KD_merged_hic_matrix_dpnII_corrected_enriched_GInteractions

################################################################################
# call loops
################################################################################
cd ~/BG3_HiC_KD
mkdir -p juicer
cd juicer
mkdir -p genome

#custom genome dm6
cd ~/genome/
bwa index dm6.fa
python /usr/local/juicer/misc/generate_site_positions.py DpnII dm6 dm6.fa
awk 'BEGIN{OFS="\t"}{print $1, $NF}'  dm6_DpnII.txt > dm6.chrom.sizes
awk 'BEGIN{OFS="\t"}{print $1, $NF}'  dm6_DpnII.txt
gawk 'BEGIN{OFS="\t"}{print $1, $NF}'  dm6_DpnII.txt
less dm6.chrom.sizes


mkdir -p BG3_WT_HiC
cd BG3_WT_HiC
mkdir -p fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX5014529_1.fastq fastq/SRX5014529_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX5014529_2.fastq fastq/SRX5014529_R2.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX5014530_1.fastq fastq/SRX5014530_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX5014530_2.fastq fastq/SRX5014530_R2.fastq
cd ..

mkdir -p BG3_BEAF_KD_HiC
cd BG3_BEAF_KD_HiC
mkdir -p fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918778_1.fastq fastq/SRX7918778_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918778_2.fastq fastq/SRX7918778_R2.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918779_1.fastq fastq/SRX7918779_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918779_2.fastq fastq/SRX7918779_R2.fastq
cd ..


mkdir -p BG3_Cp190_Chrom_KD_HiC
cd BG3_Cp190_Chrom_KD_HiC
mkdir -p fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918780_1.fastq fastq/SRX7918780_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918780_2.fastq fastq/SRX7918780_R2.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918781_1.fastq fastq/SRX7918781_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918781_2.fastq fastq/SRX7918781_R2.fastq
cd ..

mkdir -p BG3_BEAF_DREF_KD_HiC
cd BG3_BEAF_DREF_KD_HiC
mkdir -p fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918782_1.fastq fastq/SRX7918782_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918782_2.fastq fastq/SRX7918782_R2.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918783_1.fastq fastq/SRX7918783_R1.fastq
cp ~/BG3_HiC_KD/HiCExplorer/fastq/SRX7918783_2.fastq fastq/SRX7918783_R2.fastq
cd ../../

#call preprocessing pipeline
~/juicer/CPU/juicer.sh -t 30 -z ~/BG3_HiC_KD/juicer/genome/dm6.fa -p ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes -y ~/BG3_HiC_KD/juicer/genome/dm6_DpnII.txt -d ~/BG3_HiC_KD/juicer/BG3_WT_HiC

~/juicer/CPU/juicer.sh -t 30 -z ~/BG3_HiC_KD/juicer/genome/dm6.fa -p ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes -y ~/BG3_HiC_KD/juicer/genome/dm6_DpnII.txt -d ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC

~/juicer/CPU/juicer.sh -t 30 -z ~/BG3_HiC_KD/juicer/genome/dm6.fa -p ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes -y ~/BG3_HiC_KD/juicer/genome/dm6_DpnII.txt -d ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC

~/juicer/CPU/juicer.sh -t 30 -z ~/BG3_HiC_KD/juicer/genome/dm6.fa -p ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes -y ~/BG3_HiC_KD/juicer/genome/dm6_DpnII.txt -d ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC


#generate the hic files
java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pre -q 30 -r 500,1000,2000,5000,10000,20000,25000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/merged_nodups.txt ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pre -q 30 -r 500,1000,2000,5000,10000,20000,25000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/merged_nodups.txt ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pre -q 30 -r 500,1000,2000,5000,10000,20000,25000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/merged_nodups.txt ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pre -q 30 -r 500,1000,2000,5000,10000,20000,25000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/merged_nodups.txt ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic ~/BG3_HiC_KD/juicer/genome/dm6.chrom.sizes


#call chromatin loops
java -Xms512m -Xmx2048m -jar ~/myJuicerDir/scripts/common/juicer_tools.jar hiccups -k KR -r 2000 -f 0.05 -p 5 -i 10 -t 0.02,1.5,1.75,2 -d 20000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic ~/BG3_HiC_KD/juicer/BG3_WT_HiC/BG3_WT_HiC_hiccups_loops_2kb_05fdr

java -Xms512m -Xmx2048m -jar ~/myJuicerDir/scripts/common/juicer_tools.jar hiccups -k KR -r 2000 -f 0.05 -p 5 -i 10 -t 0.02,1.5,1.75,2 -d 20000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/BG3_BEAF_KD_HiC_hiccups_loops_2kb_05fdr

java -Xms512m -Xmx2048m -jar ~/myJuicerDir/scripts/common/juicer_tools.jar hiccups -k KR -r 2000 -f 0.05 -p 5 -i 10 -t 0.02,1.5,1.75,2 -d 20000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/BG3_Cp190_Chrom_KD_HiC_hiccups_loops_2kb_05fdr

java -Xms512m -Xmx2048m -jar ~/myJuicerDir/scripts/common/juicer_tools.jar hiccups -k KR -r 2000 -f 0.05 -p 5 -i 10 -t 0.02,1.5,1.75,2 -d 20000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/BG3_BEAF_DREF_KD_HiC_hiccups_loops_2kb_05fdr


################################################################################
# Juicer aggregate loops
################################################################################

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_maintained.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_maintained_BG3_WT

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_lost.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_lost_BG3_WT

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_maintained.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_maintained_BG3_BEAF_KD

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_lost.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_lost_BG3_BEAF_KD

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_maintained.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_maintained_BG3_Cp190_Chrom_KD

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_lost.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_lost_BG3_Cp190_Chrom_KD

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_maintained.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_maintained_BG3_BEAF_DREF_KD

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools_1.19.02.jar apa -r 20000,10000,5000,2000,1000,500 -u \
~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC_hiccups_loops_2kb_05fdr/BG3_WT_HiC_hiccups_loops_2kb_05fdr_merged_loops_lost.bedpe \
~/BG3_HiC_KD/juicer/apa_plots_lost_BG3_BEAF_DREF_KD


################################################################################
# call compartments BG3 WT
################################################################################
cd ~/BG3_HiC_KD/juicer/BG3_WT_HiC
mkdir -p compartments

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_eigen_10Kb_2L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_eigen_10Kb_2R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_eigen_10Kb_3L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_eigen_10Kb_3R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_eigen_10Kb_4.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_eigen_10Kb_X.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
Y BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_eigen_10Kb_Y.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_pearsons_10Kb_2L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_pearsons_10Kb_2R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_pearsons_10Kb_3L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_pearsons_10Kb_3R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_pearsons_10Kb_4.txt



java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_WT_HiC/aligned/BG3_WT_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_WT_HiC/compartments/BG3_WT_HiC_pearsons_10Kb_X.txt


################################################################################
# call compartments BG3 BEAF KD
################################################################################
cd ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC
mkdir -p compartments

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_eigen_10Kb_2L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_eigen_10Kb_2R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_eigen_10Kb_3L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_eigen_10Kb_3R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_eigen_10Kb_4.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_eigen_10Kb_X.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
Y BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_eigen_10Kb_Y.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_pearsons_10Kb_2L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_pearsons_10Kb_2R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_pearsons_10Kb_3L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_pearsons_10Kb_3R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_pearsons_10Kb_4.txt



java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/aligned/BG3_BEAF_KD_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_KD_HiC/compartments/BG3_BEAF_KD_HiC_pearsons_10Kb_X.txt



################################################################################
# call compartments BG3 Cp190 Chrom KD
################################################################################
cd ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC
mkdir -p compartments

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_eigen_10Kb_2L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_eigen_10Kb_2R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_eigen_10Kb_3L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_eigen_10Kb_3R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_eigen_10Kb_4.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_eigen_10Kb_X.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
Y BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_eigen_10Kb_Y.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_pearsons_10Kb_2L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_pearsons_10Kb_2R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_pearsons_10Kb_3L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_pearsons_10Kb_3R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_pearsons_10Kb_4.txt



java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/aligned/BG3_Cp190_Chrom_KD_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_Cp190_Chrom_KD_HiC/compartments/BG3_Cp190_Chrom_KD_HiC_pearsons_10Kb_X.txt



################################################################################
# call compartments BG3 BEAF DREF KD
################################################################################
cd ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC
mkdir -p compartments

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_eigen_10Kb_2L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_eigen_10Kb_2R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_eigen_10Kb_3L.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_eigen_10Kb_3R.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_eigen_10Kb_4.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_eigen_10Kb_X.txt

java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar eigenvector \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
Y BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_eigen_10Kb_Y.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
2L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_pearsons_10Kb_2L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
2R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_pearsons_10Kb_2R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
3L BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_pearsons_10Kb_3L.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
3R BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_pearsons_10Kb_3R.txt


java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
4 BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_pearsons_10Kb_4.txt



java -Xmx10g -jar ~/myJuicerDir/scripts/common/juicer_tools.jar pearsons \
-p KR ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/aligned/BG3_BEAF_DREF_KD_HiC.hic \
X BP 10000 ~/BG3_HiC_KD/juicer/BG3_BEAF_DREF_KD_HiC/compartments/BG3_BEAF_DREF_KD_HiC_pearsons_10Kb_X.txt
