# python2.7
# -*- coding: utf-8 -*-
'''
Based on python2.7
Need bedtools and samtools.

Contain functions for counting the transcript body covarge.

BinGtf: Create bins for RNAseq or Endseq exons

Use either pathos or for multiple processin to speed up
SplitBam: split the bam file
RNAseqCoverage, EndseqCoverage: Bedtools multicov

CountCoverage: Count bedools multicov results for each bin
'''

try:
    import time
    import numpy as np
    import pandas as pd
    import os
    import re
    from subprocess import check_call
    from collections import OrderedDict
    import toolpath # Get global variable for samtools and bedtools path
except:
    print "Module import error"
    quit()


def CreateTemp():
    '''
    return a name for a temp dir based on the time
    '''
    localtime = time.asctime(time.localtime())
    Prefix = ''.join(localtime.split()[-2].split(':')) # '151542'
    return 'CoverageTmp{}'.format(Prefix) # CoverageTmp151542

def transcriptID(x):
    return re.findall('transcript_id \"(.*?)\";',x)[0]


# Generate bed file with bins
class BinGtf:
    def __init__(self, exon_gtf, number):
        self.exon_gtf = exon_gtf # gtf annoation containing exon information, e.g. gencode.v24.exonForcoverage.gtf
        self.number = number # number-1: number of bins for counting, 51 for 50 bins
        self.dic_abovecutoff = {} # a dict containing the transcript ID with FPKM above cutoff
    
    def ExtractGeneEndseq(self):
        '''
        For 5'end sequencing, use all the transcript in gtf file for counting analysis.
        '''
        with open(self.exon_gtf) as f:
            key = (line.strip().split('\t')[5].split(';')[0] for line in f)
            self.dic_abovecutoff = dict.fromkeys(key)
        return 1

    def ExtractGeneRNAseq(self, FPKM_file, FPKM_cutoff):
        '''
        Based on RNAseq coverage, Extract gene with at least one transcript with FPMK >= FPKM_cutoff
        
        e.g. FPKM file is generated by RSeQC: 
        # chrom  st      end     accession       mRNA_size       gene_strand     Frag_count      FPM     FPKM
        Need transcript_id (accession) in column 4 and FPKM in the last column

        Create an gene annotation containing these genes -> self.gene_gtf_abovecutoff: gene_gtf_abovecutoff_temp.gtf
        '''
        # Extract exons with FPKM>=cutoff
        with open(FPKM_file) as f:
            f.readline()
            key = (line.strip().split('\t')[3] for line in f if float(line.strip().split('\t')[-1])>=FPKM_cutoff)
            self.dic_abovecutoff = dict.fromkeys(key)
        return 1

    def OrderFrom5To3(self, ls_exon, strand):
        '''
        ls_exon containing the boundary of exons ordered from 5'->3'end, e.g.
        [(exon_number 1 start, exon_number 1 end), (exon_number 2 start, exon_number 2 end) ...]
        
        return OrderedDict([(position, exon number)...]) containing positions ordered from 5'->3'end.

        negative strand [(29534, 29570), (24738, 24891), (18268, 18366)]:
        return
        OrderedDict([(29570, 1), (29569, 1), (29568, 1)..
        (29535, 1), (29534, 1), (24891, 2), (24890, 2)..
        (24739, 2), (24738, 2), (18366, 3), (18365, 3), (18364, 3)..
        (18270, 3), (18269, 3), (18268, 3)])

        positive strand [(11869, 12227), (12613, 12721), (13221, 14409)]:
        return
        OrderedDict([(11869, 1), (11870, 1)..
        (12226, 1), (12227, 1), (12613, 2), (12614, 2)..
        (12720, 2), (12721, 2), (13221, 3), (13222, 3)..
        (14408, 3), (14409, 3)])
        '''
        dic = OrderedDict()
        if strand == "+":
            i = 1 # i is the exon number
            for exon in ls_exon:
                for item in range(exon[0], exon[1]+1):
                    dic[item] = i
                i +=1
        else:
            i = 1 # i is the exon number
            for exon in ls_exon:
                for item in range(exon[1], exon[0]-1, -1):
                    dic[item] = i
                i +=1
        return dic

    def BinExon(self, dic_exon_position, number):
        '''
        Bin the positions in ls_exon_position ordered from 5'->3'end that is returned by OrderFrom5To3(ls_exon, strand).
        Generate number-1 bins using np.linspace(), e.g. 100 bins for number=101
        return list [(start, end, bin number)...]

        negative strand [(29534, 29570), (24738, 24891), (18268, 18366)] split into 50bins (number=51):
        return:
        [(29566, 29570, 'bin1'), (29560, 29565, 'bin2'), (29554, 29559, 'bin3')..  
        (18280, 18285, 'bin48'), (18274, 18279, 'bin49'), (18268, 18273, 'bin50')]
        
        positive strand [(11869, 12227), (12613, 12721), (13221, 14409)] split into 50 bins: 
        return
        [(11869, 11901, 'bin1'), (11902, 11934, 'bin2') .. 
        (12200, 12227, 'bin11'), (12613, 12617, 'bin11'), (12618, 12650, 'bin12') .. 
        (12717, 12721, 'bin15'), (13221, 13249, 'bin15') ..
        (14343, 14375, 'bin49'), (14376, 14409, 'bin50')]
        '''
        index = np.linspace(0, len(dic_exon_position.keys()), dtype=int, num=number)
        ls = [] # list containing the bourndary of each bin, break the bin for intron (the spanning bin have the same bin number), e.g. [(start, end, bin number)...]
        
        bin_ls = range(1,len(index))

        for i in range(1,len(index)): # [index[i-1]:index[i]]: interval of bin
            dic = OrderedDict() # {exon: [positions ...]}
            for key in dic_exon_position.keys()[index[i-1]:index[i]]: # key for item in each bin, dic_exon_position[key] = exon number
                if dic.has_key(dic_exon_position[key]):
                    dic[dic_exon_position[key]].append(key)
                else:
                    dic[dic_exon_position[key]] = [key]
            for x in dic:
                ls.append((min(dic[x]), max(dic[x]), 'bin{}'.format(bin_ls[i-1])))
            
        return ls
    
    def pairExon(self, line):
        '''
        input line, e.g. from gencode.v24.exon4coverage.gtf :
        chr1    .       .       11869   14409   ENST00000456328.2;1657; +       .       11869;12227;12613;12721;13221;14409;
        
        return a list containing exon start, end as int, e.g.
        [(11869, 12227), (12613, 12721), (13221, 14409)]
        '''
        line = line.strip().split('\t')
        exon = [int(x) for x in line[8].split(';')[:-1]]
        return [(exon[i], exon[i+1]) for i in range(0, len(exon), 2)]
    
    def generateGtfBin(self, output_gtf, size_cutoff):
        '''
        used to create a bed file with number-1 bins for each transcript entry having transcript size >= size_cutoff in the input_gtf.
        
        input_gtf e.g. 
        chr1	.	.	1785285	1891117	ENST00000610897.4;3145;	-	.	1890820;1891117;1825397;1825499;1817837;1817875;1815756;1815862;1806475;1806538;1804419;1804581;1793245;1793311;1790395;1790596;1789053;1789269;1787322;1787437;1785285;1787053;

        output_gtf with 100 bins, e.g.
        chr1    .       .       1891087 1891117 .       -       .       ENST00000610897.4;bin1;
        chr1    .       .       1891056 1891086 .       -       .       ENST00000610897.4;bin2;
        chr1    .       .       1891024 1891055 .       -       .       ENST00000610897.4;bin3;
        chr1    .       .       1890993 1891023 .       -       .       ENST00000610897.4;bin4;
        ..
        chr1    .       .       1785317 1785347 .       -       .       ENST00000610897.4;bin99;
        chr1    .       .       1785285 1785316 .       -       .       ENST00000610897.4;bin100;        

        number=101: 100 bins (bin1, bin2 ... bin50) correspond to 2%, 4% ... 100% from 5'->3'of each transcript
        '''

        size_cutoff = max(self.number-1, size_cutoff)
        output = open(output_gtf, 'w')

        with open(self.exon_gtf) as f:
            for line in f:
                # transcript FPKM>=cutoff and transcript size >= bin number:
                if self.dic_abovecutoff.has_key(line.strip().split('\t')[5].split(';')[0]) and int(line.strip().split('\t')[5].split(';')[1])>=size_cutoff:
                    strand = line.strip().split('\t')[6]
                    exon_ls = self.pairExon(line)
                    bin_ls = self.BinExon(self.OrderFrom5To3(exon_ls, strand), self.number)
                    for bin in bin_ls:
                        attri = [line.strip().split('\t')[5].split(';')[0], bin[2], '']
                        temp = [line.strip().split('\t')[0], '.', '.', str(bin[0]), str(bin[1]), '.', strand, '.', ';'.join(attri)]
                        print>>output, '\t'.join(temp)
                    
        output.close()
        return 1
        # an ordered dic containing transcript used for counting e.g. {id: count 0 in bin1, count 0 in bin2 ..count 0 in bin50}, {ENST00000618181.4: [0, 0 ...0]} 

# Count using Bedtools multicov, multiprocessing
def SplitBam(bam):
    '''
    Split to Read1 and Read2 bam files
    '''
   
    with open('mysamtools.sh', 'w') as script:
        print>>script, '#!/bin/sh'
        print>>script, '{}samtools view -h -b -f 64 {} > R1_temp.bam'.format(toolpath.samtoolspath, bam)
        print>>script, '{}samtools index R1_temp.bam'.format(toolpath.samtoolspath)
        print>>script, '{}samtools view -h -b -F 64 {} > R2_temp.bam'.format(toolpath.samtoolspath, bam)
        print>>script, '{}samtools index R2_temp.bam'.format(toolpath.samtoolspath)
       
    try:
        check_call('sh mysamtools.sh', shell=True)
        return 1
    except:
        print "samtools error."
        quit()

def RNAseqCoverage(gtf):
    '''
    Use bedtools multicov for counting the directional RNAseq coverage,
    for which R2 has the same direction as the RNA (-s) and R1 has the opposite direction as the RNA (-S).

    output, e.g.
    For gtf format, col10: counts in this bin
    chr1    .       .       11869   11901   .       +       transcript_id ENST00000456328.2;bin1; 98
    chr1    .       .       11902   11934   .       +       transcript_id ENST00000456328.2;bin2; 100

    gtf, e.g.: annotation.chr1
    '''

    with open('mybedtools.{}'.format(gtf), 'w') as script:
        print>>script, '#!/bin/sh'
        print>>script, '{}bedtools multicov -bams {} -bed {} -split -S > {}'.format(toolpath.bedtoolspath, 'R1_temp.bam', gtf, 'bedtools.multicov.{}.R1'.format(gtf)) # bedtools.multicov.splitannotation.chr1.R1
        print>>script, '{}bedtools multicov -bams {} -bed {} -split -s > {}'.format(toolpath.bedtoolspath, 'R2_temp.bam', gtf, 'bedtools.multicov.{}.R2'.format(gtf)) # bedtools.multicov.splitannotation.chr1.R2
        print>>script, 'cat bedtools.multicov.{}.R1 bedtools.multicov.{}.R2 > bedtools.multicov.{}'.format(gtf, gtf, gtf) # bedtools.multicov.splitannotation.chr1

    try:
        check_call('sh mybedtools.{}'.format(gtf), shell=True)
        print "Bedtools multicov for {}".format(gtf)
        return 'bedtools.multicov.{}'.format(gtf) # bedtools.multicov.splitannotation.chr1
    except:
        print "bedtools multicov error for gtf file {}.".format(gtf)
        quit()

def EndseqCoverage(bam, gtf):
    '''
    Use bedtools multicov for counting the R1 coverage of the 5'end RNAseq e.g. Cappable-seq,
    for which R1 has the same direction as the RNA (-s).

    output, e.g.
    For gtf format, col10: counts in this bin
    chr1    .       .       11869   11901   .       +       transcript_id ENST00000456328.2;bin1; 98
    chr1    .       .       11902   11934   .       +       transcript_id ENST00000456328.2;bin2; 100
    '''
    with open('mybedtools.{}'.format(gtf), 'w') as script:
        print>>script, '#!/bin/sh'
        print>>script, '{}bedtools multicov -bams {} -bed {} -split -s > {}'.format(toolpath.bedtoolspath, bam, gtf, 'bedtools.multicov.{}'.format(gtf)) # bedtools.multicov.splitannotation.chr1.R2
        
    try:
        check_call('sh mybedtools.{}'.format(gtf), shell=True)
        print "Bedtools multicov for {}".format(gtf)
        return 'bedtools.multicov.{}'.format(gtf) # bedtools.multicov.splitannotation.chr1
    except:
        print "bedtools multicov error for gtf file {}.".format(gtf)
        quit()

def BedtoolsCoveragePathos(gtf, bam, type):
    '''
    Use pathos for parallel processing on heterogeneous computing cluster, e.g. on cluster, or on a computer with multiprocessor.
    gtf: gtf file with bins, e.g. generated by generateGtfBin()
    '''
    # split the annotation file based on chr

    command = "awk -F \'\\t' \'{{print > \"splitannotation.\"$1}}\' {}".format(gtf) # split based on chr, output files: splitannotation.chr1, splitannotation.chr2 ..splitannotation.chrM
    check_call(command, shell=True)

    annotation_list = [item for item in os.listdir('./') if 'splitannotation' in item]

    try:
        from pathos.pools import ProcessPool
        print "multiple processing using pathos"
        Flag = True
    except:
        import multiprocessing
        Flag = False

    if Flag:
        if type == 'RNAseq':
            SplitBam(bam)
            pool = ProcessPool()
            multicov_ls = pool.map(RNAseqCoverage, annotation_list)
            pool.close()
            pool.join()
            # print "The following are the files used for coverage counting."
            # print multicov_ls
        else:
            pool = ProcessPool()
            bam_list = [bam for item in annotation_list] # use the same bam for all the split annotation
            multicov_ls = pool.map(EndseqCoverage, bam_list, annotation_list) # pool.map(function, arg1, arg2...)
            # print "The following are the files used for coverage counting."
            # print multicov_ls
    else:
        print "pathos module not available. Use multipleProcessing instead."
        # print "There are {} processors available.".format(multiprocessing.cpu_count())
        if type == 'RNAseq':
            SplitBam(bam)
            pool = multiprocessing.Pool(processes=multiprocessing.cpu_count()) # multiprocessing.cpu_count(): check number of CPUs
            multicov_ls = [pool.apply(RNAseqCoverage, args=(x,)) for x in annotation_list]
        else:
            pool = multiprocessing.Pool(processes=multiprocessing.cpu_count())
            multicov_ls = [pool.apply(EndseqCoverage, args=(bam,x,)) for x in annotation_list]

    return multicov_ls

# Count the counts generated by multicov in each bin from 5'->3' direction, report the normalized coverage
class CountCoverage:
    def __init__(self, multicov_ls, number, output_file):
        self.number = number # number-1 bins used for calculation
        self.multicov_ls = multicov_ls # list containing the bedtools multicov outputs, e.g. multicov for each annotation file based on chr
        self.output_file = output_file # output matrix with normalized percentage of coverage

    
    def CaculateCountInTranscript(self):
        '''
        Count the number of counts in each bin for each transcript,
        output a dataframe containing count of each bin of gene from 5'->3' direction, xaxis is bin1..binN, yaxis is transcript id 
        '''
        dic_transcript = OrderedDict()
        for input in self.multicov_ls:
            with open(input) as f:
                for line in f:
                    if not dic_transcript.has_key(line.strip().split('\t')[8].split(';')[0]):
                        dic_transcript[line.strip().split('\t')[8].split(';')[0]] = [0]*100

        df = pd.DataFrame.from_dict(dic_transcript, orient='index', columns=['bin'+str(item) for item in range(1, self.number)]) # bin1...bin50

        for input in self.multicov_ls:
            with open(input) as f:
                for line in f:
                    id, binname = line.strip().split('\t')[8].split(';')[0], line.strip().split('\t')[8].split(';')[1]
                    df.at[id, binname] = int(line.strip().split('\t')[-1]) + df.at[id, binname]
        
        df.to_csv(self.output_file, index=True, header=False,sep='\t') # Add this to output the matrix containing counts of each transcript
        return 1 
