/*
This work is licensed under the Creative Commons Attribution-Non-Commercial-ShareAlike 4.0 International License.
To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/4.0/.
For use of the software by commercial entities, please inquire with Tel Aviv University at ramot@ramot.org.
&copy; 2019 Tel Aviv University (Erez Y. Levanon, Erez.Levanon@biu.ac.il;
Eli Eisenberg, elieis@post.tau.ac.il;
Shalom Hillel Roth, shalomhillel.roth@live.biu.ac.il).
*/

package EditingIndexJavaUtils;

/*
  @author Michal Barak
 * @author 06/2018 Shalom Hillel Roth
 */

import org.apache.commons.cli.*;

import java.io.*;
import java.util.Map;
import java.util.TreeMap;

/**
 * This class generates a cmpileup file from a pileup using a genome index generated by {@link BEDGenomeIndexer}.
 * A cmpile file is a tab file with the following columns (in this order):
 *      containing region chromosome from the index
 *      containing region start from the index
 *      containing region end from the index
 *      record position
 *      reference base at the position
 *      total coverage at the position
 *      number of reads showing 'A' at the position
 *      number of reads showing 'C' at the position
 *      number of reads showing 'G' at the position
 *      number of reads showing 'T' at the position
 *      number of reads showing 'N' at the position
 *      number of reads with Phred quality score under defined threshold at the position
 */
public class PileupToCount {

    /**
     * This is the main function of the class.
     * @param args The arguments from the command line: input pileup file to convert, path for the output,
     *            Phred quality score threshold (under which bases are ignored), the offset of the Phred score, and a
     *             path to a genome index generated by {@link BEDGenomeIndexer}.
     */
    public static void main(String[] args) {
        PileupToCount pc = new PileupToCount();

        Options options = new Options();
        Option input = new Option("i", "inputPileup", true, "input pileup file path");
        input.setRequired(true);
        options.addOption(input);

        Option output = new Option("o", "outputPath", true, "cmpileup output path");
        output.setRequired(true);
        options.addOption(output);

        Option qualityThreshold = new Option("q", "qualityThreshold", true, "The Phred score threshold to use");
        qualityThreshold.setRequired(true);
        options.addOption(qualityThreshold);

        Option qualityOffset = new Option("f", "qualityOffset", true, "The PHRED score offset");
        qualityOffset.setRequired(true);
        options.addOption(qualityOffset);

        Option genomeOption = new Option("g", "genomeIndex", true, "The path to the genome index file.");
        genomeOption.setRequired(true);
        options.addOption(genomeOption);


        CommandLineParser parser = new DefaultParser();
        HelpFormatter formatter = new HelpFormatter();
        CommandLine cmd = null;


        Map<String, TreeMap<Integer, String>> genomeIndex = null;

        try {
            cmd = parser.parse(options, args);
        } catch (ParseException e) {
            System.out.println(e.getMessage());
            formatter.printHelp("PileupToCount", options);
            System.exit(1);
        }

        String inputPileupPath = cmd.getOptionValue(input.getOpt());
        String outputFilePath = cmd.getOptionValue(output.getOpt());
        int qThreshold = Integer.parseInt(cmd.getOptionValue(qualityThreshold.getOpt()));
        int qOffset = Integer.parseInt(cmd.getOptionValue(qualityOffset.getOpt()));
        String genomeIndexPath = cmd.getOptionValue(genomeOption.getOpt());

        FileInputStream fileInputStream = null;
        try {
            fileInputStream = new FileInputStream(genomeIndexPath);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            System.out.println("PileupToCount - Can't Find Genome Index File!");
            System.exit(1);
        }
        ObjectInputStream objectInputStream = null;
        try {
            objectInputStream = new ObjectInputStream(fileInputStream);
        } catch (IOException e) {
            e.printStackTrace();
            System.out.println("PileupToCount - Can't Open Genome Index File!");
            System.exit(1);
        }
        try {
            genomeIndex = (Map<String, TreeMap<Integer, String>>) objectInputStream.readObject();
        } catch (IOException | ClassNotFoundException | ClassCastException e) {
            e.printStackTrace();
            System.out.println("PileupToCount - Can't Load Genome Index File!");
            System.exit(1);
        }

        pc.AnalizePileup(inputPileupPath, outputFilePath, qThreshold, qOffset, genomeIndex);

        System.exit(0);
    }


    private void AnalizePileup(String pileupFile, String countFile, int qulTreshold, int qulOffset,
                               Map<String, TreeMap<Integer, String>> genomeIndex) {
        java.io.BufferedReader br = null;
        BufferedWriter bw = null;
        try {
            br = new java.io.BufferedReader(new java.io.FileReader(pileupFile));
            bw = new BufferedWriter(new FileWriter(countFile));
            String line;
            CountLocations cl = new CountLocations();
            while (true) {
                line = br.readLine();
                if (line == null || line.isEmpty())
                    break;

                cl.AddLocationInfo(bw, line, qulTreshold, qulOffset, genomeIndex);
            }

        } catch (IOException e) {
            System.out.println("error: " + e.getMessage());

        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (Exception e) {
                    System.out.println("final error: " + e.getMessage());
                }
                try {
                    assert bw != null;
                    bw.close();
                } catch (Exception e) {
                    System.out.println("final error: " + e.getMessage());
                }
            }
        }
    }

    class CountLocation {
        String gene;
        char strand;
        char originalNt;
        char originalRefseq;
        int count = 0;
        int As = 0;
        int Cs = 0;
        int Gs = 0;
        int Ts = 0;
        int others = 0;
        int low = 0;
        char SNP = '.';
        char mut = '.';
        int ntCount = 0;

        void setCountLocation(String inGene, char inStrand, int inCount, char inOriginal, String res, String qul, int qulTreshold, int qulOffset) {
            gene = inGene;
            strand = inStrand;
            originalNt = inOriginal;
            count = inCount;
            analyseRes(res, qul, qulTreshold, qulOffset);
        }

        void printData(BufferedWriter bw, String inChrm, int inLocation, char original, int origStart, int origEnd) {
            StringBuilder sb = new StringBuilder();

            sb.append(inChrm).append("\t");

            sb.append(origStart).append("\t");
            sb.append(origEnd).append("\t");

            sb.append(inLocation).append("\t");

            sb.append(original).append("\t");

            sb.append(count).append("\t");
            sb.append(As).append("\t");
            sb.append(Cs).append("\t");
            sb.append(Gs).append("\t");
            sb.append(Ts).append("\t");
            sb.append(others).append("\t");
            sb.append(low);

            sb.append("\n");
            try {
                bw.write(sb.toString());
            } catch (IOException e) {
                System.out.println("error: " + e.getMessage());
            }
        }

        void analyseRes(String res, String qul, int qulTreshold, int qulOffset) {
            boolean indel = false;
            int indelLen = 0;
            int qulLoc = 0;
            String sLength = "";
            int indelCount = 0;
            boolean start = false;
            for (int i = 0; i < res.length(); i++) {
                if (!start) {
                    if (indel) {
                        if (Character.isDigit(res.charAt(i))) {
                            sLength = sLength.concat(Character.toString(res.charAt(i)));
                            indelLen = Integer.valueOf(sLength);
                        } else {
                            if (res.charAt(i) == 's') {
                                indel = false;
                                indelCount = 0;
                                indelLen = 0;
                                sLength = "";
                                others++;
                            } else {
                                indelCount++;
                                if (indelCount == indelLen) {
                                    indel = false;
                                    indelCount = 0;
                                    indelLen = 0;
                                    sLength = "";
                                }
                            }
                        }
                    } else {
                        if (qulLoc > qul.length()) {
                            System.out.println("error in qul:\t" + res + "\t" + qul);
                        } else {
                            switch (res.charAt(i)) {
                                case '!':
                                case '$':
                                case ':':
                                case '(':
                                case ')':
                                case 'F':
                                case '%':
                                case '\'':
                                case '&':
                                    break;
                                case '^':
                                    start = true;
                                    break;
                                case '+':
                                case '-':
                                    indel = true;
                                    break;
                                default: {
                                    if (!Character.isDigit(res.charAt(i))) {
                                        if (((int) qul.charAt(qulLoc)) - qulOffset >= qulTreshold) {
                                            switch (res.charAt(i)) {
                                                case 'A':
                                                case 'a':
                                                    As++;
                                                    break;
                                                case 'C':
                                                case 'c':
                                                    Cs++;
                                                    break;
                                                case 'G':
                                                case 'g':
                                                    Gs++;
                                                    break;
                                                case 'T':
                                                case 't':
                                                    Ts++;
                                                    break;
                                                case '.':
                                                case ',':
                                                    ntCount++;
                                                    break;
                                                case '*':
                                                default:
                                                    others++;
                                            }
                                        } else {
                                            low++;
                                        }

                                        qulLoc++;
                                    }
                                }
                            }
                        }
                    }
                } else {
                    start = false;
                }
            }
            switch (originalNt) {
                case 'A':
                case 'a':
                    As += ntCount;
                    break;
                case 'C':
                case 'c':
                    Cs += ntCount;
                    break;
                case 'G':
                case 'g':
                    Gs += ntCount;
                    break;
                case 'T':
                case 't':
                    Ts += ntCount;
                    break;
            }
        }


    }

    class CountLocations {


        final int CHRM_COL = 2;
        final int LOCATION_COL = 3;
        final int COUNT_COL = 5;
        final int ORIGINAL_COL = 4;
        final int RES_COL = 6;
        final int QUL_COL = 7;
        final int RNA_DNA_DIFF = 2;

        void AddLocationInfo(BufferedWriter bw, String line, int qulTreshold, int qulOffset,
                             Map<String, TreeMap<Integer, String>> genomeIndex) {
            String parts[] = line.split("\\s");
            //System.out.println(String.valueOf(parts.length));


            String gene, chrm, res, qul, origSeq;
            char strand, original;
            int location, count, origStart;


            //changed 10-7-2016: added +1
            if ((parts.length < COUNT_COL - RNA_DNA_DIFF + 1) || (Integer.valueOf(parts[COUNT_COL - RNA_DNA_DIFF]) == 0))
                return;
            gene = "";
            strand = '?';
            chrm = parts[CHRM_COL - RNA_DNA_DIFF];
            location = Integer.valueOf(parts[LOCATION_COL - RNA_DNA_DIFF]);
            original = parts[ORIGINAL_COL - RNA_DNA_DIFF].charAt(0);
            count = Integer.valueOf(parts[COUNT_COL - RNA_DNA_DIFF]);
            res = parts[RES_COL - RNA_DNA_DIFF];
            qul = parts[QUL_COL - RNA_DNA_DIFF];


            try {
                Map.Entry<Integer, String> entry = genomeIndex.get(chrm).lowerEntry(location);
                origSeq = entry.getValue();
                origStart = entry.getKey();
                Character genOriginal = origSeq.charAt(location - origStart - 1);
                CountLocation cl = new CountLocation();
                cl.setCountLocation(gene, strand, count, original, res, qul, qulTreshold, qulOffset);
                //cl.findEditing(minEditing);
                if ((cl.As + cl.Cs + cl.Gs + cl.Ts) > 0) {

                    cl.printData(bw, chrm, location, genOriginal, origStart, origStart + origSeq.length());
                }

            } catch (StringIndexOutOfBoundsException | NullPointerException e) {
                System.out.println("PileupToCount - Warning: Found Coverage Out Of index! (Ignoring)");
            }


        }


    }


}
