package edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.hmm;

import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.felsenstein.likelihood.BeagleStateLikelihood;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.felsenstein.sitemodel.SiteModel;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.felsenstein.substitution.Frequencies;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.felsenstein.substitution.JukesCantor;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.felsenstein.substitution.SubstitutionModel;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.structs.ModelTree;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.test.HCGModelBuilder;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.util.Utils;
import edu.rice.cs.bioinfo.programs.phylonet.algos.MCMCcoal.felsenstein.alignment.Alignment;

import java.io.*;
import java.util.*;

/**
 * Main Hmm class; it implements a Hidden Markov Model.
 *
 * An HMM is composed of:
 * states: each state has a given probability, pi, of being initial state
 *         Each state is associated to an index; the first state
 *         is numbered 0, the last n-1 (where n is the number of states in the HMM);
 *         this number is given as an argument to the various functions to refer to
 *         the matching state.
 * transition probabilities: the probability of going from state i to state j, a[i,j].
 *
 * Created by Xinhao Liu on 12/30/19.
 */
public class HmmCore {
    private double[] pi;
    private double[][] a;
    private List<HiddenState> states;

    public HmmCore(double[] pi, double[][] a, List<HiddenState> states) {
        this.pi = pi;
        this.a = a;
        this.states = states;
        convertToLogProb();
    }

    public double[] getPi() {
        return pi;
    }

    public double[][] getA() {
        return a;
    }

    public List<HiddenState> getStates() {
        return states;
    }

    public HiddenState getState(int stateIdx) {
        return states.get(stateIdx);
    }

    public int getNumStates() {
        return states.size();
    }

    private void convertToLogProb() {
        // currently assume e is the base of beagle log likelihood
        int numStates = getNumStates();
        double[] logPi = new double[numStates];
        double[][] logA = new double[numStates][numStates];

        // convert Pi vector to logarithm
        for (int i = 0; i < numStates; i++) {
            logPi[i] = StrictMath.log(pi[i]);
        }
        // convert A matrix to logarithm
        for (int i = 0; i < numStates; i++) {
            for (int j = 0; j < numStates; j++) {
                logA[i][j] = StrictMath.log(a[i][j]);
            }
        }
        this.pi = logPi;
        this.a = logA;
    }

    /**
     * Code adapted from https://github.com/jmacglashan/generalResearch/blob/master/src/generativemodel/LogSumExp.java
     */
    public static double logSumOfExponentials(double[] exponentialTerms){
        if(exponentialTerms.length == 0){
            return Double.NEGATIVE_INFINITY;
        }

        double maxTerm = Double.NEGATIVE_INFINITY;
        for(double d : exponentialTerms){
            if(d > maxTerm){
                maxTerm = d;
            }
        }

        if(maxTerm == Double.NEGATIVE_INFINITY){
            return Double.NEGATIVE_INFINITY;
        }

        double sum = 0.;
        for(double d : exponentialTerms){
            sum += StrictMath.exp(d - maxTerm);
        }

        return maxTerm + StrictMath.log(sum);
    }

//    /**
//     * Forward algorithm
//     */
//    public double logLikelihood() {
//        int numStates = getNumStates();
//        double[] OLD = new double[numStates];
//        double[] NEW = new double[numStates];
//
//        // initialize dp matrix
//        Alignment firstColumn = Utils.DATA.get(0);
//        Frequencies firstFreq = new Frequencies(firstColumn, false);
//        SubstitutionModel firstSubstModel = new JukesCantor(firstFreq);
//        SiteModel firstSiteModel = new SiteModel(firstSubstModel); // NOTE: sitemodel gets its mutation rate from Util
//        for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
//            HiddenState state = states.get(stateIdx);
//            BeagleStateLikelihood beagle = new BeagleStateLikelihood(firstColumn, state, firstSiteModel, null);
//            double logEmissionProb = beagle.calculateLogP();
//            double matrixEntry = pi[stateIdx] + logEmissionProb;
//            OLD[stateIdx] = matrixEntry;
//        }
//
//        // forward computation
//        for (int siteIdx = 1; siteIdx < Utils.DATA.size(); siteIdx++) {
//            Alignment column = Utils.DATA.get(siteIdx);
//            Frequencies freq = new Frequencies(column, false);
//            SubstitutionModel substModel = new JukesCantor(freq);
//            SiteModel siteModel = new SiteModel(substModel);
//            for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
//                HiddenState state = states.get(stateIdx);
//                BeagleStateLikelihood beagle = new BeagleStateLikelihood(column, state, siteModel, null);
//                double logEmissionProb = beagle.calculateLogP();
//                double[] terms = new double[numStates];
//                for (int prevStateIdx = 0; prevStateIdx < numStates; prevStateIdx++) {
//                    terms[prevStateIdx] = OLD[prevStateIdx] + a[prevStateIdx][stateIdx] + logEmissionProb;
//                }
//                NEW[stateIdx] = logSumOfExponentials(terms);
//            }
//            // exchange old and new
//            double[] temp = OLD;
//            OLD = NEW;
//            NEW = temp;
//        }
//        return logSumOfExponentials(NEW);
//    }

    /**
     * Forward algorithm
     */
    public double logLikelihood() {
        int numStates = getNumStates();
        double[] OLD = new double[numStates];
        double[] NEW = new double[numStates];

        // Calculate likelihood of entire sequence alignment for all states, keep pattern log likelihoods for each state
        Alignment alignment = Utils.DATA;
        Frequencies freq = new Frequencies(alignment, false);
        SubstitutionModel substModel = new JukesCantor(freq);
        SiteModel siteModel = new SiteModel(substModel); // NOTE: sitemodel gets its mutation rate from Util
        Map<Integer, double[]> state2PatternLikelihood = new HashMap<>();
        for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
            HiddenState state = states.get(stateIdx);
            BeagleStateLikelihood beagle = new BeagleStateLikelihood(alignment, state, siteModel, null);
            double[] patternLogLikelihoods = beagle.calculateLogP();
            state2PatternLikelihood.put(stateIdx, patternLogLikelihoods);
        }

        // Initialize dp matrix
        for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
            double logEmissionProb = state2PatternLikelihood.get(stateIdx)[alignment.getPatternIndex(0)];
            double matrixEntry = pi[stateIdx] + logEmissionProb;
            OLD[stateIdx] = matrixEntry;
        }
//        System.out.println(Arrays.toString(OLD));

        // forward computation
        for (int siteIdx = 1; siteIdx < alignment.getSiteCount(); siteIdx++) {
            for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
                double logEmissionProb = state2PatternLikelihood.get(stateIdx)[alignment.getPatternIndex(siteIdx)];
                double[] terms = new double[numStates];
                for (int prevStateIdx = 0; prevStateIdx < numStates; prevStateIdx++) {
                    terms[prevStateIdx] = OLD[prevStateIdx] + a[prevStateIdx][stateIdx] + logEmissionProb;
                }
                NEW[stateIdx] = logSumOfExponentials(terms);
            }
            // exchange old and new
            double[] temp = OLD;
            OLD = NEW;
            NEW = temp;
        }
//        System.out.println(Arrays.toString(OLD));
        double result = logSumOfExponentials(OLD);
        if (Utils.ILLEGAL_LIKELIHOOD == 0 && result != Double.NEGATIVE_INFINITY) {
            Utils.ILLEGAL_LIKELIHOOD = 1.2 * result;
        }
        if (result == Double.NEGATIVE_INFINITY) {
            Utils.ILLEGAL_SAMPLE_GENERATED = true;
            return Utils.ILLEGAL_LIKELIHOOD;
        }
        return result;
//        return logSumOfExponentials(OLD);
    }

    public static void experimentBeagleLikelihood() {
        ModelTree trueModel = HCGModelBuilder.getHCGModel();
        HmmBuilder builder = new HmmBuilder(trueModel.getTree(), trueModel.getRecombRate());
        HmmCore hmm = builder.build();

        Map<String, String> omap = new HashMap<>();
        omap.put("H", "AGACATATTCGCGGCAGTTGCTCTCCGCCACAGGGGTTGGGTCCAACCCCAGTACGTACCTCATTCGTCAATCTGGGTACAATCCCAGGACAAAGGATAGTGCTCTCTGGTCAGAGCTCTTTGTACAACTTGCATTCGATTGTGAAACAAAATGGAGCTCGGTTTATGGCCTCGGTGAGTCCTGTTTATTGGACCCTAGGCGAATTAGGCACTTTATTATAGCGTGGCCCCGAGCAGGACGCGATGGCGCCCAAGGCGCCTAAACTTGATCAGTGAACGGGTGATAAACGGCTGTTCTCGGGGCTCAAGCCGATCTGTTACAGTTAACCTATAGAACCGGTTACGACAGGAAAAGTTAATCCCGATCATTGTTCTTGGAGTCTAAATACAAGAACGTTGCAGGCTGACATCGCGATGAGCAGGAGTGGATCACACTACTAAATAAATTGGTTCTCGAGGACCTAAAATTCTTGGTCATGGATGTTGACATATTAATCTATCTAAAGAAAGTAGATGGAAGTGCAAGAAGAAACACCAGGCAATTCTCATTTATTCGAGTTGACACGCCCAGATAGACTGGGACTCTCCCTAATCGTCTGGGCCTGAAAAACGTCGGATCCATAAAACGTCATTTGGCCGAGGCCCATCAGGTTAGAGTTACGCTACACACTATCTTGCATGCGAGATACGAACTGCCAAACTAGGCGTGACAATATCACCCCACAATTGTTTTCTAAGAAATCACTTACTTCGCTACGTGGGGGACTCTATAGCGCGACCACACTTGCCCTCATCTGGGTATACACGCCCAAGAGCTACAACAGTGTGGAACGTTACCATTGATGATAGCACGCAGGTCGACTTGCAGTCACCTGGTATGCATCTCTAGAGGGTACGCGAGACCAGAGCACCATAAACAAAGAGCCGACCAGAGGATAGTCGCTTCTTATCTGCAGAATGACTTCGTGCCTGTTTGAGGATCATTAACGATAGTTGGCAGTATTGAACTGATTGCTAGAGTCCTCCGCACTATGTGACGAAGGGGAGTGTTAAAAGCGGACTTTATTGAATGGTAGATGCTAACATAGAGCTGAAGTGTGGATTTGGTCTACCCTAGCTGTGCTCTAATCATTCAGTTGAGTGTTTGCGCTCCTGCGGATGGTCCCCGTTGTCTTAGTACTGCGCCCAACCAGACGATGTGAGTATCATTCGATCCTGCGAAAGTTAATTCGCTATGACCAAGATAGCAGATTGAGGTAACATAGAGGTTCCGATGCTTGTGTGCCGGTGAGCTGCTCTTGTCGTAATGGGCCCCCAAACGTCAACTACTCTAAGTACCAAAAAGGGGATTACATAGTATCAGCAACTCTGCTTTCGAGCCCTGCTAGTCGAACGCTTGTGACCTGGCGGCTGTTCGCAGATTCTGTTTATAGGCTAAGTCTACATTAGTGCTGAAGAGGTTATCTCGGCCTGGCCTAAGACCACATTAAGAAATGTCGTTCTTGCCCCTCACTAATCCATGCTGAGACAGAGGATCCGTACGGACAGATTATTCATTCCCCCCCGTGGTTACATAAACGCGTTCCTTTGTTACGATTCTACAAGAGACAGTGCGAGAGGTGGCCGATATATGTCCTTTTGCCAAACACCTAGACCTGGCGGTCTCTCAACGGGAAGGCCGCGTCGCTGCCTATAGATTCCCGAGACTCCAATGTTGCGAGAGCAAATCTGTACGTTGCCCGTTAACGCTTAAAACCGGTAGAGAGCCCGCGAACGCGCTTCATACCTGTGCCAGGCAGATATGGTATTCTTTGATAACACAGGTCGAGTCTAATCGTCGCTACTCCTATCAACCCGCGATTGGCGGATCATGTTATGGTAATGTGAACGGCTTGACTTTGTGATCCTGGCCGAATGTAATCTTCTGTCTTCTTAAGGGATTGATATCACAAAAATCAAACCCTCCTCGAATCTACACATCAGAAGTCCGGGCCAACTGGGGATCTCACTAACCTATGTGACTCGACAAATATCGCCGAACCCGATACATAGCATACGCTAGTCCGAAAGCTTGTGCTCCCAGGTGTTGACGACTACATTCGACCACTGATAGCTCGACTCTGACTCCACTCGTACAGTTAGGAGAGTTCGTCCCGTCCCTTATTACGTCTGATCGAACAGTACTGCCTGAATTCCTCCGGTACTTTAACTCCATGTTTCACACCGGAGTTGTCAAATTTGTCTTGGCCAAACCTGCGTCTCACTAATACCACTGTATATGCTGATTGGCCATATATCTCAGAATAGGGATTGTTTGGGAGGGGAGTACTACTTCACTTAGGGGGCAGTGAATGCTATAGAATTATCGACCGGACTGCGTGTATGTTCAAACCTGACGTTCAATCATACATACTGCCCCGGTAAGGTTGAACTGTGTCTGAGAAGGCCTTAGAACTAGATATGGATGACTTCGAAATGCCACCCTCTATTTAGTCGTTACTATGAGTTTTCTACTCTACACACTGAGGCGACTCCAAGCGAACTCCCGGAGCATGCCTTCGGCCCTATGAATCTTGTCATTCTCATTGCGACACTACGAGCGCCGCTAGCGCGACGGACTAAAGCTCAAAGTCATTCAGCCCTATTATGATTATGCGTCCCACCCAGGCCAAAAGGCACACATAACTTGAAAAAAGTCATCATAGATTCATGCAAGGTCCCACTACCCGGGCGGTCTTGAAGTAGCGAGCATCCTGTGAATCTAGTCGCGTCCTTGTGTCGGAATTCGTTATATAAATCCACTGCTCCGCACCCTCATCAACGTCAATCGGTGGACTACCTCGTGTGCGTATTAGACTTTTATTCTCCTTTTATGATGGCGGGCCTAGCCGGACTAACACATACACTGTGATGTATTACGTAGGGTGCCGAAGATCACGATGAGCCTGACAACTTCCCGGTGCTTAAGCCTCTTTTGTCGTACACAAAGAGAGGGCGTCTGGTCCATGAAAAGCTTACTGTACGCAATACATAGGATCGTGATCGGGGCTCCCAAACAGCCGCGCGTACGCCGTTCCCGGGATTCTAAGGTGGGGCGCCCAGTGCCACTCCATATCAATGGTAACCAATATAGGAACACTCATCTAACGGGCGTCTATGTCCAATAATAAGTGTCTACATTAGCCGTGGTAGAACGACGCAGGTGATCTCCGCTTACGCGTCGATCGAACACTGGCCGACCTACGTAATCCGTGAAACCAACCGGCGAGCCTCAGGTTCTGTTAACGATTGCGCTCATATTGGACCCGGCATGCTTTAGGGCGCGGCCTGCTTTGCAGGAATAGCCACTAGCGATGTGGCCAACATCTTTTAAGCTGGGTCGGACCGAAGGCCCGCGATGATCTGGACATAGTAGGCAAGTAAGTTTACACCGCTGGGCCGTCCGGTAACACAGCACCCTTGAACCCTCAGCATCTACAACAAGGGCGTCGTACGCTGGGCGCAAGATTAGACGCTGTATGTGTGCCAGAGGTAAAAACCGTTCTGCTTTGAACTGCGCGCTCGGCCTAAGTGGGGCGAGGGCTGTGAGGAAGATACTGCACTCGTTAAGCGAATATTTGCGGCATATTCGCCAGTTATGGAACTGCACAGCTGTAAGGGCGAATGCCGACTCTAACCCCCATCGATCTATGAGATGTCCTTTACTAAAACGGGTCGGTGGATCGCTGCAGGTCAGAGTTACTGAATACGACATTGTACCAAACCAATCGGGGCGTACCAGTAGTAAAGCTCATTTACTTAGACAAGTGGCGTTTAAATTTCGGCCACGAAAGTTGATAAACAGGACGATGCTAGGCTTGGCAGAAATTTACCTATCCCATGGCTGGACTCCGATGTACCGATGGTCCAATGCGCCTACCCTAAGCGCAGCAGACGGTCGTTCGACCTTGTGGTAGTTAACACAATGCTATCGCAATAAGCTGCCGTTCGCCCCCACTAACTACACTGAGCCGAACCTCCGAAGATGAGCAGTCGACTTAGGTAAGAACAAAAAACTAGCAGTCGAAATATGCATTATTCCCCGAGCCATATCAATTCTGTTGGGAAATATGTAGGGCTGAGGTCCCACTTAAGGCATCTAACGAGAATGCGGGTGTTATCGGCCTTGTCGTGTGCTCAATACCAAAGGCGCATGGGTCGAATCACACGAAATTGCACGGCGAAGTGAATGGCCACGAGGTATTCCAGGCCTTCTGTGCCCCAGTCTGGTAAAATCGCCAGTTACCGTGTCCGACCTATCACGCGCGTTACCAGAGTATTTCCTCCTACCAATAATGATATGCTTGTACACACTGCTTGTGTCTGATTCGCGCCCTCGCCGATTCTCAACACAATACTGGAGGTCGAGATGCCCGTTTTAAGCCCAATAGGTCAAAGGTCTTCTGTTGACCTGGGTCGGCTAAGGTATGCGGGTGCCGCCCTATTCGGTTGTCGGGTGACCTATATTTTGTGACAGCCACATGGTCGCGATTTCCGAACCAATAAGGTGCCTCTATGAGCTATAAGCGCCACGGCAGTAAGACATGTAGTTTAGCATGGAACTTCTCAGTTGGAGGATACATTTTATCGATACACATGTCTTCAGGGGCAAGACGCGCGCCAGCTATAGCGTTTGATATCACAATCACATCTGTTATCGATTAACCTGTAGCTAGAGCCCTACAGACTCCGCTGCTTCTGGCGATCCCTTTAACACCTTCTGGTATATTCGTGCCCCGTGCTACCTGGCTATGGGACCCACTCTTTATTGTACAGCGACTCAACATCGGTGTGCATCACCAATCCAGGCGCGCCTTGTCAAGGGGACTATTATAATGAGCGGCTAAATATGCCACAAGCTATCAATATATGGTAGCCGACCAGTGTGTGAGATAAGTGAGTGTCGCAACTCTAAGGTTCGGTAAGGCTTCGTCTGCCCGGTAGCGTGATATATTTAGTTGGGCCTTTGCATTCCATGCCAAATGACACGGGGCTACGGTATGGAACCTCATCGTCCTGCAACATGGGCTTCCGAACAGTTCTAGGCTGTGGCATCCCCCGCGGGGCTGGACCCAAATTCGTTAGGGACGGAGCGGTACAAGCCAAACAGTGGAGCGACTCGGTAATTTATTGGACAGAAGCGCCAGGATATAGTATTCGTGCATGACTTTGGCCGGGGCCGATCCTGTCTCTGTAATAATTACCGCCAACGCTAGCGGAGAAATGAGGTAACGTGGTTCACGCTACATAATGACTCTTATCCGCTGACGTAATGAGACTCTCGTCATGGAATCAAGAAAAAGTAACCACCTGAACGGTCTACCTGCAAATGCGGATAGAACGGCCCGGAAGAAGCTACTGTGGCAGTTTAATTCATCATAATTAGGGGTCCGAATCTTGCACATACAGATTGCGAACGCACCATGCTCACTTGAATTTACAGGATAGCGTCTGTCAGCATTGCTCCCCGAGAAGGTGATCTAATCTGCCACAGGACGCTTGACGGCAAGTTCTACCCCCAGGGAGACCTTACTTATAAGCTATGGCGGGGCACCCTTGGGCTGCCAGACGGGTGAAGCACGGACTTTAGTATACGTACTTTCAAACATTCGACGGGATATGTCGCTTTAGACGGGTGACAGATGATTGAGCCCCTCATCTCATTTAGGTCGCGGTGTTTCGGGACGCGTGACATTCGAGTTCGTGGTAGGTGTGAAAACCCTAAAGTTCATGACTACGTAGGCGGGAGTGGAGTCCCTGTTACTAGTTTTTGAGAAGATTGCAATTATGATGGCAGACGAGGCCAGCGTTTGAGACAACTTAATGTCTTCTACCTTTAATGAGAAATACGATTCCTGTTTTCTGCCATACCATGAACCGAGTCTATATCAGGTAGGATACAAAATATGCAAAGACGATGGCCTCTCGTAATGTCTTGGGTCAAGACCTACTGGCACTCCTATTCAGGTACATACCCGAAGAGTCGGTTCTGGTCTTGCACGAGGCTTAGATCCCCGTATCCTGACTTACTAACAGACTCTCCGATAACAACCTGTCTTGAATTACTAGACGCCGCGGGTCTTCCACTACCCGCTCAGCCGACGATTGTAGACGCGTTGAGATCGGTGAAATCCGGACCATTCACTCGTGTTGAAAAGCTAACCTTTGCCGCGACGACACGAACTCCGGTCTGCACAGTGGTCGTGGAGGGTAACCAGTCAACTCCTATTGACCAGCGCAGAACCACCCTCGACCTCTCTGACCTTCCAGGTTTGTATGTGCCCGAAATGTGATACGCTCTCCAAGGGACCAAGAATATTATGCGACGGGCTCATCCGGCCAGAAGGAGCGCAGTATAGCTCCAATTGTTCTAGTAAACTGATCTGAGCCGTGTTGAATTTTGACGAGTTGAAAATCCCCGACTAAGCTTGCTTATCGCTTGTCGATTGATAGTAGGGACTTTCCATGCACATTCCACAACAGTGCACGACTAGGACAACTCCCTGACTGGATCAATACATATAATACTCGCGCTCTGTCATTCCCAATGGCCCTTGCTCCTGTCAATTGCTGGTCGTGGAGCGGGTACCTGATAATTTCGCTAGCGGGACGGATTCCACCAGGATCCAACCGACATGACGCGGTAACGTCACTAAGGATACGCGGACTTGCAGGCGCCGCTCTTCACGTTTAAGAGGCCCGTCAGTGCCAATTCGTCTAAAGGTAGTCGTCTCCGTTATAGGCCTAAGCATCTCGAGCTCAGCAAGCGATGCTCCGGATAGACGGATCATGTTAATGATCACATGTGACCTAGAGCACAGTCCCCCCTTAGACAGTTCTCGTTGAACGATGTGCCAACAGAGCTCCGGATGATTTACGCAGATTACGCGCGAACGCTCGTTGTACTGTTTGAGCGCTAAGTCCTTCTCTGCCCCTGCAACGCACGGTGTTGTCGGTTGGAGATTCAACCCTGCTTTCCTCTAGGGCGAGCCGTGACCTCCAGAGTCGACCGCTAGCGTGTGGCACGCGACGCCCAGGGTACGTGCTGCTCTAACCGTTCCAATGACGAGTTTAGTGTCTTACGAATACCGTTTTTGTGGGGCTCGGACCCATGGTTATAGGGGAGGGAATTCTAGTCGCAGGCTCCTTGTCCCTCAATTTTCGATAGCAAGCACATGGTGCCGAGTTATCTTTGTGGTGTTGACGACAACCGGCCCGCAGCCTTCTTGACTTATGAAAGTTACTTGGGAATAGGATGAAGTACTACCTCTTCCCAAACGAGAACAAATGATATGTATTATGCTGTCAGACCGTAGCTGCGCTGCGCGAACATCCTAGTTGGAGGATACCTCGTGCGTCAACGGGATCCGGGACGCTGCTATGATTCATACCTTCTTTACGTGGCTCGTGAACGTGATGATGTACCAGTATCCATATAAATCCTGGGCTTAGAGCCCTTTAGGACCTCTTCGACTGGCTCTTTCTTCACTGTCTTACCGCTGACGATTAATTCGACACGGGTAAATCACCCCACACTTAAATGGTGGGCGCCACTTAACGATCTAGCGAGGCTGCCGGAGTGCCATCCTCCCCGCTTCTAAATCCGGCCCGGCCTAGGGCGCCGAATCCCCGGGATGTAGATAGGAGCTACGGGCATCGTACTGCTTGGCTATCCCGGGCTTGCGCCTGATCGCCCAAGGGTGAATTGTTCATGTCTCGCGATGTCATAGACGTCAGCTCTTAGCATATCATGAATAGCAATGCCGGGTTCTAAAAGTCTTGAAGGGCCAGCCACACTCAGGAGTTACGGTAGACATGTACAGTAGCTGTACCGCGCCTGTGGTTTGCCACCGCTCCCTTGTTTCACCAACCTACCCACCAGTAGAACTGGTGCCGTCTCGACCGAGAACGGTATCGCTAAATCAATTCTACTTAGAAAGTTCTCCCCGTATCGCATCGTTGGAATTACCATGGGCATCGGGTACGTTGTACTAACCTCGTGTTCTTCAGACCAGCGTGAGACGTGTGGCCCGCGAGAGGATGTTCTCATGTCTAGGTCGACGGACCCTGATAACGAGCAAAAGTCCTATGTAACCCGCCCATAAAATGATCACCGTAATCATCCTGGCCTGTGTAATCATAATTTTCAGGGGCGGAGGCAACCCGGGGTACGCGGTGACCCCACAATGAGTTGGCCCATATGCTCTGAACGGACAATCGTCCACTGAGGCGCAGCCTCTTCAAGATCCTTTAACCTCAAGTCTTGCGATCTACTGGTCTACGACCTTTGAATGAAAATTCGATACGCACGGCGCGACTCTAAGAAGCAGTCGGAGGCATGTGTCAATGAGTGGCCTGCTCTGGCGGAACGGTTAAGTCTCGTCGTTAGTTCTACATAAATGGTACACAAGTGCAATCGACTACGTATATGTAGCTAACGGATAGCACAGGTGTGGCGCAACGATACCACGCTTCTTACAGCTTGTACTCGCGGACGTACCGTGCTGTTTCGGTTTTTCATATTTGCACAAACGGCTCGAGAACTATATGCGACATTTTAAACCCTTATGGGGGTAGAACGTACATCAACGTTGACATAATAACGGCAGGGCTCTTCCCATGTGGGCTCAAGGTATAGCCGACACCAGCCATCTGGACATATGAAGCGGATATCTTTGTCCTATCACCACGGTCGAACTATCAAGTATCGGTCGTTGGGAATGGACGATGAAAATGATCTTCGTTTTTATAGATAAGAATATTTCGCTCTAAAACTGTGGACGGAATACTTGTACTGTAAATTGGGAACGAGACAGTGATATGTGGCGTAATGAAAACCAGCGAAGGACCTCAGTAAAAGTAACGATACCTGGAAAAGTTGCGCACCGATTTCGACCGTCTGACTATTGGATGACAAGCACGCGCGTTCCTTCTACATCTTGCCGCAATGTCTCATCGTCGGGTTATGTCTAAGCAAATACGGAGATACTGATCGATCGTCAAAGCCATGTCCGGGGCGCTTATGACTAACACGACGTAACCTGTAAAGCTCGGCACGCATGTATCAACTTTTTAAGTTAAAGCATGAGAGTAGCCTCGACCGTTCTGCGCCCTCAGCTTACAGGCAAACGCCTTACGAAAGTCGGACGATTGGCTGTGGGTAAGGCGCATTGTCCGTGACCCGCTATGGATTTTTTGACGGCGCATTATCATGCCCATTTTTACATACTGACGTGGTGTTCAAGGGTGCGACAGGAGTCATCCCGGCGGATTCCGTATCTATAGCCGGGATGATAATCTATAGGCTGGTCAGGGGGAGTCACTGGCCCTTTCTGCGCCTATGGCCCATCGCGACTGGGGTGGCAGCTCTGCGCAAGTCCCGTTCAACTTTGATCTCAATGGATATCTACAAAGCCAATGCTAAGTAGTGATTGCCATGTTTTAGGGTAGTTACTAAGCCACTCGGCCAGTGTCCGAGTAAGACCATACACTGGTCCCAATTAATGCATCGTTACTAGTCTGCATCCTATATACACTTCGTGAGCATTTAGAAACACGAGCCCCCAAGCCAGGGCACAACTTTGTATTTAGGTATTCGACAACGTTCACTGGTCTGTACCGCTTCATACAATCCTCCCACAACAGACTTAAGCTAGGAACCAACAATTGAGATAAATTGACCTAGGAAAGACAGTCGGACGAATGTCAGATTAAATCATGTTCGTAATCGAAAATACGATGTGCGAACACATTTCACACATTCACTACGTTGTGGTAGTTTGCCTCGTACCAACTTTGTGAGTTATGAAAAATGAGGCAACAAAGCCAGAATCGAACCAACTCGTCATGGACCCAGTGCGTGAAAAGTAGTGGGACGGAGGGAACCCACTAGCTTTTAAAGAGCCTATGTATAAGCAGCTTCCGTCTCCCGTTCTCGCGTTTTCTGTAGAAACCTTTAGAC");
        omap.put("C", "AGACATATCCGCGGCAGTTGCTCACCGCCACAGGGGTTGGCTCCAACCCCAGTATGTACCTCATCCGTCAAAATGGTTATATCCCCAGGACAAAGGATAGAGCTCCCTGGTCAGTGCTCTTTGTACAACTGGTATACGATTGTGAAATACAATGGGGCTCGGTTTATGGCCTCGGTGAGTCCTGTTTATTGGACCCTAGGCGACTTATGCCTTTTATTATTCCGTGGCCCTGAGCAGGACGCGTTGGCGCTCAAGGCTGCTAGACTTGATCAGTGAACGGGTGATAAGCGGCTGTTCTCGGGGCTCAAGCTGATCTCTTACAGTTAACCTATAGACCCGGTTACGACAGGAAAAGTTAATCCCGATCATTGTTCTTGGAGTCTAAACACAAAAGCGTTGCAGGCTGACCTCGGGCTGATCAGAAGTGGATCACACTACTAAATAAATTGTTTCTCGAGGACATAAAATACTTGGCCATGGATGTTGACATATTAATCTATCTTAAGAAAGTAGATTGAAGTGCAAGAAAATACAGCGGGCAATTCTAACTTTTTCGAGTATACACGCCCAGACAGACTGGGACTTTCCCTTATCGTCTGGGGCGGGTAAACGTCGGATCCATAAAACGTCATTGGGCCGAGGCCCATCAGGTTAGAGACACGCTACACACTATCTAGCATGCGAGATACGAACTACCAAAGTAGGAGTGACAATATCAACCCACACTTGTTTTCTAAGAAAGCACTTACTTCGGTACGTGGGGGACTCCATAGCGCGACCACACGTGCCCTCATCTGTGTATACACGCCCAAGAGCTACAACAGGGTGGAACGTTACCATTGATGATAGCACATAGGTCGACTTGCAGTCACCTGGTCCGCATCTCTAGAGGGTCCGCTAGACCAGAGCTCCGTAATCAAAGAGCTGAGCAGGGGATAGTCGCTTCCTATGTGCAGCATGACTTCGTGCCGGCTTGACGAACACTAACTATCGTTGGAAGTGTTGAACTCATAGCTAGAGTCCTCCGCACTATGTCGCGAAGGGGAGTGCTAAAAGCGGACTTTAATGAGGGGTAGAAGCTACCATGGTGCAGAAGCGTGGATGTGGTCTAGCCTAGCTGTGCTCTAATGGTTTATTTGAGTTTGTGCGCTCCTGCTGATAGTCCCCATTTTCTTAGTACGGTGCCCAACAAGGCCATGTGAGTAGCATTCGAGGCTGCGAAAGTTAATTCGCTGGGACCAAGATGGCAGATTGAGGTGAGATGAAGGTTCTGAGGCTAGTGTGAAAGTGAGCTGCTGTAGTCGTAATGGCCTGCCAATCGATAACTACCTTATGAACCTAAAGGAACACTACACCGTATCAGCAACTCTGATTTGGAGCCCTGCTAGTCGAATGCTTCTGACCAGTCAGCTGTAGGCAGATTCGGTTTATAGGCTAAGTCTACATTAGTGCTCAATGGGTTATCTAGGCCTGGACTAAGACCACATTGAGTAACGTTGTTCCGGACCCTCACCAATACGTTCCGAGACAGAGGATCCGTACGGACAGATTATTCGTTCCCCACCGAGGTCGCATTAAAGCGTTCCTTCGTTACGATTCTACAAGAGACAGTGCGAGAGGTGGCCGATATATGTCCTTTTGCCAAACACCTAGACCTGGCGGCTTCTGAACGGGCAAGCCGCGTCGCTGCCTATAGATTCCCGAGACTCCAATGTTGCGAGATCAAATTTGTACGTGGCCCGTTGACGCTTAAAACCGGTAGTGTGGCTGGGAACGCGTTTCATACCTGTGCCCGGAAGATATGCTATTTTTTGATAACCCAGGTCGAGTCCAATCGGCGCTACTCCTTTCAACCCGCGATTGTCGGATCATGTTATGGTAACGTGAACGGCTCGACTTTGTGATCCTTGCCGAATGTACTCTTCTGTCTTGTTAAGGGATTGATATCACAACAAACAAACCCTCATCGAATCTACACATCAATAGTACGGGCTAACTGGGTATCTCACTAACCTAAGTAACTTGACAAATGTCGCCGAACCGGATACATAGAATACGCTAGTCCGAACGCTTGTTGTCCCAGGTGTATACGACTACATGCGACCACTGTTAGATCGACTCGGACTCTCCTCGTACGGTTAGACGAGCTCGTCCCGTCCGTTATTACGGCTGATCGAATAGTACTGCCTGAATTCCTCCGCTACTGTAACTCCATGTTTCACTCCTGGGTTGTCAAATTAGTCTTGGTCAACCCTGCGTCTCACTTATACCACTGTATATGCAGATTGGGCATAACTCTCAGAATAGGGATTGTTTAGGAGGAGAGTACTAGTTCTCTTAGGGGTCAGTGACAGCTATAGAATAATCGACCGGACTGCGCGTATGTTTAAACCAGACGTTCAATCATACATACGGCCCCGGTGAGGTTGAAATGTGTCTGAGAAGGCCTTAGAACTAGCAATGGATGACTTCGAAATGCCACCCTCTATTTAGTAGTTGCTATGAAATTTCTACTCTACACACTGAGGCGACTACTAGCGTACTCCCGTAGCATGCCTTCGGCCCTGTTAATCCTGTCACTCTCATTGCGACACTACGTGCGCCGCTAGCGCGAGGGACTAAAGCTTAAAGTCCTTCAGCCCTATTATTATTATGCGTCCCATCCAGGCCAAATGGAGCATATAACTTGAAAAAAGTCATCGTAGATTCGTGCAAGCAGCGACTACGCGGACGGTATTGAAGTAGCGAGCAACCTGTGAATCTAGTCGCGTCGTTGTGTCAGAATTCGTTATATTAATTCGCTGCGCCGCACCCGCATCCACGTGCATCGGTGGACTAACTAGTGTGCGTATTAGACTTTTTGTCTCCTTTTATGATAGCGGGCCTAGCCGGACTAACACATACGCAGTGATATATTACGTAGGGTGCCAAAGATCACGATGAGCCAGACAACTTCCAAGTGCTTAAGCCAATTTTGTCGTACACATAGAGAGGGCGTCTGGTCCATGAGTAGCTTACAGTCCGCAATACATAGGATCGTGATCGGGGCTCCCAAACAGCTGCGCGTATGCGGTTCGAGGAATTCTAAGGTGGGGCGCCCAGTGCCAATCCATATCAATGGTAACCAATTTAGGAACACTCAACTAACGGGCCTCGATGTCCAATAATAAGTGTCTACTTCAGCCCTGGTAGAAGGACTGAGGTGATCTCCGCTTACGCGTCGTTCGAACACGGGCCGACCTACGTATTACGTTAACCCAACCGGCGAGACTCAGGTCGGGTAAACGATTGCTCTTATATTTGACCTGGCATGCTATAGGGCGCGGCCTGCTTTGCAGGAATAGCCACTAGCGATGTGGCCAACATGTTGTAGGCTGGGTCGGACCGAAGGCCCGCGATGTTCTGGACATAGTAGGCAAGTAAGTTTACACCGCGGAGCCGTCCGGAAACACAACGCCCTTGAGCCCTCAGCACCTACAACAAGGGCGTCGTTCGCTGGGCGCAAGATTAGACAGTGTAAGGGTGCCAGTGGTAAAAACCGTTCTCCTTTGAAGTGCGCGATCGGCCTAAGTGGGGCGAGGGCTGTCAGGAAGATCCTGCAATCGTTAACGTAATACTTGCGCCATATTCGACAGTTATGGAACTGCACAGCTGTAACGTCGAATGCCGACTCTAACCCCCATCGATCTATGAAATGTCCTTTACTAAAACGGGTCGGTGGATCGCTGAAGGTCAGGGTTTCTGAATACGACATTGTACCACACCAATCGGGGCGTACCCGTAGTAAAGCTCATTTACTAGGACAAGTGGCGTTTAAATTTCGGCCACGAAAGTTGACAAAGCGGACGATGCTAGGCTTGGCCGAATTTTACCTATCTCATGGCTGGACTCAGATGTACGGATGGTCCAATGCGTCTGCCTTAAGCGCAGCAAACGGTCGTTGGTCCTTGTGGTACTTAAAACAATACTATCGCAATAAACTTCCGTTCGCCCCCACTCACTAAACTGAACCGTGCTTGCGAAGATGAGAAGTCGACTTAGGGAAGAACAACAAACTAGTAGCCGATATATGCATTATTACCCGAGCCATATCAATTCTGTTGGGAAATATCTTGGGCTGAATTCCCACTTAATTCATCTAACGAGAATGTGGGTGTTATCGGCCTTTTCATGTGCTCATTACCAAAGGCGCATGGGTTGAATCACACGAAATGGCTAGGCGAAGTGAATGGCCACGAGGTATTCCAGGTCTTCAGTGCCCCAGTCTGGTAAAATCGCCAGTTACCGTGTCCGACCTATCACGCGCGTTACCAGAGTATCGCCTACTACCAATAATGATATGCTTGTACACAGTGCTTGTGGCCGATTCGCGCCCTCGGCGTTCGTCTCCGCTATACTGCAGGTCGAGATGCCAGGTTTAGGCCCATTAGGTCAAAAGTCTTCTGTTGACCTGGCTCGGCTAAGGTATGCGGGTGCCGCCCTATTCGATTGTCGGGTGACCGATAGTTTGTTACAGCCACCCGTTCGCGTTTGCTGAACCAATAAGGTGCCCCTATGAGCCATAAGCGCCACGGTAGTAAGACACGTAGTTTAGCACGGAACTTCTCAGTTGGCGGAGTCATGTTATCGATACTCATGTCTTCAGGTGCAAGACGCGCGCCAGCTACAGCCTTTGAGATCACATTGACATCTGTTATCGATAAACTTGTAGCTAGCGCCCTACAGAATCCGCTGCTTCTGGCGATCCCTTTAACACCTACTGGTATATCCGTGCCCCATGATACCTGGCTATGGGACCCACTCTTTATTGTACAGCGGCTCAACATCCGTGAGCATAACCGTTCCAGGCGCGCCTTGTCAAGGGGACTAGTATAATGAAAGGCTAAATCTGCCACAAGCAATTAATGTAAGGTAAACGCCCAGTGTGTGACATATGTGAGTGTCGCAACTCTAATGTTCGGTAAGCCTTCGTCCGCGCGGTAGCGTGATACATTTAGTTTGGGTTTTACATTCAATACCAAATTAGTCGGGGCTACGGTATGGACGCTGATACTCTTGCAATTTGGGCTTCCGAGCAGTTCTTGGCTGTGTCATCCCCCGCGGAGCTCGACACAAATTCGTTAGGTACGGAGCGGTACAAGCCGAACAGTTGAGTGACTCAGGAATTTGTGAGACAGAAGCGCCAGGATATACCATTCCTGCATGACTCTGGCCGGGGCCGATCCTGTCTTTGTAATAGTTACCGCCAACGCTAGCGGAGAAATGAGGTTACGTGGGTCACGATACATAATGACTCTTATCCGCTGACGTAATGAGGCTCTCGTCATGGAATCAAGAAAAAGTAACCGCCTGAATGGACTACCTGGAAGTGCGGTTAGTACGGCCCGGAAGAAGCTACTGTGCCAGTTTAATTCATCATAATTAGGGGTCCGAATCATCCTCATACAGATTGTGAACGCACCATTATCATTTGAATTAACAGGATAACGTCTGTCATCATTACTCCCCGAGAAGGTGATCTAATCTGACACATGACGCTTGCCGGCAAGTTCTACCCCCAGGAAGACCTTACTTATAAGCTATGGTGGGGCACCCTGAGCCTGATAGACCGGTGAAGCATGTAGTTTAGTATACGGACTTTCAAACACTCGACGGTACAAGCCGCTATAGGCGGGTGACAAACGATTGAGCCCCTAAACTCATTTAGTTCGCGGTGTTTCGGGACGGGTGGCATTCGAGTTCGTGGTAGGTCTAAACAACCTAAAGTTCATGACTACGTAGGAGGGGGTGGAGCCTCGGTTACTAGTTTTTGGGAAGATTGAAATTATGCTGGCAGACGAAGCCAGCGGTTGATACGACAAAATGTCTTCGACCTTTAATGAGTAATACGATTCCTGATTTCGGCCCTACCGAGAACTGAGTCTATATCAGGTTGGATACAAAATATGCAAAGACGATGTCCTCTTGTAATGGCTTGGGTCAAGACCGACTGGCACTCCTATACAGGTACATACCCGAAGAGTGGGTTCTGGTCTTGCACGAGTCTTAGATCCCCGTATCCTGACTTACTAACAGACTTTCCCATAACAACCTGTCTTGAATTACTAGAGGCCGCGGGTCTTCCACTACCCGGTCACGCGATGATGGTAGTCGCGTTGCGCTCGGTGAAATCCGGACCATTCACTCTTATTGAATAGCTAACCTTTACCGCGACGACACGAAGTCCGGGCTGCACAGTGGTCGTGGAGGGTAAACAGTTAACTCGGATTGACCAGCTCAGACCCACCCTCTGCCTCTCTGACGTTCCAGGGTTGTATGTGGCCGAAATGTGACACGCTCTGCAAGCGATCAAGTATATTATGCGAGGGGCTCATCCGGCCAGAAGGACCGCAGTATAGCTGCAATAGTTCTAAACAACTGATCGGAGCCGTGTTGAATTTTGATGAGTTGAAAATACCCGACTAAGCTTGCTTATCGCTTGTCGATGAGTAGTAGGGACTTTCCATGCACATTCCACATCAGTGCACGACTAAGACAACTCCCTGACGGGATCAATACATTTAATACTCGCGCTCTGTCATTCCCAATGACCTTTGCTCCTGTTAATTGCTGGTCGTGGTTAGGGTACCTGATAGGGCCGCTTGCGGGACGGATTCCATCAGGATCCAACCGACAAGACGCCGTAACGTCAATAAGGAAACGCGGACTTGCTGGCGCCGCTCTTCACGTTTAAGAGGCCCGTCAGTGCCAATTCGTCTAAAGGTGGTCGTCTCTGTTACTGGCCTAAGCATACTGAGCTCATCAAGCGTAGCTCCGGACAGACGGATGATGTGGATGATCACATGTCACCTACAGCACAGTCCACCATTGGCCCGTTCTCTTCGAACGATATGTCGCCAGAACTAAGAATGATTTACTTAGGGTAAACGCGAAAGCTCGTTGTACTGTTTGAGCGCTAAGCCCTTCGCTGCACATGCAACGCACGGTGTAGACGGTTGGAGATATAACCCTGCTTTGCTCTAGGGCGAGCCGTGACCTCCAGAGTCTCCCGCTAGCGTTTGGCACGCGACGCTCAGGGTACGTGCTGCTCAACCCGCTCCCATGACGAGTTTAGTGTCTCACGAGTACCGTATTTGTGGGTAACGGACCCATAGTTATAGGGGAGGGATTTCTAGTGGCAGGCTCCTTGTCCCTCAATTTTGGATAGCAAGCACATGGTGCAGAGTTATCTTTATGGTGTTGACGAAAACCGGCCCGCAGCCTTCTTGCCTTATGAAAGTTCCTTGGGAATGGGATGAAGTACTACCTCTTTCCAAACGCGCACAACTGATATCTATTATGCTGTCAGACCGTAGCTGCGCTGCGCGAACATCCAAGTTGGAGGTTACCTCGTGCGTCAACGGGATCCAGGACGCCGCTATGATTCATACCTTATATACGAGGCTCGTGATCGTGATGATGTACCAATATCCAGATAAGTCCTTAGCTTAGAGCCCTTTTGGACCTTTTCGACTGGCTCTTTCTTCACCGTCTTACCCCTGACGATTAATTCGACACGGGAACACCACCCCGGTCTTAAATAGTGGGCGCCACTTAACGATCTAGCGAGGCTGCCGGAGTGCCATCCTGCCGGCTTCTTAATCCGGCCGGGCCTAGAGCGCCGAATCCGCCGGATGTAGATAGGACCTACGGGCATCGTACTGCTTTGCTATCGCGGTATCGTGCCTGATCGCCCAAGGGTGAATTATTCATGTCTCCCGATGTTACAGACGTCAGCTCTAAGCACATCATGTATAGCAATGCCGGGATCTAGAAGTCTTGAATGGACAGCCACACTCAGGAGTTACGGTAATCATGTACAGTAGCTAGACCGCGCCTGTGGATTGTCACCGCCCACTTGTTTCACCAACCTACCCGCCAGTAGAACTGGTGTTGTCTCGTCTGAGCAAGGTAACGCTAAAACAATTCGACTTAGAAAGTTCTCCCCGAATCGCATCGTTGGATTTACCATGGGCACCGGGTACGTTGTACTGACCTCGTGTTCTTCAGACAAGAGTGAGCCGTGTGGCCCGCGAGAGGATGTACTCACGTCTAGGTCGACGGACCCTGATAACGAGCAAAAGTCCTATGTAACCCGCCCATAAAATGATCACCGTAATCATTCTGGCCTGTGTAATCATAATTTTCTGCAGCGGAGGCATCCCGGTGTACGCGGTGACCCCAAAATGAGTGGCCCCACATGCTCTGAACGGACAATGGTCCAGTGAGGCGCAGCCTTTTCAAGATGCCTTAACCTCCAGTCTTGCGATCTACTGGTCTACGACCTTTAACTGAAAATCCGATCCGCTCGGGGCGACTCTAAGAAGCACTGGGAGGCCTGTGTCAATGCGTGGCCTACTCGAGCGGAACGGTTAAGTCTCGACGTTGGTTATACTTAAATAGTACACAAGTGCAATCGACTACTTACACGTAGCTAACGGATATCACAGGTGTGGCGCAACGATACCACGCCTCTTAAAGCTTGTATTCTCTAACGTACCGTGCTGTTTAGGTTTTACATATTTGCACAAGCGGCTCGAGAACTATATGCGAAATTTTAAACCATTATGGGGGTAGACCATACATCAACGTTGACATAATAACGGGCCGGCTCTTTACATGTGGTCTCAAGGTGCAGCGGACACCAGCCATGTGGACATATGAAGCGCCTATCTTTGTCTTAACACTACTGTCGAATTATCAAGTATCGCTAGTTGGGAATGGACGATGAAAATGATCTTGGTTTTTAAAGCTAACAACATTTCGATCTAAAACTATGGACGGAATACTTGCCCGGTAACTTGCGAACGAGATAGTGATATGTGGCGTAATGAAAACCATCGAAGGACCTCAGTGAAAGTAACGATACCTTGAATAGTTGCTCACCGTTTTCGACCGTCTGACTGTCGGATGACCAACACGGGCGTTCCTTCTACAACTAGCCGCAATGTCTCAGCGTCGGGTTATGTCTAAGCCAATACGGAGATACTGATCGATCGTCAAGGCCATGTCCGGGACGCGTATGACTATCAGGACGTACCCTGTAAAGCTTGGCACGCAAGTATCAACTTTTTATGTTAAATCATGAGGGTAGCCTCGACCGTTCTGCGCTCTCAGCTTATATGCAAACGCCTTACGAAAGTCGGACGATTGGCTGTGCGTAAGGCACATTGTCCGTGATCCGCTATGGATTTTTTGACGGGGCATTATCATGCCCATTTTTACATACTGACGCAGTGTCCAAGTGTGCGACCGGAGTCATCACGGCGGATCCCGTATCTATAACCGGGAAGATAATCTATTGGCTGGTCAGGGTGAGTCACTGGCCCTTTCTGGGCCTTTGGCCCATCGCGACTGGGGTGACAGCTCTGCGCAAGTCTCGTTAAACTTTGATCTCAATTGATATATAAAAAGGCAATGCTAAGTAGGGATTGTCATGTTTTATGCTAGTTACTAAGCCACTCGGCCAGGGTTCGAGTAAGACCATACACTGGTCCCAAATAATCCATCCTTACTAGTGTGCATCCTATATACACTTCGTGAGCATTTAGAAACACGAGCCCCCAAGCTAGGGCACAATATCGTATCTTGGTATTCGACTACGTTCACGGGTCTGTACCGCGTCATACACTCCTCCCGCAACCGGCTTAAGCTAGGAACCAACAATTGAGATAAATTGACCTAGGCAAGACAGTCGGACGAATGTCAGATTCAATCATTTTCGTAATCGAAAATACCATGTGCGAACACATTTCACACATTCACTACGTTGTCGTTGTTTGTCTCTAACTTACTTTGTGAGTTATGAAGAGTGAGGTAACAATGCCAAAACCGAACCAACTCGTCATGGATCCAGTGCGTGAAAAGTAGTGGGACGGAGGGAGCCCACTAGCTTTTAAAGAAGCTATGTATAAGCAGCTTCCGTCTAGGGTTCTCGCGTTTTCTGTAGCAACCTTCAGAC");
        omap.put("G", "AGACATCTACGCTACAGTTGCTCACCGCCACAGGGGTTGGGTCCAACCCCAGTATGTACCTCATTCGGCAATATGGGTATAACCCCAGGACCAGGGATAGTCCTCTCTGGTCAGAGCCCTTTGTACAACTTGTATTCGATTGTTAAATACAATGGAGCTCGCTTTATGGCCTCGGTGAGTCATTTTTATTCGACCCTAGGCGACTCATTCACTTTATTATAACGTGGCCCAGAGCAGTACGCGTTGGCGCTCAAGGCGGCTAAACTTGCTCTGTGCACGGGTGATAAACGGCTGTTCTCGGGGCTGAAGCTGATCTCTTAGAGTTAACCTATAAACCCGGTTACGACGGGAAAAGTTATTCCCGATCATTGTTCTTGGACTCTAAACACAAAAACGATGCAGGCTGACATCGCGCTGAGCAGAAATGGAGCACAATACTAAGTAAATTGTTTCTCGAGGACGTAAAACACTTGGCCATGGATGTTGCCATATTAATCTATCTTAAGAAAGTAGATGGAAGTGCGAGAAGATACTGCAGGCAATTCTCCCTCTTTCGAGTAGTCACGCCCAGATAGACTGGGACTCTCCCTTATGGTCTGGCCCGGACTAACGTCGGATCCATAAAACGTCATTTGGCCCAGGCCCAGCAGGTTAGAGACACTCTACACACTAACTTGCATGCGAGATCGGAACTGCCAAACTAGGGGTGACAATATCACCCCACATTTGTATTCTAAGAAAGTACTTACTTCGCTAGGTGGGGGCCTCCATAGCGCGACCACACTTGCCCTCATCTGGGTATACACGCCCACGAGCTACAACAGGGTGGAACGTTGCCATTGAAGATAGCACATAGGTCGACTTGCAGTGACCTTGTGTGCATTTCTAGAGGGTACGCGAGACCAGAGCTCCGTAATCAAAGAGCTGAGCAGAGGATAGCCGGTTCCTATGTGCAGAATGAGTTCGTGCCTGCTTGAGGATCATTAACTATAGTTGGCAGTATTCAACAGATAGCTAGATTCCTCCGCAGTATGTGGCGATGGGGAGTGTTAAATGCGGACTTTATTGGTGTGTAGATGCTGACATAGAGCTGAAGCGTGGATTTCGTCTAGCCTAGCTGTGCTCTAATGATTCAGTTGCGTGTGTGCGCTCCTGCGGATGGTCCCCGTTGTCTTAGTACTGCGCCCAACCAGACGAAGTGAGTAGCAATCGATCCTGCGAAAGTAAATTCGCTGTGACCAAGATGGCAGATTGTGGTAACATAAAGGTTCCGATGATAGTGTGCCAGTGAGCTGCTCTTGTCGTAATGGGCTGCCAATCGTCAACTACGCTCAGAACCAAAAAGGGGATTACTTAGTATCAGCAACTCTGATATCGAGCCCTGCAAGTCGAATTCTTGTGAACCGTCAGCTGTAGGCAGAATCGGTTTATAGGCTGAGTCTACATTAGTGCTGACTGGTTAATGTAGGCCTGGCCTAAGACCACAGTGATTAACGTTGCTCTGGCCCCACAACAATCCGTACCGAGACAGAGGATCCGTACGGACAGATTATTCATTCCCCTCCGCGGTCGCATAAAAGCGTTCCTTTGTGACGATTCTACAAGAGCCAGTGCGTGAGGTGGCCGATATATGTCCTTTTGCCAAACACCTTGACCTTGCGGTCTCTGAACGGGAAGGCCGCGTTGCTGCCTATAGATTCACGAGACTCCAGTCTTGCGAGATCAAATCTGTACGTTGCCCGTTGACGTTTAAAACCGCTAGTGAGCCCGGGAACGCGCCTCATAGCTGTGCCCCGAAGATATGGTATTTTTTAATATCCCAGGTCGCGTCCAATCGTCGCTACTCCTCTCAACCGGCGATTGCGGGATCATGTTATCGTAACGTGAACGGCTCGACTTTGTGATCCAGGCTTAATGTACTCTTCAGTCCTGTTAAGGGCTTCAAAGCACAACAATCAAACCCTCATCGATTCTTCACATCAGAACTACGGGCCATCTGGGTATCTCACTAACCTAAGTGACTTGACAAATGTCTCCGAACCGGATACATAGAATGCGCTAGTCCGAAAGCAGGTTATCCCAGGTGTAGACGACTACATTCGACCACTGATAGCTCGACTCAGACTCCCCTCCTACAGTTGGGGGAGGTCGTCCAGTCCGTTATTATGTCTGATCGAATAGTACTGCCTGAATTCCTCCGCTCCTTTAACTCCCTTTTTCACTCCGGAGTTGTCAAATTAGTGTTCGCCCACCCTGGGTCTCACTTATACCACTGTTTATGCTGATTGGGCATAAATCTCAGAATAGGGATTGTTTGGGAGGAGAGTACTACTTCACTTAGGGGTCAGTGAATGCAATAGAACAATCTACCGGACAGCTCGTATGTTTAAACCTGACGTTCACTCAAACATATGGCCCCGGTAAGGTTGGACTGGGCTTTAAAAGACCTTAGAACTAGCAATGTCTGACTTCGAAATGCCAGCCTCTATTTAGTCGTTACTATGAAATTTGTACACTACACACTGAGTGCACTCAAAGCGTACACCCGTAGCATGCCTTCGACCCAGTGAATCTTGTCACTCTAATTGCGACAATACGAGTGCCGCTAGCGCGACGGAGTAAAGCTTGAAGTCCTTCAGCCCTACAATTATTACGCGTCCCATCCATGCCAAATGGCACAGATAACTTGACTAAACTCATCATAGATTCGTGCAAGGTGCCACTAGGCGGGCGCTATTAAAGTAGCGAGCATCCTGTGAATCTATTCGCGTCCTTGTGTCAGAATTCGTTATAAAAATCCACTGCTCCGCACCCGCATCAACGTCAATCGGTGGACTACCTAGTGTGCGTAATAGACTTTTTTTGTCCTTTTATGATAGCGGTCCTAGGCGGACTAACACATACACTGTGATATATTACGTAGGGTGCCAAAGATCACGTTGAGCCAGACAACCTCCCAGTGCTTATGCCTCCTTCGTCGTACATAAAGAGAGGGCGTCTGGTCCATGAATAGCTTACTGTATGCAATACATAGGATCGTGATCGGGGCTCCCAAACAGCCGCGCGTTTGCCGTTCCTGGAATTCTAAGGTGGGGCCCCTAGTACCGGTGCATATCAATGGTCACCAATATAGGAACACTCAACTAACGGGCGTCGGTGTCCAATAATAAGTGTCTACTTTAGCCCTGGTAGAAGGACGGAGGTGATATCCGCTTAGGCGTCGATCGAACACTGGCGGACCTACGTATTCCGTGAGACCAACGAGCGAGACTCAGGTCGGGCTAACGATTGCGCTCATATTGGAACTGGCCTGCTTTAGGGCGCGGCCTGCCTTGCAGGAAGTGCCACTAGCGATGTGGCCAACATGTTTTAGGCTGGGTCGGACCGAAGGCCCGCGATTATCTCGACATAGTAGGAAGGTAAGTTGACACCGCGGGGCCGTTTGGAAACCCAACGCCCTTGCGCCCTCGGCTTCTACAACAAGCGCGTCGTACGCTGGGCACAAGTTTAGATAGTGTATGGGTGCCAGAGGCAAAGACCGTTCTGCTTGGTATTGCGCGATCGGCCTAACTGGGGCGAGGGCTGTCAGGAAGATATTGCAATCGTGAACGTACTAGTCGAGGCATAATCGCCAGTTATGAAACTGCACTGCTGTAATGTCGAATGCGGACTCTAACCGCCCTCGATCTATGAGATGTCCCTTACTAAAACGGGTGTGTGGATCGCTGCAGGTCAGAGTTCCTGAGTAGGACAATGTACCAAAACAATCGGGGCGTTCCCGTAGTAAAGCTCATTTACCTGGACATGTGGCGTTTCAATTTCGGCCACGAAAGATGATGAACATGACGACGCTAGGCATGGCCGAAATTCACCTATTCCATGGCTGGACTCCGATCTACGGATGGTCCAATGCGACTACCCTAAGCGCAGCAGACGGTCGTTCGTCCTTGTGGTATTGAAAACGATACCATCGCCATAAACTGCCGTTCACCCCCACTAACTAAGCTGAGCCGTCCTGCCGAAGATGAAAACTCGACTTAGGGAAGAACAAAAAACTAGCAGCCGATATATGCATTATTACCCGAGCCTTATCAATTCTGTTGGGAAATATATTGGGCCGAAGTCCCACTTAAGGCACCTAACGAGAATGAGGGTGTTATCGGCCTTGTCGGGTGCTCATGACCAAAGTCGCATGGATTGAAGCACTCGAAACGGCTAAGCGAAGTGAATGGCCACGATGTATTCCAGATCTTCAGTGCCCCAGTCTGGTAAAATCGCCAGTTACCGTGTTCGACCTATCACGCGCGTTACCAGTGTCTCTCCTACTACCAATAATGAAATGCTTGTACACAGTGCTTGTATCTGATTCGGGCCCTCGGCATGCCTCTACACTATACTGGAGGTCGAGTTACCAGGTGTAGGCCCAAGAGGTCAAAGGTCTTCTTTCGACCTGAGTCGGGTAAGGTATGCTGGGGACGGCCTATTCGGTTGTCGGGTGACCAATATTATGTGATAGCCATCCTGTCGCGTTTGCTGAACCAATAGGGTGCTCCTATGAGCAATAAGCGCGACGGCAGTAAGACACTTAGTTTAGCACGGAACTTCTCAGCTGGCGGAGACATGTTATCGATACACATGTCTTCAGGGGCAAGACGCGCTCCAGCTACAGCCTTTGAGATCACATTCACATCTGTTATCGATAAACTTGTAGCTAGCGCCCTACAGAATCCGCTGCTTCTGGTGATCCCTTTAAGACCTAAGGGTATATTCGTGCCCCATGATACCTAGCTATGGGACCCAATCTCTATTGTACAGCGACTCAACGTTGGTGTACATCACCAATCCATGCGCGCCTTGTCAAGGGGACTAGTTTAATGAGCGGCTAAATCTGCCACAAGCAATTAATGTTTGATAAACGCCCAGTGTGTGACCTATGTGAGTGTCGCAACCCTAATGTTCGGTAAGTCTTCGTCCGCGCGGTAGCGTGATACAATTAGTTTGGGTTCTACATACAATACCAAAGTAGTCGGGGCTACGGTATGGACGCTCATACTCCTGCAACATGGGCTACCGAGCAGTTCTAGGCTGTGTCATCCCCGGCGGGGCTCGACACAAATTCGTCACGGACGAAGCTGTACAAGCCGAACAGTTGAGTGACTCGGTAATTTATGAGACAGAAGCGCCAGTATATAGGTACCGTGCATGACTCTGGCCTGGGCCGAGCATGTCTTTGTAATACTTACCGCCAACTCTAGCGGAGAAATGAGGTAACGTGGTTCACGATACATAATCACTCTTTTTCGCTGACGTAATGAGGCTCGCGTCATGGAATCAAGAAAATGTAACAACCTGATAGGACTACGTGGAAGTGCCGTTAATAGGGCCCGGAAGAACCTACTGTGTCAGTTTTATCCAGCTAAATTAGACGTGCGAATCAATCTCATATAAATTGCGAACGCCCCAGTTTCATTTCTACTAACAGAATAGCGTTTGTCATCATTACTCACTAAGAAGGTGATCTAATGTGGAACACCACCGTTCTAGGGAAGTTCTACCCCCAGGGCGATCTTGCTAATAAGTGATGGTTGGGCACCCTGGGCCTGCTAGACGGGTGAAGCGTGGAGTTGATTATACGTACTTTCCAACTTTCGAAGTCATAAGTCGCTTTAGGCGCTTGACAAATGAATGAGCACCCCAATTCAGTTGGGTCTCGGTGTTTCGGGACGGGTTACCTTCGAGTTCGTGGTAGGTCCGAAAACCCTAAAGTTTATGGATACGAAGAAGGGAATGGAGCCTATGGTAATACTTTTTGAGAAGATTGTAATTATGCAGTCAGACGAGGCGAGCGTTTGATATGTCATCGTGTCTTCGACTTTTAGTGAGCAATAGGATTCCTGATTTCGACCCTAGTATGAACTGTGTCTATATCAGGTGGGATACAAAATATGCAAAGACGATGGTCTCTTGTAATGTCTTGGGTCAAGACCTACTGGCACCCGTATACAGGTACATACCCGAACAGTGGGTTCTGGTCTTGCCCGAGGCTTGTATCCCCGTAGCCTGCCTAACTAACAGACTTTCCGATAACCACCGATCTTGAATTACTAGAGGCCGCGGGTCTTTCACTACTCGGTCAGGCGATGATGGTAGTCCCGTTGCGCTCGGTGAAATCAGGACCATTCACTCGTATTGAAAAGCTAACCTTTATCGCGGCGACACGAAGTACTGGCTGCACAGTGGTAGTGGAGTGTAAACAGGTATCTCGTATTGACCGGCTCAGACTCACCCTCGGCCGCCGTGACTTTCCAGGTTTGCATGTGCCCGATATGTGCTACGGTCTCCAAGGGACGAAGAGAAATATGCGACGGGCTCGTCCGGCCATAAGGACCGCAGTATAGCTACAATTGATCTAATAACCTGATCGGAGCCGTGTTGAATTTGGACGAGTCGAAAATCCCCGACTAAGCTTGCTTATCGTTTGTCGCTTAATACTAGGGACTTTTCATGCACATTCTACATCAGTGCACGACTAAGACTACTCCCTGACTGGATCAATACATTTAATACTCGTGCTCTGTCATTGCCAATGGCCTTTGCTCCTGTCAATTGCTGGTCGCGGATCAGAAACCAGATCGGGCCGCTTGCGGGACGGGGCCCATCAGGATCCAAGTGACAAGAAGCCGTAACCGCACTAAGGATAAGAGCACTTGCAGGCGCCGCTCTTAACGTTCTATAAGCCCGTCAGTGCCAATTGGTTTAAAGGTGGGCGGCTCCGTTACAGGACTAAGCATCTCAAGCTCTGCAAGCGATGCTCCGGATCGACGGATCATTTGGATGATTACATTTCTCCTAGAGCACAGTCAACCATTAGCCAGTTCTCCTCGACCGATAAGCCTCCAGAACTATGACTGATTTTGGTAGATTAAGCGCCAACGCTATTTGTATTGTTTGAGCGCTTAGCCACTCCGTGTACATGGAACGAACGTTGTATTCCGCTGAAGACAAAAACCTGCGTTCATCTAGGGCTAGCCGTGACCTCCAGAGACGGACGCTAGCTTTTGTCAAGCGAAGCGCACGGTCGGTGCTGCTACACACGCTCCAATGACAAATTGATTGTGTCTCTAGTGCCGTTTTTGTGGGTCCCCGACCCTTATTTATAGGGGTGGGAATTCTCGTGCCAGGCTCCTTGTCTATCAATTTTGGAGAGCAAGCTAATGATGCAGACCTATCTTTGTGGTGCTGACGAAAACAGCCGCGCCGCCTTCTTCATTTATGTAAGTTCCTTGGGAATGGGATAGAGTAGTAGCTCTTTCCAGACAAGAAGCAATGATATAAATTATGCTGTCAGTACTTATCTGCACGACGCGAACTTCCAATTAGAAGCTTACCGCGTGCGATAATGGGATTCAAGACACTGCTATACTTCATACTTTATATACGAAGCTCGTGGTCATGATGATGTTCCAGTATCCATATAAACCCTGGGCTTAGAGCCCTTTTAGACAATTTCGACTGGCTATTTCTTCTCTGTCTTACCCCTGACGACTAATTCGACATGGGTACTCCACCCCCGTCTTAAATAGTAGGCGCCACTAAAGGATATCGCGAGGCTCCCGGAGTGCCATCCTCCCGACTCCTAAATCCGGCCCGTCCTAGAACGCCGAATTCTCCAAATGTAGATACGAACTCCGGTCATCGTGGAGGTTGGTTATCCCCGCCTCGCGCCTGTTCGCCAACGTGCGAGTTATTAAAGTCTTTCGAGGTCATAGACCTCATATGGAAGCATATCATCAATAGCATGCCCCGGATATAAAAGTCTTAAAGGGACATCCACCCTCAGGAGTTCCGCCAATCATGTACAGTAGCTGGATCGCTACTATGGATTGTCTCCGCCCCCATGTTTCACGAACCGACCCCCCAGTGGGACTGGTTTCGTCTCATCCGTGTAAGGTATCGGTAAAACAACTCGACTTAGAAAGTTCTCCTCGAATCGAATCGTTGGATTTCCCATGGGGATAGGGTACGTTGTACTGACCTCGTGTTCTACCGACAACCGTAAGCCGTGTGTCCCCCGAGAGAAAGTACTCAGGTCTAGGTCGACGTGCCCCGTTAACGAGCAAAAGCCTTACGTAACACGCCCATACAACAATCACCGTAATCATCCTGGGCTGTGTGATCATAATTTTCTAGTGCGCTGGCTGCCCGATGTACGCGGTGGCCCCAAAATGAGTGGGCCCACATGCTCTGGACGGACAATCGTCCACTGAGGCGCAGGCTCTTTAACATCCTTTAACCTCAAGTCTTGCGTTCTATTCGTCTACGACCTTTGAATGAAAATCCGCTACGCACTGGGCGACTCTGAGGAGCAGTCGGAGGCATGTGTCAATGCGTGGCCTACTCTAGCGGAATGGTTAAGTCTCGAAGTTGGTTATACTTAAATGGTACACAAGTGCAATTGCCTACTTACATGTAGCTAATGGATATCACAGGTGTGGCGCAACACTACCTCGCCTCTTACAGCTGGTACTCTTGAACGTACCGTGCTATTTCGGTTTTTTATATTTGCACAAACGGCTCCAGAACTATATGCTTCATGTTAAACCATTATGGGGGTCGACCGTACATCAACGTTAACATAATAACGGGACGGGTCTTCAGATATGGTCTCAAGGTGTAGCCGACACCATCCATGTGGACATAATTGGCGGCTGTCTTTGTCCTATCACAACTGTCGAATTAGCAACTATCTCTCGTTCGGAATGGAGGATGAAAATGATGTTGGTTTTTATAGCTAGCAACATTTCGCTCTAAAACTATGGACAGAATCCTTGCACTGTAAATTGGGCACGAGACAGTGATATGTGGCGTAATGAAAACCAGCGAAGGACCTCAGTGAAAGTAACGATACCTTGAAAAGTGGATCACCGTTTTCGACCGTCTGACTAACGGATGACGAACACGGGCGTTCCTTCTACAACTTGCCGCAATGTCTCAGCCTCGGGACATGTCTAAGACAATACGGAGATACTCATCGATCGTCAAGGCCATGGCCGGGACGCTTATAACTAACACTACGTACCCCTTAAAGCTCGTCACGCATGTATCAACTTTTTATGTTAAAGCATGAGGCATGCCTCGAGCGGTCTGCGCGCTCAGCTTATAAGGAAACGCCTTACAAAACTAGTACGATTCGGTGTGTCTAAGGCGCATTGTCTATGCTCCGCTATGGATTTTTTGAAGGGGCTTTATGTTACCCATTCTTACATACTGACGCAGTGTCCAAGTGTGCGACGGGTGCCATCACGGCGGATTCCGTATCTATAACCGGGACGATACTCTATTGGCTGCTCAGGGTGAGTCACTGGCCCTTTCTGGGCCTTTGGTCCATCGCGGCTGGGGTGGCAGCGCCGCTCAAATCCGGTTGAGCTTTGCTTTCGATTGATATCTGAAAAGGCAATGCTAAGTTGTGATTGCCATATAGTATGCTAGTTACAAAGCCACTTGACCAGGGTTGGATTAAGACCATACACTGGTCACAAAAAATGGATCCTTACTAGTGTGCAGCCTATATACAGTTCGTGAGCATTTCGATACACGAGCCCCCAACCTAGGGCACAACTTTGTATTTTGGTATTCGACCACGTTTACTGGTCTGAACCGGGTCAGACAATCCTCCCACAACAGACTTGAGCAAGGTATCAACAATTGATATAAATAGTCCTTGGCAACCCAGTCGGACCAACGTCAGATTCAAATATTTTCGTAATCGAGAATACGATGTGCGCACTCATTTCACACATTCACTATGTTGTGGGAACTTGCCTCTAACTTACTTTGTGGGTTCTGAAGAGTAAGGCAACAAAGCCATAACCGATCCAACTCGTCATGGATCCAGTGCGTGAAACGTAGTGGCACGGAGGGAGCCCACTAACTCTTAAAGAAGCTATGTACAAGCAGTTTCCGTCTAGCGCTCACGTGTTTCCTGTAGAAAGCTTCAGAC");

        Alignment aln = new Alignment(omap);
        Frequencies freq = new Frequencies(aln, false);
        SubstitutionModel jc = new JukesCantor(freq);
        SiteModel siteModel = new SiteModel(jc); // NOTE: sitemodel gets its mutation rate from Util

        HiddenState firstState = hmm.states.get(0);
        BeagleStateLikelihood beagle = new BeagleStateLikelihood(aln, firstState, siteModel, null);
        System.out.println(Arrays.toString(beagle.calculateLogP()));

        HiddenState secondState = hmm.states.get(1);
        BeagleStateLikelihood beagle2 = new BeagleStateLikelihood(aln, secondState, siteModel, null);
        System.out.println(Arrays.toString(beagle2.calculateLogP()));

        HiddenState thirdState = hmm.states.get(2);
        BeagleStateLikelihood beagle3 = new BeagleStateLikelihood(aln, thirdState, siteModel, null);
        System.out.println(Arrays.toString(beagle3.calculateLogP()));

        HiddenState fourthState = hmm.states.get(3);
        BeagleStateLikelihood beagle4 = new BeagleStateLikelihood(aln, fourthState, siteModel, null);
        System.out.println(Arrays.toString(beagle4.calculateLogP()));
    }

    public double[][] getAInOrder() {
        String[] names = new String[] {"HC1", "HC2", "HG", "CG"};
        Map<String, Integer> name2Index = new HashMap<>();
        for (String name:names) {
            for (int i = 0; i < states.size(); i++) {
                if (states.get(i).getStateName().equals(name)) {
                    name2Index.put(name, i);
                }
            }
        }
        double[][] orderedA = new double[states.size()][states.size()];
        for (int i = 0; i < names.length; i++) {
            for (int j = 0; j < names.length; j++) {
                orderedA[i][j] = a[name2Index.get(names[i])][name2Index.get(names[j])];
            }
        }
        return orderedA;
    }

    public double[] getPiInOrder() {
        String[] names = new String[] {"HC1", "HC2", "HG", "CG"};
        Map<String, Integer> name2Index = new HashMap<>();
        for (String name:names) {
            for (int i = 0; i < states.size(); i++) {
                if (states.get(i).getStateName().equals(name)) {
                    name2Index.put(name, i);
                }
            }
        }

        double[] orderedPi = new double[states.size()];
        for (int i = 0; i < names.length; i++) {
            orderedPi[i] = pi[name2Index.get(names[i])];
        }

        return orderedPi;
    }

    public List<HiddenState> getStatesInOrder() {
        List<HiddenState> lst = new ArrayList<>();
        String[] names = new String[] {"HC1", "HC2", "HG", "CG"};
        for (String name:names) {
            for (HiddenState state : states) {
                if (state.getStateName().equals(name)) {
                    lst.add(state);
                }
            }
        }
        return lst;
    }

    /**
     * Run backward algorithm. Prepare for posterior decoding.
     * @return backward matrix
     */
    private double[][] buildBackwardMatrix() {
        int numStates = getNumStates();

        // Calculate likelihood of entire sequence alignment for all states, keep pattern log likelihoods for each state
        Alignment alignment = Utils.DATA;
        Frequencies freq = new Frequencies(alignment, false);
        SubstitutionModel substModel = new JukesCantor(freq);
        SiteModel siteModel = new SiteModel(substModel); // NOTE: sitemodel gets its mutation rate from Util
        Map<Integer, double[]> state2PatternLikelihood = new HashMap<>();
        for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
            HiddenState state = states.get(stateIdx);
            BeagleStateLikelihood beagle = new BeagleStateLikelihood(alignment, state, siteModel, null);
            double[] patternLogLikelihoods = beagle.calculateLogP();
            state2PatternLikelihood.put(stateIdx, patternLogLikelihoods);
        }
        int L = alignment.getSiteCount();

        double[][] backwardMatrix = new double[numStates][L];
        // Initialize dp matrix
        for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
            backwardMatrix[stateIdx][L - 1] = StrictMath.log(1);
        }
        // Backward computation
        for (int siteIdx = L - 2; siteIdx >= 0; siteIdx--) {
            for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
                double[] terms = new double[numStates];
                for (int nextStateIdx = 0; nextStateIdx < numStates; nextStateIdx++) {
                    double logEmissionProbNext = state2PatternLikelihood.get(nextStateIdx)[alignment.getPatternIndex(siteIdx + 1)];
                    terms[nextStateIdx] = a[stateIdx][nextStateIdx] + logEmissionProbNext + backwardMatrix[nextStateIdx][siteIdx + 1];
                }
                backwardMatrix[stateIdx][siteIdx] = logSumOfExponentials(terms);
            }
        }
        return backwardMatrix;
    }

    /**
     * Posterior decoding to find the posterior probability for each state at each site.
     * @return posterior probability matrix
     */
    public double[][] posteriorDecoding() {
        int numStates = getNumStates();
        double[] OLD = new double[numStates];
        double[] NEW = new double[numStates];

        // Calculate likelihood of entire sequence alignment for all states, keep pattern log likelihoods for each state
        Alignment alignment = Utils.DATA;
        Frequencies freq = new Frequencies(alignment, false);
        SubstitutionModel substModel = new JukesCantor(freq);
        SiteModel siteModel = new SiteModel(substModel); // NOTE: sitemodel gets its mutation rate from Util
        Map<Integer, double[]> state2PatternLikelihood = new HashMap<>();
        for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
            HiddenState state = states.get(stateIdx);
            BeagleStateLikelihood beagle = new BeagleStateLikelihood(alignment, state, siteModel, null);
            double[] patternLogLikelihoods = beagle.calculateLogP();
            state2PatternLikelihood.put(stateIdx, patternLogLikelihoods);
        }

        int L = alignment.getSiteCount();
        double logLikelihood = logLikelihood();
        double[][] backwardMatrix = buildBackwardMatrix();
        double[][] posteriorMatrix = new double[numStates][L];
        // Initialize forward dp matrix
        for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
            double logEmissionProb = state2PatternLikelihood.get(stateIdx)[alignment.getPatternIndex(0)];
            double matrixEntry = pi[stateIdx] + logEmissionProb;
            OLD[stateIdx] = matrixEntry;
            posteriorMatrix[stateIdx][0] = OLD[stateIdx] + backwardMatrix[stateIdx][0] - logLikelihood;
        }
        // Posterior decoding in the process of forward computation
        for (int siteIdx = 1; siteIdx < alignment.getSiteCount(); siteIdx++) {
            for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
                double logEmissionProb = state2PatternLikelihood.get(stateIdx)[alignment.getPatternIndex(siteIdx)];
                double[] terms = new double[numStates];
                for (int prevStateIdx = 0; prevStateIdx < numStates; prevStateIdx++) {
                    terms[prevStateIdx] = OLD[prevStateIdx] + a[prevStateIdx][stateIdx] + logEmissionProb;
                }
                NEW[stateIdx] = logSumOfExponentials(terms);
                posteriorMatrix[stateIdx][siteIdx] = NEW[stateIdx] + backwardMatrix[stateIdx][siteIdx] - logLikelihood;
            }
            // exchange old and new
            double[] temp = OLD;
            OLD = NEW;
            NEW = temp;
        }
        // Exponentiation on log posterior probability
        for (int siteIdx = 0; siteIdx < L; siteIdx++) {
            for (int stateIdx = 0; stateIdx < numStates; stateIdx++) {
                posteriorMatrix[stateIdx][siteIdx] = StrictMath.exp(posteriorMatrix[stateIdx][siteIdx]);
            }
        }
        return posteriorMatrix;
    }


    public static void main(String[] args) {
        try {
//            test1Msites();
            test100000sites();
//            test100000sitesPlotA();
//            test100000sitesPlotPi();
        } catch (IOException e) {
            e.printStackTrace();
        }
//        test10000sites();
    }

    public static void test1000sites() {
        Map<String, String> omap = new HashMap<>();
        omap.put("H", "CATACTAACGCGAACTTAGCGCAATTAGTCAGATGTCAACGGAAGGGGCGTGGACACTCCGCCTTTGCATTCAATCAGGATAGATCAAACGGGCGTTACAGGCCCCCACAGGTCCGATATAACTCGAGGTGCATGTGATGTCTCGATAGCCTATACCATGGGCAAGTTGTCCATAGTCAATGTTGGGCAGTTTAGATTCTAAATGGCTCGGCGACCGTTGCCATCAAGAAGTAATACGCTGTAAGGGTAAGCGTTTGCGGTACTGCGGCCGGCTCACAGCACGTTACCCGGAGTTTATCCATATCAGTGCAATCACGCGTCAGCGGTTATCAATTTCCTCGTGAAATTCTATGTAACAACCAGGTCGCCATTGTAGACGACTGCTTTTGGACGGAGATGTTTTGCCTCATACCGCGAGGACCAACGCACCTATGGGTCATATCGATCATAGTAAAGGCCACCAACCTTTACACAGTAAAGCCCACTTTGCAGCTGTAAGAGAGATCATACTCGGCCGCGGGAGCACGGCCTCGAGCGGGGAAGCACAGCAACATGACCACGAGATCATTCGACCAGCCACTAGCCGTAGATTGCCAATCACACATAATAGAATATGTTTGGAATTGTCGCCAGCTATAAAGGATAGTTTTCGTGAGCCGGAATCAATTTCGCACTAAGACCGAGGTAACCTTACTAAACACACCGCCCTGGTGTCCGAGTACCTGCAAAATCACGGCCTTTTGGTGGAGCTAATTCCCTGACTCTATCTGTGTTGCCGAAAATCCAGCATGAGTTCCTACATAGTCTTATCCTTGGTCCAACACAACTGAACTAGCGGATATTGTGTCACCGACGCAAACTAGTGTTGCCAGGAACTTAGTTCATATAGGTACCTCACAGCGCAATCGGCGTACAGATGGCAGTGATGTGAACAGGTCAGTGAGAATTGGTATCAGTGTTCACGTATGTCCATCGCGCCTCAAGAGGCCTGGTGGCGCCC");
        omap.put("C", "CATACTAACGCGACCTTGGCGCAATTAGTCAGATGTCAACGGAAAGGGCGTGGACACTCCGCCTTTTCATTCAATCAGAATAGATGAGACGGGCGTTACCGACACCCAAAGGTTCGAAATAACTCGAGGTGCATTTGATGTATCGATAGCCTATAGCATCGGCACGGTGTCCGTAGTCAATGTTGAGCAATTTAGATTCTAATTGGCTCGGCGACCTCTGCCATCAAGAAGGAATACGCTGTAAAGCTAAGCGTTTGCGGTACTGCGGCCGGCTGACAGCACGTTACCCGCATGTTATCCATATCAGTGAAATCACGCGTCAGCGGTTATCAATTTCCTCGTGAACTTCTTTCTTACAACCAGGTCGCCGTTGGAGACGACTGCTTTTGGACGGAGACGTTTTGCCTCATACCCCGAGGCCCAACGCACCTATGGGTCATATCGATCATAGTAATGCCCACCCACCTTTACACCGTAAAGCCCACTTTGCAGCTGTAAGAGAGATCATACTAGGCCCCGGGAGCTCCGCCTCGAGCACGGAAGCACAGCAACATGACCACGCGATCATTCTACCATCCACTAGCCGTAGATTGCCAATCACACATAATAGAATATGTCTGGAGTTGTCGCCAGCCATCAAGGATAGTTTTCGTGAGCTGGGATCCATTTCGCACTAAGACCGAGGGAACCTTACAAAACGCTCGGCCCTGGTATCCGAGTCCTTGAAAAATCACGGTCTTTTGGTGGAGCTAATTCCCCGACTCTGTCTGTGTGGCCGAAAATCCAGCATGAGTACCTACATAGTGTTATTCTTGGTGCAACACAACTGATTTATCGGATATTGTCTCACCGACCCAAACCAGTGTTGCCAGGAACTTAGTTCACATAGGTACCTCACGGCACAGGCGGAGTACTGGTGGCAGTGATGTGTACAGGTCAGTGAGAATTGGTATCAGTGTTCAGTTATATCCCTCGCGCGTCCAGAGGCCTCGTGGCACCC");
        omap.put("G", "CATACTAACTCGACCCTGGCGCAATTAATGAGATGTTAACGGAAAGGGCGTGGACACTCCGCCTTTTCATTCAATCAGATTTGATCAAACGGGAGATACCGACACGCACAGTTCCGATCTAACTCGACATGCATGAGAAGTATCGATAGCCTATATCATCGGCAAGGTGTCGATAGTCAATGTTCTGCAATTTAGATTCTAATTGGCTCGACGACCGTTGCCATCAAGAAGTATTACGCTGTAAGGGGAAGAGTTTGCGGTACTGCGGCCGGCTTACAGCGCGTTACCCGGATGCTATCCATGTCAGTGAAGTCACGCGTTAGCGGTTAGCAATTTCCTCGTGAACTTCTTTTATACAACCATTTCGCCATTGGAGACGACTGCTTTTGGACGGAGACGTTTTGCCTCATACCCCGAGGACCAACGCACCTCTGGGTCATATAGATCATAGTAATGCCCACCAACCTTTACACCGTAAAGCCCACTTTGCAGCTGTAAGAGAGATCATACTAGGCCCCGCGAGCACCGCCTCGAGCGCGGAAGCACAGCATCATGACCACGAGATCATCCGACCCGCCACTAGCCGTAGATCGCCAATCACACACCATAGAATATGTTTGGAGTTGTCGCCAGCCATAAAGGAAAGTTTTCGTGAGCTGGGATCCATTTCGCACTAAGACCGAGGGAACCGTCCTAAACACTCGGCCCGGGTGCCCGAGTACTTGAAAACTCACGGCCTTTTGCTGGAGCTAATTCCCCGACTCTGTCTGGGTTCACGAAAGTCCAGCAACAGGTCCTACATAGTGTTATTCTTGGTCCGATACAACTGATCTGGCGGATATTGTCTCACAGACGCCAGCTAGTGTTGCCAGGAACTTAGTTCATATAGGTACCTCACGGCGCAACTGCAGTACTGGTGGCAGTAATGTGTACAGGTCAGTGAGAATTGGTATCAGTGTTCACTACTATCCCTCGCGCCTCAAGAGGCCTCCTGGCACCC");

//        List<Alignment> data = new ArrayList<>();
//        for (int i = 0; i < omap.get("H").length(); i++) {
//            Map<String, String> columnMap = new HashMap<>();
//            for (String key:omap.keySet()) {
//                columnMap.put(key, Character.toString(omap.get(key).charAt(i)));
//            }
//            data.add(new Alignment(columnMap));
//        }
        Utils.DATA = new Alignment(omap);

        ModelTree trueModel = HCGModelBuilder.getHCGModel();
        HmmBuilder builder = new HmmBuilder(trueModel.getTree(), trueModel.getRecombRate());
        HmmCore hmm = builder.build();
        double likelihood = hmm.logLikelihood();
        System.out.println(likelihood);

        ModelTree badModel = HCGModelBuilder.getHCGModelWrongRecomb(1.8e-8);
        HmmBuilder builderBad = new HmmBuilder(badModel.getTree(), badModel.getRecombRate());
        HmmCore hmmBad = builderBad.build();
        double likelihoodBad = hmmBad.logLikelihood();
        System.out.println(likelihoodBad);

        ModelTree badModel2 = HCGModelBuilder.getHCGModelWrongRecomb(2.3e-8);
        HmmBuilder builderBad2 = new HmmBuilder(badModel2.getTree(), badModel2.getRecombRate());
        HmmCore hmmBad2 = builderBad2.build();
        double likelihoodBad2 = hmmBad2.logLikelihood();
        System.out.println(likelihoodBad2);

        ModelTree trueModel2 = HCGModelBuilder.getHCGModelWrongRecomb(1.5e-8);
        HmmBuilder builder2 = new HmmBuilder(trueModel2.getTree(), trueModel2.getRecombRate());
        HmmCore hmm2 = builder2.build();
        double likelihood2 = hmm2.logLikelihood();
        System.out.println(likelihood2);

        ModelTree trueModel3 = HCGModelBuilder.getHCGModel();
        HmmBuilder builder3 = new HmmBuilder(trueModel3.getTree(), trueModel3.getRecombRate());
        HmmCore hmm3 = builder3.build();
        double likelihood3 = hmm3.logLikelihood();
        System.out.println(likelihood3);

        ModelTree badModel3 = HCGModelBuilder.getHCGModelWrongRecomb(2.5e-8);
        HmmBuilder builderBad3 = new HmmBuilder(badModel3.getTree(), badModel3.getRecombRate());
        HmmCore hmmBad3 = builderBad3.build();
        double likelihoodBad3 = hmmBad3.logLikelihood();
        System.out.println(likelihoodBad3);

        // this model has wrong population size and divergence time
        ModelTree badModel4 = HCGModelBuilder.getHCGModelBad();
        HmmBuilder builderBad4 = new HmmBuilder(badModel4.getTree(), badModel4.getRecombRate());
        HmmCore hmmBad4 = builderBad4.build();
        double likelihoodBad4 = hmmBad4.logLikelihood();
        System.out.println(likelihoodBad4);

        // this model has wrong topology
        ModelTree badModel5 = HCGModelBuilder.getHCGModelWrongTopo();
        HmmBuilder builderBad5 = new HmmBuilder(badModel5.getTree(), badModel5.getRecombRate());
        HmmCore hmmBad5 = builderBad5.build();
        double likelihoodBad5 = hmmBad5.logLikelihood();
        System.out.println(likelihoodBad5);

    }

    public static void test10000sites() {
        Map<String, String> omap = new HashMap<>();
        omap.put("H", "AGACATATTCGCGGCAGTTGCTCTCCGCCACAGGGGTTGGGTCCAACCCCAGTACGTACCTCATTCGTCAATCTGGGTACAATCCCAGGACAAAGGATAGTGCTCTCTGGTCAGAGCTCTTTGTACAACTTGCATTCGATTGTGAAACAAAATGGAGCTCGGTTTATGGCCTCGGTGAGTCCTGTTTATTGGACCCTAGGCGAATTAGGCACTTTATTATAGCGTGGCCCCGAGCAGGACGCGATGGCGCCCAAGGCGCCTAAACTTGATCAGTGAACGGGTGATAAACGGCTGTTCTCGGGGCTCAAGCCGATCTGTTACAGTTAACCTATAGAACCGGTTACGACAGGAAAAGTTAATCCCGATCATTGTTCTTGGAGTCTAAATACAAGAACGTTGCAGGCTGACATCGCGATGAGCAGGAGTGGATCACACTACTAAATAAATTGGTTCTCGAGGACCTAAAATTCTTGGTCATGGATGTTGACATATTAATCTATCTAAAGAAAGTAGATGGAAGTGCAAGAAGAAACACCAGGCAATTCTCATTTATTCGAGTTGACACGCCCAGATAGACTGGGACTCTCCCTAATCGTCTGGGCCTGAAAAACGTCGGATCCATAAAACGTCATTTGGCCGAGGCCCATCAGGTTAGAGTTACGCTACACACTATCTTGCATGCGAGATACGAACTGCCAAACTAGGCGTGACAATATCACCCCACAATTGTTTTCTAAGAAATCACTTACTTCGCTACGTGGGGGACTCTATAGCGCGACCACACTTGCCCTCATCTGGGTATACACGCCCAAGAGCTACAACAGTGTGGAACGTTACCATTGATGATAGCACGCAGGTCGACTTGCAGTCACCTGGTATGCATCTCTAGAGGGTACGCGAGACCAGAGCACCATAAACAAAGAGCCGACCAGAGGATAGTCGCTTCTTATCTGCAGAATGACTTCGTGCCTGTTTGAGGATCATTAACGATAGTTGGCAGTATTGAACTGATTGCTAGAGTCCTCCGCACTATGTGACGAAGGGGAGTGTTAAAAGCGGACTTTATTGAATGGTAGATGCTAACATAGAGCTGAAGTGTGGATTTGGTCTACCCTAGCTGTGCTCTAATCATTCAGTTGAGTGTTTGCGCTCCTGCGGATGGTCCCCGTTGTCTTAGTACTGCGCCCAACCAGACGATGTGAGTATCATTCGATCCTGCGAAAGTTAATTCGCTATGACCAAGATAGCAGATTGAGGTAACATAGAGGTTCCGATGCTTGTGTGCCGGTGAGCTGCTCTTGTCGTAATGGGCCCCCAAACGTCAACTACTCTAAGTACCAAAAAGGGGATTACATAGTATCAGCAACTCTGCTTTCGAGCCCTGCTAGTCGAACGCTTGTGACCTGGCGGCTGTTCGCAGATTCTGTTTATAGGCTAAGTCTACATTAGTGCTGAAGAGGTTATCTCGGCCTGGCCTAAGACCACATTAAGAAATGTCGTTCTTGCCCCTCACTAATCCATGCTGAGACAGAGGATCCGTACGGACAGATTATTCATTCCCCCCCGTGGTTACATAAACGCGTTCCTTTGTTACGATTCTACAAGAGACAGTGCGAGAGGTGGCCGATATATGTCCTTTTGCCAAACACCTAGACCTGGCGGTCTCTCAACGGGAAGGCCGCGTCGCTGCCTATAGATTCCCGAGACTCCAATGTTGCGAGAGCAAATCTGTACGTTGCCCGTTAACGCTTAAAACCGGTAGAGAGCCCGCGAACGCGCTTCATACCTGTGCCAGGCAGATATGGTATTCTTTGATAACACAGGTCGAGTCTAATCGTCGCTACTCCTATCAACCCGCGATTGGCGGATCATGTTATGGTAATGTGAACGGCTTGACTTTGTGATCCTGGCCGAATGTAATCTTCTGTCTTCTTAAGGGATTGATATCACAAAAATCAAACCCTCCTCGAATCTACACATCAGAAGTCCGGGCCAACTGGGGATCTCACTAACCTATGTGACTCGACAAATATCGCCGAACCCGATACATAGCATACGCTAGTCCGAAAGCTTGTGCTCCCAGGTGTTGACGACTACATTCGACCACTGATAGCTCGACTCTGACTCCACTCGTACAGTTAGGAGAGTTCGTCCCGTCCCTTATTACGTCTGATCGAACAGTACTGCCTGAATTCCTCCGGTACTTTAACTCCATGTTTCACACCGGAGTTGTCAAATTTGTCTTGGCCAAACCTGCGTCTCACTAATACCACTGTATATGCTGATTGGCCATATATCTCAGAATAGGGATTGTTTGGGAGGGGAGTACTACTTCACTTAGGGGGCAGTGAATGCTATAGAATTATCGACCGGACTGCGTGTATGTTCAAACCTGACGTTCAATCATACATACTGCCCCGGTAAGGTTGAACTGTGTCTGAGAAGGCCTTAGAACTAGATATGGATGACTTCGAAATGCCACCCTCTATTTAGTCGTTACTATGAGTTTTCTACTCTACACACTGAGGCGACTCCAAGCGAACTCCCGGAGCATGCCTTCGGCCCTATGAATCTTGTCATTCTCATTGCGACACTACGAGCGCCGCTAGCGCGACGGACTAAAGCTCAAAGTCATTCAGCCCTATTATGATTATGCGTCCCACCCAGGCCAAAAGGCACACATAACTTGAAAAAAGTCATCATAGATTCATGCAAGGTCCCACTACCCGGGCGGTCTTGAAGTAGCGAGCATCCTGTGAATCTAGTCGCGTCCTTGTGTCGGAATTCGTTATATAAATCCACTGCTCCGCACCCTCATCAACGTCAATCGGTGGACTACCTCGTGTGCGTATTAGACTTTTATTCTCCTTTTATGATGGCGGGCCTAGCCGGACTAACACATACACTGTGATGTATTACGTAGGGTGCCGAAGATCACGATGAGCCTGACAACTTCCCGGTGCTTAAGCCTCTTTTGTCGTACACAAAGAGAGGGCGTCTGGTCCATGAAAAGCTTACTGTACGCAATACATAGGATCGTGATCGGGGCTCCCAAACAGCCGCGCGTACGCCGTTCCCGGGATTCTAAGGTGGGGCGCCCAGTGCCACTCCATATCAATGGTAACCAATATAGGAACACTCATCTAACGGGCGTCTATGTCCAATAATAAGTGTCTACATTAGCCGTGGTAGAACGACGCAGGTGATCTCCGCTTACGCGTCGATCGAACACTGGCCGACCTACGTAATCCGTGAAACCAACCGGCGAGCCTCAGGTTCTGTTAACGATTGCGCTCATATTGGACCCGGCATGCTTTAGGGCGCGGCCTGCTTTGCAGGAATAGCCACTAGCGATGTGGCCAACATCTTTTAAGCTGGGTCGGACCGAAGGCCCGCGATGATCTGGACATAGTAGGCAAGTAAGTTTACACCGCTGGGCCGTCCGGTAACACAGCACCCTTGAACCCTCAGCATCTACAACAAGGGCGTCGTACGCTGGGCGCAAGATTAGACGCTGTATGTGTGCCAGAGGTAAAAACCGTTCTGCTTTGAACTGCGCGCTCGGCCTAAGTGGGGCGAGGGCTGTGAGGAAGATACTGCACTCGTTAAGCGAATATTTGCGGCATATTCGCCAGTTATGGAACTGCACAGCTGTAAGGGCGAATGCCGACTCTAACCCCCATCGATCTATGAGATGTCCTTTACTAAAACGGGTCGGTGGATCGCTGCAGGTCAGAGTTACTGAATACGACATTGTACCAAACCAATCGGGGCGTACCAGTAGTAAAGCTCATTTACTTAGACAAGTGGCGTTTAAATTTCGGCCACGAAAGTTGATAAACAGGACGATGCTAGGCTTGGCAGAAATTTACCTATCCCATGGCTGGACTCCGATGTACCGATGGTCCAATGCGCCTACCCTAAGCGCAGCAGACGGTCGTTCGACCTTGTGGTAGTTAACACAATGCTATCGCAATAAGCTGCCGTTCGCCCCCACTAACTACACTGAGCCGAACCTCCGAAGATGAGCAGTCGACTTAGGTAAGAACAAAAAACTAGCAGTCGAAATATGCATTATTCCCCGAGCCATATCAATTCTGTTGGGAAATATGTAGGGCTGAGGTCCCACTTAAGGCATCTAACGAGAATGCGGGTGTTATCGGCCTTGTCGTGTGCTCAATACCAAAGGCGCATGGGTCGAATCACACGAAATTGCACGGCGAAGTGAATGGCCACGAGGTATTCCAGGCCTTCTGTGCCCCAGTCTGGTAAAATCGCCAGTTACCGTGTCCGACCTATCACGCGCGTTACCAGAGTATTTCCTCCTACCAATAATGATATGCTTGTACACACTGCTTGTGTCTGATTCGCGCCCTCGCCGATTCTCAACACAATACTGGAGGTCGAGATGCCCGTTTTAAGCCCAATAGGTCAAAGGTCTTCTGTTGACCTGGGTCGGCTAAGGTATGCGGGTGCCGCCCTATTCGGTTGTCGGGTGACCTATATTTTGTGACAGCCACATGGTCGCGATTTCCGAACCAATAAGGTGCCTCTATGAGCTATAAGCGCCACGGCAGTAAGACATGTAGTTTAGCATGGAACTTCTCAGTTGGAGGATACATTTTATCGATACACATGTCTTCAGGGGCAAGACGCGCGCCAGCTATAGCGTTTGATATCACAATCACATCTGTTATCGATTAACCTGTAGCTAGAGCCCTACAGACTCCGCTGCTTCTGGCGATCCCTTTAACACCTTCTGGTATATTCGTGCCCCGTGCTACCTGGCTATGGGACCCACTCTTTATTGTACAGCGACTCAACATCGGTGTGCATCACCAATCCAGGCGCGCCTTGTCAAGGGGACTATTATAATGAGCGGCTAAATATGCCACAAGCTATCAATATATGGTAGCCGACCAGTGTGTGAGATAAGTGAGTGTCGCAACTCTAAGGTTCGGTAAGGCTTCGTCTGCCCGGTAGCGTGATATATTTAGTTGGGCCTTTGCATTCCATGCCAAATGACACGGGGCTACGGTATGGAACCTCATCGTCCTGCAACATGGGCTTCCGAACAGTTCTAGGCTGTGGCATCCCCCGCGGGGCTGGACCCAAATTCGTTAGGGACGGAGCGGTACAAGCCAAACAGTGGAGCGACTCGGTAATTTATTGGACAGAAGCGCCAGGATATAGTATTCGTGCATGACTTTGGCCGGGGCCGATCCTGTCTCTGTAATAATTACCGCCAACGCTAGCGGAGAAATGAGGTAACGTGGTTCACGCTACATAATGACTCTTATCCGCTGACGTAATGAGACTCTCGTCATGGAATCAAGAAAAAGTAACCACCTGAACGGTCTACCTGCAAATGCGGATAGAACGGCCCGGAAGAAGCTACTGTGGCAGTTTAATTCATCATAATTAGGGGTCCGAATCTTGCACATACAGATTGCGAACGCACCATGCTCACTTGAATTTACAGGATAGCGTCTGTCAGCATTGCTCCCCGAGAAGGTGATCTAATCTGCCACAGGACGCTTGACGGCAAGTTCTACCCCCAGGGAGACCTTACTTATAAGCTATGGCGGGGCACCCTTGGGCTGCCAGACGGGTGAAGCACGGACTTTAGTATACGTACTTTCAAACATTCGACGGGATATGTCGCTTTAGACGGGTGACAGATGATTGAGCCCCTCATCTCATTTAGGTCGCGGTGTTTCGGGACGCGTGACATTCGAGTTCGTGGTAGGTGTGAAAACCCTAAAGTTCATGACTACGTAGGCGGGAGTGGAGTCCCTGTTACTAGTTTTTGAGAAGATTGCAATTATGATGGCAGACGAGGCCAGCGTTTGAGACAACTTAATGTCTTCTACCTTTAATGAGAAATACGATTCCTGTTTTCTGCCATACCATGAACCGAGTCTATATCAGGTAGGATACAAAATATGCAAAGACGATGGCCTCTCGTAATGTCTTGGGTCAAGACCTACTGGCACTCCTATTCAGGTACATACCCGAAGAGTCGGTTCTGGTCTTGCACGAGGCTTAGATCCCCGTATCCTGACTTACTAACAGACTCTCCGATAACAACCTGTCTTGAATTACTAGACGCCGCGGGTCTTCCACTACCCGCTCAGCCGACGATTGTAGACGCGTTGAGATCGGTGAAATCCGGACCATTCACTCGTGTTGAAAAGCTAACCTTTGCCGCGACGACACGAACTCCGGTCTGCACAGTGGTCGTGGAGGGTAACCAGTCAACTCCTATTGACCAGCGCAGAACCACCCTCGACCTCTCTGACCTTCCAGGTTTGTATGTGCCCGAAATGTGATACGCTCTCCAAGGGACCAAGAATATTATGCGACGGGCTCATCCGGCCAGAAGGAGCGCAGTATAGCTCCAATTGTTCTAGTAAACTGATCTGAGCCGTGTTGAATTTTGACGAGTTGAAAATCCCCGACTAAGCTTGCTTATCGCTTGTCGATTGATAGTAGGGACTTTCCATGCACATTCCACAACAGTGCACGACTAGGACAACTCCCTGACTGGATCAATACATATAATACTCGCGCTCTGTCATTCCCAATGGCCCTTGCTCCTGTCAATTGCTGGTCGTGGAGCGGGTACCTGATAATTTCGCTAGCGGGACGGATTCCACCAGGATCCAACCGACATGACGCGGTAACGTCACTAAGGATACGCGGACTTGCAGGCGCCGCTCTTCACGTTTAAGAGGCCCGTCAGTGCCAATTCGTCTAAAGGTAGTCGTCTCCGTTATAGGCCTAAGCATCTCGAGCTCAGCAAGCGATGCTCCGGATAGACGGATCATGTTAATGATCACATGTGACCTAGAGCACAGTCCCCCCTTAGACAGTTCTCGTTGAACGATGTGCCAACAGAGCTCCGGATGATTTACGCAGATTACGCGCGAACGCTCGTTGTACTGTTTGAGCGCTAAGTCCTTCTCTGCCCCTGCAACGCACGGTGTTGTCGGTTGGAGATTCAACCCTGCTTTCCTCTAGGGCGAGCCGTGACCTCCAGAGTCGACCGCTAGCGTGTGGCACGCGACGCCCAGGGTACGTGCTGCTCTAACCGTTCCAATGACGAGTTTAGTGTCTTACGAATACCGTTTTTGTGGGGCTCGGACCCATGGTTATAGGGGAGGGAATTCTAGTCGCAGGCTCCTTGTCCCTCAATTTTCGATAGCAAGCACATGGTGCCGAGTTATCTTTGTGGTGTTGACGACAACCGGCCCGCAGCCTTCTTGACTTATGAAAGTTACTTGGGAATAGGATGAAGTACTACCTCTTCCCAAACGAGAACAAATGATATGTATTATGCTGTCAGACCGTAGCTGCGCTGCGCGAACATCCTAGTTGGAGGATACCTCGTGCGTCAACGGGATCCGGGACGCTGCTATGATTCATACCTTCTTTACGTGGCTCGTGAACGTGATGATGTACCAGTATCCATATAAATCCTGGGCTTAGAGCCCTTTAGGACCTCTTCGACTGGCTCTTTCTTCACTGTCTTACCGCTGACGATTAATTCGACACGGGTAAATCACCCCACACTTAAATGGTGGGCGCCACTTAACGATCTAGCGAGGCTGCCGGAGTGCCATCCTCCCCGCTTCTAAATCCGGCCCGGCCTAGGGCGCCGAATCCCCGGGATGTAGATAGGAGCTACGGGCATCGTACTGCTTGGCTATCCCGGGCTTGCGCCTGATCGCCCAAGGGTGAATTGTTCATGTCTCGCGATGTCATAGACGTCAGCTCTTAGCATATCATGAATAGCAATGCCGGGTTCTAAAAGTCTTGAAGGGCCAGCCACACTCAGGAGTTACGGTAGACATGTACAGTAGCTGTACCGCGCCTGTGGTTTGCCACCGCTCCCTTGTTTCACCAACCTACCCACCAGTAGAACTGGTGCCGTCTCGACCGAGAACGGTATCGCTAAATCAATTCTACTTAGAAAGTTCTCCCCGTATCGCATCGTTGGAATTACCATGGGCATCGGGTACGTTGTACTAACCTCGTGTTCTTCAGACCAGCGTGAGACGTGTGGCCCGCGAGAGGATGTTCTCATGTCTAGGTCGACGGACCCTGATAACGAGCAAAAGTCCTATGTAACCCGCCCATAAAATGATCACCGTAATCATCCTGGCCTGTGTAATCATAATTTTCAGGGGCGGAGGCAACCCGGGGTACGCGGTGACCCCACAATGAGTTGGCCCATATGCTCTGAACGGACAATCGTCCACTGAGGCGCAGCCTCTTCAAGATCCTTTAACCTCAAGTCTTGCGATCTACTGGTCTACGACCTTTGAATGAAAATTCGATACGCACGGCGCGACTCTAAGAAGCAGTCGGAGGCATGTGTCAATGAGTGGCCTGCTCTGGCGGAACGGTTAAGTCTCGTCGTTAGTTCTACATAAATGGTACACAAGTGCAATCGACTACGTATATGTAGCTAACGGATAGCACAGGTGTGGCGCAACGATACCACGCTTCTTACAGCTTGTACTCGCGGACGTACCGTGCTGTTTCGGTTTTTCATATTTGCACAAACGGCTCGAGAACTATATGCGACATTTTAAACCCTTATGGGGGTAGAACGTACATCAACGTTGACATAATAACGGCAGGGCTCTTCCCATGTGGGCTCAAGGTATAGCCGACACCAGCCATCTGGACATATGAAGCGGATATCTTTGTCCTATCACCACGGTCGAACTATCAAGTATCGGTCGTTGGGAATGGACGATGAAAATGATCTTCGTTTTTATAGATAAGAATATTTCGCTCTAAAACTGTGGACGGAATACTTGTACTGTAAATTGGGAACGAGACAGTGATATGTGGCGTAATGAAAACCAGCGAAGGACCTCAGTAAAAGTAACGATACCTGGAAAAGTTGCGCACCGATTTCGACCGTCTGACTATTGGATGACAAGCACGCGCGTTCCTTCTACATCTTGCCGCAATGTCTCATCGTCGGGTTATGTCTAAGCAAATACGGAGATACTGATCGATCGTCAAAGCCATGTCCGGGGCGCTTATGACTAACACGACGTAACCTGTAAAGCTCGGCACGCATGTATCAACTTTTTAAGTTAAAGCATGAGAGTAGCCTCGACCGTTCTGCGCCCTCAGCTTACAGGCAAACGCCTTACGAAAGTCGGACGATTGGCTGTGGGTAAGGCGCATTGTCCGTGACCCGCTATGGATTTTTTGACGGCGCATTATCATGCCCATTTTTACATACTGACGTGGTGTTCAAGGGTGCGACAGGAGTCATCCCGGCGGATTCCGTATCTATAGCCGGGATGATAATCTATAGGCTGGTCAGGGGGAGTCACTGGCCCTTTCTGCGCCTATGGCCCATCGCGACTGGGGTGGCAGCTCTGCGCAAGTCCCGTTCAACTTTGATCTCAATGGATATCTACAAAGCCAATGCTAAGTAGTGATTGCCATGTTTTAGGGTAGTTACTAAGCCACTCGGCCAGTGTCCGAGTAAGACCATACACTGGTCCCAATTAATGCATCGTTACTAGTCTGCATCCTATATACACTTCGTGAGCATTTAGAAACACGAGCCCCCAAGCCAGGGCACAACTTTGTATTTAGGTATTCGACAACGTTCACTGGTCTGTACCGCTTCATACAATCCTCCCACAACAGACTTAAGCTAGGAACCAACAATTGAGATAAATTGACCTAGGAAAGACAGTCGGACGAATGTCAGATTAAATCATGTTCGTAATCGAAAATACGATGTGCGAACACATTTCACACATTCACTACGTTGTGGTAGTTTGCCTCGTACCAACTTTGTGAGTTATGAAAAATGAGGCAACAAAGCCAGAATCGAACCAACTCGTCATGGACCCAGTGCGTGAAAAGTAGTGGGACGGAGGGAACCCACTAGCTTTTAAAGAGCCTATGTATAAGCAGCTTCCGTCTCCCGTTCTCGCGTTTTCTGTAGAAACCTTTAGAC");
        omap.put("C", "AGACATATCCGCGGCAGTTGCTCACCGCCACAGGGGTTGGCTCCAACCCCAGTATGTACCTCATCCGTCAAAATGGTTATATCCCCAGGACAAAGGATAGAGCTCCCTGGTCAGTGCTCTTTGTACAACTGGTATACGATTGTGAAATACAATGGGGCTCGGTTTATGGCCTCGGTGAGTCCTGTTTATTGGACCCTAGGCGACTTATGCCTTTTATTATTCCGTGGCCCTGAGCAGGACGCGTTGGCGCTCAAGGCTGCTAGACTTGATCAGTGAACGGGTGATAAGCGGCTGTTCTCGGGGCTCAAGCTGATCTCTTACAGTTAACCTATAGACCCGGTTACGACAGGAAAAGTTAATCCCGATCATTGTTCTTGGAGTCTAAACACAAAAGCGTTGCAGGCTGACCTCGGGCTGATCAGAAGTGGATCACACTACTAAATAAATTGTTTCTCGAGGACATAAAATACTTGGCCATGGATGTTGACATATTAATCTATCTTAAGAAAGTAGATTGAAGTGCAAGAAAATACAGCGGGCAATTCTAACTTTTTCGAGTATACACGCCCAGACAGACTGGGACTTTCCCTTATCGTCTGGGGCGGGTAAACGTCGGATCCATAAAACGTCATTGGGCCGAGGCCCATCAGGTTAGAGACACGCTACACACTATCTAGCATGCGAGATACGAACTACCAAAGTAGGAGTGACAATATCAACCCACACTTGTTTTCTAAGAAAGCACTTACTTCGGTACGTGGGGGACTCCATAGCGCGACCACACGTGCCCTCATCTGTGTATACACGCCCAAGAGCTACAACAGGGTGGAACGTTACCATTGATGATAGCACATAGGTCGACTTGCAGTCACCTGGTCCGCATCTCTAGAGGGTCCGCTAGACCAGAGCTCCGTAATCAAAGAGCTGAGCAGGGGATAGTCGCTTCCTATGTGCAGCATGACTTCGTGCCGGCTTGACGAACACTAACTATCGTTGGAAGTGTTGAACTCATAGCTAGAGTCCTCCGCACTATGTCGCGAAGGGGAGTGCTAAAAGCGGACTTTAATGAGGGGTAGAAGCTACCATGGTGCAGAAGCGTGGATGTGGTCTAGCCTAGCTGTGCTCTAATGGTTTATTTGAGTTTGTGCGCTCCTGCTGATAGTCCCCATTTTCTTAGTACGGTGCCCAACAAGGCCATGTGAGTAGCATTCGAGGCTGCGAAAGTTAATTCGCTGGGACCAAGATGGCAGATTGAGGTGAGATGAAGGTTCTGAGGCTAGTGTGAAAGTGAGCTGCTGTAGTCGTAATGGCCTGCCAATCGATAACTACCTTATGAACCTAAAGGAACACTACACCGTATCAGCAACTCTGATTTGGAGCCCTGCTAGTCGAATGCTTCTGACCAGTCAGCTGTAGGCAGATTCGGTTTATAGGCTAAGTCTACATTAGTGCTCAATGGGTTATCTAGGCCTGGACTAAGACCACATTGAGTAACGTTGTTCCGGACCCTCACCAATACGTTCCGAGACAGAGGATCCGTACGGACAGATTATTCGTTCCCCACCGAGGTCGCATTAAAGCGTTCCTTCGTTACGATTCTACAAGAGACAGTGCGAGAGGTGGCCGATATATGTCCTTTTGCCAAACACCTAGACCTGGCGGCTTCTGAACGGGCAAGCCGCGTCGCTGCCTATAGATTCCCGAGACTCCAATGTTGCGAGATCAAATTTGTACGTGGCCCGTTGACGCTTAAAACCGGTAGTGTGGCTGGGAACGCGTTTCATACCTGTGCCCGGAAGATATGCTATTTTTTGATAACCCAGGTCGAGTCCAATCGGCGCTACTCCTTTCAACCCGCGATTGTCGGATCATGTTATGGTAACGTGAACGGCTCGACTTTGTGATCCTTGCCGAATGTACTCTTCTGTCTTGTTAAGGGATTGATATCACAACAAACAAACCCTCATCGAATCTACACATCAATAGTACGGGCTAACTGGGTATCTCACTAACCTAAGTAACTTGACAAATGTCGCCGAACCGGATACATAGAATACGCTAGTCCGAACGCTTGTTGTCCCAGGTGTATACGACTACATGCGACCACTGTTAGATCGACTCGGACTCTCCTCGTACGGTTAGACGAGCTCGTCCCGTCCGTTATTACGGCTGATCGAATAGTACTGCCTGAATTCCTCCGCTACTGTAACTCCATGTTTCACTCCTGGGTTGTCAAATTAGTCTTGGTCAACCCTGCGTCTCACTTATACCACTGTATATGCAGATTGGGCATAACTCTCAGAATAGGGATTGTTTAGGAGGAGAGTACTAGTTCTCTTAGGGGTCAGTGACAGCTATAGAATAATCGACCGGACTGCGCGTATGTTTAAACCAGACGTTCAATCATACATACGGCCCCGGTGAGGTTGAAATGTGTCTGAGAAGGCCTTAGAACTAGCAATGGATGACTTCGAAATGCCACCCTCTATTTAGTAGTTGCTATGAAATTTCTACTCTACACACTGAGGCGACTACTAGCGTACTCCCGTAGCATGCCTTCGGCCCTGTTAATCCTGTCACTCTCATTGCGACACTACGTGCGCCGCTAGCGCGAGGGACTAAAGCTTAAAGTCCTTCAGCCCTATTATTATTATGCGTCCCATCCAGGCCAAATGGAGCATATAACTTGAAAAAAGTCATCGTAGATTCGTGCAAGCAGCGACTACGCGGACGGTATTGAAGTAGCGAGCAACCTGTGAATCTAGTCGCGTCGTTGTGTCAGAATTCGTTATATTAATTCGCTGCGCCGCACCCGCATCCACGTGCATCGGTGGACTAACTAGTGTGCGTATTAGACTTTTTGTCTCCTTTTATGATAGCGGGCCTAGCCGGACTAACACATACGCAGTGATATATTACGTAGGGTGCCAAAGATCACGATGAGCCAGACAACTTCCAAGTGCTTAAGCCAATTTTGTCGTACACATAGAGAGGGCGTCTGGTCCATGAGTAGCTTACAGTCCGCAATACATAGGATCGTGATCGGGGCTCCCAAACAGCTGCGCGTATGCGGTTCGAGGAATTCTAAGGTGGGGCGCCCAGTGCCAATCCATATCAATGGTAACCAATTTAGGAACACTCAACTAACGGGCCTCGATGTCCAATAATAAGTGTCTACTTCAGCCCTGGTAGAAGGACTGAGGTGATCTCCGCTTACGCGTCGTTCGAACACGGGCCGACCTACGTATTACGTTAACCCAACCGGCGAGACTCAGGTCGGGTAAACGATTGCTCTTATATTTGACCTGGCATGCTATAGGGCGCGGCCTGCTTTGCAGGAATAGCCACTAGCGATGTGGCCAACATGTTGTAGGCTGGGTCGGACCGAAGGCCCGCGATGTTCTGGACATAGTAGGCAAGTAAGTTTACACCGCGGAGCCGTCCGGAAACACAACGCCCTTGAGCCCTCAGCACCTACAACAAGGGCGTCGTTCGCTGGGCGCAAGATTAGACAGTGTAAGGGTGCCAGTGGTAAAAACCGTTCTCCTTTGAAGTGCGCGATCGGCCTAAGTGGGGCGAGGGCTGTCAGGAAGATCCTGCAATCGTTAACGTAATACTTGCGCCATATTCGACAGTTATGGAACTGCACAGCTGTAACGTCGAATGCCGACTCTAACCCCCATCGATCTATGAAATGTCCTTTACTAAAACGGGTCGGTGGATCGCTGAAGGTCAGGGTTTCTGAATACGACATTGTACCACACCAATCGGGGCGTACCCGTAGTAAAGCTCATTTACTAGGACAAGTGGCGTTTAAATTTCGGCCACGAAAGTTGACAAAGCGGACGATGCTAGGCTTGGCCGAATTTTACCTATCTCATGGCTGGACTCAGATGTACGGATGGTCCAATGCGTCTGCCTTAAGCGCAGCAAACGGTCGTTGGTCCTTGTGGTACTTAAAACAATACTATCGCAATAAACTTCCGTTCGCCCCCACTCACTAAACTGAACCGTGCTTGCGAAGATGAGAAGTCGACTTAGGGAAGAACAACAAACTAGTAGCCGATATATGCATTATTACCCGAGCCATATCAATTCTGTTGGGAAATATCTTGGGCTGAATTCCCACTTAATTCATCTAACGAGAATGTGGGTGTTATCGGCCTTTTCATGTGCTCATTACCAAAGGCGCATGGGTTGAATCACACGAAATGGCTAGGCGAAGTGAATGGCCACGAGGTATTCCAGGTCTTCAGTGCCCCAGTCTGGTAAAATCGCCAGTTACCGTGTCCGACCTATCACGCGCGTTACCAGAGTATCGCCTACTACCAATAATGATATGCTTGTACACAGTGCTTGTGGCCGATTCGCGCCCTCGGCGTTCGTCTCCGCTATACTGCAGGTCGAGATGCCAGGTTTAGGCCCATTAGGTCAAAAGTCTTCTGTTGACCTGGCTCGGCTAAGGTATGCGGGTGCCGCCCTATTCGATTGTCGGGTGACCGATAGTTTGTTACAGCCACCCGTTCGCGTTTGCTGAACCAATAAGGTGCCCCTATGAGCCATAAGCGCCACGGTAGTAAGACACGTAGTTTAGCACGGAACTTCTCAGTTGGCGGAGTCATGTTATCGATACTCATGTCTTCAGGTGCAAGACGCGCGCCAGCTACAGCCTTTGAGATCACATTGACATCTGTTATCGATAAACTTGTAGCTAGCGCCCTACAGAATCCGCTGCTTCTGGCGATCCCTTTAACACCTACTGGTATATCCGTGCCCCATGATACCTGGCTATGGGACCCACTCTTTATTGTACAGCGGCTCAACATCCGTGAGCATAACCGTTCCAGGCGCGCCTTGTCAAGGGGACTAGTATAATGAAAGGCTAAATCTGCCACAAGCAATTAATGTAAGGTAAACGCCCAGTGTGTGACATATGTGAGTGTCGCAACTCTAATGTTCGGTAAGCCTTCGTCCGCGCGGTAGCGTGATACATTTAGTTTGGGTTTTACATTCAATACCAAATTAGTCGGGGCTACGGTATGGACGCTGATACTCTTGCAATTTGGGCTTCCGAGCAGTTCTTGGCTGTGTCATCCCCCGCGGAGCTCGACACAAATTCGTTAGGTACGGAGCGGTACAAGCCGAACAGTTGAGTGACTCAGGAATTTGTGAGACAGAAGCGCCAGGATATACCATTCCTGCATGACTCTGGCCGGGGCCGATCCTGTCTTTGTAATAGTTACCGCCAACGCTAGCGGAGAAATGAGGTTACGTGGGTCACGATACATAATGACTCTTATCCGCTGACGTAATGAGGCTCTCGTCATGGAATCAAGAAAAAGTAACCGCCTGAATGGACTACCTGGAAGTGCGGTTAGTACGGCCCGGAAGAAGCTACTGTGCCAGTTTAATTCATCATAATTAGGGGTCCGAATCATCCTCATACAGATTGTGAACGCACCATTATCATTTGAATTAACAGGATAACGTCTGTCATCATTACTCCCCGAGAAGGTGATCTAATCTGACACATGACGCTTGCCGGCAAGTTCTACCCCCAGGAAGACCTTACTTATAAGCTATGGTGGGGCACCCTGAGCCTGATAGACCGGTGAAGCATGTAGTTTAGTATACGGACTTTCAAACACTCGACGGTACAAGCCGCTATAGGCGGGTGACAAACGATTGAGCCCCTAAACTCATTTAGTTCGCGGTGTTTCGGGACGGGTGGCATTCGAGTTCGTGGTAGGTCTAAACAACCTAAAGTTCATGACTACGTAGGAGGGGGTGGAGCCTCGGTTACTAGTTTTTGGGAAGATTGAAATTATGCTGGCAGACGAAGCCAGCGGTTGATACGACAAAATGTCTTCGACCTTTAATGAGTAATACGATTCCTGATTTCGGCCCTACCGAGAACTGAGTCTATATCAGGTTGGATACAAAATATGCAAAGACGATGTCCTCTTGTAATGGCTTGGGTCAAGACCGACTGGCACTCCTATACAGGTACATACCCGAAGAGTGGGTTCTGGTCTTGCACGAGTCTTAGATCCCCGTATCCTGACTTACTAACAGACTTTCCCATAACAACCTGTCTTGAATTACTAGAGGCCGCGGGTCTTCCACTACCCGGTCACGCGATGATGGTAGTCGCGTTGCGCTCGGTGAAATCCGGACCATTCACTCTTATTGAATAGCTAACCTTTACCGCGACGACACGAAGTCCGGGCTGCACAGTGGTCGTGGAGGGTAAACAGTTAACTCGGATTGACCAGCTCAGACCCACCCTCTGCCTCTCTGACGTTCCAGGGTTGTATGTGGCCGAAATGTGACACGCTCTGCAAGCGATCAAGTATATTATGCGAGGGGCTCATCCGGCCAGAAGGACCGCAGTATAGCTGCAATAGTTCTAAACAACTGATCGGAGCCGTGTTGAATTTTGATGAGTTGAAAATACCCGACTAAGCTTGCTTATCGCTTGTCGATGAGTAGTAGGGACTTTCCATGCACATTCCACATCAGTGCACGACTAAGACAACTCCCTGACGGGATCAATACATTTAATACTCGCGCTCTGTCATTCCCAATGACCTTTGCTCCTGTTAATTGCTGGTCGTGGTTAGGGTACCTGATAGGGCCGCTTGCGGGACGGATTCCATCAGGATCCAACCGACAAGACGCCGTAACGTCAATAAGGAAACGCGGACTTGCTGGCGCCGCTCTTCACGTTTAAGAGGCCCGTCAGTGCCAATTCGTCTAAAGGTGGTCGTCTCTGTTACTGGCCTAAGCATACTGAGCTCATCAAGCGTAGCTCCGGACAGACGGATGATGTGGATGATCACATGTCACCTACAGCACAGTCCACCATTGGCCCGTTCTCTTCGAACGATATGTCGCCAGAACTAAGAATGATTTACTTAGGGTAAACGCGAAAGCTCGTTGTACTGTTTGAGCGCTAAGCCCTTCGCTGCACATGCAACGCACGGTGTAGACGGTTGGAGATATAACCCTGCTTTGCTCTAGGGCGAGCCGTGACCTCCAGAGTCTCCCGCTAGCGTTTGGCACGCGACGCTCAGGGTACGTGCTGCTCAACCCGCTCCCATGACGAGTTTAGTGTCTCACGAGTACCGTATTTGTGGGTAACGGACCCATAGTTATAGGGGAGGGATTTCTAGTGGCAGGCTCCTTGTCCCTCAATTTTGGATAGCAAGCACATGGTGCAGAGTTATCTTTATGGTGTTGACGAAAACCGGCCCGCAGCCTTCTTGCCTTATGAAAGTTCCTTGGGAATGGGATGAAGTACTACCTCTTTCCAAACGCGCACAACTGATATCTATTATGCTGTCAGACCGTAGCTGCGCTGCGCGAACATCCAAGTTGGAGGTTACCTCGTGCGTCAACGGGATCCAGGACGCCGCTATGATTCATACCTTATATACGAGGCTCGTGATCGTGATGATGTACCAATATCCAGATAAGTCCTTAGCTTAGAGCCCTTTTGGACCTTTTCGACTGGCTCTTTCTTCACCGTCTTACCCCTGACGATTAATTCGACACGGGAACACCACCCCGGTCTTAAATAGTGGGCGCCACTTAACGATCTAGCGAGGCTGCCGGAGTGCCATCCTGCCGGCTTCTTAATCCGGCCGGGCCTAGAGCGCCGAATCCGCCGGATGTAGATAGGACCTACGGGCATCGTACTGCTTTGCTATCGCGGTATCGTGCCTGATCGCCCAAGGGTGAATTATTCATGTCTCCCGATGTTACAGACGTCAGCTCTAAGCACATCATGTATAGCAATGCCGGGATCTAGAAGTCTTGAATGGACAGCCACACTCAGGAGTTACGGTAATCATGTACAGTAGCTAGACCGCGCCTGTGGATTGTCACCGCCCACTTGTTTCACCAACCTACCCGCCAGTAGAACTGGTGTTGTCTCGTCTGAGCAAGGTAACGCTAAAACAATTCGACTTAGAAAGTTCTCCCCGAATCGCATCGTTGGATTTACCATGGGCACCGGGTACGTTGTACTGACCTCGTGTTCTTCAGACAAGAGTGAGCCGTGTGGCCCGCGAGAGGATGTACTCACGTCTAGGTCGACGGACCCTGATAACGAGCAAAAGTCCTATGTAACCCGCCCATAAAATGATCACCGTAATCATTCTGGCCTGTGTAATCATAATTTTCTGCAGCGGAGGCATCCCGGTGTACGCGGTGACCCCAAAATGAGTGGCCCCACATGCTCTGAACGGACAATGGTCCAGTGAGGCGCAGCCTTTTCAAGATGCCTTAACCTCCAGTCTTGCGATCTACTGGTCTACGACCTTTAACTGAAAATCCGATCCGCTCGGGGCGACTCTAAGAAGCACTGGGAGGCCTGTGTCAATGCGTGGCCTACTCGAGCGGAACGGTTAAGTCTCGACGTTGGTTATACTTAAATAGTACACAAGTGCAATCGACTACTTACACGTAGCTAACGGATATCACAGGTGTGGCGCAACGATACCACGCCTCTTAAAGCTTGTATTCTCTAACGTACCGTGCTGTTTAGGTTTTACATATTTGCACAAGCGGCTCGAGAACTATATGCGAAATTTTAAACCATTATGGGGGTAGACCATACATCAACGTTGACATAATAACGGGCCGGCTCTTTACATGTGGTCTCAAGGTGCAGCGGACACCAGCCATGTGGACATATGAAGCGCCTATCTTTGTCTTAACACTACTGTCGAATTATCAAGTATCGCTAGTTGGGAATGGACGATGAAAATGATCTTGGTTTTTAAAGCTAACAACATTTCGATCTAAAACTATGGACGGAATACTTGCCCGGTAACTTGCGAACGAGATAGTGATATGTGGCGTAATGAAAACCATCGAAGGACCTCAGTGAAAGTAACGATACCTTGAATAGTTGCTCACCGTTTTCGACCGTCTGACTGTCGGATGACCAACACGGGCGTTCCTTCTACAACTAGCCGCAATGTCTCAGCGTCGGGTTATGTCTAAGCCAATACGGAGATACTGATCGATCGTCAAGGCCATGTCCGGGACGCGTATGACTATCAGGACGTACCCTGTAAAGCTTGGCACGCAAGTATCAACTTTTTATGTTAAATCATGAGGGTAGCCTCGACCGTTCTGCGCTCTCAGCTTATATGCAAACGCCTTACGAAAGTCGGACGATTGGCTGTGCGTAAGGCACATTGTCCGTGATCCGCTATGGATTTTTTGACGGGGCATTATCATGCCCATTTTTACATACTGACGCAGTGTCCAAGTGTGCGACCGGAGTCATCACGGCGGATCCCGTATCTATAACCGGGAAGATAATCTATTGGCTGGTCAGGGTGAGTCACTGGCCCTTTCTGGGCCTTTGGCCCATCGCGACTGGGGTGACAGCTCTGCGCAAGTCTCGTTAAACTTTGATCTCAATTGATATATAAAAAGGCAATGCTAAGTAGGGATTGTCATGTTTTATGCTAGTTACTAAGCCACTCGGCCAGGGTTCGAGTAAGACCATACACTGGTCCCAAATAATCCATCCTTACTAGTGTGCATCCTATATACACTTCGTGAGCATTTAGAAACACGAGCCCCCAAGCTAGGGCACAATATCGTATCTTGGTATTCGACTACGTTCACGGGTCTGTACCGCGTCATACACTCCTCCCGCAACCGGCTTAAGCTAGGAACCAACAATTGAGATAAATTGACCTAGGCAAGACAGTCGGACGAATGTCAGATTCAATCATTTTCGTAATCGAAAATACCATGTGCGAACACATTTCACACATTCACTACGTTGTCGTTGTTTGTCTCTAACTTACTTTGTGAGTTATGAAGAGTGAGGTAACAATGCCAAAACCGAACCAACTCGTCATGGATCCAGTGCGTGAAAAGTAGTGGGACGGAGGGAGCCCACTAGCTTTTAAAGAAGCTATGTATAAGCAGCTTCCGTCTAGGGTTCTCGCGTTTTCTGTAGCAACCTTCAGAC");
        omap.put("G", "AGACATCTACGCTACAGTTGCTCACCGCCACAGGGGTTGGGTCCAACCCCAGTATGTACCTCATTCGGCAATATGGGTATAACCCCAGGACCAGGGATAGTCCTCTCTGGTCAGAGCCCTTTGTACAACTTGTATTCGATTGTTAAATACAATGGAGCTCGCTTTATGGCCTCGGTGAGTCATTTTTATTCGACCCTAGGCGACTCATTCACTTTATTATAACGTGGCCCAGAGCAGTACGCGTTGGCGCTCAAGGCGGCTAAACTTGCTCTGTGCACGGGTGATAAACGGCTGTTCTCGGGGCTGAAGCTGATCTCTTAGAGTTAACCTATAAACCCGGTTACGACGGGAAAAGTTATTCCCGATCATTGTTCTTGGACTCTAAACACAAAAACGATGCAGGCTGACATCGCGCTGAGCAGAAATGGAGCACAATACTAAGTAAATTGTTTCTCGAGGACGTAAAACACTTGGCCATGGATGTTGCCATATTAATCTATCTTAAGAAAGTAGATGGAAGTGCGAGAAGATACTGCAGGCAATTCTCCCTCTTTCGAGTAGTCACGCCCAGATAGACTGGGACTCTCCCTTATGGTCTGGCCCGGACTAACGTCGGATCCATAAAACGTCATTTGGCCCAGGCCCAGCAGGTTAGAGACACTCTACACACTAACTTGCATGCGAGATCGGAACTGCCAAACTAGGGGTGACAATATCACCCCACATTTGTATTCTAAGAAAGTACTTACTTCGCTAGGTGGGGGCCTCCATAGCGCGACCACACTTGCCCTCATCTGGGTATACACGCCCACGAGCTACAACAGGGTGGAACGTTGCCATTGAAGATAGCACATAGGTCGACTTGCAGTGACCTTGTGTGCATTTCTAGAGGGTACGCGAGACCAGAGCTCCGTAATCAAAGAGCTGAGCAGAGGATAGCCGGTTCCTATGTGCAGAATGAGTTCGTGCCTGCTTGAGGATCATTAACTATAGTTGGCAGTATTCAACAGATAGCTAGATTCCTCCGCAGTATGTGGCGATGGGGAGTGTTAAATGCGGACTTTATTGGTGTGTAGATGCTGACATAGAGCTGAAGCGTGGATTTCGTCTAGCCTAGCTGTGCTCTAATGATTCAGTTGCGTGTGTGCGCTCCTGCGGATGGTCCCCGTTGTCTTAGTACTGCGCCCAACCAGACGAAGTGAGTAGCAATCGATCCTGCGAAAGTAAATTCGCTGTGACCAAGATGGCAGATTGTGGTAACATAAAGGTTCCGATGATAGTGTGCCAGTGAGCTGCTCTTGTCGTAATGGGCTGCCAATCGTCAACTACGCTCAGAACCAAAAAGGGGATTACTTAGTATCAGCAACTCTGATATCGAGCCCTGCAAGTCGAATTCTTGTGAACCGTCAGCTGTAGGCAGAATCGGTTTATAGGCTGAGTCTACATTAGTGCTGACTGGTTAATGTAGGCCTGGCCTAAGACCACAGTGATTAACGTTGCTCTGGCCCCACAACAATCCGTACCGAGACAGAGGATCCGTACGGACAGATTATTCATTCCCCTCCGCGGTCGCATAAAAGCGTTCCTTTGTGACGATTCTACAAGAGCCAGTGCGTGAGGTGGCCGATATATGTCCTTTTGCCAAACACCTTGACCTTGCGGTCTCTGAACGGGAAGGCCGCGTTGCTGCCTATAGATTCACGAGACTCCAGTCTTGCGAGATCAAATCTGTACGTTGCCCGTTGACGTTTAAAACCGCTAGTGAGCCCGGGAACGCGCCTCATAGCTGTGCCCCGAAGATATGGTATTTTTTAATATCCCAGGTCGCGTCCAATCGTCGCTACTCCTCTCAACCGGCGATTGCGGGATCATGTTATCGTAACGTGAACGGCTCGACTTTGTGATCCAGGCTTAATGTACTCTTCAGTCCTGTTAAGGGCTTCAAAGCACAACAATCAAACCCTCATCGATTCTTCACATCAGAACTACGGGCCATCTGGGTATCTCACTAACCTAAGTGACTTGACAAATGTCTCCGAACCGGATACATAGAATGCGCTAGTCCGAAAGCAGGTTATCCCAGGTGTAGACGACTACATTCGACCACTGATAGCTCGACTCAGACTCCCCTCCTACAGTTGGGGGAGGTCGTCCAGTCCGTTATTATGTCTGATCGAATAGTACTGCCTGAATTCCTCCGCTCCTTTAACTCCCTTTTTCACTCCGGAGTTGTCAAATTAGTGTTCGCCCACCCTGGGTCTCACTTATACCACTGTTTATGCTGATTGGGCATAAATCTCAGAATAGGGATTGTTTGGGAGGAGAGTACTACTTCACTTAGGGGTCAGTGAATGCAATAGAACAATCTACCGGACAGCTCGTATGTTTAAACCTGACGTTCACTCAAACATATGGCCCCGGTAAGGTTGGACTGGGCTTTAAAAGACCTTAGAACTAGCAATGTCTGACTTCGAAATGCCAGCCTCTATTTAGTCGTTACTATGAAATTTGTACACTACACACTGAGTGCACTCAAAGCGTACACCCGTAGCATGCCTTCGACCCAGTGAATCTTGTCACTCTAATTGCGACAATACGAGTGCCGCTAGCGCGACGGAGTAAAGCTTGAAGTCCTTCAGCCCTACAATTATTACGCGTCCCATCCATGCCAAATGGCACAGATAACTTGACTAAACTCATCATAGATTCGTGCAAGGTGCCACTAGGCGGGCGCTATTAAAGTAGCGAGCATCCTGTGAATCTATTCGCGTCCTTGTGTCAGAATTCGTTATAAAAATCCACTGCTCCGCACCCGCATCAACGTCAATCGGTGGACTACCTAGTGTGCGTAATAGACTTTTTTTGTCCTTTTATGATAGCGGTCCTAGGCGGACTAACACATACACTGTGATATATTACGTAGGGTGCCAAAGATCACGTTGAGCCAGACAACCTCCCAGTGCTTATGCCTCCTTCGTCGTACATAAAGAGAGGGCGTCTGGTCCATGAATAGCTTACTGTATGCAATACATAGGATCGTGATCGGGGCTCCCAAACAGCCGCGCGTTTGCCGTTCCTGGAATTCTAAGGTGGGGCCCCTAGTACCGGTGCATATCAATGGTCACCAATATAGGAACACTCAACTAACGGGCGTCGGTGTCCAATAATAAGTGTCTACTTTAGCCCTGGTAGAAGGACGGAGGTGATATCCGCTTAGGCGTCGATCGAACACTGGCGGACCTACGTATTCCGTGAGACCAACGAGCGAGACTCAGGTCGGGCTAACGATTGCGCTCATATTGGAACTGGCCTGCTTTAGGGCGCGGCCTGCCTTGCAGGAAGTGCCACTAGCGATGTGGCCAACATGTTTTAGGCTGGGTCGGACCGAAGGCCCGCGATTATCTCGACATAGTAGGAAGGTAAGTTGACACCGCGGGGCCGTTTGGAAACCCAACGCCCTTGCGCCCTCGGCTTCTACAACAAGCGCGTCGTACGCTGGGCACAAGTTTAGATAGTGTATGGGTGCCAGAGGCAAAGACCGTTCTGCTTGGTATTGCGCGATCGGCCTAACTGGGGCGAGGGCTGTCAGGAAGATATTGCAATCGTGAACGTACTAGTCGAGGCATAATCGCCAGTTATGAAACTGCACTGCTGTAATGTCGAATGCGGACTCTAACCGCCCTCGATCTATGAGATGTCCCTTACTAAAACGGGTGTGTGGATCGCTGCAGGTCAGAGTTCCTGAGTAGGACAATGTACCAAAACAATCGGGGCGTTCCCGTAGTAAAGCTCATTTACCTGGACATGTGGCGTTTCAATTTCGGCCACGAAAGATGATGAACATGACGACGCTAGGCATGGCCGAAATTCACCTATTCCATGGCTGGACTCCGATCTACGGATGGTCCAATGCGACTACCCTAAGCGCAGCAGACGGTCGTTCGTCCTTGTGGTATTGAAAACGATACCATCGCCATAAACTGCCGTTCACCCCCACTAACTAAGCTGAGCCGTCCTGCCGAAGATGAAAACTCGACTTAGGGAAGAACAAAAAACTAGCAGCCGATATATGCATTATTACCCGAGCCTTATCAATTCTGTTGGGAAATATATTGGGCCGAAGTCCCACTTAAGGCACCTAACGAGAATGAGGGTGTTATCGGCCTTGTCGGGTGCTCATGACCAAAGTCGCATGGATTGAAGCACTCGAAACGGCTAAGCGAAGTGAATGGCCACGATGTATTCCAGATCTTCAGTGCCCCAGTCTGGTAAAATCGCCAGTTACCGTGTTCGACCTATCACGCGCGTTACCAGTGTCTCTCCTACTACCAATAATGAAATGCTTGTACACAGTGCTTGTATCTGATTCGGGCCCTCGGCATGCCTCTACACTATACTGGAGGTCGAGTTACCAGGTGTAGGCCCAAGAGGTCAAAGGTCTTCTTTCGACCTGAGTCGGGTAAGGTATGCTGGGGACGGCCTATTCGGTTGTCGGGTGACCAATATTATGTGATAGCCATCCTGTCGCGTTTGCTGAACCAATAGGGTGCTCCTATGAGCAATAAGCGCGACGGCAGTAAGACACTTAGTTTAGCACGGAACTTCTCAGCTGGCGGAGACATGTTATCGATACACATGTCTTCAGGGGCAAGACGCGCTCCAGCTACAGCCTTTGAGATCACATTCACATCTGTTATCGATAAACTTGTAGCTAGCGCCCTACAGAATCCGCTGCTTCTGGTGATCCCTTTAAGACCTAAGGGTATATTCGTGCCCCATGATACCTAGCTATGGGACCCAATCTCTATTGTACAGCGACTCAACGTTGGTGTACATCACCAATCCATGCGCGCCTTGTCAAGGGGACTAGTTTAATGAGCGGCTAAATCTGCCACAAGCAATTAATGTTTGATAAACGCCCAGTGTGTGACCTATGTGAGTGTCGCAACCCTAATGTTCGGTAAGTCTTCGTCCGCGCGGTAGCGTGATACAATTAGTTTGGGTTCTACATACAATACCAAAGTAGTCGGGGCTACGGTATGGACGCTCATACTCCTGCAACATGGGCTACCGAGCAGTTCTAGGCTGTGTCATCCCCGGCGGGGCTCGACACAAATTCGTCACGGACGAAGCTGTACAAGCCGAACAGTTGAGTGACTCGGTAATTTATGAGACAGAAGCGCCAGTATATAGGTACCGTGCATGACTCTGGCCTGGGCCGAGCATGTCTTTGTAATACTTACCGCCAACTCTAGCGGAGAAATGAGGTAACGTGGTTCACGATACATAATCACTCTTTTTCGCTGACGTAATGAGGCTCGCGTCATGGAATCAAGAAAATGTAACAACCTGATAGGACTACGTGGAAGTGCCGTTAATAGGGCCCGGAAGAACCTACTGTGTCAGTTTTATCCAGCTAAATTAGACGTGCGAATCAATCTCATATAAATTGCGAACGCCCCAGTTTCATTTCTACTAACAGAATAGCGTTTGTCATCATTACTCACTAAGAAGGTGATCTAATGTGGAACACCACCGTTCTAGGGAAGTTCTACCCCCAGGGCGATCTTGCTAATAAGTGATGGTTGGGCACCCTGGGCCTGCTAGACGGGTGAAGCGTGGAGTTGATTATACGTACTTTCCAACTTTCGAAGTCATAAGTCGCTTTAGGCGCTTGACAAATGAATGAGCACCCCAATTCAGTTGGGTCTCGGTGTTTCGGGACGGGTTACCTTCGAGTTCGTGGTAGGTCCGAAAACCCTAAAGTTTATGGATACGAAGAAGGGAATGGAGCCTATGGTAATACTTTTTGAGAAGATTGTAATTATGCAGTCAGACGAGGCGAGCGTTTGATATGTCATCGTGTCTTCGACTTTTAGTGAGCAATAGGATTCCTGATTTCGACCCTAGTATGAACTGTGTCTATATCAGGTGGGATACAAAATATGCAAAGACGATGGTCTCTTGTAATGTCTTGGGTCAAGACCTACTGGCACCCGTATACAGGTACATACCCGAACAGTGGGTTCTGGTCTTGCCCGAGGCTTGTATCCCCGTAGCCTGCCTAACTAACAGACTTTCCGATAACCACCGATCTTGAATTACTAGAGGCCGCGGGTCTTTCACTACTCGGTCAGGCGATGATGGTAGTCCCGTTGCGCTCGGTGAAATCAGGACCATTCACTCGTATTGAAAAGCTAACCTTTATCGCGGCGACACGAAGTACTGGCTGCACAGTGGTAGTGGAGTGTAAACAGGTATCTCGTATTGACCGGCTCAGACTCACCCTCGGCCGCCGTGACTTTCCAGGTTTGCATGTGCCCGATATGTGCTACGGTCTCCAAGGGACGAAGAGAAATATGCGACGGGCTCGTCCGGCCATAAGGACCGCAGTATAGCTACAATTGATCTAATAACCTGATCGGAGCCGTGTTGAATTTGGACGAGTCGAAAATCCCCGACTAAGCTTGCTTATCGTTTGTCGCTTAATACTAGGGACTTTTCATGCACATTCTACATCAGTGCACGACTAAGACTACTCCCTGACTGGATCAATACATTTAATACTCGTGCTCTGTCATTGCCAATGGCCTTTGCTCCTGTCAATTGCTGGTCGCGGATCAGAAACCAGATCGGGCCGCTTGCGGGACGGGGCCCATCAGGATCCAAGTGACAAGAAGCCGTAACCGCACTAAGGATAAGAGCACTTGCAGGCGCCGCTCTTAACGTTCTATAAGCCCGTCAGTGCCAATTGGTTTAAAGGTGGGCGGCTCCGTTACAGGACTAAGCATCTCAAGCTCTGCAAGCGATGCTCCGGATCGACGGATCATTTGGATGATTACATTTCTCCTAGAGCACAGTCAACCATTAGCCAGTTCTCCTCGACCGATAAGCCTCCAGAACTATGACTGATTTTGGTAGATTAAGCGCCAACGCTATTTGTATTGTTTGAGCGCTTAGCCACTCCGTGTACATGGAACGAACGTTGTATTCCGCTGAAGACAAAAACCTGCGTTCATCTAGGGCTAGCCGTGACCTCCAGAGACGGACGCTAGCTTTTGTCAAGCGAAGCGCACGGTCGGTGCTGCTACACACGCTCCAATGACAAATTGATTGTGTCTCTAGTGCCGTTTTTGTGGGTCCCCGACCCTTATTTATAGGGGTGGGAATTCTCGTGCCAGGCTCCTTGTCTATCAATTTTGGAGAGCAAGCTAATGATGCAGACCTATCTTTGTGGTGCTGACGAAAACAGCCGCGCCGCCTTCTTCATTTATGTAAGTTCCTTGGGAATGGGATAGAGTAGTAGCTCTTTCCAGACAAGAAGCAATGATATAAATTATGCTGTCAGTACTTATCTGCACGACGCGAACTTCCAATTAGAAGCTTACCGCGTGCGATAATGGGATTCAAGACACTGCTATACTTCATACTTTATATACGAAGCTCGTGGTCATGATGATGTTCCAGTATCCATATAAACCCTGGGCTTAGAGCCCTTTTAGACAATTTCGACTGGCTATTTCTTCTCTGTCTTACCCCTGACGACTAATTCGACATGGGTACTCCACCCCCGTCTTAAATAGTAGGCGCCACTAAAGGATATCGCGAGGCTCCCGGAGTGCCATCCTCCCGACTCCTAAATCCGGCCCGTCCTAGAACGCCGAATTCTCCAAATGTAGATACGAACTCCGGTCATCGTGGAGGTTGGTTATCCCCGCCTCGCGCCTGTTCGCCAACGTGCGAGTTATTAAAGTCTTTCGAGGTCATAGACCTCATATGGAAGCATATCATCAATAGCATGCCCCGGATATAAAAGTCTTAAAGGGACATCCACCCTCAGGAGTTCCGCCAATCATGTACAGTAGCTGGATCGCTACTATGGATTGTCTCCGCCCCCATGTTTCACGAACCGACCCCCCAGTGGGACTGGTTTCGTCTCATCCGTGTAAGGTATCGGTAAAACAACTCGACTTAGAAAGTTCTCCTCGAATCGAATCGTTGGATTTCCCATGGGGATAGGGTACGTTGTACTGACCTCGTGTTCTACCGACAACCGTAAGCCGTGTGTCCCCCGAGAGAAAGTACTCAGGTCTAGGTCGACGTGCCCCGTTAACGAGCAAAAGCCTTACGTAACACGCCCATACAACAATCACCGTAATCATCCTGGGCTGTGTGATCATAATTTTCTAGTGCGCTGGCTGCCCGATGTACGCGGTGGCCCCAAAATGAGTGGGCCCACATGCTCTGGACGGACAATCGTCCACTGAGGCGCAGGCTCTTTAACATCCTTTAACCTCAAGTCTTGCGTTCTATTCGTCTACGACCTTTGAATGAAAATCCGCTACGCACTGGGCGACTCTGAGGAGCAGTCGGAGGCATGTGTCAATGCGTGGCCTACTCTAGCGGAATGGTTAAGTCTCGAAGTTGGTTATACTTAAATGGTACACAAGTGCAATTGCCTACTTACATGTAGCTAATGGATATCACAGGTGTGGCGCAACACTACCTCGCCTCTTACAGCTGGTACTCTTGAACGTACCGTGCTATTTCGGTTTTTTATATTTGCACAAACGGCTCCAGAACTATATGCTTCATGTTAAACCATTATGGGGGTCGACCGTACATCAACGTTAACATAATAACGGGACGGGTCTTCAGATATGGTCTCAAGGTGTAGCCGACACCATCCATGTGGACATAATTGGCGGCTGTCTTTGTCCTATCACAACTGTCGAATTAGCAACTATCTCTCGTTCGGAATGGAGGATGAAAATGATGTTGGTTTTTATAGCTAGCAACATTTCGCTCTAAAACTATGGACAGAATCCTTGCACTGTAAATTGGGCACGAGACAGTGATATGTGGCGTAATGAAAACCAGCGAAGGACCTCAGTGAAAGTAACGATACCTTGAAAAGTGGATCACCGTTTTCGACCGTCTGACTAACGGATGACGAACACGGGCGTTCCTTCTACAACTTGCCGCAATGTCTCAGCCTCGGGACATGTCTAAGACAATACGGAGATACTCATCGATCGTCAAGGCCATGGCCGGGACGCTTATAACTAACACTACGTACCCCTTAAAGCTCGTCACGCATGTATCAACTTTTTATGTTAAAGCATGAGGCATGCCTCGAGCGGTCTGCGCGCTCAGCTTATAAGGAAACGCCTTACAAAACTAGTACGATTCGGTGTGTCTAAGGCGCATTGTCTATGCTCCGCTATGGATTTTTTGAAGGGGCTTTATGTTACCCATTCTTACATACTGACGCAGTGTCCAAGTGTGCGACGGGTGCCATCACGGCGGATTCCGTATCTATAACCGGGACGATACTCTATTGGCTGCTCAGGGTGAGTCACTGGCCCTTTCTGGGCCTTTGGTCCATCGCGGCTGGGGTGGCAGCGCCGCTCAAATCCGGTTGAGCTTTGCTTTCGATTGATATCTGAAAAGGCAATGCTAAGTTGTGATTGCCATATAGTATGCTAGTTACAAAGCCACTTGACCAGGGTTGGATTAAGACCATACACTGGTCACAAAAAATGGATCCTTACTAGTGTGCAGCCTATATACAGTTCGTGAGCATTTCGATACACGAGCCCCCAACCTAGGGCACAACTTTGTATTTTGGTATTCGACCACGTTTACTGGTCTGAACCGGGTCAGACAATCCTCCCACAACAGACTTGAGCAAGGTATCAACAATTGATATAAATAGTCCTTGGCAACCCAGTCGGACCAACGTCAGATTCAAATATTTTCGTAATCGAGAATACGATGTGCGCACTCATTTCACACATTCACTATGTTGTGGGAACTTGCCTCTAACTTACTTTGTGGGTTCTGAAGAGTAAGGCAACAAAGCCATAACCGATCCAACTCGTCATGGATCCAGTGCGTGAAACGTAGTGGCACGGAGGGAGCCCACTAACTCTTAAAGAAGCTATGTACAAGCAGTTTCCGTCTAGCGCTCACGTGTTTCCTGTAGAAAGCTTCAGAC");

//        List<Alignment> data = new ArrayList<>();
//        for (int i = 0; i < omap.get("H").length(); i++) {
//            Map<String, String> columnMap = new HashMap<>();
//            for (String key:omap.keySet()) {
//                columnMap.put(key, Character.toString(omap.get(key).charAt(i)));
//            }
//            data.add(new Alignment(columnMap));
//        }
        Utils.DATA = new Alignment(omap);

        int numIter = 10;

//        List<Double> trueLikelihoods = new ArrayList<>();
//        for (int i = 0; i < numIter; i++) {
//            ModelTree trueModel = ModelBuilder.getHCGModel();
//            HmmBuilder builder = new HmmBuilder(trueModel.getTree(), trueModel.getRecombRate());
//            HmmCore hmm = builder.build();
//            double likelihood = hmm.logLikelihood();
//            System.out.println(likelihood);
//            System.out.println(hmm.getNumStates());
//            System.out.println(Arrays.toString(hmm.getPi()));
//            trueLikelihoods.add(likelihood);
//        }
//
//        double avgTrueLikelihood = trueLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
//        System.out.println("=================");
//        System.out.println(avgTrueLikelihood);
//        System.out.println("=================");
//
//        // this model has wrong population size and divergence time
//        ModelTree badModel = ModelBuilder.getHCGModelBad();
//        HmmBuilder builderBad = new HmmBuilder(badModel.getTree(), badModel.getRecombRate());
//        HmmCore hmmBad = builderBad.build();
//        double likelihoodBad = hmmBad.logLikelihood();
//        System.out.println(likelihoodBad);
//        System.out.println(hmmBad.getNumStates());
//        System.out.println(Arrays.toString(hmmBad.getPi()));
//        System.out.println("!!!!!!!!!!!!!!!!!!!!!!!");
//
//        // this model has wrong topology
//        ModelTree badModel2 = ModelBuilder.getHCGModelWrongTopo();
//        HmmBuilder builderBad2 = new HmmBuilder(badModel2.getTree(), badModel2.getRecombRate());
//        HmmCore hmmBad2 = builderBad2.build();
//        double likelihoodBad2 = hmmBad2.logLikelihood();
//        System.out.println(likelihoodBad2);
//        System.out.println(hmmBad2.getNumStates());
//        System.out.println(Arrays.toString(hmmBad2.getPi()));
//        System.out.println("!!!!!!!!!!!!!!!!!!!!!!!");

//        // this model is slightly wrong
//        List<Double> wrongLikelihoods = new ArrayList<>();
//        for (int i = 0; i < numIter; i++) {
//            ModelTree slightlyWrongModel = ModelBuilder.getHCGModelSlightlyWrong();
//            HmmBuilder slightlyWrongBuilder = new HmmBuilder(slightlyWrongModel.getTree(), slightlyWrongModel.getRecombRate());
//            HmmCore slightlyWrongHmm = slightlyWrongBuilder.build();
//            double slightlyWrongLikelihood = slightlyWrongHmm.logLikelihood();
//            System.out.println(slightlyWrongLikelihood);
//            System.out.println(slightlyWrongHmm.getNumStates());
//            System.out.println(Arrays.toString(slightlyWrongHmm.getPi()));
//            wrongLikelihoods.add(slightlyWrongLikelihood);
//        }
//
//        double avgWrongLikelihood = wrongLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
//        System.out.println("=================");
//        System.out.println(avgWrongLikelihood);
//        System.out.println("=================");

//        List<Integer> hcsize = new ArrayList<>();
//        List<Double> likelihoods = new ArrayList<>();
//        for (int size = 10000; size <= 100000; size += 10000) {
//            System.out.println(size);
//            // this model has wrong HC population size
//            List<Double> wrongSizeLikelihoods = new ArrayList<>();
//            for (int i = 0; i < numIter; i++) {
//                ModelTree wrongSizeModel = ModelBuilder.getHCGModelWrongHCSize(size);
//                HmmBuilder wrongSizeBuilder = new HmmBuilder(wrongSizeModel.getTree(), wrongSizeModel.getRecombRate());
//                HmmCore wrongSizeHmm = wrongSizeBuilder.build();
//                double wrongSizeLikelihood = wrongSizeHmm.logLikelihood();
//                System.out.println(wrongSizeLikelihood);
//                System.out.println(wrongSizeHmm.getNumStates());
//                System.out.println(Arrays.toString(wrongSizeHmm.getPi()));
//                wrongSizeLikelihoods.add(wrongSizeLikelihood);
//            }
//
//            double avgWrongSizeLikelihood = wrongSizeLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
//
//            likelihoods.add(avgWrongSizeLikelihood);
//            hcsize.add(size);
//        }
//
//        System.out.println(hcsize);
//        System.out.println(likelihoods);

        List<Double> rates = new ArrayList<>();
        List<Double> likelihoods = new ArrayList<>();
        for (double rate = 0.05E-8; rate < 3.0E-8; rate += 0.05E-8) {
//        for (double rate = 3E-7; rate <= 3.0E-7; rate += 0.05E-8) {
            System.out.println(rate);
            // this model has wrong recomb rate
            List<Double> wrongRecombLikelihoods = new ArrayList<>();
            for (int i = 0; i < numIter; i++) {
                ModelTree wrongRecombModel = HCGModelBuilder.getHCGModelWrongRecomb(rate);
                HmmBuilder wrongRecombBuilder = new HmmBuilder(wrongRecombModel.getTree(), wrongRecombModel.getRecombRate());
                HmmCore wrongRecombHmm = wrongRecombBuilder.build();
                double wrongRecombLikelihood = wrongRecombHmm.logLikelihood();
                System.out.println(wrongRecombLikelihood);
                System.out.println(wrongRecombHmm.getNumStates());
                System.out.println(Arrays.toString(wrongRecombHmm.getPi()));
                wrongRecombLikelihoods.add(wrongRecombLikelihood);
            }

            double avgWrongRecombLikelihood = wrongRecombLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;

            likelihoods.add(avgWrongRecombLikelihood);
            rates.add(rate);
        }

        System.out.println(rates);
        System.out.println(likelihoods);

    }

    public static void test1Msites() throws IOException {
        Map<String, String> msName2SpeciesName = new HashMap<>();
        msName2SpeciesName.put("1", "H");
        msName2SpeciesName.put("2", "C");
        msName2SpeciesName.put("3", "G");
        // Read file
        Map<String, String> omap = new HashMap<>();
        File file = new File("/Users/xinhaoliu/Desktop/Research/Scripts/HMMtestdata/1recomb10mut/HCG_1000000/aligned.fasta");
        BufferedReader br = new BufferedReader(new FileReader(file));
        String line;
        String taxon = null;
        while ((line = br.readLine()) != null) {
            if (line.length() == 2) {
                taxon = line.substring(1);
            } else {
                omap.put(msName2SpeciesName.get(taxon), line);
            }
        }

        // Build data
//        List<Alignment> data = new ArrayList<>();
//        for (int i = 0; i < omap.get("H").length(); i++) {
//            Map<String, String> columnMap = new HashMap<>();
//            for (String key:omap.keySet()) {
//                columnMap.put(key, Character.toString(omap.get(key).charAt(i)));
//            }
//            data.add(new Alignment(columnMap));
//        }
        Utils.DATA = new Alignment(omap);

        System.out.println("finished building data");

        int numIter = 10;

        List<Double> trueLikelihoods = new ArrayList<>();
        for (int i = 0; i < numIter; i++) {
            ModelTree trueModel = HCGModelBuilder.getHCGModel();
            HmmBuilder builder = new HmmBuilder(trueModel.getTree(), trueModel.getRecombRate());
            HmmCore hmm = builder.build();
            double likelihood = hmm.logLikelihood();
            System.out.println(likelihood);
            System.out.println(hmm.getNumStates());
            System.out.println(Arrays.toString(hmm.getPi()));
            trueLikelihoods.add(likelihood);
        }

        double avgTrueLikelihood = trueLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
        System.out.println("=================");
        System.out.println(avgTrueLikelihood);
        System.out.println("=================");

        // this model is slightly wrong
        List<Double> wrongLikelihoods = new ArrayList<>();
        for (int i = 0; i < numIter; i++) {
            ModelTree slightlyWrongModel = HCGModelBuilder.getHCGModelWrongHCSize(20000);
            HmmBuilder slightlyWrongBuilder = new HmmBuilder(slightlyWrongModel.getTree(), slightlyWrongModel.getRecombRate());
            HmmCore slightlyWrongHmm = slightlyWrongBuilder.build();
            double slightlyWrongLikelihood = slightlyWrongHmm.logLikelihood();
            System.out.println(slightlyWrongLikelihood);
            System.out.println(slightlyWrongHmm.getNumStates());
            System.out.println(Arrays.toString(slightlyWrongHmm.getPi()));
            wrongLikelihoods.add(slightlyWrongLikelihood);
        }

        double avgWrongLikelihood = wrongLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
        System.out.println("=================");
        System.out.println(avgWrongLikelihood);
        System.out.println("=================");
    }

    /**
     * Simulated real data here assumes different recombination rates and mutation rates.
     */
    public static void test100000sites() throws IOException {
        Map<String, String> msName2SpeciesName = new HashMap<>();
        msName2SpeciesName.put("1", "H");
        msName2SpeciesName.put("2", "C");
        msName2SpeciesName.put("3", "G");
        // Read file
        Map<String, String> omap = new HashMap<>();
        File file = new File("/Users/xinhaoliu/Desktop/Research/Scripts/HMMtestdata/20recomb20mut/HCG_100000/aligned.fasta");
        BufferedReader br = new BufferedReader(new FileReader(file));
        String line;
        String taxon = null;
        while ((line = br.readLine()) != null) {
            if (line.length() == 2) {
                taxon = line.substring(1);
            } else {
                omap.put(msName2SpeciesName.get(taxon), line);
            }
        }
        Utils.DATA = new Alignment(omap);

        int numIter = 10;

        List<Double> trueLikelihoods = new ArrayList<>();
        for (int i = 0; i < numIter; i++) {
            ModelTree trueModel = HCGModelBuilder.getHCGModel();
            HmmBuilder builder = new HmmBuilder(trueModel.getTree(), trueModel.getRecombRate());
            HmmCore hmm = builder.build();
            double likelihood = hmm.logLikelihood();
            System.out.println(likelihood);
            System.out.println(hmm.getNumStates());
            System.out.println(Arrays.toString(hmm.getPi()));
            trueLikelihoods.add(likelihood);
        }

        double avgTrueLikelihood = trueLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
        System.out.println("=================");
        System.out.println(avgTrueLikelihood);
        System.out.println("=================");

//        // this model has wrong population size and divergence time
//        ModelTree badModel = ModelBuilder.getHCGModelBad();
//        HmmBuilder builderBad = new HmmBuilder(badModel.getTree(), badModel.getRecombRate());
//        HmmCore hmmBad = builderBad.build();
//        double likelihoodBad = hmmBad.logLikelihood();
//        System.out.println(likelihoodBad);
//        System.out.println(hmmBad.getNumStates());
//        System.out.println(Arrays.toString(hmmBad.getPi()));
//        System.out.println("!!!!!!!!!!!!!!!!!!!!!!!");
//
//        // this model has wrong topology
//        ModelTree badModel2 = ModelBuilder.getHCGModelWrongTopo();
//        HmmBuilder builderBad2 = new HmmBuilder(badModel2.getTree(), badModel2.getRecombRate());
//        HmmCore hmmBad2 = builderBad2.build();
//        double likelihoodBad2 = hmmBad2.logLikelihood();
//        System.out.println(likelihoodBad2);
//        System.out.println(hmmBad2.getNumStates());
//        System.out.println(Arrays.toString(hmmBad2.getPi()));
//        System.out.println("!!!!!!!!!!!!!!!!!!!!!!!");

//        // this model is slightly wrong
//        List<Double> wrongLikelihoods = new ArrayList<>();
//        for (int i = 0; i < numIter; i++) {
//            ModelTree slightlyWrongModel = ModelBuilder.getHCGModelSlightlyWrong();
//            HmmBuilder slightlyWrongBuilder = new HmmBuilder(slightlyWrongModel.getTree(), slightlyWrongModel.getRecombRate());
//            HmmCore slightlyWrongHmm = slightlyWrongBuilder.build();
//            double slightlyWrongLikelihood = slightlyWrongHmm.logLikelihood();
//            System.out.println(slightlyWrongLikelihood);
//            System.out.println(slightlyWrongHmm.getNumStates());
//            System.out.println(Arrays.toString(slightlyWrongHmm.getPi()));
//            wrongLikelihoods.add(slightlyWrongLikelihood);
//        }
//
//        double avgWrongLikelihood = wrongLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
//        System.out.println("=================");
//        System.out.println(avgWrongLikelihood);
//        System.out.println("=================");

//        List<Integer> hcsize = new ArrayList<>();
//        List<Double> likelihoods = new ArrayList<>();
//        for (int size = 20000; size < 30000; size += 1000) {
//            System.out.println(size);
//            // this model has wrong HC population size
//            List<Double> wrongSizeLikelihoods = new ArrayList<>();
//            for (int i = 0; i < numIter; i++) {
//                ModelTree wrongSizeModel = ModelBuilder.getHCGModelWrongHCSize(size);
//                HmmBuilder wrongSizeBuilder = new HmmBuilder(wrongSizeModel.getTree(), wrongSizeModel.getRecombRate());
//                HmmCore wrongSizeHmm = wrongSizeBuilder.build();
//                double wrongSizeLikelihood = wrongSizeHmm.logLikelihood();
//                System.out.println(wrongSizeLikelihood);
//                System.out.println(wrongSizeHmm.getNumStates());
//                System.out.println(Arrays.toString(wrongSizeHmm.getPi()));
//                wrongSizeLikelihoods.add(wrongSizeLikelihood);
//            }
//
//            double avgWrongSizeLikelihood = wrongSizeLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
//
//            likelihoods.add(avgWrongSizeLikelihood);
//            hcsize.add(size);
//        }
//
//        System.out.println(hcsize);
//        System.out.println(likelihoods);

//        List<Double> rates = new ArrayList<>();
//        List<Double> likelihoods = new ArrayList<>();
//        for (double rate = 0.05E-7; rate <= 6.0E-7; rate += 0.05E-7) {
//            System.out.println(rate);
//            // this model has wrong recomb rate
//            List<Double> wrongRecombLikelihoods = new ArrayList<>();
//            for (int i = 0; i < numIter; i++) {
//                ModelTree wrongRecombModel = ModelBuilder.getHCGModelWrongRecomb(rate);
//                HmmBuilder wrongRecombBuilder = new HmmBuilder(wrongRecombModel.getTree(), wrongRecombModel.getRecombRate());
//                HmmCore wrongRecombHmm = wrongRecombBuilder.build();
//                double wrongRecombLikelihood = wrongRecombHmm.logLikelihood();
//                System.out.println(wrongRecombLikelihood);
//                System.out.println(wrongRecombHmm.getNumStates());
//                System.out.println(Arrays.toString(wrongRecombHmm.getPi()));
//                wrongRecombLikelihoods.add(wrongRecombLikelihood);
//            }
//
//            double avgWrongRecombLikelihood = wrongRecombLikelihoods.stream().mapToDouble(x -> x).sum() / numIter;
//
//            likelihoods.add(avgWrongRecombLikelihood);
//            rates.add(rate);
//        }
//
//        System.out.println(rates);
//        System.out.println(likelihoods);

    }

    /**
     * Simulated real data here assumes 20recomb20mut
     */
    public static void test100000sitesPlotA() throws IOException {
        Map<String, String> msName2SpeciesName = new HashMap<>();
        msName2SpeciesName.put("1", "H");
        msName2SpeciesName.put("2", "C");
        msName2SpeciesName.put("3", "G");
        // Read file
        Map<String, String> omap = new HashMap<>();
        File file = new File("/Users/xinhaoliu/Desktop/Research/Scripts/HMMtestdata/20recomb20mut/HCG_100000/aligned.fasta");
        BufferedReader br = new BufferedReader(new FileReader(file));
        String line;
        String taxon = null;
        while ((line = br.readLine()) != null) {
            if (line.length() == 2) {
                taxon = line.substring(1);
            } else {
                omap.put(msName2SpeciesName.get(taxon), line);
            }
        }
        Utils.DATA = new Alignment(omap);

        int numIter = 10;

        String[] names = new String[] {"HC1", "HC2", "HG", "CG"};
        List<Double> rates = new ArrayList<>();
        ArrayList<Double>[][] transitionRates = new ArrayList[4][4];
        for (int i = 0; i < 4; i++) {
            for (int j = 0; j < 4; j++) {
                transitionRates[i][j] = new ArrayList<Double>();
            }
        }
        for (double rate = 2.0E-7; rate <= 4.0E-7; rate += 0.05E-7) {
            System.out.println(rate);
            // this model has wrong recomb rate
            ArrayList<Double>[][] AForThisRate = new ArrayList[4][4];
            for (int i = 0; i < 4; i++) {
                for (int j = 0; j < 4; j++) {
                    AForThisRate[i][j] = new ArrayList<Double>();
                }
            }
            for (int i = 0; i < numIter; i++) {
                ModelTree wrongRecombModel = HCGModelBuilder.getHCGModelWrongRecomb(rate);
                HmmBuilder wrongRecombBuilder = new HmmBuilder(wrongRecombModel.getTree(), wrongRecombModel.getRecombRate());
                HmmCore wrongRecombHmm = wrongRecombBuilder.build();
                double[][] AInOrder = wrongRecombHmm.getAInOrder();
                for (int ii = 0; ii < 4; ii++) {
                    for (int jj = 0; jj < 4; jj++) {
                        AForThisRate[ii][jj].add(AInOrder[ii][jj]);
                    }
                }
            }

            for (int i = 0; i < 4; i++) {
                for (int j = 0; j < 4; j++) {
                    transitionRates[i][j].add(AForThisRate[i][j].stream().mapToDouble(x -> x).sum() / numIter);
                }
            }
            rates.add(rate);
        }
        System.out.println(rates);
        for (int i = 0; i < 4; i++) {
            for (int j = 0; j < 4; j++) {
                List<Double> ratesForThisTransition = transitionRates[i][j];
                String transitionName = names[i] + "->" + names[j];
                System.out.println(transitionName);
                System.out.println(ratesForThisTransition);
            }
        }
    }

    /**
     * Simulated real data here assumes 20recomb20mut
     */
    public static void test100000sitesPlotPi() throws IOException {
        Map<String, String> msName2SpeciesName = new HashMap<>();
        msName2SpeciesName.put("1", "H");
        msName2SpeciesName.put("2", "C");
        msName2SpeciesName.put("3", "G");
        // Read file
        Map<String, String> omap = new HashMap<>();
        File file = new File("/Users/xinhaoliu/Desktop/Research/Scripts/HMMtestdata/20recomb20mut/HCG_100000/aligned.fasta");
        BufferedReader br = new BufferedReader(new FileReader(file));
        String line;
        String taxon = null;
        while ((line = br.readLine()) != null) {
            if (line.length() == 2) {
                taxon = line.substring(1);
            } else {
                omap.put(msName2SpeciesName.get(taxon), line);
            }
        }
        Utils.DATA = new Alignment(omap);

        int numIter = 10;

        String[] names = new String[] {"HC1", "HC2", "HG", "CG"};
        List<Double> rates = new ArrayList<>();
        ArrayList<Double>[] Pis = new ArrayList[4];
        for (int i = 0; i < 4; i++) {
            Pis[i] = new ArrayList<Double>();
        }
        for (double rate = 0.05E-7; rate <= 6.0E-7; rate += 0.05E-7) {
            System.out.println(rate);
            // this model has wrong recomb rate
            ArrayList<Double>[] PiForThisRate = new ArrayList[4];
            for (int i = 0; i < 4; i++) {
                PiForThisRate[i] = new ArrayList<Double>();
            }
            for (int i = 0; i < numIter; i++) {
                ModelTree wrongRecombModel = HCGModelBuilder.getHCGModelWrongRecomb(rate);
                HmmBuilder wrongRecombBuilder = new HmmBuilder(wrongRecombModel.getTree(), wrongRecombModel.getRecombRate());
                HmmCore wrongRecombHmm = wrongRecombBuilder.build();
                double[] PiInOrder = wrongRecombHmm.getPiInOrder();
                for (int ii = 0; ii < 4; ii++) {
                    PiForThisRate[ii].add(PiInOrder[ii]);
                }
            }

            for (int i = 0; i < 4; i++) {
                Pis[i].add(PiForThisRate[i].stream().mapToDouble(x -> x).sum() / numIter);
            }
            rates.add(rate);
        }
        System.out.println(rates);
        for (int i = 0; i < 4; i++) {
            List<Double> piForThisState = Pis[i];
            String stateName = names[i];
            System.out.println(stateName);
            System.out.println(piForThisState);
        }
    }
}
