## ---- options, include = FALSE------------------------------------------------
library(knitr)
opts_chunk$set(
    cache = FALSE, 
    results = "hold"
)

## ---- bcl-data----------------------------------------------------------------
suppressMessages(library(MERINGUE))

data(BCL) # object contains counts and positions for all 4 sections. Will only need sections 1-3
head(BCL$pos)
BCL$counts[1:5,1:5]

## parse out the 3 sections
BCL1.pos <- BCL$pos[BCL$pos$slice==1,][,c("x", "y")]
BCL2.pos <- BCL$pos[BCL$pos$slice==2,][,c("x", "y")]
BCL3.pos <- BCL$pos[BCL$pos$slice==3,][,c("x", "y")]
BCL1.counts <- BCL$counts[,BCL$pos$slice==1]
BCL2.counts <- BCL$counts[,BCL$pos$slice==2]
BCL3.counts <- BCL$counts[,BCL$pos$slice==3]

## ---- bcl-combine-------------------------------------------------------------
# Get common set of genes for the sections
genes.have <- Reduce(intersect, list(
  rownames(BCL1.counts),
  rownames(BCL2.counts),
  rownames(BCL3.counts)
))

# Combine into large counts matrix. Counts of genes for all the sections being assessed
counts <- cbind(
  BCL1.counts[genes.have,],
  BCL2.counts[genes.have,],
  BCL3.counts[genes.have,]
)

# section factor
section <-  c(
  rep('L1', ncol(BCL1.counts)),
  rep('L2', ncol(BCL2.counts)),
  rep('L3', ncol(BCL3.counts))
)
names(section) <- colnames(counts)

# List of positions
posList <- list(
  BCL1.pos[colnames(BCL1.counts),],
  BCL2.pos[colnames(BCL2.counts),],
  BCL3.pos[colnames(BCL3.counts),]
)

## ---- bcl-clean, fig.width=8, fig.height=3------------------------------------
cc <- cleanCounts(counts, min.reads = 100, min.lib.size = 100, plot=TRUE)
mat <- normalizeCounts(cc, log=FALSE, verbose=TRUE)
posList[[1]] <- posList[[1]][intersect(rownames(posList[[1]]), colnames(mat)),]
posList[[2]] <- posList[[2]][intersect(rownames(posList[[2]]), colnames(mat)),]
posList[[3]] <- posList[[3]][intersect(rownames(posList[[3]]), colnames(mat)),]

## ---- bcl-init, fig.width=9, fig.height=3-------------------------------------
# Plot
par(mfrow=c(1,3), mar=rep(2,4))
plotEmbedding(posList[[1]], groups=section, main='section 1', cex=2)
plotEmbedding(posList[[2]], groups=section, main='section 2', cex=2)
plotEmbedding(posList[[3]], groups=section, main='section 3', cex=2)

# Sample 1000 genes for demonstrative purposes only
set.seed(0)
test <- sample(rownames(mat), 1000)
mat <- mat[test,]

## ---- bcl-indsection, fig.width=9, fig.height=3-------------------------------
helper <- function(pos, mat) {
  w <- getSpatialNeighbors(pos, filterDist = 3)
  plotNetwork(pos, w)
  # get spatially clustered genes
  I <- getSpatialPatterns(mat, w)
  # filter for significant hits driven by more than 10% of cells
  results.filter <- filterSpatialPatterns(mat = mat,
                                          I = I,
                                          w = w,
                                          adjustPv = TRUE,
                                          alpha = 0.05,
                                          minPercentCells = 0.05,
                                          verbose = TRUE)
  # return results
  list(I=I, sig.genes=results.filter)
}

# Analyze each section using helper function
par(mfrow=c(1,3), mar=rep(2,4))
L1 <- helper(posList[[1]], mat[, rownames(posList[[1]])])
L2 <- helper(posList[[2]], mat[, rownames(posList[[2]])])
L3 <- helper(posList[[3]], mat[, rownames(posList[[3]])])

## ---- bcl-crosssection--------------------------------------------------------
# K-mutual nearest neighbors across sections
cw <- getCrossLayerNeighbors(posList, k=3)
# Look for genes exhibiting spatial autocorrelation across layers
I <- getSpatialPatterns(mat, cw)
results.filter <- filterSpatialPatterns(mat = mat,
                                        I = I,
                                        w = cw,
                                        adjustPv = TRUE,
                                        alpha = 0.05,
                                        minPercentCells = 0.05/4,
                                        verbose = TRUE)
cross <- list(I=I, sig.genes=results.filter)

## ---- bcl-sample, fig.width=9, fig.height=3-----------------------------------
gdups <- unlist(list('L1'=L1$sig.genes, 
                     'L2'=L2$sig.genes, 
                     'L3'=L3$sig.genes
))
## genes that are significantly spatially variable in multiple sections
gdups <- gdups[duplicated(gdups)]
## genes that are significantly spatially variable across sections
gcross <- cross$sig.genes
## both 
gall <- intersect(gcross, gdups)
## order by Moran's I
gall <- gall[order(L1$I[gall,]$observed, decreasing=TRUE)]

# Plot few consistent genes
invisible(lapply(gall[1:2], function(g1) {
  par(mfrow=c(1,3), mar=rep(2,4))
  plotEmbedding(posList[[1]], 
                colors=winsorize(scale(mat[g1, rownames(posList[[1]])])[,1]), 
                main=paste0(g1, '\n section 1'),
                cex=2, verbose=FALSE)
  plotEmbedding(posList[[2]], 
                colors=winsorize(scale(mat[g1, rownames(posList[[2]])])[,1]), 
                main=paste0(g1, '\n section 2'), 
                cex=2, verbose=FALSE)
  plotEmbedding(posList[[3]], 
                colors=winsorize(scale(mat[g1, rownames(posList[[3]])])[,1]), 
                main=paste0(g1, '\n section 3'), 
                cex=2, verbose=FALSE)
}))


## ---- bcl-inconsistent, fig.width=9, fig.height=3-----------------------------
gdiff <- setdiff(gdups, gcross)
gdiff <- gdiff[order(L1$I[gdiff,]$observed, decreasing=TRUE)]

# Plot inconsistent gene
invisible(lapply(gdiff[1], function(g1) {
  par(mfrow=c(1,3), mar=rep(2,4))
  plotEmbedding(posList[[1]], 
                colors=winsorize(scale(mat[g1, rownames(posList[[1]])])[,1]), 
                main=paste0(g1, '\n section 1'),
                cex=2, verbose=FALSE)
  plotEmbedding(posList[[2]], 
                colors=winsorize(scale(mat[g1, rownames(posList[[2]])])[,1]), 
                main=paste0(g1, '\n section 2'), 
                cex=2, verbose=FALSE)
  plotEmbedding(posList[[3]], 
                colors=winsorize(scale(mat[g1, rownames(posList[[3]])])[,1]), 
                main=paste0(g1, '\n section 3'), 
                cex=2, verbose=FALSE)
}))

