#' Use scDaPars to quantify APA dynamics across single cells using scRNA-seq data
#'
#' @param raw_PDUI_file A character specifying the full path of the raw PDUI matrix generated by step1 of scDaPars;
#' @param out_dir A character specifying the full path of the output directory;
#' @param filter_gene_thre A number between 0 and 1;
#' specifying the percent of cells a gene's PDUI must be detected;
#' @param filter_cell_thre A number between 0;
#' specigying the percent of genes' PDUI a cell must be detected;
#' @return scDaPars returns the imputed PDUI_matrix with rows representing genes and columns representing cells;
#' It saves the imputed PDUI matrix as scDaPars_imputed_results.txt;
#' @export
#' @import foreach
#' @import penalized
#' @importFrom utils read.table write.table
#' @importFrom stats complete.cases prcomp quantile sd
#' @import RANN
#' @import igraph
#' @author Yipeng Gao, \email{yipeng.gao@bcm.edu}
#' @author Wei Li, \email{wei.li@uci.edu}



scDaPars =
  function(raw_PDUI_file, out_dir, filter_gene_thre, filter_cell_thre)
  {
    if(is.null(raw_PDUI_file)){
      stop("Must provide input files!'")
    }
    if(is.null(out_dir)){
      stop("'Must provide output directory!'")
    }

    print("Reading in raw PDUI matrix ...")
    dir.create(out_dir, recursive = TRUE)

    raw_PDUI_matrix = read_raw_PDUI(raw_PDUI_file)

    print("Reading finished!")

    print("Start Processing raw PDUI matrix")
    raw_PDUI_process.res = Processing_raw_PDUI_matrix(raw_PDUI = raw_PDUI_matrix,
                                                      filter_gene_thre = filter_gene_thre,
                                                      filter_cell_thre = filter_cell_thre)
    raw_PDUI_matrix_sc = raw_PDUI_process.res$raw_PDUI_matrix_sc
    raw_PDUI_matrix_sNA = raw_PDUI_process.res$raw_PDUI_matrix_sNA
    print("Pre-Processing finished")

    print("Start Imputation Steps ...")
    print("Find potential Neighboring Cells ...")

    Neighboring_Cell.res = Find_Neighboring_Cells(raw_PDUI_sNA = raw_PDUI_matrix_sNA)
    cluster_info = Neighboring_Cell.res$cluster_info
    num_clusters = Neighboring_Cell.res$num_clusters
    outliers = Neighboring_Cell.res$outliers

    print(paste0("Number of neighbors(clusters) is ", num_clusters))
    print(paste0("Outliers is/are ", outliers))

    print("Imputation with neighboring cells ...")

    Imputation.res = Impuation_model(raw_PDUI_sNA = raw_PDUI_matrix_sNA,
                                      raw_PDUI_sc = raw_PDUI_matrix_sc,
                                      num_clusters = num_clusters,
                                      cluster_info = cluster_info)

    PDUI_sc_imputed = Imputation.res
    print("Imputation Steps Finished!")

    print("Writing imputed PDUI matrix ...")

    write.table(PDUI_sc_imputed, paste0(out_dir, "/scDaPars_imputed_results.txt"), quote = FALSE)

    return(PDUI_sc_imputed)
  }
