/*
 * Decompiled with CFR 0.152.
 */
package picard.vcf;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.BlockCompressedOutputStream;
import htsjdk.samtools.util.BlockCompressedStreamConstants;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.RuntimeIOException;
import htsjdk.tribble.AbstractFeatureReader;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextComparator;
import htsjdk.variant.variantcontext.writer.Options;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterBuilder;
import htsjdk.variant.vcf.VCFContigHeaderLine;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFHeader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.VariantManipulationProgramGroup;

@CommandLineProgramProperties(summary="Gathers multiple VCF files from a scatter operation into a single VCF file. Input files must be supplied in genomic order and must not have events at overlapping positions.", oneLineSummary="Gathers multiple VCF files from a scatter operation into a single VCF file", programGroup=VariantManipulationProgramGroup.class)
@DocumentedFeature
public class GatherVcfs
extends CommandLineProgram {
    @Argument(shortName="I", doc="Input VCF file(s).")
    public List<File> INPUT;
    @Argument(shortName="O", doc="Output VCF file.")
    public File OUTPUT;
    private static final Log log = Log.getInstance(GatherVcfs.class);

    public GatherVcfs() {
        this.CREATE_INDEX = true;
    }

    @Override
    protected int doWork() {
        log.info("Checking inputs.");
        this.INPUT = IOUtil.unrollFiles(this.INPUT, IOUtil.VCF_EXTENSIONS);
        for (File f : this.INPUT) {
            IOUtil.assertFileIsReadable(f);
        }
        IOUtil.assertFileIsWritable(this.OUTPUT);
        SAMSequenceDictionary sequenceDictionary = VCFFileReader.getSequenceDictionary(this.INPUT.get(0));
        if (this.CREATE_INDEX.booleanValue() && sequenceDictionary == null) {
            throw new PicardException("In order to index the resulting VCF input VCFs must contain ##contig lines.");
        }
        log.info("Checking file headers and first records to ensure compatibility.");
        try {
            GatherVcfs.assertSameSamplesAndValidOrdering(this.INPUT);
            if (this.areAllBlockCompressed(this.INPUT) && this.areAllBlockCompressed(Collections.singletonList(this.OUTPUT))) {
                log.info("Gathering by copying gzip blocks. Will not be able to validate position non-overlap of files.");
                if (this.CREATE_INDEX.booleanValue()) {
                    log.warn("Index creation not currently supported when gathering block compressed VCFs.");
                }
                GatherVcfs.gatherWithBlockCopying(this.INPUT, this.OUTPUT);
            } else {
                log.info("Gathering by conventional means.");
                GatherVcfs.gatherConventionally(sequenceDictionary, this.CREATE_INDEX, this.INPUT, this.OUTPUT);
            }
        }
        catch (RuntimeException e) {
            log.error("There was a problem with gathering the INPUT.", e);
            try {
                Files.deleteIfExists(this.OUTPUT.toPath());
            }
            catch (Exception exception) {
                // empty catch block
            }
            return 1;
        }
        return 0;
    }

    private boolean areAllBlockCompressed(List<File> input) {
        for (File f : input) {
            if (!VCFFileReader.isBCF(f) && AbstractFeatureReader.hasBlockCompressedExtension(f)) continue;
            return false;
        }
        return true;
    }

    private static void assertSameSamplesAndValidOrdering(List<File> inputFiles) {
        VCFHeader header = new VCFFileReader(inputFiles.get(0), false).getFileHeader();
        SAMSequenceDictionary dict = header.getSequenceDictionary();
        VariantContextComparator comparator = new VariantContextComparator(header.getSequenceDictionary());
        List<String> samples = header.getGenotypeSamples();
        File lastFile = null;
        VariantContext lastContext = null;
        for (File f : inputFiles) {
            VCFFileReader in = new VCFFileReader(f, false);
            try {
                dict.assertSameDictionary(in.getFileHeader().getSequenceDictionary());
            }
            catch (AssertionError e) {
                log.error("File #1: " + inputFiles.get(0));
                log.error("File #2: " + f);
                throw e;
            }
            List<String> theseSamples = in.getFileHeader().getGenotypeSamples();
            if (!samples.equals(theseSamples)) {
                TreeSet<String> s1 = new TreeSet<String>(samples);
                TreeSet<String> s2 = new TreeSet<String>(theseSamples);
                s1.removeAll(theseSamples);
                s2.removeAll(samples);
                throw new IllegalArgumentException("VCFs do not have identical sample lists. Samples unique to first file: " + s1 + ". Samples unique to " + f.getAbsolutePath() + ": " + s2 + ".");
            }
            Iterator variantIterator = in.iterator();
            if (variantIterator.hasNext()) {
                VariantContext currentContext = (VariantContext)variantIterator.next();
                if (lastContext != null && comparator.compare(lastContext, currentContext) >= 0) {
                    throw new IllegalArgumentException("First record in file " + f.getAbsolutePath() + " is not after first record in previous file " + lastFile.getAbsolutePath());
                }
                lastContext = currentContext;
                lastFile = f;
            }
            CloserUtil.close(in);
        }
    }

    private static void gatherConventionally(SAMSequenceDictionary sequenceDictionary, boolean createIndex, List<File> inputFiles, File outputFile) {
        EnumSet<Options> options = EnumSet.copyOf(VariantContextWriterBuilder.DEFAULT_OPTIONS);
        if (createIndex) {
            options.add(Options.INDEX_ON_THE_FLY);
        } else {
            options.remove((Object)Options.INDEX_ON_THE_FLY);
        }
        VariantContextWriter out = new VariantContextWriterBuilder().setOptions(options).setOutputFile(outputFile).setReferenceDictionary(sequenceDictionary).build();
        ProgressLogger progress = new ProgressLogger(log, 10000);
        VariantContext lastContext = null;
        File lastFile = null;
        VCFHeader firstHeader = null;
        VariantContextComparator comparator = null;
        for (File f : inputFiles) {
            VariantContext vc;
            log.debug("Gathering from file: ", f.getAbsolutePath());
            VCFFileReader variantReader = new VCFFileReader(f, false);
            PeekableIterator variantIterator = new PeekableIterator(variantReader.iterator());
            VCFHeader header = variantReader.getFileHeader();
            if (firstHeader == null) {
                firstHeader = header;
                out.writeHeader(firstHeader);
                comparator = new VariantContextComparator((Collection<VCFContigHeaderLine>)firstHeader.getContigLines());
            }
            if (lastContext != null && variantIterator.hasNext() && comparator.compare(vc = (VariantContext)variantIterator.peek(), lastContext) <= 0) {
                throw new IllegalArgumentException("First variant in file " + f.getAbsolutePath() + " is at " + vc.getContig() + ":" + vc.getStart() + " but last variant in earlier file " + lastFile.getAbsolutePath() + " is at " + lastContext.getContig() + ":" + lastContext.getStart());
            }
            while (variantIterator.hasNext()) {
                lastContext = (VariantContext)variantIterator.next();
                out.add(lastContext);
                progress.record(lastContext.getContig(), lastContext.getStart());
            }
            lastFile = f;
            CloserUtil.close(variantIterator);
            CloserUtil.close(variantReader);
        }
        out.close();
    }

    private static void gatherWithBlockCopying(List<File> vcfs, File output) {
        try {
            FileOutputStream out = new FileOutputStream(output);
            boolean isFirstFile = true;
            for (File f : vcfs) {
                log.info("Gathering " + f.getAbsolutePath());
                FileInputStream in = new FileInputStream(f);
                BlockCompressedInputStream.FileTermination term = BlockCompressedInputStream.checkTermination(f);
                if (term == BlockCompressedInputStream.FileTermination.DEFECTIVE) {
                    throw new PicardException(f.getAbsolutePath() + " does not have a valid GZIP block at the end of the file.");
                }
                if (!isFirstFile) {
                    BlockCompressedInputStream blockIn = new BlockCompressedInputStream((InputStream)in, false);
                    boolean lastByteNewline = true;
                    while (blockIn.available() > 0) {
                        int blockLength = blockIn.available();
                        byte[] blockContents = new byte[blockLength];
                        int read = blockIn.read(blockContents);
                        if (blockLength == 0 || read != blockLength) {
                            throw new IllegalStateException("Could not read available bytes from BlockCompressedInputStream.");
                        }
                        int firstNonHeaderByteIndex = -1;
                        for (int i = 0; i < read; ++i) {
                            boolean thisByteNewline;
                            byte b = blockContents[i];
                            boolean bl = thisByteNewline = b == 10 || b == 13;
                            if (lastByteNewline && !thisByteNewline && b != 35) {
                                firstNonHeaderByteIndex = i;
                                break;
                            }
                            lastByteNewline = thisByteNewline;
                        }
                        if (firstNonHeaderByteIndex < 0) continue;
                        BlockCompressedOutputStream blockOut = new BlockCompressedOutputStream((OutputStream)out, (Path)null);
                        blockOut.write(blockContents, firstNonHeaderByteIndex, blockContents.length - firstNonHeaderByteIndex);
                        blockOut.flush();
                        break;
                    }
                }
                long currentPos = in.getChannel().position();
                long length = f.length();
                long skipLast = term == BlockCompressedInputStream.FileTermination.HAS_TERMINATOR_BLOCK ? (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length : 0L;
                long bytesToWrite = length - skipLast - currentPos;
                IOUtil.transferByStream(in, out, bytesToWrite);
                in.close();
                isFirstFile = false;
            }
            out.write(BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK);
            out.close();
        }
        catch (IOException ioe) {
            throw new RuntimeIOException(ioe);
        }
    }
}

