/*
 * Decompiled with CFR 0.152.
 */
package picard.arrays;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.reference.ReferenceSequenceFileFactory;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.SortingCollection;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.Genotype;
import htsjdk.variant.variantcontext.GenotypeBuilder;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextBuilder;
import htsjdk.variant.variantcontext.VariantContextComparator;
import htsjdk.variant.variantcontext.VariantContextUtils;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterBuilder;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFFilterHeaderLine;
import htsjdk.variant.vcf.VCFFormatHeaderLine;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFHeaderLine;
import htsjdk.variant.vcf.VCFHeaderLineType;
import htsjdk.variant.vcf.VCFInfoHeaderLine;
import htsjdk.variant.vcf.VCFRecordCodec;
import htsjdk.variant.vcf.VCFStandardHeaderLines;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.arrays.illumina.ArraysControlInfo;
import picard.arrays.illumina.Build37ExtendedIlluminaManifest;
import picard.arrays.illumina.Build37ExtendedIlluminaManifestRecord;
import picard.arrays.illumina.IlluminaManifestRecord;
import picard.arrays.illumina.InfiniumEGTFile;
import picard.arrays.illumina.InfiniumGTCFile;
import picard.arrays.illumina.InfiniumGTCRecord;
import picard.arrays.illumina.InfiniumNormalizationManifest;
import picard.arrays.illumina.InfiniumVcfFields;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.GenotypingArraysProgramGroup;
import picard.pedigree.Sex;

@CommandLineProgramProperties(summary="GtcToVcf takes an Illumina GTC file and converts it to a VCF file using several supporting files. A GTC file is an Illumina-specific file containing called genotypes in AA/AB/BB format. <a href='https://github.com/Illumina/BeadArrayFiles/blob/develop/docs/GTC_File_Format_v5.pdf'></a> A VCF, aka Variant Calling Format, is a text file for storing how a sequenced sample differs from the reference genome. <a href='http://software.broadinstitute.org/software/igv/book/export/html/184'></a><h4>Usage example:</h4><pre>java -jar picard.jar GtcToVcf \\<br />      INPUT=input.gtc \\<br />      REFERENCE_SEQUENCE=reference.fasta \\<br />      OUTPUT=output.vcf \\<br />      EXTENDED_ILLUMINA_MANIFEST=chip_name.extended.csv \\<br />      CLUSTER_FILE=chip_name.egt \\<br />      ILLUMINA_NORMALIZATION_MANIFEST=chip_name.bpm.csv \\<br />      SAMPLE_ALIAS=my_sample_alias \\<br /></pre>", oneLineSummary="Program to convert a GTC file to a VCF", programGroup=GenotypingArraysProgramGroup.class)
@DocumentedFeature
public class GtcToVcf
extends CommandLineProgram {
    static final String USAGE_DETAILS = "GtcToVcf takes an Illumina GTC file and converts it to a VCF file using several supporting files. A GTC file is an Illumina-specific file containing called genotypes in AA/AB/BB format. <a href='https://github.com/Illumina/BeadArrayFiles/blob/develop/docs/GTC_File_Format_v5.pdf'></a> A VCF, aka Variant Calling Format, is a text file for storing how a sequenced sample differs from the reference genome. <a href='http://software.broadinstitute.org/software/igv/book/export/html/184'></a><h4>Usage example:</h4><pre>java -jar picard.jar GtcToVcf \\<br />      INPUT=input.gtc \\<br />      REFERENCE_SEQUENCE=reference.fasta \\<br />      OUTPUT=output.vcf \\<br />      EXTENDED_ILLUMINA_MANIFEST=chip_name.extended.csv \\<br />      CLUSTER_FILE=chip_name.egt \\<br />      ILLUMINA_NORMALIZATION_MANIFEST=chip_name.bpm.csv \\<br />      SAMPLE_ALIAS=my_sample_alias \\<br /></pre>";
    private static final Log log = Log.getInstance(GtcToVcf.class);
    @Argument(shortName="I", doc="GTC file to be converted")
    public File INPUT;
    @Argument(shortName="O", doc="The output VCF file to write.")
    public File OUTPUT;
    @Argument(shortName="MANIFEST", doc="An Extended Illumina Manifest file (csv).  This is an extended version of the Illumina manifest it contains additional reference-specific fields")
    public File EXTENDED_ILLUMINA_MANIFEST;
    @Argument(shortName="CF", doc="An Illumina cluster file (egt)")
    public File CLUSTER_FILE;
    @Argument(shortName="NORM_MANIFEST", doc="An Illumina bead pool manifest (a manifest containing the Illumina normalization ids) (bpm.csv)")
    public File ILLUMINA_NORMALIZATION_MANIFEST;
    @Argument(shortName="E_GENDER", doc="The expected gender for this sample.", optional=true)
    public String EXPECTED_GENDER;
    @Argument(doc="The sample alias")
    public String SAMPLE_ALIAS;
    @Argument(doc="The analysis version of the data used to generate this VCF", optional=true)
    public Integer ANALYSIS_VERSION_NUMBER;
    @Argument(shortName="G_GTC", doc="An optional GTC file that was generated by calling the chip using a cluster file designed to optimize gender calling.", optional=true)
    public File GENDER_GTC;
    @Argument(shortName="FP_VCF", doc="The fingerprint VCF for this sample", optional=true)
    public File FINGERPRINT_GENOTYPES_VCF_FILE;
    @Argument(doc="Causes the program to fail if it finds a case where there is a call on an assay that is flagged as 'zeroed-out' in the Illumina cluster file.", optional=true)
    public boolean DO_NOT_ALLOW_CALLS_ON_ZEROED_OUT_ASSAYS = false;
    static final List<Allele> NO_CALL_ALLELES = Collections.unmodifiableList(Arrays.asList(Allele.NO_CALL, Allele.NO_CALL));
    private static InfiniumGTCFile infiniumGTCFile;
    private static InfiniumEGTFile infiniumEGTFile;
    private static String gtcGender;
    private static Sex fingerprintGender;
    private static ReferenceSequenceFile refSeq;
    private static final DecimalFormat df;
    private static final String DOT = ".";

    @Override
    protected boolean requiresReference() {
        return true;
    }

    @Override
    protected int doWork() {
        Build37ExtendedIlluminaManifest manifest = this.setupAndGetManifest();
        VCFHeader vcfHeader = this.createVCFHeader(manifest, infiniumGTCFile, gtcGender, this.CLUSTER_FILE, this.REFERENCE_SEQUENCE, refSeq.getSequenceDictionary());
        SortingCollection<VariantContext> contexts = SortingCollection.newInstance(VariantContext.class, new VCFRecordCodec(vcfHeader), new VariantContextComparator(refSeq.getSequenceDictionary()), (int)this.MAX_RECORDS_IN_RAM, (Path[])this.TMP_DIR.stream().map(File::toPath).toArray(Path[]::new));
        this.fillContexts(contexts, infiniumGTCFile, manifest, infiniumEGTFile);
        this.writeVcf(contexts, this.OUTPUT, refSeq.getSequenceDictionary(), vcfHeader);
        return 0;
    }

    @Override
    protected String[] customCommandLineValidation() {
        IOUtil.assertFileIsReadable(this.INPUT);
        IOUtil.assertFileIsReadable(this.EXTENDED_ILLUMINA_MANIFEST);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        refSeq = ReferenceSequenceFileFactory.getReferenceSequenceFile(this.REFERENCE_SEQUENCE);
        SAMSequenceDictionary sequenceDictionary = refSeq.getSequenceDictionary();
        String assembly = sequenceDictionary.getSequence(0).getAssembly();
        if (!assembly.equals("GRCh37")) {
            return new String[]{"The selected reference sequence ('" + assembly + "') is not supported.  This tool is currently only implemented to support NCBI Build 37 / HG19 Reference Sequence."};
        }
        if (this.FINGERPRINT_GENOTYPES_VCF_FILE != null) {
            IOUtil.assertFileIsReadable(this.FINGERPRINT_GENOTYPES_VCF_FILE);
        }
        if (this.GENDER_GTC != null) {
            IOUtil.assertFileIsReadable(this.GENDER_GTC);
        }
        return super.customCommandLineValidation();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Build37ExtendedIlluminaManifest setupAndGetManifest() {
        fingerprintGender = GtcToVcf.getFingerprintSex(this.FINGERPRINT_GENOTYPES_VCF_FILE);
        InfiniumNormalizationManifest infiniumNormalizationManifest = new InfiniumNormalizationManifest(this.ILLUMINA_NORMALIZATION_MANIFEST);
        try (DataInputStream gtcInputStream = new DataInputStream(new FileInputStream(this.INPUT));){
            String illuminaManifestName;
            String gtcManifestName;
            infiniumEGTFile = new InfiniumEGTFile(this.CLUSTER_FILE);
            infiniumGTCFile = new InfiniumGTCFile(gtcInputStream, infiniumNormalizationManifest);
            Build37ExtendedIlluminaManifest manifest = new Build37ExtendedIlluminaManifest(this.EXTENDED_ILLUMINA_MANIFEST);
            if (this.GENDER_GTC != null) {
                try (DataInputStream genderGtcStream = new DataInputStream(new FileInputStream(this.GENDER_GTC));){
                    gtcGender = new InfiniumGTCFile(genderGtcStream, infiniumNormalizationManifest).getGender();
                }
            }
            if (!(gtcManifestName = FilenameUtils.removeExtension(infiniumGTCFile.getSnpManifest())).equalsIgnoreCase(illuminaManifestName = FilenameUtils.removeExtension(manifest.getDescriptorFileName()))) {
                throw new PicardException("The GTC's manifest name " + gtcManifestName + " does not match the Illumina manifest name " + illuminaManifestName);
            }
            if (infiniumGTCFile.getNumberOfSnps() != manifest.getNumAssays()) {
                log.warn("The number of Assays in the GTC file: " + infiniumGTCFile.getNumberOfSnps() + " does not equal the number of Assays in the Illumina manifest file: " + manifest.getNumAssays());
            }
            Build37ExtendedIlluminaManifest build37ExtendedIlluminaManifest = manifest;
            return build37ExtendedIlluminaManifest;
        }
        catch (IOException e) {
            throw new PicardException("Error during setup", e);
        }
    }

    static Sex getFingerprintSex(File file) {
        if (file != null) {
            try (VCFFileReader reader = new VCFFileReader(file, false);){
                VCFHeader header = reader.getFileHeader();
                VCFHeaderLine gender = header.getMetaDataLine("gender");
                if (gender != null) {
                    Sex sex = Sex.valueOf(gender.getValue());
                    return sex;
                }
            }
        }
        return Sex.Unknown;
    }

    private void fillContexts(SortingCollection<VariantContext> contexts, InfiniumGTCFile gtcFile, Build37ExtendedIlluminaManifest manifest, InfiniumEGTFile egtFile) {
        ProgressLogger progressLogger = new ProgressLogger(log, 100000, "sorted");
        Iterator<Build37ExtendedIlluminaManifestRecord> iterator = manifest.extendedIterator();
        int gtcIndex = 0;
        int numVariantsWritten = 0;
        while (iterator.hasNext()) {
            Build37ExtendedIlluminaManifestRecord record = iterator.next();
            if (!record.isBad().booleanValue()) {
                InfiniumGTCRecord gtcRecord = gtcFile.getRecord(gtcIndex);
                VariantContext context = this.makeVariantContext(record, gtcRecord, egtFile, progressLogger);
                ++numVariantsWritten;
                contexts.add(context);
            }
            ++gtcIndex;
        }
        log.info(numVariantsWritten + " Variants were written to file");
        log.info(gtcFile.getNumberOfSnps() + " SNPs in the GTC file");
        log.info(manifest.getNumAssays() + " Variants on the " + manifest.getDescriptorFileName() + " genotyping array manifest file");
    }

    private VariantContext makeVariantContext(Build37ExtendedIlluminaManifestRecord record, InfiniumGTCRecord gtcRecord, InfiniumEGTFile egtFile, ProgressLogger progressLogger) {
        Allele A = record.getAlleleA();
        Allele B = record.getAlleleB();
        Allele ref = record.getRefAllele();
        String chr = record.getB37Chr();
        Integer position = record.getB37Pos();
        Integer endPosition = position + ref.length() - 1;
        Integer egtIndex = egtFile.rsNameToIndex.get(record.getName());
        if (egtIndex == null) {
            throw new PicardException("Found no record in cluster file for manifest entry '" + record.getName() + "'");
        }
        progressLogger.record(chr, position);
        ArrayList<Allele> assayAlleles = new ArrayList<Allele>();
        assayAlleles.add(ref);
        if (A.equals(B)) {
            throw new PicardException("Found same allele (" + A.getDisplayString() + ") for A and B ");
        }
        if (!ref.equals(A, true)) {
            assayAlleles.add(A);
        }
        if (!ref.equals(B, true)) {
            assayAlleles.add(B);
        }
        String sampleName = FilenameUtils.removeExtension(this.INPUT.getName());
        Genotype genotype = this.getGenotype(sampleName, gtcRecord, record, A, B);
        VariantContextBuilder builder = new VariantContextBuilder();
        builder.source(record.getName());
        builder.chr(chr);
        builder.start(position.intValue());
        builder.stop(endPosition.intValue());
        builder.alleles((Collection<Allele>)assayAlleles);
        builder.log10PError(1.0);
        builder.id(record.getName());
        builder.genotypes(genotype);
        VariantContextUtils.calculateChromosomeCounts(builder, false);
        builder.attribute("ALLELE_A", record.getAlleleA());
        builder.attribute("ALLELE_B", record.getAlleleB());
        builder.attribute("ILLUMINA_STRAND", (Object)record.getIlmnStrand());
        builder.attribute("PROBE_A", record.getAlleleAProbeSeq());
        builder.attribute("PROBE_B", record.getAlleleBProbeSeq());
        builder.attribute("BEADSET_ID", record.getBeadSetId());
        builder.attribute("ILLUMINA_CHR", record.getChr());
        builder.attribute("ILLUMINA_POS", record.getPosition());
        builder.attribute("ILLUMINA_BUILD", record.getGenomeBuild());
        builder.attribute("SOURCE", record.getSource().replace(' ', '_'));
        builder.attribute("GC_SCORE", GtcToVcf.formatFloatForVcf(egtFile.totalScore[egtIndex]));
        for (InfiniumVcfFields.GENOTYPE_VALUES gtValue : InfiniumVcfFields.GENOTYPE_VALUES.values()) {
            int ordinalValue = gtValue.ordinal();
            builder.attribute(InfiniumVcfFields.N[ordinalValue], egtFile.n[egtIndex][ordinalValue]);
            builder.attribute(InfiniumVcfFields.DEV_R[ordinalValue], GtcToVcf.formatFloatForVcf(egtFile.devR[egtIndex][ordinalValue]));
            builder.attribute(InfiniumVcfFields.MEAN_R[ordinalValue], GtcToVcf.formatFloatForVcf(egtFile.meanR[egtIndex][ordinalValue]));
            builder.attribute(InfiniumVcfFields.DEV_THETA[ordinalValue], GtcToVcf.formatFloatForVcf(egtFile.devTheta[egtIndex][ordinalValue]));
            builder.attribute(InfiniumVcfFields.MEAN_THETA[ordinalValue], GtcToVcf.formatFloatForVcf(egtFile.meanTheta[egtIndex][ordinalValue]));
            EuclideanValues genotypeEuclideanValues = this.polarToEuclidean(egtFile.meanR[egtIndex][ordinalValue], egtFile.devR[egtIndex][ordinalValue], egtFile.meanTheta[egtIndex][ordinalValue], egtFile.devTheta[egtIndex][ordinalValue]);
            builder.attribute(InfiniumVcfFields.DEV_X[ordinalValue], GtcToVcf.formatFloatForVcf(genotypeEuclideanValues.devX));
            builder.attribute(InfiniumVcfFields.MEAN_X[ordinalValue], GtcToVcf.formatFloatForVcf(genotypeEuclideanValues.meanX));
            builder.attribute(InfiniumVcfFields.DEV_Y[ordinalValue], GtcToVcf.formatFloatForVcf(genotypeEuclideanValues.devY));
            builder.attribute(InfiniumVcfFields.MEAN_Y[ordinalValue], GtcToVcf.formatFloatForVcf(genotypeEuclideanValues.meanY));
        }
        String rsid = record.getRsId();
        if (StringUtils.isNotEmpty(rsid)) {
            builder.attribute("refSNP", rsid);
        }
        if ((double)egtFile.totalScore[egtIndex] == 0.0) {
            builder.filter("ZEROED_OUT_ASSAY");
            if (genotype.isCalled()) {
                if (this.DO_NOT_ALLOW_CALLS_ON_ZEROED_OUT_ASSAYS) {
                    throw new PicardException("Found a call (genotype: " + genotype + ") on a zeroed out Assay!!");
                }
                log.warn("Found a call (genotype: " + genotype + ") on a zeroed out Assay. This could occur if you called genotypes on a different cluster file than used here.");
            }
        }
        if (record.isDupe().booleanValue()) {
            builder.filter("DUPE");
        }
        return builder.make();
    }

    EuclideanValues polarToEuclidean(float r, float rDeviation, float theta, float thetaDeviation) {
        double thetaVariance = Math.pow(thetaDeviation, 2.0);
        double rVariance = Math.pow(rDeviation, 2.0);
        double halfPi = 1.5707963267948966;
        double normalizedTheta = 1.5707963267948966 * (double)theta;
        double rOverX = 1.0 + Math.tan(normalizedTheta);
        double thetaVarianceFactorX = -1.0 * (1.5707963267948966 * (double)r) * Math.pow(rOverX * Math.cos(normalizedTheta), -2.0);
        double rVarianceFactorX = 1.0 / rOverX;
        double varianceX = Math.pow(thetaVarianceFactorX, 2.0) * thetaVariance + Math.pow(rVarianceFactorX, 2.0) * rVariance;
        double thetaVarianceFactorY = -1.0 * thetaVarianceFactorX;
        double rVarianceFactorY = 1.0 - rVarianceFactorX;
        double varianceY = Math.pow(thetaVarianceFactorY, 2.0) * thetaVariance + Math.pow(rVarianceFactorY, 2.0) * rVariance;
        double meanX = (double)r / rOverX;
        double meanY = (double)r - meanX;
        double devX = Math.pow(varianceX, 0.5);
        double devY = Math.pow(varianceY, 0.5);
        return new EuclideanValues((float)meanX, (float)meanY, (float)devX, (float)devY);
    }

    public Genotype getGenotype(String sampleName, InfiniumGTCRecord infiniumGtcRecord, IlluminaManifestRecord record, Allele A, Allele B) {
        List<Allele> alleles;
        if (infiniumGtcRecord.genotype == 0) {
            alleles = NO_CALL_ALLELES;
        } else if (infiniumGtcRecord.genotype == 1) {
            alleles = Arrays.asList(A, A);
        } else if (infiniumGtcRecord.genotype == 2) {
            alleles = Arrays.asList(A, B);
        } else if (infiniumGtcRecord.genotype == 3) {
            alleles = Arrays.asList(B, B);
        } else {
            throw new PicardException("Unexpected genotype call [" + infiniumGtcRecord.genotype + "] for SNP: " + record.getName());
        }
        HashMap<String, Object> attributes = new HashMap<String, Object>();
        attributes.put("IGC", GtcToVcf.formatFloatForVcf(infiniumGtcRecord.genotypeScore));
        attributes.put("X", infiniumGtcRecord.rawXIntensity);
        attributes.put("Y", infiniumGtcRecord.rawYIntensity);
        attributes.put("NORMX", GtcToVcf.formatFloatForVcf(infiniumGtcRecord.normalizedXIntensity));
        attributes.put("NORMY", GtcToVcf.formatFloatForVcf(infiniumGtcRecord.normalizedYIntensity));
        attributes.put("R", GtcToVcf.formatFloatForVcf(infiniumGtcRecord.RIlmn));
        attributes.put("THETA", GtcToVcf.formatFloatForVcf(infiniumGtcRecord.thetaIlmn));
        attributes.put("BAF", GtcToVcf.formatFloatForVcf(infiniumGtcRecord.bAlleleFreq));
        attributes.put("LRR", GtcToVcf.formatFloatForVcf(infiniumGtcRecord.logRRatio));
        return GenotypeBuilder.create(sampleName, alleles, attributes);
    }

    public static String formatFloatForVcf(float value) {
        if (Float.isNaN(value)) {
            return DOT;
        }
        return df.format(value);
    }

    private void writeVcf(SortingCollection<VariantContext> variants, File output, SAMSequenceDictionary dict, VCFHeader vcfHeader) {
        try (VariantContextWriter writer = new VariantContextWriterBuilder().setOutputFile(output).setReferenceDictionary(dict).setOptions(VariantContextWriterBuilder.DEFAULT_OPTIONS).build();){
            writer.writeHeader(vcfHeader);
            for (VariantContext variant : variants) {
                if (variant.getAlternateAlleles().size() > 1) {
                    variant.getCommonInfo().addFilter("TRIALLELIC");
                }
                writer.add(variant);
            }
        }
    }

    private VCFHeader createVCFHeader(Build37ExtendedIlluminaManifest manifest, InfiniumGTCFile gtcFile, String gtcGender, File clusterFile, File reference, SAMSequenceDictionary dict) {
        String inputName = this.INPUT.getName();
        String chipWellBarcode = inputName.substring(0, inputName.lastIndexOf(46));
        LinkedHashSet<VCFHeaderLine> lines = new LinkedHashSet<VCFHeaderLine>();
        lines.add(new VCFHeaderLine("fileDate", new Date().toString()));
        lines.add(new VCFHeaderLine("source", "GtcToVcf"));
        String descriptorFileName = manifest.getDescriptorFileName();
        lines.add(new VCFHeaderLine("arrayType", descriptorFileName.substring(0, descriptorFileName.lastIndexOf(DOT))));
        lines.add(new VCFHeaderLine("extendedManifestFile", this.EXTENDED_ILLUMINA_MANIFEST.getName()));
        lines.add(new VCFHeaderLine("extendedIlluminaManifestVersion", manifest.getExtendedManifestVersion()));
        lines.add(new VCFHeaderLine("chipWellBarcode", chipWellBarcode));
        if (this.ANALYSIS_VERSION_NUMBER != null) {
            lines.add(new VCFHeaderLine("analysisVersionNumber", this.ANALYSIS_VERSION_NUMBER.toString()));
        }
        lines.add(new VCFHeaderLine("sampleAlias", this.SAMPLE_ALIAS));
        if (this.EXPECTED_GENDER != null) {
            lines.add(new VCFHeaderLine("expectedGender", this.EXPECTED_GENDER));
        }
        int measurementCount = gtcFile.getRawControlXIntensities().length / ArraysControlInfo.CONTROL_INFO.length;
        for (int i = 0; i < ArraysControlInfo.CONTROL_INFO.length; ++i) {
            int offset = i * measurementCount;
            ArraysControlInfo controlInfo = ArraysControlInfo.CONTROL_INFO[i];
            int redIntensity = gtcFile.getRawControlXIntensity(offset);
            int greenIntensity = gtcFile.getRawControlYIntensity(offset);
            lines.add(new VCFHeaderLine(controlInfo.getControl(), controlInfo.toString() + "|" + redIntensity + "|" + greenIntensity));
        }
        lines.add(new VCFHeaderLine("fingerprintGender", fingerprintGender.name()));
        if (gtcGender != null) {
            lines.add(new VCFHeaderLine("autocallGender", gtcGender));
        } else {
            lines.add(new VCFHeaderLine("autocallGender", gtcFile.getGender()));
        }
        lines.add(new VCFHeaderLine("autocallDate", gtcFile.getAutoCallDate()));
        lines.add(new VCFHeaderLine("imagingDate", gtcFile.getImagingDate()));
        lines.add(new VCFHeaderLine("clusterFile", clusterFile.getName()));
        lines.add(new VCFHeaderLine("manifestFile", descriptorFileName));
        lines.add(new VCFHeaderLine("content", manifest.getManifestFile().getName()));
        lines.add(new VCFHeaderLine("autocallVersion", gtcFile.getAutoCallVersion()));
        lines.add(new VCFHeaderLine("reference", reference.getAbsolutePath()));
        lines.add(new VCFHeaderLine("picardVersion", this.getVersion()));
        lines.add(new VCFHeaderLine("p95Red", String.valueOf(gtcFile.getP95Red())));
        lines.add(new VCFHeaderLine("p95Green", String.valueOf(gtcFile.getP95Green())));
        lines.add(new VCFHeaderLine("scannerName", gtcFile.getScannerName()));
        lines.add(VCFStandardHeaderLines.getFormatLine("GT"));
        lines.add(VCFStandardHeaderLines.getInfoLine("AC"));
        lines.add(VCFStandardHeaderLines.getInfoLine("AF"));
        lines.add(VCFStandardHeaderLines.getInfoLine("AN"));
        lines.add(new VCFFormatHeaderLine("IGC", 1, VCFHeaderLineType.Float, "Illumina GenCall Confidence Score"));
        lines.add(new VCFFormatHeaderLine("X", 1, VCFHeaderLineType.Integer, "Raw X intensity"));
        lines.add(new VCFFormatHeaderLine("Y", 1, VCFHeaderLineType.Integer, "Raw Y intensity"));
        lines.add(new VCFFormatHeaderLine("NORMX", 1, VCFHeaderLineType.Float, "Normalized X intensity"));
        lines.add(new VCFFormatHeaderLine("NORMY", 1, VCFHeaderLineType.Float, "Normalized Y intensity"));
        lines.add(new VCFFormatHeaderLine("R", 1, VCFHeaderLineType.Float, "Normalized R value"));
        lines.add(new VCFFormatHeaderLine("THETA", 1, VCFHeaderLineType.Float, "Normalized Theta value"));
        lines.add(new VCFFormatHeaderLine("BAF", 1, VCFHeaderLineType.Float, "B Allele Frequency"));
        lines.add(new VCFFormatHeaderLine("LRR", 1, VCFHeaderLineType.Float, "Log R Ratio"));
        lines.add(new VCFInfoHeaderLine("ALLELE_A", 1, VCFHeaderLineType.String, "A allele"));
        lines.add(new VCFInfoHeaderLine("ALLELE_B", 1, VCFHeaderLineType.String, "B allele"));
        lines.add(new VCFInfoHeaderLine("ILLUMINA_STRAND", 1, VCFHeaderLineType.String, "Probe strand"));
        lines.add(new VCFInfoHeaderLine("PROBE_A", 1, VCFHeaderLineType.String, "Probe base pair sequence"));
        lines.add(new VCFInfoHeaderLine("PROBE_B", 1, VCFHeaderLineType.String, "Probe base pair sequence; not missing for strand-ambiguous SNPs"));
        lines.add(new VCFInfoHeaderLine("BEADSET_ID", 1, VCFHeaderLineType.Integer, "Bead set ID for normalization"));
        lines.add(new VCFInfoHeaderLine("ILLUMINA_CHR", 1, VCFHeaderLineType.String, "Chromosome in Illumina manifest"));
        lines.add(new VCFInfoHeaderLine("ILLUMINA_POS", 1, VCFHeaderLineType.Integer, "Position in Illumina manifest"));
        lines.add(new VCFInfoHeaderLine("ILLUMINA_BUILD", 1, VCFHeaderLineType.String, "Genome Build in Illumina manifest"));
        lines.add(new VCFInfoHeaderLine("SOURCE", 1, VCFHeaderLineType.String, "Probe source"));
        lines.add(new VCFInfoHeaderLine("GC_SCORE", 1, VCFHeaderLineType.Float, "Gentrain Score"));
        for (InfiniumVcfFields.GENOTYPE_VALUES gtValue : InfiniumVcfFields.GENOTYPE_VALUES.values()) {
            int ordinalValue = gtValue.ordinal();
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.N[ordinalValue], 1, VCFHeaderLineType.Integer, "Number of " + gtValue.name() + " calls in training set"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.DEV_R[ordinalValue], 1, VCFHeaderLineType.Float, "Standard deviation of normalized R for " + gtValue.name() + " cluster"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.DEV_THETA[ordinalValue], 1, VCFHeaderLineType.Float, "Standard deviation of normalized THETA for " + gtValue.name() + " cluster"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.DEV_X[ordinalValue], 1, VCFHeaderLineType.Float, "Standard deviation of normalized X for " + gtValue.name() + " cluster"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.DEV_Y[ordinalValue], 1, VCFHeaderLineType.Float, "Standard deviation of normalized Y for " + gtValue.name() + " cluster"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.MEAN_R[ordinalValue], 1, VCFHeaderLineType.Float, "Mean of normalized R for " + gtValue.name() + " cluster"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.MEAN_THETA[ordinalValue], 1, VCFHeaderLineType.Float, "Mean of normalized THETA for " + gtValue.name() + " cluster"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.MEAN_X[ordinalValue], 1, VCFHeaderLineType.Float, "Mean of normalized X for " + gtValue.name() + " cluster"));
            lines.add(new VCFInfoHeaderLine(InfiniumVcfFields.MEAN_Y[ordinalValue], 1, VCFHeaderLineType.Float, "Mean of normalized Y for " + gtValue.name() + " cluster"));
        }
        lines.add(new VCFInfoHeaderLine("refSNP", 1, VCFHeaderLineType.String, "dbSNP rsID"));
        lines.add(new VCFFilterHeaderLine("DUPE", "Duplicate assays position."));
        lines.add(new VCFFilterHeaderLine("TRIALLELIC", "Tri-allelic assay."));
        lines.add(new VCFFilterHeaderLine("FAIL_REF", "Assay failed to map to reference."));
        lines.add(new VCFFilterHeaderLine("ZEROED_OUT_ASSAY", "Assay Zeroed out (marked as uncallable) in the Illumina Cluster File"));
        VCFHeader header = new VCFHeader(lines, Collections.singletonList(chipWellBarcode));
        header.setSequenceDictionary(dict);
        return header;
    }

    static {
        gtcGender = null;
        df = new DecimalFormat();
        df.setMaximumFractionDigits(3);
    }

    class EuclideanValues {
        final float meanX;
        final float meanY;
        final float devX;
        final float devY;

        EuclideanValues(float meanX, float meanY, float devX, float devY) {
            this.meanX = meanX;
            this.meanY = meanY;
            this.devX = devX;
            this.devY = devY;
        }
    }
}

