#!/usr/bin/env python
import pandas as pd
import numpy as np
import argparse

parser = argparse.ArgumentParser(formatter_class=argparse.RawTextHelpFormatter, \
description="""

take_stat_dataframe.py

Given a tab-separated dataframe with a header (by default), for each row compute
a statistic that summarizes that row, which may be useful in time series analysis, 
for example. By default, script computes mean across rows, but can also compute
median, minimum, maximum, and signed absolute max.

""")

optional = parser._action_groups.pop()
required = parser.add_argument_group('required arguments')

##################################################
# required args:
required.add_argument("-i", "--df", 
                    help="path to input dataframe ", \
                    required=True)

required.add_argument("-o", "--out", 
                    help="path to output dataframe ", \
                    required=True)

##################################################
# optional args:

optional.add_argument("--no_header", 
                    help="if dataframe lacks a header", \
                    action='store_true')
optional.add_argument("--mean", 
                    help="take mean of dataframe across columns", \
                    action='store_true')
optional.add_argument("--median", 
                    help="take median of dataframe across columns", \
                    action='store_true')
optional.add_argument("--sum", 
                    help="take sum of dataframe across columns", \
                    action='store_true')
optional.add_argument("--min", 
                    help="take min of dataframe across columns", \
                    action='store_true')
optional.add_argument("--max", 
                    help="take max of dataframe across columns", \
                    action='store_true')
optional.add_argument("--minmax", 
                    help="take the signed absolute max of dataframe across columns", \
                    action='store_true')

##################################################
parser._action_groups.append(optional)
args = parser.parse_args()

header = 'infer' if not args.no_header else None
in_df = pd.read_csv(args.df, delim_whitespace=True, index_col=0, header=header)
    
if args.median:
    in_df = pd.DataFrame(np.median(in_df, axis=1), index=in_df.index)
elif args.min:
    in_df = in_df.min(axis=1)
elif args.max:
    in_df = in_df.max(axis=1)
elif args.sum:
    in_df = in_df.sum(axis=1)
elif args.minmax:
    index = list(in_df.index)
    in_df_npy = np.array(in_df)
    max_indices = np.abs(in_df_npy).argmax(axis=1)
    in_df = pd.DataFrame({'minmax':[in_df_npy[i,j]for i,j in enumerate(max_indices)]})
    in_df.index = index
else:
    in_df = in_df.mean(axis=1)

in_df.to_csv(args.out, sep='\t', index=True, header=False)