#!/usr/bin/env python
import pandas as pd
import numpy as np
from reddylab_utils.reddylab_plotting_utils import discrete_cmap,adjust_spines
import matplotlib
matplotlib.use('Agg')
font = {'size'   : 8}
matplotlib.rc('font', **font)
import matplotlib.pyplot as plt
import argparse

parser = argparse.ArgumentParser(formatter_class=argparse.RawTextHelpFormatter, \
description="""

Given a dataframe of proportions or counts, plot a stacked barplot where
each column in the dataframe represents a different bar and each row re

""")

optional = parser._action_groups.pop()
required = parser.add_argument_group('required arguments')

##################################################
# required args:

required.add_argument("-i", "--df", help="required, file path to dataframe", required=True)
required.add_argument("-o", "--outplot", help="required, path to outplot", required=True)

##################################################
# optional args:

optional.add_argument("--proportions", help="""optional, if indicated, then dataframe contains normalized proportions
and ymax set to 1, ymin set to 0, overrides --ymin and --ymax""", \
                    action="store_true")
optional.add_argument("--ylabel", help="optional, ylabel")
optional.add_argument("--ymax", help="optional, ymax", type=float)
optional.add_argument("--ymin", help="optional, ymin (default: %(default)s)", type=float, default=0)
optional.add_argument("--cmap", default="rainbow", 
                    help="""optional, name of matplotlib colormap, see:
                    http://matplotlib.org/examples/color/colormaps_reference.html
                    (default: %(default)s)""")
optional.add_argument("--colors", nargs="+")

##################################################
parser._action_groups.append(optional)
args = parser.parse_args()

df = pd.read_csv(args.df, sep="\t", index_col=0)

tableau20 = [(31/255., 119/255., 180/255.), 
             (174/255., 199/255., 232/255.),
             (255/255., 127/255., 14/255.),
             (255/255., 187/255., 120/255.),    
             (44/255., 160/255., 44/255.),
             (152/255., 223/255., 138/255.),
             (214/255., 39/255., 40/255.),
             (255/255., 152/255., 150/255.),    
             (148/255., 103/255., 189/255.),
             (197/255., 176/255., 213/255.),
             (140/255., 86/255., 75/255.),
             (196/255., 156/255., 148/255.),    
             (227/255., 119/255., 194/255.),
             (247/255., 182/255., 210/255.),
             (127/255., 127/255., 127/255.),
             (199/255., 199/255., 199/255.),    
             (188/255., 189/255., 34/255.),
             (219/255., 219/255., 141/255.),
             (23/255., 190/255., 207/255.),
             (158/255., 218/255., 229/255.)]    
tableau10 = tableau20[::2]

if not args.colors:
    if args.cmap == "tableau":
        if df.shape[0] <= 10:
            colors = tableau10[:df.shape[0]+1]
        else:
            colors = tableau20
    else:
        colors = discrete_cmap(df.shape[0], args.cmap)
else:
    colors = args.colors

fig, ax = plt.subplots(figsize=(1 + 0.5*df.shape[1], 3))

# plot values in dataframe as stacked barplot
x = np.arange(df.shape[1]) * 0.5 + 0.05
running_y = np.zeros_like(x)
for i, query in enumerate(df.index):
    y = np.array(df.ix[query])
    ax.bar(x, y, bottom=running_y, width=0.45, color=colors[i], label=query)
    running_y+=y


# refine plot attributes
adjust_spines(ax, ['left','bottom'])

ymax = args.ymax if args.ymax else ax.get_ylim()[1]
ymin = args.ymin if args.ymin else ax.get_ylim()[0]

ymax = ymax if not args.proportions else 1
ymin = ymin if not args.proportions else 0

ax.set_ylim((ymin, ymax))

ax.set_xticks(x)
ax.set_xticklabels(df.columns, rotation=30, ha='right')

if args.ylabel:
    ax.set_ylabel(args.ylabel, fontsize=8)


# Put a legend to the right side of the axis
handles, labels = ax.get_legend_handles_labels()
ax.legend(handles[::-1], labels[::-1], fontsize=8, loc='center left', bbox_to_anchor=(1, 0.5), frameon=False)

plt.savefig(args.outplot, bbox_inches='tight')
