#!/usr/bin/env Rscript

## Kiley Graim
## March 2018
##
## Given: 
##   A list of genes in the whole set (with scores defining the level of interest)
##   A GO annotation file (optional)
##
## Returns:
##   Enriched GO terms & some stats
##
## This script is intended for the FREYA pipeline and aims specifically at analyzing 2 way histology comparisons and CMT peps.
##   See the generic script for other usage.

###########################
###   Script setup
####################

if(!require('getopt')) {
  install.packages('getopt')
  library(getopt)
}


## usage, options and doc
argspec <- paste(get_Rscript_filename(), c(paste('runs GO enrichment using topGO. This requires a file with PEP enrichment scores and will use only the genes in that score file (include non-significant results). 
  NOTE: Unless you opt out of it, this loads humanmapping.rda and LRTtidied.rda, generated by prep_data.R, and will run pairwise histology enrichment as well as PEP enrichment.

  Usage: 
    ',get_Rscript_filename(),' -u <scores/gene name file>
  Options:
    -m <GO mapping file>      GO annotations file
    -g <human mapping file>   RDA file with the canine-human mapping (humanmapping.rda)
    -l <LRT tidied file>      RDA file with the enrichment information (LRTtidied.rda)
    -o <output directory>     Directory to write/store enrichment results
    -n <min geneset size>     Minimum number of terms necessary for a geneset to be considered for enrichment
    -c <cutoff>               Maximum value to be considered a gene of interest in a given set. Default 0.05
    -t <two-way histology>    Set this flag if you want to run the pairwise histology comparisons
        ')))

args <- commandArgs(TRUE)

## Print help if requested
if( length(args)==0 ) { args = '--help' } # If run without arguments, assume the user wants help
if ( '--help' %in% args | '-h' %in% args ) {
  write(argspec, stderr())
  quit()
}

## Set up input specs (long flag, short flag, required/optional, type)
spec <- matrix( c(
       'universe',     'u', 1, 'character',
       'mapping',      'm', 2, 'character',
       'humap',        'g', 2, 'character',
       'lrt',          'l', 2, 'character',
       'cutoff',       'c', 2, 'numeric',
       'outdir',       'o', 2, 'character',
       'two',          't', 0, 'logical',
       'num',          'n', 2, 'integer'
      ),
    ncol=4,
    byrow=TRUE
         )

opt <- getopt( spec=spec )

# Set defaults for optional parameters
if( is.null(opt$outdir) )  { opt$outdir  = './results' } 
if( is.null(opt$datadir) ) { opt$datadir = './data' } 
if( is.null(opt$cutoff) )  { opt$cutoff  = 0.05 }
if( is.null(opt$num) )     { opt$num     = 10 }
if( is.null(opt$humap) )   { opt$humap   = 'data/humanmapping.rda' }
if( is.null(opt$lrt) )     { opt$lrt     = 'user_data/LRTtidied.rda' }
opt$workingdir <- './user_data'

## If the output directory doesn't exist, create it
if(!dir.exists(opt$outdir)) {
  print(paste('Creating output directory',opt$outdir))
  system(paste('mkdir -p',opt$outdir))
}

###########################
###   Functions
####################

### Function to run topGO
run_topGO <- function(peps, pep.name, alg='weight01', ns=opt$num) {

  ## Whole gene list
  geneUniverse <- peps[, pep.name]
  # In case it's a two-hist comparison, check the column name
  if( 'EnsGene' %in% colnames(peps) ) {
    names(geneUniverse) <- peps$EnsGene
 #   names(geneUniverse) <- peps$HumanSymbol # old
  } else if ( 'gene' %in% colnames(peps) ) {
        names(geneUniverse) <- peps$gene
  }

  ## Load the GO annotations to use
  if( !is.null(opt$mapping) ) {
    geneID2GO <- try(readMappings(file=opt$mapping))
    if(inherits(geneID2GO, "try-error")) { print('ERROR: Unable to read mapping file.'); quit(save='no',status=1) }
  } else {
    #geneID2GO = getgo(peps$HumanSymbol, genome='hg19', 'geneSymbol') # old
    geneID2GO = getgo(names(geneUniverse),genome='canFam3','ensGene')
  } 

  ## How we select the genes of interest
  topDiffGenes <- function(allScore) { return(allScore < opt$cutoff) }

  ## Set the ontology (in this case we're always using BP)
  desiredOntology <- 'BP'

  ## Run topGO
  topGOdata <- new('topGOdata',
    description = paste(pep.name,desiredOntology),
    ontology = desiredOntology,
    allGenes = geneUniverse,
    geneSel = topDiffGenes,
    annot = annFUN.gene2GO,
    gene2GO = geneID2GO,
    nodeSize = ns)

  ## Calculate significance of results, adjust for multiple hypotheses
  resultKS     <- runTest(topGOdata, algorithm = alg, statistic = 'ks') # Works
  dfresult     <-GenTable(topGOdata,pvalue=resultKS,topNodes=length(resultKS@score))
  dfresult$FDR <-p.adjust(dfresult$pvalue,method='BH')
  dfresult     <- dfresult[dfresult$pvalue<0.05,]

  ## Don't return results that are significantly under-represented
  dfresult <- dfresult[ dfresult$Significant > dfresult$Expected,]

  ## Add list of genes in each set
  dfresult$Genes <- sapply(dfresult$GO.ID, function(x) { 
    paste(peps[peps$EnsGene %in% intersect(peps$EnsGene[ peps[,pep.name]<0.05], genesInTerm(topGOdata, x)[[1]]), 'HumanSymbol'], collapse=',')
    
  } )

  return( dfresult )
} # End run_topGO


###########################
###   Main function
####################

## Load libraries
print('Loading packages...');flush.console()
if(!require('topGO')) {
  install.packages('topGO')
  library(topGO)
}
if(!require('goseq')) {
  install.packages('goseq')
  library(goseq)
}
if(!require('dplyr')) {
  install.packages('dplyr')
  library(dplyr)
}

### Run topGO on each PEP
peps <- read.table(opt$universe, sep=',', header=TRUE, stringsAsFactors=FALSE) # universe?
print(paste('Successfully loaded gene universe and scores:',opt$universe)); flush.console()

pep.names <- c('Tumor_Expression_Pattern', 'Carcinoma_Expression_Pattern', 'Adenoma_Expression_Pattern')
go.res <- lapply(pep.names, function(x){ run_topGO(peps,pep.name=x) }) 
names(go.res) <- pep.names

## Run topGO on BRCA (scores generated in DESeq3.R, also provided in the repository)
print('Running the BRCA comparisons'); flush.console()
de.brca <- read.table('data/BRCA_sig_genes.csv', sep=',', header=T, row.names=1, check.names=F) # This file never changes

geneUniverse <- de.brca[,'AbsLog10FoldChange' ]
names(geneUniverse) <- rownames(de.brca)

geneID2GO = getgo(rownames(de.brca), genome='hg19', 'geneSymbol')
topDiffGenes <- function(allScore) { return(allScore < 0.5) } 
desiredOntology <- 'BP'

topGOdata <- new('topGOdata',
  description = paste('BRCA',desiredOntology),
  ontology = desiredOntology,
  allGenes = geneUniverse,
  geneSel = topDiffGenes,
  annot = annFUN.gene2GO,
  gene2GO = geneID2GO,
  nodeSize = opt$num)

resultKS          <- runTest(topGOdata, algorithm = 'weight01', statistic = 'ks') 
brca.go.res       <- GenTable(topGOdata,pvalue=resultKS,topNodes=length(resultKS@score))
brca.go.res$FDR   <- p.adjust(brca.go.res$pvalue,method='BH')
brca.go.res       <- brca.go.res[brca.go.res$pvalue<0.05,]
brca.go.res       <- brca.go.res[ brca.go.res$Significant > brca.go.res$Expected,]
brca.go.res$Genes <- sapply(brca.go.res$GO.ID, function(x) { paste(unlist(genesInTerm(topGOdata, x)), collapse=',') } )

write.table(brca.go.res, file='BRCA_GO_res.csv', sep=',', col.names=TRUE, row.names=TRUE, quote=TRUE)  # TODO: redirect to output directory
 
### Run topGO using each histology contrast m_n from LRTtidied
## Load the profile metrics
if( !is.null(opt$two) ) {
  print('Running the Histology comparisons.');flush.console()

  ## Check to make sure the required rda files exist, if yes then load them & run analysis
  ## Make sure all of the required files exist - quit if any are missing
  rda <- opt$humap
  if(!file.exists(rda)) { print(paste('ERROR: Unable to locate',rda, sep='/')); quit(save='no',status=1) }
  load(rda)
  rda <- opt$lrt
  if(!file.exists(rda)) { print(paste('ERROR: Unable to locate',rda, sep='/')); quit(save='no',status=1) }
  load(rda)

  ## Add human mappings
  LRTtidied$HumanSymbol <- NA #Not all map to human
  LRTtidied$HumanSymbol[ LRTtidied$gene %in% Map_CanEns2HumSymb_unique$Can_Ens ] <- Map_CanEns2HumSymb_unique$Hum_Symb
#  LRTtidied <- LRTtidied[! is.na(LRTtidied$HumanSymbol),]

  # Malignant vs Normal - only print significantly enriched terms (not significantly under-enriched)
  print('Processing: Malignant vs Normal');flush.console()
  dfresult <- run_topGO( as.data.frame(LRTtidied[ LRTtidied$contrast=='m_n',]),  pep.name='logFC')
  res.maligVSnormal <- dfresult # Save for later compiling
  write.table(dfresult, file=paste(opt$outdir,'2WAY_MvsN.csv',sep='/'), sep=',', col.names=TRUE, row.names=FALSE, quote=TRUE)

  # Malignant vs Benign
  print('Processing: Malignant vs Benign');flush.console()
  dfresult <- run_topGO( as.data.frame(LRTtidied[ LRTtidied$contrast=='m_b',]),  pep.name='logFC')
  write.table(dfresult, file=paste(opt$outdir,'2WAY_MvsB.csv',sep='/'), sep=',', col.names=TRUE, row.names=FALSE, quote=TRUE)


  # Benign vs Normal
  print('Processing: Benign vs Normal');flush.console()
  dfresult <- run_topGO( as.data.frame(LRTtidied[ LRTtidied$contrast=='b_n',]),  pep.name='logFC')
  write.table(dfresult, file=paste(opt$outdir,'2WAY_BvsN.csv',sep='/'), sep=',', col.names=TRUE, row.names=FALSE, quote=TRUE)
}

## Combine the results into 1 table, save to file
# P-val normal-carcinoma, Pvalue Tumor PEP, Pvalue Carcinoma PEP, Pvalue Adenoma PEP, SigGenes in Tumor PEP, SigGenes in Carcinoma PEP, SigGenes in Adenoma PEP
print('Generating the supplemental table with results.'); flush.console()

## Drop to columns we're interested in
go.res$Adenoma_Expression_Pattern   <- go.res$Adenoma_Expression_Pattern[,c('GO.ID','Term','pvalue','Genes')]
go.res$Carcinoma_Expression_Pattern <- go.res$Carcinoma_Expression_Pattern[,c('GO.ID','Term','pvalue','Genes')]
go.res$Tumor_Expression_Pattern     <- go.res$Tumor_Expression_Pattern[,c('GO.ID','Term','pvalue','Genes')]
brca.go.res                         <- brca.go.res[,c('GO.ID','Term','pvalue')]

## Change to unique column names
colnames(go.res$Adenoma_Expression_Pattern)   <- c('GO.ID','Term','PValueEnrich_in_Adenoma_Expression_Pattern','SigGenes_Adenoma_PEP')
colnames(go.res$Carcinoma_Expression_Pattern) <- c('GO.ID','Term','PValueEnrich_in_Carcinoma_Expression_Pattern','SigGenes_Carcinoma_PEP')
colnames(go.res$Tumor_Expression_Pattern)     <- c('GO.ID','Term','PValueEnrich_in_Tumor_Expression_Pattern','SigGenes_Tumor_PEP')
colnames(brca.go.res)                         <- c('GO.ID','Term','PValueEnrich_in_BRCA')

## Combine the PEP results tables into 1
res <- full_join(go.res$Tumor_Expression_Pattern, go.res$Carcinoma_Expression_Pattern, by=c('GO.ID','Term'))
res <- full_join(go.res$Adenoma_Expression_Pattern, res, by=c('GO.ID','Term'))
res <- left_join(res, brca.go.res, by=c('GO.ID','Term'))


## Add Normal vs Carcinoma comparison if option is set
## Either way, reorder columns based on which comparisons are included
if( !is.null(opt$two) ) {
  res.maligVSnormal <- res.maligVSnormal[,c('GO.ID','Term','pvalue')]
  colnames(res.maligVSnormal) <- c('GO.ID','Term','PValueEnrich_in_Normal_vs_Carcinoma')
  res <- full_join(res.maligVSnormal, res, by=c('GO.ID','Term'))
  res <- res[, c('GO.ID','Term','PValueEnrich_in_Tumor_Expression_Pattern','PValueEnrich_in_Carcinoma_Expression_Pattern','PValueEnrich_in_Adenoma_Expression_Pattern','PValueEnrich_in_BRCA','PValueEnrich_in_Normal_vs_Carcinoma','SigGenes_Tumor_PEP','SigGenes_Carcinoma_PEP','SigGenes_Adenoma_PEP')]   
} else {
  res <- res[, c('GO.ID','Term','PValueEnrich_in_Tumor_Expression_Pattern','PValueEnrich_in_Carcinoma_Expression_Pattern','PValueEnrich_in_Adenoma_Expression_Pattern','PValueEnrich_in_BRCA','SigGenes_Tumor_PEP','SigGenes_Carcinoma_PEP','SigGenes_Adenoma_PEP')]   
}

## NA gene lists to empty string
res$SigGenes_Tumor_PEP[is.na(res$SigGenes_Tumor_PEP)] <- ''
res$SigGenes_Carcinoma_PEP[is.na(res$SigGenes_Carcinoma_PEP)] <- ''
res$SigGenes_Adenoma_PEP[is.na(res$SigGenes_Adenoma_PEP)] <- ''

## NA p-values to 1
res$PValueEnrich_in_Tumor_Expression_Pattern[is.na(res$PValueEnrich_in_Tumor_Expression_Pattern)] <- 1
res$PValueEnrich_in_Carcinoma_Expression_Pattern[is.na(res$PValueEnrich_in_Carcinoma_Expression_Pattern)] <- 1
res$PValueEnrich_in_Adenoma_Expression_Pattern[is.na(res$PValueEnrich_in_Adenoma_Expression_Pattern)] <- 1
if( !is.null(opt$two) ) { res$PValueEnrich_in_Normal_vs_Carcinoma[is.na(res$PValueEnrich_in_Normal_vs_Carcinoma)] <- 1 } # Only run if two-way histology is included

## Sort by Tumor P-value
res <- res[with(res, order(PValueEnrich_in_Tumor_Expression_Pattern)),]

# Save to file
write.table(res, file=paste(opt$outdir, 'GO_Enrichment.csv', sep='/'), sep=',', col.names=TRUE, row.names=FALSE, quote=TRUE)

## Wrap it all up
print(paste('Finished, printing files to:',opt$outdir));flush.console()
