/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.variant;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import htsjdk.tribble.TribbleException;
import htsjdk.tribble.util.popgen.HardyWeinbergCalculation;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.CommonInfo;
import htsjdk.variant.variantcontext.Genotype;
import htsjdk.variant.variantcontext.GenotypeBuilder;
import htsjdk.variant.variantcontext.GenotypeLikelihoods;
import htsjdk.variant.variantcontext.GenotypesContext;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextBuilder;
import htsjdk.variant.variantcontext.VariantContextUtils;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.lang.ArrayUtils;
import org.apache.log4j.Logger;
import org.broadinstitute.gatk.utils.BaseUtils;
import org.broadinstitute.gatk.utils.GenomeLoc;
import org.broadinstitute.gatk.utils.GenomeLocParser;
import org.broadinstitute.gatk.utils.MathUtils;
import org.broadinstitute.gatk.utils.Utils;
import org.broadinstitute.gatk.utils.collections.Pair;
import org.broadinstitute.gatk.utils.exceptions.ReviewedGATKException;

public class GATKVariantContextUtils {
    private static Logger logger = Logger.getLogger(GATKVariantContextUtils.class);
    public static final int DEFAULT_PLOIDY = 2;
    public static final double SUM_GL_THRESH_NOCALL = -0.1;
    public static final String MERGE_FILTER_PREFIX = "filterIn";
    public static final String MERGE_REF_IN_ALL = "ReferenceInAll";
    public static final String MERGE_FILTER_IN_ALL = "FilteredInAll";
    public static final String MERGE_INTERSECTION = "Intersection";
    private static List<Allele>[] NOCALL_LISTS = new List[]{Collections.emptyList(), Collections.singletonList(Allele.NO_CALL), Collections.nCopies(2, Allele.NO_CALL)};

    public static boolean overlapsRegion(VariantContext variantContext, GenomeLoc region) {
        if (region == null) {
            throw new IllegalArgumentException("the active region provided cannot be null");
        }
        if (variantContext == null) {
            throw new IllegalArgumentException("the variant context provided cannot be null");
        }
        if (region.isUnmapped()) {
            return false;
        }
        if (variantContext.getEnd() < region.getStart()) {
            return false;
        }
        if (variantContext.getStart() > region.getStop()) {
            return false;
        }
        return variantContext.getChr().equals(region.getContig());
    }

    public static List<Allele> homozygousAlleleList(Allele allele, int ploidy) {
        if (allele == null || ploidy < 0) {
            throw new IllegalArgumentException();
        }
        return Collections.nCopies(ploidy, allele);
    }

    private static boolean hasIncompatibleAlleles(Collection<Allele> alleleSet1, Collection<Allele> alleleSet2) {
        Iterator<Allele> it1 = alleleSet1.iterator();
        Iterator<Allele> it2 = alleleSet2.iterator();
        while (it1.hasNext() && it2.hasNext()) {
            Allele a2;
            Allele a1 = it1.next();
            if (a1.equals(a2 = it2.next())) continue;
            return true;
        }
        return it1.hasNext() || it2.hasNext();
    }

    public static Allele determineReferenceAllele(List<VariantContext> VCs, GenomeLoc loc) {
        Allele ref = null;
        for (VariantContext vc : VCs) {
            if (!GATKVariantContextUtils.contextMatchesLoc(vc, loc)) continue;
            Allele myRef = vc.getReference();
            if (ref == null || ref.length() < myRef.length()) {
                ref = myRef;
                continue;
            }
            if (ref.length() != myRef.length() || ref.equals(myRef)) continue;
            throw new TribbleException(String.format("The provided variant file(s) have inconsistent references for the same position(s) at %s:%d, %s vs. %s", vc.getChr(), vc.getStart(), ref, myRef));
        }
        return ref;
    }

    public static int totalPloidy(VariantContext vc, int defaultPloidy) {
        if (vc == null) {
            throw new IllegalArgumentException("the vc provided cannot be null");
        }
        if (defaultPloidy < 0) {
            throw new IllegalArgumentException("the default ploidy must 0 or greater");
        }
        int result = 0;
        for (Genotype genotype : vc.getGenotypes()) {
            int declaredPloidy = genotype.getPloidy();
            result += declaredPloidy <= 0 ? defaultPloidy : declaredPloidy;
        }
        return result;
    }

    public static double getMeanAltAlleleLength(VariantContext vc) {
        double averageLength = 1.0;
        if (!vc.isSNP() && !vc.isSymbolic()) {
            int averageLengthNum = 0;
            int averageLengthDenom = 0;
            int refLength = vc.getReference().length();
            for (Allele a : vc.getAlternateAlleles()) {
                int alleleSize;
                int numAllele = vc.getCalledChrCount(a);
                if (a.length() == refLength) {
                    byte[] a_bases = a.getBases();
                    byte[] ref_bases = vc.getReference().getBases();
                    int n_mismatch = 0;
                    for (int idx = 0; idx < a_bases.length; ++idx) {
                        if (a_bases[idx] == ref_bases[idx]) continue;
                        ++n_mismatch;
                    }
                    alleleSize = n_mismatch;
                } else {
                    alleleSize = a.isSymbolic() ? 1 : Math.abs(refLength - a.length());
                }
                averageLengthNum += alleleSize * numAllele;
                averageLengthDenom += numAllele;
            }
            averageLength = (double)averageLengthNum / (double)averageLengthDenom;
        }
        return averageLength;
    }

    public static final GenomeLoc getLocation(GenomeLocParser genomeLocParser, VariantContext vc) {
        return genomeLocParser.createGenomeLoc(vc.getChr(), vc.getStart(), vc.getEnd(), true);
    }

    public static BaseUtils.BaseSubstitutionType getSNPSubstitutionType(VariantContext context) {
        if (!context.isSNP() || !context.isBiallelic()) {
            throw new IllegalStateException("Requested SNP substitution type for bialleic non-SNP " + context);
        }
        return BaseUtils.SNPSubstitutionType(context.getReference().getBases()[0], context.getAlternateAllele(0).getBases()[0]);
    }

    public static boolean isTransition(VariantContext context) {
        return GATKVariantContextUtils.getSNPSubstitutionType(context) == BaseUtils.BaseSubstitutionType.TRANSITION;
    }

    public static boolean isTransversion(VariantContext context) {
        return GATKVariantContextUtils.getSNPSubstitutionType(context) == BaseUtils.BaseSubstitutionType.TRANSVERSION;
    }

    public static boolean isTransition(Allele ref, Allele alt) {
        return BaseUtils.SNPSubstitutionType(ref.getBases()[0], alt.getBases()[0]) == BaseUtils.BaseSubstitutionType.TRANSITION;
    }

    public static boolean isTransversion(Allele ref, Allele alt) {
        return BaseUtils.SNPSubstitutionType(ref.getBases()[0], alt.getBases()[0]) == BaseUtils.BaseSubstitutionType.TRANSVERSION;
    }

    public static VariantContext reverseComplement(VariantContext vc) {
        HashMap<Allele, Allele> alleleMap = new HashMap<Allele, Allele>(vc.getAlleles().size());
        for (Allele originalAllele : vc.getAlleles()) {
            Allele newAllele = originalAllele.isNoCall() ? originalAllele : Allele.create(BaseUtils.simpleReverseComplement(originalAllele.getBases()), originalAllele.isReference());
            alleleMap.put(originalAllele, newAllele);
        }
        GenotypesContext newGenotypes = GenotypesContext.create(vc.getNSamples());
        for (Genotype genotype : vc.getGenotypes()) {
            ArrayList<Allele> newAlleles = new ArrayList<Allele>();
            for (Allele allele : genotype.getAlleles()) {
                Allele newAllele = (Allele)alleleMap.get(allele);
                if (newAllele == null) {
                    newAllele = Allele.NO_CALL;
                }
                newAlleles.add(newAllele);
            }
            newGenotypes.add(new GenotypeBuilder(genotype).alleles(newAlleles).make());
        }
        return new VariantContextBuilder(vc).alleles(alleleMap.values()).genotypes(newGenotypes).make();
    }

    @Requires(value={"vc != null", "refBasesStartingAtVCWithPad != null && refBasesStartingAtVCWithPad.length > 0"})
    public static boolean isTandemRepeat(VariantContext vc, byte[] refBasesStartingAtVCWithPad) {
        String refBasesStartingAtVCWithoutPad = new String(refBasesStartingAtVCWithPad).substring(1);
        if (!vc.isIndel()) {
            return false;
        }
        Allele ref = vc.getReference();
        for (Allele allele : vc.getAlternateAlleles()) {
            if (GATKVariantContextUtils.isRepeatAllele(ref, allele, refBasesStartingAtVCWithoutPad)) continue;
            return false;
        }
        return true;
    }

    @Requires(value={"vc != null", "refBasesStartingAtVCWithPad != null && refBasesStartingAtVCWithPad.length > 0"})
    public static Pair<List<Integer>, byte[]> getNumTandemRepeatUnits(VariantContext vc, byte[] refBasesStartingAtVCWithPad) {
        boolean VERBOSE = false;
        String refBasesStartingAtVCWithoutPad = new String(refBasesStartingAtVCWithPad).substring(1);
        if (!vc.isIndel()) {
            return null;
        }
        Allele refAllele = vc.getReference();
        byte[] refAlleleBases = Arrays.copyOfRange(refAllele.getBases(), 1, refAllele.length());
        byte[] repeatUnit = null;
        ArrayList<Integer> lengths = new ArrayList<Integer>();
        for (Allele allele : vc.getAlternateAlleles()) {
            Pair<int[], byte[]> result = GATKVariantContextUtils.getNumTandemRepeatUnits(refAlleleBases, Arrays.copyOfRange(allele.getBases(), 1, allele.length()), refBasesStartingAtVCWithoutPad.getBytes());
            int[] repetitionCount = (int[])result.first;
            if (repetitionCount[0] == 0 || repetitionCount[1] == 0) {
                return null;
            }
            if (lengths.isEmpty()) {
                lengths.add(repetitionCount[0]);
            }
            lengths.add(repetitionCount[1]);
            repeatUnit = (byte[])result.second;
        }
        return new Pair(lengths, repeatUnit);
    }

    @Deprecated
    public static Pair<int[], byte[]> getNumTandemRepeatUnits(byte[] refBases, byte[] altBases, byte[] remainingRefContext) {
        byte[] longB = altBases.length > refBases.length ? altBases : refBases;
        int repeatUnitLength = GATKVariantContextUtils.findRepeatedSubstring(longB);
        byte[] repeatUnit = Arrays.copyOf(longB, repeatUnitLength);
        int[] repetitionCount = new int[2];
        int repetitionsInRef = GATKVariantContextUtils.findNumberOfRepetitions(repeatUnit, refBases, true);
        repetitionCount[0] = GATKVariantContextUtils.findNumberOfRepetitions(repeatUnit, ArrayUtils.addAll(refBases, remainingRefContext), true) - repetitionsInRef;
        repetitionCount[1] = GATKVariantContextUtils.findNumberOfRepetitions(repeatUnit, ArrayUtils.addAll(altBases, remainingRefContext), true) - repetitionsInRef;
        return new Pair<int[], byte[]>(repetitionCount, repeatUnit);
    }

    public static int findRepeatedSubstring(byte[] bases) {
        int repLength;
        for (repLength = 1; repLength <= bases.length; ++repLength) {
            byte[] candidateRepeatUnit = Arrays.copyOf(bases, repLength);
            boolean allBasesMatch = true;
            for (int start = repLength; start < bases.length; start += repLength) {
                byte[] basePiece = Arrays.copyOfRange(bases, start, start + candidateRepeatUnit.length);
                if (Arrays.equals(candidateRepeatUnit, basePiece)) continue;
                allBasesMatch = false;
                break;
            }
            if (!allBasesMatch) continue;
            return repLength;
        }
        return repLength;
    }

    @Deprecated
    public static int findNumberOfRepetitions(byte[] repeatUnit, byte[] testString, boolean lookForward) {
        int end;
        byte[] unit;
        if (repeatUnit == null) {
            throw new IllegalArgumentException("the repeat unit cannot be null");
        }
        if (testString == null) {
            throw new IllegalArgumentException("the test string cannot be null");
        }
        int numRepeats = 0;
        if (lookForward) {
            int end2;
            byte[] unit2;
            for (int start = 0; start < testString.length && Arrays.equals(unit2 = Arrays.copyOfRange(testString, start, end2 = start + repeatUnit.length), repeatUnit); start += repeatUnit.length) {
                ++numRepeats;
            }
            return numRepeats;
        }
        for (int start = testString.length - repeatUnit.length; start >= 0 && Arrays.equals(unit = Arrays.copyOfRange(testString, start, end = start + repeatUnit.length), repeatUnit); start -= repeatUnit.length) {
            ++numRepeats;
        }
        return numRepeats;
    }

    protected static boolean isRepeatAllele(Allele ref, Allele alt, String refBasesStartingAtVCWithoutPad) {
        if (!Allele.oneIsPrefixOfOther(ref, alt)) {
            return false;
        }
        if (ref.length() > alt.length()) {
            return GATKVariantContextUtils.basesAreRepeated(ref.getBaseString(), alt.getBaseString(), refBasesStartingAtVCWithoutPad, 2);
        }
        return GATKVariantContextUtils.basesAreRepeated(alt.getBaseString(), ref.getBaseString(), refBasesStartingAtVCWithoutPad, 1);
    }

    protected static boolean basesAreRepeated(String l, String s, String ref, int minNumberOfMatches) {
        String potentialRepeat = l.substring(s.length());
        for (int i = 0; i < minNumberOfMatches; ++i) {
            int start = i * potentialRepeat.length();
            int end = (i + 1) * potentialRepeat.length();
            if (ref.length() < end) {
                return false;
            }
            String refSub = ref.substring(start, end);
            if (refSub.equals(potentialRepeat)) continue;
            return false;
        }
        return true;
    }

    public static GenotypesContext subsetAlleles(VariantContext vc, List<Allele> allelesToUse, GenotypeAssignmentMethod assignGenotypes) {
        if (vc == null) {
            throw new IllegalArgumentException("the VariantContext cannot be null");
        }
        if (allelesToUse == null) {
            throw new IllegalArgumentException("the alleles to use cannot be null");
        }
        if (allelesToUse.isEmpty()) {
            throw new IllegalArgumentException("must have alleles to use");
        }
        if (allelesToUse.get(0).isNonReference()) {
            throw new IllegalArgumentException("First allele must be the reference allele");
        }
        if (allelesToUse.size() == 1) {
            throw new IllegalArgumentException("Cannot subset to only 1 alt allele");
        }
        if (vc.getGenotypes().isEmpty()) {
            return GenotypesContext.create();
        }
        List<List<Integer>> likelihoodIndexesToUse = GATKVariantContextUtils.determineLikelihoodIndexesToUse(vc, allelesToUse);
        List<Integer> sacIndexesToUse = GATKVariantContextUtils.determineSACIndexesToUse(vc, allelesToUse);
        return GATKVariantContextUtils.createGenotypesWithSubsettedLikelihoods(vc.getGenotypes(), vc, allelesToUse, likelihoodIndexesToUse, sacIndexesToUse, assignGenotypes);
    }

    private static List<List<Integer>> determineLikelihoodIndexesToUse(VariantContext originalVC, List<Allele> allelesToUse) {
        if (originalVC == null) {
            throw new IllegalArgumentException("the original VariantContext cannot be null");
        }
        if (allelesToUse == null) {
            throw new IllegalArgumentException("the alleles to use cannot be null");
        }
        BitSet alleleIndexesToUse = GATKVariantContextUtils.getAlleleIndexBitset(originalVC, allelesToUse);
        if (alleleIndexesToUse.cardinality() == alleleIndexesToUse.size()) {
            return null;
        }
        return GATKVariantContextUtils.getLikelihoodIndexes(originalVC, alleleIndexesToUse);
    }

    public static List<Integer> determineSACIndexesToUse(VariantContext originalVC, List<Allele> allelesToUse) {
        if (originalVC == null) {
            throw new IllegalArgumentException("the original VC cannot be null");
        }
        if (allelesToUse == null) {
            throw new IllegalArgumentException("the alleles to use cannot be null");
        }
        BitSet alleleIndexesToUse = GATKVariantContextUtils.getAlleleIndexBitset(originalVC, allelesToUse);
        if (alleleIndexesToUse.cardinality() == alleleIndexesToUse.size()) {
            return null;
        }
        return GATKVariantContextUtils.getSACIndexes(alleleIndexesToUse);
    }

    private static List<List<Integer>> getLikelihoodIndexes(VariantContext originalVC, BitSet alleleIndexesToUse) {
        ArrayList<List<Integer>> likelihoodIndexesPerGenotype = new ArrayList<List<Integer>>(10);
        for (Genotype g : originalVC.getGenotypes()) {
            int numLikelihoods = GenotypeLikelihoods.numLikelihoods(originalVC.getNAlleles(), g.getPloidy());
            ArrayList<Integer> likelihoodIndexes = new ArrayList<Integer>(30);
            for (int PLindex = 0; PLindex < numLikelihoods; ++PLindex) {
                if (!GenotypeLikelihoods.getAlleles(PLindex, g.getPloidy()).stream().allMatch(i -> alleleIndexesToUse.get((int)i))) continue;
                likelihoodIndexes.add(PLindex);
            }
            likelihoodIndexesPerGenotype.add(likelihoodIndexes);
        }
        return likelihoodIndexesPerGenotype;
    }

    private static List<Integer> getSACIndexes(BitSet alleleIndexesToUse) {
        if (alleleIndexesToUse == null) {
            throw new IllegalArgumentException("the alleles to use cannot be null");
        }
        if (alleleIndexesToUse.isEmpty()) {
            throw new IllegalArgumentException("cannot have no alleles to use");
        }
        ArrayList<Integer> result = new ArrayList<Integer>(2 * alleleIndexesToUse.size());
        for (int SACindex = 0; SACindex < alleleIndexesToUse.size(); ++SACindex) {
            if (!alleleIndexesToUse.get(SACindex)) continue;
            result.add(2 * SACindex);
            result.add(2 * SACindex + 1);
        }
        return result;
    }

    private static BitSet getAlleleIndexBitset(VariantContext originalVC, List<Allele> allelesToUse) {
        if (originalVC == null) {
            throw new IllegalArgumentException("the original VC cannot be null");
        }
        if (allelesToUse == null) {
            throw new IllegalArgumentException("the alleles to use cannot be null");
        }
        int numOriginalAltAlleles = originalVC.getNAlleles() - 1;
        BitSet alleleIndexesToKeep = new BitSet(numOriginalAltAlleles + 1);
        alleleIndexesToKeep.set(0);
        for (int i = 0; i < numOriginalAltAlleles; ++i) {
            if (!allelesToUse.contains(originalVC.getAlternateAllele(i))) continue;
            alleleIndexesToKeep.set(i + 1);
        }
        return alleleIndexesToKeep;
    }

    public static int[] makeNewSACs(Genotype g, List<Integer> sacIndexesToUse) {
        if (g == null) {
            throw new IllegalArgumentException("the genotype cannot be null");
        }
        int[] oldSACs = GATKVariantContextUtils.getSACs(g);
        if (sacIndexesToUse == null) {
            return oldSACs;
        }
        int[] newSACs = new int[sacIndexesToUse.size()];
        int newIndex = 0;
        for (int oldIndex : sacIndexesToUse) {
            newSACs[newIndex++] = oldSACs[oldIndex];
        }
        return newSACs;
    }

    private static int[] getSACs(Genotype g) {
        if (g == null) {
            throw new IllegalArgumentException("the Genotype cannot be null");
        }
        if (!g.hasExtendedAttribute("SAC")) {
            throw new IllegalArgumentException("Genotype must have SAC");
        }
        if (g.getExtendedAttributes().get("SAC").getClass().equals(String.class)) {
            String SACsString = (String)g.getExtendedAttributes().get("SAC");
            ArrayList<String> stringSACs = Utils.split(SACsString, ",");
            int[] intSACs = new int[stringSACs.size()];
            int i = 0;
            for (String sac : stringSACs) {
                intSACs[i++] = Integer.parseInt(sac);
            }
            return intSACs;
        }
        if (g.getExtendedAttributes().get("SAC").getClass().equals(int[].class)) {
            return (int[])g.getExtendedAttributes().get("SAC");
        }
        throw new ReviewedGATKException("Unexpected SAC type");
    }

    private static GenotypesContext createGenotypesWithSubsettedLikelihoods(GenotypesContext originalGs, VariantContext originalVC, List<Allele> allelesToUse, List<List<Integer>> likelihoodIndexesToUse, List<Integer> sacIndexesToUse, GenotypeAssignmentMethod assignGenotypes) {
        if (originalGs == null) {
            throw new IllegalArgumentException("the original GenotypesContext cannot be null");
        }
        if (originalVC == null) {
            throw new IllegalArgumentException("the original VariantContext cannot be null");
        }
        if (allelesToUse == null) {
            throw new IllegalArgumentException("the alleles to use cannot be null");
        }
        GenotypesContext newGTs = GenotypesContext.create(originalGs.size());
        List<String> sampleIndices = originalGs.getSampleNamesOrderedByName();
        for (int k = 0; k < originalGs.size(); ++k) {
            double[] newLikelihoods;
            Genotype g = originalGs.get(sampleIndices.get(k));
            GenotypeBuilder gb = new GenotypeBuilder(g);
            if (!g.hasLikelihoods()) {
                newLikelihoods = null;
                gb.noPL();
            } else {
                int expectedNumLikelihoods = GenotypeLikelihoods.numLikelihoods(originalVC.getNAlleles(), g.getPloidy());
                double[] originalLikelihoods = g.getLikelihoods().getAsVector();
                if (likelihoodIndexesToUse == null) {
                    newLikelihoods = originalLikelihoods;
                } else if (originalLikelihoods.length != expectedNumLikelihoods) {
                    logger.debug("Wrong number of likelihoods in sample " + g.getSampleName() + " at " + originalVC + " got " + g.getLikelihoodsString() + " but expected " + expectedNumLikelihoods);
                    newLikelihoods = null;
                } else {
                    newLikelihoods = new double[likelihoodIndexesToUse.get(k).size()];
                    int newIndex = 0;
                    for (int oldIndex : likelihoodIndexesToUse.get(k)) {
                        newLikelihoods[newIndex++] = originalLikelihoods[oldIndex];
                    }
                    newLikelihoods = MathUtils.normalizeFromLog10(newLikelihoods, false, true);
                }
                if (newLikelihoods == null || originalVC.getAttributeAsInt("DP", 0) == 0 && GATKVariantContextUtils.likelihoodsAreUninformative(newLikelihoods)) {
                    gb.noPL();
                } else {
                    gb.PL(newLikelihoods);
                }
            }
            if (g.hasExtendedAttribute("SAC")) {
                int[] newSACs = GATKVariantContextUtils.makeNewSACs(g, sacIndexesToUse);
                gb.attribute("SAC", newSACs);
            }
            GATKVariantContextUtils.updateGenotypeAfterSubsetting(g.getAlleles(), g.getPloidy(), gb, assignGenotypes, newLikelihoods, allelesToUse);
            newGTs.add(gb.make());
        }
        return GATKVariantContextUtils.fixADFromSubsettedAlleles(newGTs, originalVC, allelesToUse);
    }

    private static boolean likelihoodsAreUninformative(double[] likelihoods) {
        return MathUtils.sum(likelihoods) > -0.1;
    }

    public static void updateGenotypeAfterSubsetting(List<Allele> originalGT, int ploidy, GenotypeBuilder gb, GenotypeAssignmentMethod assignmentMethod, double[] newLikelihoods, List<Allele> allelesToUse) {
        if (originalGT == null) {
            throw new IllegalArgumentException("originalGT cannot be null");
        }
        if (gb == null) {
            throw new IllegalArgumentException("gb cannot be null");
        }
        if (allelesToUse.isEmpty() || allelesToUse == null) {
            throw new IllegalArgumentException("allelesToUse cannot be empty or null");
        }
        switch (assignmentMethod) {
            case DO_NOT_ASSIGN_GENOTYPES: {
                break;
            }
            case SET_TO_NO_CALL: {
                gb.alleles(GATKVariantContextUtils.noCallAlleles(ploidy));
                gb.noGQ();
                break;
            }
            case SET_TO_NO_CALL_NO_ANNOTATIONS: {
                gb.alleles(GATKVariantContextUtils.noCallAlleles(ploidy));
                gb.noGQ();
                gb.noAD();
                gb.noPL();
                gb.noAttributes();
                break;
            }
            case USE_PLS_TO_ASSIGN: {
                if (newLikelihoods == null || GATKVariantContextUtils.likelihoodsAreUninformative(newLikelihoods)) {
                    gb.alleles(GATKVariantContextUtils.noCallAlleles(ploidy));
                    gb.noGQ();
                    break;
                }
                int PLindex = MathUtils.maxElementIndex(newLikelihoods);
                ArrayList<Allele> alleles = new ArrayList<Allele>();
                for (Integer alleleIndex : GenotypeLikelihoods.getAlleles(PLindex, ploidy)) {
                    alleles.add(allelesToUse.get(alleleIndex));
                }
                gb.alleles(alleles);
                gb.log10PError(GenotypeLikelihoods.getGQLog10FromLikelihoods(PLindex, newLikelihoods));
                break;
            }
            case BEST_MATCH_TO_ORIGINAL: {
                LinkedList<Allele> best = new LinkedList<Allele>();
                Allele ref = allelesToUse.get(0);
                for (Allele originalAllele : originalGT) {
                    best.add(allelesToUse.contains(originalAllele) || originalAllele.isNoCall() ? originalAllele : ref);
                }
                gb.alleles(best);
            }
        }
    }

    public static GenotypesContext subsetToRefOnly(VariantContext vc, int ploidy) {
        if (vc == null) {
            throw new IllegalArgumentException("vc cannot be null");
        }
        if (ploidy < 1) {
            throw new IllegalArgumentException("ploidy must be >= 1 but got " + ploidy);
        }
        GenotypesContext oldGTs = vc.getGenotypes();
        if (oldGTs.isEmpty()) {
            return oldGTs;
        }
        GenotypesContext newGTs = GenotypesContext.create(oldGTs.size());
        Allele ref = vc.getReference();
        List<Allele> diploidRefAlleles = Arrays.asList(ref, ref);
        for (Genotype g : vc.getGenotypes()) {
            int gPloidy = g.getPloidy() == 0 ? ploidy : g.getPloidy();
            List<Allele> refAlleles = Collections.nCopies(gPloidy, vc.getReference());
            GenotypeBuilder gb = new GenotypeBuilder(g.getSampleName(), refAlleles);
            if (g.hasDP()) {
                gb.DP(g.getDP());
            }
            if (g.hasGQ()) {
                gb.GQ(g.getGQ());
            }
            newGTs.add(gb.make());
        }
        return newGTs;
    }

    public static GenotypesContext assignDiploidGenotypes(VariantContext vc) {
        return GATKVariantContextUtils.subsetAlleles(vc, vc.getAlleles(), GenotypeAssignmentMethod.USE_PLS_TO_ASSIGN);
    }

    public static List<VariantContext> splitVariantContextToBiallelics(VariantContext vc) {
        return GATKVariantContextUtils.splitVariantContextToBiallelics(vc, false, GenotypeAssignmentMethod.SET_TO_NO_CALL);
    }

    public static List<VariantContext> splitVariantContextToBiallelics(VariantContext vc, boolean trimLeft, GenotypeAssignmentMethod genotypeAssignmentMethod) {
        return GATKVariantContextUtils.splitVariantContextToBiallelics(vc, trimLeft, genotypeAssignmentMethod, false);
    }

    public static List<VariantContext> splitVariantContextToBiallelics(VariantContext vc, boolean trimLeft, GenotypeAssignmentMethod genotypeAssignmentMethod, boolean keepOriginalChrCounts) {
        if (vc == null) {
            throw new IllegalArgumentException("vc cannot be null");
        }
        if (!vc.isVariant() || vc.isBiallelic()) {
            return Collections.singletonList(vc);
        }
        LinkedList<VariantContext> biallelics = new LinkedList<VariantContext>();
        GenotypeAssignmentMethod genotypeAssignmentMethodUsed = GATKVariantContextUtils.hasHetNonRef(vc.getGenotypes()) ? GenotypeAssignmentMethod.SET_TO_NO_CALL_NO_ANNOTATIONS : genotypeAssignmentMethod;
        for (Allele alt : vc.getAlternateAlleles()) {
            VariantContextBuilder builder = new VariantContextBuilder(vc);
            List<Allele> alleles = Arrays.asList(vc.getReference(), alt);
            builder.alleles((Collection<Allele>)alleles);
            for (String key : vc.getAttributes().keySet()) {
                if ((key.equals("AC") || key.equals("AF") || key.equals("AN")) && genotypeAssignmentMethodUsed != GenotypeAssignmentMethod.SET_TO_NO_CALL_NO_ANNOTATIONS) continue;
                builder.rmAttribute(key);
            }
            if (genotypeAssignmentMethodUsed != GenotypeAssignmentMethod.SET_TO_NO_CALL_NO_ANNOTATIONS && genotypeAssignmentMethodUsed != GenotypeAssignmentMethod.SET_TO_NO_CALL) {
                GATKVariantContextUtils.addInfoFiledAnnotations(vc, builder, alt, keepOriginalChrCounts);
            }
            builder.genotypes(GATKVariantContextUtils.subsetAlleles(vc, alleles, genotypeAssignmentMethodUsed));
            VariantContext trimmed = GATKVariantContextUtils.trimAlleles(builder.make(), trimLeft, true);
            biallelics.add(trimmed);
        }
        return biallelics;
    }

    private static boolean hasHetNonRef(GenotypesContext genotypesContext) {
        for (Genotype gt : genotypesContext) {
            if (!gt.isHetNonRef()) continue;
            return true;
        }
        return false;
    }

    public static Genotype removePLsAndAD(Genotype g) {
        return g.hasLikelihoods() || g.hasAD() ? new GenotypeBuilder(g).noPL().noAD().make() : g;
    }

    public static VariantContext simpleMerge(Collection<VariantContext> unsortedVCs, List<String> priorityListOfVCs, FilteredRecordMergeType filteredRecordMergeType, GenotypeMergeType genotypeMergeOptions, boolean annotateOrigin, boolean printMessages, String setKey, boolean filteredAreUncalled, boolean mergeInfoWithMaxAC) {
        int originalNumOfVCs = priorityListOfVCs == null ? 0 : priorityListOfVCs.size();
        return GATKVariantContextUtils.simpleMerge(unsortedVCs, priorityListOfVCs, originalNumOfVCs, filteredRecordMergeType, genotypeMergeOptions, annotateOrigin, printMessages, setKey, filteredAreUncalled, mergeInfoWithMaxAC);
    }

    public static VariantContext simpleMerge(Collection<VariantContext> unsortedVCs, List<String> priorityListOfVCs, int originalNumOfVCs, FilteredRecordMergeType filteredRecordMergeType, GenotypeMergeType genotypeMergeOptions, boolean annotateOrigin, boolean printMessages, String setKey, boolean filteredAreUncalled, boolean mergeInfoWithMaxAC) {
        if (unsortedVCs == null || unsortedVCs.isEmpty()) {
            return null;
        }
        if (priorityListOfVCs != null && originalNumOfVCs != priorityListOfVCs.size()) {
            throw new IllegalArgumentException("the number of the original VariantContexts must be the same as the number of VariantContexts in the priority list");
        }
        if (annotateOrigin && priorityListOfVCs == null && originalNumOfVCs == 0) {
            throw new IllegalArgumentException("Cannot merge calls and annotate their origins without a complete priority list of VariantContexts or the number of original VariantContexts");
        }
        List<VariantContext> preFilteredVCs = GATKVariantContextUtils.sortVariantContextsByPriority(unsortedVCs, priorityListOfVCs, genotypeMergeOptions);
        ArrayList<VariantContext> VCs = new ArrayList<VariantContext>();
        for (VariantContext vc : preFilteredVCs) {
            if (filteredAreUncalled && !vc.isNotFiltered()) continue;
            VCs.add(vc);
        }
        if (VCs.isEmpty()) {
            return null;
        }
        VariantContext first = (VariantContext)VCs.get(0);
        String name = first.getSource();
        Allele refAllele = GATKVariantContextUtils.determineReferenceAllele(VCs);
        LinkedHashSet<Allele> alleles = new LinkedHashSet<Allele>();
        HashSet<String> filters = new HashSet<String>();
        LinkedHashMap<String, Object> attributes = new LinkedHashMap<String, Object>();
        HashSet<String> inconsistentAttributes = new HashSet<String>();
        HashSet<String> variantSources = new HashSet<String>();
        LinkedHashSet<String> rsIDs = new LinkedHashSet<String>(1);
        HashMap<String, Integer> nonBooleanAttributeOccurrences = new HashMap<String, Integer>();
        VariantContext longestVC = first;
        int depth = 0;
        int maxAC = -1;
        LinkedHashMap<String, Object> attributesWithMaxAC = new LinkedHashMap<String, Object>();
        double log10PError = 1.0;
        boolean anyVCHadFiltersApplied = false;
        VariantContext vcWithMaxAC = null;
        GenotypesContext genotypes = GenotypesContext.create();
        int nFiltered = 0;
        boolean remapped = false;
        for (VariantContext vc : VCs) {
            if (longestVC.getStart() != vc.getStart()) {
                throw new IllegalStateException("BUG: attempting to merge VariantContexts with different start sites: first=" + first.toString() + " second=" + vc.toString());
            }
            if (VariantContextUtils.getSize(vc) > VariantContextUtils.getSize(longestVC)) {
                longestVC = vc;
            }
            nFiltered += vc.isFiltered() ? 1 : 0;
            if (vc.isVariant()) {
                variantSources.add(vc.getSource());
            }
            AlleleMapper alleleMapping = GATKVariantContextUtils.resolveIncompatibleAlleles(refAllele, vc, alleles);
            remapped = remapped || alleleMapping.needsRemapping();
            alleles.addAll(alleleMapping.values());
            GATKVariantContextUtils.mergeGenotypes(genotypes, vc, alleleMapping, genotypeMergeOptions == GenotypeMergeType.UNIQUIFY);
            if (log10PError == 1.0) {
                log10PError = vc.getLog10PError();
            }
            filters.addAll(vc.getFilters());
            anyVCHadFiltersApplied |= vc.filtersWereApplied();
            if (vc.hasAttribute("DP")) {
                depth += vc.getAttributeAsInt("DP", 0);
            }
            if (vc.hasID()) {
                rsIDs.add(vc.getID());
            }
            if (mergeInfoWithMaxAC && vc.hasAttribute("AC")) {
                String rawAlleleCounts = vc.getAttributeAsString("AC", null);
                if (rawAlleleCounts.contains(",")) {
                    List<String> list = Arrays.asList(rawAlleleCounts.substring(1, rawAlleleCounts.length() - 1).split(","));
                    for (String alleleCount : list) {
                        int ac = Integer.valueOf(alleleCount.trim());
                        if (ac <= maxAC) continue;
                        maxAC = ac;
                        vcWithMaxAC = vc;
                    }
                } else {
                    int n = Integer.valueOf(rawAlleleCounts);
                    if (n > maxAC) {
                        maxAC = n;
                        vcWithMaxAC = vc;
                    }
                }
            }
            for (Map.Entry entry : vc.getAttributes().entrySet()) {
                boolean boundIsMissingValue;
                String key = (String)entry.getKey();
                Object value = entry.getValue();
                if (!(value instanceof Boolean)) {
                    if (nonBooleanAttributeOccurrences.containsKey(key)) {
                        nonBooleanAttributeOccurrences.put(key, (Integer)nonBooleanAttributeOccurrences.get(key) + 1);
                    } else {
                        nonBooleanAttributeOccurrences.put(key, 1);
                    }
                }
                if (inconsistentAttributes.contains(key)) continue;
                boolean alreadyFound = attributes.containsKey(key);
                Object boundValue = attributes.get(key);
                boolean bl = boundIsMissingValue = alreadyFound && boundValue.equals(".");
                if (alreadyFound && !boundValue.equals(value) && !boundIsMissingValue) {
                    inconsistentAttributes.add(key);
                    attributes.remove(key);
                    continue;
                }
                if (alreadyFound && !boundIsMissingValue) continue;
                attributes.put(key, value);
            }
        }
        nonBooleanAttributeOccurrences.entrySet().stream().filter(a -> (Integer)a.getValue() < VCs.size()).map(a -> (String)a.getKey()).forEach(attributes::remove);
        for (VariantContext vc : VCs) {
            if (vc.getAlleles().size() == 1 || !GATKVariantContextUtils.hasIncompatibleAlleles(alleles, vc.getAlleles())) continue;
            if (!genotypes.isEmpty()) {
                logger.debug(String.format("Stripping PLs at %s:%d-%d due to incompatible alleles merged=%s vs. single=%s", vc.getChr(), vc.getStart(), vc.getEnd(), alleles, vc.getAlleles()));
            }
            genotypes = GATKVariantContextUtils.stripPLsAndAD(genotypes);
            VariantContextUtils.calculateChromosomeCounts(vc, attributes, true);
            break;
        }
        if (mergeInfoWithMaxAC && vcWithMaxAC != null) {
            attributesWithMaxAC.putAll(vcWithMaxAC.getAttributes());
        }
        if (filteredRecordMergeType == FilteredRecordMergeType.KEEP_IF_ANY_UNFILTERED && nFiltered != VCs.size() || filteredRecordMergeType == FilteredRecordMergeType.KEEP_UNCONDITIONAL) {
            filters.clear();
        }
        if (annotateOrigin) {
            String setValue;
            if (nFiltered == 0 && variantSources.size() == originalNumOfVCs) {
                setValue = MERGE_INTERSECTION;
            } else if (nFiltered == VCs.size()) {
                setValue = MERGE_FILTER_IN_ALL;
            } else if (variantSources.isEmpty()) {
                setValue = MERGE_REF_IN_ALL;
            } else {
                LinkedHashSet<String> s = new LinkedHashSet<String>();
                for (VariantContext vc : VCs) {
                    if (!vc.isVariant()) continue;
                    s.add(vc.isFiltered() ? MERGE_FILTER_PREFIX + vc.getSource() : vc.getSource());
                }
                setValue = Utils.join("-", s);
            }
            if (setKey != null) {
                attributes.put(setKey, setValue);
                if (mergeInfoWithMaxAC && vcWithMaxAC != null) {
                    attributesWithMaxAC.put(setKey, setValue);
                }
            }
        }
        if (depth > 0) {
            attributes.put("DP", String.valueOf(depth));
        }
        String ID = rsIDs.isEmpty() ? "." : Utils.join(",", rsIDs);
        VariantContextBuilder builder = new VariantContextBuilder().source(name).id(ID);
        builder.loc(longestVC.getChr(), longestVC.getStart(), longestVC.getEnd());
        builder.alleles(alleles);
        builder.genotypes(genotypes);
        builder.log10PError(log10PError);
        if (anyVCHadFiltersApplied) {
            builder.filters(filters.isEmpty() ? filters : new TreeSet(filters));
        }
        builder.attributes(new TreeMap(mergeInfoWithMaxAC ? attributesWithMaxAC : attributes));
        VariantContext merged = builder.make();
        VariantContextUtils.calculateChromosomeCounts(merged, attributes, true);
        if (printMessages && remapped) {
            System.out.printf("Remapped => %s%n", merged);
        }
        return merged;
    }

    public static GenotypesContext stripPLsAndAD(GenotypesContext genotypes) {
        GenotypesContext newGs = GenotypesContext.create(genotypes.size());
        for (Genotype g : genotypes) {
            newGs.add(GATKVariantContextUtils.removePLsAndAD(g));
        }
        return newGs;
    }

    public static GenotypesContext updatePLsSACsAD(VariantContext selectedVC, VariantContext originalVC) {
        int numOriginalAlleles;
        if (selectedVC == null) {
            throw new IllegalArgumentException("the selected VariantContext cannot be null");
        }
        if (originalVC == null) {
            throw new IllegalArgumentException("the original VariantContext cannot be null");
        }
        int numNewAlleles = selectedVC.getAlleles().size();
        if (numNewAlleles > (numOriginalAlleles = originalVC.getAlleles().size())) {
            throw new IllegalArgumentException("Attempting to fix PLs, SACs and AD from what appears to be a *combined* VCF and not a selected one");
        }
        GenotypesContext oldGs = selectedVC.getGenotypes();
        if (numNewAlleles == numOriginalAlleles) {
            return oldGs;
        }
        return GATKVariantContextUtils.fixGenotypesFromSubsettedAlleles(oldGs, originalVC, selectedVC.getAlleles());
    }

    private static GenotypesContext fixGenotypesFromSubsettedAlleles(GenotypesContext originalGs, VariantContext originalVC, List<Allele> allelesToUse) {
        if (originalGs == null) {
            throw new IllegalArgumentException("the selected GenotypesContext cannot be null");
        }
        if (originalVC == null) {
            throw new IllegalArgumentException("the original VariantContext cannot be null");
        }
        if (allelesToUse == null) {
            throw new IllegalArgumentException("the alleles to use cannot be null");
        }
        List<List<Integer>> likelihoodIndexesToUse = GATKVariantContextUtils.determineLikelihoodIndexesToUse(originalVC, allelesToUse);
        List<Integer> sacIndexesToUse = GATKVariantContextUtils.determineSACIndexesToUse(originalVC, allelesToUse);
        return GATKVariantContextUtils.createGenotypesWithSubsettedLikelihoods(originalGs, originalVC, allelesToUse, likelihoodIndexesToUse, sacIndexesToUse, GenotypeAssignmentMethod.DO_NOT_ASSIGN_GENOTYPES);
    }

    public static GenotypesContext fixADFromSubsettedAlleles(GenotypesContext originalGs, VariantContext originalVC, List<Allele> allelesToUse) {
        if (originalGs == null) {
            throw new IllegalArgumentException("the original Gs cannot be null");
        }
        if (originalVC == null) {
            throw new IllegalArgumentException("the original VC cannot be null");
        }
        if (allelesToUse == null) {
            throw new IllegalArgumentException("the alleles to use list cannot be null");
        }
        BitSet alleleIndexesToUse = GATKVariantContextUtils.getAlleleIndexBitset(originalVC, allelesToUse);
        GenotypesContext newGTs = GenotypesContext.create(originalGs.size());
        List<String> sampleIndices = originalGs.getSampleNamesOrderedByName();
        for (int k = 0; k < originalGs.size(); ++k) {
            Genotype g = originalGs.get(sampleIndices.get(k));
            newGTs.add(GATKVariantContextUtils.fixAD(g, alleleIndexesToUse));
        }
        return newGTs;
    }

    private static Genotype fixAD(Genotype genotype, BitSet alleleIndexesToUse) {
        if (!genotype.hasAD()) {
            return genotype;
        }
        GenotypeBuilder builder = new GenotypeBuilder(genotype);
        int[] oldAD = genotype.getAD();
        int[] newAD = new int[alleleIndexesToUse.cardinality()];
        int currentIndex = 0;
        int i = alleleIndexesToUse.nextSetBit(0);
        while (i >= 0) {
            if (i >= oldAD.length) {
                throw new IllegalStateException("AD has " + oldAD.length + " items. It should have at least " + (i + 1) + ".");
            }
            newAD[currentIndex++] = oldAD[i];
            i = alleleIndexesToUse.nextSetBit(i + 1);
        }
        return builder.AD(newAD).make();
    }

    private static Allele determineReferenceAllele(List<VariantContext> VCs) {
        return GATKVariantContextUtils.determineReferenceAllele(VCs, null);
    }

    public static boolean contextMatchesLoc(VariantContext vc, GenomeLoc loc) {
        return loc == null || loc.getStart() == vc.getStart();
    }

    private static AlleleMapper resolveIncompatibleAlleles(Allele refAllele, VariantContext vc, LinkedHashSet<Allele> allAlleles) {
        if (refAllele.equals(vc.getReference())) {
            return new AlleleMapper(vc);
        }
        Map<Allele, Allele> map = GATKVariantContextUtils.createAlleleMapping(refAllele, vc, allAlleles);
        map.put(vc.getReference(), refAllele);
        return new AlleleMapper(map);
    }

    protected static Map<Allele, Allele> createAlleleMapping(Allele refAllele, VariantContext oneVC, Collection<Allele> currentAlleles) {
        Allele myRef = oneVC.getReference();
        if (refAllele.length() <= myRef.length()) {
            throw new IllegalStateException("BUG: myRef=" + myRef + " is longer than refAllele=" + refAllele);
        }
        byte[] extraBases = Arrays.copyOfRange(refAllele.getBases(), myRef.length(), refAllele.length());
        HashMap<Allele, Allele> map = new HashMap<Allele, Allele>();
        for (Allele a : oneVC.getAlternateAlleles()) {
            if (GATKVariantContextUtils.isUsableAlternateAllele(a)) {
                Allele extended = Allele.extend(a, extraBases);
                for (Allele b : currentAlleles) {
                    if (!extended.equals(b)) continue;
                    extended = b;
                }
                map.put(a, extended);
                continue;
            }
            if (a.isReference()) continue;
            map.put(a, a);
        }
        return map;
    }

    private static boolean isUsableAlternateAllele(Allele allele) {
        return !allele.isReference() && !allele.isSymbolic() && allele != Allele.SPAN_DEL;
    }

    public static List<VariantContext> sortVariantContextsByPriority(Collection<VariantContext> unsortedVCs, List<String> priorityListOfVCs, GenotypeMergeType mergeOption) {
        if (mergeOption == GenotypeMergeType.PRIORITIZE && priorityListOfVCs == null) {
            throw new IllegalArgumentException("Cannot merge calls by priority with a null priority list");
        }
        if (priorityListOfVCs == null || mergeOption == GenotypeMergeType.UNSORTED) {
            return new ArrayList<VariantContext>(unsortedVCs);
        }
        ArrayList<VariantContext> sorted = new ArrayList<VariantContext>(unsortedVCs);
        Collections.sort(sorted, new CompareByPriority(priorityListOfVCs));
        return sorted;
    }

    private static void mergeGenotypes(GenotypesContext mergedGenotypes, VariantContext oneVC, AlleleMapper alleleMapping, boolean uniquifySamples) {
        for (Genotype g : oneVC.getGenotypes()) {
            String name = GATKVariantContextUtils.mergedSampleName(oneVC.getSource(), g.getSampleName(), uniquifySamples);
            if (mergedGenotypes.containsSample(name)) continue;
            Genotype newG = g;
            if (uniquifySamples || alleleMapping.needsRemapping()) {
                List<Allele> alleles = alleleMapping.needsRemapping() ? alleleMapping.remap(g.getAlleles()) : g.getAlleles();
                newG = new GenotypeBuilder(g).name(name).alleles(alleles).make();
            }
            mergedGenotypes.add(newG);
        }
    }

    private static synchronized void ensureNoCallListsCapacity(int capacity) {
        int currentCapacity = NOCALL_LISTS.length - 1;
        if (currentCapacity >= capacity) {
            return;
        }
        NOCALL_LISTS = Arrays.copyOf(NOCALL_LISTS, Math.max(capacity, currentCapacity << 1) + 1);
        for (int i = currentCapacity + 1; i < NOCALL_LISTS.length; ++i) {
            GATKVariantContextUtils.NOCALL_LISTS[i] = Collections.nCopies(i, Allele.NO_CALL);
        }
    }

    public static List<Allele> noCallAlleles(int ploidy) {
        if (NOCALL_LISTS.length <= ploidy) {
            GATKVariantContextUtils.ensureNoCallListsCapacity(ploidy);
        }
        return NOCALL_LISTS[ploidy];
    }

    protected static int calculatePLindexFromUnorderedIndexes(int originalIndex1, int originalIndex2) {
        return originalIndex2 < originalIndex1 ? GenotypeLikelihoods.calculatePLindex(originalIndex2, originalIndex1) : GenotypeLikelihoods.calculatePLindex(originalIndex1, originalIndex2);
    }

    public static String mergedSampleName(String trackName, String sampleName, boolean uniquify) {
        return uniquify ? sampleName + "." + trackName : sampleName;
    }

    public static VariantContext reverseTrimAlleles(VariantContext inputVC) {
        return GATKVariantContextUtils.trimAlleles(inputVC, false, true);
    }

    public static VariantContext forwardTrimAlleles(VariantContext inputVC) {
        return GATKVariantContextUtils.trimAlleles(inputVC, true, false);
    }

    @Ensures(value={"result != null"})
    public static VariantContext trimAlleles(VariantContext inputVC, boolean trimForward, boolean trimReverse) {
        if (inputVC == null) {
            throw new IllegalArgumentException("inputVC cannot be null");
        }
        if (inputVC.getNAlleles() <= 1 || inputVC.isSNP()) {
            return inputVC;
        }
        int revTrim = trimReverse ? GATKVariantContextUtils.computeReverseClipping(inputVC.getAlleles(), inputVC.getReference().getDisplayString().getBytes()) : 0;
        VariantContext revTrimVC = GATKVariantContextUtils.trimAlleles(inputVC, -1, revTrim);
        int fwdTrim = trimForward ? GATKVariantContextUtils.computeForwardClipping(revTrimVC.getAlleles()) : -1;
        VariantContext vc = GATKVariantContextUtils.trimAlleles(revTrimVC, fwdTrim, 0);
        return vc;
    }

    @Requires(value={"inputVC != null"})
    @Ensures(value={"result != null"})
    protected static VariantContext trimAlleles(VariantContext inputVC, int fwdTrimEnd, int revTrim) {
        if (fwdTrimEnd == -1 && revTrim == 0) {
            return inputVC;
        }
        LinkedList<Allele> alleles = new LinkedList<Allele>();
        HashMap<Allele, Allele> originalToTrimmedAlleleMap = new HashMap<Allele, Allele>();
        for (Allele a : inputVC.getAlleles()) {
            if (a.isSymbolic()) {
                alleles.add(a);
                originalToTrimmedAlleleMap.put(a, a);
                continue;
            }
            byte[] newBases = Arrays.copyOfRange(a.getBases(), fwdTrimEnd + 1, a.length() - revTrim);
            Allele trimmedAllele = Allele.create(newBases, a.isReference());
            alleles.add(trimmedAllele);
            originalToTrimmedAlleleMap.put(a, trimmedAllele);
        }
        AlleleMapper alleleMapper = new AlleleMapper(originalToTrimmedAlleleMap);
        GenotypesContext genotypes = GATKVariantContextUtils.updateGenotypesWithMappedAlleles(inputVC.getGenotypes(), alleleMapper);
        int start = inputVC.getStart() + (fwdTrimEnd + 1);
        VariantContextBuilder builder = new VariantContextBuilder(inputVC);
        builder.start(start);
        builder.stop(start + ((Allele)alleles.get(0)).length() - 1);
        builder.alleles((Collection<Allele>)alleles);
        builder.genotypes(genotypes);
        return builder.make();
    }

    @Requires(value={"originalGenotypes != null && alleleMapper != null"})
    protected static GenotypesContext updateGenotypesWithMappedAlleles(GenotypesContext originalGenotypes, AlleleMapper alleleMapper) {
        GenotypesContext updatedGenotypes = GenotypesContext.create(originalGenotypes.size());
        for (Genotype genotype : originalGenotypes) {
            List<Allele> updatedAlleles = alleleMapper.remap(genotype.getAlleles());
            updatedGenotypes.add(new GenotypeBuilder(genotype).alleles(updatedAlleles).make());
        }
        return updatedGenotypes;
    }

    public static int computeReverseClipping(List<Allele> unclippedAlleles, byte[] ref) {
        int clipping = 0;
        boolean stillClipping = true;
        while (stillClipping) {
            for (Allele a : unclippedAlleles) {
                if (a.isSymbolic()) continue;
                if (a.length() - clipping == 0) {
                    return clipping - 1;
                }
                if (a.length() - clipping <= 0 || a.length() == 0) {
                    stillClipping = false;
                    continue;
                }
                if (ref.length == clipping) {
                    return -1;
                }
                if (a.getBases()[a.length() - clipping - 1] == ref[ref.length - clipping - 1]) continue;
                stillClipping = false;
            }
            if (!stillClipping) continue;
            ++clipping;
        }
        return clipping;
    }

    public static int computeForwardClipping(List<Allele> unclippedAlleles) {
        if (unclippedAlleles.size() <= 1) {
            return -1;
        }
        int minAlleleLength = Integer.MAX_VALUE;
        for (Allele a : unclippedAlleles) {
            if (a.isSymbolic()) {
                return -1;
            }
            minAlleleLength = Math.min(minAlleleLength, a.length());
        }
        byte[] firstAlleleBases = unclippedAlleles.get(0).getBases();
        int indexOflastSharedBase = -1;
        int i = 0;
        while (i < minAlleleLength - 1) {
            byte base = firstAlleleBases[i];
            for (Allele allele : unclippedAlleles) {
                if (allele.getBases()[i] == base) continue;
                return indexOflastSharedBase;
            }
            indexOflastSharedBase = i++;
        }
        return indexOflastSharedBase;
    }

    public static double computeHardyWeinbergPvalue(VariantContext vc) {
        if (vc.getCalledChrCount() == 0) {
            return 0.0;
        }
        return HardyWeinbergCalculation.hwCalculate(vc.getHomRefCount(), vc.getHetCount(), vc.getHomVarCount());
    }

    public static boolean requiresPaddingBase(List<String> alleles) {
        for (String allele : alleles) {
            if (!allele.isEmpty()) continue;
            return true;
        }
        int clipping = 0;
        Character currentBase = null;
        while (true) {
            for (String allele : alleles) {
                if (allele.length() - clipping == 0) {
                    return true;
                }
                char myBase = allele.charAt(clipping);
                if (currentBase == null) {
                    currentBase = Character.valueOf(myBase);
                    continue;
                }
                if (currentBase.charValue() == myBase) continue;
                return false;
            }
            ++clipping;
            currentBase = null;
        }
    }

    private static final Map<String, Object> subsetAttributes(CommonInfo igc, Collection<String> keysToPreserve) {
        HashMap<String, Object> attributes = new HashMap<String, Object>(keysToPreserve.size());
        for (String key : keysToPreserve) {
            if (!igc.hasAttribute(key)) continue;
            attributes.put(key, igc.getAttribute(key));
        }
        return attributes;
    }

    @Deprecated
    public static VariantContext pruneVariantContext(VariantContext vc, Collection<String> keysToPreserve) {
        return GATKVariantContextUtils.pruneVariantContext(new VariantContextBuilder(vc), keysToPreserve).make();
    }

    public static VariantContextBuilder pruneVariantContext(VariantContextBuilder builder, Collection<String> keysToPreserve) {
        VariantContext vc = builder.make();
        if (keysToPreserve == null) {
            keysToPreserve = Collections.emptyList();
        }
        Map<String, Object> attributes = GATKVariantContextUtils.subsetAttributes(vc.getCommonInfo(), keysToPreserve);
        GenotypesContext genotypes = GenotypesContext.create(vc.getNSamples());
        for (Genotype g : vc.getGenotypes()) {
            GenotypeBuilder gb = new GenotypeBuilder(g);
            gb.noAD().noDP().noPL().noAttributes();
            genotypes.add(gb.make());
        }
        return builder.genotypes(genotypes).attributes(attributes);
    }

    public static boolean allelesAreSubset(VariantContext vc1, VariantContext vc2) {
        if (!vc1.getReference().equals(vc2.getReference())) {
            return false;
        }
        for (Allele a : vc1.getAlternateAlleles()) {
            if (vc2.getAlternateAlleles().contains(a)) continue;
            return false;
        }
        return true;
    }

    public static Map<VariantContext.Type, List<VariantContext>> separateVariantContextsByType(Collection<VariantContext> VCs) {
        if (VCs == null) {
            throw new IllegalArgumentException("VCs cannot be null.");
        }
        HashMap<VariantContext.Type, List<VariantContext>> mappedVCs = new HashMap<VariantContext.Type, List<VariantContext>>();
        for (VariantContext vc : VCs) {
            VariantContext.Type vcType = vc.getType();
            boolean addtoOwnList = true;
            block1: for (VariantContext.Type type : VariantContext.Type.values()) {
                if (type.equals((Object)vcType) || !mappedVCs.containsKey((Object)type)) continue;
                List<VariantContext> vcList = mappedVCs.get((Object)type);
                for (int k = 0; k < vcList.size(); ++k) {
                    VariantContext otherVC = vcList.get(k);
                    if (GATKVariantContextUtils.allelesAreSubset(otherVC, vc)) {
                        vcList.remove(k);
                        if (vcList.isEmpty()) {
                            mappedVCs.remove((Object)type);
                        }
                        if (!mappedVCs.containsKey((Object)vcType)) {
                            mappedVCs.put(vcType, new ArrayList());
                        }
                        mappedVCs.get((Object)vcType).add(otherVC);
                        continue block1;
                    }
                    if (!GATKVariantContextUtils.allelesAreSubset(vc, otherVC)) continue;
                    mappedVCs.get((Object)type).add(vc);
                    addtoOwnList = false;
                    continue block1;
                }
            }
            if (!addtoOwnList) continue;
            if (!mappedVCs.containsKey((Object)vcType)) {
                mappedVCs.put(vcType, new ArrayList());
            }
            mappedVCs.get((Object)vcType).add(vc);
        }
        return mappedVCs;
    }

    public static VariantContext purgeUnallowedGenotypeAttributes(VariantContext vc, Set<String> allowedAttributes) {
        if (allowedAttributes == null) {
            return vc;
        }
        GenotypesContext newGenotypes = GenotypesContext.create(vc.getNSamples());
        for (Genotype genotype : vc.getGenotypes()) {
            HashMap<String, Object> attrs = new HashMap<String, Object>();
            for (Map.Entry<String, Object> attr : genotype.getExtendedAttributes().entrySet()) {
                if (!allowedAttributes.contains(attr.getKey())) continue;
                attrs.put(attr.getKey(), attr.getValue());
            }
            newGenotypes.add(new GenotypeBuilder(genotype).attributes(attrs).make());
        }
        return new VariantContextBuilder(vc).genotypes(newGenotypes).make();
    }

    public static VariantContext makeFromAlleles(String name, String contig, int start, List<String> alleleStrings) {
        if (alleleStrings == null || alleleStrings.isEmpty()) {
            throw new IllegalArgumentException("alleleStrings must be non-empty, non-null list");
        }
        LinkedList<Allele> alleles = new LinkedList<Allele>();
        int length = alleleStrings.get(0).length();
        boolean first = true;
        for (String alleleString : alleleStrings) {
            alleles.add(Allele.create(alleleString, first));
            first = false;
        }
        return new VariantContextBuilder(name, contig, start, start + length - 1, alleles).make();
    }

    public static List<VariantContext> splitIntoPrimitiveAlleles(VariantContext vc) {
        byte[] alt;
        if (vc == null) {
            throw new IllegalArgumentException("Trying to break a null Variant Context into primitive parts");
        }
        if (!vc.isBiallelic()) {
            throw new IllegalArgumentException("Trying to break a multi-allelic Variant Context into primitive parts");
        }
        if (!vc.isMNP()) {
            return Arrays.asList(vc);
        }
        byte[] ref = vc.getReference().getBases();
        if (ref.length != (alt = vc.getAlternateAllele(0).getBases()).length) {
            throw new IllegalStateException("ref and alt alleles for MNP have different lengths");
        }
        ArrayList<VariantContext> result = new ArrayList<VariantContext>(ref.length);
        for (int i = 0; i < ref.length; ++i) {
            if (ref[i] == alt[i]) continue;
            Allele newRefAllele = Allele.create(ref[i], true);
            Allele newAltAllele = Allele.create(alt[i], false);
            VariantContextBuilder newVC = new VariantContextBuilder(vc).start(vc.getStart() + i).stop(vc.getStart() + i).alleles((Collection<Allele>)Arrays.asList(newRefAllele, newAltAllele));
            HashMap<Allele, Allele> alleleMap = new HashMap<Allele, Allele>();
            alleleMap.put(vc.getReference(), newRefAllele);
            alleleMap.put(vc.getAlternateAllele(0), newAltAllele);
            GenotypesContext newGenotypes = GATKVariantContextUtils.updateGenotypesWithMappedAlleles(vc.getGenotypes(), new AlleleMapper(alleleMap));
            result.add(newVC.genotypes(newGenotypes).make());
        }
        if (result.isEmpty()) {
            result.add(vc);
        }
        return result;
    }

    public static boolean equalSites(VariantContext vc1, VariantContext vc2) {
        if (vc1 == null) {
            throw new IllegalArgumentException("vc1 cannot be null");
        }
        if (vc2 == null) {
            throw new IllegalArgumentException("vc2 cannot be null");
        }
        if (vc1.getStart() != vc2.getStart()) {
            return false;
        }
        if (vc1.getEnd() != vc2.getEnd()) {
            return false;
        }
        if (!vc1.getChr().equals(vc2.getChr())) {
            return false;
        }
        return vc1.getAlleles().equals(vc2.getAlleles());
    }

    public static int indexOfAllele(VariantContext vc, Allele allele, boolean ignoreRefState, boolean considerRefAllele, boolean useEquals) {
        if (allele == null) {
            throw new IllegalArgumentException();
        }
        return useEquals ? GATKVariantContextUtils.indexOfEqualAllele(vc, allele, ignoreRefState, considerRefAllele) : GATKVariantContextUtils.indexOfSameAllele(vc, allele, considerRefAllele);
    }

    public static int indexOfAltAllele(VariantContext vc, Allele allele, boolean useEquals) {
        int absoluteIndex = GATKVariantContextUtils.indexOfAllele(vc, allele, true, false, useEquals);
        return absoluteIndex == -1 ? -1 : absoluteIndex - 1;
    }

    private static int indexOfEqualAllele(VariantContext vc, Allele allele, boolean ignoreRefState, boolean considerRefAllele) {
        int i = 0;
        for (Allele a : vc.getAlleles()) {
            if (a.equals(allele, ignoreRefState)) {
                return i == 0 ? (considerRefAllele ? 0 : -1) : i;
            }
            ++i;
        }
        return -1;
    }

    private static int indexOfSameAllele(VariantContext vc, Allele allele, boolean considerRefAllele) {
        int i = 0;
        for (Allele a : vc.getAlleles()) {
            if (a == allele) {
                return i == 0 ? (considerRefAllele ? 0 : -1) : i;
            }
            ++i;
        }
        return -1;
    }

    private static void addInfoFiledAnnotations(VariantContext vc, VariantContextBuilder builder, Allele altAllele, boolean keepOriginalChrCounts) {
        boolean keepOriginal;
        if (vc == null) {
            throw new IllegalArgumentException("the variant context cannot be null");
        }
        if (builder == null) {
            throw new IllegalArgumentException("the variant context builder cannot be null");
        }
        if (builder.getAlleles() == null) {
            throw new IllegalArgumentException("the variant context builder alleles cannot be null");
        }
        List<Allele> alleles = builder.getAlleles();
        if (alleles.size() < 2) {
            throw new IllegalArgumentException("the variant context builder must contain at least 2 alleles");
        }
        boolean bl = keepOriginal = vc.getAlleles().size() == builder.getAlleles().size();
        if (keepOriginalChrCounts) {
            if (vc.hasAttribute("AC")) {
                builder.attribute("AC_Orig", keepOriginal ? vc.getAttribute("AC") : GATKVariantContextUtils.getAltAlleleInfoFieldValue("AC", vc, altAllele));
            }
            if (vc.hasAttribute("AF")) {
                builder.attribute("AF_Orig", keepOriginal ? vc.getAttribute("AF") : GATKVariantContextUtils.getAltAlleleInfoFieldValue("AF", vc, altAllele));
            }
            if (vc.hasAttribute("AN")) {
                builder.attribute("AN_Orig", vc.getAttribute("AN"));
            }
        }
        VariantContextUtils.calculateChromosomeCounts(builder, true);
    }

    private static Object getAltAlleleInfoFieldValue(String infoFieldName, VariantContext vc, Allele altAllele) {
        Object[] splitOriginalField = GATKVariantContextUtils.getVAttributeValues(vc.getAttribute(infoFieldName));
        BitSet alleleIndexesToUse = GATKVariantContextUtils.getAlleleIndexBitset(vc, Arrays.asList(altAllele));
        for (int i = 1; i < alleleIndexesToUse.size(); ++i) {
            if (!alleleIndexesToUse.get(i)) continue;
            return splitOriginalField[i - 1];
        }
        throw new ReviewedGATKException("Alternate allele " + altAllele.toString() + " not in Variant Context " + vc.toString());
    }

    private static Object[] getVAttributeValues(Object attribute) {
        if (attribute == null) {
            throw new IllegalArgumentException("the attribute cannot be null");
        }
        Object[] tokens = attribute.getClass().isArray() ? (Object[])attribute : (List.class.isAssignableFrom(attribute.getClass()) ? ((List)attribute).toArray() : attribute.toString().split(","));
        return tokens;
    }

    public static int incrementChromosomeCountsInfo(Map<Allele, Integer> calledAltAlleles, int calledAlleles, Genotype genotype) {
        if (calledAltAlleles == null) {
            throw new IllegalArgumentException("Called alternate alleles can not be null");
        }
        if (genotype == null) {
            throw new IllegalArgumentException("Genotype can not be null");
        }
        int incrementedCalledAlleles = calledAlleles;
        if (genotype.isCalled()) {
            for (Allele allele : genotype.getAlleles()) {
                ++incrementedCalledAlleles;
                if (!allele.isNonReference()) continue;
                calledAltAlleles.put(allele, calledAltAlleles.get(allele) + 1);
            }
        }
        return incrementedCalledAlleles;
    }

    public static void updateChromosomeCountsInfo(Map<Allele, Integer> calledAltAlleles, int calledAlleles, VariantContextBuilder builder) {
        if (calledAltAlleles == null) {
            throw new IllegalArgumentException("Called alternate alleles can not be null");
        }
        if (builder == null) {
            throw new IllegalArgumentException("Variant context builder can not be null");
        }
        builder.attribute("AC", calledAltAlleles.values().toArray()).attribute("AN", calledAlleles);
        if (calledAlleles != 0) {
            LinkedHashSet<Double> alleleFrequency = new LinkedHashSet<Double>(calledAltAlleles.size());
            for (Integer value : calledAltAlleles.values()) {
                alleleFrequency.add(value.doubleValue() / (double)calledAlleles);
            }
            builder.attribute("AF", alleleFrequency.toArray());
        }
    }

    public static int calculateGQFromPLs(int[] plValues) {
        if (plValues == null) {
            throw new IllegalArgumentException("Array of PL values cannot be null.");
        }
        if (plValues.length < 2) {
            throw new IllegalArgumentException("Array of PL values must contain at least two elements.");
        }
        int first = plValues[0];
        int second = plValues[1];
        if (first > second) {
            second = first;
            first = plValues[1];
        }
        for (int i = 2; i < plValues.length; ++i) {
            int candidate = plValues[i];
            if (candidate >= second) continue;
            if (candidate <= first) {
                second = first;
                first = candidate;
                continue;
            }
            second = candidate;
        }
        return second - first;
    }

    private static class CompareByPriority
    implements Comparator<VariantContext>,
    Serializable {
        List<String> priorityListOfVCs;

        public CompareByPriority(List<String> priorityListOfVCs) {
            this.priorityListOfVCs = priorityListOfVCs;
        }

        private int getIndex(VariantContext vc) {
            int i = this.priorityListOfVCs.indexOf(vc.getSource());
            if (i == -1) {
                throw new IllegalArgumentException("Priority list " + this.priorityListOfVCs + " doesn't contain variant context " + vc.getSource());
            }
            return i;
        }

        @Override
        public int compare(VariantContext vc1, VariantContext vc2) {
            return Integer.valueOf(this.getIndex(vc1)).compareTo(this.getIndex(vc2));
        }
    }

    protected static class AlleleMapper {
        private VariantContext vc = null;
        private Map<Allele, Allele> map = null;

        public AlleleMapper(VariantContext vc) {
            this.vc = vc;
        }

        public AlleleMapper(Map<Allele, Allele> map) {
            this.map = map;
        }

        public boolean needsRemapping() {
            return this.map != null;
        }

        public Collection<Allele> values() {
            return this.map != null ? this.map.values() : this.vc.getAlleles();
        }

        public Allele remap(Allele a) {
            return this.map != null && this.map.containsKey(a) ? this.map.get(a) : a;
        }

        public List<Allele> remap(List<Allele> as) {
            ArrayList<Allele> newAs = new ArrayList<Allele>();
            for (Allele a : as) {
                newAs.add(this.remap(a));
            }
            return newAs;
        }

        public List<Allele> getUniqueMappedAlleles() {
            if (this.map == null) {
                return Collections.emptyList();
            }
            return new ArrayList<Allele>(new HashSet<Allele>(this.map.values()));
        }
    }

    public static enum GenotypeAssignmentMethod {
        SET_TO_NO_CALL,
        SET_TO_NO_CALL_NO_ANNOTATIONS,
        USE_PLS_TO_ASSIGN,
        BEST_MATCH_TO_ORIGINAL,
        DO_NOT_ASSIGN_GENOTYPES;

    }

    public static enum MultipleAllelesMergeType {
        BY_TYPE,
        MIX_TYPES;

    }

    public static enum FilteredRecordMergeType {
        KEEP_IF_ANY_UNFILTERED,
        KEEP_IF_ALL_UNFILTERED,
        KEEP_UNCONDITIONAL;

    }

    public static enum GenotypeMergeType {
        UNIQUIFY,
        PRIORITIZE,
        UNSORTED,
        REQUIRE_UNIQUE;

    }
}

