/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.genotyper;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.variant.variantcontext.Allele;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.math3.stat.descriptive.rank.Median;
import org.broadinstitute.gatk.utils.GenomeLoc;
import org.broadinstitute.gatk.utils.Utils;
import org.broadinstitute.gatk.utils.downsampling.AlleleBiasedDownsamplingUtils;
import org.broadinstitute.gatk.utils.genotyper.AlleleList;
import org.broadinstitute.gatk.utils.genotyper.AlleleListUtils;
import org.broadinstitute.gatk.utils.genotyper.IndexedAlleleList;
import org.broadinstitute.gatk.utils.genotyper.IndexedSampleList;
import org.broadinstitute.gatk.utils.genotyper.PerReadAlleleLikelihoodMap;
import org.broadinstitute.gatk.utils.genotyper.SampleList;
import org.broadinstitute.gatk.utils.genotyper.SampleListUtils;
import org.broadinstitute.gatk.utils.sam.GATKSAMRecord;
import org.broadinstitute.gatk.utils.variant.GATKVCFConstants;

public class ReadLikelihoods<A extends Allele>
implements SampleList,
AlleleList<A>,
Cloneable {
    private GATKSAMRecord[][] readsBySampleIndex;
    private double[][][] valuesBySampleIndex;
    private final SampleList samples;
    private AlleleList<A> alleles;
    private List<A> alleleList;
    private List<String> sampleList;
    private final Object2IntMap<GATKSAMRecord>[] readIndexBySampleIndex;
    private int referenceAlleleIndex = -1;
    private int nonRefAlleleIndex = -1;
    private final List<GATKSAMRecord>[] readListBySampleIndex;
    private final Matrix<A>[] sampleMatrices;

    public ReadLikelihoods(SampleList samples, AlleleList<A> alleles, Map<String, List<GATKSAMRecord>> reads) {
        if (alleles == null) {
            throw new IllegalArgumentException("allele list cannot be null");
        }
        if (samples == null) {
            throw new IllegalArgumentException("sample list cannot be null");
        }
        if (reads == null) {
            throw new IllegalArgumentException("read map cannot be null");
        }
        this.samples = samples;
        this.alleles = alleles;
        int sampleCount = samples.sampleCount();
        int alleleCount = alleles.alleleCount();
        this.readsBySampleIndex = new GATKSAMRecord[sampleCount][];
        this.readListBySampleIndex = new List[sampleCount];
        this.valuesBySampleIndex = new double[sampleCount][][];
        this.referenceAlleleIndex = this.findReferenceAllele(alleles);
        this.nonRefAlleleIndex = this.findNonRefAllele(alleles);
        this.readIndexBySampleIndex = new Object2IntMap[sampleCount];
        this.setupIndexes(reads, sampleCount, alleleCount);
        this.sampleMatrices = new Matrix[sampleCount];
    }

    private void setupIndexes(Map<String, List<GATKSAMRecord>> reads, int sampleCount, int alleleCount) {
        for (int i = 0; i < sampleCount; ++i) {
            this.setupSampleData(i, reads, alleleCount);
        }
    }

    private void setupSampleData(int sampleIndex, Map<String, List<GATKSAMRecord>> readsBySample, int alleleCount) {
        String sample = this.samples.sampleAt(sampleIndex);
        List<GATKSAMRecord> reads = readsBySample.get(sample);
        this.readsBySampleIndex[sampleIndex] = reads == null ? new GATKSAMRecord[]{} : reads.toArray(new GATKSAMRecord[reads.size()]);
        int sampleReadCount = this.readsBySampleIndex[sampleIndex].length;
        double[][] sampleValues = new double[alleleCount][sampleReadCount];
        this.valuesBySampleIndex[sampleIndex] = sampleValues;
    }

    public ReadLikelihoods<A> clone() {
        int sampleCount = this.samples.sampleCount();
        int alleleCount = this.alleles.alleleCount();
        double[][][] newLikelihoodValues = new double[sampleCount][alleleCount][];
        Object2IntMap[] newReadIndexBySampleIndex = new Object2IntMap[sampleCount];
        GATKSAMRecord[][] newReadsBySampleIndex = new GATKSAMRecord[sampleCount][];
        for (int s = 0; s < sampleCount; ++s) {
            newReadsBySampleIndex[s] = (GATKSAMRecord[])this.readsBySampleIndex[s].clone();
            for (int a = 0; a < alleleCount; ++a) {
                newLikelihoodValues[s][a] = (double[])this.valuesBySampleIndex[s][a].clone();
            }
        }
        return new ReadLikelihoods<A>(this.alleles, this.samples, newReadsBySampleIndex, newReadIndexBySampleIndex, newLikelihoodValues);
    }

    private ReadLikelihoods(AlleleList alleles, SampleList samples, GATKSAMRecord[][] readsBySampleIndex, Object2IntMap<GATKSAMRecord>[] readIndex, double[][][] values) {
        this.samples = samples;
        this.alleles = alleles;
        this.readsBySampleIndex = readsBySampleIndex;
        this.valuesBySampleIndex = values;
        this.readIndexBySampleIndex = readIndex;
        int sampleCount = samples.sampleCount();
        this.readListBySampleIndex = new List[sampleCount];
        this.referenceAlleleIndex = this.findReferenceAllele(alleles);
        this.sampleMatrices = new Matrix[sampleCount];
    }

    private int findReferenceAllele(AlleleList<A> alleles) {
        int alleleCount = alleles.alleleCount();
        for (int i = 0; i < alleleCount; ++i) {
            if (!((Allele)alleles.alleleAt(i)).isReference()) continue;
            return i;
        }
        return -1;
    }

    private int findNonRefAllele(AlleleList<A> alleles) {
        int alleleCount = alleles.alleleCount();
        for (int i = alleleCount - 1; i >= 0; --i) {
            if (!((Allele)alleles.alleleAt(i)).equals(GATKVCFConstants.NON_REF_SYMBOLIC_ALLELE)) continue;
            return i;
        }
        return -1;
    }

    @Override
    public int sampleIndex(String sample) {
        return this.samples.sampleIndex(sample);
    }

    @Override
    public int sampleCount() {
        return this.samples.sampleCount();
    }

    @Override
    public String sampleAt(int sampleIndex) {
        return this.samples.sampleAt(sampleIndex);
    }

    @Override
    public int alleleIndex(A allele) {
        return this.alleles.alleleIndex(allele);
    }

    @Override
    public int alleleCount() {
        return this.alleles.alleleCount();
    }

    @Override
    public A alleleAt(int alleleIndex) {
        return this.alleles.alleleAt(alleleIndex);
    }

    public List<GATKSAMRecord> sampleReads(int sampleIndex) {
        this.checkSampleIndex(sampleIndex);
        List<GATKSAMRecord> extantList = this.readListBySampleIndex[sampleIndex];
        if (extantList == null) {
            this.readListBySampleIndex[sampleIndex] = Collections.unmodifiableList(Arrays.asList(this.readsBySampleIndex[sampleIndex]));
            return this.readListBySampleIndex[sampleIndex];
        }
        return extantList;
    }

    public Matrix<A> sampleMatrix(int sampleIndex) {
        this.checkSampleIndex(sampleIndex);
        Matrix<A> extantResult = this.sampleMatrices[sampleIndex];
        if (extantResult != null) {
            return extantResult;
        }
        this.sampleMatrices[sampleIndex] = new SampleMatrix(sampleIndex);
        return this.sampleMatrices[sampleIndex];
    }

    public void normalizeLikelihoods(boolean bestToZero, double maximumLikelihoodDifferenceCap) {
        if (maximumLikelihoodDifferenceCap >= 0.0 || Double.isNaN(maximumLikelihoodDifferenceCap)) {
            throw new IllegalArgumentException("the minimum reference likelihood fall cannot be positive");
        }
        if (maximumLikelihoodDifferenceCap == Double.NEGATIVE_INFINITY && !bestToZero) {
            return;
        }
        int alleleCount = this.alleles.alleleCount();
        if (alleleCount == 0) {
            return;
        }
        if (alleleCount == 1 && !bestToZero) {
            return;
        }
        for (int s = 0; s < this.valuesBySampleIndex.length; ++s) {
            double[][] sampleValues = this.valuesBySampleIndex[s];
            int readCount = this.readsBySampleIndex[s].length;
            for (int r = 0; r < readCount; ++r) {
                this.normalizeLikelihoodsPerRead(bestToZero, maximumLikelihoodDifferenceCap, sampleValues, s, r);
            }
        }
    }

    private void normalizeLikelihoodsPerRead(boolean bestToZero, double maximumBestAltLikelihoodDifference, double[][] sampleValues, int sampleIndex, int readIndex) {
        BestAllele bestAlternativeAllele = this.searchBestAllele(sampleIndex, readIndex, false);
        double worstLikelihoodCap = bestAlternativeAllele.likelihood + maximumBestAltLikelihoodDifference;
        double referenceLikelihood = this.referenceAlleleIndex == -1 ? Double.NEGATIVE_INFINITY : sampleValues[this.referenceAlleleIndex][readIndex];
        double bestAbsoluteLikelihood = Math.max(bestAlternativeAllele.likelihood, referenceLikelihood);
        int alleleCount = this.alleles.alleleCount();
        if (bestToZero) {
            if (bestAbsoluteLikelihood == Double.NEGATIVE_INFINITY) {
                for (int a = 0; a < alleleCount; ++a) {
                    sampleValues[a][readIndex] = 0.0;
                }
            } else if (worstLikelihoodCap != Double.NEGATIVE_INFINITY) {
                for (int a = 0; a < alleleCount; ++a) {
                    sampleValues[a][readIndex] = (sampleValues[a][readIndex] < worstLikelihoodCap ? worstLikelihoodCap : sampleValues[a][readIndex]) - bestAbsoluteLikelihood;
                }
            } else {
                for (int a = 0; a < alleleCount; ++a) {
                    double[] dArray = sampleValues[a];
                    int n = readIndex;
                    dArray[n] = dArray[n] - bestAbsoluteLikelihood;
                }
            }
        } else {
            for (int a = 0; a < alleleCount; ++a) {
                if (!(sampleValues[a][readIndex] < worstLikelihoodCap)) continue;
                sampleValues[a][readIndex] = worstLikelihoodCap;
            }
        }
    }

    public List<String> samples() {
        return this.sampleList == null ? (this.sampleList = SampleListUtils.asList(this.samples)) : this.sampleList;
    }

    public List<A> alleles() {
        return this.alleleList == null ? (this.alleleList = AlleleListUtils.asList(this.alleles)) : this.alleleList;
    }

    private BestAllele searchBestAllele(int sampleIndex, int readIndex, boolean canBeReference) {
        return this.searchBestAllele(sampleIndex, readIndex, canBeReference, this.alleles);
    }

    private BestAllele searchBestAllele(int sampleIndex, int readIndex, boolean canBeReference, AlleleList<A> allelesToConsider) {
        int alleleCount = this.alleles.alleleCount();
        if (alleleCount == 0 || alleleCount == 1 && this.referenceAlleleIndex == 0 && !canBeReference) {
            return new BestAllele(sampleIndex, readIndex, -1, Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY);
        }
        double[][] sampleValues = this.valuesBySampleIndex[sampleIndex];
        int bestAlleleIndex = canBeReference || this.referenceAlleleIndex != 0 ? 0 : 1;
        double bestLikelihood = sampleValues[bestAlleleIndex][readIndex];
        double secondBestLikelihood = Double.NEGATIVE_INFINITY;
        for (int a = bestAlleleIndex + 1; a < alleleCount; ++a) {
            if (!canBeReference && this.referenceAlleleIndex == a || this.nonRefAlleleIndex == a || allelesToConsider.alleleIndex(this.alleles.alleleAt(a)) < 0) continue;
            double candidateLikelihood = sampleValues[a][readIndex];
            if (candidateLikelihood > bestLikelihood) {
                bestAlleleIndex = a;
                secondBestLikelihood = bestLikelihood;
                bestLikelihood = candidateLikelihood;
                continue;
            }
            if (!(candidateLikelihood > secondBestLikelihood)) continue;
            secondBestLikelihood = candidateLikelihood;
        }
        return new BestAllele(sampleIndex, readIndex, bestAlleleIndex, bestLikelihood, secondBestLikelihood);
    }

    public void changeReads(Map<GATKSAMRecord, GATKSAMRecord> readRealignments) {
        int sampleCount = this.samples.sampleCount();
        for (int s = 0; s < sampleCount; ++s) {
            GATKSAMRecord[] sampleReads = this.readsBySampleIndex[s];
            Object2IntMap<GATKSAMRecord> readIndex = this.readIndexBySampleIndex[s];
            int sampleReadCount = sampleReads.length;
            for (int r = 0; r < sampleReadCount; ++r) {
                GATKSAMRecord read = sampleReads[r];
                GATKSAMRecord replacement = readRealignments.get(read);
                if (replacement == null) continue;
                sampleReads[r] = replacement;
                if (readIndex == null) continue;
                readIndex.remove((Object)read);
                readIndex.put((Object)replacement, r);
            }
        }
    }

    public boolean addMissingAlleles(Collection<A> candidateAlleles, double defaultLikelihood) {
        if (candidateAlleles == null) {
            throw new IllegalArgumentException("the candidateAlleles list cannot be null");
        }
        if (candidateAlleles.isEmpty()) {
            return false;
        }
        ArrayList<Allele> allelesToAdd = new ArrayList<Allele>(candidateAlleles.size());
        for (Allele allele : candidateAlleles) {
            if (this.alleles.alleleIndex(allele) != -1) continue;
            allelesToAdd.add(allele);
        }
        if (allelesToAdd.isEmpty()) {
            return false;
        }
        int oldAlleleCount = this.alleles.alleleCount();
        int newAlleleCount = this.alleles.alleleCount() + allelesToAdd.size();
        this.alleleList = null;
        int referenceIndex = this.referenceAlleleIndex;
        int nonRefIndex = this.nonRefAlleleIndex;
        Allele[] newAlleles = new Allele[newAlleleCount];
        for (int a = 0; a < oldAlleleCount; ++a) {
            newAlleles[a] = this.alleleAt(a);
        }
        int newIndex = oldAlleleCount;
        for (Allele allele : allelesToAdd) {
            if (allele.isReference()) {
                if (referenceIndex != -1) {
                    throw new IllegalArgumentException("there cannot be more than one reference allele");
                }
                referenceIndex = newIndex;
            } else if (allele.equals(GATKVCFConstants.NON_REF_SYMBOLIC_ALLELE)) {
                if (this.nonRefAlleleIndex != -1) {
                    throw new IllegalArgumentException(String.format("there cannot be more than one %s allele", "NON_REF"));
                }
                nonRefIndex = newIndex;
            }
            newAlleles[newIndex++] = allele;
        }
        this.alleles = new IndexedAlleleList(newAlleles);
        int sampleCount = this.samples.sampleCount();
        for (int s = 0; s < sampleCount; ++s) {
            int sampleReadCount = this.readsBySampleIndex[s].length;
            double[][] newValuesBySampleIndex = (double[][])Arrays.copyOf(this.valuesBySampleIndex[s], newAlleleCount);
            for (int a = oldAlleleCount; a < newAlleleCount; ++a) {
                newValuesBySampleIndex[a] = new double[sampleReadCount];
                if (defaultLikelihood == 0.0) continue;
                Arrays.fill(newValuesBySampleIndex[a], defaultLikelihood);
            }
            this.valuesBySampleIndex[s] = newValuesBySampleIndex;
        }
        if (referenceIndex != -1) {
            this.referenceAlleleIndex = referenceIndex;
        }
        if (nonRefIndex != -1) {
            this.nonRefAlleleIndex = nonRefIndex;
            this.updateNonRefAlleleLikelihoods();
        }
        return true;
    }

    public void dropAlleles(Set<A> allelesToDrop) {
        int i;
        if (allelesToDrop == null) {
            throw new IllegalArgumentException("the input allele to drop set cannot be null");
        }
        if (allelesToDrop.isEmpty()) {
            return;
        }
        boolean[] indicesToDrop = new boolean[this.alleles.alleleCount()];
        for (Allele allele : allelesToDrop) {
            int index = this.alleles.alleleIndex(allele);
            if (index < 0) {
                throw new IllegalArgumentException("unknown allele: " + allele);
            }
            indicesToDrop[index] = true;
        }
        Allele[] newAlleles = new Allele[this.alleles.alleleCount() - allelesToDrop.size()];
        int[] newAlleleIndices = new int[newAlleles.length];
        int nextIndex = 0;
        for (i = 0; i < this.alleles.alleleCount(); ++i) {
            if (indicesToDrop[i]) continue;
            newAlleleIndices[nextIndex] = i;
            newAlleles[nextIndex++] = this.alleles.alleleAt(i);
        }
        for (i = 0; i < this.samples.sampleCount(); ++i) {
            double[][] oldSampleValues = this.valuesBySampleIndex[i];
            double[][] newSampleValues = new double[newAlleles.length][];
            for (int j = 0; j < newAlleles.length; ++j) {
                newSampleValues[j] = oldSampleValues[newAlleleIndices[j]];
            }
            this.valuesBySampleIndex[i] = newSampleValues;
        }
        this.alleleList = Collections.unmodifiableList(Arrays.asList(newAlleles));
        this.alleles = new IndexedAlleleList<A>(this.alleleList);
        if (this.nonRefAlleleIndex >= 0) {
            this.nonRefAlleleIndex = this.findNonRefAllele(this.alleles);
            this.updateNonRefAlleleLikelihoods();
        }
        if (this.referenceAlleleIndex >= 0) {
            this.referenceAlleleIndex = this.findReferenceAllele(this.alleles);
        }
    }

    public void retainAlleles(Set<A> subset) {
        if (this.alleles == null) {
            throw new IllegalArgumentException("the retain subset must not be null");
        }
        if (!this.alleles().containsAll(subset) || subset.size() > this.alleleCount()) {
            throw new IllegalArgumentException("some of the alleles to retain are not present in the read-likelihoods collection");
        }
        if (subset.isEmpty() || subset.size() == 1 && subset.contains(GATKVCFConstants.NON_REF_SYMBOLIC_ALLELE)) {
            throw new IllegalArgumentException("there must be at least one allele to retain");
        }
        HashSet<A> allelesToDrop = new HashSet<A>(this.alleles());
        allelesToDrop.removeAll(subset);
        this.dropAlleles(allelesToDrop);
    }

    public <B extends Allele> ReadLikelihoods<B> marginalize(Map<B, List<A>> newToOldAlleleMap) {
        if (newToOldAlleleMap == null) {
            throw new IllegalArgumentException("the input allele mapping cannot be null");
        }
        Allele[] newAlleles = newToOldAlleleMap.keySet().toArray(new Allele[newToOldAlleleMap.size()]);
        int oldAlleleCount = this.alleles.alleleCount();
        int newAlleleCount = newAlleles.length;
        int[] oldToNewAlleleIndexMap = this.oldToNewAlleleIndexMap(newToOldAlleleMap, newAlleles, oldAlleleCount, newAlleleCount);
        double[][][] newLikelihoodValues = this.marginalLikelihoods(oldAlleleCount, newAlleleCount, oldToNewAlleleIndexMap, null);
        int sampleCount = this.samples.sampleCount();
        Object2IntMap[] newReadIndexBySampleIndex = new Object2IntMap[sampleCount];
        GATKSAMRecord[][] newReadsBySampleIndex = new GATKSAMRecord[sampleCount][];
        for (int s = 0; s < sampleCount; ++s) {
            newReadsBySampleIndex[s] = (GATKSAMRecord[])this.readsBySampleIndex[s].clone();
        }
        return new ReadLikelihoods<A>(new IndexedAlleleList(newAlleles), this.samples, newReadsBySampleIndex, newReadIndexBySampleIndex, newLikelihoodValues);
    }

    public <B extends Allele> ReadLikelihoods<B> marginalize(Map<B, List<A>> newToOldAlleleMap, GenomeLoc overlap) {
        if (overlap == null) {
            return this.marginalize(newToOldAlleleMap);
        }
        if (newToOldAlleleMap == null) {
            throw new IllegalArgumentException("the input allele mapping cannot be null");
        }
        Allele[] newAlleles = newToOldAlleleMap.keySet().toArray(new Allele[newToOldAlleleMap.size()]);
        int oldAlleleCount = this.alleles.alleleCount();
        int newAlleleCount = newAlleles.length;
        int[] oldToNewAlleleIndexMap = this.oldToNewAlleleIndexMap(newToOldAlleleMap, newAlleles, oldAlleleCount, newAlleleCount);
        int[][] readsToKeep = this.overlappingReadIndicesBySampleIndex(overlap);
        double[][][] newLikelihoodValues = this.marginalLikelihoods(oldAlleleCount, newAlleleCount, oldToNewAlleleIndexMap, readsToKeep);
        int sampleCount = this.samples.sampleCount();
        Object2IntMap[] newReadIndexBySampleIndex = new Object2IntMap[sampleCount];
        GATKSAMRecord[][] newReadsBySampleIndex = new GATKSAMRecord[sampleCount][];
        for (int s = 0; s < sampleCount; ++s) {
            int[] sampleReadsToKeep = readsToKeep[s];
            int newSampleReadCount = sampleReadsToKeep.length;
            GATKSAMRecord[] oldSampleReads = this.readsBySampleIndex[s];
            int oldSampleReadCount = oldSampleReads.length;
            if (newSampleReadCount == oldSampleReadCount) {
                newReadsBySampleIndex[s] = (GATKSAMRecord[])oldSampleReads.clone();
                continue;
            }
            newReadsBySampleIndex[s] = new GATKSAMRecord[newSampleReadCount];
            for (int i = 0; i < newSampleReadCount; ++i) {
                newReadsBySampleIndex[s][i] = oldSampleReads[sampleReadsToKeep[i]];
            }
        }
        return new ReadLikelihoods<A>(new IndexedAlleleList(newAlleles), this.samples, newReadsBySampleIndex, newReadIndexBySampleIndex, newLikelihoodValues);
    }

    private int[][] overlappingReadIndicesBySampleIndex(GenomeLoc overlap) {
        if (overlap == null) {
            return null;
        }
        int sampleCount = this.samples.sampleCount();
        int[][] result = new int[sampleCount][];
        IntArrayList buffer = new IntArrayList(200);
        int referenceIndex = overlap.getContigIndex();
        int overlapStart = overlap.getStart();
        int overlapEnd = overlap.getStop();
        for (int s = 0; s < sampleCount; ++s) {
            buffer.clear();
            GATKSAMRecord[] sampleReads = this.readsBySampleIndex[s];
            int sampleReadCount = sampleReads.length;
            buffer.ensureCapacity(sampleReadCount);
            for (int r = 0; r < sampleReadCount; ++r) {
                if (!ReadLikelihoods.unclippedReadOverlapsRegion(sampleReads[r], referenceIndex, overlapStart, overlapEnd)) continue;
                buffer.add(r);
            }
            result[s] = buffer.toIntArray();
        }
        return result;
    }

    public static boolean unclippedReadOverlapsRegion(GATKSAMRecord read, GenomeLoc region) {
        return ReadLikelihoods.unclippedReadOverlapsRegion(read, region.getContigIndex(), region.getStart(), region.getStop());
    }

    private static boolean unclippedReadOverlapsRegion(GATKSAMRecord sampleRead, int referenceIndex, int start, int end) {
        int readReference = sampleRead.getReferenceIndex();
        if (readReference != referenceIndex) {
            return false;
        }
        int readStart = sampleRead.getUnclippedStart();
        if (readStart > end) {
            return false;
        }
        int readEnd = sampleRead.getReadUnmappedFlag() ? sampleRead.getUnclippedEnd() : Math.max(sampleRead.getUnclippedEnd(), sampleRead.getUnclippedStart());
        return readEnd >= start;
    }

    private double[][][] marginalLikelihoods(int oldAlleleCount, int newAlleleCount, int[] oldToNewAlleleIndexMap, int[][] readsToKeep) {
        int sampleCount = this.samples.sampleCount();
        double[][][] result = new double[sampleCount][][];
        for (int s = 0; s < sampleCount; ++s) {
            int sampleReadCount = this.readsBySampleIndex[s].length;
            double[][] oldSampleValues = this.valuesBySampleIndex[s];
            int[] sampleReadToKeep = readsToKeep == null || readsToKeep[s].length == sampleReadCount ? null : readsToKeep[s];
            int newSampleReadCount = sampleReadToKeep == null ? sampleReadCount : sampleReadToKeep.length;
            double[][] newSampleValues = result[s] = new double[newAlleleCount][newSampleReadCount];
            for (int a = 0; a < newAlleleCount; ++a) {
                Arrays.fill(newSampleValues[a], Double.NEGATIVE_INFINITY);
            }
            for (int r = 0; r < newSampleReadCount; ++r) {
                for (int a = 0; a < oldAlleleCount; ++a) {
                    double likelihood;
                    int oldReadIndex = newSampleReadCount == sampleReadCount ? r : sampleReadToKeep[r];
                    int newAlleleIndex = oldToNewAlleleIndexMap[a];
                    if (newAlleleIndex == -1 || !((likelihood = oldSampleValues[a][oldReadIndex]) > newSampleValues[newAlleleIndex][r])) continue;
                    newSampleValues[newAlleleIndex][r] = likelihood;
                }
            }
        }
        return result;
    }

    public static ReadLikelihoods<Allele> fromPerAlleleReadLikelihoodsMap(Map<String, PerReadAlleleLikelihoodMap> map) {
        IndexedSampleList sampleList = new IndexedSampleList(map.keySet());
        LinkedHashSet<Allele> alleles = new LinkedHashSet<Allele>(10);
        HashMap<String, List<GATKSAMRecord>> sampleToReads = new HashMap<String, List<GATKSAMRecord>>(sampleList.sampleCount());
        for (Map.Entry<String, PerReadAlleleLikelihoodMap> entry : map.entrySet()) {
            String sample = entry.getKey();
            PerReadAlleleLikelihoodMap sampleLikelihoods = entry.getValue();
            alleles.addAll(sampleLikelihoods.getAllelesSet());
            sampleToReads.put(sample, new ArrayList<GATKSAMRecord>(sampleLikelihoods.getLikelihoodReadMap().keySet()));
        }
        IndexedAlleleList alleleList = new IndexedAlleleList(alleles);
        ReadLikelihoods<Allele> result = new ReadLikelihoods<Allele>(sampleList, alleleList, sampleToReads);
        for (Map.Entry<String, PerReadAlleleLikelihoodMap> sampleEntry : map.entrySet()) {
            Matrix<Allele> sampleMatrix = result.sampleMatrix(result.sampleIndex(sampleEntry.getKey()));
            for (Map.Entry<GATKSAMRecord, Map<Allele, Double>> readEntry : sampleEntry.getValue().getLikelihoodReadMap().entrySet()) {
                GATKSAMRecord read = readEntry.getKey();
                int readIndex = sampleMatrix.readIndex(read);
                for (Map.Entry<Allele, Double> alleleEntry : readEntry.getValue().entrySet()) {
                    int alleleIndex = result.alleleIndex(alleleEntry.getKey());
                    sampleMatrix.set(alleleIndex, readIndex, alleleEntry.getValue());
                }
            }
        }
        return result;
    }

    private <B extends Allele> int[] oldToNewAlleleIndexMap(Map<B, List<A>> newToOldAlleleMap, B[] newAlleles, int oldAlleleCount, int newAlleleCount) {
        int[] oldToNewAlleleIndexMap = new int[oldAlleleCount];
        Arrays.fill(oldToNewAlleleIndexMap, -1);
        for (int i = 0; i < newAlleleCount; ++i) {
            B newAllele = newAlleles[i];
            if (newAllele == null) {
                throw new IllegalArgumentException("input alleles cannot be null");
            }
            List<A> oldAlleles = newToOldAlleleMap.get(newAllele);
            if (oldAlleles == null) {
                throw new IllegalArgumentException("no new allele list can be null");
            }
            for (Allele oldAllele : oldAlleles) {
                if (oldAllele == null) {
                    throw new IllegalArgumentException("old alleles cannot be null");
                }
                int oldAlleleIndex = this.alleleIndex(oldAllele);
                if (oldAlleleIndex == -1) {
                    throw new IllegalArgumentException("missing old allele " + oldAllele + " in likelihood collection ");
                }
                if (oldToNewAlleleIndexMap[oldAlleleIndex] != -1) {
                    throw new IllegalArgumentException("collision: two new alleles make reference to the same old allele");
                }
                oldToNewAlleleIndexMap[oldAlleleIndex] = i;
            }
        }
        return oldToNewAlleleIndexMap;
    }

    public void filterToOnlyOverlappingUnclippedReads(GenomeLoc location) {
        if (location == null) {
            throw new IllegalArgumentException("the location cannot be null");
        }
        if (location.isUnmapped()) {
            throw new IllegalArgumentException("the location cannot be unmapped");
        }
        int sampleCount = this.samples.sampleCount();
        int locContig = location.getContigIndex();
        int locStart = location.getStart();
        int locEnd = location.getStop();
        int alleleCount = this.alleles.alleleCount();
        IntArrayList removeIndices = new IntArrayList(10);
        for (int s = 0; s < sampleCount; ++s) {
            GATKSAMRecord[] sampleReads = this.readsBySampleIndex[s];
            int sampleReadCount = sampleReads.length;
            for (int r = 0; r < sampleReadCount; ++r) {
                if (ReadLikelihoods.unclippedReadOverlapsRegion(sampleReads[r], locContig, locStart, locEnd)) continue;
                removeIndices.add(r);
            }
            this.removeSampleReads(s, removeIndices, alleleCount);
            removeIndices.clear();
        }
    }

    public void filterPoorlyModeledReads(double maximumErrorPerBase) {
        if (this.alleles.alleleCount() == 0) {
            throw new IllegalStateException("unsupported for read-likelihood collections with no alleles");
        }
        if (Double.isNaN(maximumErrorPerBase) || maximumErrorPerBase <= 0.0) {
            throw new IllegalArgumentException("the maximum error per base must be a positive number");
        }
        int sampleCount = this.samples.sampleCount();
        int alleleCount = this.alleles.alleleCount();
        IntArrayList removeIndices = new IntArrayList(10);
        for (int s = 0; s < sampleCount; ++s) {
            GATKSAMRecord[] sampleReads = this.readsBySampleIndex[s];
            int sampleReadCount = sampleReads.length;
            for (int r = 0; r < sampleReadCount; ++r) {
                GATKSAMRecord read = sampleReads[r];
                if (!this.readIsPoorlyModelled(s, r, read, maximumErrorPerBase)) continue;
                removeIndices.add(r);
            }
            this.removeSampleReads(s, removeIndices, alleleCount);
            removeIndices.clear();
        }
    }

    private boolean readIsPoorlyModelled(int sampleIndex, int readIndex, GATKSAMRecord read, double maxErrorRatePerBase) {
        double maxErrorsForRead = Math.min(2.0, Math.ceil((double)read.getReadLength() * maxErrorRatePerBase));
        double log10QualPerBase = -4.0;
        double log10MaxLikelihoodForTrueAllele = maxErrorsForRead * -4.0;
        int alleleCount = this.alleles.alleleCount();
        double[][] sampleValues = this.valuesBySampleIndex[sampleIndex];
        for (int a = 0; a < alleleCount; ++a) {
            if (!(sampleValues[a][readIndex] >= log10MaxLikelihoodForTrueAllele)) continue;
            return false;
        }
        return true;
    }

    public void addReads(Map<String, List<GATKSAMRecord>> readsBySample, double initialLikelihood) {
        for (Map.Entry<String, List<GATKSAMRecord>> entry : readsBySample.entrySet()) {
            String sample = entry.getKey();
            List<GATKSAMRecord> newSampleReads = entry.getValue();
            int sampleIndex = this.samples.sampleIndex(sample);
            if (sampleIndex == -1) {
                throw new IllegalArgumentException("input sample " + sample + " is not part of the read-likelihoods collection");
            }
            if (newSampleReads == null || newSampleReads.size() == 0) continue;
            int sampleReadCount = this.readsBySampleIndex[sampleIndex].length;
            int newSampleReadCount = sampleReadCount + newSampleReads.size();
            this.appendReads(newSampleReads, sampleIndex, sampleReadCount, newSampleReadCount);
            this.extendsLikelihoodArrays(initialLikelihood, sampleIndex, sampleReadCount, newSampleReadCount);
        }
    }

    private void extendsLikelihoodArrays(double initialLikelihood, int sampleIndex, int sampleReadCount, int newSampleReadCount) {
        int a;
        double[][] sampleValues = this.valuesBySampleIndex[sampleIndex];
        int alleleCount = this.alleles.alleleCount();
        for (a = 0; a < alleleCount; ++a) {
            sampleValues[a] = Arrays.copyOf(sampleValues[a], newSampleReadCount);
        }
        if (initialLikelihood != 0.0) {
            for (a = 0; a < alleleCount; ++a) {
                Arrays.fill(sampleValues[a], sampleReadCount, newSampleReadCount, initialLikelihood);
            }
        }
    }

    private void appendReads(List<GATKSAMRecord> newSampleReads, int sampleIndex, int sampleReadCount, int newSampleReadCount) {
        this.readsBySampleIndex[sampleIndex] = Arrays.copyOf(this.readsBySampleIndex[sampleIndex], newSampleReadCount);
        GATKSAMRecord[] sampleReads = this.readsBySampleIndex[sampleIndex];
        int nextReadIndex = sampleReadCount;
        Object2IntMap<GATKSAMRecord> sampleReadIndex = this.readIndexBySampleIndex[sampleIndex];
        for (GATKSAMRecord newRead : newSampleReads) {
            if (sampleReadIndex != null) {
                sampleReadIndex.put((Object)newRead, nextReadIndex);
            }
            sampleReads[nextReadIndex++] = newRead;
        }
    }

    public void addNonReferenceAllele(A nonRefAllele) {
        if (nonRefAllele == null) {
            throw new IllegalArgumentException("non-ref allele cannot be null");
        }
        if (!((Allele)nonRefAllele).equals(GATKVCFConstants.NON_REF_SYMBOLIC_ALLELE)) {
            throw new IllegalArgumentException("the non-ref allele is not valid");
        }
        if (this.addMissingAlleles(Collections.singleton(nonRefAllele), Double.NEGATIVE_INFINITY)) {
            this.updateNonRefAlleleLikelihoods();
        }
    }

    public void updateNonRefAlleleLikelihoods() {
        this.updateNonRefAlleleLikelihoods(this.alleles);
    }

    public void updateNonRefAlleleLikelihoods(AlleleList<A> allelesToConsider) {
        if (this.nonRefAlleleIndex < 0) {
            return;
        }
        int alleleCount = this.alleles.alleleCount();
        int nonRefAlleleIndex = this.alleleIndex(GATKVCFConstants.NON_REF_SYMBOLIC_ALLELE);
        int concreteAlleleCount = nonRefAlleleIndex < 0 ? alleleCount : alleleCount - 1;
        double[] qualifiedAlleleLikelihoods = new double[concreteAlleleCount];
        Median medianCalculator = new Median();
        for (int s = 0; s < this.samples.sampleCount(); ++s) {
            double[][] sampleValues = this.valuesBySampleIndex[s];
            int readCount = sampleValues[0].length;
            for (int r = 0; r < readCount; ++r) {
                BestAllele bestAllele = this.searchBestAllele(s, r, true);
                int numberOfQualifiedAlleleLikelihoods = 0;
                for (int i = 0; i < alleleCount; ++i) {
                    double alleleLikelihood = sampleValues[i][r];
                    if (i == nonRefAlleleIndex || !(alleleLikelihood < bestAllele.likelihood) || Double.isNaN(alleleLikelihood) || allelesToConsider.alleleIndex(this.alleles.alleleAt(i)) == -1) continue;
                    qualifiedAlleleLikelihoods[numberOfQualifiedAlleleLikelihoods++] = alleleLikelihood;
                }
                double nonRefLikelihood = medianCalculator.evaluate(qualifiedAlleleLikelihoods, 0, numberOfQualifiedAlleleLikelihoods);
                sampleValues[nonRefAlleleIndex][r] = !Double.isNaN(nonRefLikelihood) ? nonRefLikelihood : (concreteAlleleCount <= 1 ? Double.NaN : bestAllele.likelihood);
            }
        }
    }

    public void contaminationDownsampling(Map<String, Double> perSampleDownsamplingFraction) {
        int sampleCount = this.samples.sampleCount();
        IntArrayList readsToRemove = new IntArrayList(10);
        int alleleCount = this.alleles.alleleCount();
        for (int s = 0; s < sampleCount; ++s) {
            double fraction;
            String sample = this.samples.sampleAt(s);
            Double fractionDouble = perSampleDownsamplingFraction.get(sample);
            if (fractionDouble == null || Double.isNaN(fraction = fractionDouble.doubleValue()) || fraction <= 0.0) continue;
            if (fraction >= 1.0) {
                int sampleReadCount = this.readsBySampleIndex[s].length;
                readsToRemove.ensureCapacity(sampleReadCount);
                for (int r = 0; r < sampleReadCount; ++r) {
                    readsToRemove.add(r);
                }
                this.removeSampleReads(s, readsToRemove, alleleCount);
                readsToRemove.clear();
                continue;
            }
            Map<A, List<GATKSAMRecord>> readsByBestAllelesMap = this.readsByBestAlleleMap(s);
            this.removeSampleReads(s, AlleleBiasedDownsamplingUtils.selectAlleleBiasedReads(readsByBestAllelesMap, fraction), alleleCount);
        }
    }

    @VisibleForTesting
    static ReadLikelihoods<Allele> fromPerAlleleReadLikelihoodsMap(AlleleList<Allele> alleleList, Map<String, PerReadAlleleLikelihoodMap> map) {
        IndexedSampleList sampleList = new IndexedSampleList(map.keySet());
        int alleleCount = alleleList.alleleCount();
        HashMap<String, List<GATKSAMRecord>> sampleToReads = new HashMap<String, List<GATKSAMRecord>>(sampleList.sampleCount());
        for (Map.Entry<String, PerReadAlleleLikelihoodMap> entry : map.entrySet()) {
            String sample = entry.getKey();
            PerReadAlleleLikelihoodMap sampleLikelihoods = entry.getValue();
            sampleToReads.put(sample, new ArrayList<GATKSAMRecord>(sampleLikelihoods.getLikelihoodReadMap().keySet()));
        }
        ReadLikelihoods<Allele> result = new ReadLikelihoods<Allele>(sampleList, alleleList, sampleToReads);
        for (Map.Entry<String, PerReadAlleleLikelihoodMap> sampleEntry : map.entrySet()) {
            Matrix<Allele> sampleMatrix = result.sampleMatrix(result.sampleIndex(sampleEntry.getKey()));
            for (Map.Entry<GATKSAMRecord, Map<Allele, Double>> readEntry : sampleEntry.getValue().getLikelihoodReadMap().entrySet()) {
                GATKSAMRecord read = readEntry.getKey();
                int readIndex = sampleMatrix.readIndex(read);
                Map<Allele, Double> alleleToLikelihoodMap = readEntry.getValue();
                for (int a = 0; a < alleleCount; ++a) {
                    Allele allele = alleleList.alleleAt(a);
                    Double likelihood = alleleToLikelihoodMap.get(allele);
                    if (likelihood == null) {
                        throw new IllegalArgumentException("there is no likelihood for allele " + allele + " and read " + read);
                    }
                    sampleMatrix.set(a, readIndex, likelihood);
                }
            }
        }
        return result;
    }

    public Collection<BestAllele> bestAlleles() {
        ArrayList<BestAllele> result = new ArrayList<BestAllele>(100);
        int sampleCount = this.samples.sampleCount();
        for (int s = 0; s < sampleCount; ++s) {
            GATKSAMRecord[] sampleReads = this.readsBySampleIndex[s];
            int readCount = sampleReads.length;
            for (int r = 0; r < readCount; ++r) {
                result.add(this.searchBestAllele(s, r, true));
            }
        }
        return result;
    }

    @VisibleForTesting
    Map<A, List<GATKSAMRecord>> readsByBestAlleleMap(int sampleIndex) {
        this.checkSampleIndex(sampleIndex);
        int alleleCount = this.alleles.alleleCount();
        int sampleReadCount = this.readsBySampleIndex[sampleIndex].length;
        LinkedHashMap result = new LinkedHashMap(alleleCount);
        for (int a = 0; a < alleleCount; ++a) {
            result.put(this.alleles.alleleAt(a), new ArrayList(sampleReadCount));
        }
        this.readsByBestAlleleMap(sampleIndex, result);
        return result;
    }

    public Map<A, List<GATKSAMRecord>> readsByBestAlleleMap() {
        int alleleCount = this.alleles.alleleCount();
        HashMap result = new HashMap(alleleCount);
        int totalReadCount = this.readCount();
        for (int a = 0; a < alleleCount; ++a) {
            result.put(this.alleles.alleleAt(a), new ArrayList(totalReadCount));
        }
        int sampleCount = this.samples.sampleCount();
        for (int s = 0; s < sampleCount; ++s) {
            this.readsByBestAlleleMap(s, result);
        }
        return result;
    }

    private void readsByBestAlleleMap(int sampleIndex, Map<A, List<GATKSAMRecord>> result) {
        GATKSAMRecord[] reads = this.readsBySampleIndex[sampleIndex];
        int readCount = reads.length;
        for (int r = 0; r < readCount; ++r) {
            BestAllele bestAllele = this.searchBestAllele(sampleIndex, r, true);
            if (!bestAllele.isInformative()) continue;
            result.get(bestAllele.allele).add(bestAllele.read);
        }
    }

    public int readIndex(int sampleIndex, GATKSAMRecord read) {
        Object2IntMap<GATKSAMRecord> readIndex = this.readIndexBySampleIndex(sampleIndex);
        if (readIndex.containsKey((Object)read)) {
            return this.readIndexBySampleIndex(sampleIndex).getInt((Object)read);
        }
        return -1;
    }

    public int readCount() {
        int sum = 0;
        int sampleCount = this.samples.sampleCount();
        for (int i = 0; i < sampleCount; ++i) {
            sum += this.readsBySampleIndex[i].length;
        }
        return sum;
    }

    public int sampleReadCount(int sampleIndex) {
        this.checkSampleIndex(sampleIndex);
        return this.readsBySampleIndex[sampleIndex].length;
    }

    private void removeSampleReads(int sampleIndex, IntArrayList indexToRemove, int alleleCount) {
        int removeCount = indexToRemove.size();
        if (removeCount == 0) {
            return;
        }
        GATKSAMRecord[] sampleReads = this.readsBySampleIndex[sampleIndex];
        int sampleReadCount = sampleReads.length;
        Object2IntMap<GATKSAMRecord> indexByRead = this.readIndexBySampleIndex[sampleIndex];
        if (indexByRead != null) {
            for (int i = 0; i < removeCount; ++i) {
                indexByRead.remove((Object)sampleReads[indexToRemove.getInt(i)]);
            }
        }
        boolean[] removeIndex = new boolean[sampleReadCount];
        int firstDeleted = indexToRemove.get(0);
        for (int i = 0; i < removeCount; ++i) {
            removeIndex[indexToRemove.get((int)i).intValue()] = true;
        }
        int newSampleReadCount = sampleReadCount - removeCount;
        GATKSAMRecord[] oldSampleReads = this.readsBySampleIndex[sampleIndex];
        GATKSAMRecord[] newSampleReads = new GATKSAMRecord[newSampleReadCount];
        System.arraycopy(oldSampleReads, 0, newSampleReads, 0, firstDeleted);
        Utils.skimArray(oldSampleReads, firstDeleted, newSampleReads, firstDeleted, removeIndex, firstDeleted);
        double[][] oldSampleValues = this.valuesBySampleIndex[sampleIndex];
        double[][] newSampleValues = new double[alleleCount][newSampleReadCount];
        for (int a = 0; a < alleleCount; ++a) {
            System.arraycopy(oldSampleValues[a], 0, newSampleValues[a], 0, firstDeleted);
            Utils.skimArray(oldSampleValues[a], firstDeleted, newSampleValues[a], firstDeleted, removeIndex, firstDeleted);
        }
        this.valuesBySampleIndex[sampleIndex] = newSampleValues;
        this.readsBySampleIndex[sampleIndex] = newSampleReads;
        this.readListBySampleIndex[sampleIndex] = null;
    }

    private void removeSampleReads(int sampleIndex, Collection<GATKSAMRecord> readsToRemove, int alleleCount) {
        GATKSAMRecord[] sampleReads = this.readsBySampleIndex[sampleIndex];
        int sampleReadCount = sampleReads.length;
        Object2IntMap<GATKSAMRecord> indexByRead = this.readIndexBySampleIndex(sampleIndex);
        boolean[] removeIndex = new boolean[sampleReadCount];
        int removeCount = 0;
        int firstDeleted = sampleReadCount;
        Iterator<GATKSAMRecord> readsToRemoveIterator = readsToRemove.iterator();
        while (readsToRemoveIterator.hasNext()) {
            GATKSAMRecord read = readsToRemoveIterator.next();
            if (!indexByRead.containsKey((Object)read)) continue;
            int index = indexByRead.getInt((Object)read);
            if (firstDeleted > index) {
                firstDeleted = index;
            }
            ++removeCount;
            removeIndex[index] = true;
            readsToRemoveIterator.remove();
            indexByRead.remove((Object)read);
        }
        if (removeCount == 0) {
            return;
        }
        int newSampleReadCount = sampleReadCount - removeCount;
        GATKSAMRecord[] oldSampleReads = this.readsBySampleIndex[sampleIndex];
        GATKSAMRecord[] newSampleReads = new GATKSAMRecord[newSampleReadCount];
        System.arraycopy(oldSampleReads, 0, newSampleReads, 0, firstDeleted);
        Utils.skimArray(oldSampleReads, firstDeleted, newSampleReads, firstDeleted, removeIndex, firstDeleted);
        for (int r = firstDeleted; r < newSampleReadCount; ++r) {
            indexByRead.put((Object)newSampleReads[r], r);
        }
        double[][] oldSampleValues = this.valuesBySampleIndex[sampleIndex];
        double[][] newSampleValues = new double[alleleCount][newSampleReadCount];
        for (int a = 0; a < alleleCount; ++a) {
            System.arraycopy(oldSampleValues[a], 0, newSampleValues[a], 0, firstDeleted);
            Utils.skimArray(oldSampleValues[a], firstDeleted, newSampleValues[a], firstDeleted, removeIndex, firstDeleted);
        }
        this.valuesBySampleIndex[sampleIndex] = newSampleValues;
        this.readsBySampleIndex[sampleIndex] = newSampleReads;
        this.readListBySampleIndex[sampleIndex] = null;
    }

    private Object2IntMap<GATKSAMRecord> readIndexBySampleIndex(int sampleIndex) {
        if (this.readIndexBySampleIndex[sampleIndex] == null) {
            GATKSAMRecord[] sampleReads = this.readsBySampleIndex[sampleIndex];
            int sampleReadCount = sampleReads.length;
            this.readIndexBySampleIndex[sampleIndex] = new Object2IntOpenHashMap(sampleReadCount);
            for (int r = 0; r < sampleReadCount; ++r) {
                this.readIndexBySampleIndex[sampleIndex].put((Object)sampleReads[r], r);
            }
        }
        return this.readIndexBySampleIndex[sampleIndex];
    }

    @Deprecated
    public Map<String, PerReadAlleleLikelihoodMap> toPerReadAlleleLikelihoodMap() {
        int sampleCount = this.samples.sampleCount();
        HashMap<String, PerReadAlleleLikelihoodMap> result = new HashMap<String, PerReadAlleleLikelihoodMap>(sampleCount);
        for (int s = 0; s < sampleCount; ++s) {
            result.put(this.samples.sampleAt(s), this.toPerReadAlleleLikelihoodMap(s));
        }
        return result;
    }

    @Deprecated
    public PerReadAlleleLikelihoodMap toPerReadAlleleLikelihoodMap(int sampleIndex) {
        this.checkSampleIndex(sampleIndex);
        PerReadAlleleLikelihoodMap result = new PerReadAlleleLikelihoodMap();
        int alleleCount = this.alleles.alleleCount();
        GATKSAMRecord[] sampleReads = this.readsBySampleIndex[sampleIndex];
        int sampleReadCount = sampleReads.length;
        for (int a = 0; a < alleleCount; ++a) {
            A allele = this.alleles.alleleAt(a);
            double[] readLikelihoods = this.valuesBySampleIndex[sampleIndex][a];
            for (int r = 0; r < sampleReadCount; ++r) {
                result.add(sampleReads[r], (Allele)allele, (Double)readLikelihoods[r]);
            }
        }
        return result;
    }

    private void checkSampleIndex(int sampleIndex) {
        if (sampleIndex < 0 || sampleIndex >= this.samples.sampleCount()) {
            throw new IllegalArgumentException("invalid sample index: " + sampleIndex);
        }
    }

    private class SampleMatrix
    implements Matrix<A> {
        private final int sampleIndex;

        private SampleMatrix(int sampleIndex) {
            this.sampleIndex = sampleIndex;
        }

        @Override
        public List<GATKSAMRecord> reads() {
            return ReadLikelihoods.this.sampleReads(this.sampleIndex);
        }

        @Override
        public List<A> alleles() {
            return ReadLikelihoods.this.alleles();
        }

        @Override
        public void set(int alleleIndex, int readIndex, double value) {
            ((ReadLikelihoods)ReadLikelihoods.this).valuesBySampleIndex[this.sampleIndex][alleleIndex][readIndex] = value;
        }

        @Override
        public double get(int alleleIndex, int readIndex) {
            return ReadLikelihoods.this.valuesBySampleIndex[this.sampleIndex][alleleIndex][readIndex];
        }

        @Override
        public int alleleIndex(A allele) {
            return ReadLikelihoods.this.alleleIndex(allele);
        }

        @Override
        public int readIndex(GATKSAMRecord read) {
            return ReadLikelihoods.this.readIndex(this.sampleIndex, read);
        }

        @Override
        public int alleleCount() {
            return ReadLikelihoods.this.alleles.alleleCount();
        }

        @Override
        public int readCount() {
            return ReadLikelihoods.this.readsBySampleIndex[this.sampleIndex].length;
        }

        @Override
        public A alleleAt(int alleleIndex) {
            return ReadLikelihoods.this.alleleAt(alleleIndex);
        }

        @Override
        public GATKSAMRecord readAt(int readIndex) {
            if (readIndex < 0) {
                throw new IllegalArgumentException("the read-index cannot be negative");
            }
            GATKSAMRecord[] sampleReads = ReadLikelihoods.this.readsBySampleIndex[this.sampleIndex];
            if (readIndex >= sampleReads.length) {
                throw new IllegalArgumentException("the read-index is beyond the read count of the sample");
            }
            return sampleReads[readIndex];
        }

        @Override
        public void copyAlleleLikelihoods(int alleleIndex, double[] dest, int offset) {
            System.arraycopy(ReadLikelihoods.this.valuesBySampleIndex[this.sampleIndex][alleleIndex], 0, dest, offset, this.readCount());
        }
    }

    public class BestAllele {
        public static final double INFORMATIVE_THRESHOLD = 0.2;
        public final A allele;
        public final String sample;
        public final GATKSAMRecord read;
        public final double likelihood;
        public final double confidence;

        private BestAllele(int sampleIndex, int readIndex, int bestAlleleIndex, double likelihood, double secondBestLikelihood) {
            this.allele = bestAlleleIndex == -1 ? null : ReadLikelihoods.this.alleles.alleleAt(bestAlleleIndex);
            this.likelihood = likelihood;
            this.sample = ReadLikelihoods.this.samples.sampleAt(sampleIndex);
            this.read = ReadLikelihoods.this.readsBySampleIndex[sampleIndex][readIndex];
            this.confidence = likelihood == secondBestLikelihood ? 0.0 : likelihood - secondBestLikelihood;
        }

        public boolean isInformative() {
            return this.confidence > 0.2;
        }
    }

    public static interface Matrix<A extends Allele>
    extends AlleleList<A> {
        public List<GATKSAMRecord> reads();

        public List<A> alleles();

        public void set(int var1, int var2, double var3);

        public double get(int var1, int var2);

        @Override
        public int alleleIndex(A var1);

        public int readIndex(GATKSAMRecord var1);

        @Override
        public int alleleCount();

        public int readCount();

        @Override
        public A alleleAt(int var1);

        public GATKSAMRecord readAt(int var1);

        public void copyAlleleLikelihoods(int var1, double[] var2, int var3);
    }
}

