/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.file;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.AsynchronousCloseException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.FileLockInterruptionException;
import java.nio.channels.OverlappingFileLockException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.log4j.Logger;
import org.broadinstitute.gatk.utils.exceptions.UserException;

public class FSLockWithShared {
    private static final Logger logger = Logger.getLogger(FSLockWithShared.class);
    private final File file;
    private FileLock lock = null;
    private FileChannel channel = null;
    private int lockAcquisitionTimeout;
    public static final int DEFAULT_LOCK_ACQUISITION_TIMEOUT_IN_MILLISECONDS = 30000;
    public static final int THREAD_TERMINATION_TIMEOUT_IN_MILLISECONDS = 30000;

    public FSLockWithShared(File file) {
        this.file = file;
        this.lockAcquisitionTimeout = 30000;
    }

    public FSLockWithShared(File file, int lockAcquisitionTimeout) {
        this.file = file;
        this.lockAcquisitionTimeout = lockAcquisitionTimeout;
    }

    public boolean sharedLock() {
        return this.acquireLockWithTimeout(true);
    }

    public boolean exclusiveLock() {
        return this.acquireLockWithTimeout(false);
    }

    private boolean acquireLockWithTimeout(boolean acquireSharedLock) {
        ExecutorService executor = Executors.newSingleThreadExecutor(new ThreadFactory(){

            @Override
            public Thread newThread(Runnable r) {
                Thread lockAcquisitionThread = new Thread(r);
                lockAcquisitionThread.setDaemon(true);
                return lockAcquisitionThread;
            }
        });
        FutureTask<Boolean> lockAcquisitionTask = new FutureTask<Boolean>(new LockAcquisitionTask(acquireSharedLock));
        boolean lockAcquired = false;
        try {
            executor.execute(lockAcquisitionTask);
            lockAcquired = lockAcquisitionTask.get(this.lockAcquisitionTimeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            throw new UserException.FileSystemInabilityToLockException(String.format("Timeout of %d milliseconds was reached while trying to acquire a lock on file %s. Since the GATK uses non-blocking lock acquisition calls that are not supposed to wait, this implies a problem with the file locking support in your operating system.", this.lockAcquisitionTimeout, this.file.getAbsolutePath()));
        }
        catch (ExecutionException e) {
            logger.warn(String.format("WARNING: Unable to lock file %s because exception %s occurred with error message %s", this.file.getAbsolutePath(), e.getCause() != null ? e.getCause().getClass().getSimpleName() : "unknown", e.getCause() != null ? e.getCause().getMessage() : "none"));
            lockAcquired = false;
        }
        catch (InterruptedException e) {
            logger.warn(String.format("WARNING: interrupted while attempting to acquire a lock for file %s", this.file.getAbsolutePath()));
            lockAcquired = false;
        }
        catch (Exception e) {
            logger.warn(String.format("WARNING: error while attempting to acquire a lock for file %s. Error message: %s", this.file.getAbsolutePath(), e.getMessage()));
            lockAcquired = false;
        }
        this.shutdownLockAcquisitionTask(executor);
        if (!lockAcquired) {
            this.unlock();
        }
        return lockAcquired;
    }

    private void shutdownLockAcquisitionTask(ExecutorService executor) {
        boolean shutdownAttemptSucceeded;
        try {
            executor.shutdownNow();
            shutdownAttemptSucceeded = executor.awaitTermination(30000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            shutdownAttemptSucceeded = false;
        }
        if (!shutdownAttemptSucceeded) {
            throw new UserException(String.format("Failed to terminate lock acquisition thread while trying to lock file %s. Exiting because it's not safe to proceed with this run of the GATK.", this.file.getAbsolutePath()));
        }
    }

    public void unlock() {
        this.releaseLock();
        this.closeChannel();
    }

    private void releaseLock() {
        try {
            if (this.lock != null) {
                this.lock.release();
            }
        }
        catch (ClosedChannelException closedChannelException) {
        }
        catch (IOException e) {
            throw new UserException(String.format("An error occurred while releasing the lock for file %s", this.file.getAbsolutePath()), e);
        }
    }

    private void closeChannel() {
        try {
            if (this.channel != null) {
                this.channel.close();
            }
        }
        catch (IOException e) {
            throw new UserException(String.format("An error occurred while closing channel for file %s", this.file.getAbsolutePath()), e);
        }
    }

    private class LockAcquisitionTask
    implements Callable<Boolean> {
        private final boolean acquireSharedLock;

        public LockAcquisitionTask(boolean acquireSharedLock) {
            this.acquireSharedLock = acquireSharedLock;
        }

        @Override
        public Boolean call() {
            try {
                FSLockWithShared.this.channel = new RandomAccessFile(FSLockWithShared.this.file, this.acquireSharedLock ? "r" : "rw").getChannel();
            }
            catch (IOException e) {
                logger.warn(String.format("WARNING: Unable to lock file %s because we could not open a file channel", FSLockWithShared.this.file.getAbsolutePath()));
                return false;
            }
            boolean lockAcquired = false;
            try {
                FSLockWithShared.this.lock = FSLockWithShared.this.channel.tryLock(0L, Long.MAX_VALUE, this.acquireSharedLock);
                lockAcquired = FSLockWithShared.this.lock != null;
            }
            catch (AsynchronousCloseException e) {
                logger.warn(String.format("WARNING: Unable to lock file %s because the file channel was closed by another thread", FSLockWithShared.this.file.getAbsolutePath()));
                lockAcquired = false;
            }
            catch (ClosedChannelException e) {
                logger.warn(String.format("WARNING: Unable to lock file %s because the file channel is closed.", FSLockWithShared.this.file.getAbsolutePath()));
                lockAcquired = false;
            }
            catch (OverlappingFileLockException e) {
                logger.warn(String.format("WARNING: Unable to lock file %s because you already have a lock on this file.", FSLockWithShared.this.file.getAbsolutePath()));
                lockAcquired = false;
            }
            catch (FileLockInterruptionException e) {
                logger.warn(String.format("WARNING: Interrupted while attempting to lock file %s", FSLockWithShared.this.file.getAbsolutePath()));
                lockAcquired = false;
            }
            catch (IOException e) {
                logger.warn(String.format("WARNING: Unable to lock file %s because an IOException occurred with message: %s.", FSLockWithShared.this.file.getAbsolutePath(), e.getMessage()));
                lockAcquired = false;
            }
            return lockAcquired;
        }
    }
}

