/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.fasta;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.reference.FastaSequenceIndex;
import htsjdk.samtools.reference.IndexedFastaSequenceFile;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.sra.SRAAccession;
import htsjdk.samtools.sra.SRAIndexedSequenceFile;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.Arrays;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
import org.broadinstitute.gatk.utils.BaseUtils;
import org.broadinstitute.gatk.utils.exceptions.ReviewedGATKException;
import org.broadinstitute.gatk.utils.exceptions.UserException;

public class CachingIndexedFastaSequenceFile
extends IndexedFastaSequenceFile {
    protected static final Logger logger = Logger.getLogger(CachingIndexedFastaSequenceFile.class);
    private static final boolean PRINT_EFFICIENCY = false;
    private static final int PRINT_FREQUENCY = 10000;
    public static final long DEFAULT_CACHE_SIZE = 1000000L;
    private final long cacheSize;
    private final long cacheMissBackup;
    private final boolean preserveCase;
    private final boolean preserveIUPAC;
    long cacheHits = 0L;
    long cacheMisses = 0L;
    private ThreadLocal<Cache> cache = new ThreadLocal<Cache>(){

        @Override
        protected Cache initialValue() {
            return new Cache();
        }
    };

    public CachingIndexedFastaSequenceFile(File fasta, FastaSequenceIndex index, long cacheSize, boolean preserveCase, boolean preserveIUPAC) {
        super(fasta, index);
        if (cacheSize < 0L) {
            throw new IllegalArgumentException("cacheSize must be > 0");
        }
        this.cacheSize = cacheSize;
        this.cacheMissBackup = Math.max(cacheSize / 1000L, 1L);
        this.preserveCase = preserveCase;
        this.preserveIUPAC = preserveIUPAC;
    }

    public CachingIndexedFastaSequenceFile(File fasta, long cacheSize, boolean preserveCase, boolean preserveIUPAC) throws FileNotFoundException {
        super(fasta);
        if (cacheSize < 0L) {
            throw new IllegalArgumentException("cacheSize must be > 0");
        }
        this.cacheSize = cacheSize;
        this.cacheMissBackup = Math.max(cacheSize / 1000L, 1L);
        this.preserveCase = preserveCase;
        this.preserveIUPAC = preserveIUPAC;
    }

    public CachingIndexedFastaSequenceFile(File fasta, FastaSequenceIndex index, long cacheSize) {
        this(fasta, index, cacheSize, false, false);
    }

    public CachingIndexedFastaSequenceFile(File fasta) throws FileNotFoundException {
        this(fasta, false);
    }

    public CachingIndexedFastaSequenceFile(File fasta, boolean preserveCase) throws FileNotFoundException {
        this(fasta, 1000000L, preserveCase, false);
    }

    public CachingIndexedFastaSequenceFile(File fasta, boolean preserveCase, boolean preserveIUPAC) throws FileNotFoundException {
        this(fasta, 1000000L, preserveCase, preserveIUPAC);
    }

    public static ReferenceSequenceFile checkAndCreate(File fastaFile) {
        if (fastaFile == null) {
            throw new IllegalArgumentException("Fasta file is null");
        }
        if (SRAAccession.isValid(fastaFile.getPath())) {
            return new SRAIndexedSequenceFile(new SRAAccession(fastaFile.getPath()));
        }
        if (!fastaFile.exists()) {
            throw new UserException("The fasta file you specified (" + fastaFile.getAbsolutePath() + ") does not exist.");
        }
        boolean isGzipped = fastaFile.getAbsolutePath().endsWith(".gz");
        if (isGzipped) {
            throw new UserException.CannotHandleGzippedRef();
        }
        File indexFile = new File(fastaFile.getAbsolutePath() + ".fai");
        String fastaExt = fastaFile.getAbsolutePath().endsWith("fa") ? "\\.fa$" : "\\.fasta$";
        File dictFile = new File(fastaFile.getAbsolutePath().replaceAll(fastaExt, ".dict"));
        if (!indexFile.exists()) {
            throw new UserException.MissingReferenceFaiFile(indexFile, fastaFile);
        }
        if (!dictFile.exists()) {
            throw new UserException.MissingReferenceDictFile(dictFile, fastaFile);
        }
        try {
            return new CachingIndexedFastaSequenceFile(fastaFile);
        }
        catch (IllegalArgumentException e) {
            throw new UserException.CouldNotReadInputFile(fastaFile, "Could not read reference sequence.  The FASTA must have either a .fasta or .fa extension", e);
        }
        catch (Exception e) {
            throw new UserException.CouldNotReadInputFile(fastaFile, e);
        }
    }

    public CachingIndexedFastaSequenceFile(File fasta, long cacheSize) throws FileNotFoundException {
        this(fasta, cacheSize, false, false);
    }

    public void printEfficiency(Priority priority) {
        logger.log(priority, String.format("### CachingIndexedFastaReader: hits=%d misses=%d efficiency %.6f%%", this.cacheHits, this.cacheMisses, this.calcEfficiency()));
    }

    public double calcEfficiency() {
        return 100.0 * (double)this.cacheHits / ((double)this.cacheMisses + (double)this.cacheHits * 1.0);
    }

    public long getCacheHits() {
        return this.cacheHits;
    }

    public long getCacheMisses() {
        return this.cacheMisses;
    }

    public long getCacheSize() {
        return this.cacheSize;
    }

    public boolean isPreservingCase() {
        return this.preserveCase;
    }

    public boolean isUppercasingBases() {
        return !this.isPreservingCase();
    }

    public boolean isPreservingIUPAC() {
        return this.preserveIUPAC;
    }

    @Override
    public ReferenceSequence getSubsequenceAt(String contig, long start, long stop) {
        ReferenceSequence result;
        Cache myCache = this.cache.get();
        if (stop - start >= this.cacheSize) {
            ++this.cacheMisses;
            result = super.getSubsequenceAt(contig, start, stop);
            if (!this.preserveCase) {
                StringUtil.toUpperCase(result.getBases());
            }
            if (!this.preserveIUPAC) {
                BaseUtils.convertIUPACtoN(result.getBases(), true, start < 1L);
            }
        } else {
            SAMSequenceRecord contigInfo = super.getSequenceDictionary().getSequence(contig);
            if (stop > (long)contigInfo.getSequenceLength()) {
                throw new SAMException("Query asks for data past end of contig");
            }
            if (start < myCache.start || stop > myCache.stop || myCache.seq == null || myCache.seq.getContigIndex() != contigInfo.getSequenceIndex()) {
                ++this.cacheMisses;
                myCache.start = Math.max(start - this.cacheMissBackup, 0L);
                myCache.stop = Math.min(start + this.cacheSize + this.cacheMissBackup, (long)contigInfo.getSequenceLength());
                myCache.seq = super.getSubsequenceAt(contig, myCache.start, myCache.stop);
                if (!this.preserveCase) {
                    StringUtil.toUpperCase(myCache.seq.getBases());
                }
                if (!this.preserveIUPAC) {
                    BaseUtils.convertIUPACtoN(myCache.seq.getBases(), true, myCache.start == 0L);
                }
            } else {
                ++this.cacheHits;
            }
            int cacheOffsetStart = (int)(start - myCache.start);
            int cacheOffsetStop = (int)(stop - start + (long)cacheOffsetStart + 1L);
            try {
                result = new ReferenceSequence(myCache.seq.getName(), myCache.seq.getContigIndex(), Arrays.copyOfRange(myCache.seq.getBases(), cacheOffsetStart, cacheOffsetStop));
            }
            catch (ArrayIndexOutOfBoundsException e) {
                throw new ReviewedGATKException(String.format("BUG: bad array indexing.  Cache start %d and end %d, request start %d end %d, offset start %d and end %d, base size %d", myCache.start, myCache.stop, start, stop, cacheOffsetStart, cacheOffsetStop, myCache.seq.getBases().length), e);
            }
        }
        return result;
    }

    private static class Cache {
        long start = -1L;
        long stop = -1L;
        ReferenceSequence seq = null;

        private Cache() {
        }
    }
}

