/*  $Id: test_histogram_binning.cpp 407707 2013-07-23 19:09:51Z kornbluh $
 * ===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *   This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 * Author: Michael Kornbluh
 *
 * File Description:
 *   Test program for CHistogramBinning class
 *
 */

#include <ncbi_pch.hpp>

#include <util/histogram_binning.hpp>
#include <util/random_gen.hpp>
#include <corelib/ncbitime.hpp>

#define BOOST_AUTO_TEST_MAIN
#include <corelib/test_boost.hpp>

#include <common/test_assert.h>  // This header must go last


USING_NCBI_SCOPE;

// private functions namespace
namespace {

    void s_GenericListOfBinsCheck(
        const CHistogramBinning::TListOfBins & list_of_bins,
        const Uint8 expected_num_bins,
        const Uint8 expected_total_data_points )
    {
        if( expected_num_bins > 0 ) {
            BOOST_CHECK( list_of_bins.size() <= expected_num_bins);
        }
        if( list_of_bins.empty() ) {
            return;
        }
        Uint8 total_data_points = 0;
        ITERATE(CHistogramBinning::TListOfBins, bin_iter, list_of_bins) {
            const CHistogramBinning::SBin & bin = *bin_iter;
            BOOST_CHECK( bin.first_number <= bin.last_number );
            BOOST_CHECK( bin.total_appearances > 0 );
            total_data_points += bin.total_appearances;
        }
        BOOST_CHECK_EQUAL(total_data_points, expected_total_data_points );

        // check that the last of each bin is less than the first of the
        // next bin
        CHistogramBinning::TListOfBins::const_iterator this_bin_iter =
            list_of_bins.begin();
        CHistogramBinning::TValue last_value = this_bin_iter->last_number;
        while( ++this_bin_iter != list_of_bins.end() ) {
            const CHistogramBinning::SBin & bin = *this_bin_iter;
            BOOST_CHECK( last_value < bin.first_number );
        }
    }

    void s_PrintListOfBins(
        const CHistogramBinning::TListOfBins & list_of_bins)
    {
        cerr << "BEGIN HISTOGRAM" << endl;
        ITERATE(CHistogramBinning::TListOfBins, bin_iter, list_of_bins) {
            const CHistogramBinning::SBin & bin = *bin_iter;
            cerr << "\t(" << bin.first_number << "-" << bin.last_number
                 << "): " << bin.total_appearances << endl;
        }
        cerr << "END HISTOGRAM" << endl;
    }

    AutoPtr<CHistogramBinning::TListOfBins>
    s_TestArray(const Int8 numbers[], 
                     const Uint8 num_numbers, 
                     const Uint8 num_bins,
                     CHistogramBinning::EHistAlgo eHistAlgo = 
                        CHistogramBinning::eHistAlgo_Default )
    {
        CHistogramBinning binner(num_bins);

        ITERATE_0_IDX(ii, num_numbers) {
            binner.AddNumber(numbers[ii], 1);
        }

        AutoPtr<CHistogramBinning::TListOfBins> list_of_bins = 
            binner.CalcHistogram(eHistAlgo);
        s_PrintListOfBins(*list_of_bins);

        s_GenericListOfBinsCheck(*list_of_bins, num_bins, num_numbers);

        return list_of_bins;
    }

    // helper macro to make s_TestArray slightly easier to use
#define TEST_ARRAY(_array, _num_bins) \
        s_TestArray(_array, ArraySize(_array), _num_bins)

}

// private data namespace
namespace {

    // these were generated by a random
    // uniform distribution of [1, 100]
    Int8 uniform_random_numbers[] = {
        18,  60, 57, 7,   13, 
        19,  85, 13, 96,  32, 
        61,  6,  41, 76,  29, 
        16,  39, 19, 29,  77, 
        78,  66, 98, 14,  31, 
        25,  71, 25, 93,  40, 
        5,   83, 44, 47,  52, 
        34,  70, 70, 47,  29, 
        52,  73, 1,  59,  6, 
        4,   71, 74, 98,  6, 
        100, 58, 5,  17,  89, 
        67,  94, 61, 94,  93, 
        7,   42, 83, 20,  9, 
        57,  63, 91, 84,  95, 
        32,  34, 79, 62,  85, 
        24,  28, 13, 80,  54, 
        87,  9,  32, 24,  71, 
        64,  91, 85, 60,  57, 
        65,  94, 85, 100, 87, 
        65,  28, 39, 24,  41
    };

}

BOOST_AUTO_TEST_CASE(TestBasic)
{
    TEST_ARRAY(uniform_random_numbers, 0);
    TEST_ARRAY(uniform_random_numbers, 1);
    TEST_ARRAY(uniform_random_numbers, 2);
    TEST_ARRAY(uniform_random_numbers, 9);
    TEST_ARRAY(uniform_random_numbers, 23);
    TEST_ARRAY(uniform_random_numbers, 100);
    TEST_ARRAY(uniform_random_numbers, 1000);
}

BOOST_AUTO_TEST_CASE(TestNoData) {
    Int8 dummy = 80; // 80 has no significance whatsoever

    s_TestArray(&dummy, 0, 0);
    s_TestArray(&dummy, 0, 1);
    s_TestArray(&dummy, 0, 2);
}

BOOST_AUTO_TEST_CASE(TestOutlier) {
    Int8 data_with_outlier[] = { 8, 90, 4, 7, 7, 12 };

    AutoPtr<CHistogramBinning::TListOfBins> pListOfBins = 
        TEST_ARRAY(data_with_outlier, 2);
    
    const CHistogramBinning::SBin & first_bin = pListOfBins->front();
    const CHistogramBinning::SBin & last_bin = pListOfBins->back();

    BOOST_CHECK_EQUAL(first_bin.first_number, 4);
    BOOST_CHECK_EQUAL(first_bin.last_number, 12);
    BOOST_CHECK_EQUAL(last_bin.first_number, 90);
    BOOST_CHECK_EQUAL(last_bin.last_number,  90);
}

BOOST_AUTO_TEST_CASE(TestEfficiency) {
    // speed should be about O(n lg n), so even a huge
    // number of data points should
    // be reasonably quick

    vector<CHistogramBinning::TValue> value_vec;

    CStopWatch stop_watch;

    const Uint8 kNumDataPoints = 1000000;

    // generate the pseudo-random data
    cerr << "Generating " << kNumDataPoints << " random numbers" << endl;
    stop_watch.Restart();
    CRandom randgen; // do NOT seed, we want it to be deterministic
    ITERATE_0_IDX(ii, kNumDataPoints) {
        value_vec.push_back( randgen.GetRand() % kNumDataPoints );
    }
    cerr << "Time to generate the numbers, in seconds: " 
         << stop_watch.Elapsed() << endl;


    cerr << "Creating histogram" << endl;
    stop_watch.Restart();
    s_TestArray(&value_vec[0], value_vec.size(), 0);
    cerr << "Time taken to create histogram, in seconds: " 
         << stop_watch.Elapsed() << endl;

    cerr << "Creating histogram that aims to be even" << endl;
    stop_watch.Restart();
    s_TestArray(&value_vec[0], value_vec.size(), 0, 
        CHistogramBinning::eHistAlgo_TryForSameNumDataInEachBin);
    cerr << "Time taken to create supposedly-even histogram, in seconds: " 
         << stop_watch.Elapsed() << endl;
}

