/*  $Id: open_view_dlg.cpp 40532 2018-03-02 21:28:41Z katargir $
 * ===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 * Authors:
 *
 * File Description:
 *
 */

#include <ncbi_pch.hpp>

#include <gui/core/open_view_dlg.hpp>

#include <gui/framework/view_manager_service.hpp>
#include <gui/framework/item_selection_panel.hpp>

#include <gui/widgets/wx/wx_utils.hpp>

#include <gui/objutils/registry.hpp>
#include <gui/objutils/reg_settings.hpp>

#include <gui/objutils/interface_registry.hpp>
#include <gui/objutils/gui_object_info.hpp>

#include <wx/sizer.h>
#include <wx/button.h>
#include <wx/bitmap.h>
#include <wx/icon.h>
#include <wx/checkbox.h>
#include <wx/valgen.h>


BEGIN_NCBI_SCOPE


///////////////////////////////////////////////////////////////////////////////
/// CToolManagerItem
class CViewFactoryItem :
    public CObject,
    public IItemWithDescription
{
public:
    CViewFactoryItem(IProjectViewFactory& factory, bool showDefault = false)
    :   m_Factory(&factory), m_ShowDefault(showDefault)
    {
    }

    virtual const IUIObject&  GetDescriptor() const
    {
        return m_Factory->GetViewTypeDescriptor();
    }

    virtual string    GetCategory()
    {
        return m_Factory->GetViewTypeDescriptor().GetCategory();
    }

    virtual bool ShowDefault() const { return m_ShowDefault; }


    CIRef<IProjectViewFactory>  m_Factory;
    bool    m_ShowDefault;
};

///////////////////////////////////////////////////////////////////////////////
/// COpenViewDlg

IMPLEMENT_DYNAMIC_CLASS( COpenViewDlg, CDialog )

BEGIN_EVENT_TABLE( COpenViewDlg, CDialog )
////@begin COpenViewDlg event table entries
    EVT_INIT_DIALOG( COpenViewDlg::OnInitDialog )

    EVT_BUTTON( wxID_BACKWARD, COpenViewDlg::OnBackwardClick )

    EVT_BUTTON( wxID_FORWARD, COpenViewDlg::OnForwardClick )

////@end COpenViewDlg event table entries

    EVT_CHECKBOX( ID_ONLY, COpenViewDlg::OnOnlyClick )
    EVT_LISTBOX_DCLICK(wxID_ANY, COpenViewDlg::OnViewSelected)
    EVT_LIST_ITEM_ACTIVATED(wxID_ANY, COpenViewDlg::OnListItemActivate)
    EVT_MENU( wxID_SELECTALL, COpenViewDlg::OnSelectAll )
END_EVENT_TABLE()


COpenViewDlg::COpenViewDlg()
{
    Init();
}


COpenViewDlg::COpenViewDlg( wxWindow* parent, wxWindowID id, const wxString& caption, const wxPoint& pos, const wxSize& size, long style )
{
    Init();
    Create(parent, id, caption, pos, size, style);
}


bool COpenViewDlg::Create( wxWindow* parent, wxWindowID id, const wxString& caption, const wxPoint& pos, const wxSize& size, long style )
{
////@begin COpenViewDlg creation
    SetExtraStyle(wxWS_EX_BLOCK_EVENTS);
    CDialog::Create( parent, id, caption, pos, size, style );

    CreateControls();
    if( GetSizer() ){
        GetSizer()->SetSizeHints(this);
    }
    Centre();
////@end COpenViewDlg creation

    m_DefaultManager->InitUI();
    m_DefaultManager->SetParentWindow( m_Panel );

    return true;
}

COpenViewDlg::~COpenViewDlg()
{
////@begin COpenViewDlg destruction
////@end COpenViewDlg destruction
}


void COpenViewDlg::Init()
{
////@begin COpenViewDlg member initialisation
    m_Panel = NULL;
    m_BackBtn = NULL;
    m_NextBtn = NULL;
////@end COpenViewDlg member initialisation

    m_ItemPanel = NULL;
    m_ViewListPanel = NULL;
    m_CurrPanel = NULL;
    m_CompatibleOnly = false;
    
    m_DefaultManager = new COpenViewManager();
}


void COpenViewDlg::CreateControls()
{
////@begin COpenViewDlg content construction
    // Generated by DialogBlocks, 01/07/2009 13:07:46 (unregistered)

    COpenViewDlg* itemCDialog1 = this;

    wxBoxSizer* itemBoxSizer2 = new wxBoxSizer(wxVERTICAL);
    itemCDialog1->SetSizer(itemBoxSizer2);

    m_Panel = new wxPanel( itemCDialog1, ID_OPENVIEWPANEL, wxDefaultPosition, wxSize(400, 300), wxNO_BORDER|wxFULL_REPAINT_ON_RESIZE|wxTAB_TRAVERSAL );
    itemBoxSizer2->Add(m_Panel, 1, wxGROW, 5);

    wxBoxSizer* itemBoxSizer4 = new wxBoxSizer(wxVERTICAL);
    m_Panel->SetSizer(itemBoxSizer4);

    wxStaticLine* itemStaticLine5 = new wxStaticLine( itemCDialog1, wxID_STATIC, wxDefaultPosition, wxDefaultSize, wxLI_HORIZONTAL );
    itemBoxSizer2->Add(itemStaticLine5, 0, wxGROW|wxALL, 5);

    wxBoxSizer* itemBoxSizer6 = new wxBoxSizer(wxHORIZONTAL);
    itemBoxSizer2->Add(itemBoxSizer6, 0, wxGROW|wxLEFT|wxRIGHT, 5);

    wxHyperlinkCtrl* itemHyperlinkCtrl7 = new wxHyperlinkCtrl( itemCDialog1, ID_HELPLINK, _("Help"), wxEmptyString, wxDefaultPosition, wxDefaultSize, wxHL_DEFAULT_STYLE );
    itemHyperlinkCtrl7->SetForegroundColour(wxColour(192, 192, 192));
    itemHyperlinkCtrl7->Enable(false);
    itemBoxSizer6->Add(itemHyperlinkCtrl7, 0, wxALIGN_CENTER_VERTICAL|wxALL, 5);

    itemBoxSizer6->Add(5, 5, 1, wxALIGN_CENTER_VERTICAL|wxALL, 5);

    wxBoxSizer* itemBoxSizer9 = new wxBoxSizer(wxHORIZONTAL);
    itemBoxSizer6->Add(itemBoxSizer9, 0, wxALIGN_CENTER_VERTICAL, 5);

    m_BackBtn = new wxButton( itemCDialog1, wxID_BACKWARD, _("< &Back"), wxDefaultPosition, wxDefaultSize, 0 );
    itemBoxSizer9->Add(m_BackBtn, 0, wxALIGN_CENTER_VERTICAL|wxLEFT|wxTOP|wxBOTTOM, 5);

    // in wxCocoa, we get a little button overlap here without the spacer.  I believe
    // this is a bug, but we need to pad for now. The variable is added
    // to create a syntax error if this is overwritten by DialogBlocks
    bool spacer_code_included = true;
#ifdef __WXOSX_COCOA__   
    itemBoxSizer9->AddSpacer(6);
#endif

    m_NextBtn = new wxButton( itemCDialog1, wxID_FORWARD, _("&Next >"), wxDefaultPosition, wxDefaultSize, 0 );
    m_NextBtn->SetDefault();
    itemBoxSizer9->Add(m_NextBtn, 0, wxALIGN_CENTER_VERTICAL|wxRIGHT|wxTOP|wxBOTTOM, 5);

    wxButton* itemButton12 = new wxButton( itemCDialog1, wxID_CANCEL, _("&Cancel"), wxDefaultPosition, wxDefaultSize, 0 );
    itemBoxSizer6->Add(itemButton12, 0, wxALIGN_CENTER_VERTICAL|wxALL, 5);

////@end COpenViewDlg content construction

    if (!spacer_code_included) {
        // If spacer_code_included is not defined, it has been overwritten 
        // so you must add the following code after m_BackBtn is added and
        // before m_NextBtn is added.  We need this since that section of code
        // can be overridden by DialogBlocks.
        /*
    // in wxCocoa, we get a little button overlap here without the spacer.  I believe
    // this is a bug, but we need to pad for now. The variable is added
    // to create a syntax error if this is overwritten by DialogBlocks
    bool spacer_code included = true;
#ifdef __WXOSX_COCOA__   
    itemBoxSizer9->AddSpacer(6);
#endif
        */
    }

    m_ViewListPanel = new wxPanel( m_Panel, wxID_ANY, wxDefaultPosition, wxSize(400, 300) );
    wxBoxSizer* itemBoxSizer13 = new wxBoxSizer(wxVERTICAL);
    m_ViewListPanel->SetSizer(itemBoxSizer13);

    m_ItemPanel = new CItemSelectionPanel(m_ViewListPanel, wxID_ANY);
    m_ItemPanel->SetItemTypeLabel("view");
    CMapControl* map_control = m_ItemPanel->m_MapWidget->GetMapControl();
    CMapControl::SProperties& props = map_control->GetProperties();
    props.m_SingleColumn = true;
    itemBoxSizer13->Add(m_ItemPanel, 1, wxGROW, 5);

    wxCheckBox* onlyCheckBox = new wxCheckBox( m_ViewListPanel, ID_ONLY, wxT("Show only compatible views"), wxDefaultPosition, wxDefaultSize, 0 );
    onlyCheckBox->SetValue(m_CompatibleOnly);
    itemBoxSizer13->Add(onlyCheckBox, 0, wxALIGN_LEFT|wxALL, 5);
    onlyCheckBox->SetValidator( wxGenericValidator(& m_CompatibleOnly) );

    x_SetCurrentPanel(m_ViewListPanel);
    x_UpdateButtons();
}


void COpenViewDlg::SetFactories(TFactories& factories)
{
    m_Factories = factories;
    for( size_t i =0; i < m_Factories.size(); i++ ){
        IOpenViewManager* manager = m_Factories[i]->GetOpenViewManager();
        if( manager ){
            manager->InitUI();
            manager->SetParentWindow( m_Panel );

            if( !m_RegPath.empty() ){
                IRegSettings* rgs = dynamic_cast<IRegSettings*>( manager );
                if( rgs ){
                    string sub_section = m_Factories[i]->GetViewTypeDescriptor().GetLabel();
                    rgs->SetRegistryPath( m_RegPath + "." + sub_section );
                }
            }
        }
    }

    m_CompatibleFactories.clear();
    m_CompatibleEmpty = true;

    x_UpdateViewPanel();
}

void COpenViewDlg::SetInputObjects(const TConstScopedObjects& input_objects)
{
    m_InputObjects = input_objects;

    if (m_InputObjects.size() == 1) {
        CIRef<IGuiObjectInfo> gui_info(CreateObjectInterface<IGuiObjectInfo>(m_InputObjects[0], NULL));
		if (!gui_info)
			return;
        m_InputCategory = gui_info->GetViewCategory();
    }

    // Enable Default view functionality
    m_ItemPanel->m_MapWidget->SetInputCategory(m_InputCategory);

}

COpenViewDlg::TFactoryRef COpenViewDlg::GetSelectedFactory()
{
    CIRef<IItemWithDescription> item = m_ItemPanel->GetSelectedItemRef();
    CViewFactoryItem* fc_item = dynamic_cast<CViewFactoryItem*>(item.GetPointer());
    if( fc_item != NULL ){
        return fc_item->m_Factory;
    } else {
        return TFactoryRef();
    }
}


void COpenViewDlg::OnInitDialog( wxInitDialogEvent& event )
{
    //x_UpdateViewPanel();
    wxDialog::OnInitDialog(event);

}


void COpenViewDlg::OnOnlyClick( wxCommandEvent& event )
{
    m_ViewListPanel->TransferDataFromWindow();

    x_UpdateViewPanel();
}


void COpenViewDlg::OnBackwardClick( wxCommandEvent& event )
{
    x_DoTransition( IOpenViewManager::eBack );
}


void COpenViewDlg::OnForwardClick( wxCommandEvent& event )
{
    if(m_CurrPanel == m_ViewListPanel ){
        // this is the first step
        OnViewSelected( event );
    } else {
        x_DoTransition( IOpenViewManager::eNext );
    }
}

void COpenViewDlg::OnViewSelected( wxCommandEvent& event )
{
    TFactoryRef factory = GetSelectedFactory();
    if( factory.IsNull() ){
        //! should be replaced by NcbiMessageBox
        wxMessageBox(
            wxT("Please select a view to be opened."),
            wxT("Open View"), wxOK | wxICON_EXCLAMATION, this
        );

        return;
    }

    m_CurrFactory = factory;

    IOpenViewManager* manager = factory->GetOpenViewManager();
    if( !manager ){
        manager = m_DefaultManager.GetPointer();
        m_DefaultManager->SetProjectViewFactory( m_CurrFactory.GetPointer() );
    }
    m_CurrManager = manager;

    m_CurrManager->SetInputObjects( m_InputObjects );
    x_DoTransition( IOpenViewManager::eNext );
}


void COpenViewDlg::OnListItemActivate( wxListEvent &event )
{
    x_DoTransition( IOpenViewManager::eNext );
}

void COpenViewDlg::OnSelectAll( wxCommandEvent &anEvt )
{
    anEvt.Skip();
}

void COpenViewDlg::SetRegistryPath(const string& path)
{
    m_RegPath = path; // store for later use
    m_ItemPanel->SetRegistryPath( path + ".ItemsPanel" );
    m_DefaultManager->SetRegistryPath( path + ".DefaultManager" );

    /// set Registry paths for all Managers
    /// each Manager will get a section located under the dialog's section
    for( size_t i =0; i < m_Factories.size(); i++ ){
        IOpenViewManager* manager = m_Factories[i]->GetOpenViewManager();
        if( manager ){
            IRegSettings* rgs = dynamic_cast<IRegSettings*>( manager );
            if( rgs ){
                string sub_section = m_Factories[i]->GetViewTypeDescriptor().GetLabel();
                rgs->SetRegistryPath( m_RegPath + "." + sub_section );
            }
        }
    }
}

void COpenViewDlg::EndModal( int ret_code )
{
    SaveSettings();

    for (size_t i =0; i < m_Factories.size(); i++) {
        IOpenViewManager* manager = m_Factories[i]->GetOpenViewManager();
        if( manager ){
            manager->CleanUI();
        }
    }

    m_DefaultManager->SetProjectViewFactory( NULL );
    m_DefaultManager->CleanUI();

    wxDialog::EndModal( ret_code );
}


static const char* kOnlyCompatibleTag = "OnlyCompatible";
static const char* kSelOptionTag = "SelectedOption";
static const char* kDefaultItem = "DefaultView.";

void COpenViewDlg::x_SaveSettings(CRegistryWriteView view) const
{
    if( !m_RegPath.empty() ){
        // save Items Panel
        m_ItemPanel->SaveSettings();
        view.Set( kOnlyCompatibleTag, m_CompatibleOnly );

        // remember the selected option
        if( m_CurrFactory ){
            string mgr_label = m_CurrFactory->GetViewTypeDescriptor().GetLabel();
            view.Set(kSelOptionTag, mgr_label);
        }

        // save Managers' settigns
        m_DefaultManager->SaveSettings();

        for( size_t i =0; i < m_Factories.size(); i++ ){
            const IRegSettings* rgs = 
                dynamic_cast<const IRegSettings*>(m_Factories[i].GetPointer())
            ;
            if( rgs ){
                rgs->SaveSettings();
            }
        }

        if (!m_InputCategory.empty()) {
            string defaultItem = m_ItemPanel->m_MapWidget->GetDefaultItem();
            view.Set(kDefaultItem + m_InputCategory, defaultItem);
        }
    }
}

void COpenViewDlg::x_LoadSettings(const CRegistryReadView& view)
{
    if( !m_RegPath.empty() ){
        // load Items panel settings
        m_ItemPanel->LoadSettings();
        m_CompatibleOnly = view.GetBool(kOnlyCompatibleTag, m_CompatibleOnly);

        // load the default option
        string sel_mgr_label = view.GetString(kSelOptionTag, "empty");

        // load Managers' settigns and find the selected manager
        m_DefaultManager->LoadSettings();

        for (size_t i =0; i < m_Factories.size(); i++) {
            IRegSettings* rgs = dynamic_cast<IRegSettings*>(m_Factories[i].GetPointer());
            if( rgs ){
                rgs->LoadSettings();
            }
            // check whether this is the selected manager
            string mgr_label = m_Factories[i]->GetViewTypeDescriptor().GetLabel();
            if( mgr_label == sel_mgr_label ){
                m_CurrFactory = m_Factories[i];

                IOpenViewManager* manager = m_CurrFactory->GetOpenViewManager();
                if( !manager ){
                    manager = m_DefaultManager.GetPointer();
                    m_DefaultManager->SetProjectViewFactory( m_CurrFactory );
                }

                m_CurrManager = manager; // found the selected manager
            }
        }

        if (!m_InputCategory.empty()) {
            string defaultItem = view.GetString(kDefaultItem + m_InputCategory);
            m_ItemPanel->m_MapWidget->SetDefaultItem(defaultItem);
        }

        m_ViewListPanel->TransferDataToWindow();

        x_UpdateViewPanel();
    }
}

string COpenViewDlg::GetDefaultFactoryLabel(const string& reg_path, TConstScopedObjects& objects)
{
    if (objects.size() != 1)
        return "";

    CIRef<IGuiObjectInfo> gui_info(CreateObjectInterface<IGuiObjectInfo>(objects[0], NULL));
    if (!gui_info)
        return "";

    string inputCategory = gui_info->GetViewCategory();

    return CGuiRegistry::GetInstance().GetString(reg_path + "." + kDefaultItem + inputCategory);
}

bool COpenViewDlg::ShowToolTips()
{
    return true;
}


wxBitmap COpenViewDlg::GetBitmapResource( const wxString& name )
{
    // Bitmap retrieval
////@begin COpenViewDlg bitmap retrieval
    wxUnusedVar(name);
    return wxNullBitmap;
////@end COpenViewDlg bitmap retrieval
}


wxIcon COpenViewDlg::GetIconResource( const wxString& name )
{
    // Icon retrieval
////@begin COpenViewDlg icon retrieval
    wxUnusedVar(name);
    return wxNullIcon;
////@end COpenViewDlg icon retrieval
}

void COpenViewDlg::x_UpdateViewPanel()
{
    TFactories* factories = NULL;
    if( m_CompatibleOnly ){
        if( m_CompatibleEmpty ){
            // build a list of compatible factories
            int good_mask = IProjectViewFactory::fCanShowSome | IProjectViewFactory::fCanShowAll;
            for( size_t i = 0; i < m_Factories.size(); i++ ){
                TFactoryRef& factory_ref = m_Factories[i];
                int test = factory_ref->TestInputObjects( m_InputObjects );
                if( test & good_mask ){
                    m_CompatibleFactories.push_back( factory_ref );
                }
            }
            m_CompatibleEmpty = false; // do this only once
        }
        factories = &m_CompatibleFactories;
    } else {
        factories = &m_Factories;
    }

    // create map items
    typedef CIRef<IItemWithDescription> TItemRef;
    vector<TItemRef> items;
    for( size_t i = 0; i < factories->size(); i++ ){
        bool showDefault = false;

        TFactoryRef& factory_ref = (*factories)[i];

        if (m_InputObjects.size() == 1) {
            int test = factory_ref->TestInputObjects( m_InputObjects );
            if (test&IProjectViewFactory::fCanShowAll)
                showDefault = true;
        }

        TItemRef item( new CViewFactoryItem( *factory_ref, showDefault ) );
        items.push_back( item );
    }

    // reset panel
    m_ItemPanel->SetItems( items );
}


void COpenViewDlg::x_SetCurrentPanel( wxPanel* panel, const wxString& caption )
{
    if (panel == m_CurrPanel) {
        return;
    }

    if (m_CurrPanel) {
        m_CurrPanel->Hide();
        m_CurrPanel = NULL;
        SetTitle( wxT("") );
    }
    if (panel) {
        if (panel->GetContainingSizer() != m_Panel->GetSizer()) {
            m_Panel->GetSizer()->Add(panel, 1, wxEXPAND | wxALL, 5);
        }
        m_CurrPanel = panel;
        SetTitle(caption);

        m_CurrPanel->Show();
        m_Panel->Layout();
        m_CurrPanel->SetFocus();
    }
    Layout();
}


void COpenViewDlg::x_DoTransition( IOpenViewManager::EAction action )
{
    if( m_CurrManager->DoTransition( action ) ){

        if( m_CurrManager->IsCompletedState() ){
            try {
                // we are done.
                m_OutputObjects = m_CurrManager->GetOutputObjects();
                m_OutputIndices = m_CurrManager->GetOutputIndices(); 

                EndModal( wxID_OK );

            } catch( CException& ex ){
                EndModal( wxID_CANCEL );

                //!should be replaced by NcbiMessageBox
                wxMessageBox(
                    wxT("Particular kind of this input object is not supported:\n")
                    + ToWxString( ex.GetMsg() ),
                    wxT("Open View"), wxOK | wxICON_EXCLAMATION
                );
            }

        } else {
            wxPanel* new_panel = m_CurrManager->GetCurrentPanel();

            if( new_panel == NULL ){
                // pre-initial state
                x_SetCurrentPanel( m_ViewListPanel );

            } else {
                wxString title = SYMBOL_COPENVIEWDLG_TITLE;
                title += wxT(" - ");

                if( m_CurrFactory.NotNull() ){
                    title += ToWxString(m_CurrFactory->GetViewTypeDescriptor().GetLabel());
                }

                x_SetCurrentPanel( new_panel, title );
            }
        }
    }

    x_UpdateButtons();
}

void COpenViewDlg::x_UpdateButtons()
{
    bool can_back = false;
    bool can_next = true;
    bool final = false;

    if( m_CurrManager ){
        can_back = m_CurrManager->CanDo( IOpenViewManager::eBack );
        can_next = m_CurrManager->CanDo( IOpenViewManager::eNext );
        final = m_CurrManager->AtFinalStep();
    }

    m_BackBtn->Show( can_back );
    m_NextBtn->Show( can_next );
    m_NextBtn->SetLabel( final? wxT("Finish") : wxT("Next >") );

    Layout();
}

END_NCBI_SCOPE



