/*  $Id: test_ncbidiag_p.cpp 461720 2015-03-12 13:00:58Z ivanov $
 * ===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 * Author:  Vyacheslav Kononenko
 *
 * File Description:
 *   Test for "NCBIDIAG_P"
 *
 */

#include <ncbi_pch.hpp>
#include "test_ncbidiag_p.hpp"
#include <corelib/ncbiapp.hpp>
#include "../ncbidiag_p.hpp"

#include <iomanip>

#include <common/test_assert.h>  /* This header must go last */


USING_NCBI_SCOPE;


////////////////////////////////
// Test application
//

class CTest : public CNcbiApplication
{
public:
    typedef vector<CNcbiDiag*> TNcbiDiags;
    void Init(void);
    int  Run(void);
private:
    void x_TestBadFormats(void);
    void x_TestString(const char* str, int expects[]);
    void x_TestPathes(void);
    void x_TestFormats(void);
    void x_CheckDiag(CNcbiDiag *diag, const string& desc, bool module_expected);
    void x_TestDiagCompileInfo(void);
    void x_TestSeverity(void);
    void x_TestErrCode(void);

    int        m_Result;
    TNcbiDiags m_Diags;
};


void CTest::Init(void)
{
    m_Result = 0;
}


void CTest::x_TestBadFormats(void)
{
    const char* badFormats[] = { "!!", "!module!", "module:", "function(", "module::class::function::", "module::function()::", 0 };
    CDiagFilter tester;
    
    NcbiCout << "Testing bad format strings" << NcbiEndl;

    for( const char** str = badFormats; *str; ++str ){
        NcbiCout << setw(61) << string("\"") + *str + "\"" << " - ";
        bool failed = false;
        try { 
            tester.Fill(*str);
        }
        catch(const CException&){
            failed = true;
        }
        NcbiCout << ( failed ? "PASS" : "FAIL" ) << NcbiEndl;
        if(!failed)
            m_Result = 1;
    }
    NcbiCout << "------------------------------------" << NcbiEndl;
}


ostream& operator<<(ostream& os, const CNcbiDiag &diag) 
{
    return os << diag.GetFile() << " " << diag.GetModule() << "::" 
              << diag.GetClass() << "::" << diag.GetFunction() << "()";
}

void CTest::x_TestString(const char* str, int expects[])
{
    NcbiCout << "Testing string \"" << str << "\" CDiagFilter:\n";

    CDiagFilter tester;
    tester.Fill(str);
    tester.Print(NcbiCout);

    NcbiCout << "-------\n";

    for (unsigned int i = 0;  i < m_Diags.size();  ++i ) {
        CNcbiOstrstream out;
        out << *m_Diags[i];

        NcbiCout << setw(45) << (string) CNcbiOstrstreamToString(out) << " "
                 << ( expects[i] ? "accept" : "reject" ) << " expected - ";
        if ( (tester.Check(*m_Diags[i]) == eDiagFilter_Accept)
             == (expects[i] != 0)) {
            NcbiCout << "PASS";
        } else {
            NcbiCout << "FAIL";
            m_Result = 2;
        }
        NcbiCout << NcbiEndl;
    }
    NcbiCout << "--------------------------\n";
}


void CTest::x_TestPathes()
{
    CNcbiDiag p1( CDiagCompileInfo("somewhere/cpp/src/corelib/file.cpp", 0, 
                                   NCBI_CURRENT_FUNCTION) );
    CNcbiDiag p2( CDiagCompileInfo("somewhere/include/corelib/file.cpp", 0, 
                                   NCBI_CURRENT_FUNCTION) );
    CNcbiDiag p3( CDiagCompileInfo("somewhere/cpp/src/corelib/int/file.cpp", 0, 
                                   NCBI_CURRENT_FUNCTION) );
    CNcbiDiag p4( CDiagCompileInfo("somewhere/include/corelib/int/file.cpp", 0,
                                   NCBI_CURRENT_FUNCTION) );
    CNcbiDiag p5( CDiagCompileInfo("somewhere/foo/corelib/file.cpp", 0,
                                   NCBI_CURRENT_FUNCTION) );

    m_Diags.push_back(&p1);
    m_Diags.push_back(&p2);
    m_Diags.push_back(&p3);
    m_Diags.push_back(&p4);
    m_Diags.push_back(&p5);

    NcbiCout << "Testing file paths" << NcbiEndl;
    {
        int expects[] = { 1, 1, 1, 1, 0 };
        x_TestString( "/corelib", expects );
    }
    {
        int expects[] = { 1, 1, 0, 0, 0 };
        x_TestString( "/corelib/", expects );
    }
    {
        int expects[] = { 0, 0, 1, 1, 0 };
        x_TestString( "/corelib/int", expects );
    }

    m_Diags.clear();
}

void CTest::x_TestFormats(void)
{
    const char* module1   = "module";
    const char* module2   = "foo";
    const char* class1    = "class";
    const char* function1 = "function";

    CNcbiDiag m0_c0_f0;
    CNcbiDiag m1_c0_f0;
    CNcbiDiag m2_c0_f0;
    CNcbiDiag m1_c1_f0;
    CNcbiDiag m0_c1_f0;
    CNcbiDiag m1_c0_f1;
    CNcbiDiag m0_c0_f1;
    CNcbiDiag m0_c1_f1;
    CNcbiDiag m1_c1_f1;

    m1_c0_f0 << MDiagModule(module1);
    m2_c0_f0 << MDiagModule(module2);
    m1_c1_f0 << MDiagModule(module1);
    m1_c0_f1 << MDiagModule(module1);
    m1_c1_f1 << MDiagModule(module1);

    m1_c1_f0 << MDiagClass(class1);
    m0_c1_f0 << MDiagClass(class1);
    m0_c1_f1 << MDiagClass(class1);
    m1_c1_f1 << MDiagClass(class1);

    m1_c0_f1 << MDiagFunction(function1);
    m0_c0_f1 << MDiagFunction(function1);
    m0_c1_f1 << MDiagFunction(function1);
    m1_c1_f1 << MDiagFunction(function1);

    m_Diags.push_back(&m0_c0_f0);
    m_Diags.push_back(&m1_c0_f0);
    m_Diags.push_back(&m2_c0_f0);
    m_Diags.push_back(&m1_c1_f0);
    m_Diags.push_back(&m0_c1_f0);
    m_Diags.push_back(&m1_c0_f1);
    m_Diags.push_back(&m0_c0_f1);
    m_Diags.push_back(&m0_c1_f1);
    m_Diags.push_back(&m1_c1_f1);


    NcbiCout << "Testing good format strings" << NcbiEndl;
    {
        int expects[] = { 1, 1, 1, 1, 1, 1, 1, 1, 1 };
        x_TestString( "", expects );
    }
    {
        int expects[] = { 0, 1, 0, 1, 0, 1, 0, 0, 1 };
        x_TestString( "module", expects );
    }
    {
        int expects[] = { 0, 1, 0, 1, 0, 1, 0, 0, 1 };
        x_TestString( "[Warning]module", expects );
    }

    {
        int expects[] = { 1, 0, 1, 0, 1, 0, 1, 1, 0 };
        x_TestString( "!module", expects );
    }
    {
        int expects[] = { 0, 1, 1, 1, 0, 1, 0, 0, 1 };
        x_TestString( "module foo", expects );
    }
    {
        int expects[] = { 1, 0, 0, 0, 1, 0, 1, 1, 0 };
        x_TestString( "?", expects );
    }
    {
        int expects[] = { 0, 0, 0, 1, 0, 0, 0, 0, 1 };
        x_TestString( "module::class", expects );
    }
    {
        int expects[] = { 1, 1, 1, 0, 1, 1, 1, 1, 0 };
        x_TestString( "!module::class", expects );
    }
    {
        int expects[] = { 0, 0, 0, 1, 1, 0, 0, 1, 1 };
        x_TestString( "::class", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 1, 0, 0, 1, 0 };
        x_TestString( "?::class", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 0, 0, 0, 0, 1 };
        x_TestString( "module::class::function", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 0, 0, 0, 0, 1 };
        x_TestString( "module::class::function()", expects );
    }
    {
        int expects[] = { 1, 1, 1, 1, 1, 1, 1, 1, 0 };
        x_TestString( "!module::class::function", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 0, 1, 0, 0, 1 };
        x_TestString( "module::function()", expects );
    }
    {
        int expects[] = { 1, 1, 1, 1, 1, 0, 1, 1, 0 };
        x_TestString( "!module::function()", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 0, 1, 1, 1, 1 };
        x_TestString( "function()", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 0, 0, 1, 1, 0 };
        x_TestString( "?::function()", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 0, 0, 0, 1, 1 };
        x_TestString( "::class::function()", expects );
    }
    {
        int expects[] = { 0, 0, 0, 0, 0, 1, 1, 0, 0 };
        x_TestString( "::?::function()", expects );
    }
    m_Diags.clear();
}


CNcbiDiag *YesDefineFromHeader();
CNcbiDiag *YesDefineFromCpp();
CNcbiDiag *NoDefineFromHeader();
CNcbiDiag *NoDefineFromCpp();

void CTest::x_CheckDiag(CNcbiDiag *   diag, 
                        const string& desc, 
                        bool          module_expected)
{
    NcbiCout << setw(64) << string("Test \"") + desc 
        + "\" module name" + ( module_expected ? "" : " not" ) 
        + " expected - ";

    if ( string(diag->GetModule()).empty() == module_expected ) {
        NcbiCout << "FAIL";
        m_Result = 3;
    }
    else NcbiCout << "PASS";
    NcbiCout << NcbiEndl;
        
    delete diag;
}

void CTest::x_TestDiagCompileInfo()
{
    NcbiCout << "Testing DIAG_COMPILE_INFO\n";

    x_CheckDiag(YesDefineFromHeader(), "Defined from the header",     false);
    x_CheckDiag(YesDefineFromCpp(),    "Defined from the source",     true );
    x_CheckDiag(NoDefineFromHeader(),  "Not defined from the header", false);
    x_CheckDiag(NoDefineFromCpp(),     "Not defined from the header", false);

    NcbiCout << "--------------------------\n";
}

void CTest::x_TestSeverity()
{
    SetDiagPostLevel(eDiag_Info);
    SetDiagFilter(eDiagFilter_All, "[Error]module [Info]!module");

    LOG_POST(Warning << MDiagModule("module")  << "Test error 1");
    LOG_POST(Error   << MDiagModule("module")  << "Test error 2");
    LOG_POST(Warning << MDiagModule("module2") << "Test error 3");
    LOG_POST(Info    << MDiagModule("module3") << "Test error 4");
}

void CTest::x_TestErrCode()
{
    NcbiCout << "Testing ErrCodes\n";

    CNcbiDiag p1( CDiagCompileInfo("", 0, NCBI_CURRENT_FUNCTION) );
    p1.SetErrorCode(-6,-10);
    m_Diags.push_back(&p1);

    CNcbiDiag p2( CDiagCompileInfo("", 0, NCBI_CURRENT_FUNCTION) );
    p2.SetErrorCode(-6,10);
    m_Diags.push_back(&p2);

    CNcbiDiag p3( CDiagCompileInfo("", 0, NCBI_CURRENT_FUNCTION) );
    p3.SetErrorCode(20,1);
    m_Diags.push_back(&p3);

    CNcbiDiag p4( CDiagCompileInfo("", 0, NCBI_CURRENT_FUNCTION) );
    p4.SetErrorCode(21,1);
    m_Diags.push_back(&p4);
 
    CNcbiDiag p5( CDiagCompileInfo("", 0, NCBI_CURRENT_FUNCTION) );
    p5.SetErrorCode(31,21);
    m_Diags.push_back(&p5);

    CNcbiDiag p6( CDiagCompileInfo("", 0, NCBI_CURRENT_FUNCTION) );
    p6.SetErrorCode(40,1);
    m_Diags.push_back(&p6);
   
    {
        int expects[] = { 1, 1, 1, 0, 0, 1 };
        x_TestString("(-10--5,20,30-40.) !(31.1-100)",expects);
    }

    {
        int expects[] = { 1, 0, 0, 1, 1, 0 };
        x_TestString("!(-10--5,20,30-40.1-20)",expects);
    }

    {
        int expects[] = { 1, 0, 0, 1, 1, 0 };
        x_TestString("!(20.) !(30-40.1-20) !(-6.1-100)",expects);
    }
    m_Diags.clear();
}

int CTest::Run(void)
{
    x_TestBadFormats();
    x_TestPathes();
    x_TestFormats();
    x_TestDiagCompileInfo();
    x_TestSeverity();
    x_TestErrCode();

    NcbiCout << "**********************************************************************" 
             << NcbiEndl;
    NcbiCout << "                                                          " 
             << "TEST: " << ( m_Result == 0 ? "PASS" : "FAIL" ) << NcbiEndl;
    NcbiCout << "**********************************************************************" 
             << NcbiEndl;
    return m_Result;
}



/////////////////////////////////////////////////////////////////////////////
//  MAIN

int main(int argc, const char* argv[]) 
{
    return CTest().AppMain(argc, argv);
}
