#ifndef GUI_WIDGETS_SEQ_GRAPHIC___GRAPH_CACHE__HPP
#define GUI_WIDGETS_SEQ_GRAPHIC___GRAPH_CACHE__HPP

/*  
 * ===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 * Authors:  Andrei Shkeda
 *
 * File Description:
 *
 */

#include <corelib/obj_pool.hpp>
#include <util/cache/icache.hpp>
#include <connect/services/neticache_client.hpp>
#include <corelib/ncbidiag.hpp>
#include <corelib/request_ctx.hpp>

#include <thread>
#include <atomic>
#include <mutex>
#include <queue>
#include <condition_variable>

BEGIN_NCBI_SCOPE

///< Storage factory provides ICache instance
///< for permanent storage of cached data 
class CGraphCacheFactory 
{
public:
    CGraphCacheFactory(const string& param)
        : m_Param(param)
    {
    }
    ICache* CreateObject(void) {
        CNcbiRegistry& reg = CNcbiApplication::Instance()->GetConfig();
        string nc_service = reg.GetString(m_Param, "service", "NC_SV_GRAPH_DEV");
        string cache_name = reg.GetString(m_Param, "cache", "graph");
        string client = reg.GetString(m_Param, "client", "sviewer");
        return new CNetICacheClient(nc_service, cache_name, client);
    }
        
    void DeleteObject(ICache* obj) {
    }
private:
    string m_Param;
};

using TGraphCachePool = CObjPool<ICache,CGraphCacheFactory>;

/////////////////////////////////////////////////////////////////////////////
///
///  CGraphCache --
///
/// in-memeory cache for graph data 
/// Provides concurrent access and asynchroneous storage operation for TData
///
/// TData is expected to provide:
/// Init() - method to be called to initialize data
/// Save() - method to serialize data
///
/// ICache storage is not used by default
/// the caller needs to invoke EnableICache() to enable it
///
/// if ICache is enabled GraphCache start a backround thread (m_Thread) on first save request
/// The consecutive save request sends a copy of data to m_Queue for asynchroneous Save. 


template<class TData>
class CGraphCache : public CObject
{
public:
    
    static CGraphCache& GetInstance()
    {
        static CSafeStatic<CGraphCache> s_Cache;
        return s_Cache.Get();
    }
        
    virtual ~CGraphCache() {
        try {
            Stop();
        } catch (exception& e) {
            ERR_POST(Error << e.what());
        }
    }
    
    void Stop() {
        if (m_Thread.joinable()) {
            m_StopRequested = true;
            m_CV.notify_all();
            m_Thread.join();
        }
    }

    /// Retrieves TData from in-memory cache 
    /// @param data_key
    ///    memory access key generated by TData
    /// @return
    ///    instance of TData or NULL if there are no cached data
    CRef<TData> GetCachedData(const string& data_key)
    {
        CRef<TData> data;
        {
            lock_guard<mutex> lock(m_Lock);
            if (m_Cache.count(data_key) != 0)
                data = m_Cache[data_key];
        }
        return data;
    }
    
    /// Retrieves TData from in-memory cache or if not found calls TData::Init to initialize new TData instance
    /// @param data_key
    ///     memory access key generated by TData
    /// @return
    ///     instance of TData or NULL if there are no cached data and TData does not have ICache 
    CRef<TData> GetData(const string& data_key)
    {
        CRef<TData> data;
        {
            lock_guard<mutex> lock(m_Lock);
            if (m_Cache.count(data_key) > 0) 
                data = m_Cache[data_key];
            else {
                data = Ref(new TData(data_key));
                m_Cache[data_key] = data;
            }
        }
        if (m_HasICache)
            data->Init(m_Storage);
        return data;
    }


    /// Removes Data from in-memory cache 
    /// @param data_key
    ///     memory access key generated by TData
    void RemoveData(const string& data_key)
    {
        lock_guard<mutex> lock(m_Lock);
        auto it = m_Cache.find(data_key);
        if (it != m_Cache.end()) {
            m_Cache.erase(it);
        }
    }
    
    /// Clones TData, puts it into Save Queue for asynchroneous storage operation 
    /// @param data 
    ///     Instance of TData
    void SaveData(CRef<TData> data)
    {
        if (!data)
            return;
        if (!m_HasICache)
            return;
        
        {
            if (!m_Thread.joinable()) {
                CRef<CRequestContext> context(CDiagContext::GetRequestContext().Clone());
                // start worker thread 
                m_Thread = move(thread(&CGraphCache::x_Run, this, context));
            }
            lock_guard<mutex> lock(m_Lock);
            m_Queue.emplace(new TData(*data));
        }
        m_CV.notify_all();
    }

    void EnableICache(bool value) {
        m_HasICache = value;
    }
private:
    CGraphCache() :
        m_Storage(CGraphCacheFactory(TData::GetStorageName()))
    {
    }
    friend class CSafeStatic_Allocator<CGraphCache>;
    CGraphCache(const CGraphCache&)  = delete;
    CGraphCache& operator=(const CGraphCache&)  = delete;

private:
    void x_Run(CRef<CRequestContext> context) {
        if (!context)
            CDiagContext::SetRequestContext(context);
        while (true) {
            {
                unique_lock<mutex> lock(m_Lock);
                m_CV.wait(lock, [&] { return m_StopRequested || !m_Queue.empty();});
            }
            if (m_StopRequested)
                break;
            CRef<TData> data;
            {
                lock_guard<mutex> lock(m_Lock);
                data = m_Queue.front();
                m_Queue.pop();
                
            }
            data->Save(m_Storage);
        }
    }
        
    mutex               m_Lock; ///< Saving Queue and Cache lock

    queue<CRef<TData>>  m_Queue; ///< Saving Queue for storage operations

    bool                m_HasICache = false; ///< Flag indicates if data should be saved into ICache

    using TDataKey = string;
    map<TDataKey, CRef<TData>> m_Cache;  ///< in-memory data cache 

    TGraphCachePool    m_Storage; ///< ICache pool
    
    condition_variable m_CV;      ///< wake-up signal for save requests
    thread             m_Thread;  ///< background thread for save processing
    atomic<bool>       m_StopRequested{false}; ///< flag to stop the backround thread
    
};




END_NCBI_SCOPE

#endif  // GUI_WIDGETS_SEQ_GRAPHIC___GRAPH_CACHE__HPP
