/*
 * protein_grouping_result.cpp
 *
 *  Created on: 25 mars 2013
 *      Author: valot
 */

#include "protein_grouping_result.h"
#include <QDebug>
#include <cmath>

ProteinGroupingResult::ProteinGroupingResult(GpParams * params) :
		peptideId(0) {
	_writer = new QXmlStreamWriter(params->getOutputFile());
	_writer->setAutoFormatting(true);
	this->printHeader();
	this->printParams(params);
}

ProteinGroupingResult::~ProteinGroupingResult() {
	qDebug() << "Delete writer";
	this->printFooter();
	delete (_writer);
	_sampleId.clear();
	_proteinId.clear();
}

void ProteinGroupingResult::printHeader() {
	qDebug() << "Print Header";
	_writer->writeStartDocument();
	_writer->writeStartElement("grouping_protein");
}

void ProteinGroupingResult::printParams(GpParams * params) {
	qDebug() << "Print params";
	_writer->writeStartElement("params");
	_writer->writeEmptyElement("peptide_reproductibility");
	_writer->writeAttribute("value", QString().setNum(params->getPepRepro()));
	_writer->writeEmptyElement("peptide_by_prot");
	_writer->writeAttribute("value", QString().setNum(params->getPepByProt()));
	_writer->writeEmptyElement("remove_database");
	if (params->isFastaFile())
		_writer->writeAttribute("value", params->getFastaFile()->fileName());
	//End params element
	_writer->writeEndElement();
}

void ProteinGroupingResult::printGroupingResult(GpEngine * engine) {
	qDebug() << "Begin print GpEngine result";
	this->printSamples(engine);
	this->printProteins(engine);
	this->printGroups(engine->getGroupSet());
}

void ProteinGroupingResult::printSamples(GpEngine * engine) {
	qDebug() << "Print Samples";
	_writer->writeStartElement("sample_list");
	std::map<QString, Sample *>::const_iterator samples;
	std::vector<QString>::const_iterator files;
	Sample * samp;
	QString id;
	int sampleCount = 1;
	for (samples = engine->getSamplesBegin();
			samples != engine->getSamplesEnd(); samples++) {
		samp = samples->second;
		id.clear();
		id.append("samp").append(QString().setNum(sampleCount));
		_writer->writeStartElement("sample");
		_writer->writeAttribute("id", id);
		_writer->writeAttribute("name", samp->getName());
		for (files = samp->getFiles().begin(); files != samp->getFiles().end();
				files++) {
			_writer->writeEmptyElement("xtandem");
			_writer->writeAttribute("file", (*files));
		}
		//end Sample
		_writer->writeEndElement();
		//Store id
		_sampleId.insert(std::pair<Sample *, QString>(samp, id));
		sampleCount++;
	}
	//end Sample_list
	_writer->writeEndElement();
}

void ProteinGroupingResult::printProteins(GpEngine * engine) {
	qDebug() << "Print Proteins";
	_writer->writeStartElement("protein_list");
	std::vector<const Group *>::const_iterator groups;
	std::vector<SubGroup *>::const_iterator subgroups;
	std::vector<const ProteinMatch *>::const_iterator prots;
	const Group * group;
	SubGroup * subgroup;
	const ProteinMatch * prot;
	QString id;
	int protCount, groupcount, subgroupcount;
	groupcount = 1;
	for (groups = engine->getGroupSet().getOrderedGroups().begin();
			groups != engine->getGroupSet().getOrderedGroups().end(); groups++) {
		group = *groups;
		subgroupcount = 1;
		for (subgroups = group->getSubGroupSet().getOrderedSubgroups().begin();
				subgroups != group->getSubGroupSet().getOrderedSubgroups().end();
				subgroups++) {
			subgroup = *subgroups;
			protCount = 1;
			//SubGroupId
			id.clear();
			id.append(this->getId(groupcount));
			id.append(".").append(this->getId(subgroupcount));
			_subgroupId.insert(std::pair<SubGroup *, QString>(subgroup, id));
			for (prots = subgroup->getOrderedProteinMatchs().begin();
					prots != subgroup->getOrderedProteinMatchs().end();
					prots++) {
				prot = *prots;
				id.clear();
				id.append(this->getId(groupcount));
				id.append(".").append(this->getId(subgroupcount));
				id.append(".").append(this->getId(protCount));
				_writer->writeEmptyElement("protein");
				_writer->writeAttribute("id", id);
				_writer->writeAttribute("access", prot->getAccession());
				_writer->writeAttribute("desc", prot->getDescription());
				//Store id
				_proteinId.insert(std::pair<const ProteinMatch *, QString>(prot, id));
				protCount++;
			}
			subgroupcount++;
		}
		groupcount++;
	}
	//end of prot_list
	_writer->writeEndElement();
}

void ProteinGroupingResult::printGroups(const GroupSet & groupSet) {
	_writer->writeStartElement("group_list");
	std::vector<const Group *>::const_iterator groups;
	const Group * group;
	int groupcount;
	groupcount = 1;
	for (groups = groupSet.getOrderedGroups().begin(); groups != groupSet.getOrderedGroups().end();
			groups++) {
		group = *groups;
		this->printGroup(group, groupcount);
		groupcount++;
	}
	//end of group_list
	_writer->writeEndElement();
}

void ProteinGroupingResult::printGroup(const Group * group, int groupid) {
	_writer->writeStartElement("group");
	_writer->writeAttribute("id", this->getId(groupid));
	_writer->writeAttribute("number", QString().setNum(groupid));
	_writer->writeAttribute("count",
			QString().setNum(group->getPeptideSet().getTotalSpectra()));
	_writer->writeStartElement("subgroup_list");

	std::vector<SubGroup *>::const_iterator subgroups;
	std::vector<const ProteinMatch *>::const_iterator prots;
	SubGroup * subgroup;
	const ProteinMatch * prot;
	int subgroupcount = 1;
	QString protIds;
	for (subgroups = group->getSubGroupSet().getOrderedSubgroups().begin();
			subgroups != group->getSubGroupSet().getOrderedSubgroups().end();
			subgroups++) {
		subgroup = *subgroups;
		_writer->writeEmptyElement("subgroup");
		_writer->writeAttribute("id", _subgroupId.find(subgroup)->second);
		_writer->writeAttribute("count",
				QString().setNum(subgroup->getPeptideSet().getTotalSpectra()));
		_writer->writeAttribute("number", QString().setNum(subgroupcount));

		//ProtIds
		protIds.clear();
		for (prots = subgroup->getOrderedProteinMatchs().begin();
				prots != subgroup->getOrderedProteinMatchs().end(); prots++) {
			prot = *prots;
			protIds.append(_proteinId.find(prot)->second);
			protIds.append(" ");
		}
		_writer->writeAttribute("protIds", protIds.trimmed());
		subgroupcount++;
	}
	//end of subgroupList
	_writer->writeEndElement();
	//print PeptideList of this group
	this->printPeptideList(group);
	//end of group
	_writer->writeEndElement();
}

void ProteinGroupingResult::printPeptideList(const Group * group) {
	_writer->writeStartElement("peptide_list");
	std::set<Peptide *>::const_iterator pepIt;
	std::vector<SubGroup *>::const_iterator subIt;
	std::map<QString, int>::const_iterator seqIt;
	std::map<Sample*, std::map<int, int> >::const_iterator sampleIt;
	std::map<int, int>::const_iterator scansIt;
	Peptide * pep;
	SubGroup * sub;
	QString subgroupIds;
	for (pepIt = group->getPeptideSet().getList().begin();
			pepIt != group->getPeptideSet().getList().end(); pepIt++) {
		peptideId++;
		pep = *pepIt;
		_writer->writeStartElement("peptide");
		_writer->writeAttribute("id", "pep." + this->getId(peptideId));
		_writer->writeAttribute("seqLI", pep->getSequenceLi());
		_writer->writeAttribute("mhTheo",
				QString().setNum(pep->getMh(), 'g', 8));
		//SubGroupIDs
		subgroupIds.clear();
		for (subIt = group->getSubGroupSet().getOrderedSubgroups().begin();
				subIt != group->getSubGroupSet().getOrderedSubgroups().end();
				subIt++) {
			sub = *subIt;
			if (sub->getPeptideSet().contains(pep))
				subgroupIds.append(_subgroupId.find(sub)->second).append(" ");
		}
		_writer->writeAttribute("subgroupIds", subgroupIds.trimmed());
		//Sequence Modifs
		_writer->writeStartElement("sequence_modifs_list");
		for (seqIt = pep->getSequenceModifsBegin();
				seqIt != pep->getSequenceModifsEnd(); seqIt++) {
			_writer->writeEmptyElement("sequence_modifs");
			QStringList list = seqIt->first.split("|");
			_writer->writeAttribute("seq", list.value(0));
			_writer->writeAttribute("mods", list.value(1));
			_writer->writeAttribute("count", QString().setNum(seqIt->second));
		}
		//end sequence modif list$
		_writer->writeEndElement();
		//spectra
		_writer->writeStartElement("spectra");
		for (sampleIt = pep->getHashSampleScanBegin();
				sampleIt != pep->getHashSampleScanEnd(); sampleIt++) {
			QString sampId = _sampleId.find(sampleIt->first)->second;
			for (scansIt = sampleIt->second.begin();
					scansIt != sampleIt->second.end(); scansIt++) {
				_writer->writeEmptyElement("spectrum");
				_writer->writeAttribute("sample", sampId);
				_writer->writeAttribute("scan",
						QString().setNum(scansIt->first));
				_writer->writeAttribute("charge",
						QString().setNum(scansIt->second));
			}
		}
		//end spectra
		_writer->writeEndElement();
		//end of Peptide
		_writer->writeEndElement();
	}
	//end of peptideList
	_writer->writeEndElement();
}

void ProteinGroupingResult::printFooter() {
//End of protein_grouping element
	_writer->writeEndElement();
	_writer->writeEndDocument();
}

QString ProteinGroupingResult::getId(int num) {
	int size = log10(num);
	size += 97;
	QString base(size);
	base.append(QString().setNum(num));
	return (base);
}
