/*
 *
 * Protein Grouper
 * Copyright (C) 2014 Olivier Langella, Benoit Valot, Michel Zivy.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *

 * threadsafe_group_set.cpp
 *
 *  Created on: 22 mars 2013
 *      Author: valot
 */

#include "threadsafe_group_set.h"
#include <QDebug>
#include <QtConcurrentMap>
#include "group.h"

#include <cstdio>
#include <iostream>

TsGroupSet::TsGroupSet() {
    tagFusioningGroups_f = std::bind(&TsGroupSet::tagFusioningGroups, this, std::placeholders::_1);
}

TsGroupSet::~TsGroupSet() {
  qDebug()  << "TsGroupSet::~TsGroupSet ";
  _mapPeptide2group.clear();
}


void TsGroupSet::deindexGroup(Group* p_group) {
    for (std::set<Peptide*>::const_iterator it = p_group->getPeptideSet().getList().begin();
            it != p_group->getPeptideSet().getList().end(); it++) {
        _mapPeptide2group.erase(*it);
    }
}
void TsGroupSet::indexGroup(Group* p_group) {
    for (std::set<Peptide*>::const_iterator it = p_group->getPeptideSet().getList().begin();
            it != p_group->getPeptideSet().getList().end(); it++) {
        _mapPeptide2group.insert(std::pair<const Peptide *,Group *>(*it, p_group));
    }
}

void TsGroupSet::mapFusioningGroups(Group* p_fromgroup, Group* p_togroup) {
    this->_mutex.lock();
    QMap<Group*,QList<Group *>>::iterator it = _map_fusioning_groups.find(p_fromgroup);
    if (it == _map_fusioning_groups.end()) {
        //nothing in map for this key
        QList<Group *> listGroup;
        listGroup.append(p_togroup);
        this->_map_fusioning_groups.insert(p_fromgroup,listGroup);
    }
    else {
        it.value().append(p_togroup);
    }
    this->_mutex.unlock();
}

void TsGroupSet::tagFusioningGroups(Group* & p_incoming_group) {
    //look in all the group list to find common peptides
    //if there are common peptides :
    // the group must be tagged for a differed fusion
    std::set<Group*> groupsToFusion;
    for (std::set<Peptide*>::const_iterator it =p_incoming_group->getPeptideSet().getList().begin(); it != p_incoming_group->getPeptideSet().getList().end(); it++) {
        std::map<const Peptide*,Group*>::const_iterator itMapGroupToFusion = _mapPeptide2group.find(*it);
        if (itMapGroupToFusion != _mapPeptide2group.end()) {
            groupsToFusion.insert(itMapGroupToFusion->second);
        }
    }
    
    //fusioning now :


    std::set<Group *>::const_iterator it;
    for (it = groupsToFusion.begin(); it != groupsToFusion.end(); ++it) {
            //the current must be tagged for later fusion
            this->mapFusioningGroups(p_incoming_group, *it);
    }
}


void TsGroupSet::groupingSubGroupList(const QList<SubGroup *> & qlSubgroups) {

    QList<Group *> qlGroups;

    //1) convert subgroups to groups and add it to the main group list:
    for (QList<SubGroup *>::const_iterator it = qlSubgroups.begin(); it != qlSubgroups.end(); it++) {
        Group * group = new Group(*it);
        qlGroups.append(group);
	this->indexGroup(group);
    }
    qDebug()  << "groupingSubGroupList : inserting " << qlSubgroups.size() << " groups ";
    this->_groupSet.insert(_groupSet.end(), qlGroups.begin(), qlGroups.end());

//2) launch concurrent map on qlGroups to add to identify fusioning groups
    qDebug()  << "launching fusion tags on "<< qlGroups.size() ;
    QFuture<void> futur;
    futur = QtConcurrent::map(qlGroups, tagFusioningGroups_f);
    futur.waitForFinished();
    qDebug()  << "fusion tags finished " ;

    //3) we know which groups has to fusion with, let's do it
    //keep in mind group pointer tracability :
    qDebug()  << "fusioning groups";

    QMap<Group*,Group *> _map_group_to_newgroup;
    for ( QMap<Group*,QList<Group *>>::iterator itFusionMap = _map_fusioning_groups.begin(); itFusionMap != _map_fusioning_groups.end();  ++itFusionMap) {
        Group * eaterGroup = itFusionMap.key();
        //if the eaterGroup was already eated :
        QMap<Group*,Group *>::iterator itTrGroup = _map_group_to_newgroup.find(eaterGroup);
        if (itTrGroup != _map_group_to_newgroup.end()) {
            eaterGroup = *itTrGroup;
        }
        qDebug()  << " map of eater group " << itFusionMap.key() << "eater  group" << eaterGroup;
        for (QList<Group *>::iterator itEatedGroups = itFusionMap.value().begin(); itEatedGroups != itFusionMap.value().end(); ++itEatedGroups) {
            qDebug()  << "eated  group" << *itEatedGroups;
            if (*itEatedGroups != itFusionMap.key()) {
                if (*itEatedGroups != eaterGroup) {
                    qDebug()  << "eat group";
                    eaterGroup->eatGroup(*itEatedGroups);
                    _map_group_to_newgroup.insert(*itEatedGroups, eaterGroup);
                }
            }
        }
    }
    _map_fusioning_groups.clear();
    //4) remove eated groups
    qDebug()  << "removing eated groups " <<_groupSet.size()  ;
    for (QMap<Group*,Group *>::iterator itEatMapGroup = _map_group_to_newgroup.begin(); itEatMapGroup != _map_group_to_newgroup.end();  ++itEatMapGroup) {
        _groupSet.remove(itEatMapGroup.key());
	deindexGroup(itEatMapGroup.key());
        qlGroups.removeAll(itEatMapGroup.key());
    }
    //5) remove non informative redundancy from eater groups
    /* qDebug()  << "removing non informative redundancy in eater groups" <<_groupSet.size();
     for (QList<Group *>::iterator itGroup= qlGroups.begin(); itGroup!=qlGroups.end(); ++itGroup) {
         (*itGroup)->removeNonInformative();
     }*/
}

