/*
 *
 * Protein Grouper
 * Copyright (C) 2014 Olivier Langella, Benoit Valot, Michel Zivy.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *

 * sub_group_set.cpp
 *
 *  Created on: 22 mars 2013
 *      Author: valot
 */

#include "sub_group_set.h"
#include "../gp_error.h"
#include <QDebug>

SubGroupSet::SubGroupSet() {

}

SubGroupSet::~SubGroupSet() {
    qDebug()<<"SubGroupSet::~SubGroupSet() HARD DESTROY " << _subgroups.size();
    _subgroupOrder.clear();
    std::list<SubGroup *>::iterator it;
    SubGroup * sg;
    for(it=_subgroups.begin(); it!=_subgroups.end(); it++) {
        sg = *it;
        delete(sg);
    }
    _subgroups.clear();
}


void SubGroupSet::deindexPeptide2SubGroup(const SubGroup * p_sg) {
    for (std::set<Peptide*>::const_iterator it = p_sg->getPeptideSet().getList().begin();
            it != p_sg->getPeptideSet().getList().end(); it++) {
        _mapPeptide2subGroup.erase(*it);
    }
}

bool SubGroupSet::indexPeptide2SubGroup(SubGroup * p_sg) {
    bool newPeptide = false;
    for (std::set<Peptide*>::const_iterator it = p_sg->getPeptideSet().getList().begin();
            it != p_sg->getPeptideSet().getList().end(); it++) {
        std::map<const Peptide*,std::set<SubGroup*>>::iterator itMapPeptideSg = _mapPeptide2subGroup.find(*it);
        if (itMapPeptideSg == _mapPeptide2subGroup.end()) {
            std::set<SubGroup*> sgSet;
            sgSet.insert(p_sg);
            _mapPeptide2subGroup.insert(std::pair<const Peptide *,std::set<SubGroup*>>(*it, sgSet));
            newPeptide = true;
        } else {
            itMapPeptideSg->second.insert(p_sg);
        }
    }
    return newPeptide;
}

void SubGroupSet::fillSgSet(std::set<SubGroup *> & subGroupListCheck, const std::set<Peptide *> & peptideList) {
    for (std::set<Peptide*>::const_iterator it = peptideList.begin();
            it != peptideList.end(); it++) {
        std::map<const Peptide*,std::set<SubGroup*>>::iterator itMapPeptideSg = _mapPeptide2subGroup.find(*it);
        if (itMapPeptideSg != _mapPeptide2subGroup.end()) {
            subGroupListCheck.insert(itMapPeptideSg->second.begin(), itMapPeptideSg->second.end());
        }
    }
}

void SubGroupSet::addProteinMatch(const ProteinMatch * proteinMatch) {
    bool createNewSubGroup = true;
    SubGroup * sg;

    std::set<SubGroup *>::iterator subgroupIt;
    std::set<SubGroup *> subGroupListCheck;

    this->fillSgSet(subGroupListCheck, proteinMatch->getPeptideSet().getList());

    qDebug()<<"SubGroupSet::addProteinMatch begin " << _subgroups.size() << " " << subGroupListCheck.size();

    for(subgroupIt = subGroupListCheck.begin(); subgroupIt!=subGroupListCheck.end(); subgroupIt++) {
        sg = *subgroupIt;
        if(sg->addProteinMatch(proteinMatch)) {
            indexPeptide2SubGroup(sg);
            createNewSubGroup = false;
            break;
        }
    }
    if (createNewSubGroup) {
        qDebug()<<"SubGroupSet::addProteinMatch totally new sg ";
        sg = new SubGroup(proteinMatch);
        _subgroups.push_back(sg);
        indexPeptide2SubGroup(sg);
    }
    qDebug()<<"SubGroupSet::addProteinMatch end ";
}

void SubGroupSet::addSubGroup(SubGroup * subGroup) {
    _subgroups.push_back(subGroup);
}

void SubGroupSet::removeSubGroup(SubGroup * subGroup) {
    qDebug()<<"Remove non informative subgroup "+ subGroup->getFirstAccession();
    _subgroups.remove(subGroup);
}

QString SubGroupSet::getFirstAccession() const {
    if (_subgroups.empty()) {
        throw new GpError("something is wrong : this subgroupset has no subgroup");
    }
    QString access, temp;
    std::list<SubGroup *>::const_iterator subgroupIt;
    SubGroup * sg;
    for(subgroupIt = _subgroups.begin(); subgroupIt!=_subgroups.end(); subgroupIt++) {
        sg = *subgroupIt;
        temp = sg->getFirstAccession();
        if(access.isEmpty())
            access=temp;
        else if(temp<access)
            access=temp;
    }
    return access;
}

void SubGroupSet::indexAndOrderSubGroup() {
    _subgroupOrder.clear();

    std::map<const QString, std::map<const QString, SubGroup *> > indexMap;
    for (std::list<SubGroup *>::iterator it = _subgroups.begin(); it != _subgroups.end(); it++) {
        indexMap[(*it)->getOrderString()].insert(std::pair<QString, SubGroup *>((*it)->getFirstAccession(), (*it)));
    }

    qDebug() << "SubGroupSet::indexAndOrderSubGroup Indexing";
    _subgroupOrder.reserve(_subgroups.size());
    for (std::map<const QString, std::map<const QString, SubGroup *> >::reverse_iterator itm = indexMap.rbegin(); itm != indexMap.rend(); itm++) {
        //qDebug() << "GroupSet::indexAndOrderGroup getFirstAccession "<< accessIt->first;
        for (std::map<const QString, SubGroup *>::iterator it = itm->second.begin(); it != itm->second.end(); it++) {
            _subgroupOrder.push_back(it->second);

            it->second->indexAndOrderProteinMatch();
        }
    }

}

const std::list<SubGroup *> & SubGroupSet::getSubgroups() const {
    if (_subgroups.empty()) {
       // throw new GpError("subgroup list is empty in the current subgroupset");
    }
    return _subgroups;
}



const std::vector<SubGroup *> & SubGroupSet::getOrderedSubgroups() const {
    if (_subgroupOrder.empty()) {
        throw new GpError("ordered subgroup list is empty. use SubGroupSet::indexAndOrderSubGroup() first.");
    }
    return _subgroupOrder;
}
