/*
* Protein Grouper
* Copyright (C) 2014 Olivier Langella, Benoit Valot, Michel Zivy.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "groupng.h"

#include <iostream>
#include "../gp_error.h"

GroupNg::GroupNg()
{

}

GroupNg::GroupNg(std::set<SubGroup*> & alreadyLinkedSg) {
    for (std::set<SubGroup *>::iterator itSg = alreadyLinkedSg.begin(); itSg != alreadyLinkedSg.end(); ++itSg) {
        this->push_back_subgroup(*itSg);
    }
}


GroupNg::~GroupNg()
{
    _mapPeptide2subGroup.clear();
}

void GroupNg::deindexPeptide2SubGroup(SubGroup * p_sg) {
    for (std::set<Peptide*>::const_iterator it = p_sg->getPeptideSet().getList().begin();
            it != p_sg->getPeptideSet().getList().end(); it++) {
        std::map<const Peptide*,std::set<SubGroup*>>::iterator itMapPeptideSg = _mapPeptide2subGroup.find(*it);
        if (itMapPeptideSg == _mapPeptide2subGroup.end()) {
            throw GpError(QObject::tr("deindexPeptide2SubGroup error : subgroup to deindex is not indexed"));
        } else {
            //qDebug() << "GroupNg::deindexPeptide2SubGroup pep "<< (*it)->getSequenceLi();
            itMapPeptideSg->second.erase(p_sg);
            if (itMapPeptideSg->second.size() == 0) {
                throw GpError(QObject::tr("deindexPeptide2SubGroup error : no more occurence of a peptide"));
            }
        }
    }
}

bool GroupNg::indexPeptide2SubGroup(SubGroup * p_sg) {
    bool newPeptide = false;
    for (std::set<Peptide*>::const_iterator it = p_sg->getPeptideSet().getList().begin();
            it != p_sg->getPeptideSet().getList().end(); it++) {
        std::map<const Peptide*,std::set<SubGroup*>>::iterator itMapPeptideSg = _mapPeptide2subGroup.find(*it);
        if (itMapPeptideSg == _mapPeptide2subGroup.end()) {
            std::set<SubGroup*> sgSet;
            sgSet.insert(p_sg);
            _mapPeptide2subGroup.insert(std::pair<const Peptide *,std::set<SubGroup*>>(*it, sgSet));
            newPeptide =  true;
        } else {
            itMapPeptideSg->second.insert(p_sg);
        }
    }
    return newPeptide;
}


void GroupNg::push_back_subgroup(SubGroup * subGroup) {
    qDebug() << "GroupNg::push_back_subgroup begin ";
    _peptideSet.addAll(subGroup->getPeptideSet());
    _subGroupSet.addSubGroup(subGroup);

    indexPeptide2SubGroup(subGroup);
    qDebug() << "GroupNg::push_back_subgroup end ";
}


void GroupNg::removeSubGroup(SubGroup * subGroup) {
    _subGroupSet.removeSubGroup(subGroup);
    deindexPeptide2SubGroup(subGroup);
}


bool GroupNg::contains(const SubGroup* p_sg) const {
    qDebug() << "GroupNg::contains begin ";
    std::list<SubGroup *>::const_iterator it;
    for(it = _subGroupSet.getSubgroups().begin(); it != _subGroupSet.getSubgroups().end(); it++) {
        if (p_sg == *it) {
            qDebug() << "GroupNg::contains end true ";
            return true;
        }
    }
    qDebug() << "GroupNg::contains end false ";
    return false;
}

bool GroupNg::contains(const Peptide* p_sg) const {
    std::map<const Peptide*,std::set<SubGroup*>>::const_iterator itMapPeptideSg = _mapPeptide2subGroup.find(p_sg);
    if (itMapPeptideSg == _mapPeptide2subGroup.end()) {
        return false;
    }
    return true;
}

bool GroupNg::makeItGrow(std::set<SubGroup*> & alreadyLinkedSg)  {
    qDebug() << "GroupNg::makeItGrow begin ";
    uint initialSize = alreadyLinkedSg.size();

    std::set<SubGroup *>::const_iterator itSg;
    std::set< Peptide* >::const_iterator itPep;
    for (itSg = alreadyLinkedSg.begin(); itSg != alreadyLinkedSg.end() ; itSg++) {
        for (itPep = (*itSg)->getPeptideSet().getList().begin(); itPep != (*itSg)->getPeptideSet().getList().end(); ++itPep) {
            std::map<const Peptide*,std::set<SubGroup*>>::iterator itMapPeptideSg = _mapPeptide2subGroup.find(*itPep);
            if (itMapPeptideSg == _mapPeptide2subGroup.end()) {
                throw new GpError("GroupNg::makeItGrow ERROR : pep not found in map");
            }
            alreadyLinkedSg.insert((*itMapPeptideSg).second.begin(),(*itMapPeptideSg).second.end());
        }
    }


    if (alreadyLinkedSg.size() > initialSize) {
        return true;
    }
    else {
        return false;
    }
}

bool GroupNg::pep1pep2sharingSubgroup(std::set<SubGroup*> & alreadyLinkedSg, const Peptide* p_pep1, const Peptide* p_pep2)  {
    std::map<const Peptide*,std::set<SubGroup*>>::iterator itMapPeptideSg = _mapPeptide2subGroup.find(p_pep1);
    if (itMapPeptideSg == _mapPeptide2subGroup.end()) {
        throw new GpError("GroupNg::pep1pep2sharingSubgroup ERROR : pep1 not found in map");
    }
    std::set<SubGroup*> & sgList1 = itMapPeptideSg->second;
    itMapPeptideSg = _mapPeptide2subGroup.find(p_pep2);
    if (itMapPeptideSg == _mapPeptide2subGroup.end()) {
        throw new GpError("GroupNg::pep1pep2sharingSubgroup ERROR : pep2 not found in map");
    }
    std::set<SubGroup*> & sgList2 = itMapPeptideSg->second;

    if (alreadyLinkedSg.size() == 0) {
        alreadyLinkedSg.insert(sgList2.begin(), sgList2.end());
    }
    std::set<SubGroup *>::const_iterator itSg;
    for (itSg = sgList1.begin(); itSg != sgList1.end() ; itSg++) {
        if (alreadyLinkedSg.find(*itSg) != alreadyLinkedSg.end()) {
            qDebug() << "GroupNg::pep1pep2sharingSubgroup there is something in common between pep1 pep2";
            alreadyLinkedSg.insert(sgList1.begin(), sgList1.end());
            return true;
        }
    }
    qDebug() << "GroupNg::pep1pep2sharingSubgroup there is something NOTHING in common between pep1 pep2";

    //deep search is needed
    if (makeItGrow(alreadyLinkedSg)) {
        qDebug() << "GroupNg::pep1pep2sharingSubgroup there is something make it grow";
        return pep1pep2sharingSubgroup(alreadyLinkedSg, p_pep1, p_pep2);
    }
    else {
        qDebug() << "GroupNg::pep1pep2sharingSubgroup there is something NO make it grow";
        return false;
    }

}


bool GroupNg::isBondingPeptide(std::set<SubGroup*> & alreadyLinkedSg, const SubGroup* p_sg) {
    std::set< Peptide* >::const_iterator itPep2 = p_sg->getPeptideSet().getList().begin();
    std::set< Peptide* >::const_iterator itPep1 = p_sg->getPeptideSet().getList().begin()++;
    std::set< Peptide* >::const_reverse_iterator itPepR = p_sg->getPeptideSet().getList().rbegin();
    qDebug() << "GroupNg::isBondingPeptide begin peptideSet size " << p_sg->getPeptideSet().getList().size();

    bool alreadyLinked = true;

    if (pep1pep2sharingSubgroup(alreadyLinkedSg, *itPepR, *itPep2)) {
        qDebug() << "GroupNg::isBondingPeptide begin *itPepR, *itPep2 OK";
        for (; itPep1 != p_sg->getPeptideSet().getList().end(); ++itPep1,++itPep2) {
            if (pep1pep2sharingSubgroup(alreadyLinkedSg, *itPep1, *itPep2)) {
                qDebug() << "GroupNg::isBondingPeptide begin *itPep1, *itPep2 OK";
            }
            else {
                alreadyLinked = false;
                break;
            }
        }
        qDebug() << "GroupNg::isBondingPeptide " << !alreadyLinked;
    }
    else {
        qDebug() << "GroupNg::isBondingPeptide true";
        alreadyLinked =  false;
    }
    //qDebug() << "GroupNg::isBondingPeptide true some peptides does not have direct link... deep link check needed ";
    return !alreadyLinked;

}


void GroupNg::removeNonInformativeGroup(std::set<GroupNg *> * p_groupSet) {
    std::set<GroupNg *> groupList;
    GroupNg * p_cleanGroup;
    if (this->_subGroupSet.size() > 10) {
        std::cerr << "the current group contains " << this->_subGroupSet.size() << " subgroups" << std::endl;
    }
    p_cleanGroup = removeNonInformativeSubGroup(&groupList);
    while ((p_cleanGroup == 0) && (groupList.size() == 0)) {
        p_cleanGroup = removeNonInformativeSubGroup(&groupList);
    }
    if (p_cleanGroup != 0) {
        p_groupSet->insert(p_cleanGroup);
        return;
    }

    std::cerr << "group split in " << groupList.size() << " new groups" << std::endl;
    int i=1;
    for (std::set< GroupNg* >::iterator itg = groupList.begin(); itg != groupList.end(); itg++,i++) {
        std::cerr << "group " << i << " contains " << (*itg)->getSubGroupSet().size() << " subgroups" << std::endl;
    }

    for (std::set< GroupNg* >::iterator itg = groupList.begin(); itg != groupList.end(); itg++) {
        (*itg)->removeNonInformativeGroup(p_groupSet);
    }



}

/** remove non informative subgroups inside a group
 *  the resulting group if any is a "clean" group : with informative subgroups
 *  if the result is 0, then removal is not finished on this subgroup
 *  if the result is 0 AND p_groupSet contains new group : the current group was splitted, it needs removal again
 *
 * - it detects non informative subgroups (no specific peptide) if there is no non informative subgroups : the current group is clean return this
 * - it chooses only one subgroup to remove at a time (determined ordered not informative subgroups)
 * - it checks if this subgroup to remove is a bonding one (if it is the last bond : it will create new groups)
 * - it removes the non informative subgroup
 * - if it was NOT a bonding peptide : proceed to the next subgroup to remove, return 0
 * - if this is a bonding peptide : create new groups and to split current group and return 0
 *
 */
GroupNg * GroupNg::removeNonInformativeSubGroup(std::set<GroupNg *> * p_groupSet) {
    std::set<SubGroup *> notInfoSubGroup;
    notInfoSubGroup.insert(_subGroupSet.getSubgroups().begin(), _subGroupSet.getSubgroups().end());
    //SubGroupSet _subGroupSet;
    qDebug() << "GroupNg::removeNonInformative begin notInfoSubGroup.size():" << notInfoSubGroup.size() << " _peptideSet.size():" << _peptideSet.size()<< " _mapPeptide2subGroup.size():" << _mapPeptide2subGroup.size()<< "  " << this;
    //qDebug() << this->printSg();
    std::set<SubGroup *>::const_iterator sgIt;
    // _mapPeptide2subGroup.clear();
    //for (sgIt = _subGroupSet.getSubgroups().begin(); sgIt != _subGroupSet.getSubgroups().end(); sgIt++) {
    //(*sgIt)->debugPeptide();
    ///indexPeptide2SubGroup(*sgIt);
    // }
    //qDebug() << "GroupNg::removeNonInformative begin " << notInfoSubGroup.size() << _peptideSet.size() << _mapPeptide2subGroup.size();

    std::map<const Peptide*,std::set<SubGroup*>>::iterator itMapPeptideSg;
    for (itMapPeptideSg = _mapPeptide2subGroup.begin(); itMapPeptideSg != _mapPeptide2subGroup.end(); itMapPeptideSg++) {
        //qDebug() << "itMapPeptideSg->second.size() " <<itMapPeptideSg->second.size();
        if (itMapPeptideSg->second.size() == 1) {
            //only one subgroup has this peptide, this peptide is specific to this subgroup so this subgroup is informative
            notInfoSubGroup.erase(*(itMapPeptideSg->second.begin()));
            //qDebug() << "specific peptide " << itMapPeptideSg->first->getSequenceLi() << " " << *(itMapPeptideSg->second.begin());
        }
    }
    qDebug() << "GroupNg::removeNonInformative notInfoSubGroup.size():" << notInfoSubGroup.size()<< " _subGroupSet.size():" << _subGroupSet.size();
    if (notInfoSubGroup.size() > 0) {

        SubGroup * toRemove = NULL;
        SubGroup * sg;
        //there is no scan at all ... we make the difference given the alphabetical order of peptide+mh
        QString peptideStr;
        for (sgIt = notInfoSubGroup.begin(); sgIt != notInfoSubGroup.end(); sgIt++) {
            sg = *sgIt;
            if (peptideStr.isEmpty()) {
                peptideStr = sg->getNonInformativeOrderString();
                toRemove = sg;
            }
            else {
                if (sg->getNonInformativeOrderString() > peptideStr) {
                    peptideStr = sg->getNonInformativeOrderString();
                    toRemove = sg;
                }
            }
            // qDebug() << " Group::removeNonInformative peptideStr : " << peptideStr;
        }

        qDebug() << "GroupNg::removeNonInformative one sg chosen " << toRemove;

        this->removeSubGroup(toRemove);

        std::set<SubGroup*> alreadyLinkedSg;

        if (!this->isBondingPeptide(alreadyLinkedSg, toRemove)) {
            //there is no split, continue to remov non informative subgroup :
            qDebug() << "this is NOT a bonding peptide no need to regroup";
            //removeNonInformativeGroup(p_groupSet);
        }
        else {
            qDebug() << "this is a bonding peptide heavy grouping needed";

            std::list<GroupNg *> groupSet;
            std::set< Peptide* >::iterator itPep = toRemove->getPeptideSet().getList().begin();

            while(this->makeItGrow(alreadyLinkedSg)) {
            }
            GroupNg * p_grp = new GroupNg(alreadyLinkedSg);
            //p_grp->fillFromPeptide(_mapPeptide2subGroup, *itPep);
            groupSet.push_back(p_grp);
            itPep++;
            for (; itPep != toRemove->getPeptideSet().getList().end(); ++itPep) {
                //qDebug() << "GroupNg::removeNonInformative " << (*itPep)->getSequenceLi();
                bool createNewGroup = true;
                for (std::list<GroupNg *>::iterator itGrp = groupSet.begin(); itGrp != groupSet.end(); ++itGrp) {
                    if ((*itGrp)->contains(*itPep)) {
                        createNewGroup = false;
                        break;
                    }
                }
                if (createNewGroup) {
                    alreadyLinkedSg.clear();
                    qDebug() << "GroupNg::removeNonInformative create a new group " << (*itPep)->getSequenceLi();
                    //bootstrap subgroup :
                    SubGroup*const& bootstrapSg = *(_mapPeptide2subGroup.find(*itPep)->second.begin());
                    alreadyLinkedSg.insert(bootstrapSg);
                    while(this->makeItGrow(alreadyLinkedSg)) {
                    }
                    GroupNg * p_grp = new GroupNg(alreadyLinkedSg);
                    //p_grp->fillFromPeptide(_mapPeptide2subGroup, *itPep);
                    groupSet.push_back(p_grp);
                }

            }
            //delete p_grp;
            qDebug() << "GroupNg::removeNonInformative groupSet.size() " << groupSet.size()<<" "<< this;
            if(groupSet.size() == 1) {
                //it needs a deep bind search
                //removeNonInformativeGroup(p_groupSet);
                //there is no split : this should have been detected by isBondingPeptide function
                throw new GpError("GroupNg::removeNonInformativeGroup ERROR : there is no split : this should have been detected by isBondingPeptide function");
            }
            else {
                qDebug() << "GroupNg::removeNonInformative split "<< this;
                std::cerr << "non informative peptide sequence " << peptideStr.toStdString() << " removed" << std::endl;

                //there was a split
                //forget current Group
                this->_mapPeptide2subGroup.clear();
                this->_peptideSet.clear();
                this->_subGroupSet.clear();
                std::list<GroupNg *>::iterator itG;
                for (itG = groupSet.begin(); itG != groupSet.end(); itG++) {
                    //(*itG)->removeNonInformativeGroup(p_groupSet);
                    p_groupSet->insert(*itG);
                }

            }
        }
        //return;
        //this->fragmentGroup(p_groupSet, toRemove);
        // throw GpError(QObject::tr("test stop"));
    }
    else {
        qDebug() << "GroupNg::removeNonInformative the group is clean. no not informative subgroup."<< this;
        //p_groupSet->insert(this);
        return this;
    }
    return 0;
    qDebug() << "GroupNg::removeNonInformative end " << this;
}
