/*
 *
 * Protein Grouper
 * Copyright (C) 2014 Olivier Langella, Benoit Valot, Michel Zivy.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *

 * group.cpp
 *
 *  Created on: 22 mars 2013 by Author: valot
 *
 */

#include "group.h"
#include <vector>
#include <QDebug>
#include "../pg_utils.h"

#include "../gp_error.h"


Group::Group() {
}

Group::Group(SubGroup * subGroup) {
    _peptideSet.addAll(subGroup->getPeptideSet());
    _subGroupSet.addSubGroup(subGroup);
}

Group::~Group() {
    // qDebug() << "Group::~Group " ;
    _peptideSet.clear();
    _subGroupSet.clear();
}

const QString Group::getOrderString() const {

    QString index = PgUtils::getLexicalOrderedString(this->getPeptideSet().getTotalSpectra());
    index.append(PgUtils::getLexicalOrderedString(this->getSubGroupSet().size()));
    //total number of protein match
    //index.append(this->getSubGroupSet().getFirstAccession());
    qDebug() << "QString Group::getOrderString() " << index;
    return index;
}

bool Group::isEmpty() const {
    if (_subGroupSet.size()==0) {
        qDebug() << "Group::isEmpty() _subGroupSet empty " ;
        return true;
    }
    if (_peptideSet.size()==0) {
        qDebug() << "Group::isEmpty() _peptideSet empty " ;
        return true;
    }
    return false;
}

const PeptideSet& Group::getPeptideSet() const {
    if (_peptideSet.size()==0) {
        throw new GpError("_peptideSet is empty in group.");
    }
    return _peptideSet;
}
const SubGroupSet& Group::getSubGroupSet() const {
    if (_subGroupSet.size()==0) {
        throw new GpError("_subGroupSet is empty in group.");
    }
    return (_subGroupSet);
}

void Group::eatGroup(const Group * groupIncoming) {
    if (this == groupIncoming)
        return;
    qDebug() << "Group::eatGroup begin " << this << " " << groupIncoming;
    qDebug() << "Group::eatGroup group size " << this->getSubGroupSet().size();

    /*
    std::set<SubGroup *>::iterator it;
    SubGroup * sg;
    for (it = _subGroupSet.getSubgroups().begin(); it != _subGroupSet.getSubgroups().end(); it++) {
        sg = *it;
        qDebug() << "Group::eatGroup " << sg->getNumberOfProteinMatch();
        sg->debugProtein();
    }
    */

    _peptideSet.addAll(groupIncoming->getPeptideSet());
    _subGroupSet.addAll(groupIncoming->getSubGroupSet());
    qDebug() << "Group::eatGroup new  group size "
             << this->getSubGroupSet().size();
    /*
    for (std::set<SubGroup *>::iterator it = _subGroupSet.getSubgroups().begin(); it != _subGroupSet.getSubgroups().end(); it++) {
        qDebug() << "Group::eatGroup after " << (*it)->getNumberOfProteinMatch();
        (*it)->debugProtein();
    }
    */
}

bool Group::fusionWith(Group * groupIncoming) {
    bool fusion = groupIncoming->getPeptideSet().containsOnePeptideIn(
                      _peptideSet);

    if (fusion) {
        _peptideSet.addAll(groupIncoming->getPeptideSet());
        _subGroupSet.addAll(groupIncoming->getSubGroupSet());
    }
    return fusion;
}

bool Group::removeNonInformative() {
    qDebug() << "Group::removeNonInformative begin " << this;
    qDebug() << "Group::removeNonInformative _subGroupSet.size() " << _subGroupSet.size()<< this;
    if (_subGroupSet.size() < 3) {
        // qDebug() << "Group::removeNonInformative < 3 " << this;
        return false;
    }
    std::set<SubGroup *> notInfoSubGroup;
    std::list<SubGroup *>::const_iterator it;
    SubGroup * sg;
    for (it = _subGroupSet.getSubgroups().begin(); it != _subGroupSet.getSubgroups().end(); it++) {
        sg = *it;
        if (!this->hasSpecificPeptideInSubGroup(sg)) {
            notInfoSubGroup.insert(sg);
        }
    }
    qDebug() << "Group::removeNonInformative begin notInfoSubGroup.size() "<< notInfoSubGroup.size() << this;
    if (notInfoSubGroup.size() == 0) {
        //qDebug() << "Group::removeNonInformative begin notInfoSubGroup.size() == 0 "<< notInfoSubGroup.size() << this;
        return false;
    } else if (notInfoSubGroup.size() == 1) {
        //qDebug() << "Simple non informative case";
        _subGroupSet.removeSubGroup(*notInfoSubGroup.begin());
        return true;
    }
    //Multiple non informative, choose the less number of scans (the smallest subgroup in term of scans)
    qDebug() << "Multiple Non informative case : " << notInfoSubGroup.size();
    std::set<SubGroup *>::iterator notInfoIt;
    SubGroup * toRemove = NULL;
    //there is no scan at all ... we make the difference given the alphabetical order of peptide+mh
    QString peptideStr;
    for (notInfoIt = notInfoSubGroup.begin();
            notInfoIt != notInfoSubGroup.end(); notInfoIt++) {
        sg = *notInfoIt;
        if (peptideStr.isEmpty()) {
            peptideStr = sg->getNonInformativeOrderString();
            toRemove = sg;
        }
        else {
            if (sg->getNonInformativeOrderString() < peptideStr) {
                peptideStr = sg->getNonInformativeOrderString();
                toRemove = sg;
            }
        }
        // qDebug() << " Group::removeNonInformative peptideStr : " << peptideStr;
    }

    //c
    _subGroupSet.removeSubGroup(toRemove);
    removeNonInformative();
    //  qDebug() << "end 1 Group::removeNonInformative _subGroupSet.size() " << _subGroupSet.size()<< this;
    // qDebug() << this->getSubGroupSet().getFirstAccession();
    // qDebug() << "end 2 Group::removeNonInformative _subGroupSet.size() " << _subGroupSet.size()<< this;
    return true;
}

bool Group::hasSpecificPeptideInSubGroup(SubGroup * subGroup) const {
    std::list<SubGroup *>::const_iterator it;
    std::set<Peptide *>::const_iterator peptideIt;
    Peptide* pep;
    SubGroup* sg;
    for (peptideIt = subGroup->getPeptideSet().getList().begin();
            peptideIt != subGroup->getPeptideSet().getList().end();
            peptideIt++) {
        pep = *peptideIt;
        bool thisPeptideExistInAnOtherSg = false;
        for (it = _subGroupSet.getSubgroups().begin();
                (thisPeptideExistInAnOtherSg == false)
                && (it != _subGroupSet.getSubgroups().end()); it++) {
            sg = *it;
            if ((sg != subGroup) && (sg->getPeptideSet().contains(pep))) {
                thisPeptideExistInAnOtherSg = true;
            }
        }
        //View one time == specific
        if (thisPeptideExistInAnOtherSg == false) {
            return true;
        }
    }
    return (false);
}
